#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import (
   timestampToStr,
   utcTimestampToStr,
)
import Arnet
from ArnetModel import MacAddress
from CliPlugin.CfmCliLib import (
   mepDirectionShowStr,
   pmOperModeKwStr,
   mdFormatLegend,
   maFormatLegend,
)
from CliPlugin import IntfCli
from CliModel import (
      Bool,
      Dict,
      Enum,
      Float,
      Int,
      List,
      Model,
      Str,
      Submodel,
)
from CfmLib import (
   CcmDefectAlarmEnum,
   ccmTxIntervalToCliToken,
   CcmTxIntervalWithUnit,
   getDefectAlarmNameFromEnum,
)
from CfmTypes import (
   tacCfmDefectCode,
   tacCcmDefectAlarm,
   tacMdNameFormat,
   tacMaNameFormat
)
import Ethernet
from IntfModels import Interface
import TableOutput
import Tac
import TacSigint
from Toggles.CfmToggleLib import (
   toggleCfmSwUpMepCcmEnabled,
   toggleCfmProgStateEnabled,
   toggleCfmPmMiEnabled,
)
from TypeFuture import TacLazyType

MepDirection = TacLazyType( 'Cfm::MepDirection' )
DefectCodeEnum = TacLazyType( 'Cfm::DefectCode' )
MiSuspectEnum = TacLazyType( 'Cfm::Smash::MiSuspect' )
pmRmepStatusEnum = ( 'failed', 'inProgress', 'notConfigured', 'suspendedForLoc' )

class CcSummaryDefectAlarmModel( Model ):
   defectAlarm = Enum( values=CcmDefectAlarmEnum.attributes, help="CCM defect" )
   reason = Enum( values=( "unsupported", ),
                  help="Reason for inactive CCM defect", optional=True )

class CfmSummaryProfileModel( Model ):
   maCount = Int( help="Number of MAs associated with this profile" )
   ccmTxInterval = Float( help="Operational CCM TX interval in seconds",
                          optional=True )
   activeDefectAlarms = List( valueType=CcSummaryDefectAlarmModel,
                              help="Continuity check defects that will raise an "
                                   "alarm", optional=True )
   inactiveDefectAlarms = List( valueType=CcSummaryDefectAlarmModel,
                                help="Continuity check defects that will not raise "
                                     "an alarm", optional=True )

   def defectAlarmsToString( self, defectAlarmModels ):
      if not defectAlarmModels:
         return "none"
      defectAlarms = []
      for defectAlarmModel in sorted( defectAlarmModels,
                                      key=lambda x:
                                       ( tacCcmDefectAlarm( x.defectAlarm ) ) ):
         defectAlarmStr = getDefectAlarmNameFromEnum( defectAlarmModel.defectAlarm )
         if defectAlarmModel.reason is not None:
            defectAlarmStr += f" ({defectAlarmModel.reason})"
         defectAlarms.append( defectAlarmStr )
      return ", ".join( defectAlarms )

   def render( self ):
      print( f"Maintenance associations: {self.maCount}" )
      if not self.maCount:
         print( "Operational CCM TX interval: n/a" )
         print( "Active defect alarms: n/a" )
         print( "Inactive defect alarms: n/a" )
         return
      ccmTxIntervalWithUnit = CcmTxIntervalWithUnit( 0, 'seconds' )
      if self.ccmTxInterval in ccmTxIntervalToCliToken:
         ccmTxIntervalWithUnit = ccmTxIntervalToCliToken[ self.ccmTxInterval ]
      print( "Operational CCM TX interval: %s %s" %
             ( ccmTxIntervalWithUnit.ccmTxInterval, ccmTxIntervalWithUnit.unit ) )
      print( "Active defect alarms: "
             f"{self.defectAlarmsToString( self.activeDefectAlarms )}" )
      print( "Inactive defect alarms: "
             f"{self.defectAlarmsToString( self.inactiveDefectAlarms )}" )

class CcSummaryLocActionModel( Model ):
   action = Enum( values=( "logging", "interfaceRoutingDisable" ),
                  help="Action on loss of connectivity" )
   reason = Enum( values=( "unsupported", ),
                  help="Reason for inactive action on loss of continuity action",
                  optional=True )

class CfmSummaryModel( Model ):
   ccActive = Bool( help="Continuity check is active" )
   ccInactiveReason = Enum( values=( "unsupported", ),
                            help="Reason for inactive continuity check",
                            optional=True )
   ccHwOffloadActive = Bool( help="CCM hardware offload is active" )
   ccHwOffloadInactiveReason = Enum( values=( "unsupported", ),
                                     help="Reason for inactive CCM hardware offload",
                                     optional=True )
   locActions = List( valueType=CcSummaryLocActionModel,
                      help="Actions taken on loss of connectivity",
                      optional=True )
   inactiveLocActions = List( valueType=CcSummaryLocActionModel,
                              help="Actions not taken on loss of "
                                   "connectivity", optional=True )
   mdCount = Int( help="Number of MDs" )
   maCount = Int( help="Number of MAs" )
   mepCount = Int( help="Number of MEPs" )
   profiles = Dict( keyType=str, valueType=CfmSummaryProfileModel,
                    help="Mapping between profile name and its status" )

   def locActionsToString( self, locActionModels ):
      if not locActionModels:
         return "none"
      locActions = []
      for locActionModel in sorted( locActionModels,
                                    key=lambda x: ( x.action ) ):
         locAction = locActionModel.action
         if locAction == "interfaceRoutingDisable":
            # Un-camel case the LOC action name.
            locAction = "interface routing disable"
         if locActionModel.reason is not None:
            locAction += f" ({locActionModel.reason})"
         locActions.append( locAction )
      return ", ".join( locActions )

   def render( self ):
      ccStatus = "active" if self.ccActive else "inactive"
      if self.ccInactiveReason is not None:
         ccStatus += f" ({self.ccInactiveReason})"
      print( f"Continuity check status: {ccStatus}" )
      hwOffloadStatus = "active" if self.ccHwOffloadActive else "inactive"
      if self.ccHwOffloadInactiveReason is not None:
         hwOffloadStatus += f" ({self.ccHwOffloadInactiveReason})"
      print( f"CCM hardware offload: {hwOffloadStatus}" )
      print( "Action on loss of connectivity: "
             f"{self.locActionsToString( self.locActions )}" )
      print( "Inactive action on loss of connectivity: "
             f"{self.locActionsToString( self.inactiveLocActions )}" )
      print( f"Maintenance domains: {self.mdCount}" )
      print( f"Maintenance associations: {self.maCount}" )
      print( f"Maintenance end points: {self.mepCount}" )
      for profile, profileModel in sorted( self.profiles.items() ):
         print()
         print( f"Profile: {profile}" )
         profileModel.render()

class LocalMepCcCountersModel( Model ):
   __public__ = toggleCfmSwUpMepCcmEnabled()
   intfId = Str( help="The interface ID" )
   vlanId = Int( help="The VLAN ID" )
   txCount = Int( help="Number of CCM packets sent" )
   txFailureCount = Int( help="Number of CCM packets that failed to send" )
   rxCount = Int( help="Number of CCM packets received" )
   rxDropCount = Int( help="Number of CCM packets that were dropped" )
   ccmPacketParseError = Int( help="Number of CCM packets that were dropped due to "
                                   "an error in packet parsing" )
   invalidMdName = Int( help="Number of CCM packets that were dropped due to an "
                             "invalid MD name" )
   invalidMaName = Int( help="Number of CCM packets that were dropped due to an "
                             "invalid MA name" )
   directionMismatch = Int( help="Number of CCM packets that were dropped due to a "
                                 "direction mismatch" )
   ccmIntervalMismatch = Int( help="Number of CCM packets that were dropped due to a"
                                   " CCM interval mismatch" )
   invalidRemoteMepId = Int( help="Number of CCM packets that were dropped due to an"
                                  " invalid remote MEP ID" )
   duplicateRemoteMep = Int( help="Number of CCM packets that were dropped due to a "
                                  "duplicate remote MEP" )
   unconfiguredRemoteMep = Int( help="Number of CCM packets that were dropped due "
                                     "to an unconfigured remote MEP" )

class MaCcCountersModel( Model ):
   __public__ = toggleCfmSwUpMepCcmEnabled()
   localMeps = Dict( keyType=int, valueType=LocalMepCcCountersModel,
                     help="Local MEPs, keyed by their ID" )

class MdCcCountersModel( Model ):
   __public__ = toggleCfmSwUpMepCcmEnabled()
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaCcCountersModel,
                        help="MAs, keyed by their name" )

def renderMep( domainName, mdModel ):
   for maName in sorted( mdModel.associations ):
      maModel = mdModel.associations[ maName ]
      for mepId in sorted( maModel.localMeps ):
         localMepModel = maModel.localMeps[ mepId ]
         print( f"Maintenance domain: {domainName}, Level: {mdModel.level}" )
         print( f"Maintenance association: {maName}" )
         print( f"Maintenance end point ID: {mepId}, Interface: "
                  f"{localMepModel.intfId}, VLAN: {localMepModel.vlanId}" )
         print( f"Total TX packets: {localMepModel.txCount}" )
         print( f"Total TX packet failures: {localMepModel.txFailureCount}" )
         print( f"Total RX packets: {localMepModel.rxCount}" )
         print( f"Total RX packet discards: {localMepModel.rxDropCount}" )
         print( f"RX - CCM packet parse errors: "
                  f"{localMepModel.ccmPacketParseError}" )
         print( f"RX - Domain name invalid: {localMepModel.invalidMdName}" )
         print( f"RX - Association name invalid: {localMepModel.invalidMaName}" )
         print( f"RX - Traffic direction mismatch: "
                  f"{localMepModel.directionMismatch}" )
         print( f"RX - CCM interval invalid: "
                  f"{localMepModel.ccmIntervalMismatch}" )
         print( f"RX - Remote MEP ID invalid: "
                  f"{localMepModel.invalidRemoteMepId}" )
         print( f"RX - Duplicate remote MEP: {localMepModel.duplicateRemoteMep}" )
         print( f"RX - Unconfigured remote MEP: "
                  f"{localMepModel.unconfiguredRemoteMep}" )
         print()

class CcCountersModel( Model ):
   __public__ = toggleCfmSwUpMepCcmEnabled()
   domains = Dict( keyType=str, valueType=MdCcCountersModel,
                   help="Maintenance domains, keyed by their name" )

   def render( self ):
      for domainName in sorted( self.domains ):
         renderMep( domainName, self.domains[ domainName ] )

class IntfVlanCcCountersModel( CcCountersModel ):
   __public__ = toggleCfmSwUpMepCcmEnabled()
   txCount = Int( help="Number of CCM packets that were sent" )
   txFailureCount = Int( help="Number of CCM packets that failed to send" )
   rxCount = Int( help="Number of CCM packets that were received" )
   rxDropCount = Int( help="Number of CCM packets that were dropped" )
   noMepConfigured = Int( help="Number of CCM packets that were dropped due to no "
                               "MEP being configured" )
   ethHeaderParseError = Int( help="Number of CCM packets that were dropped due to "
                                   "an error in ethernet header parsing" )
   cfmHeaderParseError = Int( help="Number of CCM packets that were dropped due to "
                                   "an error in CFM header parsing" )
   mdLevelMismatch = Int( help="Number of CCM packets that were dropped due to an "
                               "error in MD level mismatch" )
   firstTlvOffsetInvalid = Int( help="Number of CCM packets that were dropped due to"
                                     " an invalid first TLV offset" )

   def render( self ):
      print( f"Total TX packets: {self.txCount}" )
      print( f"Total TX packet failures: {self.txFailureCount}" )
      print( f"Total RX packets: {self.rxCount}" )
      print( f"Total RX packet discards: {self.rxDropCount}" )
      print( f"RX - No MEP configured: {self.noMepConfigured}" )
      print( f"RX - Ethernet header parse errors: {self.ethHeaderParseError}" )
      print( f"RX - CFM header parse errors: {self.cfmHeaderParseError}" )
      print( f"RX - Domain level mismatch: {self.mdLevelMismatch}" )
      print( f"RX - TLV offset invalid: {self.firstTlvOffsetInvalid}" )
      print()
      for domainName in sorted( self.domains ):
         renderMep( domainName, self.domains[ domainName ] )

class IntfCcCountersModel( Model ):
   __public__ = toggleCfmSwUpMepCcmEnabled()
   vlans = Dict( keyType=int,
                 valueType=IntfVlanCcCountersModel,
                 help="CC counters, keyed by VLAN ID" )

class CcCountersDetailModel( Model ):
   __public__ = toggleCfmSwUpMepCcmEnabled()
   interfaces = Dict( keyType=Interface,
                      valueType=IntfCcCountersModel,
                      help="CC counters, keyed by interface" )
   noIntfVlanMapping = Submodel( valueType=CcCountersModel,
                       help="CC counters without interface VLAN mapping",
                       optional=True )

   def render( self ):
      for intfId in Arnet.sortIntf( self.interfaces ):
         intfModel = self.interfaces[ intfId ]
         for vlanId in sorted( intfModel.vlans ):
            print( f"Interface: {intfId}, VLAN: {vlanId}" )
            intfModel.vlans[ vlanId ].render()
      if self.noIntfVlanMapping is not None:
         print( "Interface: none, VLAN: 0" )
         print( "Total TX packets: n/a" )
         print( "Total TX packet failures: n/a" )
         print( "Total RX packets: n/a" )
         print( "Total RX packet discards: n/a" )
         print( "RX - No MEP configured: n/a" )
         print( "RX - Ethernet header parse errors: n/a" )
         print( "RX - CFM header parse errors: n/a" )
         print( "RX - Domain level mismatch: n/a" )
         print( "RX - TLV offset invalid: n/a" )
         print( "" )
         self.noIntfVlanMapping.render()

class CfmIntfCounter( Model ):
   inPkts = Int( help="Number of CFM packets received on the interface" )
   outPkts = Int( help="Number of CFM packets sent from the interface" )
   errorPkts = Int(
      help="Number of invalid CFM packets received on the interface" )

class CfmCounter( Model ):
   interfaces = Dict( keyType=Interface,
      valueType=CfmIntfCounter,
      help="Maps CFM counters to the corresponding interfaces" )

   def render( self ):
      headings = ( "Interface", "InPkts", "OutPkts", "ErrorPkts" )
      table = TableOutput.createTable( headings )
      fmtName = TableOutput.Format( justify="left" )
      fmtName.noPadLeftIs( True )
      fmtName.padLimitIs( True )
      fmtPkts = TableOutput.Format( justify="right" )
      fmtPkts.noPadLeftIs( True )
      fmtPkts.padLimitIs( True )
      table.formatColumns( fmtName, fmtPkts, fmtPkts, fmtPkts )

      for intf in Arnet.sortIntf( self.interfaces ):
         model = self.interfaces[ intf ]
         table.newRow( IntfCli.Intf.getShortname( intf ),
                       model.inPkts, model.outPkts, model.errorPkts )
      print( table.output() )

class CfmMaintenanceDomain( Model ):
   intermediatePoint = Dict( keyType=str, valueType=bool,
                              help='Maps maintenance domains to intermediate point' )

   def render( self ):
      table = TableOutput.createTable( ( 'Domain', 'Intermediate Point' ) )
      fmtName = TableOutput.Format( justify='right' )
      fmtName.noPadLeftIs( True )
      fmtName.padLimitIs( True )
      fmtMip = TableOutput.Format( justify='right' )
      fmtMip.noPadLeftIs( True )
      fmtMip.padLimitIs( True )
      table.formatColumns( fmtName, fmtMip )

      for ( domain, defaultMip ) in sorted( self.intermediatePoint.items() ):
         table.newRow( domain, 'enabled' if defaultMip else 'disabled' )
      print( table.output() )

def renderDomainStr( domainName, md ):
   print( "Maintenance domain: %s, Level: %d" % ( domainName, md.level ) )

def renderRemoteMepStatusStr( remoteMepId, remoteMep ):
   if remoteMep.enabled:
      print( "Remote MEP ID: %d, Status: enabled" % remoteMepId )
   else:
      reason = "Unknown"
      if remoteMep.reason == "failed":
         reason = "Hardware programming has failed"
      elif remoteMep.reason == "inProgress":
         reason = "Hardware programming is in progress"
      elif remoteMep.reason == "notConfigured":
         reason = "Measurement is not configured"
      elif remoteMep.reason == "suspendedForLoc":
         reason = "Measurement is suspended due to LOC"
      print( "Remote MEP ID: %d, Status: disabled, Reason: %s" %
                                     ( remoteMepId, reason ) )

def renderMaStr( maName, ma, withCcmTx=True, withPm=False ):
   if withCcmTx:
      ccmTxIntervalWithUnit = CcmTxIntervalWithUnit( 0, 'seconds' )
      if ma.ccmTxInterval in ccmTxIntervalToCliToken:
         ccmTxIntervalWithUnit = ccmTxIntervalToCliToken[ ma.ccmTxInterval ]
      print( "Maintenance association: %s, Direction: %s, CCM TX interval: %s %s"
            % ( maName, ma.direction, ccmTxIntervalWithUnit.ccmTxInterval,
                ccmTxIntervalWithUnit.unit ) )
   else:
      print( "Maintenance association: {}, Direction: {}".format( maName,
      ma.direction ) )
   if withPm:
      print( "Measurement type: %s, TX interval: %s milliseconds"
             % ( ma.operMode, ma.txInterval ) )

def renderNoMep( domainName, md, maName, ma, withCcmTx=True, withPm=False ):
   if not ma.localMeps:
      renderDomainStr( domainName, md )
      renderMaStr( maName, ma, withCcmTx=withCcmTx, withPm=withPm )
      print( "No maintenance end points" )
      print( "" )
      return True
   return False

def renderNoMa( domainName, md ):
   if not md.associations:
      renderDomainStr( domainName, md )
      print( "No maintenance associations" )
      print( "" )
      return True
   return False

def getMepProgReasonStr( localMepDetailStatus ):
   reasonStr = ""
   rxReason = localMepDetailStatus.rxReason
   txReason = localMepDetailStatus.txReason
   if rxReason and txReason:
      if rxReason == txReason:
         if rxReason == "waiting":
            reasonStr = "waiting to be programmed"
         elif rxReason == "programmingFailed":
            reasonStr = "programming failed"
      else:
         if rxReason == "waiting":
            reasonStr = "waiting to be programmed as receiver"
         elif rxReason == "programmingFailed":
            reasonStr = "unable to program as receiver"
         if txReason == "waiting":
            reasonStr += ", waiting to be programmed as transmitter"
         elif txReason == "programmingFailed":
            reasonStr += ", unable to program as transmitter"
   elif rxReason:
      if rxReason == "waiting":
         reasonStr = "waiting to be programmed as receiver"
      elif rxReason == "programmingFailed":
         reasonStr = "unable to program as receiver"
   elif txReason:
      if txReason == "waiting":
         reasonStr = "waiting to be programmed as transmitter"
      elif txReason == "programmingFailed":
         reasonStr = "unable to program as transmitter"
   return reasonStr

def getMepProgStatusStr( localMepDetailStatus ):
   statusStr = ""
   if localMepDetailStatus is not None:
      rxProgrammed = localMepDetailStatus.rxProgrammed
      txProgrammed = localMepDetailStatus.txProgrammed
      if rxProgrammed is None and txProgrammed is None:
         statusStr += "n/a"
      else:
         if rxProgrammed:
            statusStr += "receiver"
            if txProgrammed:
               statusStr += ", transmitter"
         else:
            if txProgrammed:
               statusStr += "transmitter"
            else:
               statusStr += "inactive"
         reasonStr = getMepProgReasonStr( localMepDetailStatus )
         if reasonStr:
            statusStr += " (" + reasonStr + ")"
   else:
      statusStr += "n/a"
   return statusStr

def getRemoteMepReasonStr( reason ):
   reasonStr = ""
   if reason == "waiting":
      reasonStr = "waiting to be programmed"
   elif reason == "programmingFailed":
      reasonStr = "programming failed"
   return reasonStr

def getRemoteMepStatusStr( remoteMepDetailStatus ):
   statusStr = "n/a"
   if remoteMepDetailStatus.status is not None:
      statusStr = remoteMepDetailStatus.status
      if remoteMepDetailStatus.totalSessions and \
         remoteMepDetailStatus.activeSessions is not None:
         # If we have valid data about active and configured
         # sessions, then display it.
         statusStr += " " + str( remoteMepDetailStatus.activeSessions ) + \
                         "/" + str( remoteMepDetailStatus.totalSessions )
      if remoteMepDetailStatus.reason:
         statusStr += " (" + getRemoteMepReasonStr(
                      remoteMepDetailStatus.reason ) + ")"
   return statusStr

class RemoteMepDetailModel( Model ):
   __public__ = toggleCfmProgStateEnabled()
   # status is optional because it will be populated only if:
   # - the corresponding status exists, and/or
   # - the corresponding protocol is configured
   # If the protocol is not configured and there
   # is no programming status, then it will not be populated.
   status = Enum( values=( "active", "inactive" ),
                  help="Remote maintenance end point status",
                  optional=True )
   reason = Enum( values=( "waiting", "programmingFailed" ),
                  help="Reason for remote maintenance end point status",
                  optional=True )
   # Following are optional because only a subset of CFM protocols
   # support more than one session
   totalSessions = Int( help="Total number of SLM sessions", optional=True )
   activeSessions = Int( help="Total number of active SLM sessions", optional=True )

class RemoteMepModel( Model ):
   if toggleCfmProgStateEnabled():
      status = Enum( values=( "active", "inactive" ),
                     help="Remote maintenance end point status" )
      # Only needed for "detailed" version of the command, hence optional
      ccStatus = Submodel( valueType=RemoteMepDetailModel,
                           help="Remote maintenance end point detailed CC status",
                           optional=True )
      lmStatus = Submodel( valueType=RemoteMepDetailModel,
                           help="Remote maintenance end point detailed LM status",
                           optional=True )
      slmStatus = Submodel( valueType=RemoteMepDetailModel,
                           help="Remote maintenance end point detailed SLM status",
                           optional=True )
      dmStatus = Submodel( valueType=RemoteMepDetailModel,
                           help="Remote maintenance end point detailed DM status",
                           optional=True )
   else:
      status = Enum( values=( "active", "inactive", "rdi", "unreachable" ),
                     help="Remote maintenance end point status" )
   reason = Str( help="Reason for remote MEP status", optional=True )

class LocalMepVlanModel( Model ):
   activeVlans = List( valueType=int, help="Active VLANs", optional=True )
   primaryVlan = Int( help="Primary VLAN", optional=True )
   inactiveVlans = List( valueType=int, help="Inactive VLANs", optional=True )

class LocalMepDetailModel( Model ):
   __public__ = toggleCfmProgStateEnabled()
   # rxProgrammed and txProgrammed are optional because they will be
   # populated only if:
   # - the corresponding status exists, and/or
   # - the corresponding protocol is configured
   # If the protocol is not configured and there
   # is no programming status, then it will not be populated.
   rxProgrammed = Bool( help="Programmed as receiver", optional=True )
   txProgrammed = Bool( help="Programmed as transmitter", optional=True )
   rxReason = Enum( values=( "waiting", "programmingFailed" ),
                    help="Reason for receiver programming state",
                    optional=True )
   txReason = Enum( values=( "waiting", "programmingFailed" ),
                    help="Reason for transmitter programming state",
                    optional=True )

class LocalMepModel( Model ):
   intf = Interface( help="Local maintenance end point interface" )
   status = Enum( values=( "active", "inactive" ),
                  help="Local maintenance end point status" )
   vlanStatus = Submodel( valueType=LocalMepVlanModel,
                          help="Local maintenance end point VLAN information",
                          optional=True )
   remoteMeps = Dict( keyType=int, valueType=RemoteMepModel,
                      help="Remote MEPs, keyed by remote MEP ID" )
   if toggleCfmProgStateEnabled():
      # Only needed for "detailed" version of the command
      ccStatus = Submodel(
         valueType=LocalMepDetailModel,
         help="Local maintenance end point detailed CC status",
         optional=True )
      aisStatus = Submodel(
         valueType=LocalMepDetailModel,
         help="Local maintenance end point detailed AIS status",
         optional=True )

class MaModel( Model ):
   direction = Enum( help="MEP direction",
                     values=list( mepDirectionShowStr.values() ) )
   ccmTxInterval = Float( help="Continuity check transmission interval in seconds" )
   localMeps = Dict( keyType=int, valueType=LocalMepModel,
                     help="Local MEPs, keyed by local MEP ID" )

class MdModel( Model ):
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaModel,
                        help="A mapping between MA name and MA" )

class EndPointModel( Model ):
   domains = Dict( keyType=str, valueType=MdModel,
                   help="A mapping between MD name and MD" )

   def renderMep( self, domainName, md, maName, ma, localMepId, localMep ):
      renderDomainStr( domainName, md )
      renderMaStr( maName, ma )
      print( "Maintenance end point ID: %d, Status: %s, Interface: %s"
             % ( localMepId, localMep.status, localMep.intf.stringValue ) )
      # If localMep.vlanStatus is None, it means that L2 eth intfs are not
      # supported yet and we don't need to show the VLAN since there is no
      # concept of VLAN in CFM other than for L2 ports.
      if localMep.vlanStatus is not None:
         activeVlanStr = "none"
         primaryVlanStr = "none"
         inactiveVlanStr = "none"
         if localMep.vlanStatus.activeVlans:
            activeVlanStr = ""
            for activeVlan in sorted( localMep.vlanStatus.activeVlans ):
               if activeVlanStr:
                  activeVlanStr += ", "
               activeVlanStr += str( activeVlan )
         if "," in activeVlanStr:
            activeVlanStr = "Active VLANs: " + activeVlanStr
         else:
            activeVlanStr = "Active VLAN: " + activeVlanStr
         if localMep.vlanStatus.inactiveVlans:
            inactiveVlanStr = ""
            for inactiveVlan in sorted( localMep.vlanStatus.inactiveVlans ):
               if inactiveVlanStr:
                  inactiveVlanStr += ", "
               inactiveVlanStr += str( inactiveVlan )
         if "," in inactiveVlanStr:
            inactiveVlanStr = "Inactive VLANs: " + inactiveVlanStr
         else:
            inactiveVlanStr = "Inactive VLAN: " + inactiveVlanStr
         if localMep.vlanStatus.primaryVlan:
            primaryVlanStr = \
               "Primary VLAN: " + str( localMep.vlanStatus.primaryVlan )
         else:
            primaryVlanStr = "Primary VLAN: " + primaryVlanStr
         print( f"{activeVlanStr}, {primaryVlanStr}, {inactiveVlanStr}" )

      # -----------------------------------------------------------
      # Detailed local MEP status
      # -----------------------------------------------------------
      detailedOutput = False
      if toggleCfmProgStateEnabled() and ( localMep.ccStatus is not None or
         localMep.aisStatus is not None ):
         detailedOutput = True
         ccStatusStr = getMepProgStatusStr( localMep.ccStatus )
         print( "Continuity check:", ccStatusStr )
         aisStatusStr = getMepProgStatusStr( localMep.aisStatus )
         print( "Alarm indication:", aisStatusStr )

      if not localMep.remoteMeps:
         print( "No remote maintenance end points" )
         print( "" )
         return

      if detailedOutput:
         print( "" ) # new line before first Remote MEP
         for remoteMepId in sorted( localMep.remoteMeps ):
            remoteMep = localMep.remoteMeps[ remoteMepId ]
            # Overall status
            statusStr = "Remote MEP ID: " + str( remoteMepId ) + \
                        ", Status: " + remoteMep.status
            print( statusStr )
            # CCM status
            ccStatusStr = getRemoteMepStatusStr( remoteMep.ccStatus )
            print( "Continuity check:", ccStatusStr )
            # LM status
            lmStatusStr = getRemoteMepStatusStr( remoteMep.lmStatus )
            print( "Loss measurement:", lmStatusStr )
            # SLM status
            slmStatusStr = getRemoteMepStatusStr( remoteMep.slmStatus )
            print( "Synthetic loss measurement:", slmStatusStr )
            # DM status
            dmStatusStr = getRemoteMepStatusStr( remoteMep.dmStatus )
            print( "Delay measurement:", dmStatusStr )
            print( "" ) # new line at the end of each remote MEP
      else:
         # non-detailed remote MEP output
         headings = ( "Remote MEP ID", "Status" )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )
         formatRight = TableOutput.Format( justify="right" )

         table = TableOutput.createTable( headings )
         table.formatColumns( formatRight, formatLeft )
         for remoteMepId in sorted( localMep.remoteMeps ):
            table.newRow( remoteMepId, localMep.remoteMeps[ remoteMepId ].status )
            TacSigint.check()
         print( table.output() )

   def renderMa( self, domainName, md, maName, ma ):
      if renderNoMep( domainName, md, maName, ma ):
         return
      for mepId in sorted( ma.localMeps ):
         self.renderMep( domainName, md, maName, ma, mepId, ma.localMeps[ mepId ] )

   def renderMd( self, domainName, md ):
      if renderNoMa( domainName, md ):
         return
      for maName in sorted( md.associations ):
         self.renderMa( domainName, md, maName, md.associations[ maName ] )

   def render( self ):
      for domainName in sorted( self.domains ):
         self.renderMd( domainName, self.domains[ domainName ] )

class LocAndTimeChangeModel( Model ):
   loc = Bool( help="End point connectivity state" )
   lastDetectedTime = Float( help="UTC time of last LOC state detected" )
   lastClearedTime = Float( help="UTC time of last LOC state cleared" )

class RdiAndTimeChangeModel( Model ):
   rdi = Bool( help="End point remote defect indication" )
   lastDetectedTime = Float( help="UTC time of last RDI state detected" )
   lastClearedTime = Float( help="UTC time of last RDI state cleared" )

class RemoteMepCcModel( Model ):
   locState = Submodel( valueType=LocAndTimeChangeModel,
                  help="End point connectivity state and its last change timestamp" )
   mismatchCcmTxInterval = Float(
      help="Mismatching Continuity check transmission interval from remote MEP",
      optional=True )
   ccmPduReceived = Int( help="Number of CCM PDUs received", optional=True )

class CcmDefectModel( Model ):
   level = Int( help="MD level" )
   maName = Str( help="Maintenance association" )
   mdName = Str( help="Maintenance domain" )
   mdNameFormat = Enum( values=tacMdNameFormat.attributes, help="MD name format" )
   maNameFormat = Enum( values=tacMaNameFormat.attributes, help="MA name format" )
   remoteMepId = Int( help="Remote MEP ID" )
   cfmDefectPkts = Int( help="Number of defective CFM packets received on " +
                             "the interface" )
   defectId = Enum( values=DefectCodeEnum.attributes, help="Defect ID" )
   firstSeenTime = Float( help="UTC time of first time the defect was detected" )
   lastSeenTime = Float( help="UTC time of last time the defect was detected" )

class LocalMepCcModel( Model ):
   intf = Interface( help="Local maintenance end point interface" )
   remoteMeps = Dict( keyType=int, valueType=RemoteMepCcModel,
                      help="Remote MEPs, keyed by remote MEP ID" )
   ccmDefects = List( valueType=CcmDefectModel,
                      help="CCM defects",
                      optional=True )
   rdiTxCondition = Submodel( valueType=RdiAndTimeChangeModel,
      help="Local MEP Tx RDI Bit status and its last change timestamp" )
   ccmPduSent = Int( help="Number of CCM PDUs transmitted", optional=True )

class MaCcModel( Model ):
   direction = Enum( help="MEP direction",
                     values=list( mepDirectionShowStr.values() ) )
   ccmTxInterval = Float( help="Continuity check transmission interval in seconds" )
   localMeps = Dict( keyType=int, valueType=LocalMepCcModel,
                     help="Local MEPs, keyed by local MEP ID" )

class MdCcModel( Model ):
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaCcModel,
                        help="A mapping between MA name and MA continuity check" )

class EndPointCcModel( Model ):
   domains = Dict( keyType=str, valueType=MdCcModel,
                   help="A mapping between MD name and MD continuity check" )

   def renderMepDefects( self, localMep ):
      print( "Unknown RMEPs" )
      print( "Format" )
      print( "N = No MD name present, D = Domain name based string, " +
             "M = MAC address + 2-octet integer, C = Character string, " +
             "P = Primary VID, I = Short integer, V = RFC2685 VPN ID " )
      headings = ( "MD Level", "MD Name (format)", "MA Name (format)",
                   "Remote MEP ID", "Received", "Error Status",
                   "First CCM Received", "Last CCM Received" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )

      table = TableOutput.createTable( headings )
      table.formatColumns( formatRight, formatLeft, formatLeft, formatRight,
                           formatRight, formatLeft, formatRight, formatRight )

      for ccmDefect in sorted( localMep.ccmDefects,
                               key=lambda x: ( x.level,
                                               x.mdName,
                                               x.maName,
                                               x.remoteMepId,
                                               tacCfmDefectCode( x.defectId ) ) ):
         mdLevel = ccmDefect.level
         mdName = ( ccmDefect.mdName, ccmDefect.mdNameFormat )
         maName = ( ccmDefect.maName, ccmDefect.maNameFormat )
         rMepId = ccmDefect.remoteMepId
         count = ccmDefect.cfmDefectPkts
         errorStatus = ""
         if ccmDefect.defectId == DefectCodeEnum.defectCodeMaidMismatch:
            errorStatus = "MAID mismatch"
         elif ccmDefect.defectId == DefectCodeEnum.defectCodeMdlMismatch:
            errorStatus = "MD Level mismatch"
         elif ccmDefect.defectId == DefectCodeEnum.defectCodeUnconfiguredMep:
            errorStatus = "Unconfigured RMEP"
         elif ccmDefect.defectId == DefectCodeEnum.defectCodeDuplicateMep:
            errorStatus = "Duplicate RMEP"
         else:
            # We need to render only the above defects, rest need not be rendered.
            continue

         firstCcmTimeStr = timestampToStr( ccmDefect.firstSeenTime,
                                           now=Tac.utcNow() )
         lastCcmTimeStr = timestampToStr( ccmDefect.lastSeenTime,
                                          now=Tac.utcNow() )
         mdNameFormatted = mdName[ 0 ] \
         + " (" + mdFormatLegend[ mdName[ 1 ] ] + ") "
         maNameFormatted = maName[ 0 ] \
         + " (" + maFormatLegend[ maName[ 1 ] ] + ") "
         table.newRow( mdLevel, mdNameFormatted, maNameFormatted, rMepId, count,
                       errorStatus, firstCcmTimeStr, lastCcmTimeStr )
         TacSigint.check()
      print( table.output() )

   def renderMep( self, domainName, md, maName, ma, localMepId, localMep ):
      renderDomainStr( domainName, md )
      renderMaStr( maName, ma )
      # The presence of ccmPduSent in the localMep model can be used to implicitly
      # indicate that SW CFM is supported and that the tx/rx stats will be displayed.
      if toggleCfmSwUpMepCcmEnabled() and localMep.ccmPduSent is not None:
         print( "Maintenance end point ID: %d, Interface: %s, CCM PDUs sent: %d"
                % ( localMepId, localMep.intf.stringValue, localMep.ccmPduSent ) )
      else:
         print( "Maintenance end point ID: %d, Interface: %s"
             % ( localMepId, localMep.intf.stringValue ) )

      # Add RDI Tx State for LocalMep
      rdiTxClearedTimeStr = timestampToStr(
                                       localMep.rdiTxCondition.lastClearedTime,
                                       now=Tac.utcNow() )
      rdiTxSetTimeStr = timestampToStr(
                                       localMep.rdiTxCondition.lastDetectedTime,
                                       now=Tac.utcNow() )
      rdiTxConditionStr = "TX RDI state: "
      if localMep.rdiTxCondition.rdi:
         rdiTxConditionStr += "true, Last TX RDI set: " + rdiTxSetTimeStr
      else:
         rdiTxConditionStr += "false, Last TX RDI cleared: " + rdiTxClearedTimeStr
      print( rdiTxConditionStr )

      if not localMep.remoteMeps:
         print( "No remote maintenance end points" )
         print( "" )
      else:
         headings = ( "Remote MEP ID", "Connectivity", "Last LOC Detected",
                      "Last LOC Cleared" )
         if toggleCfmSwUpMepCcmEnabled() and localMep.ccmPduSent is not None:
            # Since ccmPduSent is present in the localMep model, the ccmPduReceived
            # attribute must also be present in the remoteMep model
            headings += ( "Rx Count", )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )
         formatRight = TableOutput.Format( justify="right" )

         table = TableOutput.createTable( headings )
         table.formatColumns( formatRight, formatLeft, formatRight, formatRight )
         for remoteMepId in sorted( localMep.remoteMeps ):
            ccEntry = localMep.remoteMeps[ remoteMepId ]
            connectivity = 'reachable'
            if ccEntry.locState.loc:
               connectivity = 'unreachable'
               if ccEntry.mismatchCcmTxInterval:
                  ccmTxIntervalWithUnit = CcmTxIntervalWithUnit( 0, 'seconds' )
                  if ccEntry.mismatchCcmTxInterval in ccmTxIntervalToCliToken:
                     ccmTxIntervalWithUnit = \
                           ccmTxIntervalToCliToken[ ccEntry.mismatchCcmTxInterval ]
                     connectivity = 'unreachable (received CCM TX interval: %s %s)' \
                                     % ( ccmTxIntervalWithUnit.ccmTxInterval,
                                         ccmTxIntervalWithUnit.unit )

            locDetectedTimeStr = timestampToStr( ccEntry.locState.lastDetectedTime,
                                                 now=Tac.utcNow() )
            locClearedTimeStr = timestampToStr( ccEntry.locState.lastClearedTime,
                                                 now=Tac.utcNow() )
            if toggleCfmSwUpMepCcmEnabled() and localMep.ccmPduSent is not None:
               # If ccmPduSent is present but the ccmPduReceived is not, there must
               # have been an issue with the creation of the count tables.
               assert not ccEntry.ccmPduReceived is None
               table.newRow( remoteMepId, connectivity, locDetectedTimeStr,
                             locClearedTimeStr, ccEntry.ccmPduReceived )
            else:
               table.newRow( remoteMepId, connectivity, locDetectedTimeStr,
                             locClearedTimeStr )
            TacSigint.check()
         print( table.output() )

      if localMep.ccmDefects:
         self.renderMepDefects( localMep )

   def renderMa( self, domainName, md, maName, ma ):
      if renderNoMep( domainName, md, maName, ma ):
         return
      for mepId in sorted( ma.localMeps ):
         self.renderMep( domainName, md, maName, ma, mepId, ma.localMeps[ mepId ] )

   def renderMd( self, domainName, md ):
      if renderNoMa( domainName, md ):
         return
      for maName in sorted( md.associations ):
         self.renderMa( domainName, md, maName, md.associations[ maName ] )

   def render( self ):
      for domainName in sorted( self.domains ):
         self.renderMd( domainName, self.domains[ domainName ] )

class RemoteMepRdiModel( Model ):
   rdiState = Submodel( valueType=RdiAndTimeChangeModel,
                 help="End point remote defect state and its last change timestamp" )

class LocalMepRdiModel( Model ):
   intf = Interface( help="Local maintenance end point interface" )
   rdiCondition = Submodel( valueType=RdiAndTimeChangeModel,
      help="Local MEP remote defect condition and its last change timestamp" )
   remoteMeps = Dict( keyType=int, valueType=RemoteMepRdiModel,
                      help="Remote MEPs, keyed by remote MEP ID" )
   rdiTxCondition = Submodel( valueType=RdiAndTimeChangeModel,
      help="Local MEP Tx RDI bit status and its last change timestamp" )

class MaRdiModel( Model ):
   direction = Enum( help="MEP direction",
                     values=list( mepDirectionShowStr.values() ) )
   localMeps = Dict( keyType=int, valueType=LocalMepRdiModel,
                     help="Local MEPs, keyed by local MEP ID" )

class MdRdiModel( Model ):
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaRdiModel,
                        help="A mapping between MA name and MA RDI" )

class EndPointRdiModel( Model ):
   domains = Dict( keyType=str, valueType=MdRdiModel,
                   help="A mapping between MD name and MD RDI" )

   def renderMep( self, domainName, md, maName, ma, localMepId, localMep ):
      renderDomainStr( domainName, md )
      renderMaStr( maName, ma, withCcmTx=False )
      print( "Maintenance end point ID: %d, Interface: %s"
             % ( localMepId, localMep.intf.stringValue ) )
      rdiDetectedTimeStr = timestampToStr( localMep.rdiCondition.lastDetectedTime,
                                           now=Tac.utcNow() )
      rdiClearedTimeStr = timestampToStr( localMep.rdiCondition.lastClearedTime,
                                           now=Tac.utcNow() )
      rdiConditionStr = "RDI condition: "
      if localMep.rdiCondition.rdi:
         rdiConditionStr += "detected"
      elif not localMep.rdiCondition.lastClearedTime and \
           not localMep.rdiCondition.lastDetectedTime:
         rdiConditionStr += "undetected"
      else:
         rdiConditionStr += "cleared"

      # In either of the cases: "detected" and "cleared", if we have
      # the information of lastDetectedTime and lastClearedTime, we should
      # display it.
      if localMep.rdiCondition.lastDetectedTime:
         rdiConditionStr += ", last detected " + rdiDetectedTimeStr
      if localMep.rdiCondition.lastClearedTime:
         rdiConditionStr += ", last cleared " + rdiClearedTimeStr
      print( rdiConditionStr )

      # Add RDI Tx State for LocalMep
      rdiTxClearedTimeStr = timestampToStr(
                                       localMep.rdiTxCondition.lastClearedTime,
                                       now=Tac.utcNow() )
      rdiTxSetTimeStr = timestampToStr(
                                       localMep.rdiTxCondition.lastDetectedTime,
                                       now=Tac.utcNow() )
      rdiTxConditionStr = "TX RDI state: "
      if localMep.rdiTxCondition.rdi:
         rdiTxConditionStr += "true, Last TX RDI set: " + rdiTxSetTimeStr
      else:
         rdiTxConditionStr += "false, Last TX RDI cleared: " + rdiTxClearedTimeStr
      print( rdiTxConditionStr )

      if not localMep.remoteMeps:
         print( "No remote maintenance end points" )
         print( "" )
         return
      headings = ( "Remote MEP ID", "RDI State", "RDI Detected", "RDI Cleared" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )

      table = TableOutput.createTable( headings )
      table.formatColumns( formatRight, formatLeft, formatRight, formatRight )
      for remoteMepId in sorted( localMep.remoteMeps ):
         rdiEntry = localMep.remoteMeps[ remoteMepId ]
         rdiStateStr = 'undetected'
         if rdiEntry.rdiState.lastClearedTime:
            rdiStateStr = 'cleared'
         if rdiEntry.rdiState.rdi:
            rdiStateStr = 'detected'
         rdiDetectedTimeStr = timestampToStr( rdiEntry.rdiState.lastDetectedTime,
                                              now=Tac.utcNow() )
         rdiClearedTimeStr = timestampToStr( rdiEntry.rdiState.lastClearedTime,
                                              now=Tac.utcNow() )
         table.newRow( remoteMepId, rdiStateStr, rdiDetectedTimeStr,
                       rdiClearedTimeStr )
         TacSigint.check()
      print( table.output() )

   def renderMa( self, domainName, md, maName, ma ):
      if renderNoMep( domainName, md, maName, ma, withCcmTx=False ):
         return
      for mepId in sorted( ma.localMeps ):
         self.renderMep( domainName, md, maName, ma, mepId, ma.localMeps[ mepId ] )

   def renderMd( self, domainName, md ):
      if renderNoMa( domainName, md ):
         return
      for maName in sorted( md.associations ):
         self.renderMa( domainName, md, maName, md.associations[ maName ] )

   def render( self ):
      for domainName in sorted( self.domains ):
         self.renderMd( domainName, self.domains[ domainName ] )

class MiStatsModel( Model ):
   __public__ = toggleCfmPmMiEnabled()
   startTime = Float( help="UTC start time of measurement" )
   endTime = Float( help="UTC end time of measurement", optional=True )
   measurementInterval = Int( help="Measurement interval in minutes", optional=True )
   suspect = Enum( values=MiSuspectEnum.attributes,
                   help="Suspect status of measurement", optional=True )

class DmMiStatsModel( MiStatsModel ):
   __public__ = toggleCfmPmMiEnabled()
   numSamples = Int( help="Number of delay measurement frames in measurement "
                          "interval" )
   minDelay = Float( help="Minimum measured delay in measurement interval" )
   maxDelay = Float( help="Maximum measured delay in measurement interval" )
   avgDelay = Float( help="Average measured delay in measurement interval" )
   minVariation = Float( help="Minimum measured variation in measurement interval" )
   maxVariation = Float( help="Maximum measured variation in measurement interval" )
   avgVariation = Float( help="Average measured variation in measurement interval" )

class DmStatsModel( Model ):
   delay = Float( help="Measured delay in this delay measurement frame" )
   updateTime = Float( help="UTC time when this DM stat entry was updated" )

class RemoteMepDmModel( Model ):
   enabled = Bool( help="Delay measurement is enabled" )
   reason = Enum( values=pmRmepStatusEnum,
                  help="Reason for delay measurement failure", optional=True )
   numSamples = Int( help="Number of delay measurement frames", optional=True )
   startTime = Float( help="UTC start time of delay measurement", optional=True )
   avgDelay = Float( help="Average measured delay", optional=True )
   avgJitter = Float( help="Average measured jitter", optional=True )
   bestDelay = Float( help="Best measured delay", optional=True )
   bestDelayTime = Float( help="UTC time when best delay was observed",
                          optional=True )
   worstDelay = Float( help="Worst measured delay", optional=True )
   worstDelayTime = Float( help="UTC time when worst delay was observed",
                           optional=True )
   delayHistory = List( valueType=DmStatsModel,
                        help="Measured delay in the recent delay measurement frames",
                        optional=True )
   if toggleCfmPmMiEnabled():
      miEnabled = Bool( help="Measurement interval is configured" )
      currentMiStats = Submodel( valueType=DmMiStatsModel,
                                 help="Current measurement interval statistics for "
                                      "delay measurement", optional=True )
      historicMiStats = List( valueType=DmMiStatsModel,
                          help="Historic measurement interval statistics for delay "
                               "measurement", optional=True )

class LocalMepDmModel( Model ):
   intf = Interface( help="Local maintenance end point interface" )
   remoteMeps = Dict( keyType=int, valueType=RemoteMepDmModel,
                      help="Remote MEPs, keyed by remote MEP ID" )

class MaDmModel( Model ):
   direction = Enum( help="MEP direction",
                     values=list( mepDirectionShowStr.values() ) )
   operMode = Enum( help="Delay measurement type",
                    values=list( pmOperModeKwStr.values() ) )
   txInterval = Float(
               help="Delay measurement frame transmission interval in milliseconds" )
   localMeps = Dict( keyType=int, valueType=LocalMepDmModel,
                     help="Local MEPs, keyed by local MEP ID" )

class MdDmModel( Model ):
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaDmModel,
                        help="A mapping between MA name and MA" )

class EndPointDmModel( Model ):
   domains = Dict( keyType=str, valueType=MdDmModel,
                    help="A mapping between MD name and MD" )

   def renderRemoteMep( self, remoteMepId, remoteMep ):
      startTimeStr = timestampToStr( remoteMep.startTime, now=Tac.utcNow() )
      bestDelayTimeStr = timestampToStr( remoteMep.bestDelayTime, now=Tac.utcNow() )
      worstDelayTimeStr = timestampToStr( remoteMep.worstDelayTime,
                                          now=Tac.utcNow() )
      renderRemoteMepStatusStr( remoteMepId, remoteMep )
      dmTablePresent = False
      if remoteMep.enabled:
         print( "Start time: %s" % startTimeStr )
         print( "Number of samples: %d" % remoteMep.numSamples )
         if remoteMep.numSamples:
            print( "Average two-way delay: %.3f usec" % remoteMep.avgDelay )
            print( "Average two-way delay variation: %.3f usec"
                   % remoteMep.avgJitter )
            print( "Best case two-way delay: %.3f usec at %s"
                   % ( remoteMep.bestDelay, bestDelayTimeStr ) )
            print( "Worst case two-way delay: %.3f usec at %s"
                   % ( remoteMep.worstDelay, worstDelayTimeStr ) )

            if remoteMep.delayHistory:
               headings = ( "Timestamp", "Two-way Delay (usec)" )
               formatRight = TableOutput.Format( justify="right" )

               table = TableOutput.createTable( headings )
               table.formatColumns( formatRight, formatRight )
               for delayEntry in remoteMep.delayHistory:
                  table.newRow( timestampToStr( delayEntry.updateTime,
                                                relative=False, now=Tac.utcNow() ),
                                                ( "%.3f" % delayEntry.delay ) )
                  TacSigint.check()
               print( table.output() )
               dmTablePresent = True
      if toggleCfmPmMiEnabled():
         miEnabled = remoteMep.miEnabled
         currentMiStats = remoteMep.currentMiStats
         if not dmTablePresent and ( currentMiStats or miEnabled ):
            print( "" )
         if currentMiStats:
            print( "Current measurement interval statistics" )
            print( "Start time: %s, Elapsed time: %d seconds" %
                   ( utcTimestampToStr( currentMiStats.startTime ),
                     Tac.utcNow() - currentMiStats.startTime ) )
            miStr = '%d minutes' % currentMiStats.measurementInterval if \
                    currentMiStats.measurementInterval else 'n/a'
            print( "Measurement interval: %s, Number of samples: %d" %
                   ( miStr, currentMiStats.numSamples ) )
            print( "Two-way delay (usec) min/max/avg: %.3f/%.3f/%.3f" %
                   ( currentMiStats.minDelay,
                     currentMiStats.maxDelay,
                     currentMiStats.avgDelay ) )
            print( "Two-way delay variation (usec) min/max/avg: %.3f/%.3f/%.3f"
                   % ( currentMiStats.minVariation,
                       currentMiStats.maxVariation,
                       currentMiStats.avgVariation ) )
         elif miEnabled:
            print( "No current measurement interval statistics" )
         historicMiStats = remoteMep.historicMiStats
         # Following is needed only for the detailed version of the command, in which
         # case historicMiStats will be not None
         if historicMiStats is not None:
            if len( historicMiStats ) or miEnabled:
               print( "" )
            if len( historicMiStats ):
               print( "Historic measurement interval statistics" )
               headings = ( "Start time", "End time", "Samples", "Suspect",
                            "Two-way delay (usec)\nmin/max/avg",
                            "Two-way delay variation (usec)\nmin/max/avg" )
               formatLeft = TableOutput.Format( justify="left" )
               formatLeft.noPadLeftIs( True )
               formatRight = TableOutput.Format( justify="right" )
               miHistoryTable = TableOutput.createTable( headings )
               miHistoryTable.formatColumns( formatLeft, formatLeft, formatRight,
                                             formatLeft, formatLeft, formatLeft )
               for miHistory in sorted( historicMiStats, reverse=True,
                                        key=lambda x: ( x.startTime ) ):
                  startTime = utcTimestampToStr( miHistory.startTime )
                  endTime = utcTimestampToStr( miHistory.endTime )
                  samples = miHistory.numSamples
                  suspect = 'n/a'
                  if miHistory.suspect == MiSuspectEnum.loc:
                     suspect = 'LOC'
                  elif miHistory.suspect == MiSuspectEnum.incomplete:
                     suspect = 'incomplete'
                  variationStr = "%.3f/%.3f/%.3f" % ( miHistory.minVariation,
                                                      miHistory.maxVariation,
                                                      miHistory.avgVariation )
                  delayStr = "%.3f/%.3f/%.3f" % ( miHistory.minDelay,
                                                  miHistory.maxDelay,
                                                  miHistory.avgDelay )
                  miHistoryTable.newRow( startTime, endTime, samples, suspect,
                                         delayStr, variationStr )
                  TacSigint.check()
               print( miHistoryTable.output() )
            elif miEnabled:
               print( "No historic measurement interval statistics" )

   def renderMep( self, domainName, md, maName, ma, localMepId, localMep ):
      renderDomainStr( domainName, md )
      renderMaStr( maName, ma, withCcmTx=False, withPm=True )
      print( "Maintenance end point ID: %d, Interface: %s"
             % ( localMepId, localMep.intf.stringValue ) )
      if not localMep.remoteMeps:
         print( "No remote maintenance end points" )
         print( "" )
         return
      for remoteMepId in sorted( localMep.remoteMeps ):
         remoteMep = localMep.remoteMeps[ remoteMepId ]
         self.renderRemoteMep( remoteMepId, remoteMep )
         print( "" )
         TacSigint.check()

   def renderMa( self, domainName, md, maName, ma ):
      if renderNoMep( domainName, md, maName, ma, withCcmTx=False, withPm=True ):
         return
      for mepId in sorted( ma.localMeps ):
         self.renderMep( domainName, md, maName, ma, mepId, ma.localMeps[ mepId ] )

   def renderMd( self, domainName, md ):
      if renderNoMa( domainName, md ):
         return
      for maName in sorted( md.associations ):
         self.renderMa( domainName, md, maName, md.associations[ maName ] )

   def render( self ):
      for domainName in sorted( self.domains ):
         self.renderMd( domainName, self.domains[ domainName ] )

class LmStatsModel( Model ):
   localTx = Int( help="Packets sent from local MEP" )
   localRx = Int( help="Packets received on local MEP" )
   remoteRx = Int( help="Packets received on remote MEP" )
   remoteTx = Int( help="Packets sent from remote MEP" )
   nearEndLoss = Int( help="Packets lost on local MEP" )
   farEndLoss = Int( help="Packets lost on remote MEP" )
   updateTime = Float( help="UTC time when this LM stat entry was updated" )

class RemoteMepLmModel( Model ):
   enabled = Bool( help="Loss measurement is enabled" )
   reason = Enum( values=pmRmepStatusEnum,
                  help="Reason for loss measurement failure", optional=True )
   numSamples = Int( help="Number of loss measurement frames", optional=True )
   startTime = Float( help="UTC start time of loss measurement", optional=True )
   avgFarEndFrameLoss = Int( help="Average far-end frame loss", optional=True )
   avgNearEndFrameLoss = Int( help="Average near-end frame loss", optional=True )
   lmHistory = List( valueType=LmStatsModel,
                     help="Measurement stats from recent loss measurement frames",
                     optional=True )

class LocalMepLmModel( Model ):
   intf = Interface( help="Local maintenance end point interface" )
   remoteMeps = Dict( keyType=int, valueType=RemoteMepLmModel,
                      help="Remote MEPs, keyed by remote MEP ID" )

class MaLmModel( Model ):
   direction = Enum( help="MEP direction",
                     values=list( mepDirectionShowStr.values() ) )
   operMode = Enum( help="Loss measurement type",
                    values=list( pmOperModeKwStr.values() ) )
   txInterval = Float(
               help="Loss measurement frame transmission interval in milliseconds" )
   localMeps = Dict( keyType=int, valueType=LocalMepLmModel,
                     help="Local MEPs, keyed by local MEP ID" )

class MdLmModel( Model ):
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaLmModel,
                        help="A mapping between MA name and MA" )

class EndPointLmModel( Model ):
   domains = Dict( keyType=str, valueType=MdLmModel,
                    help="A mapping between MD name and MD" )

   def renderRemoteMep( self, remoteMepId, remoteMep ):
      startTimeStr = timestampToStr( remoteMep.startTime, now=Tac.utcNow() )
      renderRemoteMepStatusStr( remoteMepId, remoteMep )
      if remoteMep.enabled:
         print( "Start time: %s" % startTimeStr )
         print( "Number of samples: %d" % remoteMep.numSamples )
         if remoteMep.numSamples:
            print( "Average far-end frame loss: %d" % remoteMep.avgFarEndFrameLoss )
            print( "Average near-end frame loss: %d" %
                                                   remoteMep.avgNearEndFrameLoss )
            if remoteMep.lmHistory:
               headings = ( "Timestamp", "Near-end Frame Loss", "Near-end Rx",
                            "Far-end Tx", "Far-end Frame Loss", "Near-end Tx",
                            "Far-end Rx" )
               formatRight = TableOutput.Format( justify="right" )

               table = TableOutput.createTable( headings )
               table.formatColumns( formatRight, formatRight, formatRight,
                              formatRight, formatRight, formatRight, formatRight )
               for lmEntry in remoteMep.lmHistory:
                  table.newRow( timestampToStr( lmEntry.updateTime, relative=False,
                                                now=Tac.utcNow() ),
                                lmEntry.nearEndLoss, lmEntry.localRx,
                                lmEntry.remoteTx, lmEntry.farEndLoss,
                                lmEntry.localTx, lmEntry.remoteRx )
                  TacSigint.check()
               print( table.output() )

   def renderMep( self, domainName, md, maName, ma, localMepId, localMep ):
      renderDomainStr( domainName, md )
      renderMaStr( maName, ma, withCcmTx=False, withPm=True )
      print( "Maintenance end point ID: %d, Interface: %s"
             % ( localMepId, localMep.intf.stringValue ) )
      if not localMep.remoteMeps:
         print( "No remote maintenance end points" )
         print( "" )
         return
      for remoteMepId in sorted( localMep.remoteMeps ):
         remoteMep = localMep.remoteMeps[ remoteMepId ]
         self.renderRemoteMep( remoteMepId, remoteMep )
         print( "" )
         TacSigint.check()

   def renderMa( self, domainName, md, maName, ma ):
      if renderNoMep( domainName, md, maName, ma, withCcmTx=False, withPm=True ):
         return
      for mepId in sorted( ma.localMeps ):
         self.renderMep( domainName, md, maName, ma, mepId, ma.localMeps[ mepId ] )

   def renderMd( self, domainName, md ):
      if renderNoMa( domainName, md ):
         return
      for maName in sorted( md.associations ):
         self.renderMa( domainName, md, maName, md.associations[ maName ] )

   def render( self ):
      for domainName in sorted( self.domains ):
         self.renderMd( domainName, self.domains[ domainName ] )

class SlmMiStatsModel( MiStatsModel ):
   __public__ = toggleCfmPmMiEnabled()
   numMeasurements = \
      Int( help="Number of measurements taken in measurement interval" )
   minFarEndFrameLossPercentage = Float( help="Minimum far-end synthetic frame loss "
                                              "percentage" )
   maxFarEndFrameLossPercentage = Float( help="Maximum far-end synthetic frame loss "
                                              "percentage" )
   avgFarEndFrameLossPercentage = Float( help="Average far-end synthetic frame loss "
                                              "percentage" )
   minNearEndFrameLossPercentage = Float( help="Minimum near-end synthetic frame "
                                               "loss percentage" )
   maxNearEndFrameLossPercentage = Float( help="Maximum near-end synthetic frame "
                                               "loss percentage" )
   avgNearEndFrameLossPercentage = Float( help="Average near-end synthetic frame "
                                               "loss percentage" )

class SlmStatsModel( Model ):
   localTx = Int( help="Packets sent from local MEP" )
   localRx = Int( help="Packets received on local MEP" )
   remoteRx = Int( help="Packets received on remote MEP" )
   updateTime = Float( help="UTC time of the measurement period" )

class RemoteMepSlmModel( Model ):
   enabled = Bool( help="Synthetic loss measurement is enabled" )
   reason = Enum( values=pmRmepStatusEnum,
                  help="Reason for synthetic loss measurement failure",
                  optional=True )
   numMeasurements = Int( help="Number of measurements taken", optional=True )
   measurementPeriod = Float(
                           help="Synthetic loss measurement period in milliseconds",
                           optional=True )
   startTime = Float( help="UTC start time of synthetic loss measurement",
                      optional=True )
   avgFarEndFrameLoss = Float( help="Average far-end synthetic frame loss",
                               optional=True )
   avgNearEndFrameLoss = Float( help="Average near-end synthetic frame loss",
                                optional=True )
   slmHistory = List( valueType=SlmStatsModel,
            help="Measurement stats from recent synthetic loss measurement frames",
            optional=True )
   if toggleCfmPmMiEnabled():
      miEnabled = Bool( help="Measurement interval is configured" )
      currentMiStats = Submodel( valueType=SlmMiStatsModel,
                                 help="Current measurement interval statistics for "
                                      "synthetic loss measurement", optional=True )
      historicMiStats = List( valueType=SlmMiStatsModel,
                              help="Historic measurement interval statistics for "
                                   "synthetic loss measurement", optional=True )

class LocalMepSlmModel( Model ):
   mepId = Int( help="Local MEP ID" )
   intf = Interface( help="Local MEP interface" )
   cos = Int( help="COS value used for Synthetic Loss measurement frames" )
   remoteMeps = Dict( keyType=int, valueType=RemoteMepSlmModel,
                      help="Remote MEPs, keyed by remote MEP ID" )

class MaSlmModel( Model ):
   direction = Enum( help="MEP direction",
                     values=list( mepDirectionShowStr.values() ) )
   operMode = Enum( help="Synthetic Loss measurement type",
                    values=list( pmOperModeKwStr.values() ) )
   txInterval = Float(
      help="Synthetic Loss measurement frame transmission interval in milliseconds" )
   localMeps = List( valueType=LocalMepSlmModel,
                     help="List of Local MEPs and their COS value" )

class MdSlmModel( Model ):
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaSlmModel,
                        help="A mapping between MA name and MA" )

class EndPointSlmModel( Model ):
   domains = Dict( keyType=str, valueType=MdSlmModel,
                    help="A mapping between MD name and MD" )

   def renderRemoteMep( self, remoteMepId, remoteMep ):
      renderRemoteMepStatusStr( remoteMepId, remoteMep )
      startTimeStr = timestampToStr( remoteMep.startTime, now=Tac.utcNow() )
      slmTablePresent = False
      if remoteMep.enabled:
         print( "Start time: %s" % startTimeStr )
         print( "Number of measurements: %d, Measurement period: %d milliseconds" %
                ( remoteMep.numMeasurements, remoteMep.measurementPeriod ) )
         if remoteMep.numMeasurements:
            print( "Average far-end synthetic frame loss: %.3f" %
                                                      remoteMep.avgFarEndFrameLoss )
            print( "Average near-end synthetic frame loss: %.3f" %
                                                      remoteMep.avgNearEndFrameLoss )
            if remoteMep.slmHistory:
               headings = ( "Timestamp", "Near-end Rx", "Near-end Tx", "Far-end Rx" )
               formatRight = TableOutput.Format( justify="right" )

               table = TableOutput.createTable( headings )
               table.formatColumns( formatRight, formatRight, formatRight,
                                    formatRight )
               for slmEntry in remoteMep.slmHistory:
                  table.newRow( timestampToStr( slmEntry.updateTime,
                                                relative=False, now=Tac.utcNow() ),
                                slmEntry.localRx, slmEntry.localTx,
                                slmEntry.remoteRx )
                  TacSigint.check()
               print( table.output() )
               slmTablePresent = True
      if toggleCfmPmMiEnabled():
         miEnabled = remoteMep.miEnabled
         currentMiStats = remoteMep.currentMiStats
         if not slmTablePresent and ( currentMiStats or miEnabled ):
            print( "" )
         if currentMiStats:
            print( "Current measurement interval statistics" )
            print( "Start time: %s, Elapsed time: %d seconds" %
                   ( utcTimestampToStr( currentMiStats.startTime ),
                     Tac.utcNow() - currentMiStats.startTime ) )
            miStr = '%d minutes' % currentMiStats.measurementInterval if \
                    currentMiStats.measurementInterval else 'n/a'
            print( "Measurement interval: %s, Number of measurements: %d" %
                   ( miStr, currentMiStats.numMeasurements ) )
            print( "Far-end frame loss ratio (percent) min/max/avg: %.3f/%.3f/%.3f" %
                   ( currentMiStats.minFarEndFrameLossPercentage,
                     currentMiStats.maxFarEndFrameLossPercentage,
                     currentMiStats.avgFarEndFrameLossPercentage ) )
            print( "Near-end frame loss ratio (percent) min/max/avg: %.3f/%.3f/%.3f"
                   % ( currentMiStats.minNearEndFrameLossPercentage,
                       currentMiStats.maxNearEndFrameLossPercentage,
                       currentMiStats.avgNearEndFrameLossPercentage ) )
         elif miEnabled:
            print( "No current measurement interval statistics" )
         historicMiStats = remoteMep.historicMiStats
         # Following is needed only for the detailed version of the command, in which
         # case historicMiStats will be not None
         if historicMiStats is not None:
            if len( historicMiStats ) or miEnabled:
               print( "" )
            if len( historicMiStats ):
               print( "Historic measurement interval statistics" )
               headings = ( "Start time", "End time", "Measurements", "Suspect",
                            "Far-end FLR (percent)\nmin/max/avg",
                            "Near-end FLR (percent)\nmin/max/avg" )
               formatLeft = TableOutput.Format( justify="left" )
               formatLeft.noPadLeftIs( True )
               formatRight = TableOutput.Format( justify="right" )
               miHistoryTable = TableOutput.createTable( headings )
               miHistoryTable.formatColumns( formatLeft, formatLeft, formatRight,
                                             formatLeft, formatLeft, formatLeft )
               for miHistory in sorted( historicMiStats, reverse=True,
                                        key=lambda x: ( x.startTime ) ):
                  startTime = utcTimestampToStr( miHistory.startTime )
                  endTime = utcTimestampToStr( miHistory.endTime )
                  suspect = 'n/a'
                  if miHistory.suspect == MiSuspectEnum.loc:
                     suspect = 'LOC'
                  elif miHistory.suspect == MiSuspectEnum.incomplete:
                     suspect = 'incomplete'
                  farEndFlrStr = "%.3f/%.3f/%.3f" % \
                     ( miHistory.minFarEndFrameLossPercentage,
                       miHistory.maxFarEndFrameLossPercentage,
                       miHistory.avgFarEndFrameLossPercentage )
                  nearEndFlrStr = "%.3f/%.3f/%.3f" % \
                     ( miHistory.minNearEndFrameLossPercentage,
                       miHistory.maxNearEndFrameLossPercentage,
                       miHistory.avgNearEndFrameLossPercentage )
                  miHistoryTable.newRow( startTime, endTime,
                                         miHistory.numMeasurements, suspect,
                                         farEndFlrStr, nearEndFlrStr )
                  TacSigint.check()
               print( miHistoryTable.output() )
            elif miEnabled:
               print( "No historic measurement interval statistics" )

   def renderMep( self, domainName, md, maName, ma, localMep ):
      print( "Maintenance end point ID: %d, Interface: %s, COS: %d"
             % ( localMep.mepId, localMep.intf.stringValue, localMep.cos ) )
      if not localMep.remoteMeps:
         print( "No remote maintenance end points" )
         print( "" )
         return
      for remoteMepId in sorted( localMep.remoteMeps ):
         remoteMep = localMep.remoteMeps[ remoteMepId ]
         self.renderRemoteMep( remoteMepId, remoteMep )
         print( "" )
         TacSigint.check()

   def renderMa( self, domainName, md, maName, ma ):
      if renderNoMep( domainName, md, maName, ma, withCcmTx=False, withPm=True ):
         return
      prevMepId = 0
      for localMep in ma.localMeps:
         if localMep.mepId != prevMepId:
            # show the MD and MA header only once for every local Mep and not for
            # the same local Mep repeated with a different COS value (localMep.cos)
            renderDomainStr( domainName, md )
            renderMaStr( maName, ma, withCcmTx=False, withPm=True )
         self.renderMep( domainName, md, maName, ma, localMep )
         prevMepId = localMep.mepId

   def renderMd( self, domainName, md ):
      if renderNoMa( domainName, md ):
         return
      for maName in sorted( md.associations ):
         self.renderMa( domainName, md, maName, md.associations[ maName ] )

   def render( self ):
      for domainName in sorted( self.domains ):
         self.renderMd( domainName, self.domains[ domainName ] )

class AisDefectTimeChangeModel( Model ):
   detected = Bool( help="End point has an AIS defect" )
   lastDetectedTime = Float( help="UTC time of last AIS state detected" )
   lastClearedTime = Float( help="UTC time of last AIS state cleared" )

class RemoteAisRxEntryModel( Model ):
   remoteMac = MacAddress( help="MAC address of remote MEP that detected a fault" )
   txInterval = Int( help="AIS transmission interval in seconds" )
   count = Int( help="Number of AIS PDUs that were received" )
   lastReceivedTime = Float( help="UTC time of last received AIS PDU" )

class LocalMepAisModel( Model ):
   intf = Interface( help="Local maintenance end point interface" )
   aisCondition = Submodel( valueType=AisDefectTimeChangeModel,
      help="Local MEP AIS defect condition and its last change timestamp" )
   clientDomainLevel = Int( help="Client MD level for AIS transmission",
                            optional=True )
   clientDomainLevelError = Enum(
      values=( 'invalid', 'notConfigured' ),
      help="Reason for client domain level configuration error", optional=True )
   aisTxInterval = Int( help="AIS transmission interval in seconds",
                        optional=True )
   aisDefects = Dict( keyType=str, valueType=AisDefectTimeChangeModel,
                      help="AIS defects keyed by their name" )
   # AIS Rx entries sorted based on last received timestamp.
   remoteAisRxEntries = List( valueType=RemoteAisRxEntryModel,
                              help="List of AIS information received from "
                              "remote MEPs in order of last received" )

class MaAisModel( Model ):
   direction = Enum( help="MEP direction",
                     values=list( mepDirectionShowStr.values() ) )
   localMeps = Dict( keyType=int, valueType=LocalMepAisModel,
                     help="Local MEPs, keyed by local MEP ID" )

class MdAisModel( Model ):
   level = Int( help="MD level" )
   associations = Dict( keyType=str, valueType=MaAisModel,
                        help="A mapping between MA name and MA AIS" )

class EndPointAisModel( Model ):
   domains = Dict( keyType=str, valueType=MdAisModel,
                   help="A mapping between MD name and MD AIS" )

   def renderContinuityCheckDefectNote( self ):
      print( "NOTE: Please refer to \"show cfm continuity-check end-point detail\" "
         "for continuity check defects" )

   def renderMep( self, domainName, md, maName, ma, localMepId, localMep ):
      renderDomainStr( domainName, md )
      renderMaStr( maName, ma, withCcmTx=False )
      print( "Maintenance end point ID: %d, Interface: %s"
             % ( localMepId, localMep.intf.stringValue ) )
      aisStateStr = "AIS defect condition: "
      if localMep.aisCondition.detected:
         aisStateStr += "detected"
      elif not localMep.aisCondition.lastClearedTime and \
           not localMep.aisCondition.lastDetectedTime:
         aisStateStr += "undetected"
      else:
         aisStateStr += "cleared"
      aisDetectedTimeStr = timestampToStr( localMep.aisCondition.lastDetectedTime,
                                           now=Tac.utcNow() )
      aisClearedTimeStr = timestampToStr( localMep.aisCondition.lastClearedTime,
                                          now=Tac.utcNow() )
      aisStateStr += ", last entered: " + aisDetectedTimeStr
      aisStateStr += ", last cleared: " + aisClearedTimeStr
      print( aisStateStr )
      if localMep.clientDomainLevel is not None:
         aisTxInfoStr = "AIS defect client domain level: {}".format(
            localMep.clientDomainLevel )
         aisTxInfoStr += ", TX interval: {} second".format(
            localMep.aisTxInterval )
      else:
         clientDomainErrorStr = 'Invalid client domain configuration' \
            if localMep.clientDomainLevelError == 'invalid' else 'Not configured'
         aisTxInfoStr = "AIS defect client domain level: n/a (Reason: {})".format(
            clientDomainErrorStr )
         aisTxInfoStr += ", TX interval: n/a"
      print( aisTxInfoStr )
      headings = ( "Defect Condition", "State", "Defect Detected", "Defect Cleared" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      defectTable = TableOutput.createTable( headings )
      defectTable.formatColumns( formatLeft, formatLeft, formatRight, formatRight )
      for defectCondition in sorted( localMep.aisDefects ):
         defectEntry = localMep.aisDefects[ defectCondition ]
         defectStateStr = 'undetected'
         if defectEntry.detected:
            defectStateStr = 'detected'
         elif defectEntry.lastClearedTime:
            defectStateStr = "cleared"
         defectDetectedTimeStr = timestampToStr( defectEntry.lastDetectedTime,
                                                 now=Tac.utcNow() )
         defectClearedTimeStr = timestampToStr( defectEntry.lastClearedTime,
                                                now=Tac.utcNow() )
         defectTable.newRow( defectCondition, defectStateStr,
                             defectDetectedTimeStr, defectClearedTimeStr )
         TacSigint.check()
      print( defectTable.output() )
      if not localMep.remoteAisRxEntries:
         print( "No AIS received information" )
         return
      print( "AIS received information" )
      headings = ( "Remote MAC Address", "TX Interval", "Count", "Last Received" )
      statsTable = TableOutput.createTable( headings )
      statsTable.formatColumns( formatLeft, formatRight, formatRight, formatRight )
      for aisRxEntry in localMep.remoteAisRxEntries:
         lastReceivedTimeStr = timestampToStr( aisRxEntry.lastReceivedTime,
                                               now=Tac.utcNow() )
         macStr = Ethernet.convertMacAddrCanonicalToDisplay(
            aisRxEntry.remoteMac.stringValue )
         txIntervalStr = "%d second" % aisRxEntry.txInterval
         statsTable.newRow( macStr, txIntervalStr,
                            aisRxEntry.count, lastReceivedTimeStr )
         TacSigint.check()
      print( statsTable.output() )

   def renderMa( self, domainName, md, maName, ma ):
      if renderNoMep( domainName, md, maName, ma, withCcmTx=False ):
         return
      for mepId in sorted( ma.localMeps ):
         self.renderMep( domainName, md, maName, ma, mepId, ma.localMeps[ mepId ] )

   def renderMd( self, domainName, md ):
      if renderNoMa( domainName, md ):
         return
      for maName in sorted( md.associations ):
         self.renderMa( domainName, md, maName, md.associations[ maName ] )

   def render( self ):
      for domainName in sorted( self.domains ):
         self.renderMd( domainName, self.domains[ domainName ] )
      mepRendered = False
      for domainName in self.domains:
         for association in self.domains[ domainName ].associations:
            maAisModel = self.domains[ domainName ].associations[ association ]
            if maAisModel.localMeps:
               mepRendered = True
               break
      if mepRendered:
         self.renderContinuityCheckDefectNote()
