# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
import Tac
import Logging
import Url
from ClassificationLib import ( addOrDeleteRange,
                                fieldSetTypeToStr,
                                numericalRangeToSet,
                                rangeSetToNumericalRange,
                                listUrlLocalSchemes,
                                listUrlNetworkSchemes,
                                getFieldSetConfigForType,
                                getTcpFlagAndMasksEst,
                                getTcpFlagAndMaskInit )
import CliSession
from socket import IPPROTO_TCP
from UrlPlugin.NetworkUrl import NetworkUrl, ScpUrl

appConstants = Tac.Value( 'Classification::Constants' )
ProtocolRange = Tac.Type( 'Classification::ProtocolRange' )
IpGenPrefix = Tac.Type( 'Arnet::IpGenPrefix' )
FragmentType = Tac.Type( 'Classification::FragmentType' )
ContentsSource = Tac.Type( 'Classification::ContentsSource' )
matchOffset = FragmentType.matchOffset
IcmpTypeRange = Tac.Type( 'Classification::IcmpTypeRange' )
UniqueId = Tac.Type( "Ark::UniqueId" )
UrlConfig = Tac.Type( "Classification::UrlConfig" )
FieldSetLimit = Tac.Type( "Classification::FieldSetLimit" )
macEntry = Tac.Value( 'Arnet::EthAddr' )
UdfValueAndMask = Tac.Type( "Classification::UdfValueAndMask" )

fieldSetConfigValidCallbacks = []

def registerFieldSetConfigValidCallbacks( callback ):
   # Register feature-specific methods that are used to check
   # upon a field-set update that the feature has propagated
   # the change to hardware. In case any user of the field-set
   # reports a failure, the field-set configuration is
   # rolled back, if the CLI mode supports it.
   #
   # Callback parameters:
   # @mode: CLI mode
   # @fieldSetName: Name of field-set being updated.
   # @fieldSetType: Type of field-set being updated.
   #                One of ( 'ipv4', 'ipv6', 'l4-port' )
   # Return values:
   # ( isValid, issues ) :
   # ( bool, dict( 'error' : <String with programming errors>,
   #               'warning' : <Warning strings to add to CLI>. ) )
   #   ==> On success, status == True.
   #   ==> On failure, status == False and
   #                   'issues' will describe any programming issues.
   #
   fieldSetConfigValidCallbacks.append( callback )

def fieldSetCommitIssueStr( fieldSetType, fieldSetName, errStr=None,
                            warningStr=None ):
   fieldSetTypeStr = fieldSetTypeToStr[ fieldSetType ]

   if errStr:
      return ( "Error: Failed to commit {} field-set {} - {}".format(
               fieldSetTypeStr, fieldSetName, errStr ) )
   elif warningStr:
      return ( "Warning: {} field-set {} - {}".format(
               fieldSetTypeStr, fieldSetName, warningStr ) )
   else:
      assert False, 'Either an error or warning string must be provided'
      return ''

class ProtocolContextMixin:
   def updateOredProtocolAttr( self, rangeType, rangeSet, protoWithOtherField,
                               add=True ):
      currProtos = self.getProto() # pylint: disable=no-member
      # 'no protocol' removes all proto fields.
      if not add and not rangeSet:
         currProtos.clear()
         return
      protoRanges = rangeSetToNumericalRange( rangeSet, rangeType )
      # If adding proto value with other fields (sport/dport/icmp type/tcp flag), add
      # this proto value instead of merge with other protocol values.
      if protoWithOtherField:
         assert add
         for protoRange in protoRanges:
            currProtos.newMember( protoRange )
      # If adding/removing proto alone, merge with other stand-alone protocol values.
      else:
         newProtos = addOrDeleteRange( currProtos, rangeSet, rangeType, add )
         for currProto in currProtos:
            currProtoField = currProtos[ currProto ]
            if currProto not in newProtos and \
               not ( currProtoField.port or currProtoField.icmpType or
                     currProtoField.tcpFlagAndMask ):
               del currProtos[ currProto ]
         for newProto in newProtos:
            currProtos.newMember( newProto )

   def addOredPortRangeAttr( self, protoSet, sportSet, dportSet, clearPrev=False ):
      """
      Implements the ORed logic for adding l4 ports for a given protocol. e.g.,
         (config)protocol tcp source port 10 destination port 20
         (config)protocol tcp source port 30 destination port 40
         (config)protocol tcp source port 10 destination port 50
      After the second config, the resulting config is:
         protocol tcp source port 10 destination port 20
         protocol tcp source port 30 destination port 40
         which means (sport 10 AND dport 20) OR (sport 30 AND dport 40) for tcp.
      After the third config, the resulting config is:
         protocol tcp source port 10 destination port 20,50
         protocol tcp source port 30 destination port 40
         which means (sport 10 AND (dport 20 OR 50)) OR (sport 30 AND dport 40)
      """
      def updatePortSet( curPortSet, newPortList ):
         for curPort in curPortSet:
            if curPort not in newPortList:
               del curPortSet[ curPort ]
         for aRange in newPortList:
            if aRange not in curPortSet:
               curPortSet.add( aRange )

      protoColl = self.getProto() # pylint: disable=no-member
      protos = rangeSetToNumericalRange( protoSet, "Classification::ProtocolRange" )
      sportList = rangeSetToNumericalRange( sportSet, "Classification::PortRange" )
      dportList = rangeSetToNumericalRange( dportSet, "Classification::PortRange" )
      newPortField = Tac.newInstance( 'Classification::PortField', UniqueId() )
      for sport in sportList:
         newPortField.sport.add( sport )
      for dport in dportList:
         newPortField.dport.add( dport )
      for proto in protos:
         if clearPrev:
            protoColl[ proto ].port.clear()
         # When 'port' collection is empty, add the newPortField to 'port' collection
         if not protoColl[ proto ].port:
            portField = protoColl[ proto ].port.newMember( newPortField.id )
            portField.copy( newPortField )
            continue
         if sportSet and dportSet:
            mergeAttr = ''
            for portField in protoColl[ proto ].port.values():
               # When the same 'sport' is found, merge the 'dport'
               if portField.hasEqualField( newPortField, 'sport' ):
                  mergeAttr = 'dport'
                  mergePortSet = dportSet
               elif portField.hasEqualField( newPortField, 'dport' ):
                  mergeAttr = 'sport'
                  mergePortSet = sportSet
               if mergeAttr:
                  curPortSet = getattr( portField, mergeAttr, set() )
                  newPortList = addOrDeleteRange( curPortSet, mergePortSet,
                                                  "Classification::PortRange",
                                                  add=True )
                  updatePortSet( curPortSet, newPortList )
                  break
            # If no same 'sport'/'dport' field, add the newPortField object to 'port'
            if not mergeAttr:
               portField = protoColl[ proto ].port.newMember( newPortField.id )
               portField.copy( newPortField )
               continue
         # Handle condition that only configures sport or dport. e.g.,
         # When config 'protocol tcp source port 1-3 [destination port all]',
         # search if previously have config with the same sport, e.g.,
         # `protocol tcp source port 1-3 destination port 10`,
         # then merge them to 'protocol tcp source port 1-3'. If not, search if has
         # previous config with empty dport `protocol tcp source port XX`, if yes,
         # merge sport XX and sport (1-3).
         else:
            portSet, portAttr, matchAllPortAttr = ( sportSet, 'sport', 'dport' ) if \
                                       sportSet else ( dportSet, 'dport', 'sport' )
            hasEqualField = False
            for portField in protoColl[ proto ].port.values():
               # pylint: disable-next=no-else-break
               if portField.hasEqualField( newPortField, portAttr ):
                  hasEqualField = True
                  getattr( portField, matchAllPortAttr ).clear()
                  break
               elif getattr( portField, portAttr ) and \
                    portField.hasEqualField( newPortField, matchAllPortAttr ):
                  hasEqualField = True
                  curPortSet = getattr( portField, portAttr, set() )
                  newPortList = addOrDeleteRange( curPortSet, portSet,
                                                  "Classification::PortRange",
                                                  add=True )
                  updatePortSet( curPortSet, newPortList )
                  break
            # If no same 'sport'/'dport' set, create a new portField object
            if not hasEqualField:
               portField = protoColl[ proto ].port.newMember( newPortField.id )
               portField.copy( newPortField )

   def maybeUpdateProto( self, protocolRangeSet ):
      """
      Only called when removing additional protocol fields (sport/dport/flags).
      When all additional fields have been removed, we removed the protocol too
      """
      protos = rangeSetToNumericalRange( protocolRangeSet,
                                         "Classification::ProtocolRange" )
      protoColl = self.getProto() # pylint: disable=no-member
      for proto in protos:
         if proto not in protoColl:
            continue
         for portId, portField in protoColl[ proto ].port.items():
            sport = portField.sport
            sportFieldSet = portField.sportFieldSet
            dport = portField.dport
            dportFieldSet = portField.dportFieldSet
            if sport or sportFieldSet or dport or dportFieldSet:
               continue
            del protoColl[ proto ].port[ portId ]
         protoField = protoColl[ proto ]
         tcpFlagAndMask = protoField.tcpFlagAndMask
         icmpType = protoField.icmpType
         port = protoField.port
         if ( tcpFlagAndMask and list( tcpFlagAndMask ).pop().tcpFlagMask ) or\
            port or icmpType:
            continue
         del protoColl[ proto ]

   def removeOredPortRangeAttr( self, protoSet, sportSet, dportSet,
                                clearPrev=False ):
      """
      Removes l4 ports from the ORed ports configuration for a given protocol. e.g.,
         (config)protocol tcp source port 10-12 destination port 20-25
         (config)no protocol tcp source port 10-12 destination port 23
         (config)no protocol tcp source port 10-12 destination port 20-25
      The second config is ignored.
      The third config matches the previous config and removes the first config.
      """
      protoColl = self.getProto() # pylint: disable=E1101

      protos = rangeSetToNumericalRange( protoSet, "Classification::ProtocolRange" )
      sportList = rangeSetToNumericalRange( sportSet, "Classification::PortRange" )
      dportList = rangeSetToNumericalRange( dportSet, "Classification::PortRange" )
      newPortField = Tac.newInstance( 'Classification::PortField', UniqueId() )
      for sport in sportList:
         newPortField.sport.add( sport )
      for dport in dportList:
         newPortField.dport.add( dport )
      # When removing l4 port config, if not both sport and sport match the exsiting
      # l4 port configs, the removing is ignored.
      isIgnored = True
      for proto in protos:
         if proto not in protoColl:
            continue
         for portField in protoColl[ proto ].port.values():
            if portField.isEqual( newPortField ):
               isIgnored = False
               del protoColl[ proto ].port[ portField.id ]
               break
      return isIgnored

class MatchRuleBaseContext( ProtocolContextMixin ):
   def __init__( self, ruleName, matchOption ):
      self.ruleName = ruleName
      self.matchOption = matchOption
      self.filter = Tac.newInstance( "Classification::StructuredFilter", "" )

   def setAction( self, actionType, actionValue=None, no=False, clearActions=None ):
      raise NotImplementedError

   def addAppProfileMatchToSf( self, appProfileSf ):
      if appProfileSf.appProfile is None:
         return
      for appProfName in appProfileSf.appProfile:
         self.filter.appProfile.add( appProfName )

   def removeAppProfileMatchFromSf( self, appProfileSf ):
      if not appProfileSf.appProfile:
         self.filter.appProfile.clear()
         return
      for appProfName in appProfileSf.appProfile:
         del self.filter.appProfile[ appProfName ]

   def delAllAction( self ):
      '''
      called when 'no actions' is configured.
      '''
      raise NotImplementedError

   def copyEditMatchRule( self, ruleName, seqnum ):
      raise NotImplementedError

   def newEditMatchRule( self, ruleName, seqnum ):
      raise NotImplementedError

   def hasOtherInnerMatch( self, newMatchType ):
      """
      Returns True if existing inner match af differs from new af
      """
      if self.filter.innerMatch and self.filter.innerMatch.af != newMatchType:
         return True
      return False

   def hasOtherEncapType( self, newEncapType ):
      """
      Returns True if existing encap Type differs from new encapType
      """
      if self.filter.encapField and self.filter.encapField.encapType != newEncapType:
         return True
      return False

   def isValidConfig( self, conflictType ):
      if not self.filter:
         return False
      return self.filter.isValidConfig( conflictType )

   def addOrRemovePrefix( self, prefixes, filterType, add ):
      if filterType not in [ 'source', 'destination' ]:
         return
      # remove any previously configured prefixes
      if filterType == 'source':
         self.filter.srcPrefixSet.clear()
         self.filter.sourceLpm.clear()
         self.filter.srcLpmPrefixSet.clear()
      else:
         self.filter.dstPrefixSet.clear()
         self.filter.destinationLpm.clear()
         self.filter.dstLpmPrefixSet.clear()
         self.removeDestinationSelfFromSf()
      prefixList = getattr( self.filter, filterType )
      for prefix in prefixes:
         prefix = IpGenPrefix( str( prefix ) )
         if add:
            prefixList[ prefix ] = True
         else:
            del prefixList[ prefix ]

   def addMacFromSf( self, macSf ):
      if macSf.srcMac:
         # Cannot have src mac field-sets and src mac at the same time
         self.filter.srcMacAddrSet.clear()
         for macAddr in macSf.srcMac:
            self.filter.srcMac.add( macAddr )
      if macSf.srcMacAddrSet:
         # Cannot have src mac field-sets and src mac at the same time
         self.filter.srcMac.clear()
         for fsName in macSf.srcMacAddrSet:
            self.filter.srcMacAddrSet.add( fsName )
      if macSf.dstMac:
         # Cannot have dst mac field-sets and dst mac at the same time
         self.filter.dstMacAddrSet.clear()
         for macAddr in macSf.dstMac:
            self.filter.dstMac.add( macAddr )
      if macSf.dstMacAddrSet:
         # Cannot have dst mac field-sets and dst mac at the same time
         self.filter.dstMac.clear()
         for fsName in macSf.dstMacAddrSet:
            self.filter.dstMacAddrSet.add( fsName )

   def removeMacFromSf( self, macSf ):
      for macAddr in macSf.srcMac:
         del self.filter.srcMac[ macAddr ]
      for macAddr in macSf.dstMac:
         del self.filter.dstMac[ macAddr ]
      for fsName in macSf.srcMacAddrSet:
         del self.filter.srcMacAddrSet[ fsName ]
      for fsName in macSf.dstMacAddrSet:
         del self.filter.dstMacAddrSet[ fsName ]

   def addPrefixFromSf( self, prefixSf ):
      if prefixSf.source:
         # Cannot have (Lpm) field-sets and prefixes
         self.filter.srcPrefixSet.clear()
         self.filter.sourceLpm.clear()
         self.filter.srcLpmPrefixSet.clear()
         for prefix in prefixSf.source:
            self.filter.source.add( prefix )
      if prefixSf.destination:
         # Cannot have (Lpm) field-sets and prefixes
         self.filter.dstPrefixSet.clear()
         self.filter.destinationLpm.clear()
         self.filter.dstLpmPrefixSet.clear()
         self.removeDestinationSelfFromSf()
         for prefix in prefixSf.destination:
            self.filter.destination.add( prefix )
      if prefixSf.srcPrefixSet:
         # Cannot have prefixes and (Lpm) field-sets
         self.filter.source.clear()
         self.filter.sourceLpm.clear()
         self.filter.srcLpmPrefixSet.clear()
         for fsName in prefixSf.srcPrefixSet:
            self.filter.srcPrefixSet.add( fsName )
      if prefixSf.dstPrefixSet:
         # Cannot have prefixes and (Lpm) field-sets
         self.filter.destination.clear()
         self.filter.destinationLpm.clear()
         self.filter.dstLpmPrefixSet.clear()
         self.removeDestinationSelfFromSf()
         for fsName in prefixSf.dstPrefixSet:
            self.filter.dstPrefixSet.add( fsName )
      if prefixSf.sourceLpm:
         # Cannot have (Lpm) field-sets and prefixes
         self.filter.source.clear()
         self.filter.srcPrefixSet.clear()
         self.filter.srcLpmPrefixSet.clear()
         for prefix in prefixSf.sourceLpm:
            self.filter.sourceLpm.add( prefix )
      if prefixSf.srcLpmPrefixSet:
         # Cannot have prefixes and (Lpm) field-sets
         self.filter.source.clear()
         self.filter.sourceLpm.clear()
         self.filter.srcPrefixSet.clear()
         for fsName in prefixSf.srcLpmPrefixSet:
            self.filter.srcLpmPrefixSet.add( fsName )
      if prefixSf.destinationLpm:
         # Cannot have (Lpm) field-sets and prefixes
         self.filter.destination.clear()
         self.filter.dstPrefixSet.clear()
         self.filter.dstLpmPrefixSet.clear()
         self.removeDestinationSelfFromSf()
         for prefix in prefixSf.destinationLpm:
            self.filter.destinationLpm.add( prefix )
      if prefixSf.dstLpmPrefixSet:
         # Cannot have prefixes and (Lpm) field-sets
         self.filter.destination.clear()
         self.filter.destinationLpm.clear()
         self.filter.dstPrefixSet.clear()
         self.removeDestinationSelfFromSf()
         for fsName in prefixSf.dstLpmPrefixSet:
            self.filter.dstLpmPrefixSet.add( fsName )
      if prefixSf.toCpu:
         self.filter.destination.clear()
         self.filter.destinationLpm.clear()
         self.filter.dstPrefixSet.clear()
         self.filter.dstLpmPrefixSet.clear()
         self.filter.toCpu = prefixSf.toCpu

   def removePrefixFromSf( self, prefixSf ):
      self.filter.toCpu = prefixSf.toCpu
      for prefix in prefixSf.source:
         del self.filter.source[ prefix ]
      for prefix in prefixSf.destination:
         del self.filter.destination[ prefix ]
      for fsName in prefixSf.srcPrefixSet:
         del self.filter.srcPrefixSet[ fsName ]
      for fsName in prefixSf.dstPrefixSet:
         del self.filter.dstPrefixSet[ fsName ]
      for prefix in prefixSf.sourceLpm:
         del self.filter.sourceLpm[ prefix ]
      for fsName in prefixSf.srcLpmPrefixSet:
         del self.filter.srcLpmPrefixSet[ fsName ]
      for prefix in prefixSf.destinationLpm:
         del self.filter.destinationLpm[ prefix ]
      for fsName in prefixSf.dstLpmPrefixSet:
         del self.filter.dstLpmPrefixSet[ fsName ]

   def updatePrefixFieldSet( self, source=True, names=None, add=True ):
      # clear all source and destination configurations
      srcFilterTypes = [ 'source', 'sourceExcept', 'sourceLpm', 'sourceLpmExcept' ]
      dstFilterTypes = [ 'destination', 'destinationExcept', 'destinationLpm',
                         'destinationLpmExcept' ]
      clearFilterTypes = srcFilterTypes if source else dstFilterTypes
      for f in clearFilterTypes:
         prefixList = getattr( self.filter, f )
         prefixList.clear()
      # add or remove fieldSet
      # Cannot have both LPM and non-LPM field sets. This method is only called
      # from the non V2 commands so we know we need to clear the LPM field sets.
      if source:
         fieldSetList = self.filter.srcPrefixSet
         self.filter.srcLpmPrefixSet.clear()
      else:
         fieldSetList = self.filter.dstPrefixSet
         self.filter.dstLpmPrefixSet.clear()
      if add:
         for n in names:
            fieldSetList.add( n )
      else:
         if names:
            for n in names:
               del fieldSetList[ n ]
         else:
            fieldSetList.clear()

   def updateServiceFieldSet( self, names=None, add=True ):
      # cannot have protocols defined along with service field-sets or vice-versa
      protoList = getattr( self.filter, 'proto' )
      protoList.clear()
      fieldSetList = self.filter.serviceSet
      # add or remove fieldSet
      if add:
         for n in names:
            fieldSetList.add( n )
      else:
         if names:
            for n in names:
               del fieldSetList[ n ]
         else:
            fieldSetList.clear()

   def getProto( self ):
      return self.filter.proto

   def clearServiceFieldSet( self ):
      self.filter.serviceSet.clear()

   @staticmethod
   def updateTcpFlags( sf, tcpFlagsDict, add=True ):
      """
      Updates TcpFlagAndMask for a given match rule.
      e.g., for 'no protocol tcp syn not-rst ack not-psh',
      tcpFlagsDict = { 'yesFlags' : [syn, ack],
                       'notFlags' : [rst, psh] },
      add = False
      """
      # Using 'isUpdatingIgnored' to indicate if updating TCP flag config is ignored,
      # will skip updating l4 port config when 'isUpdatingIgnored=True'.
      isUpdatingIgnored = True
      yesFlags = tcpFlagsDict.get( 'yesFlags', [] )
      notFlags = tcpFlagsDict.get( 'notFlags', [] )
      acceptedFlags = [ 'est', 'init', 'fin', 'syn', 'rst', 'psh', 'ack', 'urg' ]
      tcpFlags = yesFlags + notFlags
      for tcpFlag in tcpFlags:
         if tcpFlag not in acceptedFlags:
            return isUpdatingIgnored
      # When yesFlags and notFlags are not disjoint, return early.
      if not set( yesFlags ).isdisjoint( set( notFlags ) ):
         return isUpdatingIgnored
      tcpProto = ProtocolRange( IPPROTO_TCP, IPPROTO_TCP )
      protoField = sf.proto.get( tcpProto )
      if not protoField:
         return isUpdatingIgnored
      # Removing from empty tcp flag set will be ignore.
      if not add and not protoField.tcpFlagAndMask:
         return isUpdatingIgnored

      # Handle special case: (no) protocol tcp flags established initial, treat it as
      # established OR initial.
      if sorted( yesFlags ) == [ 'est', 'init' ]:
         newTcpFlagAndMasks = []
         for tcpFlag in yesFlags:
            if tcpFlag == 'est':
               # ack | rst
               newTcpFlagAndMasks += getTcpFlagAndMasksEst()
            else:
               assert tcpFlag == 'init'
               newTcpFlagAndMasks.append( getTcpFlagAndMaskInit() )
         if add:
            for newTcpFlagAndMask in newTcpFlagAndMasks:
               if newTcpFlagAndMask.tcpFlagMask.value and \
                  newTcpFlagAndMask not in protoField.tcpFlagAndMask:
                  protoField.tcpFlagAndMask.add( newTcpFlagAndMask )
            isUpdatingIgnored = False
         else:
            if newTcpFlagAndMasks[ 0 ] in protoField.tcpFlagAndMask and \
               newTcpFlagAndMasks[ 1 ] in protoField.tcpFlagAndMask:
               for newTcpFlagAndMask in newTcpFlagAndMasks:
                  del protoField.tcpFlagAndMask[ newTcpFlagAndMask ]
               isUpdatingIgnored = False
         return isUpdatingIgnored

      newTcpFlag = Tac.Value( "Classification::TcpFlag" )
      newTcpFlagMask = Tac.Value( "Classification::TcpFlag" )
      newFlagAndMasks = []
      for tcpFlag in yesFlags:
         if tcpFlag == 'est':
            newFlagAndMasks += getTcpFlagAndMasksEst()
         elif tcpFlag == 'init':
            newFlagAndMasks.append( getTcpFlagAndMaskInit() )
         else:
            setattr( newTcpFlag, tcpFlag, True )
            setattr( newTcpFlagMask, tcpFlag, True )
      for tcpFlag in notFlags:
         assert tcpFlag not in [ 'est', 'init' ]
         setattr( newTcpFlag, tcpFlag, False )
         setattr( newTcpFlagMask, tcpFlag, True )
      # It is possible the TCP flags only contain "notFlags", hence "or" -- if either
      # the mask or the flags are non-default, update the filter.
      if newTcpFlag:
         # It is nonsensical for the mask to be unset but have some flags be set.
         assert newTcpFlagMask
      if newTcpFlagMask:
         # non-init/est flags
         newTcpFlagAndMask = Tac.Value( "Classification::TcpFlagAndMask", newTcpFlag,
                                        newTcpFlagMask )
         newFlagAndMasks.append( newTcpFlagAndMask )
      for newTcpFlagAndMask in newFlagAndMasks:
         if add:
            if newTcpFlagAndMask.tcpFlagMask.value and \
               newTcpFlagAndMask not in protoField.tcpFlagAndMask:
               protoField.tcpFlagAndMask.add( newTcpFlagAndMask )
               isUpdatingIgnored = False
         else:
            if newTcpFlagAndMask in protoField.tcpFlagAndMask:
               del protoField.tcpFlagAndMask[ newTcpFlagAndMask ]
               isUpdatingIgnored = False
      return isUpdatingIgnored

   def updateProtoRangeFromSf( self, protoSf ):
      '''This function will overwrite the current filter's proto state with the
         proto fields from the input filter'''
      self.filter.proto.clear()
      for protoRange in protoSf.proto:
         self.filter.proto.newMember( protoRange )

   def clearProtoRangeFromSf( self ):
      '''This function will delete all the proto range entries in the current
         structuredFilter'''
      self.filter.proto.clear()

   def updateRangeAttr( self, attrName, rangeType, rangeSet, add=True ):
      acceptedAttr = [ 'ttl', 'sport', 'dport', 'proto', 'dscp', 'length',
                       'fragmentOffset', 'vlanId', 'vlanTag', 'innerVlanTag',
                       'cos', 'dzGrePolicyId', 'dzGrePortId', 'dzGreSwitchId' ]
      if attrName not in acceptedAttr:
         return
      # retrieve the current collection of NumericalRange objs
      if attrName == "fragmentOffset":
         fragment = getattr( self.filter, "fragment" )
         if fragment:
            currentAttrList = fragment.offset
         else:
            currentAttrList = ()
      else:
         currentAttrList = getattr( self.filter, attrName )
      newRangeList = addOrDeleteRange( currentAttrList,
                                       rangeSet, rangeType, add )
      # Cannot delete all members and re-add for protocols because protocols have
      # additional protoFields
      for currAttr in currentAttrList:
         if currAttr not in newRangeList:
            del currentAttrList[ currAttr ]
      for aRange in newRangeList:
         if aRange not in currentAttrList:
            if attrName == 'proto':
               currentAttrList.newMember( aRange )
            else:
               currentAttrList.add( aRange )

   def updateIcmpRangeAttr( self, icmpValue, rangeType, rangeSet, add=True ):
      # Each ICMP type may or may not have ICMP codes configured. For example, we
      # have type [1,2], type [3,3] code [5,10], type [10,20], when type [2,4] is
      # added. Firstly merge type [2,4] with type [1,2] and [10,20]. Then remove
      # type [ 3, 3 ] code [ 5, 10 ] because it covered by the added type [ 2, 4 ].
      icmpProto = ProtocolRange( icmpValue, icmpValue )
      if self.filter.proto.get( icmpProto ) is None:
         return
      icmpTypeRangeColl = self.filter.proto[ icmpProto ].icmpType
      # Remove all ICMP types if type is not specified
      if not rangeSet:
         icmpTypeRangeColl.clear()
         return
      typeRangeWithCodeList = []
      typeRangeWithoutCodeList = []
      for aTypeRange in icmpTypeRangeColl:
         if icmpTypeRangeColl[ aTypeRange ].icmpCode:
            typeRangeWithCodeList.append( aTypeRange )
         else:
            typeRangeWithoutCodeList.append( aTypeRange )
      newRangeList = addOrDeleteRange( typeRangeWithoutCodeList, rangeSet,
                                       rangeType, add )
      for currRange in typeRangeWithoutCodeList:
         if currRange not in newRangeList:
            del icmpTypeRangeColl[ currRange ]
      for aRange in newRangeList:
         if aRange not in typeRangeWithoutCodeList:
            icmpTypeRangeColl.newMember( aRange )
      for aRange in typeRangeWithCodeList:
         aRangeValue = numericalRangeToSet( [ aRange ] ).pop()
         if aRangeValue in rangeSet:
            # e.g., when 'protocol icmp type 3 code all' is configured after
            # 'protocol icmp type 3 code 1', we have aRangeValue=3 and
            # rangeSet = set( [ 3 ] ), so update to type [3,3] code all.
            if len( rangeSet ) == 1:
               icmpTypeRangeColl[ aRange ].icmpCode.clear()
            # e.g., when 'protocol icmp type 1-3 code all' is configured after
            # 'protocol icmp type 3 code 1', we have aRangeValue=3 and
            # rangeSet = set( [ 1,2,3 ] ), so remove type [3,3] code 1.
            else:
               del icmpTypeRangeColl[ aRange ]

   def addIcmpTypeCodeRangeAttr( self, icmpValue, typeRangeType, typeValue,
                                 codeRangeType, codeSet ):
      icmpProto = ProtocolRange( icmpValue, icmpValue )
      if self.filter.proto.get( icmpProto ) is None:
         return
      currIcmpTypeRangeSet = self.filter.proto[ icmpProto ].icmpType
      currIcmpTypeValueSet = numericalRangeToSet( currIcmpTypeRangeSet )
      icmpType = IcmpTypeRange( typeValue, typeValue )
      # e.g., IcmpType [3, 3], currIcmpTypeRangeSet [1, 2], add the new ICMP type and
      # its codes.
      if typeValue not in currIcmpTypeValueSet:
         currIcmpTypeRangeSet.newMember( icmpType )
         icmpCodeRangeList = rangeSetToNumericalRange( codeSet, codeRangeType )
         for icmpCodeRange in icmpCodeRangeList:
            currIcmpTypeRangeSet[ icmpType ].icmpCode.add( icmpCodeRange )
      # e.g., icmpType [3, 3], currIcmpTypeRangeSet [1, 2], [3, 3], only add the ICMP
      # codes when it's not code all.
      elif icmpType in currIcmpTypeRangeSet:
         currIcmpCodeRangeSet = currIcmpTypeRangeSet[ icmpType ].icmpCode
         if not currIcmpCodeRangeSet:
            return
         newCodeRangeSet = addOrDeleteRange( currIcmpCodeRangeSet, codeSet,
                                             codeRangeType, add=True )
         for currCode in currIcmpCodeRangeSet:
            if currCode not in newCodeRangeSet:
               del currIcmpCodeRangeSet[ currCode ]
         for aCode in newCodeRangeSet:
            currIcmpCodeRangeSet.add( aCode )
      # e.g., icmpType [3, 3], currIcmpTypeRangeSet [1, 4], [10, 20], ignore it
      # because it is coverd by code all.

   def removeIcmpTypeCodeRangeAttr( self, icmpValue, typeRangeType, typeValue,
                                    codeRangeType, codeSet ):
      icmpProto = ProtocolRange( icmpValue, icmpValue )
      if icmpProto not in self.filter.proto:
         return
      currIcmpTypeRangeSet = self.filter.proto[ icmpProto ].icmpType
      icmpType = IcmpTypeRange( typeValue, typeValue )
      # If currIcmpTypeRangeSet has icmpType and it's not code all, remove codes of
      # this ICMP type, otherwise do nothing.
      # e.g., icmpType [ 3, 3 ], currIcmpTypeRangeSet [ 1, 2 ], [ 3, 3 ], if the
      # current ICMP type has code all, do nothing. If not, remove the codes from the
      # current codes.
      if icmpType in currIcmpTypeRangeSet:
         currIcmpCodeRangeSet = currIcmpTypeRangeSet[ icmpType ].icmpCode
         if not currIcmpCodeRangeSet:
            return
         newCodeRangeSet = addOrDeleteRange( currIcmpCodeRangeSet, codeSet,
                                             codeRangeType, add=False )
         for currCode in currIcmpCodeRangeSet:
            if currCode not in newCodeRangeSet:
               del currIcmpCodeRangeSet[ currCode ]
         for aCode in newCodeRangeSet:
            if aCode not in currIcmpCodeRangeSet:
               currIcmpCodeRangeSet.add( aCode )
         # When all ICMP codes are removed, also remove the corresponding ICMP type.
         if not currIcmpCodeRangeSet:
            del currIcmpTypeRangeSet[ icmpType ]

   def updatePortFieldSetAttr( self, attrName, fieldSetNames, add=True,
                               protoSet=None ):
      """
      Adds/removes l4 field-sets for a given protocol. When field-set is applied,
      all configured l4 ports are clear. (both sport and dport)
      Example:
         original config: protocol tcp source port 10 destination port 50
         new config: protocol tcp source port field-set sample
         result: protocol tcp source port field-set sample
                 both source port of 10 and destination port of 50 are removed
      """
      if attrName not in [ 'sportFieldSet', 'dportFieldSet' ]:
         return
      protoColl = getattr( self.filter, 'proto' )

      protos = rangeSetToNumericalRange( protoSet, "Classification::ProtocolRange" )
      # for each protocol remove the l4 ports, if no more port remaining remove proto
      # XXXBUG494947
      for proto in protos:
         if add and not protoColl[ proto ].port:
            if fieldSetNames:
               portField = protoColl[ proto ].port.newMember( UniqueId() )
            for fs in fieldSetNames:
               getattr( portField, attrName ).add( fs )
            continue
         # There's only one element in protoColl[ proto ].port
         portField = list( protoColl[ proto ].port.values() ).pop()
         portProtoAttrList = getattr( portField, attrName )
         # After 'protocol tcp source port 10 destination port 20', when
         # 'no protocol tcp source port field-set bar destination port field-set bar'
         # is configured, 'sport' and 'dport' should not be cleared.
         if not add and not portProtoAttrList:
            return
         # Ensure l4 ports are cleared. Cannot have l4 ports and field-set
         # configured in the same line
         portField.sport.clear()
         portField.dport.clear()
         for fs in fieldSetNames:
            if add:
               getattr( portField, attrName ).add( fs )
            else:
               getattr( portField, attrName ).remove( fs )

   def updatePortRangeAttr( self, attrName, protoSet, portSet, add=True,
                            clearPrev=False ):
      """
      Adds/removes l4 ports for a given protocol. When port is applied,
      all configured l4 field-sets are clear for that type (sport/dport), e.g.,
        (config)protocol tcp source port 10 destination port 20
        (config)protocol tcp source port 30 destination port 40
        (config)protocol tcp source port field-set foo destination port field-set bar
      After the second config, the resulting config for tcp is:
        protocol tcp source port 10,30 destination port 20,40
        which means (sport 10 OR 30 ) AND (dport 20 OR 40)
      After the third config, the resulting config for tcp is:
        protocol tcp source port field-set foo destination port field-set bar
        which means (sport foo) AND (dport bar)
      """
      if attrName not in [ 'sport', 'dport' ]:
         return
      protoColl = getattr( self.filter, 'proto' )

      protos = rangeSetToNumericalRange( protoSet, "Classification::ProtocolRange" )
      # for each protocol remove the l4 ports, if no more port remaining remove proto
      # XXXBUG494947
      for proto in protos:
         # Bug 548845. If proto not in collection, return. Can be the case where
         # ports for multiple protocols are configured and we are removing all
         # ports for a particular protocol
         if proto not in protoColl:
            continue
         if add and not protoColl[ proto ].port:
            newPortList = rangeSetToNumericalRange( portSet,
                                                    "Classification::PortRange" )
            if newPortList:
               portField = protoColl[ proto ].port.newMember( UniqueId() )
            for pRange in newPortList:
               getattr( portField, attrName ).add( pRange )
            continue
         # There's only one element in protoColl[ proto ].port
         portField = list( protoColl[ proto ].port.values() ).pop()
         portProtoAttrList = getattr( portField, attrName )
         if clearPrev:
            portProtoAttrList.clear()
         # After 'protocol tcp source port field-set bar destination port field-set
         # bar 20', when 'no protocol tcp source port 10 destination port 20' is
         # configured, 'sportFieldSet' and 'dportFieldSet' should not be cleared.
         if not add and not portProtoAttrList:
            continue
         # Ensure field-set cleared. Cannot have field-set and l4 ports configured
         portField.sportFieldSet.clear()
         portField.dportFieldSet.clear()
         newPortList = addOrDeleteRange( portProtoAttrList, portSet,
                                         "Classification::PortRange", add )
         getattr( portField, attrName ).clear()
         for pRange in newPortList:
            getattr( portField, attrName ).add( pRange )

   def addOredPortFieldSetAttr( self, protoSet, sportFieldSetNames,
                                dportFieldSetNames ):
      """
      Implement the ORed logic for adding l4 ports field-set for a given protocol.
      e.g.,
        (config)protocol tcp source port field-set fs1 destination port field-set fs2
        (config)protocol tcp source port field-set fs1 destination port field-set fs3
        (config)protocol tcp source port field-set fs2 destination port field-set fs4
      After the second config, the resulting config is:
        protocol tcp source port field-set fs1 destination port field-set fs2,fs3
        which means (sport fs1) AND (dport fs2 OR fs3)
      After the third config, the resulting config is:
        protocol tcp source port field-set fs1 destination port field-set fs2,fs3
        protocol tcp source port field-set fs2 destination port field-set fs4
        which means ((sport fs1) AND (dport fs2 OR fs3)) OR (sport fs2 AND dport fs4)
      """
      protoColl = self.filter.proto

      protos = rangeSetToNumericalRange( protoSet, "Classification::ProtocolRange" )
      newPortField = Tac.newInstance( 'Classification::PortField', UniqueId() )
      for sport in sportFieldSetNames:
         newPortField.sportFieldSet.add( sport )
      for dport in dportFieldSetNames:
         newPortField.dportFieldSet.add( dport )
      for proto in protos:
         # When 'port' collection is empty, add the newPortField to 'port' collection
         if not protoColl[ proto ].port:
            portField = protoColl[ proto ].port.newMember( newPortField.id )
            portField.copy( newPortField )
            continue
         if sportFieldSetNames and dportFieldSetNames:
            mergeAttr = ''
            for portField in protoColl[ proto ].port.values():
               if portField.hasEqualField( newPortField, 'sportFieldSet' ):
                  mergeAttr = 'dportFieldSet'
                  mergePortFieldSetNames = dportFieldSetNames
               elif portField.hasEqualField( newPortField, 'dportFieldSet' ):
                  mergeAttr = 'sportFieldSet'
                  mergePortFieldSetNames = sportFieldSetNames
               if mergeAttr:
                  curPortFieldSet = getattr( portField, mergeAttr, set() )
                  for fieldSetName in mergePortFieldSetNames:
                     curPortFieldSet.add( fieldSetName )
                  break
            # If no same 'sportField'/'dportFieldSet' field, add the newPortField
            # object to 'port' collection.
            if not mergeAttr:
               portField = protoColl[ proto ].port.newMember( newPortField.id )
               portField.copy( newPortField )
               continue
         # Handle condition that only one of sportFieldSetNames and
         # dportFieldSetNames exists.
         # When config 'protocol tcp source port field-set foo', if previously have
         # l4 port config with the same sport, e.g.,
         # `protocol tcp source port field-set foo destination port field-set bar`,
         # then merge them to 'protocol tcp source port field-set foo'. If not,
         # search if we have previous config with empty dport, like
         # `protocol tcp source port field-set XX`,
         # if yes, merge sport field-set XX and sport foo.
         else:
            portFieldSetNames, portAttr, matchAllPortAttr = \
            ( sportFieldSetNames, 'sportFieldSet', 'dportFieldSet' ) if \
               sportFieldSetNames else \
               ( dportFieldSetNames, 'dportFieldSet', 'sportFieldSet' )
            portFieldSetNames = sportFieldSetNames if sportFieldSetNames else \
                                dportFieldSetNames
            portAttr = 'sportFieldSet' if sportFieldSetNames else 'dportFieldSet'
            matchAllPortAttr = 'dportFieldSet' if sportFieldSetNames else \
                               'sportFieldSet'
            hasEqualField = False
            for portField in protoColl[ proto ].port.values():
               # pylint: disable-next=no-else-break
               if portField.hasEqualField( newPortField, portAttr ):
                  hasEqualField = True
                  getattr( portField, matchAllPortAttr ).clear()
                  break
               elif getattr( portField, portAttr, set() ) and \
                    portField.hasEqualField( newPortField, matchAllPortAttr ):
                  hasEqualField = True
                  curPortFieldSet = getattr( portField, portAttr, set() )
                  for fieldSetName in portFieldSetNames:
                     curPortFieldSet.add( fieldSetName )
                  break
            if not hasEqualField:
               portField = protoColl[ proto ].port.newMember( newPortField.id )
               portField.copy( newPortField )

   def removeOredPortFieldSetAttr( self, protoSet, sportFieldSetNames,
                                   dportFieldSetNames ):
      """
      Removes l4 ports field-set from the ORed ports configuration for a given
      protocol. e.g.,
     (config)protocol tcp source port field-set fs1 destination port field-set fs2
     (config)no protocol tcp source port field-set fs1 destination port field-set fs3
     (config)no protocol tcp source port field-set fs1 destination port field-set fs2
      The second config is ignored.
      The third config matches the previous config and removes the previous config.
      """
      protoColl = self.filter.proto

      protos = rangeSetToNumericalRange( protoSet, "Classification::ProtocolRange" )
      newPortField = Tac.newInstance( 'Classification::PortField', UniqueId() )
      for sport in sportFieldSetNames:
         newPortField.sportFieldSet.add( sport )
      for dport in dportFieldSetNames:
         newPortField.dportFieldSet.add( dport )
      isIgnored = True
      for proto in protos:
         if proto not in protoColl:
            continue
         for portField in protoColl[ proto ].port.values():
            if portField.isEqual( newPortField ):
               isIgnored = False
               del protoColl[ proto ].port[ portField.id ]
               break
      return isIgnored

   def updateMatchAllFragment( self, add=False ):
      self.filter.matchAllFragments = add

   def updateMatchIpOptions( self, add=False ):
      self.filter.matchIpOptions = add

   def commit( self ):
      raise NotImplementedError

   def updateFilterAttr( self, attrName, attrValue ):
      setattr( self.filter, attrName, attrValue )

   def updateFragmentType( self, fragmentType ):
      if self.filter.fragment is None:
         self.filter.fragment = ( fragmentType, )
      elif self.filter.fragment.fragmentType != fragmentType:
         self.filter.fragment = None
         self.filter.fragment = ( fragmentType, )

   def clearFragment( self ):
      self.filter.fragment = None

   def maybeDelFragment( self ):
      if self.filter.fragment and self.filter.fragment.fragmentType == matchOffset \
         and not self.filter.fragment.offset:
         self.filter.fragment = None

   def addVlanTagFromSf( self, vlanTagSf ):
      for vlanRangeSet, dot1QMatch in vlanTagSf.vlanTag.items():
         if vlanRangeSet not in self.filter.vlanTag:
            outerVlan = self.filter.vlanTag.newMember( vlanRangeSet )
         else:
            outerVlan = self.filter.vlanTag[ vlanRangeSet ]
         if dot1QMatch.innerVlanTag:
            for vlanTag in dot1QMatch.innerVlanTag:
               if vlanTag not in outerVlan.innerVlanTag:
                  outerVlan.innerVlanTag.add( vlanTag )

   def removeVlanTagFromSf( self, vlanTagSf ):
      for vlanRangeSet, dot1QMatch in vlanTagSf.vlanTag.items():
         if vlanRangeSet not in self.filter.vlanTag:
            continue
         outerVlan = self.filter.vlanTag[ vlanRangeSet ]
         if not outerVlan.innerVlanTag and not dot1QMatch.innerVlanTag:
            del self.filter.vlanTag[ vlanRangeSet ]
         if outerVlan.innerVlanTag:
            for vlanTag in dot1QMatch.innerVlanTag:
               del outerVlan.innerVlanTag[ vlanTag ]
            if not outerVlan.innerVlanTag:
               del self.filter.vlanTag[ vlanRangeSet ]

   def removeDestinationSelfFromSf( self ):
      # replacing toCpu with default values
      toCpu = Tac.Value( 'Classification::ToCpuMatch' )
      self.filter.toCpu = toCpu

   def addVlanMatchToSf( self, vlan ):
      if vlan is None:
         return
      vlanRangeList = rangeSetToNumericalRange( vlan[ 0 ],
                                               "Classification::VlanRange" )
      for vlanRange in vlanRangeList:
         self.filter.vlanId.add( vlanRange )

   def removeVlanMatchFromSf( self, vlan ):
      if vlan is None:
         return
      vlanRangeList = rangeSetToNumericalRange( vlan[ 0 ],
                                               "Classification::VlanRange" )
      for vlanRange in vlanRangeList:
         del self.filter.vlanId[ vlanRange ]

   def addNhgMatchToSf( self, nexthopGroupName ):
      if nexthopGroupName is None:
         return
      if not isinstance( nexthopGroupName, list ):
         nexthopGroupName = [ nexthopGroupName ]
      for nhgName in nexthopGroupName:
         self.filter.nexthopGroup.add( nhgName )

   def removeNhgMatchFromSf( self, nexthopGroupName ):
      if nexthopGroupName is None:
         self.filter.nexthopGroup.clear()
         return
      if not isinstance( nexthopGroupName, list ):
         nexthopGroupName = [ nexthopGroupName ]
      for nhgName in nexthopGroupName:
         del self.filter.nexthopGroup[ nhgName ]

   def addLocationMatchToSf( self, locationAliasName, matchValue, matchMask ):
      if locationAliasName is None or matchValue is None or matchMask is None:
         return
      # if locationAliasName exists newMember returns that
      udfInfo = self.filter.udf.newMember( locationAliasName )
      udfValueAndMask = UdfValueAndMask( matchValue, matchMask )
      udfInfo.udfValueAndMask.add( udfValueAndMask )

   def removeLocationMatchFromSf( self, locationAliasName, matchValue, matchMask ):
      if locationAliasName is None or locationAliasName not in self.filter.udf:
         return
      if matchValue is not None and matchMask is None:
         return
      if matchValue is None and matchMask is not None:
         return
      udfInfo = self.filter.udf[ locationAliasName ]
      if matchValue is None:
         del self.filter.udf[ locationAliasName ]
      else:
         udfValueAndMask = UdfValueAndMask( matchValue, matchMask )
         # del is safe even if udfValueAndMask doesn't exist
         del udfInfo.udfValueAndMask[ udfValueAndMask ]
         if not udfInfo.udfValueAndMask:
            # delete unused alias
            del self.filter.udf[ locationAliasName ]

   def updateEthTypeFromSf( self, ethTypeSf, add ):
      if ethTypeSf.ethType:
         currEthTypeList = self.filter.ethType
         modifyEthTypeSet = numericalRangeToSet( ethTypeSf.ethType )
         newEthTypeList = addOrDeleteRange( currEthTypeList, modifyEthTypeSet,
                                            'Classification::EthTypeRange', add )
         for ethType in currEthTypeList:
            if ethType not in newEthTypeList:
               del currEthTypeList[ ethType ]
         for ethType in newEthTypeList:
            if ethType not in currEthTypeList:
               currEthTypeList.add( ethType )

   def updateTeidFromSf( self, teidSf, add ):
      if teidSf.teid:
         # teids and teid-field-sets are mutually exclusive.
         if add:
            self.filter.teidFieldSet.clear()
         currTeidList = self.filter.teid
         modifyTeidSet = numericalRangeToSet( teidSf.teid )
         newTeidList = addOrDeleteRange( currTeidList, modifyTeidSet,
                                         'Classification::TeidRange', add )
         for teid in currTeidList:
            if teid not in newTeidList:
               del currTeidList[ teid ]
         for teid in newTeidList:
            if teid not in currTeidList:
               currTeidList.add( teid )
      elif teidSf.teidFieldSet:
         # teids and teid-field-sets are mutually exclusive.
         if add:
            self.filter.teid.clear()
         currTeidFsList = self.filter.teidFieldSet
         for teidFs in teidSf.teidFieldSet:
            if add:
               currTeidFsList.add( teidFs )
            else:
               del currTeidFsList[ teidFs ]

#------------------------------------------------------------------------------------
# Context
# A context is created for FieldSet and stores the command till the user exits the
# mode or aborts the changes.
#
# If the FieldSet exists already, the context contains an editable copy of the
# contents; else it contains a new (editable) copy.
#------------------------------------------------------------------------------------
class FieldSetBaseContext:
   limitSupported = False
   limitExpanded = False
   def __init__( self, fieldSetName, fieldSetConfig, childMode,
                 featureName=None,
                 rollbackSupported=False,
                 urlFieldSetConfig=None,
                 urlLoadInitialMode=None,
                 urlRollbackSupported=False,
                 urlImportFailSyslogHandle=None,
                 urlImportSuccessSyslogHandle=None ):
      self.fieldSetName = fieldSetName
      # Entity holding field-sets input from CLI.
      self.fieldSetConfig = fieldSetConfig

      self.urlFieldSetConfig = urlFieldSetConfig
      self.childMode = childMode
      self.featureName = featureName
      self.mode_ = None
      self.setType = None

      # Collection of field-sets of type `self.setType` input
      # from CLI.
      self.fieldSetColl = None # set in child class

      self.urlFieldSetColl = None # set in child class
      self.prevFieldSet = None # set in child class
      self.editFieldSet = None # set in child class
      self.editFieldSetVersion = None # set in child class
      self.rollbackSupported = rollbackSupported
      # Initial mode for loading URL's field-sets
      self.urlLoadInitialMode = urlLoadInitialMode
      # Support for rollback of field-set config when we fail to load the
      # URL's field-set contents.
      self.urlRollbackSupported = urlRollbackSupported
      self.urlImportFailSyslogHandle = urlImportFailSyslogHandle
      self.urlImportSuccessSyslogHandle = urlImportSuccessSyslogHandle
      self.prevUrlFieldSet = None
      # Set to true when rollback has been completed in Sysdb.
      # The config session commit handler are called twice if
      # there is a config session rollback, so if we have rolled
      # back in the first handler call (due to checkStatus returning
      # a failure, then there's no work to do on the second call.
      self.rollbackComplete = False
      # We mark the context as successfully committed a config session
      # so that if some other feature causes the config session to
      # roll back, we can go ahead and complete the rollback on the
      # second session commit handler call.
      self.commitCfgSuccess = False

   def copyEditFieldSet( self ):
      raise NotImplementedError

   def newEditFieldSet( self ):
      raise NotImplementedError

   def updateFieldSet( self, data, add, **kwargs ):
      raise NotImplementedError

   def isFieldSetIdentical( self, left, right ):
      raise NotImplementedError

   def isFieldSetEmpty( self ):
      raise NotImplementedError

   def copy( self, src, dst ):
      raise NotImplementedError

   def cacheInitialUrlFieldSet( self ):
      if self.urlFieldSetsOnOwnMount():
         urlFsConfig = self.urlFieldSetColl.get( self.fieldSetName )
         if urlFsConfig:
            self.prevUrlFieldSet = urlFsConfig.currCfg

   def hasMacAddrFieldSet( self, name ):
      if self.fieldSetColl is None:
         return False
      return name in self.fieldSetColl

   def hasPrefixFieldSet( self, name, af ):
      if self.fieldSetColl is None:
         return False
      fieldSet = self.fieldSetColl.get( name )
      if fieldSet:
         return fieldSet.af == af
      return False

   def hasL4PortFieldSet( self, name ):
      if self.fieldSetColl is None:
         return False
      return name in self.fieldSetColl

   def hasServiceFieldSet( self, name ):
      if self.fieldSetColl is None:
         return False
      return name in self.fieldSetColl

   def hasVlanFieldSet( self, name ):
      if self.fieldSetColl is None:
         return False
      return name in self.fieldSetColl

   def hasIntegerFieldSet( self, name ):
      if self.fieldSetColl is None:
         return False
      return name in self.fieldSetColl

   def _newFieldSet( self ):
      # Used by `commit` to add a new config to the FieldSetColl.
      raise NotImplementedError

   def _newUrlFieldSet( self ):
      # Used by `commitUrlFieldSet ` to add a new config to the UrlFieldSetColl.
      raise NotImplementedError

   @property
   def callbackKey( self ):
      # Used to register a commit handler for config sessions.
      assert self.featureName
      assert self.setType
      return "{}-field-set-{}-{}".format( self.featureName,
                                          self.setType,
                                          self.fieldSetName )


   def delFieldSet( self, name, mode, fromNoHandler ):
      # Remove the field-set config entity corresponding to 'name'
      # from all input configs in Sysdb. If called from the context of
      # a `no` handler i.e., `no field-set`, or `no traffic-policies`,
      # we need to ensure that we do not modify non-config mounts if we
      # are within a config session. Instead, we register a commit handler
      # to delete the field-set when the session is committed.
      if self.fieldSetColl is None:
         # If the CLI mountpoint is not present, we should not expect the URL
         # mountpoint to have any field-sets.
         assert not self.urlFieldSetColl
         return

      # fieldSetConfig tracks the field-sets that are used by and-results field-set
      # and hence the mapping needs to be updated when any and-results field-set is
      # deleted
      if self.setType in ( "ipv4", "ipv6" ) and len( self.fieldSetColl[ name ].
         currCfg.intersectFieldSets ):
         for fsName in self.fieldSetColl[ self.fieldSetName ].\
               currCfg.intersectFieldSets:
            usedInAndResultsFieldSet = self.getUsedInAndResultsFieldSet()
            if fsName in usedInAndResultsFieldSet:
               usedInAndResultsFieldSet[ fsName ] -= 1
               count = usedInAndResultsFieldSet[ fsName ]
               if not count:
                  del usedInAndResultsFieldSet[ fsName ]

      del self.fieldSetColl[ name ]

      if self.urlFieldSetsOnOwnMount():
         if fromNoHandler:
            assert mode

         if mode and mode.session_.inConfigSession() and fromNoHandler:
            def handler( mode, onSessionCommit=True ):
               # Deleting a field-set is not expected to fail, so
               # we don't need to check for programming status.
               del self.urlFieldSetColl[ name ]
               return ''

            CliSession.registerSessionOnCommitHandler(
                  mode.session_.entityManager,
                  self.callbackKey,
                  handler )
         else:
            del self.urlFieldSetColl[ name ]

   def _delFieldSetFromConfigMount( self ):
      # Used internally to remove a field-set from the config-mount used by the
      # current CLI context (i.e., Sysdb, or config session).
      raise NotImplementedError

   def urlFieldSetsOnOwnMount( self ):
      return self.urlFieldSetConfig is not None

   def modeIs( self, mode ):
      self.mode_ = mode

   def tmpFieldSetExceedsLimit( self, tmpFieldSetConfig ):
      if self.editFieldSet.limit == FieldSetLimit.null:
         return False
      fieldSetCfg = getFieldSetConfigForType( self.setType, self.fieldSetName,
                                              tmpFieldSetConfig )
      if self.limitSupported and fieldSetCfg and fieldSetCfg.currCfg:
         return fieldSetCfg.currCfg.size() > self.editFieldSet.limit
      return False

   def copyTmpFieldSetToEditFieldSet( self, tmpFieldSetConfig ):
      fieldSetCfg = getFieldSetConfigForType( self.setType, self.fieldSetName,
                                              tmpFieldSetConfig )
      self.delAllContent() # Clears editFieldSet
      if fieldSetCfg and fieldSetCfg.currCfg:
         self.editFieldSet.copyContents( fieldSetCfg.currCfg )
         return True
      return False

   def fetchFieldSetContentsFromSessionMount( self, sessionName ):
      '''
      Copies content from the load-field-set session to the scratchpad in the
      current context.
      @in
      sessionName: Expected name of the temporary session where the field-set
                   is being loaded.
      @out
      Returns boolean representing if the field-set FSNAME of type FSTYPE was
      loaded from the session mount.
      '''
      mode = self.mode_
      em = mode.entityManager
      currSessionName = CliSession.currentSession( em )
      assert sessionName == currSessionName

      # We need to reference the config-mount, self.fieldSetConfig,
      # while in the temporary load-session to point to the session's
      # copy of the path.
      return self.copyTmpFieldSetToEditFieldSet( self.fieldSetConfig )

   def loadFieldSetFromUrl( self ):
      '''
      Copies contents from the field-set configured URL into the context's
      scratchpad i.e., self.editFieldSet.
      @out
      bool : Represents success/failure from loading the URL's contents.
      '''
      mode = self.mode_
      fsUrl = self.editFieldSet.urlConfig.url
      fsVrfName = self.editFieldSet.urlConfig.vrfName

      urlSchemes = listUrlLocalSchemes.union( listUrlNetworkSchemes )
      if isinstance( fsUrl, str ):
         fsUrl = Url.parseUrl( fsUrl,
                               Url.Context( *Url.urlArgsFromMode( mode ) ),
                               fsFunc=lambda fs: fs.scheme in urlSchemes )
      assert isinstance( fsUrl, Url.Url )
      if fsVrfName:
         assert isinstance( fsUrl, NetworkUrl )
         fsUrl.setUrlVrfName( fsVrfName )

      if isinstance( fsUrl, ScpUrl ):
         fsUrl.setAsRoot( True )

      loadSuccessful = True
      try:
         url = fsUrl.url
         scheme = url[ : url.find( ':' ) + 1 ]
         filename = None

         try:
            if scheme in listUrlLocalSchemes:
               filename = fsUrl.localFilename()
            else:
               # For external URLs, follow pattern of 'Url.open' in
               # /src/Cli/Url.py except we do not want to open the file in
               # python, the cpp parser is expected to open file. Python is
               # only responsible for fetching and storing the contents of
               # an external URL in a temp file.
               assert scheme in listUrlNetworkSchemes
               filename = fsUrl.makeLocalFile()
            tmpConfig = Tac.newInstance( "Classification::FieldSetConfig", '' )
            helper = Tac.newInstance( "ClassificationCli::FieldSetLoader" )
            errorString = helper.loadFieldSetFromFile( tmpConfig, filename )
            if errorString:
               # Failed to parse via cpp parser
               raise Exception( errorString )

            if self.tmpFieldSetExceedsLimit( tmpConfig ):
               exp = ' expanded' if self.limitExpanded else ''
               raise Exception( "Error: Limit of %d entries%s exceeded" %
                                ( self.editFieldSet.limit, exp ) )

            if self.copyTmpFieldSetToEditFieldSet( tmpConfig ):
               mode.addMessage(
                  "%s field-set '%s' was successfully loaded from %s"
                  % ( fieldSetTypeToStr[ self.setType ],
                      self.fieldSetName,
                      fsUrl.url ) )
               Logging.log( self.urlImportSuccessSyslogHandle,
                            fieldSetTypeToStr[ self.setType ],
                            self.fieldSetName,
                            fsUrl.url )
            else:
               raise Exception( "%s field-set '%s' not found in the URL" %
                                ( fieldSetTypeToStr[ self.setType ],
                                  self.fieldSetName ) )
         finally:
            fsUrl.cleanupLocalFile( filename )
      except Exception as e: # pylint: disable=broad-except
         mode.addError( f'Failed to import {fsUrl.url}: {e}' )
         Logging.log( self.urlImportFailSyslogHandle,
                      fieldSetTypeToStr[ self.setType ],
                      self.fieldSetName,
                      fsUrl.url,
                      e )
         if mode.session_.startupConfig():
            mode.addWarning( "%s field-set '%s' will be defined as an empty "
                             "field-set" % ( fieldSetTypeToStr[ self.setType ],
                                             self.fieldSetName ) )
         loadSuccessful = False

      return loadSuccessful

   def refreshFieldSet( self ):
      self.copyEditFieldSet()
      if not self.isUsingUrl():
         self.mode_.addWarning( "%s field-set '%s' was not refreshed because "
                                "it does not have an URL as source" % (
                                fieldSetTypeToStr[ self.setType ],
                                self.fieldSetName ) )
         return

      self.importAndCommitUrlFieldSet( refreshCtx=True )

   def importAndCommitUrlFieldSet( self, refreshCtx=False ):
      '''
      Used when committing a URL field-set when exiting the field-set config mode.
      '''
      assert self.editFieldSet.urlConfig.url

      # Only needed while using the same mount for URL field-sets. Once we remove
      # move to separate mounts, this can be removed.
      self.delAllContent()

      loadFieldSetSuccess = self.loadFieldSetFromUrl()
      # During startup-config, we do not support roll-back. If the URL failed to
      # load, there's no config to roll back to, so we keep a defined empty
      # field-set.
      if ( loadFieldSetSuccess or self.mode_.session_.startupConfig() or
           not self.urlRollbackSupported ):
         # On a failed startup, an empty field-set gets added.
         self.copyUrlFieldSetToSysdb()

      if loadFieldSetSuccess:
         self.checkHwStatus( refreshCtx=refreshCtx )

   def handleFieldSetOnSessionCommit( self ):
      '''
      Invoked by the config session commit handler.
      Returns an empty string if the field-set is successfully imported
      and programmed in hardware (via _checkStatus) or an error string.
      If it returns an error, the config session will rollback.
      '''
      successStr = ''

      if not self.urlFieldSetsOnOwnMount():
         # Maintain legacy behavior - if only using the CLI mount,
         # it's always _okay_ to fetch on commit of the mode.
         return self._checkStatus()

      if self.rollbackComplete:
         # This must be the second time the commit handler is called -
         # that means that the URL mount has already been rolled back.
         # There's no point on fetching the previous URL.
         assert not self.commitCfgSuccess
         return successStr

      if self.commitCfgSuccess:
         # The field-set was committed successfully, and now we are
         # calling this handler again to roll it back. This can only
         # happen if some other feature caused the session to roll back.
         self.completeConfigSessionRollbackOnUrlMount()
         return self._checkStatus()

      if not self.editFieldSet.urlConfig.url:
         # Field-sets without a URL do not need to fetch any contents.
         return self._checkStatus()

      if not self.prevUrlFieldSet or ( self.editFieldSet.urlConfig.url !=
         self.prevUrlFieldSet.urlConfig.url ):
         self.delAllContent()
         loadFieldSetSuccess = self.loadFieldSetFromUrl()
         if loadFieldSetSuccess:
            self.copyUrlFieldSetToSysdb()
            return self._checkStatus()
         else:
            # There is an import error - so the URL mount wasn't modified.
            # Mark this context as rollbackComplete so that if the commit
            # handler is invoked a second time, we do not modify the URL mount.
            self.rollbackComplete = True
            return 'Failed to import field-set from %s' % (
                      self.editFieldSet.urlConfig.url )
      else:
         # No need to fetch if the URL hasn't changed.
         return self._checkStatus()

   def copyUrlFieldSetToSysdb( self ):
      # Entity at the CLI mountpoint
      fieldSetCfg = self.fieldSetColl.get( self.fieldSetName )

      urlFieldSetsOnOwnMount = self.urlFieldSetsOnOwnMount()
      # Push subconfig to Sysdb.
      if not fieldSetCfg:
         fieldSetCfg = self._newFieldSet()

      if urlFieldSetsOnOwnMount:
         urlFieldSetCfg = self.urlFieldSetColl.get( self.fieldSetName )
         if not urlFieldSetCfg:
            urlFieldSetCfg = self._newUrlFieldSet()

      editSubCfg = fieldSetCfg.subConfig.get( self.editFieldSetVersion )

      if not editSubCfg:
         editSubCfg = fieldSetCfg.subConfig.newMember( self.fieldSetName,
                                                       self.editFieldSetVersion )
      # Update the timestamp in the scratchpad, before committing to Sysdb.
      self.editFieldSet.urlConfig = (
            UrlConfig( self.editFieldSet.urlConfig.url,
                       self.editFieldSet.urlConfig.vrfName,
                       Tac.now() ) )

      if urlFieldSetsOnOwnMount:
         assert urlFieldSetCfg
         urlFieldSetNewSubCfg = (
               urlFieldSetCfg.subConfig.newMember( self.fieldSetName,
                                                   self.editFieldSetVersion ) )
         self.copy( self.editFieldSet, urlFieldSetNewSubCfg )
         # Only copy source, urlConfig, and limit to the subConfig at CLI mountpoint.
         editSubCfg.source = self.editFieldSet.source
         editSubCfg.urlConfig = self.editFieldSet.urlConfig
         editSubCfg.limit = self.editFieldSet.limit

         # We ensure copy order is URL then CLI
         urlFieldSetCfg.currCfg = urlFieldSetNewSubCfg
         fieldSetCfg.currCfg = editSubCfg
      else:
         self.copy( self.editFieldSet, editSubCfg )
         # Switch pointer to new sub-config to complete the commit.
         fieldSetCfg.currCfg = editSubCfg

      if not self.urlRollbackSupported:
         # When URL config rollback is not supported, we always
         # switch to the new sub-config and if the URL failed to load, we
         # commit an empty field-set.
         if self.prevFieldSet:
            self.cleanupSubConfig( 'cli', self.prevFieldSet.version )
         if self.prevUrlFieldSet:
            self.cleanupSubConfig( 'url', self.prevUrlFieldSet.version )

   def validateAndResultsFieldSet( self ):
      '''
      Validate for an and-results field-set that there is no cycle
      formation by providing source as another and-results field-set
      '''
      intersectFieldSets = self.editFieldSet.intersectFieldSets
      AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
      af = AddressFamily.ipunknown
      if self.setType == "ipv4":
         af = AddressFamily.ipv4
      elif self.setType == "ipv6":
         af = AddressFamily.ipv6
      if self.fieldSetName in intersectFieldSets:
         errorStr = f"and-results field-set {self.fieldSetName} cannot be used "\
                    "inside itself"
         return errorStr
      for fsName in intersectFieldSets:
         if self.hasPrefixFieldSet( fsName, af ):
            fsConfig = self.fieldSetColl.get( fsName )
            if fsConfig.currCfg.source == "intersect":
               errorStr = f"and-results field-set {fsName} cannot be used inside"\
                           " another and-results field-set"
               return errorStr
      return ""

   def commit( self ):
      # commit to parent context
      if self.editFieldSet is None:
         return

      fieldSetCfg = self.fieldSetColl.get( self.fieldSetName )
      currCfg = fieldSetCfg.currCfg if fieldSetCfg else None

      # Check if field-set contents or source/urlConfig has changed.
      onlyLimitChanged = False
      currLimit = currCfg.limit if currCfg else FieldSetLimit.null
      editLimit = self.editFieldSet.limit
      self.editFieldSet.limit = currLimit
      if self.isFieldSetIdentical( currCfg, self.editFieldSet ):
         if currLimit != editLimit:
            # Only the limit changed
            self.editFieldSet.limit = editLimit
            onlyLimitChanged = True
         else:
            # No config change - nothing to update
            return
      # Set limit back
      self.editFieldSet.limit = editLimit

      # We do not re-fetch if only the limit has changed
      if ( self.editFieldSet.urlConfig.url and not
           ( self.urlFieldSetsOnOwnMount() and
             self.mode_.session_.inConfigSession() ) and not onlyLimitChanged ):
         # If we are in config session, checkHwStatus should register a callback
         # to fetch and commit the URL field-set on session commit.
         self.importAndCommitUrlFieldSet()
         return
      if self.editFieldSet.source == 'intersect':
         errorStr = self.validateAndResultsFieldSet()
         if errorStr:
            self.delIntersectFieldSets( self.editFieldSet.intersectFieldSets )
            self.mode_.addError( errorStr )
            return

      # Field-set has static contents that need to be committed.
      if not fieldSetCfg:
         fieldSetCfg = self._newFieldSet()
      editSubCfg = fieldSetCfg.subConfig.newMember( self.fieldSetName,
                                                    self.editFieldSetVersion )
      self.copy( self.editFieldSet, editSubCfg )

      # Switch pointer to new sub-config to complete the commit.
      fieldSetCfg.currCfg = editSubCfg

      self.checkHwStatus()

   def completeConfigSessionRollbackOnUrlMount( self ):
      '''
      Suppose we have a config-session:

      checkpointCfg -> commitCfg

      When a config session is committed, we check the hwStatus and if
      there is some failure associated with _this_ field-set, we roll
      back the configuration. But, suppose _this_ field-set got
      programmed successfully, but some other feature caused the config
      session to roll back. Then, the URL mount will still have the
      commitCfg instead of the checkpointCfg.

      This function will ONLY change the URL mount - and shouldn't edit
      the CLI mount because the config session
      rollback already set the CLI mount in the state at
      checkpointCfg. The reason we haven't rolled back the URL mount at
      that point is that on config sessions, the URL mount gets edited
      when you call the session commit handler.
      '''
      assert not self.rollbackComplete
      if self.prevUrlFieldSet:
         urlFsConfig = self.urlFieldSetColl.get( self.fieldSetName )
         if not urlFsConfig:
            # Rolling back a URL->CLI change. In this case, the URL
            # mount was cleaned up as part of _checkStatus succeeding
            # on the first commit handler call, so we need to
            # re-create it.
            assert self.editFieldSet.source != ContentsSource.url
            urlFsConfig = self._newUrlFieldSet()
         urlFsSubConfig = urlFsConfig.subConfig.newMember(
            self.fieldSetName, self.prevUrlFieldSet.version )
         self.copy( self.prevUrlFieldSet, urlFsSubConfig )
         urlFsConfig.currCfg = urlFsSubConfig
         # Rolling back URL1->URL2 change; remove URL2 sub-config
         # which was committed as part of commitCfg.
         del urlFsConfig.subConfig[ self.editFieldSet.version ]
      else:
         # Either there was no previous field-set or the prevFieldSet
         # source was not URL. Clean up the URL mount.
         if self.prevFieldSet:
            assert self.prevFieldSet.source != ContentsSource.url
         del self.urlFieldSetColl[ self.fieldSetName ]
      self.rollbackComplete = True

   def checkHwStatus( self, refreshCtx=False ):
      '''
      Verify whether the field-set was programmed successfully if we are
      not in config session. If we are in a config session, register a callback
      to verify the status when the session is committed.
      '''
      refreshAllCallbacks = self.mode_.session_.sessionData( 'refreshAllCallbacks',
                                                             None )
      if refreshAllCallbacks:
         refreshAllCallbacks.registerStatusCallback(
               ( self.mode_, self.fieldSetName, self.setType ),
               fieldSetConfigValidCallbacks )
         refreshAllCallbacks.registerRollbackCallback( self.rollback )
         if self.prevUrlFieldSet:
            refreshAllCallbacks.registerCleanupCallback(
                  self.cleanupSubConfig, self.prevUrlFieldSet.version )
         return

      if ( self.rollbackSupported and self.mode_.session_.inConfigSession()
           and not refreshCtx ):
         handler = (
            lambda mode, onSessionCommit=True: self.handleFieldSetOnSessionCommit()
         )
         CliSession.registerSessionOnCommitHandler(
               self.mode_.session_.entityManager,
               self.callbackKey,
               handler )
         return

      if self.rollbackSupported and ( refreshCtx or not (
            self.mode_.session_.inConfigSession() or
            self.mode_.session_.startupConfig() or
            self.mode_.session_.isStandalone() ) ):
         self._checkStatus()
      else:
         if self.prevFieldSet:
            self.cleanupSubConfig( 'cli', self.prevFieldSet.version )
         if self.prevUrlFieldSet:
            self.cleanupSubConfig( 'url', self.prevUrlFieldSet.version )

   def rollback( self ):
      if not self.rollbackSupported or self.rollbackComplete:
         return
      if self.prevFieldSet:
         fieldSetCfg = self.fieldSetColl[ self.fieldSetName ]
         currSource = fieldSetCfg.currCfg.source
         # Field-set is being updated; rollback to previous subCfg.
         fieldSetCfg.currCfg = self.prevFieldSet
         self.cleanupSubConfig( 'cli', self.editFieldSetVersion )
         if self.urlFieldSetsOnOwnMount():

            # Reverting from URL2 to URL1, or from non-URL source to URL.
            if self.prevFieldSet.source == ContentsSource.url:
               urlFieldSetCfg = self.urlFieldSetColl[ self.fieldSetName ]
               urlFieldSetCfg.currCfg = (
                     urlFieldSetCfg.subConfig[ self.prevUrlFieldSet.version ] )
               if self.editFieldSet.source == ContentsSource.url:
                  self.cleanupSubConfig( 'url', self.editFieldSet.version )
            elif ( currSource == ContentsSource.url and
                   self.prevFieldSet.source != ContentsSource.url ):
               # Reverting from URL to non-URL source
               del self.urlFieldSetColl[ self.fieldSetName ]
      elif self.prevUrlFieldSet:
         # No field-set in CLI proxy mount but there is a field-set in the urlMount.
         # This must be a config-replace and we need to rollback to
         # self.prevUrlFieldSet
         urlFsConfig = self.urlFieldSetColl[ self.fieldSetName ]
         urlFsConfig.currCfg = self.prevUrlFieldSet
         if self.editFieldSet.source == ContentsSource.url:
            self.cleanupSubConfig( 'url', self.editFieldSet.version )
      else:
         self.delFieldSet( self.fieldSetName, self.mode_, fromNoHandler=False )

      self.rollbackComplete = True

   def cleanupSubConfig( self, mount, version ):
      fieldSetCfg = None

      if mount == 'cli':
         fieldSetCfg = self.fieldSetColl.get( self.fieldSetName )
      elif mount == 'url':
         if self.urlFieldSetsOnOwnMount():
            fieldSetCfg = self.urlFieldSetColl.get( self.fieldSetName )
            if fieldSetCfg and fieldSetCfg.currCfg.version == version:
               # Don't remove the currCfg. This is possible if doing
               # a config replace to the same config - we do not
               # re-fetch the URL but we end up calling cleanupSubConfig
               # for the previous URL subconfig.
               fieldSetCfg = None

      if fieldSetCfg:
         del fieldSetCfg.subConfig[ version ]

   def _checkStatus( self ):
      '''
      Checks with registered features if the field-set commit caused
      an error. If an error occurs, either roll the config back if
      this is called from a CLI context or return the commit error if
      called from a config session context.
      '''
      successStr = ''
      for fieldSetConfigValidCallback in fieldSetConfigValidCallbacks:
         ( isValid, issues ) = fieldSetConfigValidCallback( self.mode_,
                                                            self.fieldSetName,
                                                            self.setType )

         issuesStr = issues.get( 'error', 'unknown' ) if issues else 'unknown'

         if isValid is False:
            self.mode_.addError( fieldSetCommitIssueStr( self.setType,
                                                         self.fieldSetName,
                                                         issuesStr ) )
            self.rollback()
            return issuesStr
         else:
            if issues:
               warning = issues.get( 'warning' )
               if warning:
                  self.mode_.addWarning(
                     fieldSetCommitIssueStr( self.setType,
                                             self.fieldSetName,
                                             warning ) )

      if self.rollbackComplete:
         # The first time the commit handler is called, we have completed the
         # rollback. If the same commit handler is called again, we do not
         # need to cleanup because we are solely pushing in the config in
         # the CLI mount back into Sysdb.
         return successStr

      # No issues found - safe to remove previous subConfig.
      if self.prevUrlFieldSet:
         self.cleanupSubConfig( 'url', self.prevUrlFieldSet.version )
      if self.prevFieldSet:
         self.cleanupSubConfig( 'cli', self.prevFieldSet.version )
         if ( self.prevFieldSet.source == ContentsSource.url and
              self.editFieldSet.source != ContentsSource.url and
              self.urlFieldSetsOnOwnMount() ):
            del self.urlFieldSetColl[ self.fieldSetName ]

      # Mark the field-set as committed successfully. We might still
      # need to rollback in a config-session if some other feature
      # causes the session to roll back even when the field-set was
      # programmed successfully.
      self.commitCfgSuccess = True
      return successStr

   def abort( self ):
      raise NotImplementedError

   def delAllContent( self ):
      raise NotImplementedError

   def isUsingUrl( self ):
      if self.urlFieldSetsOnOwnMount():
         return self.editFieldSet.source == ContentsSource.url
      return ( self.editFieldSet.source == ContentsSource.cli and
               self.editFieldSet.urlConfig )

   def isUsingCli( self ):
      if self.urlFieldSetsOnOwnMount():
         return self.editFieldSet.source == ContentsSource.cli
      return ( self.editFieldSet.source == ContentsSource.cli and
               not self.editFieldSet.urlConfig )

   def isUsingBgp( self ):
      return ( self.editFieldSet.source == ContentsSource.bgp and
               not self.editFieldSet.urlConfig )

   def isMatchingSource( self, source, url ):
      return ( self.editFieldSet.source == source and
               self.editFieldSet.urlConfig.url == url )

   def canUpdateFieldSet( self, isUser=True ):
      # Only in CLI mode without URL can a user update field-set OR
      # In CLI mode with URL set can ConfigAgent update field-set
      return self.isUsingCli() or ( self.isUsingUrl() and not isUser )

   def validateSource( self, source, url, field ):
      if self.isMatchingSource( source, url ):
         # Does not take vrf into account because if the URL matches vrf can change
         return ''
      if self.isUsingCli() and not self.isFieldSetEmpty():
         return 'Cannot change source while %s are defined.' % field
      if self.editFieldSet.limit != FieldSetLimit.null:
         if url or source == 'cli':
            return ''
         errorMsg = \
               "Error: Cannot configure 'source %s' with 'limit entries' configured"
         if source == 'bgp':
            return errorMsg % "bgp"
         elif source == 'intersect':
            return errorMsg % "and-results"
         else:
            return errorMsg % "unknown"
      if source == 'intersect':
         usedInAndResultsFieldSet = self.getUsedInAndResultsFieldSet()
         if self.fieldSetName in usedInAndResultsFieldSet:
            errorMsg = "Error: Cannot configure and-results field-set when "\
                       f"{self.fieldSetName} is used inside another and-results "\
                       "field-set(s)"
            return errorMsg
      return ''

   def updateSource( self, source ):
      if source != self.editFieldSet.source:
         self.delAllContent()
      self.editFieldSet.source = source

   def setUrlConfig( self, url, vrfName ):
      url = str( url )
      if ( url and
           ( self.editFieldSet.urlConfig.url != url or
             self.editFieldSet.urlConfig.vrfName != vrfName ) ):
         # Only update the URL if it differs
         self.editFieldSet.urlConfig = UrlConfig( url, vrfName, Tac.now() )
         if not self.urlFieldSetsOnOwnMount():
            # editFieldSet will contain prev URL contents, be sure to clear
            self.delAllContent()

   def removeUrlConfig( self ):
      if self.prevFieldSet and self.prevFieldSet.urlConfig:
         self.delAllContent()
      self.editFieldSet.urlConfig = UrlConfig()

   def setLimitEntries( self, limit ):
      if limit is None:
         # Removing limit
         self.editFieldSet.limit = FieldSetLimit.null
         return
      self.editFieldSet.limit = limit

   def limitEntriesValid( self ):
      if not self.rollbackSupported or not self.limitSupported:
         return ( "Error: 'limit entries' not supported", "" )

      if self.editFieldSet.limit == FieldSetLimit.null:
         # No limit is configured
         return ( "", "" )
      if self.isUsingBgp():
         errorMsg = ( "Error: Cannot configure 'limit entries' "
                      "with 'source bgp' configured" )
         warningMsg = ""
         return ( errorMsg, warningMsg )

      warningMsg = "Warning: Field set has %d entries exceeding limit of %d"

      if self.prevUrlFieldSet and \
         self.editFieldSet.source == ContentsSource.url and \
         self.prevUrlFieldSet.urlConfig.url == self.editFieldSet.urlConfig.url:
         # Limit was added to a URL based field-set after we have already
         # successfully fetched content and the URL contained more entries then
         # the limit we are currently setting
         currSize = self.prevUrlFieldSet.size()
      else:
         currSize = self.editFieldSet.size()

      if currSize > self.editFieldSet.limit:
         return ( "", warningMsg % ( currSize, self.editFieldSet.limit ) )
      return ( "", "" )

   def addIntersectFieldSets( self, fieldSetNames ):
      for fsName in fieldSetNames:
         self.editFieldSet.intersectFieldSets.add( fsName )
         usedInAndResultsFieldSet = self.getUsedInAndResultsFieldSet()
         if fsName in usedInAndResultsFieldSet:
            usedInAndResultsFieldSet[ fsName ] += 1
         else:
            usedInAndResultsFieldSet[ fsName ] = 1


   def delIntersectFieldSets( self, fieldSetNames ):
      for fsName in fieldSetNames:
         self.editFieldSet.intersectFieldSets.remove( fsName )
         usedInAndResultsFieldSet = self.getUsedInAndResultsFieldSet()
         if fsName in usedInAndResultsFieldSet:
            usedInAndResultsFieldSet[ fsName ] -= 1
            count = usedInAndResultsFieldSet[ fsName ]
            if not count:
               del usedInAndResultsFieldSet[ fsName ]
         if not self.editFieldSet.intersectFieldSets:
            self.editFieldSet.source = ContentsSource.cli

   def getUsedInAndResultsFieldSet( self ):
      usedInAndResultsFieldSet = self.fieldSetConfig.ipPrefixFsUsedInAndResult
      if self.setType == "ipv6":
         usedInAndResultsFieldSet = self.fieldSetConfig.\
                                       ip6PrefixFsUsedInAndResult
      return usedInAndResultsFieldSet

class IntegerFieldSetContext( FieldSetBaseContext ):
   limitSupported = True
   limitExpanded = True
   def __init__( self, fieldSetIntegerName, fieldSetConfig, childMode=None,
                 featureName=None,
                 rollbackSupported=False,
                 urlFieldSetConfig=None,
                 urlLoadInitialMode=None,
                 urlRollbackSupported=False,
                 urlImportFailSyslogHandle=None,
                 urlImportSuccessSyslogHandle=None ):
      super().__init__(
         fieldSetIntegerName, fieldSetConfig, childMode,
         featureName=featureName,
         rollbackSupported=rollbackSupported,
         urlFieldSetConfig=urlFieldSetConfig,
         urlLoadInitialMode=urlLoadInitialMode,
         urlRollbackSupported=urlRollbackSupported,
         urlImportFailSyslogHandle=urlImportFailSyslogHandle,
         urlImportSuccessSyslogHandle=urlImportSuccessSyslogHandle )
      self.fieldSetColl = self.fieldSetConfig.fieldSetInteger
      if self.urlFieldSetsOnOwnMount():
         self.urlFieldSetColl = self.urlFieldSetConfig.fieldSetInteger
      self.setType = 'integer'

   def copyEditFieldSet( self ):
      self.prevFieldSet = self.fieldSetColl[ self.fieldSetName ].currCfg
      self.cacheInitialUrlFieldSet()

      self.editFieldSet = Tac.newInstance(
         'Classification::FieldSetIntegerSubConfig', self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.copy( self.prevFieldSet, self.editFieldSet )

   def newEditFieldSet( self ):
      self.editFieldSet = Tac.newInstance(
         'Classification::FieldSetIntegerSubConfig', self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.editFieldSet.source = ContentsSource.cli
      # Handles config replace as URL field-sets, if not config mounted, might
      # still be present as we replace to the target configuration.
      self.cacheInitialUrlFieldSet()

   def updateFieldSet( self, data, add, **kwargs ):
      currentSet = numericalRangeToSet( self.editFieldSet.integerRange )
      updatedSet = set()
      if add:
         updatedSet = currentSet | data
      else:
         updatedSet = currentSet - data
      if add and self.editFieldSet.limit != FieldSetLimit.null:
         if currentSet == data:
            # All ranges overlap with existing ranges
            return ""
         if len( updatedSet ) > self.editFieldSet.limit:
            return ( "Error: Limit of %d entries expanded exceeded" %
                     self.editFieldSet.limit )

      self.editFieldSet.integerRange.clear()
      newRangeList = rangeSetToNumericalRange( updatedSet,
                                               "Classification::NumericalRange" )
      for aRange in newRangeList:
         self.editFieldSet.integerRange.add( aRange )
      return ""

   def isFieldSetEmpty( self, ):
      return not self.editFieldSet.integerRange

   def _delFieldSetFromConfigMount( self ):
      # Used internally to remove a field-set from the config-mount used by the
      # current CLI context (i.e., Sysdb, or config session).
      # Needs to reference `self.fieldSetConfig` to properly use the
      # config-mount proxy.
      del self.fieldSetConfig.fieldSetInteger[ self.fieldSetName ]

   def delAllContent( self ):
      self.editFieldSet.integerRange.clear()

   def isFieldSetIdentical( self, left, right ):
      if not left and not right:
         return True
      if not left or not right:
         return False
      return left.isEqual( right )

   def copy( self, src, dst ):
      if not src or not dst:
         return
      if self.isFieldSetIdentical( src, dst ):
         return
      dst.copy( src )

   def _newFieldSet( self ):
      return self.fieldSetColl.newMember( self.fieldSetName )

   def _newUrlFieldSet( self ):
      return self.urlFieldSetColl.newMember( self.fieldSetName )

   def abort( self ):
      self.fieldSetName = None
      self.editFieldSet = None
      self.prevFieldSet = None
      self.editFieldSetVersion = None

class VlanFieldSetContext( FieldSetBaseContext ):
   limitSupported = True
   limitExpanded = True
   def __init__( self, fieldSetVlanName, fieldSetConfig, childMode=None,
                 featureName=None,
                 rollbackSupported=False,
                 urlFieldSetConfig=None,
                 urlLoadInitialMode=None,
                 urlRollbackSupported=False,
                 urlImportFailSyslogHandle=None,
                 urlImportSuccessSyslogHandle=None ):
      super().__init__(
         fieldSetVlanName, fieldSetConfig, childMode,
         featureName=featureName,
         rollbackSupported=rollbackSupported,
         urlFieldSetConfig=urlFieldSetConfig,
         urlLoadInitialMode=urlLoadInitialMode,
         urlRollbackSupported=urlRollbackSupported,
         urlImportFailSyslogHandle=urlImportFailSyslogHandle,
         urlImportSuccessSyslogHandle=urlImportSuccessSyslogHandle )
      self.fieldSetColl = self.fieldSetConfig.fieldSetVlan
      if self.urlFieldSetsOnOwnMount():
         self.urlFieldSetColl = self.urlFieldSetConfig.fieldSetVlan
      self.setType = 'vlan'

   def copyEditFieldSet( self ):
      self.prevFieldSet = self.fieldSetColl[ self.fieldSetName ].currCfg
      self.cacheInitialUrlFieldSet()

      self.editFieldSet = Tac.newInstance( 'Classification::FieldSetVlanSubConfig',
                                           self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.copy( self.prevFieldSet, self.editFieldSet )

   def newEditFieldSet( self ):
      self.editFieldSet = Tac.newInstance( 'Classification::FieldSetVlanSubConfig',
                                           self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.editFieldSet.source = ContentsSource.cli
      # Handles config replace as URL field-sets, if not config mounted, might
      # still be present as we replace to the target configuration.
      self.cacheInitialUrlFieldSet()

   def updateFieldSet( self, data, add, **kwargs ):
      allVlans = kwargs.get( 'allVlans', False )
      if allVlans:
         allVlanRange = Tac.Value( 'Classification::VlanRange', 1,
                                   appConstants.maxVlan - 1 )
         if not add:
            self.editFieldSet.vlans.clear()
            return ""
         if allVlanRange.rangeSize() > self.editFieldSet.limit:
            return ( "Error: Limit of %d entries expanded exceeded" %
                     self.editFieldSet.limit )
         self.editFieldSet.vlans.clear()
         self.editFieldSet.vlans.add( allVlanRange )
         return ""

      currentSet = numericalRangeToSet( list( self.editFieldSet.vlans ) )
      updatedSet = set()
      if add:
         updatedSet = currentSet | data
      else:
         updatedSet = currentSet - data
      if add and self.editFieldSet.limit != FieldSetLimit.null:
         if currentSet == data:
            # All ranges overlap with existing ranges
            return ""
         if len( updatedSet ) > self.editFieldSet.limit:
            return ( "Error: Limit of %d entries expanded exceeded" %
                     self.editFieldSet.limit )
      self.editFieldSet.vlans.clear()

      newRangeList = rangeSetToNumericalRange( updatedSet,
                                               "Classification::VlanRange" )
      for aRange in newRangeList:
         self.editFieldSet.vlans.add( aRange )
      return ""

   def isFieldSetEmpty( self, ):
      return not self.editFieldSet.vlans

   def _delFieldSetFromConfigMount( self ):
      # Used internally to remove a field-set from the config-mount used by the
      # current CLI context (i.e., Sysdb, or config session).
      # Needs to reference `self.fieldSetConfig` to properly use the
      # config-mount proxy.
      del self.fieldSetConfig.fieldSetVlan[ self.fieldSetName ]

   def delAllContent( self ):
      self.editFieldSet.vlans.clear()

   def isFieldSetIdentical( self, left, right ):
      if not left and not right:
         return True
      if not left or not right:
         return False

      return left.isEqual( right )

   def copy( self, src, dst ):
      if not src or not dst:
         return

      if self.isFieldSetIdentical( src, dst ):
         return

      dst.copy( src )

   def _newFieldSet( self ):
      return self.fieldSetColl.newMember( self.fieldSetName )

   def _newUrlFieldSet( self ):
      return self.urlFieldSetColl.newMember( self.fieldSetName )

   def abort( self ):
      self.fieldSetName = None
      self.editFieldSet = None
      self.prevFieldSet = None
      self.editFieldSetVersion = None

class L4PortFieldSetContext( FieldSetBaseContext ):
   limitSupported = True
   limitExpanded = True
   def __init__( self, fieldSetL4PortName, fieldSetConfig, childMode=None,
                 featureName=None,
                 rollbackSupported=False,
                 urlFieldSetConfig=None,
                 urlLoadInitialMode=None,
                 urlRollbackSupported=False,
                 urlImportFailSyslogHandle=None,
                 urlImportSuccessSyslogHandle=None ):
      super().__init__(
            fieldSetL4PortName,
            fieldSetConfig,
            childMode,
            featureName=featureName,
            rollbackSupported=rollbackSupported,
            urlFieldSetConfig=urlFieldSetConfig,
            urlLoadInitialMode=urlLoadInitialMode,
            urlRollbackSupported=urlRollbackSupported,
            urlImportFailSyslogHandle=urlImportFailSyslogHandle,
            urlImportSuccessSyslogHandle=urlImportSuccessSyslogHandle )
      self.fieldSetColl = self.fieldSetConfig.fieldSetL4Port
      if self.urlFieldSetsOnOwnMount():
         self.urlFieldSetColl = self.urlFieldSetConfig.fieldSetL4Port
      self.setType = 'l4-port'

   def copyEditFieldSet( self ):
      self.prevFieldSet = self.fieldSetColl[ self.fieldSetName ].currCfg
      self.cacheInitialUrlFieldSet()
      self.editFieldSet = Tac.newInstance( 'Classification::FieldSetL4PortSubConfig',
                                           self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.copy( self.prevFieldSet, self.editFieldSet )

   def newEditFieldSet( self ):
      self.editFieldSet = Tac.newInstance( 'Classification::FieldSetL4PortSubConfig',
                                           self.fieldSetName, UniqueId() )
      self.editFieldSet.source = ContentsSource.cli
      self.editFieldSetVersion = self.editFieldSet.version
      # Handles config replace as URL field-sets, if not config mounted, might
      # still be present as we replace to the target configuration.
      self.cacheInitialUrlFieldSet()

   def updateFieldSet( self, data, add, **kwargs ):
      allPorts = kwargs.get( 'allPorts', False )
      if allPorts:
         allPortRange = Tac.Value( 'Classification::PortRange', 0,
                                   appConstants.maxL4Port )
         if not add:
            self.editFieldSet.ports.clear()
            return ""
         if allPortRange.rangeSize() > self.editFieldSet.limit:
            return ( "Error: Limit of %d entries expanded exceeded" %
                     self.editFieldSet.limit )
         self.editFieldSet.ports.clear()
         self.editFieldSet.ports.add( allPortRange )
         return ""

      currentSet = numericalRangeToSet( list( self.editFieldSet.ports ) )
      updatedSet = set()
      if add:
         updatedSet = currentSet | data
      else:
         updatedSet = currentSet - data
      if add and self.editFieldSet.limit != FieldSetLimit.null:
         if currentSet == data:
            # All ranges overlap with existing ranges
            return ""
         if len( updatedSet ) > self.editFieldSet.limit:
            return ( "Error: Limit of %d entries expanded exceeded" %
                     self.editFieldSet.limit )
      self.editFieldSet.ports.clear()

      newRangeList = rangeSetToNumericalRange( updatedSet,
                                               "Classification::PortRange" )
      for aRange in newRangeList:
         self.editFieldSet.ports.add( aRange )
      return ""

   def _delFieldSetFromConfigMount( self ):
      # Used internally to remove a field-set from the config-mount used by the
      # current CLI context (i.e., Sysdb, or config session).
      # Needs to reference `self.fieldSetConfig` to properly use the
      # config-mount proxy.
      del self.fieldSetConfig.fieldSetL4Port[ self.fieldSetName ]

   def isFieldSetIdentical( self, left, right ):
      if not left and not right:
         return True
      if not left or not right:
         return False

      return left.isEqual( right )

   def isFieldSetEmpty( self ):
      return not self.editFieldSet.ports

   def delAllContent( self ):
      self.editFieldSet.ports.clear()

   def copy( self, src, dst ):
      if not src or not dst:
         return

      if self.isFieldSetIdentical( src, dst ):
         return

      dst.copy( src )

   def _newFieldSet( self ):
      return self.fieldSetColl.newMember( self.fieldSetName )

   def _newUrlFieldSet( self ):
      return self.urlFieldSetColl.newMember( self.fieldSetName )

   def abort( self ):
      self.fieldSetName = None
      self.editFieldSet = None
      self.prevFieldSet = None
      self.editFieldSetVersion = None

class ServiceFieldSetContext( FieldSetBaseContext, ProtocolContextMixin ):
   def __init__( self, fieldSetServiceName, fieldSetConfig, childMode=None,
                 featureName=None,
                 rollbackSupported=False,
                 urlFieldSetConfig=None,
                 urlLoadInitialMode=None,
                 urlRollbackSupported=False,
                 urlImportFailSyslogHandle=None,
                 urlImportSuccessSyslogHandle=None ):
      super().__init__(
            fieldSetServiceName,
            fieldSetConfig,
            childMode,
            featureName=featureName,
            rollbackSupported=rollbackSupported,
            urlFieldSetConfig=urlFieldSetConfig,
            urlLoadInitialMode=urlLoadInitialMode,
            urlRollbackSupported=urlRollbackSupported,
            urlImportFailSyslogHandle=urlImportFailSyslogHandle,
            urlImportSuccessSyslogHandle=urlImportSuccessSyslogHandle )
      self.fieldSetColl = self.fieldSetConfig.fieldSetService
      if self.urlFieldSetsOnOwnMount():
         self.urlFieldSetColl = self.urlFieldSetConfig.fieldSetService
      self.setType = 'service'

   def copyEditFieldSet( self ):
      self.prevFieldSet = self.fieldSetColl[ self.fieldSetName ].currCfg
      self.cacheInitialUrlFieldSet()
      self.editFieldSet = Tac.newInstance(
                              'Classification::FieldSetServiceSubConfig',
                              self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.copy( self.prevFieldSet, self.editFieldSet )

   def newEditFieldSet( self ):
      self.editFieldSet = Tac.newInstance(
                              'Classification::FieldSetServiceSubConfig',
                              self.fieldSetName, UniqueId() )
      self.editFieldSet.source = ContentsSource.cli
      self.editFieldSetVersion = self.editFieldSet.version
      # Handles config replace as URL field-sets, if not config mounted, might
      # still be present as we replace to the target configuration.
      self.cacheInitialUrlFieldSet()

   # updateFieldSet() is an abstract method inherited from FieldSetBaseContext. while
   # configuring a service we are using the ProtocolListServiceFieldSetCmd command
   # class, so a service update would use api like updateOredProtocolAttr() and
   # addOredPortFieldSetAttr() instead of updateFieldSet() as used by other existing
   # field-sets. since we are not using this api have defined it as a no-op.
   def updateFieldSet( self, data, add, **kwargs ):
      pass

   def isFieldSetIdentical( self, left, right ):
      if not left and not right:
         return True
      if not left or not right:
         return False

      return left.isEqual( right )

   def _newFieldSet( self ):
      return self.fieldSetColl.newMember( self.fieldSetName )

   def _newUrlFieldSet( self ):
      return self.urlFieldSetColl.newMember( self.fieldSetName )

   def isFieldSetEmpty( self ):
      return not self.editFieldSet.proto

   def copy( self, src, dst ):
      if not src or not dst:
         return

      if self.isFieldSetIdentical( src, dst ):
         return

      dst.copy( src )

   # In a ServiceFieldSetContext the api clearServiceFieldSet has to be a no-op.
   # The reason being that ProtocolMixin command handler calls clearServiceFieldSet()
   # to clear any serviceFieldSets configured when configuring a raw protocol match
   # statement in the MatchRuleBaseContext (raw protocol and service field-set
   # defined protocol cannot coexist). now since configuring a service field-set
   # also re-uses ProtocolMixin we implement implement it as a no-op as in that
   # context this api doesnt apply.
   def clearServiceFieldSet( self ):
      return

   def _delFieldSetFromConfigMount( self ):
      # Used internally to remove a field-set from the config-mount used by the
      # current CLI context (i.e., Sysdb, or config session).
      # Needs to reference `self.fieldSetConfig` to properly use the
      # config-mount proxy.
      del self.fieldSetConfig.fieldSetService[ self.fieldSetName ]

   def delAllContent( self ):
      self.editFieldSet.proto.clear()

   def isValidConfig( self, conflictOther=None ):
      return True

   def updateRangeAttr( self, attrName, rangeSet, rangeType, add ):
      acceptedAttr = [ 'sport', 'dport', 'proto' ]
      if attrName not in acceptedAttr:
         return
      currentAttrList = getattr( self.editFieldSet, attrName )
      newRangeList = addOrDeleteRange( currentAttrList,
                                       rangeSet, rangeType, add )
      # Cannot delete all members and re-add for protocols because protocols have
      # additional protoFields
      for currAttr in currentAttrList:
         if currAttr not in newRangeList:
            del currentAttrList[ currAttr ]
      for aRange in newRangeList:
         if aRange not in currentAttrList:
            if attrName == 'proto':
               currentAttrList.newMember( aRange )
            else:
               currentAttrList.add( aRange )

   def getProto( self ):
      return self.editFieldSet.proto

   def abort( self ):
      self.fieldSetName = None
      self.editFieldSet = None
      self.prevFieldSet = None
      self.editFieldSetVersion = None

class IpPrefixFieldSetContext( FieldSetBaseContext ):
   limitSupported = True
   def __init__( self, fieldSetIpPrefixName, fieldSetConfig, setType='ipv4',
                 childMode=None,
                 featureName=None,
                 rollbackSupported=False,
                 urlFieldSetConfig=None,
                 urlLoadInitialMode=None,
                 urlRollbackSupported=False,
                 urlImportFailSyslogHandle=None,
                 urlImportSuccessSyslogHandle=None ):
      super().__init__(
         fieldSetIpPrefixName,
         fieldSetConfig,
         childMode,
         featureName=featureName,
         rollbackSupported=rollbackSupported,
         urlFieldSetConfig=urlFieldSetConfig,
         urlLoadInitialMode=urlLoadInitialMode,
         urlRollbackSupported=urlRollbackSupported,
         urlImportFailSyslogHandle=urlImportFailSyslogHandle,
         urlImportSuccessSyslogHandle=urlImportSuccessSyslogHandle )
      if setType == "ipv4":
         self.fieldSetColl = self.fieldSetConfig.fieldSetIpPrefix
      else:
         self.fieldSetColl = self.fieldSetConfig.fieldSetIpv6Prefix

      if self.urlFieldSetsOnOwnMount():
         if setType == "ipv4":
            self.urlFieldSetColl = self.urlFieldSetConfig.fieldSetIpPrefix
         else:
            self.urlFieldSetColl = self.urlFieldSetConfig.fieldSetIpv6Prefix
      self.setType = setType

   def copyEditFieldSet( self ):
      self.prevFieldSet = self.fieldSetColl[ self.fieldSetName ].currCfg
      self.cacheInitialUrlFieldSet()

      self.editFieldSet = (
         Tac.newInstance( 'Classification::FieldSetIpPrefixSubConfig',
                          self.fieldSetName, UniqueId() ) )
      self.editFieldSetVersion = self.editFieldSet.version
      self.copy( self.prevFieldSet, self.editFieldSet )

   def newEditFieldSet( self ):
      self.editFieldSet = (
            Tac.newInstance( 'Classification::FieldSetIpPrefixSubConfig',
                             self.fieldSetName, UniqueId() ) )
      self.editFieldSet.source = ContentsSource.cli
      self.editFieldSetVersion = self.editFieldSet.version
      # Handles config replace as URL field-sets, if not config mounted, might
      # still be present as we replace to the target configuration.
      self.cacheInitialUrlFieldSet()

   def updateFieldSet( self, data, add=True, **kwargs ):
      prefixes = [ IpGenPrefix( str( d ) ) for d in data ]
      if kwargs.get( 'updateExcept' ):
         prefixSet = self.editFieldSet.exceptPrefix
      else:
         prefixSet = self.editFieldSet.prefixes
         if add and self.editFieldSet.limit != FieldSetLimit.null:
            currNumPrefixes = len( prefixSet )
            # We do not have to check prevUrlFieldSet here because this is only
            # called when we are updating field-sets from CLI which implies source
            # is static and we are using self.editFieldSet
            pendingCount = \
               len( [ p for p in prefixes if p not in prefixSet ] )
            if pendingCount == 0:
               # All prefixes overlap with existing prefixes
               return ""
            finalCount = currNumPrefixes + pendingCount
            if finalCount > self.editFieldSet.limit:
               return "Error: Limit of %d entries exceeded" % self.editFieldSet.limit

      op = prefixSet.add if add else prefixSet.remove
      for prefix in prefixes:
         op( prefix )
      return ""

   def _delFieldSetFromConfigMount( self ):
      # Used internally to remove a field-set from the config-mount used by the
      # current CLI context (i.e., Sysdb, or config session).
      # Needs to reference `self.fieldSetConfig` to properly use the
      # config-mount proxy.
      if self.setType == "ipv4":
         del self.fieldSetConfig.fieldSetIpPrefix[ self.fieldSetName ]
      else:
         del self.fieldSetConfig.fieldSetIpv6Prefix[ self.fieldSetName ]

   def isFieldSetEmpty( self ):
      return not self.editFieldSet.prefixes and not self.editFieldSet.exceptPrefix

   def delAllContent( self ):
      self.editFieldSet.prefixes.clear()
      self.editFieldSet.exceptPrefix.clear()
      self.delIntersectFieldSets( self.editFieldSet.intersectFieldSets )

   def isFieldSetIdentical( self, left, right ):
      if not left and not right:
         return True
      if not left or not right:
         return False

      return left.isEqual( right )

   def copy( self, src, dst ):
      if not src or not dst:
         return

      if self.isFieldSetIdentical( src, dst ):
         return

      dst.copy( src )

   def _newFieldSet( self ):
      return self.fieldSetColl.newMember( self.fieldSetName,
                                          self.setType )

   def _newUrlFieldSet( self ):
      assert self.urlFieldSetsOnOwnMount()
      return self.urlFieldSetColl.newMember( self.fieldSetName,
                                             self.setType )

   def abort( self ):
      self.fieldSetName = None
      self.editFieldSet = None
      self.prevFieldSet = None
      self.editFieldSetVersion = None

class MacAddrFieldSetContext( FieldSetBaseContext ):
   def __init__( self, fieldSetMacAddrName, fieldSetConfig, childMode=None,
                 featureName=None,
                 rollbackSupported=False,
                 urlFieldSetConfig=None,
                 urlLoadInitialMode=None,
                 urlRollbackSupported=False,
                 urlImportFailSyslogHandle=None,
                 urlImportSuccessSyslogHandle=None ):
      super().__init__(
         fieldSetMacAddrName, fieldSetConfig, childMode,
         featureName=featureName,
         rollbackSupported=rollbackSupported,
         urlFieldSetConfig=urlFieldSetConfig,
         urlLoadInitialMode=urlLoadInitialMode,
         urlRollbackSupported=urlRollbackSupported,
         urlImportFailSyslogHandle=urlImportFailSyslogHandle,
         urlImportSuccessSyslogHandle=urlImportSuccessSyslogHandle )
      self.fieldSetColl = self.fieldSetConfig.fieldSetMacAddr
      if self.urlFieldSetsOnOwnMount():
         self.urlFieldSetColl = self.urlFieldSetConfig.fieldSetMacAddr
      self.setType = 'mac'

   def copyEditFieldSet( self ):
      self.prevFieldSet = self.fieldSetColl[ self.fieldSetName ].currCfg
      self.cacheInitialUrlFieldSet()

      self.editFieldSet = Tac.newInstance(
            'Classification::FieldSetMacAddrSubConfig',
            self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.copy( self.prevFieldSet, self.editFieldSet )

   def newEditFieldSet( self ):
      self.editFieldSet = Tac.newInstance(
            'Classification::FieldSetMacAddrSubConfig',
            self.fieldSetName, UniqueId() )
      self.editFieldSetVersion = self.editFieldSet.version
      self.editFieldSet.source = ContentsSource.cli
      # Handles config replace as URL field-sets, if not config mounted, might
      # still be present as we replace to the target configuration.
      self.cacheInitialUrlFieldSet()

   def updateFieldSet( self, data, add, **kwargs ):
      macSet = self.editFieldSet.macAddrs
      op = macSet.add if add else macSet.remove
      for datum in data:
         macEntry.stringValue = str( datum )
         op( macEntry )
      return ""

   def isFieldSetEmpty( self, ):
      return not self.editFieldSet.macAddrs

   def _delFieldSetFromConfigMount( self ):
      # Used internally to remove a field-set from the config-mount used by the
      # current CLI context (i.e., Sysdb, or config session).
      # Needs to reference `self.fieldSetConfig` to properly use the
      # config-mount proxy.
      del self.fieldSetConfig.fieldSetMacAddr[ self.fieldSetName ]

   def delAllContent( self ):
      self.editFieldSet.macAddrs.clear()

   def isFieldSetIdentical( self, left, right ):
      if not left and not right:
         return True
      if not left or not right:
         return False

      return left.isEqual( right )

   def copy( self, src, dst ):
      if not src or not dst:
         return

      if self.isFieldSetIdentical( src, dst ):
         return

      dst.copy( src )

   def _newFieldSet( self ):
      return self.fieldSetColl.newMember( self.fieldSetName )

   def _newUrlFieldSet( self ):
      return self.urlFieldSetColl.newMember( self.fieldSetName )

   def abort( self ):
      self.fieldSetName = None
      self.editFieldSet = None
      self.prevFieldSet = None
      self.editFieldSetVersion = None
