#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac
import Tracing
import BasicCliUtil
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import CliPlugin.MacAddr as MacAddr # pylint: disable=consider-using-from-import
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.ClassificationCliContextLib import (
   IpPrefixFieldSetContext,
   L4PortFieldSetContext,
   ServiceFieldSetContext,
   VlanFieldSetContext,
   IntegerFieldSetContext,
   MacAddrFieldSetContext, )
from AclLib import tcpServiceByName, udpServiceByName
from Arnet import EthAddr
from MultiRangeRule import MultiRangeMatcher, NumberFormat
from CliMode.Classification import ( AppConfigModeIpv4, AppConfigModeL4,
                                     AppProfileConfigMode, CategoryConfigMode )
import CliToken.Refresh
from ClassificationLib import ( numericalRangeToSet, rangeSetToNumericalRange,
                                numericalRangeToRangeString, extraIpv4Protocols,
                                genericIpProtocols, tcpUdpProtocols, getKeywordMap,
                                getProtectedFieldSetNames, icmpV4Types, icmpV6Types,
                                icmpV4Codes, icmpV6Codes, icmpV4TypeWithValidCodes,
                                icmpV6TypeWithValidCodes, fieldSetTypeToStr,
                                listUrlLocalSchemes, listUrlNetworkSchemes, )
from socket import IPPROTO_TCP, IPPROTO_ICMP, IPPROTO_ICMPV6
from Url import UrlMatcher

__defaultTraceHandle__ = Tracing.Handle( 'ClassificationCli' )
t0 = Tracing.trace0

tacFieldConflict = Tac.Type( 'Classification::FieldConflict' )
conflictNeighborProtocol = tacFieldConflict.conflictNeighborProtocol
conflictMatchL4Protocol = tacFieldConflict.conflictMatchL4Protocol
conflictMatchAllFragments = tacFieldConflict.conflictMatchAllFragments
conflictFragmentOffset = tacFieldConflict.conflictFragmentOffset
conflictOther = tacFieldConflict.conflictOther
FragmentType = Tac.Type( 'Classification::FragmentType' )
ContentsSource = Tac.Type( 'Classification::ContentsSource' )
matchAll = FragmentType.matchAll
matchNone = FragmentType.matchNone
matchOffset = FragmentType.matchOffset
UniqueId = Tac.Type( 'Ark::UniqueId' )
IpGenPrefix = Tac.Type( 'Arnet::IpGenPrefix' )
EthAddrWithMask = Tac.Type( 'Arnet::EthAddrWithMask' )
VlanRange = Tac.Type( 'Classification::VlanRange' )
AppTransport = Tac.Type( "Classification::AppTransport" )

configConflictMsg = (
      "The '%s' subcommand is not "
      "supported when either the 'protocol neighbors' or "
      "'protocol bgp' subcommand are configured" )
invalidPortConflictMsg = (
      "The '%s' subcommand is not supported if protocols other than "
      "'{tcp|udp|tcp udp}' are configured" )
invalidProtocolConflictMsg = (
      "The 'protocol' subcommand only supports 'tcp' or 'udp' if"
      " '{source|destination} port' is configured" )
invalidL4PortConflictMsg = (
      "The '{source|destination} port' command is not supported when "
      "'fragment' or 'fragment offset' command is configured" )
invalidFragmentConflictMsg = (
      "The 'fragment' command is not supported when 'source port' or "
      "'destination port' command is configured" )
invalidFragOffsetConflictMsg = (
      "The 'fragment offset' command is not supported when 'source port' or "
      "'destination port' command is configured" )

appConstants = Tac.Value( 'Classification::Constants' )
FieldSetLimit = Tac.Type( "Classification::FieldSetLimit" )
tcpKeywordMatcher = CliMatcher.KeywordMatcher( "tcp", helpdesc="tcp",
                                               value=lambda mode,
                                               _:{ IPPROTO_TCP } )
tcpFlagCombinationTokens = {
   'established': 'Match on Established',
   'initial': 'Match on Initial'
}
tcpFlagTokens = {
   'fin': 'Match on Fin',
   'syn': 'Match on Syn',
   'rst': 'Match on Rst',
   'psh': 'Match on Psh',
   'ack': 'Match on Ack',
   'urg': 'Match on Urg',
}
notTcpFlagTokens = {
   'not-fin': 'Match on not Fin',
   'not-syn': 'Match on not Syn',
   'not-rst': 'Match on not Rst',
   'not-psh': 'Match on not Psh',
   'not-ack': 'Match on not Ack',
   'not-urg': 'Match on not Urg',
}
tcpFlagKeyWordMapping = {
   'established': 'est',
   'initial': 'init',
   'fin': 'fin',
   'syn': 'syn',
   'rst': 'rst',
   'psh': 'psh',
   'ack': 'ack',
   'urg': 'urg',
}

def populateSymbolicDscpValueMap( mapDict ):
   # AF<x><y> values: <x> << 3 | <y> << 1
   for x in range( 1, 5 ):
      for y in range( 1, 4 ):
         strVal = f'af{x}{y}'
         intVal = ( x << 3 ) | ( y << 1 )
         mapDict[ intVal ] = strVal
   # CS<x> values: <x> << 3
   for x in range( 0, 8 ):
      strVal = f'cs{x}'
      intVal = ( x << 3 )
      mapDict[ intVal ] = strVal
   mapDict[ 46 ] = 'ef'

classifConstants = Tac.Type( "Classification::ClassificationConstants" )
DEFAULT_SERVICE = classifConstants.defaultServiceName
DEFAULT_CATEGORY = classifConstants.defaultCategoryName

fieldSetUrlLocalMatcher = UrlMatcher(
   lambda fs: fs.scheme in listUrlLocalSchemes,
   "Local URL to load field-set entries",
   allowAllPaths=True, notAllowed=[ 'and-results' ] )

fieldSetUrlNetworkMatcher = UrlMatcher(
   lambda fs: fs.scheme in listUrlNetworkSchemes,
   "Remote URL to load field-set entries", acceptSimpleFile=False,
   priority=CliParser.PRIO_LOW )

def nameAndNumberExpression( name, rangeMatcher, nameMatcher,
                             nameToValueMaps, mapFn=None ):
   """nameToValueMaps is a list of maps that have the format of
   name : ( value, description )

   or

   name : DescriptionObject

   For the first case, the default `mapFn` will extract the "value"
   from the "( value, description )" mappings.

   For the second case, provide a `mapFn` that tells this method how
   to extract the value from the "DescriptionObject".


   This function generates an expression that allows:
   1. A numerical range
   2. A list of names

   And sets the set of numeric values in the specified argument.
   """
   rangeName = name + "_RANGE"
   nameName = name + "_NAME"

   class NameAndNumberExpression( CliCommand.CliExpression ):
      expression = f"{rangeName} | {{ {nameName} }}"
      data = {
         rangeName: rangeMatcher,
         nameName: nameMatcher
         }

      @staticmethod
      def adapter( mode, args, argsList ):
         nonlocal mapFn
         names = args.pop( nameName, None )
         if names:
            valueSet = set()
            if mapFn is None:
               def mapFn( entry ):
                  return entry[ 0 ]

            for n in names:
               for nmap in nameToValueMaps:
                  entry = nmap.get( n )
                  if entry is not None:
                     valueSet.add( mapFn( entry ) )
         else:
            valueSet = args.pop( rangeName, None )
         if valueSet is not None:
            args[ name ] = valueSet
   return NameAndNumberExpression

# port number/range
@Tac.memoize
def _portNames():
   """Accepts one udp or tcp port keyword"""
   protoMaps = [ tcpServiceByName, udpServiceByName ]
   return {
      name: f'{desc.helpdesc} ({desc.port})' for protoMap in protoMaps
      for name, desc in protoMap.items()
   }

portNameMatcher = CliMatcher.DynamicKeywordMatcher( lambda mode: _portNames() )

def portExpression( name ):
   def mapFn( entry ):
      return entry.port

   return nameAndNumberExpression( name, portRangeMatcher, portNameMatcher,
                                   ( tcpServiceByName, udpServiceByName ),
                                   mapFn=mapFn )

def generateMultiRangeMatcher( name, maxVal, minVal=0, helpdesc='' ):
   if not helpdesc:
      helpdesc = f'{name} values(s) or range(s) of {name} values'
   return MultiRangeMatcher( rangeFn=lambda: ( minVal, maxVal ),
                             noSingletons=False,
                             helpdesc=helpdesc,
                             value=lambda mode, grList: set( grList.values() ) )

# XXXBUG641114, to avoid MemoryError happend in grList.values(), create a new matcher
# for 32-bit TEID and field-set integer ranges.
def generateMultiRangeMatcherV2( name, maxVal, minVal=0,
                                 numberFormat=NumberFormat.DEFAULT, helpdesc='' ):
   if not helpdesc:
      helpdesc = f'{name} values(s) or range(s) of {name} values'
   return MultiRangeMatcher( rangeFn=lambda: ( minVal, maxVal ),
                             noSingletons=False,
                             numberFormat=numberFormat,
                             helpdesc=helpdesc,
                             value=lambda mode, grList: set( grList.ranges() ) )

def generateLimitEntryMatcher( maxValue ):
   return CliMatcher.IntegerMatcher( 1, maxValue,
             helpdesc="Max number of entries that can be configured on a field set" )

ipLengthRangeMatcher = generateMultiRangeMatcher( 'length', appConstants.maxLength )
fragOffsetRangeMatcher = generateMultiRangeMatcher( 'fragment offset',
                                                    appConstants.maxFragment )
portRangeMatcher = generateMultiRangeMatcher( 'port', appConstants.maxL4Port )
protoRangeMatcher = generateMultiRangeMatcher( 'protocol',
                                               appConstants.maxProto, 1 )
vlanTagRangeMatcher = generateMultiRangeMatcher(
   'vlan tag', appConstants.maxVlan - 1, 1, helpdesc='Identifier for a Virtual LAN' )
vlanRangeMatcher = generateMultiRangeMatcher(
   'vlan', appConstants.maxVlan - 1, 1, helpdesc='Identifier for a Virtual LAN' )
portKwNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'port', helpdesc='Port' ),
   noResult=True )
fieldSetKwNode = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'field-set', helpdesc='Field set' ),
   noResult=True )
sourceKwMatcher = CliMatcher.KeywordMatcher(
   'source', helpdesc='Source that specifies the contents of this field set' )
dot1QKwMatcher = CliMatcher.KeywordMatcher(
   'dot1q', helpdesc='802.1q encapsulation' )
vlanKwMatcher = CliMatcher.KeywordMatcher( 'vlan', helpdesc='Configure VLAN' )
macKwMatcher = CliMatcher.KeywordMatcher( 'mac',
                                          helpdesc='Configure Ethernet address' )
fieldSetConfigKwMatcher = CliMatcher.KeywordMatcher( 'field-set',
                                                     helpdesc='Configure field set' )
fieldSetRefreshKwMatcher = CliMatcher.KeywordMatcher( 'field-set',
                                                      helpdesc='Refresh field set' )
icmpTypeRangeMatcher = generateMultiRangeMatcher( 'icmp type',
                                                  appConstants.maxIcmpType )
icmpCodeRangeMatcher = generateMultiRangeMatcher( 'icmp code',
                                                  appConstants.maxIcmpCode )
pMapConstants = Tac.Value( 'Classification::Constants' )
protoValueMatcher = CliMatcher.IntegerMatcher( pMapConstants.minProtoV4,
                                               pMapConstants.maxProto,
                                               helpdesc='Protocol value' )
teidKwMatcher = CliMatcher.KeywordMatcher(
   'teid', helpdesc='Configure GTPv1 TEID match criteria' )
teidRangeMatcher = generateMultiRangeMatcherV2( 'TEID', appConstants.maxTeid,
                                             helpdesc='Tunnel endpoint identifier' )
integerKwMatcher = CliMatcher.KeywordMatcher(
   'integer', helpdesc='Configure a field-set of integer ranges' )
integerRangeMatcher = generateMultiRangeMatcherV2( 'integer',
                                                   appConstants.maxInteger )
nexthopKwMatcher = CliMatcher.KeywordMatcher(
   'next-hop', helpdesc='Match on next-hop group' )
nexthopGroupKwMatcher = CliMatcher.KeywordMatcher(
   'group', helpdesc='Add next-hop group name' )
locationKwMatcher = CliMatcher.KeywordMatcher(
   'location', helpdesc='Match packets based on location' )
locationValueMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                                helpname='<0x00000000-0xFFFFFFFF>',
                                                helpdesc='Pattern to match' )
locationMaskMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                              helpname='<0x00000000-0xFFFFFFFF>',
                                              helpdesc='Mask for pattern to match' )

def getIcmpCodeMap( mode, context, icmpTypeWithValidCodes, icmpCodes ):
   icmpTypeName = context.sharedResult.get( 'TYPE_NAME' )
   for name, value in icmpTypeWithValidCodes.items():
      if name == icmpTypeName:
         icmpTypeValue = value[ 0 ]
         break
   if icmpTypeValue is None:
      return {}
   icmpCodeMap = getKeywordMap( icmpCodes.get( icmpTypeValue, {} ) )
   return icmpCodeMap

icmpV4CodeNameMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode, context: getIcmpCodeMap( mode, context, icmpV4TypeWithValidCodes,
                                         icmpV4Codes ), passContext=True )
icmpV6CodeNameMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode, context: getIcmpCodeMap( mode, context, icmpV6TypeWithValidCodes,
                                         icmpV6Codes ), passContext=True )
icmpProtocols = {
   'icmp': ( IPPROTO_ICMP, 'Internet Control Message Protocol' ),
   'icmpv6': ( IPPROTO_ICMPV6, 'Internet Control Message Protocol version 6' ),
}
icmpV4KwMatcher = CliMatcher.KeywordMatcher(
   'icmp', helpdesc=f"{icmpProtocols['icmp'][1]} ({int(icmpProtocols['icmp'][0])})" )
icmpV6KwMatcher = CliMatcher.KeywordMatcher(
   'icmpv6', helpdesc='%s (%d)' % ( icmpProtocols[ 'icmpv6' ][ 1 ],
                                    icmpProtocols[ 'icmpv6' ][ 0 ] ) )

# This CliExpression only matches on TCP flags combinations: established, initial,
# because we have CLIs to configure TCP flag combinations and l4 ports together in
# previous release. Use this to keep backward compatibilty.
def generateTcpFlagCombinationExpression():
   class TcpFieldsExpression( CliCommand.CliExpression ):
      expression = "tcp flags { TCP_FLAGS_COMBINATION }"
      data = {
         'tcp': tcpKeywordMatcher,
         'flags': 'flags',
         'TCP_FLAGS_COMBINATION': CliMatcher.EnumMatcher( tcpFlagCombinationTokens ),
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         yesFlags = []
         flags = args.get( 'TCP_FLAGS_COMBINATION', [] )
         for flag in flags:
            yesFlags.append( flag )
         # process flags
         yesFlags = [ tcpFlagKeyWordMapping[ f ] for f in yesFlags ]
         args[ 'FLAGS_EXPR' ] = { 'yesFlags': yesFlags }

   return TcpFieldsExpression

# This CliExpression matches on TCP flags combinations (established, initial) or
# individual flags (syn, ack, rst,...).
def generateTcpFlagExpression( tcpFlagsSupported=False, notAllowed=False,
                               guard=None ):
   class TcpFieldsExpression( CliCommand.CliExpression ):
      if tcpFlagsSupported:
         if notAllowed:
            tcpFlagTokens.update( notTcpFlagTokens )
         expression = "tcp flags { TCP_FLAGS_COMBINATION } | { TCP_FLAGS }"
         data = {
            'tcp': tcpKeywordMatcher,
            'flags': CliCommand.guardedKeyword( 'flags',
                                                helpdesc='Match on tcp flags',
                                                guard=guard ),
            'TCP_FLAGS_COMBINATION': CliMatcher.EnumMatcher(
               tcpFlagCombinationTokens ),
            'TCP_FLAGS': CliMatcher.EnumMatcher( tcpFlagTokens )
         }
      else:
         expression = ""
         data = {}

      @staticmethod
      def adapter( mode, args, argsList ):
         notFlags = []
         yesFlags = []
         flags = args.get( 'TCP_FLAGS_COMBINATION', args.get( 'TCP_FLAGS', [] ) )
         for flag in flags:
            if flag.startswith( "not-" ):
               flag = flag.split( '-' )[ 1 ]
               notFlags.append( flag )
            else:
               yesFlags.append( flag )
         # process flags
         notFlags = [ tcpFlagKeyWordMapping[ f ] for f in notFlags ]
         yesFlags = [ tcpFlagKeyWordMapping[ f ] for f in yesFlags ]
         args[ 'FLAGS_EXPR' ] = { 'notFlags': notFlags, 'yesFlags': yesFlags }

   return TcpFieldsExpression

def generateIpProtoExpression( name, genericRules, extraRules ):
   # Generate protocol expression that accepts both names and ranges
   protoMap = getKeywordMap( genericRules, extraRules )
   protoName = name + "_PROTO"
   protoRange = name + "_RANGE"

   class IpProtoExpression( CliCommand.CliExpression ):
      expression = f"{{ {protoName} }} | {protoRange}"
      data = { protoName: CliMatcher.DynamicKeywordMatcher( lambda mode:
                                                            protoMap ),
               protoRange: protoRangeMatcher }

      @staticmethod
      def adapter( mode, args, argsList ):
         names = args.pop( protoName, None )
         if names:
            valueSet = { ( genericRules.get( n, None ) or
                              extraRules[ n ] )[ 0 ] for n in names }
         else:
            valueSet = args.pop( protoRange, None )
         if valueSet is not None:
            args[ name ] = valueSet

   return IpProtoExpression

def generateIpProtoSingleExpression( genericRules, extraRules ):
   '''Generate protocol expression that accepts either a protocol value or a
      well know protocol name'''
   protoMap = getKeywordMap( genericRules, extraRules )

   class IpProtoSingleExpression( CliCommand.CliExpression ):
      expression = "PROTOCOL_NAME | PROTOCOL_VALUE"
      data = { 'PROTOCOL_NAME': CliMatcher.DynamicKeywordMatcher( lambda mode:
                                                                   protoMap ),
               'PROTOCOL_VALUE': protoValueMatcher,
             }

      @staticmethod
      def adapter( mode, args, argsList ):
         ipProtoVal = args.get( 'PROTOCOL_VALUE' )
         if ipProtoVal is None:
            protoStr = args.get( 'PROTOCOL_NAME' )
            if protoStr is None:
               return
            protoTuple = genericRules.get( protoStr, None ) or extraRules[ protoStr ]
            ipProtoVal = protoTuple[ 0 ]
         protoRange = Tac.Value( "Classification::ProtocolRange", ipProtoVal,
                                 ipProtoVal )
         sf = Tac.newInstance( "Classification::StructuredFilter", "" )
         sf.proto.newMember( protoRange )
         # Update the PROTO_FIELDS_SF arg with a structured filter containing the
         # proto range
         args[ 'PROTO_FIELDS_SF' ] = sf
   return IpProtoSingleExpression

def generateTcpUdpProtoExpression( name, tcpUdpRules, allowMultiple=True ):
   # generate an expression to have a set of tcp/udp:
   # "tcp | udp | ( tcp udp ) | ( udp | tcp )"
   matchers = { name + '_' + proto: CliCommand.singleKeyword(
      proto, helpdesc=val[ 1 ] ) for proto, val in tcpUdpRules.items() }

   class TcpUdpExpression( CliCommand.CliExpression ):
      expression = ' | '.join( m for m in matchers )
      if allowMultiple:
         expression = '{' + expression + '}'
      data = matchers

      @staticmethod
      def adapter( mode, args, argsList ):
         for m in matchers:
            arg = args.pop( m, None )
            if arg:
               args.setdefault( name, set() )
               args[ name ].add( tcpUdpRules[ arg ][ 0 ] )

   return TcpUdpExpression

def generateIcmpTypeRangeExpression( name, icmpTypeRules ):
   # Generate icmp type expression that accepts a range of names or a range of values
   icmpTypeMap = getKeywordMap( icmpTypeRules )
   icmpTypeName = name + "_NAME"
   icmpTypeRange = name + "_RANGE"

   class IcmpTypeRangeExpression( CliCommand.CliExpression ):
      expression = f"{{ {icmpTypeName} }} | {icmpTypeRange}"
      data = { icmpTypeName: CliMatcher.DynamicKeywordMatcher( lambda mode:
                                                               icmpTypeMap ),
               icmpTypeRange: icmpTypeRangeMatcher }

      @staticmethod
      def adapter( mode, args, argsList ):
         typeNames = args.pop( icmpTypeName, None )
         if typeNames:
            valueSet = { icmpTypeRules[ n ][ 0 ] for n in typeNames }
         else:
            valueSet = args.pop( icmpTypeRange, None )
         if valueSet is not None:
            args[ name ] = valueSet

   return IcmpTypeRangeExpression

def generateIcmpTypeSingleExpression( name, icmpTypeRules ):
   # Generate icmp type expression that acceptes a single name or a single value
   icmpTypeMap = getKeywordMap( icmpTypeRules )
   icmpTypeMatcher = CliMatcher.DynamicKeywordMatcher( lambda mode: icmpTypeMap )
   icmpTypeName = name + "_NAME"

   class IcmpTypeSingleExpression( CliCommand.CliExpression ):
      expression = f"{icmpTypeName}"
      data = { icmpTypeName: CliCommand.Node( icmpTypeMatcher,
                                              storeSharedResult=True ) }
      @staticmethod
      def adapter( mode, args, argsList ):
         typeName = args.pop( icmpTypeName, None )
         value = icmpTypeRules[ typeName ][ 0 ]
         args[ name ] = value

   return IcmpTypeSingleExpression

def generateIcmpCodeExpression( name, icmpCodeRules, icmpCodeNameMatcher ):
   # Generate icmp code expression that accepts names or ranges
   icmpCodeName = name + "_NAME"
   icmpCodeRange = name + "_RANGE"

   class IcmpCodeExpression( CliCommand.CliExpression ):
      expression = f"{{ {icmpCodeName} }} | {icmpCodeRange}"
      data = { icmpCodeName: icmpCodeNameMatcher,
               icmpCodeRange: icmpCodeRangeMatcher }

      @staticmethod
      def adapter( mode, args, argsList ):
         codeNames = args.pop( icmpCodeName, None )
         valueSet = set()
         codeNameValueMap = dict() # pylint: disable=use-dict-literal
         if codeNames:
            for codeMap in icmpCodeRules.values():
               codeNameValueMap.update( codeMap )
            for codeName in codeNames:
               for n, value in codeNameValueMap.items():
                  if n == codeName:
                     valueSet.add( value[ 0 ] )
                     break
         else:
            valueSet = args.pop( icmpCodeRange, None )
         if valueSet:
            args[ name ] = valueSet

   return IcmpCodeExpression

icmpV4TypeRangeExpr = generateIcmpTypeRangeExpression( 'TYPE', icmpV4Types )
icmpV6TypeRangeExpr = generateIcmpTypeRangeExpression( 'TYPE', icmpV6Types )

icmpV4TypeSingleExpr = generateIcmpTypeSingleExpression( 'TYPE',
                                                         icmpV4TypeWithValidCodes )
icmpV6TypeSingleExpr = generateIcmpTypeSingleExpression( 'TYPE',
                                                         icmpV6TypeWithValidCodes )

icmpV4CodeExpr = generateIcmpCodeExpression( 'CODE', icmpV4Codes,
                                             icmpV4CodeNameMatcher )
icmpV6CodeExpr = generateIcmpCodeExpression( 'CODE', icmpV6Codes,
                                             icmpV6CodeNameMatcher )

tcpUdpProtoExpr = generateTcpUdpProtoExpression( 'TCP_UDP', tcpUdpProtocols )

ipv4ProtoExpr = generateIpProtoExpression( 'PROTOCOL', genericIpProtocols,
                                           extraIpv4Protocols )
ipv4ProtoSingleExpr = generateIpProtoSingleExpression( genericIpProtocols,
                                                       extraIpv4Protocols )

def generateFieldSetExpression( nameMatcher, name, allowMultiple=True ):
   class FieldSetExpression( CliCommand.CliExpression ):
      expression = name
      if allowMultiple:
         expression = '{' + expression + '}'
      data = {
         name: nameMatcher
      }
   return FieldSetExpression

def sameCategory( src, dst ):
   # don't copy if no change in any fields
   if set( src.appService ) != set( dst.appService ):
      return False
   for appName in src.appService:
      services = dst.appService.get( appName )
      if services is None:
         return False

      if set( src.appService[ appName ].service ) != set( services.service ):
         return False

   return True

def appIsEqual( src, dst ):
   srcProtoStr = numericalRangeToRangeString( src.proto )
   dstProtoStr = numericalRangeToRangeString( dst.proto )
   srcDscpStr = numericalRangeToRangeString( src.dscp )
   dstDscpStr = numericalRangeToRangeString( dst.dscp )
   if src.srcPrefixFieldSet == dst.srcPrefixFieldSet and \
      src.dstPrefixFieldSet == dst.dstPrefixFieldSet and \
      src.srcPortFieldSet == dst.srcPortFieldSet and \
      src.dstPortFieldSet == dst.dstPortFieldSet and \
      src.af == dst.af and \
      srcProtoStr == dstProtoStr and \
      srcDscpStr == dstDscpStr and \
      sorted( src.dscpSymbolic.items() ) == sorted( dst.dscpSymbolic.items() ) and \
      src.appId == dst.appId and \
      src.defaultApp == dst.defaultApp and \
      sorted( src.defaultServiceCategory.items() ) == \
         sorted( dst.defaultServiceCategory.items() ):
      return True
   return False

def appCopy( src, dst ):
   dst.af = src.af
   dst.srcPrefixFieldSet = src.srcPrefixFieldSet
   dst.dstPrefixFieldSet = src.dstPrefixFieldSet
   dst.srcPortFieldSet = src.srcPortFieldSet
   dst.dstPortFieldSet = src.dstPortFieldSet
   dst.proto.clear()
   for protoRange in src.proto:
      dst.proto.add( protoRange )
   dst.dscp.clear()
   for dscp in src.dscp:
      dst.dscp.add( dscp )
   dst.dscpSymbolic.clear()
   for dscp, useName in src.dscpSymbolic.items():
      dst.dscpSymbolic[ dscp ] = useName
   dst.defaultApp = src.defaultApp
   dst.appId = src.appId
   for service, category in src.defaultServiceCategory.items():
      dst.defaultServiceCategory[ service ] = category
   dst.version = src.version

class AppRecognitionContext:
   def __init__( self, appRecognitionConfig, fieldSetConfig ):
      self.appRecognitionCurrConfig = appRecognitionConfig
      self.fieldSetCurrConfig = fieldSetConfig
      self.appRecognitionEditConfig = None
      self.fieldSetEditConfig = None
      self.mode_ = None

   def modeIs( self, mode ):
      self.mode_ = mode

   def copyEditAppRecognitionConfig( self ):
      self.appRecognitionEditConfig = Tac.newInstance(
              'Classification::AppRecognitionConfig', 'appRecognitionConfig' )
      self.fieldSetEditConfig = Tac.newInstance(
              'Classification::FieldSetConfig', 'fieldSetConfig' )
      self.copyFieldSet( toSysdb=False )
      self.copyAppRec( toSysdb=False )

   def copyFieldSetL4Port( self, src, dst ):
      for name, srcFsCfg in src.items():
         dstFsCfg = dst.get( name )
         srcSubCfg = srcFsCfg.currCfg
         if not dstFsCfg:
            dstFsCfg = dst.newMember( name )
            prevDstCurrCfg = None
         else:
            prevDstCurrCfg = dstFsCfg.currCfg
            # don't copy if no change in ports set
            if srcSubCfg.isEqual( prevDstCurrCfg ):
               continue

         dstSubCfg = dstFsCfg.subConfig.newMember( name, UniqueId() )
         dstSubCfg.copy( srcSubCfg )
         dstFsCfg.currCfg = dstSubCfg

         if prevDstCurrCfg is not None:
            assert prevDstCurrCfg.version in dstFsCfg.subConfig
            del dstFsCfg.subConfig[ prevDstCurrCfg.version ]

      # Delete stale entries
      for name in dst:
         if name not in src:
            del dst[ name ]

   def copyFieldSetIpPrefix( self, src, dst ):
      for name, srcFsCfg in src.items():
         dstFsCfg = dst.get( name )
         srcSubCfg = srcFsCfg.currCfg
         if not dstFsCfg:
            # XXX - We're always copying the same AF, right?
            dstFsCfg = dst.newMember( name, srcFsCfg.af )
            prevDstCurrCfg = None
         else:
            prevDstCurrCfg = dstFsCfg.currCfg
            if srcSubCfg.isEqual( prevDstCurrCfg ):
               continue
         dstSubCfg = dstFsCfg.subConfig.newMember( name, UniqueId() )
         dstSubCfg.copy( srcSubCfg )
         dstFsCfg.currCfg = dstSubCfg

         if prevDstCurrCfg is not None:
            assert prevDstCurrCfg.version in dstFsCfg.subConfig
            del dstFsCfg.subConfig[ prevDstCurrCfg.version ]

      # Delete stale entries
      for name in dst:
         if name not in src:
            del dst[ name ]

   def copyAppProfile( self, src, dst ):
      for name in src:
         if name not in dst:
            dst.newMember( name )

         # don't copy if no change in app set
         if sorted( src[ name ].app.keys() ) == sorted( dst[ name ].app.keys() ):
            continue

         # clear all the apps before copying to avoid stale entry
         dst[ name ].app.clear()
         for appName in src[ name ].app:
            app = dst[ name ].app.newMember( appName )
            app.service.add( 'all' )

         dst[ name ].version = src[ name ].version

      # Delete Stale entries
      for name in dst:
         if name not in src:
            del dst[ name ]

   def copyAppProfileWithAppService( self, src, dst ):
      # Add new app profiles
      for name in src:
         if name not in dst:
            dst.newMember( name )

         # Delete stale apps
         for appName in dst[ name ].app:
            if appName not in src[ name ].app:
               del dst[ name ].app[ appName ]

         # Add/update apps
         for appName in src[ name ].app:
            srcApp = src[ name ].app[ appName ]
            dstApp = dst[ name ].app.newMember( appName )
            # Delete stale app services
            for service in dstApp.service:
               if service not in srcApp.service:
                  dstApp.service.remove( service )
            # Add new app services
            for service in srcApp.service:
               dstApp.service.add( service )

         # Delete stale categories
         for categoryName in dst[ name ].category:
            if categoryName not in src[ name ].category:
               del dst[ name ].category[ categoryName ]

         # Add/update categories
         for categoryName in src[ name ].category:
            srcCategory = src[ name ].category[ categoryName ]
            dstCategory = dst[ name ].category.newMember( categoryName )
            # Delete stale category services
            for service in dstCategory.service:
               if service not in srcCategory.service:
                  dstCategory.service.remove( service )
            # Add new category services
            for service in srcCategory.service:
               dstCategory.service.add( service )

         # Delete stale app transports
         for appName in dst[ name ].appTransport:
            if appName not in src[ name ].appTransport:
               del dst[ name ].appTransport[ appName ]

         # Add/update app transports.
         for appName in src[ name ].appTransport:
            dst[ name ].appTransport.add( appName )

         dst[ name ].version = src[ name ].version

      # Delete stale app profiles
      for name in dst:
         if name not in src:
            del dst[ name ]

   def copyApp( self, src, dst ):
      for name, srcApp in src.items():
         dstApp = dst.newMember( name )
         dstApp.readonly = srcApp.readonly
         if srcApp.readonly:
            continue # don't copy readonly applications as we won't be editing them

         # don't copy if no change in any fields
         if appIsEqual( srcApp, dstApp ):
            continue

         appCopy( srcApp, dstApp )

      # Delete stale entries
      for name in dst:
         if name not in src:
            del dst[ name ]

   def copyCategory( self, src, dst ):
      # Delete Stale entries
      for name in dst:
         if name not in src:
            del dst[ name ]

      for name, srcCategory in src.items():
         dstCategory = dst.newMember( name )
         dstCategory.defaultCategory = srcCategory.defaultCategory
         dstCategory.categoryId = srcCategory.categoryId

         # Delete stale app service
         for appName, dstApp in dstCategory.appService.items():
            if appName not in srcCategory.appService:
               del dstCategory.appService[ appName ]
               continue
            srcApp = srcCategory.appService[ appName ]
            for service in dstApp.service:
               if service not in srcApp.service:
                  dstApp.service.remove( service )
         # Add new app service
         for appName, srcApp in srcCategory.appService.items():
            dstApp = dstCategory.appService.newMember( appName )
            for service in srcApp.service:
               if service not in dstApp.service:
                  dstApp.service.add( service )

   def copyFieldSet( self, toSysdb=False ):
      # When copying to Sysdb, copy the "scratchpad" EditConfig to the current
      # configuration "CurrConfig". Otherwise, the direction is reversed.
      if toSysdb:
         src = self.fieldSetEditConfig
         dst = self.fieldSetCurrConfig
      else:
         src = self.fieldSetCurrConfig
         dst = self.fieldSetEditConfig
      self.copyFieldSetL4Port( src.fieldSetL4Port, dst.fieldSetL4Port )
      self.copyFieldSetIpPrefix( src.fieldSetIpPrefix, dst.fieldSetIpPrefix )

   def copyAppRec( self, toSysdb=False ):
      # When copying to Sysdb, copy the "scratchpad" EditConfig to the current
      # configuration "CurrConfig". Otherwise, the direction is reversed.
      if toSysdb:
         src = self.appRecognitionEditConfig
         dst = self.appRecognitionCurrConfig
      else:
         src = self.appRecognitionCurrConfig
         dst = self.appRecognitionEditConfig
      self.copyApp( src.app, dst.app )
      self.copyAppProfileWithAppService( src.appProfile, dst.appProfile )
      self.copyCategory( src.category, dst.category )

   def abort( self ):
      self.appRecognitionEditConfig = None
      self.fieldSetEditConfig = None

   def commit( self ):
      if self.fieldSetEditConfig:
         self.copyFieldSet( toSysdb=True )

      if self.appRecognitionEditConfig:
         self.copyAppRec( toSysdb=True )

class AppProfileContext:
   def __init__( self, appProfileName, parentContext ):
      self.childMode = AppProfileConfigMode
      self.appProfileName = appProfileName
      self.appProfile = parentContext.appRecognitionEditConfig.appProfile
      self.appProfileEdit = None
      self.mode_ = None

   def copyEditAppProfile( self ):
      self.appProfileEdit = Tac.newInstance( 'Classification::AppProfile',
                                             self.appProfileName )
      self.copyAppServices( self.appProfile[ self.appProfileName ],
                            self.appProfileEdit )
      self.copyCategoryServices( self.appProfile[ self.appProfileName ],
                                 self.appProfileEdit )
      self.copyAppTransports( self.appProfile[ self.appProfileName ],
                              self.appProfileEdit )

   def newEditAppProfile( self ):
      self.appProfileEdit = Tac.newInstance( 'Classification::AppProfile',
                                             self.appProfileName )

   def updateApp( self, appName, add=True ):
      if add:
         app = self.appProfileEdit.app.newMember( appName )
         app.service.add( 'all' )
      else:
         del self.appProfileEdit.app[ appName ]

   def updateAppService( self, appName, serviceName=None, add=True ):
      if add:
         app = self.appProfileEdit.app.newMember( appName )
         if not serviceName:
            # If service is not specified, add service='all'
            app.service.clear()
            app.service.add( 'all' )
         else:
            # Ignore service specific config, if service='all'
            # has already been configured
            if 'all' not in app.service:
               app.service.add( serviceName )
      else:
         if appName in self.appProfileEdit.app:
            if not serviceName:
               # If service is not specified, remove all services
               del self.appProfileEdit.app[ appName ]
            else:
               app = self.appProfileEdit.app[ appName ]
               if serviceName in app.service:
                  app.service.remove( serviceName )
                  if not app.service:
                     # all service of app removed, so remove the app
                     del self.appProfileEdit.app[ appName ]

   def addAppTransport( self, appName ):
      self.appProfileEdit.appTransport.add( appName )

   def delAppTransport( self, appName ):
      del self.appProfileEdit.appTransport[ appName ]

   def updateCategoryService( self, categoryName, serviceName=None, add=True ):
      if add:
         category = self.appProfileEdit.category.newMember( categoryName )
         if serviceName:
            if 'all' not in category.service:
               category.service.add( serviceName )
         else:
            category.service.clear()
            category.service.add( 'all' )
      else:
         categories = self.appProfileEdit.category
         if serviceName:
            category = categories.get( categoryName )
            if category:
               if serviceName in category.service:
                  category.service.remove( serviceName )
                  if not category.service:
                     del categories[ categoryName ]
         else:
            del categories[ categoryName ]

   def copyApps( self, src, dst, commit=False ):
      if sorted( dst.app.keys() ) == sorted( src.app.keys() ):
         return

      dst.app.clear()
      for appName in src.app:
         app = dst.app.newMember( appName )
         app.service.add( 'all' )

      if commit:
         # increment the version because there is some config change
         dst.version += 1

   def copyAppTransports( self, src, dst, commit=False ):
      if set( dst.appTransport.keys() ) == set( src.appTransport.keys() ):
         return

      dst.appTransport.clear()
      for appName in src.appTransport:
         dst.appTransport.add( appName )

      if commit:
         # increment the version because there is some config change
         dst.version += 1

   def copyAppServices( self, src, dst, commit=False ):
      changed = False

      # Delete stale apps
      for appName in dst.app:
         if appName not in src.app:
            del dst.app[ appName ]
            changed = True

      for appName in src.app:
         # Add new apps
         if appName not in dst.app:
            dst.app.newMember( appName )
            changed = True

         srcApp = src.app[ appName ]
         dstApp = dst.app[ appName ]
         # Delete stale app services
         for service in dstApp.service:
            if service not in srcApp.service:
               dstApp.service.remove( service )
               changed = True
         # Add new app services
         for service in srcApp.service:
            if service not in dstApp.service:
               dstApp.service.add( service )
               changed = True

      if commit and changed:
         # increment the version because there is some config change
         dst.version += 1

   def copyCategoryServices( self, src, dst, commit=False ):
      changed = False

      # Delete stale categories
      for categoryName in dst.category:
         if categoryName not in src.category:
            del dst.category[ categoryName ]
            changed = True

      for categoryName in src.category:
         # Add new categories
         if categoryName not in dst.category:
            dst.category.newMember( categoryName )
            changed = True

         srcCategory = src.category[ categoryName ]
         dstCategory = dst.category[ categoryName ]
         # Delete stale category services
         for service in dstCategory.service:
            if service not in srcCategory.service:
               dstCategory.service.remove( service )
               changed = True
         # Add new category services
         for service in srcCategory.service:
            if service not in dstCategory.service:
               dstCategory.service.add( service )
               changed = True

      if commit and changed:
         # increment the version because there is some config change
         dst.version += 1

   def hasAppProfile( self, name ):
      return name in self.appProfile

   def delAppProfile( self, name ):
      del self.appProfile[ name ]

   def modeIs( self, mode ):
      self.mode_ = mode

   def commit( self ):
      # commit to parent context
      if self.appProfileEdit:
         if self.appProfileName not in self.appProfile:
            self.appProfile.newMember( self.appProfileName )

         self.copyAppServices( self.appProfileEdit,
                               self.appProfile[ self.appProfileName ],
                               commit=True )
         self.copyCategoryServices( self.appProfileEdit,
                                    self.appProfile[ self.appProfileName ],
                                    commit=True )
         self.copyAppTransports( self.appProfileEdit,
                                 self.appProfile[ self.appProfileName ],
                                 commit=True )

   def abort( self ):
      self.appProfileName = None
      self.appProfileEdit = None

_afToChildMode = {
   'ipv4': AppConfigModeIpv4,
   'bothIpv4AndIpv6': AppConfigModeL4,
}

class AppContext:

   def __init__( self, appName, parentContext, af ):
      self.af = af
      self.appName = appName
      self.app = parentContext.appRecognitionEditConfig.app
      self.appEdit = None
      self.mode_ = None

   @property
   def childMode( self ):
      return _afToChildMode[ self.af ]

   def copyEditApp( self ):
      self.appEdit = Tac.newInstance( 'Classification::AppConfig', self.appName )
      self.appEdit.af = self.af
      self.copyAppFields( self.app[ self.appName ], self.appEdit )

   def newEditApp( self ):
      self.appEdit = Tac.newInstance( 'Classification::AppConfig', self.appName )
      self.appEdit.af = self.af
      self.appEdit.defaultApp = False
      self.appEdit.appId = Tac.Type( "Classification::ApplicationId" ).invalid
      self.appEdit.defaultServiceCategory[ DEFAULT_SERVICE ] = DEFAULT_CATEGORY

   def updatePrefixFieldSet( self, source=True, names=None, add=True ):
      assert len( names ) <= 1
      if source:
         if add:
            self.appEdit.srcPrefixFieldSet = names[ 0 ]
         else:
            self.appEdit.srcPrefixFieldSet = ""
      else:
         if add:
            self.appEdit.dstPrefixFieldSet = names[ 0 ]
         else:
            self.appEdit.dstPrefixFieldSet = ""

   def portAndProtoConfigured( self ):
      protoSet = numericalRangeToSet( self.appEdit.proto )
      port = not ( self.appEdit.srcPortFieldSet == '' and
                   self.appEdit.dstPortFieldSet == '' )
      return ( protoSet, port )

   def updatePortFieldSetAttr( self, attrName, fieldSetName='', add=True, **kwargs ):
      if attrName not in [ 'srcPortFieldSet', 'dstPortFieldSet' ]:
         return
      if add:
         setattr( self.appEdit, attrName, fieldSetName )
      else:
         setattr( self.appEdit, attrName, '' )

   def updateTcpFlags( self, tcpFlags, notFlags, add=True ):
      """
      Does not support tcp flags yet
      """
      pass # pylint: disable=unnecessary-pass

   def maybeUpdateProto( self, protocolRangeSet ):
      """
      Takes in a set of protocol ranges where additional fields have been clear
      if all fields have been cleared (sport/dport/flags etc) we delete the protocol
      """
      pass # pylint: disable=unnecessary-pass

   def updateRangeAttr( self, attrName, rangeSet, rangeType, add=True ):
      currRange = getattr( self.appEdit, attrName )
      currentSet = numericalRangeToSet( currRange )
      updatedSet = set()
      if add:
         updatedSet = currentSet | rangeSet
      else:
         if not rangeSet:
            updatedSet = set()
         else:
            updatedSet = currentSet - rangeSet
      newRangeList = rangeSetToNumericalRange( updatedSet,
                                               rangeType )

      currRange.clear()
      for aRange in newRangeList:
         currRange.add( aRange )

      # delete source and dest port when no proto is present
      if attrName == 'proto' and not currRange:
         self.updatePortFieldSetAttr( 'srcPortFieldSet', add=False )
         self.updatePortFieldSetAttr( 'dstPortFieldSet', add=False )

      if attrName == 'dscp' and not currRange:
         self.appEdit.dscpSymbolic.clear()

   def getProto( self ):
      return self.appEdit.proto

   def updateDscpSymbolicValues( self, dscp, dscpUseName ):
      dscp = Tac.Value( "Arnet::DscpValue", dscp )
      self.appEdit.dscpSymbolic[ dscp ] = dscpUseName

   def clearDscpSymbolicValues( self, dscp ):
      dscp = Tac.Value( "Arnet::DscpValue", dscp )
      del self.appEdit.dscpSymbolic[ dscp ]

   def copyAppFields( self, src, dst, commit=False ):
      if appIsEqual( src, dst ):
         # no attribute has changed
         return

      appCopy( src, dst )

      if commit:
         # increment the version because there is some config change
         dst.version += 1

   def hasApp( self, name ):
      return name in self.app

   def getApp( self, name ):
      return self.app.get( name )

   def delApp( self, name ):
      del self.app[ name ]

   def modeIs( self, mode ):
      self.mode_ = mode

   def commit( self ):
      # commit to parent context
      if self.appEdit:
         if self.appName not in self.app:
            self.app.newMember( self.appName )
         self.copyAppFields( self.appEdit, self.app[ self.appName ], commit=True )

   def abort( self ):
      self.appName = None
      self.appEdit = None

class CategoryContext:
   def __init__( self, categoryName, parentContext ):
      self.childMode = CategoryConfigMode
      self.categoryName = categoryName
      self.category = parentContext.appRecognitionEditConfig.category
      self.categoryEdit = None
      self.mode_ = None

   def copyEditCategory( self ):
      self.categoryEdit = Tac.newInstance( 'Classification::CategoryConfig',
                                           self.categoryName )
      self.categoryEdit.defaultCategory = \
         self.category[ self.categoryName ].defaultCategory
      self.categoryEdit.categoryId = \
         self.category[ self.categoryName ].categoryId
      self.copyAppServices( self.category[ self.categoryName ], self.categoryEdit )

   def newEditCategory( self ):
      self.categoryEdit = Tac.newInstance( 'Classification::CategoryConfig',
                                           self.categoryName )
      self.categoryEdit.defaultCategory = False
      self.categoryEdit.categoryId = \
         Tac.Type( "Classification::CategoryId" ).invalid

   def addCategoryApps( self, appName, serviceName ):
      if appName not in self.categoryEdit.appService:
         self.categoryEdit.appService.newMember( appName )
      appServiceEdit = self.categoryEdit.appService[ appName ].service
      if serviceName:
         if "all" in appServiceEdit:
            appServiceEdit.remove( "all" )
         appServiceEdit.add( serviceName )
      else:
         appServiceEdit.clear()
         appServiceEdit.add( "all" )

   def delCategoryApps( self, appName, serviceName ):
      if appName not in self.categoryEdit.appService:
         return
      appServiceEdit = self.categoryEdit.appService[ appName ].service
      if serviceName:
         if serviceName in appServiceEdit:
            appServiceEdit.remove( serviceName )
      else:
         del self.categoryEdit.appService[ appName ]

   def copyAppServices( self, src, dst, commit=False ):
      if sameCategory( src, dst ):
         return

      for appName in src.appService:
         if appName not in dst.appService:
            dst.appService.newMember( appName )
         if "all" in src.appService[ appName ].service:
            dst.appService[ appName ].service.clear()
            dst.appService[ appName ].service.add( "all" )
         else:
            if "all" in dst.appService[ appName ].service:
               dst.appService[ appName ].service.remove( "all" )
            for service in src.appService[ appName ].service:
               dst.appService[ appName ].service.add( service )

      # Delete stale app services
      for app in dst.appService:
         if app not in src.appService:
            del dst.appService[ app ]
            return
         for service in dst.appService[ app ].service:
            if service not in src.appService[ app ].service:
               dst.appService[ app ].service.remove( service )

   # This function is to handle moving application from one category
   # to another category by removing possible stale entry in previous
   # category.
   def maybeDeleteStaleEntry( self, src, dst ):
      for categoryName, dstCategory in dst.items():
         if categoryName == src.categoryName:
            continue
         for appName, srcApp in src.appService.items():
            if appName not in dstCategory.appService:
               continue
            dstApp = dstCategory.appService[ appName ]
            for service in srcApp.service:
               # pylint: disable-next=no-else-break
               if service == 'all' or 'all' in dstApp.service:
                  del dstCategory.appService[ appName ]
                  break
               elif service in dstApp.service:
                  dstApp.service.remove( service )

   def hasCategory( self, name ):
      return name in self.category

   def getCategory( self, name ):
      return self.category.get( name )

   def delCategory( self, name ):
      del self.category[ name ]

   def clearCategoryApps( self, name ):
      self.category[ name ].appService.clear()

   def modeIs( self, mode ):
      self.mode_ = mode

   def commit( self ):
      # commit to parent context
      if self.categoryEdit:
         if self.categoryName not in self.category:
            self.category.newMember( self.categoryName )

         self.category[ self.categoryName ].defaultCategory = \
               self.categoryEdit.defaultCategory
         self.category[ self.categoryName ].categoryId = \
               self.categoryEdit.categoryId
         self.copyAppServices( self.categoryEdit,
                               self.category[ self.categoryName ],
                               commit=True )
         self.maybeDeleteStaleEntry( self.categoryEdit, self.category )

   def abort( self ):
      self.categoryName = None
      self.categoryEdit = None

#------------------------------------------------------------------------------------
# protocol service field-set FIELD_SET
# -----------------------------------------------------------------------------------
class ServiceFieldSetCmdBase( CliCommand.CliCommandClass ):
   syntax = 'protocol service field-set FIELD_SET'
   noOrDefaultSyntax = 'protocol service field-set [ FIELD_SET ]'
   _baseData = {
      'protocol': 'Protocol',
      'service': 'service',
      'field-set': 'Field set',
   }

   @classmethod
   def handler( cls, mode, args ):
      fieldSetName = args.get( 'FIELD_SET' )
      context = mode.getContext()
      # FIELD_SET may or may not allow for multiples
      if not isinstance( fieldSetName, list ):
         fieldSetName = [ fieldSetName ]
      context.updateServiceFieldSet( names=fieldSetName, add=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      fieldSetName = args.get( 'FIELD_SET', [] )
      context = mode.getContext()
      # FIELD_SET may or may not allow for multiples
      if not isinstance( fieldSetName, list ):
         fieldSetName = [ fieldSetName ]
      context.updateServiceFieldSet( names=fieldSetName, add=False )

# -----------------------------------------------------------------------------------
# (source | destination) prefix field-set FIELD_SET
#------------------------------------------------------------------------------------
class PrefixFieldSetCmdBase( CliCommand.CliCommandClass ):
   syntax = '( source | destination ) prefix field-set FIELD_SET'
   noOrDefaultSyntax = '( source | destination ) prefix field-set [ FIELD_SET ]'

   _baseData = {
      'source': 'Source',
      'destination': 'Destination',
      'prefix': 'Prefix',
      'field-set': 'Field set',
   }

   @classmethod
   def handler( cls, mode, args ):
      source = args.get( 'source', False )
      fieldSetName = args.get( 'FIELD_SET' )
      context = mode.getContext()
      # FIELD_SET may or may not allow for multiples
      if not isinstance( fieldSetName, list ):
         fieldSetName = [ fieldSetName ]
      context.updatePrefixFieldSet( source=source, names=fieldSetName, add=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      source = args.get( 'source', False )
      fieldSetName = args.get( 'FIELD_SET', [] )
      context = mode.getContext()
      # FIELD_SET may or may not allow for multiples
      if not isinstance( fieldSetName, list ):
         fieldSetName = [ fieldSetName ]
      context.updatePrefixFieldSet( source=source, names=fieldSetName, add=False )

#------------------------------------------------------------------------------------
# [ except <exceptPrefix> ]
#------------------------------------------------------------------------------------
def generateExceptExpression( matcher=None, exceptSupported=False,
                              allowMultiple=True ):
   class ExceptExpression( CliCommand.CliExpression ):
      if exceptSupported:
         expression = "except "
         if allowMultiple:
            expression += "{ EXCEPT_ITEMS }"
         else:
            expression += "EXCEPT_ITEMS"
         data = {
            'except': 'except',
            'EXCEPT_ITEMS': matcher
         }
      else:
         expression = ""
         data = {}

      @staticmethod
      def adapter( mode, args, argsList ):
         exceptPrefixes = args.get( 'EXCEPT_ITEMS', [] )
         if not isinstance( exceptPrefixes, list ):
            exceptPrefixes = [ exceptPrefixes ]
         args[ 'EXCEPT_ITEMS' ] = exceptPrefixes
   return ExceptExpression

#------------------------------------------------------------------------------------
# (source | destination) prefix [longest-prefix] <prefix>
#------------------------------------------------------------------------------------
class PrefixAdapterBase( CliCommand.CliExpression ):
   @staticmethod
   def adapter( mode, args, argList ):
      prefixes = args.get( 'PREFIX', [] )
      fieldSetNames = args.get( 'FIELD_SET', [] )
      longestPrefix = 'longest-prefix' in args
      if not isinstance( prefixes, list ):
         prefixes = [ prefixes ]
      if not isinstance( fieldSetNames, list ):
         fieldSetNames = [ fieldSetNames ]
      prefixes = [ IpGenPrefix( str( prefix ) ) for prefix in prefixes ]
      sf = Tac.newInstance( "Classification::StructuredFilter", "" )
      if 'source' in args:
         prefixColl = sf.sourceLpm if longestPrefix else sf.source
         fieldSetColl = sf.srcLpmPrefixSet if longestPrefix else sf.srcPrefixSet
      else:
         prefixColl = sf.destinationLpm if longestPrefix else sf.destination
         fieldSetColl = sf.dstLpmPrefixSet if longestPrefix else sf.dstPrefixSet
      for prefix in prefixes:
         prefixColl.add( prefix )
      for fsName in fieldSetNames:
         fieldSetColl.add( fsName )
      args[ 'PREFIX_SF' ] = sf

class MacAddrAdapterBase( CliCommand.CliExpression ):
   @staticmethod
   def adapter( mode, args, argList ):
      macAddresses = args.get( 'MAC_ADDR', [] )
      fieldSetNames = args.get( 'FIELD_SET', [] )
      if not isinstance( macAddresses, list ):
         macAddresses = [ macAddresses ]
      if not isinstance( fieldSetNames, list ):
         fieldSetNames = [ fieldSetNames ]
      # BUG706397: macAddrMask entry is populated when we support MAC address mask
      # and currently we set it as default value
      defaultMask = EthAddr( 'ffff.ffff.ffff' )
      macAddresses = [ EthAddrWithMask( EthAddr( mac ), defaultMask )
                       for mac in macAddresses ]
      sf = Tac.newInstance( "Classification::StructuredFilter", "" )
      if 'source' in args:
         macAddrColl = sf.srcMac
         fieldSetColl = sf.srcMacAddrSet
      else:
         macAddrColl = sf.dstMac
         fieldSetColl = sf.dstMacAddrSet
      for mac in macAddresses:
         macAddrColl.add( mac )
      for fsName in fieldSetNames:
         fieldSetColl.add( fsName )
      args[ 'MAC_ADDR_SF' ] = sf

class SrcDstMacAddrCmdMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, allowMultiple, allowSource=True, allowDestination=True ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.allowMultiple = allowMultiple
      self.allowSource = allowSource
      self.allowDestination = allowDestination

   def generate( self, name ):
      class MacAddrExpression( MacAddrAdapterBase ):
         assert self.allowSource or self.allowDestination
         if self.allowSource and self.allowDestination:
            _srcDstStr = '( source | destination )'
         elif self.allowSource:
            _srcDstStr = 'source'
         else:
            _srcDstStr = 'destination'
         _baseExpr = f'{_srcDstStr} mac'
         if self.allowMultiple:
            expression = _baseExpr + '{ MAC_ADDR }'
         else:
            expression = _baseExpr + 'MAC_ADDR'

         data = {
            'source': 'source',
            'destination': 'destination',
            'mac': 'Ethernet address',
            'MAC_ADDR': MacAddr.macAddrMatcher,
         }
      return MacAddrExpression

class SrcDstMacAddrFieldSetCmdMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, allowMultiple, fieldSetMatcher,
                 allowSource=True, allowDestination=True ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.allowMultiple = allowMultiple
      self.fieldSetMatcher = fieldSetMatcher
      self.allowSource = allowSource
      self.allowDestination = allowDestination

   def generate( self, name ):
      class MacAddrFieldSetExpression( MacAddrAdapterBase ):
         assert self.allowSource or self.allowDestination
         if self.allowSource and self.allowDestination:
            _srcDstStr = '( source | destination )'
         elif self.allowSource:
            _srcDstStr = 'source'
         else:
            _srcDstStr = 'destination'
         _baseExpr = f'{_srcDstStr} mac field-set'
         if self.allowMultiple:
            expression = _baseExpr + '{ FIELD_SET }'
         else:
            expression = _baseExpr + 'FIELD_SET'

         data = {
            'source': 'source',
            'destination': 'destination',
            'mac': 'Ethernet address',
            'field-set': 'field set',
            'FIELD_SET': self.fieldSetMatcher,
         }
      return MacAddrFieldSetExpression

class PrefixCmdMatcher( CliCommand.CliExpressionFactory ):
   """
   addrType - either 'ipv4' or 'ipv6'. Used to determine which prefix expr we use
   allowMultiple - flag to determine if we match multiple prefixes or a single prefix
                   Some features using this expression only allow single prefix
                   matching rather than matching a list of prefixes.
   longestPrefix - allow longest-prefix keyword
   allowSource - flag to allow source prefix field-set
   allowDestination - flag to allow destination prefix field-set
   """
   def __init__( self, addrType, allowMultiple, longestPrefix=False,
                 allowSource=True, allowDestination=True ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.addrType = addrType
      self.allowMultiple = allowMultiple
      self.longestPrefix = longestPrefix
      self.allowSource = allowSource
      self.allowDestination = allowDestination

   def generate( self, name ):
      if self.addrType == 'ipv4':
         prefixExpr = IpAddrMatcher.ipPrefixExpr(
            'Prefix address',
            'Prefix mask',
            'Prefix',
            overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT,
            raiseError=True )
      else:
         prefixExpr = Ip6AddrMatcher.ip6PrefixExpr(
            'Prefix address',
            'Prefix mask',
            'Prefix',
            overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT, raiseError=True )

      class PrefixExpression( PrefixAdapterBase ):
         assert self.allowSource or self.allowDestination
         if self.allowSource and self.allowDestination:
            _srcDstStr = '( source | destination )'
         elif self.allowSource:
            _srcDstStr = 'source'
         else:
            _srcDstStr = 'destination'
         _baseExpr = f'{_srcDstStr} prefix'
         if self.longestPrefix:
            _baseExpr += ' longest-prefix'
         if self.allowMultiple:
            expression = _baseExpr + '{ PREFIX }'
         else:
            expression = _baseExpr + ' PREFIX'

         data = {
            'source': 'source',
            'destination': 'destination',
            'prefix': 'prefix',
            'longest-prefix': 'longest prefix match',
            'PREFIX': prefixExpr,
         }
      return PrefixExpression

class PrefixFieldSetCmdMatcher( CliCommand.CliExpressionFactory ):
   """
   allowMultiple - flag to determine if we match multiple field-sets or a single
                   field-set. Some features using this expression only allow single
                   field-set matching rather than matching a list of prefixes.
   longestPrefix - allow longest-prefix keyword
   allowSource - flag to allow source prefix field-set
   allowDestination - flag to allow destination prefix field-set
   """
   def __init__( self, allowMultiple, fieldSetMatcher, longestPrefix=False,
                 allowSource=True, allowDestination=True ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.allowMultiple = allowMultiple
      self.fieldSetMatcher = fieldSetMatcher
      self.longestPrefix = longestPrefix
      self.allowSource = allowSource
      self.allowDestination = allowDestination

   def generate( self, name ):
      class PrefixFieldSetExpression( PrefixAdapterBase ):
         assert self.allowSource or self.allowDestination
         if self.allowSource and self.allowDestination:
            _srcDstStr = '( source | destination )'
         elif self.allowSource:
            _srcDstStr = 'source'
         else:
            _srcDstStr = 'destination'
         _baseExpr = '{} prefix{} field-set '.format( _srcDstStr,
                                 ' longest-prefix' if self.longestPrefix else '' )
         if self.allowMultiple:
            expression = _baseExpr + '{ FIELD_SET }'
         else:
            expression = _baseExpr + 'FIELD_SET'

         data = {
            'source': 'source',
            'destination': 'destination',
            'prefix': 'prefix',
            'longest-prefix': 'longest prefix match',
            'field-set': 'field set',
            'FIELD_SET': self.fieldSetMatcher,
         }
      return PrefixFieldSetExpression

class MacCmdBase( CliCommand.CliCommandClass ):
   syntax = "MAC_EXPR"
   noOrDefaultSyntax = syntax

   @classmethod
   def handler( cls, mode, args ):
      macSf = args[ "MAC_ADDR_SF" ]
      context = mode.getContext()
      context.addMacFromSf( macSf )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      macSf = args[ "MAC_ADDR_SF" ]
      context = mode.getContext()
      context.removeMacFromSf( macSf )

class PrefixCmdBaseV2( CliCommand.CliCommandClass ):
   syntax = "PREFIX_EXPR"
   noOrDefaultSyntax = syntax

   @classmethod
   def handler( cls, mode, args ):
      prefixSf = args[ "PREFIX_SF" ]
      context = mode.getContext()
      context.addPrefixFromSf( prefixSf )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      prefixSf = args[ "PREFIX_SF" ]
      context = mode.getContext()
      context.removePrefixFromSf( prefixSf )

class PrefixIpv4CmdV2( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR": PrefixCmdMatcher( addrType="ipv4", allowMultiple=True )
   }

class PrefixIpv6CmdV2( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR": PrefixCmdMatcher( addrType="ipv6", allowMultiple=True )
   }

class PrefixLpmIpv4Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR": PrefixCmdMatcher( addrType="ipv4", allowMultiple=True,
                                       longestPrefix=True )
   }

class PrefixLpmIpv6Cmd( PrefixCmdBaseV2 ):
   data = {
      "PREFIX_EXPR": PrefixCmdMatcher( addrType="ipv6", allowMultiple=True,
                                       longestPrefix=True )
   }

class PrefixCmdBase( CliCommand.CliCommandClass ):
   _baseSyntax = '( source | destination ) prefix'

   _baseData = { 'source': 'source',
                 'destination': 'destination',
                 'prefix': 'prefix' }

   # Set to True if each invocation of this command should first clear the previous
   # values.
   _overwrite = False

   @classmethod
   def _updatePrefix( cls, mode, args, add ):
      context = mode.getContext()
      filterType = "source" if 'source' in args else "destination"
      if add and cls._overwrite:
         context.addOrRemovePrefix( getattr( context.filter, filterType ),
                                    filterType=filterType, add=False )

      prefixes = args[ 'PREFIX' ]
      if not isinstance( prefixes, list ):
         prefixes = [ prefixes ]
      if prefixes:
         context.addOrRemovePrefix( prefixes, filterType=filterType,
                                    add=add )

   @classmethod
   def handler( cls, mode, args ):
      # Check to see if the structuredFilter contains conflicting config.
      context = mode.getContext()
      if context.isValidConfig( conflictOther ):
         cls._updatePrefix( mode, args, add=True )
         return
      attr = 'source prefix' if 'source' in args else 'destination prefix'
      mode.addError( configConflictMsg % attr )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      cls._updatePrefix( mode, args, add=False )

class PrefixIpv4Cmd( PrefixCmdBase ):
   syntax = PrefixCmdBase._baseSyntax + ' { PREFIX }'
   noOrDefaultSyntax = syntax
   data = {
      'PREFIX': IpAddrMatcher.ipPrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT,
         raiseError=True )
   }
   data.update( PrefixCmdBase._baseData )

class PrefixIpv4SingletonCmd( PrefixCmdBase ):
   syntax = PrefixCmdBase._baseSyntax + ' PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'PREFIX': IpAddrMatcher.ipPrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT,
         raiseError=True )
   }
   data.update( PrefixCmdBase._baseData )
   _overwrite = True

class PrefixIpv6Cmd( PrefixCmdBase ):
   syntax = PrefixCmdBase._baseSyntax + ' { PREFIX }'
   noOrDefaultSyntax = syntax
   data = {
      'PREFIX': Ip6AddrMatcher.ip6PrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT, raiseError=True )
   }
   data.update( PrefixCmdBase._baseData )

class PrefixIpv6SingletonCmd( PrefixCmdBase ):
   syntax = PrefixCmdBase._baseSyntax + ' PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'PREFIX': Ip6AddrMatcher.ip6PrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT, raiseError=True )
   }
   data.update( PrefixCmdBase._baseData )
   _overwrite = True

# --------------------------------------------------------------------------
# The "application-profile APP_PROFILE_NAME" command
# --------------------------------------------------------------------------
class AppProfileMatchBaseConfigCmd ( CliCommand.CliCommandClass ):
   syntax = 'application-profile APP_PROFILE_NAME'
   noOrDefaultSyntax = 'application-profile [ APP_PROFILE_NAME ]'
   _baseData = {
      'application-profile': "Configure application profile",
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      appProfiles = args.get( 'APP_PROFILE_NAME', [] )
      if not isinstance( appProfiles, list ):
         appProfiles = [ appProfiles ]
      sf = Tac.newInstance( "Classification::StructuredFilter", "" )
      for profile in appProfiles:
         sf.appProfile.add( profile )
      args[ 'APP_PROFILE_NAME' ] = sf

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      context.addAppProfileMatchToSf( args[ 'APP_PROFILE_NAME' ] )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      context.removeAppProfileMatchFromSf( args[ 'APP_PROFILE_NAME' ] )

# --------------------------------------------------------------------------
# The "next-hop group NEXTHOP_GROUP_NAME" command
# --------------------------------------------------------------------------
class NexthopGroupMatchBaseConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hop group NEXTHOP_GROUP_NAME'
   noOrDefaultSyntax = syntax
   _feature = "app"
   _baseData = {
      'next-hop': nexthopKwMatcher,
      'group': nexthopGroupKwMatcher,
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      nexthopGroupName = args[ 'NEXTHOP_GROUP_NAME' ]
      context.addNhgMatchToSf( nexthopGroupName )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      nexthopGroupName = args.get( 'NEXTHOP_GROUP_NAME' )
      context.removeNhgMatchFromSf( nexthopGroupName )

# --------------------------------------------------------------------------
# The "destination prefix self unicast" command
# --------------------------------------------------------------------------
class DestinationSelfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'destination prefix self unicast'
   noOrDefaultSyntax = 'destination prefix self ...'
   data = {
      'destination': 'Destination',
      'prefix': 'Prefix',
      'self': 'Self',
      'unicast': 'Unicast',
   }

   @staticmethod
   def adapter( mod, args, argsList ):
      selfIpUnicast = args.pop( "unicast", False )
      matchFlag = Tac.Value( "Classification::ToCpuMatch::MatchFlags" )
      if selfIpUnicast:
         matchFlag.selfIpUnicast = True
      toCpu = Tac.Value( "Classification::ToCpuMatch" )
      toCpu.matchFlag = matchFlag
      sf = Tac.newInstance( "Classification::StructuredFilter", "" )
      sf.toCpu = toCpu
      args[ "toCpuSf" ] = sf

   @classmethod
   def handler( cls, mode, args ):
      sf = args[ "toCpuSf" ]
      context = mode.getContext()
      context.addPrefixFromSf( sf )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      sf = args[ "toCpuSf" ]
      context = mode.getContext()
      context.removePrefixFromSf( sf )

# --------------------------------------------------------------------------
# The "location LOCATION_ALIAS_NAME value HEX_VALUE mask HEX_MASK" command
# --------------------------------------------------------------------------
class LocationValueMatchBaseConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'location LOCATION_ALIAS_NAME value HEX_VALUE mask HEX_MASK'
   noOrDefaultSyntax = syntax
   _baseData = {
      'location': locationKwMatcher,
      'value': 'Hex value to match',
      'HEX_VALUE': locationValueMatcher,
      'mask': 'Hex mask for value to match',
      'HEX_MASK': locationMaskMatcher,
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      locationAliasName = args[ 'LOCATION_ALIAS_NAME' ]
      matchValue = args[ 'HEX_VALUE' ]
      matchMask = args[ 'HEX_MASK' ]
      context.addLocationMatchToSf( locationAliasName, matchValue, matchMask )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      locationAliasName = args[ 'LOCATION_ALIAS_NAME' ]
      matchValue = args[ 'HEX_VALUE' ]
      matchMask = args[ 'HEX_MASK' ]
      context.removeLocationMatchFromSf( locationAliasName, matchValue, matchMask )

# --------------------------------------------------------------------------
# The "field-set integer FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetIntegerBaseConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'field-set integer FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "app"
   _integerContext = IntegerFieldSetContext
   _baseData = {
      'field-set': fieldSetConfigKwMatcher,
      'integer': integerKwMatcher,
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetIntegerName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._integerContext( **contextKwargs )

      if context.hasIntegerFieldSet( name ):
         context.copyEditFieldSet()
      else:
         context.newEditFieldSet()

      childMode = mode.childMode( context.childMode, context=context,
                                  feature=cls._feature )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._integerContext( **contextKwargs )
      if context.hasIntegerFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

   @classmethod
   def _removeFieldSet( cls, mode, name ):
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._integerContext( **contextKwargs )
      if context.hasIntegerFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

# --------------------------------------------------------------------------
# The "[remove] ( INTEGER_RANGE )" command
# --------------------------------------------------------------------------
class FieldSetIntegerConfigCmds( CliCommand.CliCommandClass ):
   syntax = '[ remove ] INTEGER_RANGE'
   data = {
      'remove': 'Remove integer range(s) from integer field-set',
      'INTEGER_RANGE': integerRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      tokenRemove = 'remove' in args
      integersToUpdate = args[ 'INTEGER_RANGE' ]
      ( integersToUpdate, errorMsg ) = expandRanges( integersToUpdate )
      if errorMsg:
         mode.addError( errorMsg )
         return
      addIntegers = not tokenRemove
      context = mode.getContext()
      if not context.canUpdateFieldSet():
         error = "Cannot change INTEGERs unless (contents) source is 'static'"
      else:
         error = context.updateFieldSet( integersToUpdate, add=addIntegers )

      if error:
         mode.addError( error )

# --------------------------------------------------------------------------
# The "field-set vlan FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetVlanBaseConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'field-set vlan FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "app"
   _vlanContext = VlanFieldSetContext
   _baseData = {
      'field-set': fieldSetConfigKwMatcher,
      'vlan': vlanKwMatcher,
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetVlanName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._vlanContext( **contextKwargs )

      if context.hasVlanFieldSet( name ):
         context.copyEditFieldSet()
      else:
         context.newEditFieldSet()

      childMode = mode.childMode( context.childMode, context=context,
                                  feature=cls._feature )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._vlanContext( **contextKwargs )
      if context.hasVlanFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

   @classmethod
   def _removeFieldSet( cls, mode, name ):
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._vlanContext( **contextKwargs )
      if context.hasVlanFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

# --------------------------------------------------------------------------
# The "[remove] ( all | VLAN )" command
# --------------------------------------------------------------------------
class FieldSetVlanConfigCmds( CliCommand.CliCommandClass ):
   syntax = '[remove] ( all | VLAN )'
   data = {
      'remove': 'Remove VLAN tag(s) from VLAN field-set',
      'all': f'All VLAN from 1-{appConstants.maxVlan - 1}',
      'VLAN': vlanTagRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      tokenRemove = 'remove' in args
      tokenAll = 'all' in args
      vlansToUpdate = args.get( 'VLAN', set() )
      addVlans = not tokenRemove
      context = mode.getContext()
      if not context.canUpdateFieldSet():
         error = "Cannot change VLANs unless (contents) source is 'static'"
      else:
         error = context.updateFieldSet( vlansToUpdate, add=addVlans,
                                         allVlans=tokenAll )
      if error:
         mode.addError( error )

# --------------------------------------------------------------------------
# The "field-set l4-port PORT_SET_NAME" command
# --------------------------------------------------------------------------
def protectedFieldSetNamesRegex( field ):
   excludePattern = ''.join( BasicCliUtil.notAPrefixOf( k )
                             for k in getProtectedFieldSetNames( field ) )
   return excludePattern + r'[A-Za-z0-9_:{}\[\]-]*'

class FieldSetL4PortBaseConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'field-set l4-port FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "app"
   _l4PortContext = L4PortFieldSetContext
   _baseData = {
      'field-set': fieldSetConfigKwMatcher,
      'l4-port': 'Layer 4 port',
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._l4PortContext( **contextKwargs )

      if context.hasL4PortFieldSet( name ):
         context.copyEditFieldSet()
      else:
         context.newEditFieldSet()

      childMode = mode.childMode( context.childMode, context=context,
                                  feature=cls._feature )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._l4PortContext( **contextKwargs )
      if context.hasL4PortFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

   @classmethod
   def _removeFieldSet( cls, mode, name ):
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._l4PortContext( **contextKwargs )
      if context.hasL4PortFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

# --------------------------------------------------------------------------
# The "[remove] ( all | PORT )" command
# --------------------------------------------------------------------------
class FieldSetL4PortConfigBase( CliCommand.CliCommandClass ):
   syntax = '[ remove ] ( all | PORT )'
   data = {
      'remove': 'Remove l4 port(s) from port set',
      'all': f'All l4 ports from 0-{appConstants.maxL4Port}',
      'PORT': portRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      context = mode.getContext()
      if not context.canUpdateFieldSet():
         error = "Cannot change L4-ports unless (contents) source is 'static'"
      else:
         addPorts = 'remove' not in args
         allPorts = 'all' in args
         portsToUpdate = args.get( 'PORT', set() )
         error = context.updateFieldSet( portsToUpdate, add=addPorts,
                                         allPorts=allPorts )
      if error:
         mode.addError( error )

class FieldSetL4PortConfigCmds( FieldSetL4PortConfigBase ):
   data = FieldSetL4PortConfigBase.data.copy()

# --------------------------------------------------------------------------
# The "field-set service FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetServiceBaseConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'field-set service FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "app"
   _serviceContext = ServiceFieldSetContext
   _baseData = {
      'field-set': fieldSetConfigKwMatcher,
      'service': 'Networking Service',
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._serviceContext( **contextKwargs )

      if context.hasServiceFieldSet( name ):
         context.copyEditFieldSet()
      else:
         context.newEditFieldSet()

      childMode = mode.childMode( context.childMode, context=context,
                                  feature=cls._feature )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._serviceContext( **contextKwargs )
      if context.hasServiceFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

   @classmethod
   def _removeFieldSet( cls, mode, name ):
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._serviceContext( **contextKwargs )
      if context.hasServiceFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

# --------------------------------------------------------------------------
# The "field-set (ipv4 | ipv6) prefix FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetIpPrefixBaseConfigCmd( CliCommand.CliCommandClass ):
   _feature = "app"
   _ipPrefixFieldSetContext = IpPrefixFieldSetContext
   _baseData = {
      'field-set': fieldSetConfigKwMatcher,
      'ipv4': 'IPv4',
      'ipv6': 'IPv6',
      'prefix': 'IP prefixes',
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      setType = args.get( 'ipv4' ) or args.get( 'ipv6' )
      contextKwargs = cls._getContextKwargs( name, setType, mode )
      context = cls._ipPrefixFieldSetContext( **contextKwargs )

      if context.hasPrefixFieldSet( name, setType ):
         context.copyEditFieldSet()
      else:
         context.newEditFieldSet()

      childMode = mode.childMode( context.childMode, context=context,
                                  feature=cls._feature )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      setType = args.get( 'ipv4' ) or args.get( 'ipv6' )
      contextKwargs = cls._getContextKwargs( name, setType, mode )
      context = cls._ipPrefixFieldSetContext( **contextKwargs )
      if context.hasPrefixFieldSet( name, setType ):
         context.delFieldSet( name, mode, fromNoHandler=True )

   @classmethod
   def _removeFieldSet( cls, mode, name, setType ):
      contextKwargs = cls._getContextKwargs( name, setType, mode )
      context = cls._ipPrefixFieldSetContext( **contextKwargs )
      if context.hasPrefixFieldSet( name, setType ):
         context.delFieldSet( name, mode, fromNoHandler=True )

# --------------------------------------------------------------------------
# The "[ ( no | remove ) ] { PREFIXES }" command
# --------------------------------------------------------------------------
class FieldSetPrefixConfigCmdsBase( CliCommand.CliCommandClass ):
   syntax = '[ remove ] { PREFIXES }'
   noOrDefaultSyntax = '{ PREFIXES }'
   _baseData = {
      'remove': 'Remove IP prefix from IP prefix set'
   }

   @staticmethod
   def handler( mode, args ):
      prefixes = args.get( 'PREFIXES' )
      add = 'remove' not in args
      context = mode.getContext()
      if not context.canUpdateFieldSet():
         mode.addError(
               "Cannot change prefixes unless (contents) source is 'static'" )
         return
      error = context.updateFieldSet( prefixes, add=add )
      if error:
         mode.addError( error )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      prefixes = args.get( 'PREFIXES' )
      context = mode.getContext()
      context.updateFieldSet( prefixes, add=False )

# --------------------------------------------------------------------------
# The "except { PREFIXES }" command
# --------------------------------------------------------------------------
class FieldSetPrefixExceptConfigCmdsBase( CliCommand.CliCommandClass ):
   syntax = 'EXCEPT_ITEMS'
   noOrDefaultSyntax = syntax

   @staticmethod
   def handler( mode, args ):
      context = mode.getContext()
      exceptPrefixes = args.get( 'EXCEPT_ITEMS' )
      if not context.canUpdateFieldSet():
         mode.addError(
               "Cannot change prefixes unless (contents) source is 'static'" )
         return
      error = context.updateFieldSet( exceptPrefixes, add=True, updateExcept=True )
      if error:
         mode.addError( error )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      context = mode.getContext()
      exceptPrefixes = args.get( 'EXCEPT_ITEMS' )
      context.updateFieldSet( exceptPrefixes, add=False, updateExcept=True )

# --------------------------------------------------------------------------
# The "except PORTS" command
# --------------------------------------------------------------------------
class FieldSetL4PortExceptConfigBase( CliCommand.CliCommandClass ):
   syntax = 'EXCEPT_ITEMS'
   data = {
      'EXCEPT_ITEMS': generateExceptExpression( portRangeMatcher,
                                                exceptSupported=True,
                                                allowMultiple=False )
   }

   @staticmethod
   def handler( mode, args ):
      context = mode.getContext()
      exceptPorts = args.get( 'EXCEPT_ITEMS', set() )
      if not isinstance( exceptPorts, set ):
         exceptPorts = exceptPorts[ 0 ]
      context.updateFieldSet( exceptPorts, add=False )

class FieldSetL4PortExceptConfigCmds( FieldSetL4PortExceptConfigBase ):
   data = FieldSetL4PortExceptConfigBase.data.copy()

# --------------------------------------------------------------------------
# The "except VLANS" command
# --------------------------------------------------------------------------
class FieldSetVlanExceptConfigCmds( CliCommand.CliCommandClass ):
   syntax = 'EXCEPT_ITEMS'
   data = {
      'EXCEPT_ITEMS': generateExceptExpression( vlanTagRangeMatcher,
                                                exceptSupported=True,
                                                allowMultiple=False )
   }

   @staticmethod
   def handler( mode, args ):
      context = mode.getContext()
      exceptVlans = args.get( 'EXCEPT_ITEMS', set() )
      if not isinstance( exceptVlans, set ):
         exceptVlans = exceptVlans[ 0 ]
      context.updateFieldSet( exceptVlans, add=False )

# --------------------------------------------------------------------------
# The "except INTEGERS" command
# --------------------------------------------------------------------------
class FieldSetIntegerExceptConfigCmds( CliCommand.CliCommandClass ):
   syntax = 'EXCEPT_ITEMS'
   data = {
      'EXCEPT_ITEMS': generateExceptExpression( integerRangeMatcher,
                                                exceptSupported=True,
                                                allowMultiple=False )
   }

   @staticmethod
   def handler( mode, args ):
      context = mode.getContext()
      exceptIntegers = args.get( 'EXCEPT_ITEMS', set() )
      if not isinstance( exceptIntegers, set ):
         exceptIntegers = exceptIntegers[ 0 ]
      ( exceptIntegers, errorMsg ) = expandRanges( exceptIntegers )
      if errorMsg:
         mode.addError( errorMsg )
         return
      context.updateFieldSet( exceptIntegers, add=False )

class NumericalRangeConfigCmdBase( CliCommand.CliCommandClass ):
   _attrName = None
   _rangeType = None
   _argListName = None

   @classmethod
   def handler( cls, mode, args ):
      if mode.context.filter.isValidConfig( conflictOther ):
         rangeSet = args.get( cls._argListName )
         cls._updateRange( mode, rangeSet )
         return
      mode.addError( configConflictMsg % cls._attrName )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      rangeSet = args.get( cls._argListName )
      if rangeSet is None:
         rangeSet = set()
      cls._updateRange( mode, rangeSet, add=False )

   @classmethod
   def _updateRange( cls, mode, rangeSet, add=True ):
      mode.context.updateRangeAttr( attrName=cls._attrName,
                                    rangeSet=rangeSet,
                                    rangeType=cls._rangeType,
                                    add=add )

# The following commands are EXEC commands to refresh field-sets defined for
# particular features. The implementation should include a token for the
# ---------------------------------------------------------------------------
# The "refresh FEATURE field-set all" command
# ---------------------------------------------------------------------------
class RefreshAllCallback:
   def __init__( self ):
      self.rollbackCallback = []
      self.statusCallback = {}
      self.cleanupCallback = []

   def registerRollbackCallback( self, rollbackFn ):
      self.rollbackCallback.append( rollbackFn )

   def registerStatusCallback( self, key, statusFn ):
      self.statusCallback[ key ] = statusFn

   def registerCleanupCallback( self, cleanupFn, version ):
      self.cleanupCallback.append( ( cleanupFn, version ) )

class FieldSetRefreshAllCmdBase( CliCommand.CliCommandClass ):
   _ipPrefixFieldSetContext = IpPrefixFieldSetContext
   _l4PortContext = L4PortFieldSetContext
   _vlanContext = VlanFieldSetContext
   _integerContext = IntegerFieldSetContext
   _macAddrContext = MacAddrFieldSetContext
   _baseData = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'field-set': fieldSetRefreshKwMatcher,
      'all': 'refresh all field set types',
   }

   @classmethod
   def _freezeProcessing( cls, freeze ):
      raise NotImplementedError

   @classmethod
   def _getFieldSetConfig( cls ):
      raise NotImplementedError

   @classmethod
   def _getIpPrefixContextKwargs( cls, fieldSetName, setType, mode=None ):
      raise NotImplementedError

   @classmethod
   def _getIpv6PrefixContextKwargs( cls, fieldSetName, setType, mode=None ):
      raise NotImplementedError

   @classmethod
   def _getVlanContextKwargs( cls, fieldSetName, mode=None ):
      raise NotImplementedError

   @classmethod
   def _getL4PortContextKwargs( cls, fieldSetName, mode=None ):
      raise NotImplementedError

   @classmethod
   def _getIntegerContextKwargs( cls, fieldSetName, mode=None ):
      raise NotImplementedError

   @classmethod
   def _getMacAddrContextKwargs( cls, fieldSetName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      callback = RefreshAllCallback()
      mode.session.sessionDataIs( 'refreshAllCallbacks', callback )

      fieldSetConfig = cls._getFieldSetConfig()

      t0( 'Pause batch processing for refresh-all' )
      cls._freezeProcessing( True )
      t0( 'Refresh ip prefix field-sets' )
      for prefixFsName in fieldSetConfig.fieldSetIpPrefix:
         contextKwargs = cls._getIpPrefixContextKwargs( prefixFsName, 'ipv4', mode )
         context = cls._ipPrefixFieldSetContext( **contextKwargs )
         context.modeIs( mode )
         if context.hasPrefixFieldSet( prefixFsName, 'ipv4' ):
            context.refreshFieldSet()
      t0( 'Refresh ipv6 prefix field-sets' )
      for prefixFsName in fieldSetConfig.fieldSetIpv6Prefix:
         contextKwargs = cls._getIpPrefixContextKwargs( prefixFsName, 'ipv6', mode )
         context = cls._ipPrefixFieldSetContext( **contextKwargs )
         context.modeIs( mode )
         if context.hasPrefixFieldSet( prefixFsName, 'ipv6' ):
            context.refreshFieldSet()
      t0( 'Refresh l4-port field-sets' )
      for fsName in fieldSetConfig.fieldSetL4Port:
         contextKwargs = cls._getL4PortContextKwargs( fsName, mode )
         context = cls._l4PortContext( **contextKwargs )
         context.modeIs( mode )
         if context.hasL4PortFieldSet( fsName ):
            context.refreshFieldSet()
      t0( 'Refresh vlan field-sets' )
      for fsName in fieldSetConfig.fieldSetVlan:
         contextKwargs = cls._getVlanContextKwargs( fsName, mode )
         context = cls._vlanContext( **contextKwargs )
         context.modeIs( mode )
         if context.hasVlanFieldSet( fsName ):
            context.refreshFieldSet()
      t0( 'Refresh integer field-sets' )
      for fsName in fieldSetConfig.fieldSetInteger:
         contextKwargs = cls._getIntegerContextKwargs( fsName, mode )
         context = cls._integerContext( **contextKwargs )
         context.modeIs( mode )
         if context.hasIntegerFieldSet( fsName ):
            context.refreshFieldSet()
      t0( 'Refresh mac field-sets' )
      for fsName in fieldSetConfig.fieldSetMacAddr:
         contextKwargs = cls._getMacAddrContextKwargs( fsName, mode )
         context = cls._macAddrContext( **contextKwargs )
         context.modeIs( mode )
         if context.hasMacAddrFieldSet( fsName ):
            context.refreshFieldSet()

      t0( 'Unpause batch processing for refresh all' )
      cls._freezeProcessing( False )
      mode.session.sessionDataIs( 'refreshAllCallbacks', None )

      needsRollback = False
      errorStr = None
      t0( 'Run field-set callbacks' )
      for k, statusFn in callback.statusCallback.items():
         for fsCallback in statusFn:
            ( isValid, issues ) = fsCallback( *k )
            issuesStr = issues.get( 'error', 'unknown' ) if issues else 'unknown'

            if not isValid:
               t0( f'Error for field-set {k[1]} of type {k[2]}' )
               errorStr = issuesStr
               needsRollback = True
               break
         if needsRollback:
            # Hit an error, no need to check other statuses.
            break
      if needsRollback:
         t0( 'Error found, start rollbacks' )
         mode.addError( f"Failed to commit 1 or more field-sets : {errorStr}" )
         for rollbackFn in callback.rollbackCallback:
            rollbackFn()
         mode.addWarning( "Rolled back 'refresh traffic-policy field-set all'" )
      else:
         t0( 'Refresh-all completed, delete previous subconfigs' )
         for ( cleanupFn, version ) in callback.cleanupCallback:
            cleanupFn( 'url', version )

# FEATURE name between the 'refresh' and 'field-set' tokens.
# ---------------------------------------------------------------------------
# The "refresh FEATURE field-set (ipv4 | ipv6) prefix FIELD_SET_NAME" command
# ---------------------------------------------------------------------------
class FieldSetIpPrefixRefreshCmdBase( CliCommand.CliCommandClass ):
   _ipPrefixFieldSetContext = IpPrefixFieldSetContext
   _baseData = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'field-set': fieldSetRefreshKwMatcher,
      'ipv4': 'IPv4',
      'ipv6': 'IPv6',
      'prefix': 'IP prefixes',
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      setType = args.get( 'ipv4' ) or args.get( 'ipv6' )
      contextKwargs = cls._getContextKwargs( name, setType, mode )
      context = cls._ipPrefixFieldSetContext( **contextKwargs )
      context.modeIs( mode )

      if context.hasPrefixFieldSet( name, setType ):
         context.refreshFieldSet()
      else:
         mode.addWarning( "{} field-set '{}' is not defined".format(
                          fieldSetTypeToStr[ setType ], name ) )

# --------------------------------------------------------------------------
# The "refresh FEATURE field-set l4-port FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetL4PortRefreshCmdBase( CliCommand.CliCommandClass ):
   _l4PortContext = L4PortFieldSetContext
   _baseData = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'field-set': fieldSetRefreshKwMatcher,
      'l4-port': 'Layer 4 port',
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._l4PortContext( **contextKwargs )
      context.modeIs( mode )

      if context.hasL4PortFieldSet( name ):
         context.refreshFieldSet()
      else:
         mode.addWarning( "{} field-set '{}' is not defined".format(
                          fieldSetTypeToStr[ context.setType ], name ) )

# --------------------------------------------------------------------------
# The "refresh FEATURE field-set vlan FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetVlanRefreshCmdBase( CliCommand.CliCommandClass ):
   _vlanContext = VlanFieldSetContext
   _baseData = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'field-set': fieldSetRefreshKwMatcher,
      'vlan': vlanKwMatcher,
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetVlanName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._vlanContext( **contextKwargs )
      context.modeIs( mode )

      if context.hasVlanFieldSet( name ):
         context.refreshFieldSet()
      else:
         mode.addWarning( "{} field-set '{}' is not defined".format(
                          fieldSetTypeToStr[ context.setType ], name ) )

# --------------------------------------------------------------------------
# The "refresh FEATURE field-set integer FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetIntegerRefreshCmdBase( CliCommand.CliCommandClass ):
   _integerContext = IntegerFieldSetContext
   _baseData = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'field-set': fieldSetRefreshKwMatcher,
      'integer': integerKwMatcher,
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetIntegerName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._integerContext( **contextKwargs )
      context.modeIs( mode )

      if context.hasIntegerFieldSet( name ):
         context.refreshFieldSet()
      else:
         mode.addWarning( "{} field-set '{}' is not defined".format(
                          fieldSetTypeToStr[ context.setType ], name ) )

# --------------------------------------------------------------------------
# The "refresh FEATURE field-set mac FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetMacAddrRefreshCmdBase( CliCommand.CliCommandClass ):
   _macAddrContext = MacAddrFieldSetContext
   _baseData = {
      'refresh': CliToken.Refresh.refreshMatcherForExec,
      'field-set': fieldSetRefreshKwMatcher,
      'mac': macKwMatcher,
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetMacAddrName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._macAddrContext( **contextKwargs )
      context.modeIs( mode )

      if context.hasMacAddrFieldSet( name ):
         context.refreshFieldSet()
      else:
         mode.addWarning( "{} field-set '{}' is not defined".format(
                          fieldSetTypeToStr[ context.setType ], name ) )

# End of 'refresh FEATURE field-set' base commands.

#--------------------------------------------------
# The "ip length LENGTH" command for traffic-policy
#--------------------------------------------------
class IpLengthConfigCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'length'
   _rangeType = 'Classification::PacketLengthRange'
   _argListName = 'LENGTH'

   syntax = '''ip length LENGTH'''
   noOrDefaultSyntax = '''ip length [ LENGTH ]'''
   data = {
      'ip': 'IP',
      'length': 'Configure ip length match criteria',
      'LENGTH': ipLengthRangeMatcher
   }

# --------------------------------------------------------------------------
# The "[ ( no | default ) ] source { static | bgp | URL | and-results }" command
# --------------------------------------------------------------------------
class SourceMatcher( CliCommand.CliExpressionFactory ):
   """
   Generic source matcher to be used by all field-sets. Sources may vary
   by field-set type and this is accounted for by ctor flags for the matcher.
   """
   def __init__( self, supportsBgp, supportsUrl, supportsController=False,
         supportsIntersect=False, fieldSetMatcher=None, isRemove=False ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.tokens = set()
      self.isRemove = isRemove
      if not isRemove:
         self.tokens.add( 'static' )
      if supportsBgp:
         self.tokens.add( 'bgp' )
      if supportsController:
         self.tokens.add( 'controller' )
      if supportsUrl:
         self.tokens.add( 'URL' )
         self.tokens.add( '( URL_NETWORK [ VRF_NAME ] )' )
      if supportsIntersect:
         self.tokens.add( '( and-results { FIELD_SET } )' )
         self.fieldSetMatcher = fieldSetMatcher

   def generate( self, name ):
      class SourceExpression( CliCommand.CliExpression ):
         expression = "source"
         _tokens = f" ({'|'.join(self.tokens)})"
         if self.isRemove and self.tokens:
            expression += _tokens.replace( '(', '[(' ).replace( ')', ')]' )
         elif self.tokens:
            expression += _tokens
         else:
            expression += " ..."
         data = {
            'source': 'Source',
         }
         if 'static' in self.tokens:
            data[ 'static' ] = 'Static configuration'
         if 'bgp' in self.tokens:
            data[ 'bgp' ] = 'BGP'
         if 'controller' in self.tokens:
            data[ 'controller' ] = 'Controller to push field-set entries'
         if 'URL' in self.tokens:
            data[ 'URL' ] = fieldSetUrlLocalMatcher
            data[ 'URL_NETWORK' ] = fieldSetUrlNetworkMatcher
            data[ 'VRF_NAME' ] = VrfExprFactory( helpdesc="VRF instance name" )
         if 'and-results' in expression:
            data[ 'and-results' ] = 'AND Operation'
            data[ 'FIELD_SET' ] = self.fieldSetMatcher
         @staticmethod
         def adapter( mode, args, argList ):
            static = 'static' in args
            bgp = 'bgp' in args
            controller = 'controller' in args
            url = args.get( 'URL' ) or args.get( 'URL_NETWORK' )
            intersect = 'and-results' in args
            if static:
               args[ 'SOURCE' ] = ContentsSource.cli
            elif url:
               context = mode.getContext()
               if context.urlFieldSetsOnOwnMount():
                  args[ 'SOURCE' ] = ContentsSource.url
               else:
                  args[ 'SOURCE' ] = ContentsSource.cli
               args[ 'URL' ] = str( url )
            elif bgp:
               args[ 'SOURCE' ] = ContentsSource.bgp
            elif controller:
               args[ 'SOURCE' ] = ContentsSource.controller
            elif intersect:
               args[ 'SOURCE' ] = ContentsSource.intersect
            else:
               # Default is CLI
               args[ 'SOURCE' ] = ContentsSource.cli
      return SourceExpression

class FieldSetSourceConfigCmdBase( CliCommand.CliCommandClass ):
   syntax = 'SOURCE_EXPR'
   noOrDefaultSyntax = 'REMOVE_SOURCE_EXPR'
   _field = ""

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      source = args.get( 'SOURCE' )
      url = args.get( 'URL', '' )
      vrfName = args.get( 'VRF_NAME', '' )
      intersect = args.get( 'and-results', '' )
      fieldSetNames = args.get( 'FIELD_SET', [] )
      error = context.validateSource( source, url, cls._field )
      if error:
         mode.addError( error )
         return
      context.updateSource( source )
      if url:
         context.setUrlConfig( url, vrfName )
      elif intersect:
         context.addIntersectFieldSets( fieldSetNames )
      else:
         context.removeUrlConfig()

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      context.updateSource( args.get( 'SOURCE' ) )
      if args.get( 'SOURCE' ) == "intersect":
         fieldSetNames = args.get( 'FIELD_SET', [] )
         context.delIntersectFieldSets( fieldSetNames )
      context.removeUrlConfig()

class FieldSetLimitExpr( CliCommand.CliExpressionFactory ):
   def __init__( self, isExpanded, maxValue=FieldSetLimit.null - 1 ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.isExpanded = isExpanded
      self.maxValue = maxValue

   def generate( self, name ):
      class FieldSetLimitExpression( CliCommand.CliExpression ):
         _baseExpr = 'limit entries'
         if self.isExpanded:
            _baseExpr += ' expanded'
         expression = f'{_baseExpr} LIMIT'
         data = {
            'limit': 'Limit configuration within a field set',
            'entries': 'Limit the number of entries',
            'expanded': 'Limit the number of entries of expanded range',
            'LIMIT': generateLimitEntryMatcher( self.maxValue ),
         }

         @staticmethod
         def adapter( mode, args, argList ):
            args[ 'EXPANDED' ] = 'expanded' in args
      return FieldSetLimitExpression

class FieldSetLimitConfigBaseCmd( CliCommand.CliCommandClass ):
   syntax = 'LIMIT_EXPR'
   noOrDefaultSyntax = 'limit ...'

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      limit = args.get( "LIMIT" )
      context.setLimitEntries( limit )
      error, warning = context.limitEntriesValid()
      if error:
         # Revert limit application and print error
         context.setLimitEntries( None )
         mode.addError( error )
      if warning:
         mode.addWarning( warning )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      context.setLimitEntries( None )

class FieldSetLimitConfigCmd( FieldSetLimitConfigBaseCmd ):
   data = {
      'LIMIT_EXPR': FieldSetLimitExpr( isExpanded=False ),
   }

class FieldSetRangeLimitConfigCmd( FieldSetLimitConfigBaseCmd ):
   data = {
      'LIMIT_EXPR': FieldSetLimitExpr( isExpanded=True ),
   }

class FieldSetL4RangeLimitConfigCmd( FieldSetLimitConfigBaseCmd ):
   data = {
      'LIMIT_EXPR': FieldSetLimitExpr( isExpanded=True,
                                       maxValue=appConstants.maxL4Port ),
   }

class FieldSetVlanRangeLimitConfigCmd( FieldSetLimitConfigBaseCmd ):
   data = {
      'LIMIT_EXPR': FieldSetLimitExpr( isExpanded=True,
                                      maxValue=appConstants.maxVlan - 1 ),
   }

class FieldSetIpPrefixConfigBase( FieldSetPrefixConfigCmdsBase ):
   data = FieldSetPrefixConfigCmdsBase._baseData | {
      'PREFIXES': IpAddrMatcher.ipPrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT,
         maskKeyword=False,
         raiseError=True )
   }

class FieldSetIpPrefixConfigCmds( FieldSetIpPrefixConfigBase ):
   data = FieldSetIpPrefixConfigBase.data.copy()

class FieldSetIpv6PrefixConfigCmds( FieldSetPrefixConfigCmdsBase ):
   data = FieldSetPrefixConfigCmdsBase._baseData | {
      'PREFIXES': Ip6AddrMatcher.ip6PrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT, raiseError=True )
   }

class FieldSetIpPrefixExceptConfigCmds( FieldSetPrefixExceptConfigCmdsBase ):
   data = {
      'EXCEPT_ITEMS': generateExceptExpression( IpAddrMatcher.ipPrefixExpr(
         'Prefix address', 'Prefix mask', 'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT, raiseError=True ),
         exceptSupported=True )
   }

class FieldSetIpv6PrefixExceptConfigCmds( FieldSetPrefixExceptConfigCmdsBase ):
   data = {
      'EXCEPT_ITEMS': generateExceptExpression( Ip6AddrMatcher.ip6PrefixExpr(
         'Prefix address',
         'Prefix mask',
         'Prefix',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT, raiseError=True ),
         exceptSupported=True )
   }

class ProtocolMixin( CliCommand.CliCommandClass ):
   """
   This class defines numerous helper methods for "protocol" commands. These
   are commands like 'protocol udp source port 10-50' as well as the field-set
   variant 'protocol tcp source port field-set FIELD_SET_NAME
   """
   _protoArgsListName = 'PROTOCOL'
   _tcpUdpArgsListName = 'TCP_UDP'
   _tcpFlagArgsListName = 'tcp'
   _sportFieldSetAttr = ""
   _dportFieldSetAttr = ""
   # If this is True, all commands _overwrite_ whatever configuration is currently
   # present.
   _overwrite = False

   @classmethod
   def _maybeHandleErrors( cls, mode, args, proto, source=False, destination=False,
                           fieldSet=False ):
      hasError = False
      context = mode.getContext()
      if not context.isValidConfig( conflictOther ):
         mode.addError( configConflictMsg % 'protocol PROTOCOL' )
         hasError = True
      if not context.isValidConfig( conflictMatchAllFragments ) or not \
         context.isValidConfig( conflictFragmentOffset ):
         mode.addError( invalidL4PortConflictMsg )
         hasError = True
      if not hasError:
         # No error, safe to keep changes
         return
      # Errors found, revert changes made
      cls._updateProtoAndPort( mode, args, proto, source, destination,
                               fieldSet=fieldSet, add=False )

   @classmethod
   def _maybeHandleErrorsOred( cls, mode, args, proto, source=False,
                               destination=False, fieldSet=False ):
      hasError = False
      context = mode.getContext()
      if not context.isValidConfig( conflictOther ):
         mode.addError( configConflictMsg % 'protocol PROTOCOL' )
         hasError = True
      if not context.isValidConfig( conflictMatchAllFragments ) or not \
         context.isValidConfig( conflictFragmentOffset ):
         mode.addError( invalidL4PortConflictMsg )
         hasError = True
      if not hasError:
         # No error, safe to keep changes
         return
      # Errors found, revert changes made
      cls._updateOredProtoAndPort( mode, args, proto, source, destination,
                                   fieldSet=fieldSet, add=False )

   @classmethod
   def _updateProtoAndPort( cls, mode, args, proto,
                            source=False, destination=False,
                            flags=False, fieldSet=False, add=True ):
      context = mode.getContext()
      if proto:
         if cls._protoArgsListName in args:
            argListName = cls._protoArgsListName
         elif cls._tcpFlagArgsListName in args:
            argListName = cls._tcpFlagArgsListName
         else:
            argListName = cls._tcpUdpArgsListName
         if add or ( not add and not source and not destination and not flags ):
            cls._updateProtocol( mode, args, argListName, add=add )

      if flags:
         cls._updateTcpFlags( mode, args, add=add )

      if not context.getProto():
         # all ports removed, this will remove all sport/dport/icmp etc
         return

      # When only proto is removed, no need to update sport/dport
      if source:
         if fieldSet:
            cls._updatePortFieldSet( mode, args, cls._sportFieldSetAttr,
                                     'SRC_FIELD_SET_NAME', add=add )
         else:
            cls._updatePort( mode, args, 'sport', 'SPORT', add=add )

      if destination:
         if fieldSet:
            cls._updatePortFieldSet( mode, args, cls._dportFieldSetAttr,
                                     'DST_FIELD_SET_NAME', add=add )
         else:
            cls._updatePort( mode, args, 'dport', 'DPORT', add=add )

      # Remove protocol once all additional fields have been removed
      if proto and ( source or destination or flags ):
         protocolRangeSet = args.get( argListName, set() )
         context.maybeUpdateProto( protocolRangeSet )

   @classmethod
   def _updateOredProtoAndPort( cls, mode, args, proto,
                                source=False, destination=False,
                                flags=False, fieldSet=False, add=True ):
      clearPrev = add and cls._overwrite
      context = mode.getContext()
      if proto:
         if cls._protoArgsListName in args:
            argListName = cls._protoArgsListName
         elif cls._tcpFlagArgsListName in args:
            argListName = cls._tcpFlagArgsListName
         else:
            argListName = cls._tcpUdpArgsListName
         # only need to update protocol when it's proto add cmd or pure proto remove.
         if add or ( not add and not source and not destination and not flags ):
            cls._updateOredProtocol( mode, args, argListName, add=add )

      if flags:
         isUpdatingIgnored = cls._updateTcpFlags( mode, args, add=add )
         # If updating TCP flag config is ignored, skip updating l4 port config.
         if isUpdatingIgnored:
            return

      if not context.getProto():
         return

      isIgnored = True
      if source or destination:
         protoRangeSet = args.get( argListName, set() )
         if fieldSet:
            sportFieldSet = args.get( 'SRC_FIELD_SET_NAME', set() )
            dportFieldSet = args.get( 'DST_FIELD_SET_NAME', set() )
            if add:
               context.addOredPortFieldSetAttr( protoRangeSet, sportFieldSet,
                                                dportFieldSet )
            else:
               isIgnored = context.removeOredPortFieldSetAttr(
                  protoRangeSet, sportFieldSet, dportFieldSet )
         else:
            sportRangeSet = args.get( 'SPORT', set() )
            dportRangeSet = args.get( 'DPORT', set() )
            if add:
               context.addOredPortRangeAttr( protoRangeSet, sportRangeSet,
                                             dportRangeSet, clearPrev=clearPrev )
            else:
               isIgnored = context.removeOredPortRangeAttr(
                  protoRangeSet, sportRangeSet, dportRangeSet, clearPrev=clearPrev )
         # When removing the TCP flags together with l4 ports, if removing l4 ports
         # is ignored, then add back the TCP flags.
         if flags and not add and isIgnored:
            cls._updateTcpFlags( mode, args, add=True )
      # Remove protocol once all additional fields have been removed
      if proto and ( source or destination or flags ):
         context.maybeUpdateProto( protoRangeSet )

   @classmethod
   def _updateProtocol( cls, mode, args, argListName, add=True ):
      rangeType = 'Classification::ProtocolRange'
      context = mode.getContext()
      if add and cls._overwrite:
         originalSet = numericalRangeToSet( context.getProto() )
         context.updateRangeAttr( attrName='proto', rangeSet=originalSet,
                                  rangeType=rangeType, add=False )
      # do not remove protocol when removing tcp flags
      protocolRangeSet = args.get( argListName, set() )
      context.updateRangeAttr( attrName='proto',
                               rangeType=rangeType,
                               rangeSet=protocolRangeSet,
                               add=add )

   @classmethod
   def _updateOredProtocol( cls, mode, args, argListName, add=True ):
      rangeType = 'Classification::ProtocolRange'
      context = mode.getContext()
      if add and cls._overwrite:
         originalSet = numericalRangeToSet( context.getProto() )
         context.updateRangeAttr( attrName='proto', rangeSet=originalSet,
                                  rangeType=rangeType, add=False )
      protocolRangeSet = args.get( argListName, set() )
      fieldKeywords = [ 'source', 'destination', 'flags', 'type' ]
      protoWithOtherField = any( kw in args for kw in fieldKeywords )
      context.updateOredProtocolAttr( rangeType=rangeType, rangeSet=protocolRangeSet,
                                      protoWithOtherField=protoWithOtherField,
                                      add=add )

   @classmethod
   def _updateTcpFlags( cls, mode, args, add=True ):
      context = mode.getContext()
      if not args.get( 'flags' ):
         return True
      return context.updateTcpFlags( context.filter, args.get( "FLAGS_EXPR" ),
                                     add=add )

   @classmethod
   def _updatePortFieldSet( cls, mode, args, attrName, argListName, add=True ):
      protoRangeSet = args.get( cls._tcpUdpArgsListName, set() ) or \
                      args.get( cls._tcpFlagArgsListName, set() )
      fieldSetNames = args.get( argListName, set() )
      context = mode.getContext()
      context.updatePortFieldSetAttr( attrName, fieldSetNames, add=add,
                                      protoSet=protoRangeSet )

   @classmethod
   def _updatePort( cls, mode, args, attrName, argListName, add=True ):
      clearPrev = add and cls._overwrite
      # If the protocol set is either TCP or UDP, or both -- simply add the
      # port qualifier.
      protoRangeSet = args.get( cls._tcpUdpArgsListName, set() ) or \
                      args.get( "tcp", set() )
      portRangeSet = args.get( argListName, set() )
      context = mode.getContext()
      context.updatePortRangeAttr( attrName, protoRangeSet,
                                   portRangeSet, add=add, clearPrev=clearPrev )

#--------------------------------------------------------------------------------
# The "protocol icmp | icmpv6 type TYPE code all" command for traffic-policy
#--------------------------------------------------------------------------------
class ProtocolIcmpConfigBase( ProtocolMixin ):
   _icmpName = 'icmp'
   _typeArgsListName = 'TYPE'

   _baseData = {
      'protocol': 'Protocol',
      'type': 'Configure ICMP type',
      'code': 'Configure ICMP code',
      'all': 'Configure ALL ICMP codes'
   }

   @classmethod
   def handler( cls, mode, args ):
      args.pop( cls._icmpName )
      icmpValue = icmpProtocols[ cls._icmpName ][ 0 ]
      args[ cls._icmpName ] = { icmpValue }
      cls._updateOredProtocol( mode, args, argListName=cls._icmpName, add=True )
      context = mode.getContext()
      icmpTypeSet = args[ cls._typeArgsListName ]
      if icmpTypeSet:
         rangeType = 'Classification::IcmpTypeRange'
         context.updateIcmpRangeAttr( icmpValue, rangeType, icmpTypeSet, add=True )
      # cannot have protocols defined along with service field-sets and vice-versa
      context.clearServiceFieldSet()

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      args.pop( cls._icmpName )
      icmpValue = icmpProtocols[ cls._icmpName ][ 0 ]
      args[ cls._icmpName ] = { icmpValue }
      context = mode.getContext()
      icmpTypeSet = args.get( cls._typeArgsListName, set() )
      rangeType = 'Classification::IcmpTypeRange'
      context.updateIcmpRangeAttr( icmpValue, rangeType, icmpTypeSet, add=False )
      # Remove protocol once all additional fields have been removed
      protocolRangeSet = args.get( cls._icmpName, set() )
      context.maybeUpdateProto( protocolRangeSet )

class ProtocolIcmpV4ConfigCmd( ProtocolIcmpConfigBase ):
   syntax = 'protocol icmp type TYPE code all'
   noOrDefaultSyntax = ( 'protocol icmp type [ TYPE [ code all ] ]' )
   data = {
      'icmp': icmpV4KwMatcher,
      'TYPE': icmpV4TypeRangeExpr,
   }
   data.update( ProtocolIcmpConfigBase._baseData )

class ProtocolIcmpV6ConfigCmd( ProtocolIcmpConfigBase ):
   _icmpName = 'icmpv6'

   syntax = 'protocol icmpv6 type TYPE code all'
   noOrDefaultSyntax = ( 'protocol icmpv6 type [ TYPE [ code all ] ]' )
   data = {
      'icmpv6': icmpV6KwMatcher,
      'TYPE': icmpV6TypeRangeExpr,
   }
   data.update( ProtocolIcmpConfigBase._baseData )

#--------------------------------------------------------------------------------
# The "protocol icmp | icmpv6 type TYPE code CODE" command for traffic-policy
#--------------------------------------------------------------------------------
class ProtocolIcmpTypeCodeConfigBase( ProtocolMixin ):
   _icmpName = 'icmp'
   _typeArgsListName = 'TYPE'
   _codeArgsListName = 'CODE'

   _baseData = {
      'protocol': 'Protocol',
      'type': 'Configure ICMP type',
      'code': 'Configure ICMP code',
   }

   @classmethod
   def handler( cls, mode, args ):
      args.pop( cls._icmpName )
      icmpValue = icmpProtocols[ cls._icmpName ][ 0 ]
      args[ cls._icmpName ] = { icmpValue }
      cls._updateOredProtocol( mode, args, argListName=cls._icmpName, add=True )
      context = mode.getContext()
      icmpTypeValue = args.get( cls._typeArgsListName )
      icmpCodeSet = args.get( cls._codeArgsListName )
      if icmpTypeValue and icmpCodeSet:
         typeRangeType = 'Classification::IcmpTypeRange'
         codeRangeType = 'Classification::IcmpCodeRange'
         context.addIcmpTypeCodeRangeAttr( icmpValue, typeRangeType, icmpTypeValue,
                                           codeRangeType, icmpCodeSet )
      # cannot have protocols defined along with service field-sets and vice-versa
      context.clearServiceFieldSet()

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      args.pop( cls._icmpName )
      icmpValue = icmpProtocols[ cls._icmpName ][ 0 ]
      args[ cls._icmpName ] = { icmpValue }
      context = mode.getContext()
      icmpTypeValue = args.get( cls._typeArgsListName )
      icmpCodeSet = args.get( cls._codeArgsListName, set() )
      if icmpTypeValue:
         typeRangeType = 'Classification::IcmpTypeRange'
         codeRangeType = 'Classification::IcmpCodeRange'
         context.removeIcmpTypeCodeRangeAttr( icmpValue, typeRangeType,
                                              icmpTypeValue, codeRangeType,
                                              icmpCodeSet )

class ProtocolIcmpV4TypeCodeConfigCmd( ProtocolIcmpTypeCodeConfigBase ):
   syntax = 'protocol icmp type TYPE code CODE'
   noOrDefaultSyntax = 'protocol icmp type TYPE code [ CODE ]'

   data = {
      'icmp': icmpV4KwMatcher,
      'TYPE': icmpV4TypeSingleExpr,
      'CODE': icmpV4CodeExpr
   }
   data.update( ProtocolIcmpTypeCodeConfigBase._baseData )

class ProtocolIcmpV6TypeCodeConfigCmd( ProtocolIcmpTypeCodeConfigBase ):
   _icmpName = 'icmpv6'

   syntax = 'protocol icmpv6 type TYPE code CODE'
   noOrDefaultSyntax = 'protocol icmpv6 type TYPE code [ CODE ]'
   data = {
      'icmpv6': icmpV6KwMatcher,
      'TYPE': icmpV6TypeSingleExpr,
      'CODE': icmpV6CodeExpr
   }
   data.update( ProtocolIcmpTypeCodeConfigBase._baseData )

#  ProtocolBase is to implement the non-ORed L4 ports configuration.
#------------------------------------------------------------------------------------
# The "protocol PROTOCOL | (tcp [ flags [ not ] TCP_FLAGS ] | udp) source port SPORT
#      destination port DPORT" command for traffic-policy
#------------------------------------------------------------------------------------

class ProtocolBase( ProtocolMixin ):
   syntax = '''
      protocol ( PROTOCOL
               | ( ( TCP_UDP | FLAGS_EXPR )
                     ( ( source port SPORT [ destination port DPORT ] )
                     | ( destination port DPORT ) ) ) )'''
   noOrDefaultSyntax = '''
      protocol [ PROTOCOL
               | ( ( TCP_UDP | FLAGS_EXPR )
                     ( ( source port [ SPORT [ destination port DPORT ] ] )
                     | ( destination port [ DPORT ] ) ) ) ]'''

   _baseData = {
      'protocol': 'Protocol',
      'TCP_UDP': tcpUdpProtoExpr,
      'port': portKwNode,
      'source': 'Source',
      'SPORT': portExpression( 'SPORT' ),
      'destination': 'Destination',
      'DPORT': portExpression( 'DPORT' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      hasProto = ( args.get( cls._protoArgsListName ) or
                   args.get( cls._tcpUdpArgsListName ) or
                   args.get( cls._tcpFlagArgsListName ) )
      assert hasProto, args
      hasSource = 'source' in args
      hasDestination = 'destination' in args
      hasFlags = 'flags' in args
      cls._updateProtoAndPort( mode, args, hasProto,
                               hasSource, hasDestination, flags=hasFlags, add=True )
      cls._maybeHandleErrors( mode, args, hasProto, hasSource, hasDestination )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      hasProto = ( args.get( cls._protoArgsListName ) or
                   args.get( cls._tcpUdpArgsListName ) or
                   args.get( cls._tcpFlagArgsListName ) )
      hasSource = 'source' in args
      hasDestination = 'destination' in args
      hasFlags = 'flags' in args
      if not hasProto and not hasSource and not hasDestination:
         # 'no protocol' removes all protocols
         hasProto = True
      cls._updateProtoAndPort( mode, args, hasProto,
                               hasSource, hasDestination, flags=hasFlags, add=False )

#------------------------------------------------------------------------------------
# The "protocol PROTOCOL | (tcp [ flags [ not ] TCP_FLAGS ] | udp) source port
#      (SPORT | all) destination port (DPORT | all)" command for traffic-policy
#------------------------------------------------------------------------------------
class ProtocolOredBase( ProtocolMixin ):
   """
   ProtocolOredBase is to implement the ORed L4 ports configuration. e.g.,
      (config) protocol tcp source port 10 destination port 20
      (config) protocol tcp source port 30 destination port 40
   The resulting config is:
      (protocol tcp source port 10 destination port 20) OR
      (protocol tcp source port 30 destination port 40)
   """
   syntax = '''
protocol ( PROTOCOL
         | ( ( TCP_UDP | FLAGS_EXPR )
               ( ( source port SPORT [ destination port ( DPORT | all ) ] )
               | ( [ source port all ] destination port DPORT ) ) ) )'''
   noOrDefaultSyntax = '''
protocol [ PROTOCOL
         | ( ( TCP_UDP | FLAGS_EXPR )
               ( ( source port [ SPORT [ destination port ( DPORT | all ) ] ] )
               | ( source port all destination port DPORT )
               | ( destination port [ DPORT ] ) ) ) ]'''
   _baseData = {
      'protocol': 'Protocol',
      'TCP_UDP': tcpUdpProtoExpr,
      'port': portKwNode,
      'source': 'Source',
      'SPORT': portExpression( 'SPORT' ),
      'destination': 'Destination',
      'DPORT': portExpression( 'DPORT' ),
      'all': 'Match all ports',
      }

   @classmethod
   def handler( cls, mode, args ):
      hasProto = ( args.get( cls._protoArgsListName ) or
                   args.get( cls._tcpUdpArgsListName ) or
                   args.get( cls._tcpFlagArgsListName ) )
      if not hasProto and not args:
         return
      hasSource = 'source' in args
      hasDestination = 'destination' in args
      hasFlags = 'flags' in args
      context = mode.getContext()
      cls._updateOredProtoAndPort( mode, args, hasProto, hasSource, hasDestination,
                                   flags=hasFlags, add=True )
      cls._maybeHandleErrorsOred( mode, args, hasProto, hasSource, hasDestination )
      # cannot have protocols defined along with service field-sets and vice-versa
      context.clearServiceFieldSet()

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      hasProto = ( args.get( cls._protoArgsListName ) or
                   args.get( cls._tcpUdpArgsListName ) or
                   args.get( cls._tcpFlagArgsListName ) )
      hasSource = 'source' in args
      hasDestination = 'destination' in args
      hasFlags = 'flags' in args
      if not hasProto and not hasSource and not hasDestination:
         # 'no protocol' removes all protocols
         hasProto = True
      cls._updateOredProtoAndPort( mode, args, hasProto, hasSource, hasDestination,
                                   flags=hasFlags, add=False )

#------------------------------------------------------------------------------------
# The "protocol (tcp [ flags [ not ] TCP_FLAGS ] | udp) source port field-set
# FIELD_SET destination port field-set FIELD_SET" command for traffic-policy
#------------------------------------------------------------------------------------
class ProtocolFieldSetBaseCmd( ProtocolMixin ):
   syntax = '''
protocol ( TCP_UDP | FLAGS_EXPR )
           ( ( source port field-set SRC_FIELD_SET_NAME
                      [ destination port field-set DST_FIELD_SET_NAME ] )
           | ( destination port field-set DST_FIELD_SET_NAME ) )'''
   noOrDefaultSyntax = '''
protocol ( TCP_UDP | FLAGS_EXPR )
           ( ( source port field-set [ SRC_FIELD_SET_NAME ]
                      [ destination port field-set [ DST_FIELD_SET_NAME ] ] )
           | ( destination port field-set [ DST_FIELD_SET_NAME ] ) )'''
   _baseData = {
      'protocol': 'Protocol',
      'TCP_UDP': tcpUdpProtoExpr,
      'field-set': fieldSetKwNode,
      'port': portKwNode,
      'source': 'Source',
      'destination': 'Destination',
   }

   @classmethod
   def _handleInternal( cls, mode, args, updateProtoAndPort, maybeHandleErrors ):
      hasProto = ( args.get( cls._tcpUdpArgsListName ) or
                   args.get( cls._tcpFlagArgsListName ) )
      if not hasProto and not args:
         return
      hasSource = 'source' in args
      hasDestination = 'destination' in args
      hasFlags = 'flags' in args
      updateProtoAndPort( mode, args, hasProto, hasSource, hasDestination,
                          flags=hasFlags, fieldSet=True, add=True )
      maybeHandleErrors( mode, args, hasProto, hasSource, hasDestination,
                         fieldSet=True )

   @classmethod
   def handler( cls, mode, args ):
      cls._handleInternal( mode, args, cls._updateProtoAndPort,
                           cls._maybeHandleErrors )

   @classmethod
   def _noOrDefHandleInternal( cls, mode, args, updateProtoAndPort ):
      hasProto = ( args.get( cls._tcpUdpArgsListName ) or
                   args.get( cls._tcpFlagArgsListName ) )
      hasSource = 'source' in args
      hasDestination = 'destination' in args
      hasFlags = 'flags' in args
      if not hasProto and not hasSource and not hasDestination:
         # 'no protocol' removes all protocols
         hasProto = True
      updateProtoAndPort( mode, args, hasProto, hasSource, hasDestination,
                          flags=hasFlags, fieldSet=True, add=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      cls._noOrDefHandleInternal( mode, args, cls._updateProtoAndPort )

#------------------------------------------------------------------------------------
# The "protocol (tcp [ flags [ not ] TCP_FLAGS ] | udp) source port field-set
# FIELD_SET destination port field-set FIELD_SET" command for traffic-policy
#------------------------------------------------------------------------------------
class ProtocolFieldSetOredBaseCmd( ProtocolFieldSetBaseCmd ):
   """
   ProtocolFieldSetOredBaseCmd is to implement the ORed L4 ports field-set
   configuration. e.g.,
      (config) protocol tcp source port field-set fs1 destination port field-set fs2
      (config) protocol tcp source port field-set fs3 destination port field-set fs4
   The resulting config is:
      (protocol tcp source port field-set fs1 destination port field-set fs2) OR
      (protocol tcp source port field-set fs3 destination port field-set fs4)
   """

   @classmethod
   def handler( cls, mode, args ):
      cls._handleInternal( mode, args, cls._updateOredProtoAndPort,
                           cls._maybeHandleErrorsOred )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      cls._noOrDefHandleInternal( mode, args, cls._updateOredProtoAndPort )

#--------------------------------------------------
# The "fragment" command for traffic-policy
#--------------------------------------------------
class MatchAllFragmentConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''fragment'''
   noOrDefaultSyntax = syntax
   data = {
     'fragment': 'fragment'
   }

   @classmethod
   def _maybeHandleErrors( cls, mode, args ):
      hasError = False
      context = mode.getContext()
      if not context.isValidConfig( conflictMatchAllFragments ):
         mode.addError( invalidFragmentConflictMsg )
         hasError = True
      if not hasError:
         return
      context.clearFragment()

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      context.updateFragmentType( matchAll )
      cls._maybeHandleErrors( mode, args )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      context.clearFragment()

#--------------------------------------------------------
# The "fragment offset OFFSET" command for traffic-policy
#--------------------------------------------------------
class FragmentOffsetConfigCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'fragmentOffset'
   _rangeType = 'Classification::FragmentOffsetRange'
   _argListName = 'OFFSET'

   syntax = '''fragment offset OFFSET'''
   noOrDefaultSyntax = '''fragment offset [ OFFSET ]'''
   data = {
      'fragment': 'fragment',
      'offset': 'Offset keyword',
      'OFFSET': fragOffsetRangeMatcher
   }

   @classmethod
   def _maybeHandleErrors( cls, mode, args ):
      hasError = False
      context = mode.getContext()
      if not context.isValidConfig( conflictFragmentOffset ):
         mode.addError( invalidFragOffsetConflictMsg )
         hasError = True
      if not hasError:
         return
      context.clearFragment()

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      context.updateFragmentType( matchOffset )
      super().handler( mode, args )
      cls._maybeHandleErrors( mode, args )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      super().noOrDefaultHandler( mode, args )
      context = mode.getContext()
      context.maybeDelFragment()

#--------------------------------------------------
# The "ip options" command for traffic-policy
#--------------------------------------------------
class IpOptionsConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''ip options'''
   noOrDefaultSyntax = syntax
   data = {
      'ip': 'IP',
      'options': 'Match packets with IPv4 options',
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      context.updateMatchIpOptions( add=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      context.updateMatchIpOptions( add=False )

#-------------------------------------------------------
# The "protocol <protocol name or value>" match command
#-------------------------------------------------------
class ProtocolIPv4SingleMatchConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( 'protocol PROTOCOL' )
   noOrDefaultSyntax = ( 'protocol [ PROTOCOL ]' )

   data = {
      'protocol': 'Protocol',
      'PROTOCOL': ipv4ProtoSingleExpr,
   }

   @staticmethod
   def handler( mode, args ):
      sf = args.get( 'PROTO_FIELDS_SF' )
      context = mode.getContext()
      context.updateProtoRangeFromSf( sf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      context = mode.getContext()
      context.clearProtoRangeFromSf()

class VlanTagConfigCmdBase( CliCommand.CliCommandClass ):
   syntax = 'VLANTAG_EXPR'
   noOrDefaultSyntax = syntax

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      if 'VLANTAG_SF' not in args:
         return
      vlanTagSf = args[ 'VLANTAG_SF' ]
      context.addVlanTagFromSf( vlanTagSf )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      if 'VLANTAG_SF' not in args:
         return
      vlanTagSf = args[ 'VLANTAG_SF' ]
      context.removeVlanTagFromSf( vlanTagSf )

class VlanTagAdapterBase( CliCommand.CliExpression ):
   @staticmethod
   def adapter( mode, args, argList ):
      vlanTagRangeSet = args.get( 'VLANTAG', set() )
      vlanTagFsSet = args.get( 'VLANTAG_FIELD_SET', set() )
      innerVlanTagRange = args.get( 'INNERVLANTAG', set() )
      if ( vlanTagRangeSet and vlanTagFsSet ) or \
         ( not vlanTagRangeSet and not vlanTagFsSet ):
         return
      sf = Tac.newInstance( "Classification::StructuredFilter", "" )
      newVlanRangeSet = Tac.Value( "Classification::VlanRangeSet" )
      if vlanTagRangeSet:
         vlanTagList = rangeSetToNumericalRange( vlanTagRangeSet,
                                                 "Classification::VlanRange" )
         for vlanTag in vlanTagList:
            newVlanRangeSet.vlan.add( vlanTag )
      if vlanTagFsSet:
         for vlanTagFs in vlanTagFsSet:
            newVlanRangeSet.vlanFieldSet.add( vlanTagFs )
      outerVlan = sf.vlanTag.newMember( newVlanRangeSet )
      if innerVlanTagRange:
         innerVlanTagList = rangeSetToNumericalRange( innerVlanTagRange,
                                                      "Classification::VlanRange" )
         for vlanTag in innerVlanTagList:
            outerVlan.innerVlanTag.add( vlanTag )

      args[ 'VLANTAG_SF' ] = sf

class VlanTagCmdMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, isVlanFieldSet, isInnerVlanFieldSet, fsExpr=None ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.isVlanFieldSet = isVlanFieldSet
      self.isInnerVlanFieldSet = isInnerVlanFieldSet
      self.fsExpr = fsExpr

   def generate( self, name ):
      class VlanTagExpression( VlanTagAdapterBase ):
         _baseExpr = 'dot1q vlan '
         if self.isVlanFieldSet:
            expression = _baseExpr + 'field-set VLANTAG_FIELD_SET'
         else:
            expression = _baseExpr + 'VLANTAG'
         if self.isInnerVlanFieldSet:
            expression += '[ inner inner-field-set INNER_VLANTAG_FIELD_SET ]'
         else:
            expression += '[ inner INNERVLANTAG ]'
         data = {
            'dot1q': dot1QKwMatcher,
            'vlan': vlanKwMatcher,
            'VLANTAG': vlanTagRangeMatcher,
            'field-set': fieldSetKwNode,
            'inner': 'inner',
            'INNERVLANTAG': vlanTagRangeMatcher,
            'inner-field-set': fieldSetKwNode,
         }
         if self.fsExpr:
            data.update( { 'VLANTAG_FIELD_SET': self.fsExpr } )
            data.update( { 'INNER_VLANTAG_FIELD_SET': self.fsExpr } )
      return VlanTagExpression

#----------------------------------------------------------------------------
# The "dot1q vlan VLANTAG" command for traffic-policy
#----------------------------------------------------------------------------
class VlanTagConfigCmd( VlanTagConfigCmdBase ):
   data = {
      'VLANTAG_EXPR': VlanTagCmdMatcher( isVlanFieldSet=False,
                                         isInnerVlanFieldSet=False )
   }

# ----------------------------------------------------------------------------
# The "vlan VLAN" command for traffic-policy
# ----------------------------------------------------------------------------
class VlanConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan { VLAN }'
   noOrDefaultSyntax = syntax
   data = {
         'vlan': 'vlan',
         'VLAN': vlanRangeMatcher,
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      vlan = args[ 'VLAN' ]
      context.addVlanMatchToSf( vlan )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      vlan = args[ 'VLAN' ]
      context.removeVlanMatchFromSf( vlan )


# modelet to be added to modes that support commit/abort
class CommitAbortModelet( CliParser.Modelet ):
   pass

class CommitCommand( CliCommand.CliCommandClass ):
   syntax = "commit"
   data = { "commit": "Commit all changes" }

   @staticmethod
   def handler( mode, args ):
      mode.commit()

class AbortCommand( CliCommand.CliCommandClass ):
   syntax = "abort"
   data = { "abort": "Abandon all changes" }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

CommitAbortModelet.addCommandClass( CommitCommand )
CommitAbortModelet.addCommandClass( AbortCommand )

# --------------------------------------------------------------------------
# The "field-set mac FIELD_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetMacAddrBaseConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'field-set mac FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "app"
   _macAddrContext = MacAddrFieldSetContext
   _baseData = {
      'field-set': fieldSetConfigKwMatcher,
      'mac': macKwMatcher,
   }

   @classmethod
   def _getContextKwargs( cls, fieldSetMacAddrName, mode=None ):
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._macAddrContext( **contextKwargs )

      if context.hasMacAddrFieldSet( name ):
         context.copyEditFieldSet()
      else:
         context.newEditFieldSet()

      childMode = mode.childMode( context.childMode, context=context,
                                  feature=cls._feature )
      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      name = args[ 'FIELD_SET_NAME' ]
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._macAddrContext( **contextKwargs )
      if context.hasMacAddrFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

   @classmethod
   def _removeFieldSet( cls, mode, name ):
      contextKwargs = cls._getContextKwargs( name, mode )
      context = cls._macAddrContext( **contextKwargs )
      if context.hasMacAddrFieldSet( name ):
         context.delFieldSet( name, mode, fromNoHandler=True )

# --------------------------------------------------------------------------
# The "[ remove ] { MACS }" command
# --------------------------------------------------------------------------
class FieldSetMacAddrConfigBase( CliCommand.CliCommandClass ):
   syntax = '[ remove ] { MACS }'
   data = {
      'remove': 'Remove Ethernet address from MAC field-set',
      'MACS': MacAddr.macAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      add = 'remove' not in args
      macAddrToUpdate = args[ 'MACS' ]
      context = mode.getContext()
      if context.canUpdateFieldSet():
         context.updateFieldSet( macAddrToUpdate, add=add )
         return
      mode.addError( "Cannot change MAC unless (contents) source is 'static'" )

class FieldSetMacAddrConfigCmds( FieldSetMacAddrConfigBase ):
   data = FieldSetMacAddrConfigBase.data.copy()

def expandRanges( ranges ):
   # XXXBUG641114: throw an error when the configured range size exceeds a limit.
   mergedVals = set()
   error = ""
   for r in ranges:
      if len( mergedVals ) + ( r[ 1 ] - r[ 0 ] + 1 ) > 200000:
         error = "Total range size permitted per configuration line cannot "\
            "exceed 200K"
         break
      mergedVals.update( range( r[ 0 ], r[ 1 ] + 1 ) )
   return ( mergedVals, error )

class TeidAdapterBase( CliCommand.CliExpression ):
   @staticmethod
   def adapter( mode, args, argList ):
      teidRangeSet = args.get( 'TEID', set() )
      teidFsSet = args.get( 'TEID_FIELD_SET', set() )
      assert ( teidRangeSet and not teidFsSet ) or ( not teidRangeSet and teidFsSet )
      sf = Tac.newInstance( "Classification::StructuredFilter", "" )
      if teidRangeSet:
         ( mergedTeidVals, errorMsg ) = expandRanges( teidRangeSet )
         if errorMsg:
            args[ 'error' ] = errorMsg
            return
         teidRangeList = rangeSetToNumericalRange( mergedTeidVals,
                                                   "Classification::TeidRange" )
         for teid in teidRangeList:
            sf.teid.add( teid )
      if teidFsSet:
         for teidFs in teidFsSet:
            sf.teidFieldSet.add( teidFs )
      args[ 'TEID_SF' ] = sf

class TeidCmdMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, isTeidFieldSet, fsExpr=None ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.isTeidFieldSet = isTeidFieldSet
      self.fsExpr = fsExpr

   def generate( self, name ):
      class TeidExpression( TeidAdapterBase ):
         _baseExpr = 'teid '
         if self.isTeidFieldSet:
            expression = _baseExpr + 'field-set integer TEID_FIELD_SET'
         else:
            expression = _baseExpr + 'TEID'
         data = {
            'teid': teidKwMatcher,
            'TEID': teidRangeMatcher,
            'field-set': fieldSetKwNode,
            'integer': integerKwMatcher,
         }
         if self.isTeidFieldSet and self.fsExpr:
            data.update( { 'TEID_FIELD_SET': self.fsExpr } )
      return TeidExpression

class TeidConfigCmdBase( CliCommand.CliCommandClass ):
   syntax = 'TEID_EXPR'
   noOrDefaultSyntax = syntax

   @classmethod
   def handler( cls, mode, args ):
      errorMsg = args.get( 'error', '' )
      if errorMsg:
         mode.addError( errorMsg )
         return
      context = mode.getContext()
      teidSf = args[ 'TEID_SF' ]
      context.updateTeidFromSf( teidSf, add=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      teidSf = args[ 'TEID_SF' ]
      context.updateTeidFromSf( teidSf, add=False )

# --------------------------------------------------------------------------
# The "teid TEIDS" command for traffic-policy
# --------------------------------------------------------------------------
class TeidConfigCmd( TeidConfigCmdBase ):
   data = {
      'TEID_EXPR': TeidCmdMatcher( isTeidFieldSet=False )
   }

# Features that may use the `application [ipv4|ipv6|l4] APPNAME` commands
# add functions of signature "ipv4" | "ipv6" | "l4" -> bool, with
# True indicating that the corresponding mode should be enabled
applicationModeGuards = CliExtensions.CliHook()

# --------------------------------------------------------------------------
# The "ethertype ETHTYPE" command for traffic-policy
# --------------------------------------------------------------------------
class EthTypeConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ethertype TYPE'
   noOrDefaultSyntax = syntax
   data = {
      'ethertype': 'Configure ethernet type',
      'TYPE': generateMultiRangeMatcherV2( 'ethertype',
                    appConstants.maxEthType,
                    numberFormat=NumberFormat.HEX ),
   }

   @staticmethod
   def adapter( mode, args, argList ):
      ethTypeRangeSet = args[ 'TYPE' ]
      ( mergedEthTypeVals, errorMsg ) = expandRanges( ethTypeRangeSet )
      if errorMsg:
         mode.addErrorAndStop( errorMsg )
      ethTypeRangeList = rangeSetToNumericalRange(
         mergedEthTypeVals, "Classification::EthTypeRange" )
      sf = Tac.newInstance( "Classification::StructuredFilter", "" )
      for ethType in ethTypeRangeList:
         sf.ethType.add( ethType )
      args[ 'TYPE' ] = sf

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      ethTypeSf = args[ 'TYPE' ]
      add = not CliCommand.isNoOrDefaultCmd( args )
      context.updateEthTypeFromSf( ethTypeSf, add=add )

   noOrDefaultHandler = handler
