# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import TableOutput
from CliModel import (
   Enum,
   Str,
   Dict,
   Model,
)

classifierNameToModel = { "dpi": "dpi", "custom-dpi": "customDpi", "ipv4": "ipv4" }
classifierModelToName = { "dpi": "dpi", "customDpi": "custom-dpi", "ipv4": "ipv4" }

class ServiceModel( Model ):
   category = Str( help="Category assigned to the application service" )

class ApplicationModel( Model ):
   classifier = Enum( values=( "dpi", "customDpi", "ipv4" ),
                               help="The classifier used for the application" )
   services = \
      Dict( keyType=str, valueType=ServiceModel,
           help="Services available for the application, keyed by service name" )

class ShowApplications( Model ):
   applications = \
     Dict( keyType=str, valueType=ApplicationModel,
           help="Applications configured on the switch, keyed by application name" )

   def render( self ):

      headings = [ "Application Name", "Classifier", "Service", "Category" ]
      table = TableOutput.createTable( headings )

      f1 = TableOutput.Format( justify="left" )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      table.formatColumns( f1, f1, f1, f1 )

      applications = self.applications
      for appName, app in sorted( applications.items() ):
         for serviceName, service in sorted( app.services.items() ):
            table.newRow( appName, classifierModelToName[ app.classifier ],
                    serviceName, service.category )

      print( table.output() )
