#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Bool, Dict, Enum, Int, Model, Str, Submodel, DeferredModel
from ShowRunOutputModel import Mode
import TableOutput

# Each Fragment in Fragments is keyed by the fragment's name.
class Fragment( Model ):
   pendingSession = Str( help="Name of pending session",
                         optional=True )
   committedSession = Str( help="Name of committed session",
                           optional=True )
   description = Str( help="Description of the fragment",
                      optional=True )
   commitGroupId = Int(
      help="Identity of commit group, if part of commit in progress",
      optional=True )
   # Will be used if/when god-mode is implemented
   # overriding = Bool( help="True if this fragment (uniquely) overrides others",
   #                    optional=True )
   # Will be used if/when we implement the notion of controllers over multiple
   # fragments
   # controller = Str( help="Identity of controller of this fragment",
   #                   optional=True )

class Fragments( Model ):
   fragmentConfigStatus = Enum(
      values=( "notLoaded", "inProgress", "incomplete", "completed" ),
      help="State of process of loading fragment config files from disk" )
   fragments = Dict( keyType=str, valueType=Fragment,
                     help="A mapping between the Fragment name and its state" )
   _hasDescription = Bool( "Determines whether to render description column",
                           default=False )
   def render( self ):
      # Before the table of fragments:
      print( f"Loading of fragment config files: {self.fragmentConfigStatus}" )
      # Format the table
      # headers = [ "", "Name", ] # overriding, name
      headers = [ "Name", ] # overriding, name
      fmt_header = TableOutput.Format( isHeading=True, border=True )
      fmt_header.borderStartPosIs( 1 )
      # fmt_star = TableOutput.Format( justify="left", minWidth=1, maxWidth=1,
      #                                isHeading=True )
      # fmt_star.padLimitIs( True )
      fmt_name = TableOutput.Format( justify="left", isHeading=True )
      fmt_name.padLimitIs( True )
      fmt = TableOutput.Format( justify="left" )
      fmtr = TableOutput.Format( justify="right" )
      # formats = [ fmt_star, fmt_name ]
      formats = [ fmt_name ]
      headers += [ "Version Name", "Session", ]
      formats += [ fmt, fmt ]
      headers.append( "Commit Group" )
      formats.append( fmtr )
      # headers.append( "Controller" )
      # formats.append( fmt )
      if self._hasDescription:
         headers.append( "Description" )
         fmt_description = TableOutput.Format( justify="left", wrap=True )
         formats.append( fmt_description )

      # Create table and create headings row
      tbl = TableOutput.TableFormatter( 0, None )
      tbl.startRow( fmt_header )
      for h in headers:
         tbl.newCell( h )
      tbl.formatColumns( *tuple( formats ) )

      if not self.fragments:
         tbl.newRow( '', '' )

      # Fill in the table contents:
      for name, fragment in self.fragments.items():
         # godMode = '*' if fragment.overriding else ''
         # tbl.newRow( godMode, name )
         tbl.newRow( name )
         if fragment.committedSession:
            tbl.newCell( "committed", fragment.committedSession )
         elif fragment.pendingSession:
            tbl.newCell( "pending", fragment.pendingSession )
         else:
            tbl.newCell( '', '' )
         tbl.newCell( str( fragment.commitGroupId or '' ) )
         # tbl.newCell( fragment.controller or '' )
         tbl.newCell( fragment.description or '' )
         if fragment.pendingSession and fragment.committedSession:
            # Add an extra row for 2nd session.
            # Include name here, to make it easier for ArosTestPlugin to
            # extract data (it would probably be prettier to have '' instead
            # of name).
            # tbl.newRow( '', name, "pending", fragment.pendingSession, '' )
            tbl.newRow( name, "pending", fragment.pendingSession, '' )
      print( tbl.output() )

class FragmentContent( DeferredModel ):
   pending = Bool( help="""
True if the config for this fragment comes from the pending session, False if
from the committed session""",
                   optional=True )
   config = Submodel( valueType=Mode,
                      optional=True,
                      help='Dictionary of config commands in this fragment. '
                      'The key is '
                     'a config command.  The config command might be:\n'
                     '- a simple command and in this case the value is NULL. '
                     ' Example: \'cmds\': { \'no ip routing\' : NULL } \n'
                       '- a command leading to a sub-mode and in this case the '
                       'value is a new Mode consisting of potential new config '
                       'commands.  Example: '
                       '\'cmds\': { \'interface Ethernet2\' : { '
                       '                                  \'cmds\': { } }' )

class ContentsOfFragments( DeferredModel ):
   contents = Dict( keyType=str,
                    valueType=FragmentContent,
                    valueOptional=True,
                    help="A mapping between the Fragment name and its config" )
   versionName = Enum( values=( "committed", "pending" ),
                       help="""Display only contents of committed sessions, or
                       prefer 'pending' when both exist""",
                       optional=True )
