#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import CliFragmentCli
from CliPlugin import CliFragmentModel
from CliPlugin.ConfigSessionShowCli import configKwMatcher
import CommonGuards
import ShowCommand
import ShowRunOutputModel

#-----------------------------------------------------------------------------------
# show configuration fragments
# show configuration fragments [ committed | pending ] ( all | named <name> )
# show configuration fragments [ committed | pending ] combined
#-----------------------------------------------------------------------------------

fragmentsKwMatcher = CliMatcher.KeywordMatcher( 'fragments',
      helpdesc='Show configuration fragment information' )

allKwMatcher = CliMatcher.KeywordMatcher( 'all',
      helpdesc="Show individual content of each non-empty fragment" )
combinedKwMatcher = CliMatcher.KeywordMatcher( 'combined',
      helpdesc="Show the combined config of the union of all fragments" )
namedKwMatcher = CliMatcher.KeywordMatcher( 'named',
      helpdesc="Show the config of the single named fragment" )
committedKwMatcher = CliMatcher.KeywordMatcher( 'committed',
      helpdesc="Show only contents of committed fragments" )
pendingKwMatcher = CliMatcher.KeywordMatcher( 'pending',
      helpdesc="""Show contents of pending session, if it exists. Otherwise
      only contents of previously committed fragments""" )

# The basic command to show fragment configuration/status:
# show configuration fragments
class ShowFragmentConfig( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show configuration fragments' )
   data = { 'configuration' : configKwMatcher,
            # For now, only allow on active supervisor
            'fragments': CliCommand.Node( matcher=fragmentsKwMatcher,
                                           guard=CommonGuards.ssoStandbyGuard,
                                           hidden=True )
           }
   cliModel = CliFragmentModel.Fragments
   privileged = True
   handler = CliFragmentCli.showFragmentConfigs

BasicCli.addShowCommandClass( ShowFragmentConfig )

# If you add any tokens past the basic command, we show the fragment
# contents instead of the fragment configuration itself.
# show configuration fragments [ committed | pending ]  [ all | named <name> ]
class ShowFragmentConfigContents( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show configuration fragments '
              '[ committed | pending ]'
              '( all | ( named FRAGMENT ) ) ' )
   data = { 'configuration' : configKwMatcher,
            'fragments': CliCommand.Node( matcher=fragmentsKwMatcher,
                                           hidden=True ),
            'all' : allKwMatcher,
            'named' : namedKwMatcher,
            'FRAGMENT' : CliFragmentCli.fragmentNamesMatcher,
            'committed' : committedKwMatcher,
            'pending' : pendingKwMatcher,
           }
   cliModel = CliFragmentModel.ContentsOfFragments
   privileged = True
   handler = CliFragmentCli.showFragmentConfigContents

BasicCli.addShowCommandClass( ShowFragmentConfigContents )

# This command displays the combined config from all of the fragments,
# without including any configuration in running-config that is *not* derived
# from fragment.
# show configuration fragments [ committed | pending ] combined
class ShowFragmentConfigCombinedContents( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show configuration fragments '
              '[ committed | pending ]'
              'combined ' )
   data = { 'configuration': configKwMatcher,
            'fragments': CliCommand.Node( matcher=fragmentsKwMatcher,
                                           hidden=True ),
            'combined': combinedKwMatcher,
            'committed': committedKwMatcher,
            'pending': pendingKwMatcher,
           }
   cliModel = ShowRunOutputModel.Mode
   privileged = True
   handler = CliFragmentCli.showFragmentConfigCombinedContents

BasicCli.addShowCommandClass( ShowFragmentConfigCombinedContents )


