#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import io
import re
import sys

import Ark
import ArPyUtils
import BasicCliSession
import BasicCliModes
import CliPlugin.TechSupportCli
from CliPlugin import ConfigSessionModel
from CliPlugin import FileCli
import CliCommand
import CliMatcher
import CliModel
import CliSave
import CliSaveBlock
import CliSession
import CliToken.Configure
import ConfigMount
import ConfigSessionCommon
import GitCliLib
import GitLib
import FileCliUtil
import FileUrl
import SessionUrlUtil
import ShowRunOutputModel
import TableOutput
import Tac
import Tracing
import Url
import UrlPlugin.SessionUrl as USU # pylint: disable-msg=F0401

t0 = Tracing.t0

cliConfig = None

# configure replace <source>
# configure replace <source> force      [Hidden command, for backward compatibility]
# configure replace <source> ignore-errors
# <source> : url (session:, flash:, clean-config) | 'named <session name>'

@Ark.synchronized( CliSession.sessionLock )
def configureReplace( mode, surl=None, ignoreErrors=False, debugAttrChanges=False,
                      md5=None, replace=True, noCkp=False ):
   # durl isn't literally running-config, but after commit that's what
   # we're going to end up modifying.  So, if surl is running-config, then...
   durl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )
   if surl == durl and not debugAttrChanges:
      mode.addError( "Source and destination are the same file" )
      return None

   if not BasicCliSession.CONFIG_LOCK.canRunConfigureCmds():
      raise ConfigMount.ConfigChangeProhibitedError(
         ConfigMount.ConfigChangeProhibitedError.CONFIG_LOCKED )

   em = mode.session_.entityManager_
   CliSession.exitSession( em )

   targetStr = "unknown"
   configSource = ""
   configSourceUrl = ""
   if ignoreErrors is None:
      ignoreErrors = False
   if surl:
      targetStr = surl
      configSource = "url"
      configSourceUrl = str( surl )

   if CliSession.isCommitTimerInProgress():
      if configSourceUrl == CliSession.sessionStatus.preCommitConfig:
         # We allow replace back to pre-commit-config
         pass
      else:
         mode.addError( 'Cannot configure replace while session %s is pending '
                        'commit timer' % CliSession.commitTimerSessionName() )
         return None

   # Generate the session used for config replace
   result = ConfigSessionCommon.configReplaceSession( mode,
                                                      surl,
                                                      ignoreErrors=ignoreErrors,
                                                      md5=md5,
                                                      replace=replace )
   targetSessionName, response = result
   if response:
      mode.addError( response )
      ConfigSessionCommon.doLog( mode,
            ConfigSessionCommon.SYS_CONFIG_REPLACE_FAILURE, targetStr,
            loggingTerminal=cliConfig.loggingTerminal )
      return None

   changeId = None
   if not noCkp:
      changeId = GitCliLib.savePreCommitRunningConfig( mode, targetSessionName,
            maxHistorySize=cliConfig.sysdbObj().maxCheckpoints )

   # Commit
   t0( 'Committing session', targetSessionName )
   response = CliSession.commitSession( em, debugAttrChanges, mode=mode,
                                        sessionName=targetSessionName )
   if response:
      mode.addError( response )
      if replace:
         ConfigSessionCommon.doLog( mode,
            ConfigSessionCommon.SYS_CONFIG_REPLACE_FAILURE, targetStr,
            loggingTerminal=cliConfig.loggingTerminal )
      return None
   else:
      # Invoke commit handlers and rollback if error
      onCommitStatus = ConfigSessionCommon.invokeOnCommitHandlersOrRevert( mode,
                         targetSessionName, changeId, cliConfig.loggingTerminal )
      if onCommitStatus is None:
         if replace:
            ConfigSessionCommon.doLog( mode,
               ConfigSessionCommon.SYS_CONFIG_REPLACE_SUCCESS, targetStr,
               loggingTerminal=cliConfig.loggingTerminal )
         if not noCkp:
            GitCliLib.savePostCommitRunningConfig( mode, targetSessionName,
                  description=f'Configure Replace ({surl})',
                  maxHistorySize=cliConfig.sysdbObj().maxCheckpoints )
         CliSession.runPostCommitHandlers( mode, targetSessionName )
         mode.session_.addToHistoryEventTable( "commandLine", configSource,
                                               "running", configSourceUrl, "" )
         return True
      else:
         mode.addError( onCommitStatus )
         return False

class ConfigureReplace( CliCommand.CliCommandClass ):
   syntax = ( "configure replace SURL [ ignore-errors | force ] "
              "[ debug-attribute ] [ md5 MD5 ] [ skip-checkpoint ]" )
   data = {
      "configure" : CliToken.Configure.configureParseNode,
      "replace" : 'Replace configuration state',
      "SURL" : FileCliUtil.copySourceUrlExpr(),
      "ignore-errors" : 'Replace config, ignoring any errors in loading the config',
      "force" : CliCommand.Node( CliMatcher.KeywordMatcher( "force",
                                                            helpdesc="force" ),
                                 hidden=True ),
      "debug-attribute" : CliCommand.Node(
         CliMatcher.KeywordMatcher( "debug-attribute",
                                    helpdesc="Record attribute changes" ),
         hidden=True ),
      "md5" : 'Validate input config file content against provided MD5 digest',
      'MD5' : CliMatcher.PatternMatcher( r'[0-9a-f]{32}', helpname='MD5SUM',
                                        helpdesc='32-digit hexadecimal MD5 digest' ),
      "skip-checkpoint" : 'Skip checkpointing of the current config'
      }
   @staticmethod
   def handler( mode, args ):
      surl = args[ "SURL" ]
      ignoreErrors = 'ignore-errors' in args or 'force' in args
      debugAttrChanges = 'debug-attribute' in args
      md5 = args.get( 'MD5' )
      noCkp = 'skip-checkpoint' in args
      configureReplace( mode, surl=surl, ignoreErrors=ignoreErrors,
                        debugAttrChanges=debugAttrChanges,
                        md5=md5, noCkp=noCkp )

BasicCliModes.EnableMode.addCommandClass( ConfigureReplace )

def _getSessionName( mode, args ):
   sessionName = args.get( 'SESSION_NAME', '' )
   if not sessionName:
      sessionName = CliSession.currentSession( mode.entityManager )
   if not sessionName:
      mode.addError( "Not currently in a session" )
      return None
   pcs = CliSession.sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if not pcs:
      mode.addError( "Session %s does not exist" % sessionName )
      return None
   return pcs

def showSessionConfig( mode, args ):
   pcs = _getSessionName( mode, args )
   if pcs is None:
      return None

   if not pcs.mergeOnCommit and ( 'ancestor' in args or 'merged' in args ):
      mode.addError( 'Session %s does not have commit merge enabled' % pcs.name )
      return None

   saveConfigFn = CliSave.saveSessionConfig
   showCmd = 'show session-configuration named %s'
   if 'ancestor' in args:
      saveConfigFn = CliSave.saveAncestorSessionConfig
      showCmd = 'show session-configuration ancestor named %s'
   elif 'merged' in args:
      saveConfigFn = CliSave.saveMergedSessionConfig
      showCmd = 'show session-configuration merged named %s'

   if mode.session_.shouldPrint():
      # if aren't in JSON mode print the command time
      FileCli.showCommandAndTime( mode, showCmd % pcs.name )

   try:
      saveConfigFn( mode.entityManager, sys.stdout, pcs.name,
            showSanitized='sanitized' in args,
            showJson=not mode.session_.shouldPrint() )
   except CliSaveBlock.CliMergeConflict as e:
      _handleMergeConflict( mode, pcs.name, e )
      return None

   # the config should be able to be JSON and we return the model type.
   # the result itself would have been streamed out
   return CliModel.noValidationModel( ShowRunOutputModel.Mode )

def _handleMergeConflict( mode, sessionName, cliMergeConflict ):
   stream = io.StringIO()
   cliMergeConflict.renderConflictMsg( stream,
         'session:/%s-ancestor-config' % sessionName,
         'system:/running-config',
         'session:/%s-session-config' % sessionName )
   mode.addError( stream.getvalue() )

def showSessionConfigDiffs( mode, args ):
   # Print out the diffs for this config session according to the rules below:
   # Legend: S = session-config, A = common ancestor-config, R = running-config
   #
   # show session-config diffs -> diff(A,S) if mergeOnCommit else diff(R,S)
   # show session-config merged diffs -> diff(R, M)
   # show running-config diff session-config ancestor -> diff(A, R)

   pcs = _getSessionName( mode, args )
   if pcs is None:
      return

   if not pcs.mergeOnCommit and ( 'ancestor' in args or 'merged' in args ):
      mode.addError( 'Session %s does not have commit merge enabled' % pcs.name )
      return

   theirSource = 'running-config'
   mySource = 'session-config'
   if 'merged' in args:
      theirSource = 'running-config'
      mySource = 'merged-config'
   elif 'ancestor' in args:
      theirSource = 'ancestor-config'
      mySource = 'running-config'
   elif 'reparse' in args:
      theirSource = 'session-config'
      mySource = 'running-config'
   elif pcs.mergeOnCommit:
      # if we are in a session-config and a diff is requested and we have a
      # commit merge enabled then diff against the common ancestor-config rather
      assert 'merged' not in args and 'ancestor' not in args
      theirSource = 'ancestor-config'
      mySource = 'session-config'

   stream = io.StringIO()
   try:
      CliSave.saveSessionDiffs( mode.entityManager, stream, pcs.name,
            theirSource, mySource,
            showNoSeqNum='ignore-sequence-number' in args )
   except CliSaveBlock.CliMergeConflict as e:
      _handleMergeConflict( mode, pcs.name, e )
      return
   _printDiffs( pcs.name, theirSource, mySource, stream.getvalue() )

def showSessionConfigCommitDiffs( mode, args ):
   commitHash1 = args[ 'COMMIT_HASH1' ]
   commitHash2 = args.get( 'COMMIT_HASH2' )
   commit1 = GitLib.getGitCommits( mode.entityManager.sysname(),
         commitHash=commitHash1 )
   if not commit1:
      mode.addErrorAndStop( f'Unable to find commit hash: {commitHash1}' )

   commit2 = GitLib.getGitCommits( mode.entityManager.sysname(),
         commitHash=( commitHash2 if commitHash2 else f'{commitHash1}^1' ) )
   if not commit2:
      if commitHash2:
         mode.addErrorAndStop( f'Unable to find commit hash: {commitHash2}' )
      else:
         mode.addErrorAndStop(
               f'Unable to find the predecessor of commit hash: {commitHash1}' )

   commitHash2 = commit2[ 0 ][ "commitHash" ]
   if 'reparse' not in args:
      diff = GitLib.gitDiff( mode.entityManager.sysname(), commitHash1, commitHash2 )
      if diff:
         print( f'--- checkpoint:/{commitHash2}' )
         print( f'+++ checkpoint:/{commitHash1}' )
         print( diff.strip() )
   else:
      _printReparseCommitDiff( mode, args, commitHash1, commitHash2 )

def _printReparseCommitDiff( mode, args, commitHash1, commitHash2 ):
   ctx = Url.Context( *Url.urlArgsFromMode( mode ) )
   hash1Url = Url.parseUrl( f'checkpoint:/{commitHash1}', ctx )
   hash2Url = Url.parseUrl( f'checkpoint:/{commitHash2}', ctx )
 
   with ArPyUtils.StdoutAndStderrInterceptor():
      result = ConfigSessionCommon.configReplaceSession( mode, hash1Url,
            ignoreErrors=True, ignoreENOENT=True, replace=True )
   sessionName1, error = result
   if error:
      mode.addErrorAndStop( error )

   try:
      with ArPyUtils.StdoutAndStderrInterceptor():
         result = ConfigSessionCommon.configReplaceSession( mode, hash2Url,
               ignoreErrors=True, ignoreENOENT=True, replace=True )
      sessionName2, error = result
      if error:
         mode.addErrorAndStop( error )
      try:
         stream = io.StringIO()
         CliSave.saveSessionDiffs( mode.entityManager, stream,
               sessionName=sessionName2,
               mySessionName=sessionName1,
               theirSource='session-config',
               mySource='session-config',
               showNoSeqNum='ignore-sequence-number' in args )
         diffs = stream.getvalue()
         if diffs:
            print( f'--- checkpoint:/{commitHash2}' )
            print( f'+++ checkpoint:/{commitHash1}' )
            print( diffs.strip() )
      finally:
         CliSession.abortSession( mode.entityManager, sessionName=sessionName2 )
   finally:
      CliSession.abortSession( mode.entityManager, sessionName=sessionName1 )
      CliSession.exitSession( mode.entityManager )

def showSessionConfigCommit( mode, args ):
   commitHash = args[ 'COMMIT_HASH' ]
   contents = GitLib.gitShow( mode.entityManager.sysname(),
         commitHash, GitLib.CONFIG_FILE_NAME )
   if contents is None:
      mode.addError( f"Unable to show commit '{commitHash}'" )
      return
   print( contents )

def showRunningConfigDiffsBySession( mode, args ):
   surl = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
   result = ConfigSessionCommon.configReplaceSession( mode,
                                                      surl,
                                                      ignoreErrors=True,
                                                      ignoreENOENT=True,
                                                      replace=True )
   sessionName, error = result
   if error:
      mode.addErrorAndStop( error )

   try:
      args[ 'SESSION_NAME' ] = sessionName
      showSessionConfigDiffs( mode, args )
   finally:
      CliSession.abortSession( mode.entityManager )
      CliSession.exitSession( mode.entityManager )

def _getFilepath( configName, sessionName ):
   if configName == 'running-config':
      return 'system:/running-config'
   assert configName in ( 'session-config', 'ancestor-config', 'merged-config' ), (
      'Unsupported configName: %s' % configName )
   return f'session:/{sessionName}-{configName}'

def _printDiffs( sessionName, theirSource, mySource, diffs ):
   if not diffs:
      return # if there are no diffs then dont print anything
   print( '--- %s' % _getFilepath( theirSource, sessionName ) )
   print( '+++ %s' % _getFilepath( mySource, sessionName ) )
   print( diffs.strip() )

def showConfigSessions( mode, args ):
   detail = 'detail' in args
   sessions = {}
   openSession = {}
   renderDescription = False
   tty = CliSession.getPrettyTtyName()
   for oss in CliSession.handlerDir( mode.session.entityManager ).\
       openSessionStatusDir.openSession.values():
      pcc = oss.session
      if pcc:
         pidList = openSession.setdefault( pcc.name, [] )
         pidList.append( oss.processId )

   statusSet = CliSession.sessionStatus.pendingChangeStatusDir.status
   h = CliSession.handlerDir( mode.session.entityManager )

   for sessionName, sessionPcs in statusSet.items():
      pidList = openSession.get( sessionName ) or []
      state = CliSession.sessionStatus.sessionStateDir.state.get( sessionName,
                                                                  'pending' )

      if ( ( ( state == 'committed' and
               not CliSession.isSessionPendingCommitTimer( sessionName )
               and not pidList ) or
             CliSession.internalSessionName( sessionName ) )
           # but if 'detail' is passed in, print *all* sessions even if they
           # are internal or completed
           and not detail ):
         continue

      instances = {}
      for pid in pidList:
         osc = h.openSessionConfigDir.openSession.get( pid )
         user = osc.user if osc else ""
         terminal = osc.terminal if osc else ""
         instances[ pid ] = ConfigSessionModel.Instance( user=user,
                                 terminal=terminal,
                                 currentTerminal=( tty and tty == terminal ) )

      description = ''
      pccDir = CliSession.sessionConfig.pendingChangeConfigDir.config
      pcc = pccDir.get( sessionName )
      if pcc and pcc.userString:
         description = pcc.userString
         renderDescription = True

      isCompleted = ( sessionPcs.sessionState != "pending" and
                      sessionPcs.endTime < Tac.endOfTime )
      completedTime = ( Ark.switchTimeToUtc( sessionPcs.endTime )
                        if isCompleted else None )

      sessions[ sessionName ] = ConfigSessionModel.Session(
            state=state,
            instances=instances,
            commitUser=sessionPcs.commitUser,
            completedTime=completedTime,
            description=description )

   commitTimerSessionName = \
      CliSession.sessionStatus.sessionStateDir.commitTimerSessionName or None
   commitTimerExpireTimeStatus = CliSession.sessionStatus.commitTimerExpireTime
   if commitTimerSessionName and commitTimerExpireTimeStatus != Tac.endOfTime:
      commitTimerExpireTime = Ark.switchTimeToUtc( commitTimerExpireTimeStatus )
   else:
      commitTimerExpireTime = None

   return ConfigSessionModel.Sessions( sessions=sessions,
                                    maxSavedSessions=cliConfig.maxSavedSessions,
                                    maxOpenSessions=cliConfig.maxOpenSessions,
                                    mergeOnCommit=cliConfig.mergeOnCommit,
                                    saveToStartupConfigOnCommit=
                                    cliConfig.saveToStartupConfigOnCommit,
                                    commitTimerSessionName=commitTimerSessionName,
                                    commitTimerExpireTime=commitTimerExpireTime,
                                    _renderDetail=bool( detail ),
                                    _renderDescriptions=renderDescription )

def showConfigSessionsHistory( mode, args ):
   result = ConfigSessionModel.Commits()
   for commit in GitLib.getGitCommits(
         mode.entityManager.sysname(),
         trailerKeys=(
            'Session-name',
            'Commit-type',
            'Timestamp-ns',
            'Description',
            'Commit-time-expiry' ) ):
      # all commits with a session name should also contain the type of commit
      assert 'Commit-type' in commit[ 'trailers' ]
      if commit[ 'trailers' ][ 'Commit-type' ] != 'post-commit':
         # only look at commits that are post commit, ignoring pre-commit ones
         continue

      commitTimerExpiry = int( commit[ 'trailers' ][ 'Commit-time-expiry' ] )
      description = commit[ 'trailers' ][ 'Description' ]
      commitTime = float( commit[ 'trailers' ][ 'Timestamp-ns' ] ) / 1000000000
      result.commits.append( ConfigSessionModel.Commit(
            commitHash=commit[ 'commitHash' ],
            authorName=commit[ 'author' ],
            commitTime=commitTime,
            sessionName=commit[ 'trailers' ][ 'Session-name' ],
            description=description if description else None,
            commitTimerExpireTime=( float( commitTimerExpiry ) + commitTime
               if commitTimerExpiry else None )
         ) )

   return result

def showConfigSessionsMemory( mode, args ):
   sessions = {}
   statusSet = CliSession.sessionStatus.pendingChangeStatusDir.status
   totalMem = 0
   for name in statusSet:
      mem = int( CliSession.sessionSize( mode.session.entityManager, name,
                 warnAfter=None ) )
      totalMem += mem
      sessions[ name ] = ConfigSessionModel.SessionMemory( memory=mem )
   return ConfigSessionModel.SessionMemoryTable( sessions=sessions,
                                                 totalMemory=totalMem )

def rootWalk( mode, args ):
   """Walk all config roots and match the specified filters."""
   detail = 'detail' in args
   attrPattern = args.get( 'ATTR_PATTERN' )
   if attrPattern:
      attrPattern = attrPattern[ 0 ]
   typePattern = args.get( 'TYPE_PATTERN' )
   if typePattern:
      typePattern = typePattern[ 0 ]
   configTypes = {}

   rootTable = TableOutput.createTable( [ 'Attribute', 'Type', 'Parent Type' ] )
   f1 = TableOutput.Format( justify='left', maxWidth=30, wrap=True )
   f1.noPadLeftIs( True )
   rootTable.formatColumns( f1, f1, f1 )

   configRoot = CliSession.configRoot( mode.entityManager )
   if detail:
      # All roots
      print( '\n'.join( configRoot.root ) )
   for root in configRoot.root:
      # maybe skip if the root is ConfigMounted 'r'?
      t = Tac.typeNode( configRoot.rootTrie.prefixTypename( root ) )
      walk( t, attrPattern, typePattern, configTypes, rootTable )
   print( rootTable.output() )

def walk( t, attrPattern, typePattern, configTypes, rootTable ):
   """Walk a given type, recursively."""

   typeName = t.fullTypeName

   if configTypes.get( typeName, False ):
      # already visited
      return

   configTypes[ typeName ] = True

   if typeName.startswith( "Tac::" ):
      # Not interested in children of Tac
      return

   for attr in t.attributeQ:
      attrName = attr.name
      attrType = attr.memberType.fullTypeName
      # .memberType gives you just the base type even if it's a collection or a Ptr
      # isPtr seems to be true even when it's not a ::Ptr
      if attr.isPtr:
         attrType += '::Ptr'
      if attr.isCollection:
         attrType += '[]'
      if ( ( attrPattern is None or
             re.search( attrPattern, attrName, re.IGNORECASE ) ) and
           ( typePattern is None or
             re.search( typePattern, attrType, re.IGNORECASE ) ) ):
         # maybe skip if the attribute has a filter or is not .isLogging
         rootTable.newRow( attrName, attrType, typeName )
      # Don't walk Ptrs (that are not part of any config type?)?
      walk( attr.memberType, attrPattern, typePattern, configTypes, rootTable )

def debugEntityCopy( mode, args ):
   editType = args[ 'EDIT_TYPE' ]
   if editType == 'delete':
      editType = 'del'

   if args[ 'ACTION' ] == 'assert':
      action = 'actionAssert'
   elif args[ 'ACTION' ] == 'trace':
      action = 'actionTrace'
   else:
      assert args[ 'ACTION' ] == 'none'
      action = None

   pathname = args[ 'PATH_PATTERN' ] if 'PATH_PATTERN' in args else ''
   attr = args[ 'ATTR_PATTERN' ] if 'ATTR_PATTERN' in args else ''
   entityType = args[ 'TYPE_PATTERN' ] if 'TYPE_PATTERN' in args else ''

   userConfig = CliSession.sessionConfig.userConfig
   assert userConfig == ConfigMount.force( cliConfig )
      
   CliSession.registerEntityCopyDebug( mode.entityManager,
                                       attr, entityType=entityType,
                                       pathname=pathname, editType=editType,
                                       action=action )   
   
#--------------------------------------------------------------------------------
# debug configuration sessions
#               [ { ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) |
#                   ( pathname PATH_PATTERN ) } ]
#               EDIT_TYPE ACTION
#--------------------------------------------------------------------------------
regexMatcher = CliMatcher.PatternMatcher( pattern='\\S+',
      helpdesc='Python regular expression', helpname='WORD' )

class DebugConfigurationSessionsCmd( CliCommand.CliCommandClass ):
   syntax = ( 'debug configuration sessions '
              '[ {  ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) | '
                   '( pathname PATH_PATTERN ) } ]'
              'EDIT_TYPE ACTION' )
   data = {
      'debug': 'Debug command',
      'configuration': 'Debug configuration',
      'sessions': 'Debug entity copy of obj/type/attr changes',
      'attribute': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'attribute',
            helpdesc='Match attribute' ),
         maxMatches=1 ),
      'ATTR_PATTERN': regexMatcher,
      'type': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'type', helpdesc='Match type' ),
         maxMatches=1 ),
      'TYPE_PATTERN': regexMatcher,
      'pathname': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'pathname', helpdesc='Match pathname' ),
         maxMatches=1 ),
      'PATH_PATTERN': regexMatcher,
      'EDIT_TYPE': CliMatcher.EnumMatcher(
         { 'delete': 'Match if deleted',
           'set': 'Match if attr is set'} ),
      'ACTION': CliMatcher.EnumMatcher(
         { 'assert': 'Assert if match description',
           'trace': 'Trace if match description',
           'none': 'Delete description' } )
   }
   handler = debugEntityCopy
   hidden = True

BasicCliModes.EnableMode.addCommandClass( DebugConfigurationSessionsCmd )

def showEntityCopyDebug( mode, args ):
   if cliConfig.debugEntityCopy:
      headings = [( "Pathname.attr", "lh" ), ( "Operation", "l" ), 
                  ( "Action", "l" ) ]
      tbl = TableOutput.createTable( headings )
      for pathname, debugConfigCollection in cliConfig.debugEntityCopy.items():
         for attrEdit, debugConfig in debugConfigCollection.debugConfig.items():
            tbl.newRow( f"{pathname}.{attrEdit.attributeName}",
                        attrEdit.editType, debugConfig.action )
      print( tbl.output() )
      print()

   if cliConfig.debugPerTypeEntityCopy:
      headings = [( "Type::attr", "lh" ), ( "Operation", "l" ), 
                  ( "[subtree]" ), ( "Action", "l" ) ]
      tbl = TableOutput.createTable( headings )
      for attrEdit, debugTypeConfig in cliConfig.debugPerTypeEntityCopy.items():
         for pathname, action in debugTypeConfig.action.items():
            tbl.newRow( "{}.{}".format( attrEdit.attributeType, 
                                    attrEdit.attributeName ),
                        attrEdit.editType, pathname, action )
      print( tbl.output() )
      print()


def getSessionConfigFromUrl( mode, match ):
   sessionName = CliSession.currentSession( mode.session_.entityManager_ )
   return SessionUrlUtil.sessionConfig( *Url.urlArgsFromMode( mode ),
                                         sessionName=sessionName )
def getCleanConfig( mode, match ):
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   pathname = "/clean-session-config" 
   url = "session:" + pathname
   fs = Url.getFilesystem( "session:" )
   return USU.SessionUrl( fs, url, pathname, pathname, context, clean=True )

FileCliUtil.registerCopySource(
   'session-cofig',
   CliCommand.Node(
      CliMatcher.KeywordMatcher( 'session-config',
                                helpdesc='Copy from session configuration',
                                value=getSessionConfigFromUrl ),
      guard=ConfigSessionCommon.sessionGuard ),
   'session-config' )
FileCliUtil.registerCopySource(
   'clean-config',
   CliMatcher.KeywordMatcher( 'clean-config',
                              helpdesc='Copy from clean, default, configuration',
                              value=getCleanConfig ),
   'clean-config' )

FileCliUtil.registerCopyDestination(
   'session-cofig',
   CliCommand.Node(
      CliMatcher.KeywordMatcher( 'session-config',
                 helpdesc='Update (merge with) current session configuration',
                                value=getSessionConfigFromUrl ),
      guard=ConfigSessionCommon.sessionSsoGuard ),
   'session-config' )

def copyRunningBySession( mode, surl ):
   return configureReplace( mode, surl=surl, ignoreErrors=True,
                            replace=False )

def maybeAutoSaveToStartupConfig( mode, sessionName ):
   if not cliConfig.saveToStartupConfigOnCommit:
      return

   running = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )
   startup = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
   FileCliUtil.copyFile( None, mode, running, startup )

def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager, "cli/session/input/config", 
                                  "Cli::Session::CliConfig", "w" )

   CliSession.registerPostCommitHandler( maybeAutoSaveToStartupConfig )

   # register configure replace handler
   FileCli.copyRunningHandler = copyRunningBySession

   CliPlugin.TechSupportCli.registerShowTechSupportCmd(
      '2015-02-06 00:00:47',
      cmds=[ 'show configuration sessions detail' ] )
