#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import ArPyUtils
from CliModel import Bool, Dict, Enum, Int, Model, Str, Float, List
import TableOutput
import Tac

textField = TableOutput.Format( justify='left' )
textField.noPadLeftIs( True )
textField.padLimitIs( True )

class Instance( Model ):
   user = Str( help="User that opened the session" )
   terminal = Str( help="Terminal where the session is open" )
   currentTerminal = Bool( help="The session corresponds to the current terminal" )

# show configuration sessions [detail]
class Session( Model ):
   state = Enum( help="State of the session",
                 values=Tac.Type( 'Cli::Session::SessionState' ).attributes )
   completedTime = Float( help="Timestamp of when the session completed",
         optional=True )
   commitUser = Str( help="The user that committed this session",
         optional=True )
   description = Str( help="Description of the session" )
   instances = Dict( keyType=int, valueType=Instance,
         help="Instances where the session is open, indexed by process ID" )

class Sessions( Model ):
   sessions = Dict( keyType=str, valueType=Session,
                    help="A mapping between the Session name and information" ) 

   maxSavedSessions = Int( help="Maximum number of saved sessions" )

   maxOpenSessions = Int( help="Maximum number of open sessions" )

   mergeOnCommit = Bool( help="New sessions will perform a commit merge" )

   saveToStartupConfigOnCommit = Bool( help="Successfully committed session will "
                                      "autosave to startup-config" )
   commitTimerSessionName = \
      Str( help="Config session name with pending commit timer", optional=True )
   commitTimerExpireTime = \
      Float( help="Timestamp of when pending commit timer expires", optional=True )

   _renderDetail = Bool(
      help="Determines whether additional information should be rendered",
      default=False )

   _renderDescriptions = Bool( 
      help="Determines whether descriptions should be rendered i.e. at least one " 
      "session has a description" )

   def render( self ):
      # Format the table
      headings = [ "", "Name", "State", "User", "Terminal" ]
      fmtHeader = TableOutput.Format( isHeading=True, border=True )
      fmtHeader.borderStartPosIs( 1 )
      fmtStar = TableOutput.Format( justify="left", minWidth=1, maxWidth=1,
                                     isHeading=True )
      fmtStar.padLimitIs( True )
      fmtName = TableOutput.Format( justify="left", isHeading=True )
      fmtName.padLimitIs( True )
      fmt = TableOutput.Format( justify="left" )
      formats = [ fmtStar, fmtName, fmt, fmt, fmt ]

      # Add additional headings to the table as necessary
      if self._renderDetail:
         for hdrName in ( 'Completed Time', 'Committed By' ):
            headings.append( hdrName )
            formats.append( TableOutput.Format( justify='right' ) )

      if self._renderDescriptions:
         headings.append( "Description" )
         fmtDescription = TableOutput.Format( justify="left", wrap=True )
         formats.append( fmtDescription )

      # Create table and create headings row
      tbl = TableOutput.TableFormatter( 0, None )
      tbl.startRow( fmtHeader )
      for h in headings:
         tbl.newCell( h )

      tbl.formatColumns( *tuple( formats ) )

      # Fill the table 
      for name, session in sorted( self.sessions.items() ):
         if not session.instances:
            instances = ( Instance( user='', terminal='', currentTerminal=False ), )
         else:
            instances = session.instances.values()

         for inst in instances:
            star = '*' if inst.currentTerminal else ''
            tbl.newRow( star, name, session.state, inst.user, inst.terminal )
            if self._renderDetail:

               completedTime = ( Ark.utcTimestampToStr( session.completedTime )
                     if session.completedTime else '' )
               tbl.newCell( completedTime )

               tbl.newCell( session.commitUser or '' )
            if self._renderDescriptions:
               if session.description:
                  tbl.newCell( session.description ) 

      print( 'Maximum number of completed sessions:', self.maxSavedSessions )
      print( 'Maximum number of pending sessions:', self.maxOpenSessions )
      print( 'Merge on commit is %s' % # pylint: disable=consider-using-f-string
            ( 'enabled' if self.mergeOnCommit else 'disabled' ) )
      # pylint: disable-next=consider-using-f-string
      print( 'Autosave to startup-config on commit is %s' %
            ( 'enabled' if self.saveToStartupConfigOnCommit else 'disabled' ) )
      print()
      if self.commitTimerSessionName:
         print( 'Session with pending commit timer:',
                self.commitTimerSessionName )
      if self.commitTimerExpireTime:
         now = Tac.utcNow()
         if self.commitTimerExpireTime >= now:
            print( 'Time left until commit timer expiry:',
                   ArPyUtils.formatSeconds( self.commitTimerExpireTime - now ) )
      if self.commitTimerSessionName or self.commitTimerExpireTime:
         print()
      print( tbl.output() )

class SessionMemory( Model ):
   memory = Int( help='Memory in bytes used by the session' )

class SessionMemoryTable( Model ):
   sessions = Dict( keyType=str, valueType=SessionMemory,
                    help="A mapping between the Session name and information" ) 
   totalMemory = Int( help='Memory in bytes used by all sessions' )

   def render( self ):
      headings = [ "Name", "Memory (Bytes)" ]
      fmtHeader = TableOutput.Format( isHeading=True, border=True )
      fmtName = TableOutput.Format( justify="left", isHeading=True )
      fmtName.padLimitIs( True )
      fmtMemory = TableOutput.Format( justify="right" )
      formats = [ fmtName, fmtMemory ]

      tbl = TableOutput.TableFormatter( 0, None )
      tbl.startRow( fmtHeader )
      for h in headings:
         tbl.newCell( h )
      tbl.formatColumns( *tuple( formats ) )
      for name, session in sorted( self.sessions.items() ):
         tbl.newRow( name, session.memory )

      print( tbl.output() )
      print( 'Total memory used by configuration sessions:', self.totalMemory,
         'bytes' )

class Checkpoint( Model ):
   name = Str( help='A name given to this checkpoint', optional=True )
   fileUser = Str( help="The user who created the checkpoint file" )
   fileDate = Float( help="The utc timestamp of when the checkpoitn file created" )
   
class Checkpoints( Model ):
   """
   A model that represents information about checkpoint direcotory. This is 
   returned on execution of command 'show configure checkpoint'
   """
   checkpoints = Dict( keyType=str, valueType=Checkpoint,
                       help="A mapping between the checkpoint name and info" )
   maxCheckpoints = Int( help="Maximum number of checkpoints" )

   def render( self ):
      print( 'Maximum number of checkpoints:', self.maxCheckpoints )

      headings = [ 'Hash', 'Name', 'Date', 'User' ]
      table = TableOutput.createTable( headings )
      table.formatColumns( *( textField for _ in headings ) )
      if not self.checkpoints:
         return
      
      for commitHash, info in sorted( self.checkpoints.items(), 
                                key=lambda x: x[ 1 ].fileDate ):
         # convert the utc timestamp into local time
         fileDateStr = ( Ark.utcTimestampToStr( info.fileDate )
               if info.fileDate else '' )
         table.newRow(
               commitHash,
               info.name if info.name else '',
               fileDateStr,
               info.fileUser )
      print( table.output() )

class Commit( Model ):
   commitHash = Str( help="Commit Hash" )
   authorName = Str( help="The user that committed this session" )
   commitTime = Float( help="Timestamp of when the commit was completed" )
   sessionName = Str( help="Name of the session" )
   description = Str( help="User description of the session", optional=True )
   commitTimerExpireTime = Float( help="Timestamp of when the commit timer expires",
         optional=True )

class Commits( Model ):
   commits = List( valueType=Commit, help="In order list of commits" )

   def render( self ):
      headings = [ "Commit Hash", "Session Name", "User", "Description",
                  "Commit Time", "Commit Cancel\nTimer Expiry" ]
      table = TableOutput.createTable( headings )
      table.formatColumns( *( textField for _ in headings ) )
      for commit in self.commits:
         table.newRow( commit.commitHash, commit.sessionName, commit.authorName,
                    commit.description if commit.description else 'n/a',
                    Ark.utcTimestampToStr( commit.commitTime ),
                    ( commit.commitTimerExpireTime - commit.commitTime
                       if commit.commitTimerExpireTime else 'n/a' ) )
      print( table.output() )
