# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import ConfigSessionCli
from CliPlugin import ConfigSessionModel
from CliPlugin import FileCli
import CommonGuards
import GitLib
import ShowCommand
import ShowRunOutputModel

configKwMatcher = CliMatcher.KeywordMatcher( 'configuration',
      helpdesc='Show configuration information' )
sessionsKwMatcher = CliMatcher.KeywordMatcher( 'sessions',
      helpdesc='List configuration sessions' )
sessionConfigKwMatcher = CliMatcher.KeywordMatcher( 'session-config',
      helpdesc='Operating configuration of session' )
ancestorKwMatcher = CliMatcher.KeywordMatcher( 'ancestor',
      helpdesc='Operating configuration of the ancestor' )
mergedKwMatcher = CliMatcher.KeywordMatcher( 'merged',
      helpdesc='Three way merge of common-ancestor, session, and running config' )
namedKwMatcher = CliMatcher.KeywordMatcher( 'named',
      helpdesc='Session named' )
sessionNameMatcher = CliMatcher.PatternMatcher( r'.+',
      helpname='WORD', helpdesc='Name for session' )
pythonReMatcher = CliMatcher.PatternMatcher( r'.+',
      helpname='WORD', helpdesc='Python regular expression' )
commitKwMatcher = CliMatcher.KeywordMatcher( 'commit',
      helpdesc='Configuration session commit hash' )

def getCommitHashes( mode ):
   result = []
   for commit in GitLib.getGitCommits(
         mode.entityManager.sysname(),
         trailerKeys=(
            'Session-name',
            'Commit-type' ) ):
      # all commits with a session name should also contain the type of commit
      assert 'Commit-type' in commit[ 'trailers' ]
      if commit[ 'trailers' ][ 'Commit-type' ] != 'post-commit':
         # only look at commits that are post commit, ignoring pre-commit ones
         continue
      result.append( commit[ 'commitHash' ] )

   return result

commitIdMatcher = CliMatcher.DynamicNameMatcher( getCommitHashes,
      helpdesc='Commit Hash' )

#-----------------------------------------------------------------------------------
# show session-config [ ancestor | merged ] [ named SESSION_NAME ] [ sanitized ]
#-----------------------------------------------------------------------------------
class ShowSessionConfig( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show session-config [ ancestor | merged ] [ named SESSION_NAME ] '
              '[ sanitized ]' )
   data = { 
            'session-config': sessionConfigKwMatcher,
            'ancestor': ancestorKwMatcher,
            'merged': mergedKwMatcher,
            'named': namedKwMatcher,
            'SESSION_NAME': sessionNameMatcher,
            'sanitized': 'Sanitized Output'
          }
   cliModel = ShowRunOutputModel.Mode
   privileged = True
   handler = ConfigSessionCli.showSessionConfig

BasicCli.addShowCommandClass( ShowSessionConfig )

#-----------------------------------------------------------------------------------
# show session-config [ ancestor ] [ named SESSION_NAME ] diffs 
#                                                       [ ignore-sequence-number ]
#-----------------------------------------------------------------------------------
class ShowSessionConfigDiffs( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show session-config [ merged ] [ named SESSION_NAME ] '
              'diffs [ ignore-sequence-number ]' )
   data = { 
            'session-config': sessionConfigKwMatcher,
            'merged': mergedKwMatcher,
            'named': namedKwMatcher,
            'SESSION_NAME': sessionNameMatcher,
            'diffs': 'Differences from running-config',
            'ignore-sequence-number': 'Ignore sequence numbers in certain commands',
          }
   handler = ConfigSessionCli.showSessionConfigDiffs
   privileged = True

BasicCli.addShowCommandClass( ShowSessionConfigDiffs )

#-----------------------------------------------------------------------------------
# show session-config commit COMMIT_HASH1 diffs [ COMMIT_HASH2 ]
#                                              [ reparse [ ignore-sequence-number ] ]
#-----------------------------------------------------------------------------------
class ShowSessionConfigCommitDiffs( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show session-config commit COMMIT_HASH1 diffs '
                        '[ COMMIT_HASH2 ] [ reparse [ ignore-sequence-number ] ]' )
   data = {
            'session-config': sessionConfigKwMatcher,
            'commit': commitKwMatcher,
            'COMMIT_HASH1': commitIdMatcher,
            'diffs': 'Differences in configuration session commit[s]',
            'COMMIT_HASH2': commitIdMatcher,
            'reparse' : 'Reparsing configs to detect diffs (slower)',
            'ignore-sequence-number': 'Ignore sequence numbers in certain commands',
          }
   handler = ConfigSessionCli.showSessionConfigCommitDiffs
   privileged = True

BasicCli.addShowCommandClass( ShowSessionConfigCommitDiffs )

#-----------------------------------------------------------------------------------
# show session-config commit COMMIT_HASH
#-----------------------------------------------------------------------------------
class ShowSessionConfigCommit( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show session-config commit COMMIT_HASH' )
   data = {
            'session-config': sessionConfigKwMatcher,
            'commit': commitKwMatcher,
            'COMMIT_HASH': commitIdMatcher,
          }
   handler = ConfigSessionCli.showSessionConfigCommit
   privileged = True

BasicCli.addShowCommandClass( ShowSessionConfigCommit )

#-----------------------------------------------------------------------------------
# show running-config diffs session-config ancestor
#-----------------------------------------------------------------------------------
class ShowRunningConfigDiffAncestor( ShowCommand.ShowCliCommandClass ):
   syntax = 'show running-config diffs session-config ancestor'
   data = {
      'running-config': FileCli.runningConfigAfterShowKw,
      'diffs': FileCli.runningConfigDiffsKw,
      'session-config': sessionConfigKwMatcher,
      'ancestor': ancestorKwMatcher
   }

   handler = ConfigSessionCli.showSessionConfigDiffs
   privileged = True

BasicCli.addShowCommandClass( ShowRunningConfigDiffAncestor )

#-----------------------------------------------------------------------------------
# show configuration sessions [ detail ]
#-----------------------------------------------------------------------------------
class ShowConfigSession( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration sessions [ detail ]'
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'detail': 'List configuration session details',
          }
   cliModel = ConfigSessionModel.Sessions
   handler = ConfigSessionCli.showConfigSessions

BasicCli.addShowCommandClass( ShowConfigSession )

#-----------------------------------------------------------------------------------
# show configuration sessions history
#-----------------------------------------------------------------------------------
class ShowConfigSessionHistory( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration sessions history'
   data = {
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'history': 'List configuration session history',
          }
   cliModel = ConfigSessionModel.Commits
   handler = ConfigSessionCli.showConfigSessionsHistory

BasicCli.addShowCommandClass( ShowConfigSessionHistory )

#-----------------------------------------------------------------------------------
# show configuration sessions memory
#-----------------------------------------------------------------------------------
class ShowConfigSessionMem( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration sessions memory'
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'memory': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'memory',
                  helpdesc='Show memory utilization (may take time to compute)' ),
               guard=CommonGuards.ssoStandbyGuard )
          }
   cliModel = ConfigSessionModel.SessionMemoryTable
   handler = ConfigSessionCli.showConfigSessionsMemory

BasicCli.addShowCommandClass( ShowConfigSessionMem )

#-----------------------------------------------------------------------------------
# show configuration sessions roots
# [ { ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) } ] [ detail ]
#-----------------------------------------------------------------------------------
class ShowConfigSessionsRoots( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show configuration sessions roots '
              '[ { ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) } ] '
              '[ detail ]' )
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'roots': 'roots',
            'attribute': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'attribute',
                  helpdesc='match attribute' ),
               maxMatches=1 ),
            'type': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'type', helpdesc='match type' ),
               maxMatches=1 ),
            'ATTR_PATTERN': pythonReMatcher,
            'TYPE_PATTERN': pythonReMatcher,
            'detail': 'show root mount paths',
          }
   hidden = True
   handler = ConfigSessionCli.rootWalk

BasicCli.addShowCommandClass( ShowConfigSessionsRoots )

#-----------------------------------------------------------------------------------
# show configuration sessions debug
#-----------------------------------------------------------------------------------
class ShowConfigSessionDebug( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration sessions debug'
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'debug': 'Debug command',
          }
   hidden = True
   handler = ConfigSessionCli.showEntityCopyDebug

BasicCli.addShowCommandClass( ShowConfigSessionDebug )

#-----------------------------------------------------------------------------------
#    show running-config diffs reparse
#-----------------------------------------------------------------------------------
class ShowRunningConfigDiffsReparse( ShowCommand.ShowCliCommandClass ):
   syntax = 'show running-config diffs reparse'
   data = {
      'running-config' : FileCli.runningConfigAfterShowKw,
      'diffs' : FileCli.runningConfigDiffsKw,
      'reparse' : 'Reparsing startup-config to detect diffs (slower)',
   }
   privileged = True
   handler = ConfigSessionCli.showRunningConfigDiffsBySession

BasicCli.addShowCommandClass( ShowRunningConfigDiffsReparse )
