# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
import AclCliLib
import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
from CliMode.ConnectivityMonitor import ConnectivityMonitorBaseMode
from CliMode.ConnectivityMonitor import ConnectivityMonitorHostMode
from CliMode.ConnectivityMonitor import ConnectivityMonitorVrfMode
from CliMode.ConnectivityMonitor import ConnectivityMonitorClientMode
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
from CliPlugin.VrfCli import VrfExprFactory, getVrfNames
import CliToken.Monitor
import ConfigMount
import HostnameCli
import LazyMount
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
from TypeFuture import TacLazyType
import Tac
import IpLibConsts
from IpLibTypes import VrfNameType
from Toggles.ConnectivityMonitorTypesToggleLib import (
      toggleOCConnectivityMonitorSupportEnabled )
from Toggles.ConnectivityMonitorToggleLib import (
      toggleCMTcpProbeEnabled,
      toggleCMIcmpPingCountEnabled,
)
import urllib.parse

# Declare the dependency explicitly
# pkgdeps: library ConnectivityMonitorTypes
# To be able to mount connectivityMonitor/config when loading plugin:
# pkgdeps: rpmwith %{_libdir}/preinit/ConnectivityMonitor

AddrFamily = TacLazyType( 'Arnet::AddressFamily' )

config = None
status = None
defaultClientConfig = None
defaultClientStatus = None
configDir = None
statusDir = None
internalNamespace = None

def getVrf( mode ):
   vrf = getattr( mode, 'vrfName', None )
   if vrf is None:
      # Try to fetch from the parent mode. This is for the case when this
      # function is called from host config mode inside vrf mode
      return getattr( mode.parent_, 'vrfName', IpLibConsts.DEFAULT_VRF )
   return vrf

def getHostNames( mode=None ):
   vrfName = getVrf( mode )
   members = []
   _config = getConfig( mode )
   if _config is None:
      return members

   for host in _config.hostConfig:
      if host.vrfName == vrfName:
         members.append( host.hostName )
   return members

def getClientName( mode ):
   client = getattr( mode, 'clientName', None )
   if not client:
      return getattr( mode.parent_, 'clientName', None )
   return client

def getConfig( mode ):
   client = getClientName( mode )
   if client:
      if client in configDir:
         return configDir[ client ]
      else:
         return None
   return defaultClientConfig

clientHostPattern = r'[.A-Za-z0-9_:{}\[\]-]+'
matcherHostName = CliMatcher.DynamicNameMatcher( getHostNames, 'Name of the host',
                                                 pattern=clientHostPattern )

#------------------------------------------------------------------------------------
# (config)# monitor connectivity
#------------------------------------------------------------------------------------
class ConnMonitorConfigMode( ConnectivityMonitorBaseMode, BasicCli.ConfigModeBase ):
   name = 'Connectivity monitor configuration'

   def __init__( self, parent, session ):
      ConnectivityMonitorBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ConnMonitorConfigClientMode( ConnectivityMonitorClientMode,
                                   BasicCli.ConfigModeBase ):
   name = "Connectivity monitor client configuration"

   def __init__( self, parent, session, clientName ):
      self.clientName = clientName
      ConnectivityMonitorClientMode.__init__( self, clientName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ClientConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'client CLIENT'
   noOrDefaultSyntax = syntax
   data = {
         'client': 'Configure client for connectivity probes',
         'CLIENT': CliMatcher.PatternMatcher( pattern=clientHostPattern,
                                              helpdesc='Name of the client',
                                              helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      client = args[ 'CLIENT' ]
      if toggleOCConnectivityMonitorSupportEnabled() and client == 'default':
         mode.addErrorAndStop( 'Keyword default is a reserved word and cannot be '
                        'used as client name.' )
      configDir.newEntity( 'ConnectivityMonitor::ConfigDir', client )
      childMode = mode.childMode( ConnMonitorConfigClientMode, clientName=client )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      client = args[ 'CLIENT' ]
      clientConfig = configDir.get( client )
      if clientConfig:
         clientConfig.hostConfig.clear()
         configDir.deleteEntity( client )

ConnMonitorConfigMode.addCommandClass( ClientConfigCmd )

def allNameServerGroupNames( mode ):
   allNameServerGroups = {}
   # Get the list of NameServerGroup names from Sysdb
   for nameServer in internalNamespace:
      allNameServerGroups[ nameServer ] = nameServer
   return allNameServerGroups

class ConnMonitorConfigVrfMode( ConnectivityMonitorVrfMode,
                                BasicCli.ConfigModeBase ):
   name = "Connectivity monitor vrf configuration"

   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      ConnectivityMonitorVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getVrf( self ):
      if self.vrfName in config.vrf:
         return config.vrf[ self.vrfName ]
      else:
         # Could be deleted from a concurrent session
         self.addWarning( 'Configuration for %s has been deleted.' % self.vrfName )
         return None

   def addDescription( self, args ):
      # Sanity check the Description.
      if 'DESCRIPTION' in args:
         text = args[ 'DESCRIPTION' ]
      else:
         text = BasicCliUtil.getSingleLineInput(
                self, 'Enter Vrf description (max 140 characters): ' )

      if len( text ) > 140:
         self.addError( 'Description text is too long' )
         return
      vrf = self.getVrf()
      if not vrf:
         return
      vrf.description = text

   def noDescription( self, args ):
      vrf = self.getVrf()
      if not vrf:
         return
      vrf.description = ''

class VrfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfExprFactory( helpdesc='Configure VRF for connectivity probes' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrf = args[ 'VRF' ]
      if vrf not in getVrfNames():
         mode.addWarning( 'Invalid vrf %s. Hosts will not be operational.' % vrf )
      if vrf not in config.vrf:
         config.vrf.newMember( vrf )
      childMode = mode.childMode( ConnMonitorConfigVrfMode, vrfName=vrf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrf = args[ 'VRF' ]
      if vrf in config.vrf:
         for key in config.intfSet:
            if key.vrfName == vrf:
               del config.intfSet[ key ]
         for key in defaultClientConfig.hostConfig:
            if key.vrfName == vrf:
               del defaultClientConfig.hostConfig[ key ]
         del config.vrf[ vrf ]

ConnMonitorConfigMode.addCommandClass( VrfConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description
#--------------------------------------------------------------------------------
class VrfDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description [DESCRIPTION]'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description' : 'Configure a brief description of the VRF',
      'DESCRIPTION': CliMatcher.StringMatcher( helpdesc='Description of the VRF',
                                               helpname='description' ),
   }

   handler = ConnMonitorConfigVrfMode.addDescription
   noOrDefaultHandler = ConnMonitorConfigVrfMode.noDescription

ConnMonitorConfigVrfMode.addCommandClass( VrfDescriptionCmd )

def getIntfSets( mode ):
   vrf = getVrf( mode )
   return sorted( key.setName for key in config.intfSet if key.vrfName == vrf )

def createIntfSetKey( vrf, setName ):
   return Tac.newInstance( 'ConnectivityMonitor::IntfSetKey', vrf, setName )

def createIntfSetType( intfKey, useSrcIp ):
   intfSetType = Tac.newInstance( 'ConnectivityMonitor::IntfSetType', intfKey )
   intfSetType.useSrcIp = useSrcIp
   return intfSetType

def getDefaultIntfSetKey():
   # Is this the right way to set the default value ?
   return createIntfSetKey( '', '' )

def getDefaultIntfSetType():
   intfSetType = createIntfSetType( getDefaultIntfSetKey(), False )
   return intfSetType

setNameMatcher = CliMatcher.DynamicNameMatcher( getIntfSets, 'Interface sets',
     pattern=r'[.A-Za-z0-9_:{}\[\]-]+' )

def doSetIntfKeyInHosts( clientConfig, vrfToCompare, setKeyToCompare, valueToSet ):
   for key, host in clientConfig.hostConfig.items():
      if key.vrfName == vrfToCompare and host.intfSetKey == setKeyToCompare:
         host.intfSetKey = valueToSet

def setIntfKeyInHosts( vrfToCompare, setKeyToCompare, valueToSet ):
   doSetIntfKeyInHosts( defaultClientConfig, vrfToCompare, setKeyToCompare,
                        valueToSet )
   for clientConfig in configDir.values():
      doSetIntfKeyInHosts( clientConfig, vrfToCompare, setKeyToCompare, valueToSet )

def doSetIntfTypesInHosts( clientConfig, vrfToCompare, intfSetTypeToCompare,
                           valueToSet, onlyCompareKey ):
   for key, host in clientConfig.hostConfig.items():
      # if onlyCompareKey is true, check for consistency between just the
      # intfSetKey, don't worry about useSrcIp
      hostCompareVal = host.intfSetType.key if onlyCompareKey else host.intfSetType
      if onlyCompareKey:
         passedInCompareVal = intfSetTypeToCompare.key
      else:
         passedInCompareVal = intfSetTypeToCompare
      if key.vrfName == vrfToCompare and hostCompareVal == passedInCompareVal:
         host.intfSetType = valueToSet

def setIntfTypesInHosts( vrfToCompare, intfSetTypeToCompare, valueToSet,
                         onlyCompareKey=False ):
   doSetIntfTypesInHosts( defaultClientConfig, vrfToCompare, intfSetTypeToCompare,
                          valueToSet, onlyCompareKey )
   for clientConfig in configDir.values():
      doSetIntfTypesInHosts( clientConfig, vrfToCompare, intfSetTypeToCompare,
                             valueToSet, onlyCompareKey )

class LocalInterfacesExpression( CliCommand.CliExpression ):
   expression = 'local-interfaces SET'
   data = {
      'local-interfaces': 'Configure the default interface set for probing',
      'SET': setNameMatcher
   }

#------------------------------------------------------------------------------------
# (config-mon-connectivity)# no|default local-interfaces SET [ address-only ] default
#------------------------------------------------------------------------------------
class LocalInterfacesConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'EXPR [ address-only ] default'
   noOrDefaultSyntax = 'EXPR ...'
   data = {
      'EXPR': LocalInterfacesExpression,
      'address-only': 'Use interfaces for source addresses only',
      'default': 'Configure the given interface set as default'
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = getVrf( mode )
      vrf = VrfNameType( vrfName )
      useSrcIp = 'address-only' in args
      setName = args[ 'SET' ]
      intfKey = createIntfSetKey( vrf, setName )
      if intfKey in config.intfSet:
         vrfConfig = config.vrf[ vrf ]
         defaultIntfSet = vrfConfig.defaultIntfSet
         vrfConfig.defaultIntfSet = intfKey
         setIntfKeyInHosts( vrfName, defaultIntfSet, intfKey )

         staleIntfSetType = vrfConfig.defaultIntfSetType
         intfSetType = createIntfSetType( intfKey, useSrcIp )
         vrfConfig.defaultIntfSetType = intfSetType
         setIntfTypesInHosts( vrfName, staleIntfSetType, intfSetType )
      else:
         mode.addError( 'Invalid set %s' % setName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setName = args[ 'SET' ]
      vrfName = getVrf( mode )
      vrf = VrfNameType( vrfName )
      vrfConfig = config.vrf[ vrf ]
      defaultIntfSet = vrfConfig.defaultIntfSet
      defaultKey = getDefaultIntfSetKey()

      staleIntfSetType = vrfConfig.defaultIntfSetType
      defaultIntfSetType = getDefaultIntfSetType()
      if defaultIntfSet != defaultKey and defaultIntfSet.setName == setName:
         intfKey = createIntfSetKey( vrf, setName )
         setIntfKeyInHosts( vrfName, intfKey, defaultKey )
         vrfConfig.defaultIntfSet = defaultKey

         intfSetType = createIntfSetType( intfKey, staleIntfSetType.useSrcIp )
         setIntfTypesInHosts( vrfName, intfSetType, defaultIntfSetType )
         vrfConfig.defaultIntfSetType = defaultIntfSetType

ConnMonitorConfigMode.addCommandClass( LocalInterfacesConfigCmd )
ConnMonitorConfigVrfMode.addCommandClass( LocalInterfacesConfigCmd )

def getHostKey( mode ):
   return getattr( mode, 'hostKey', None )

#------------------------------------------------------------------------------------
# (config-mc-<host>)# no|default local-interfaces SET [ address-only ]
#------------------------------------------------------------------------------------
class LocalInterfacesHostConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'EXPR [ address-only ]'
   noOrDefaultSyntax = syntax
   data = {
      'EXPR': LocalInterfacesExpression,
      'address-only': 'Use interfaces for source addresses only'
   }

   @staticmethod
   def handler( mode, args ):
      hostKey = getHostKey( mode )
      if hostKey:
         setName = args[ 'SET' ]
         useSrcIp = 'address-only' in args
         key = createIntfSetKey( hostKey.vrfName, setName )
         intfSetType = createIntfSetType( key, useSrcIp )
         _config = getConfig( mode )
         if key in config.intfSet:
            _config.hostConfig[ hostKey ].intfSetKey = key
            _config.hostConfig[ hostKey ].intfSetType = intfSetType
         else:
            mode.addError( 'Invalid set %s' % setName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostKey = getHostKey( mode )
      if hostKey:
         setName = args[ 'SET' ]
         _config = getConfig( mode )
         host = _config.hostConfig[ hostKey ]
         if host.intfSetKey.setName == setName:
            host.intfSetKey = config.vrf[ hostKey.vrfName ].defaultIntfSet
            host.intfSetType = config.vrf[ hostKey.vrfName ].defaultIntfSetType

#------------------------------------------------------------------------------------
# (config-mon-connectivity)# no|default interface set SET INTFS
#------------------------------------------------------------------------------------
class InterfaceSetConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'interface set SET INTFS'
   noOrDefaultSyntax = 'interface set SET ...'
   data = {
      'interface': 'Configure a set of source interfaces',
      'set': 'Name of the interface set',
      'SET': setNameMatcher,
      'INTFS': IntfRange.intfRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      vrf = getVrf( mode )
      setName = args[ 'SET' ]
      key = createIntfSetKey( VrfNameType( vrf ), setName )
      sysdbIntfs = config.intfSet.newMember( key ).intf
      cliIntfs = set( args[ 'INTFS' ] )
      for intf in sysdbIntfs:
         if intf not in cliIntfs:
            sysdbIntfs.remove( intf )
      for intf in cliIntfs:
         sysdbIntfs.add( intf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = getVrf( mode )
      vrf = VrfNameType( vrfName )
      setName = args[ 'SET' ]
      key = createIntfSetKey( vrf, setName )
      # useSrcIp value doesn't matter, as we will just be comparing the key
      intfSetType = createIntfSetType( key, False )
      if key in config.intfSet:
         defaultIntfSet = config.vrf[ vrf ].defaultIntfSet
         defaultIntfSetType = config.vrf[ vrf ].defaultIntfSetType
         if defaultIntfSet == key:
            defaultIntfSet = getDefaultIntfSetKey()
            defaultIntfSetType = getDefaultIntfSetType()
            # If the interface set being deleted is the default interface set, make
            # sure to update that in config
            config.vrf[ vrf ].defaultIntfSet = defaultIntfSet
            config.vrf[ vrf ].defaultIntfSetType = defaultIntfSetType
         setIntfKeyInHosts( vrfName, key, defaultIntfSet )
         # If we delete the interface set, it doesn't matter what useSrcIp value is
         # in the host, we want to set the intfSetType of all hosts to be its
         # default value unconditionally.
         setIntfTypesInHosts( vrfName, intfSetType, defaultIntfSetType,
                              onlyCompareKey=True )
         del config.intfSet[ key ]

ConnMonitorConfigMode.addCommandClass( InterfaceSetConfigCmd )
ConnMonitorConfigVrfMode.addCommandClass( InterfaceSetConfigCmd )

class ConnMonitorConfigHostMode( ConnectivityMonitorHostMode,
      BasicCli.ConfigModeBase ):
   name = 'Connectivity monitor host configuration'

   def __init__( self, parent, session, clientName, hostKey ):
      self.clientName = clientName
      self.hostKey = hostKey
      ConnectivityMonitorHostMode.__init__( self, ( hostKey.hostName,
                                                    hostKey.vrfName,
                                                    clientName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getHost( self ):
      if self.clientName:
         _config = configDir[ self.clientName ]
      else:
         _config = defaultClientConfig
      if self.hostKey in _config.hostConfig:
         host = _config.hostConfig[ self.hostKey ]
         if not host.icmpConfig:
            host.icmpConfig = ( "icmpConfig", )
         if not host.httpConfig:
            host.httpConfig = ( "httpConfig", )
         if not host.tcpConfig:
            host.tcpConfig = ( "tcpConfig", )
         return host
      else:
         # Could be deleted from a concurrent session
         self.addWarning( 'Configuration for %s has been deleted.' % self.hostName )
         return None

   def setIcmpEnabled( self, disable=True ):
      if host := self.getHost():
         host.icmpConfig.icmpEnabled = not disable

   def setAddressFamily( self, probeConfig, af=None ):
      if not probeConfig:
         return
      if af:
         probeConfig.addressFamily = getattr( AddrFamily, af )
      else:
         probeConfig.addressFamily = getattr( AddrFamily, 'ipunknown' )

   def setIpOrHostname( self, probeConfig, addr=None, hostname=None, af=None ):
      if not probeConfig:
         return
      if addr is not None and not addr.isAddrZero:
         # clear hostname destination and address family
         probeConfig.hostname = ''
         self.setAddressFamily( probeConfig )
         # configure ip address
         probeConfig.ipAddr = Arnet.IpGenAddr( str( addr ) )
      elif hostname is not None:
         # clear ip address and address family
         probeConfig.ipAddr = Arnet.IpGenAddr()
         self.setAddressFamily( probeConfig )
         # configure hostname
         probeConfig.hostname = hostname
         # Since we are using hostname, set the af
         self.setAddressFamily( probeConfig, af=af )
      else:
         probeConfig.ipAddr = Arnet.IpGenAddr()
         probeConfig.hostname = ''
         self.setAddressFamily( probeConfig )

   def setIpOrHostnameIcmp( self, addr=None, hostname=None, af=None ):
      if not ( host := self.getHost() ):
         return
      icmpCfg = host.icmpConfig
      self.setIpOrHostname( icmpCfg, addr=addr, hostname=hostname, af=af )

   def setIpOrHostnameTcp( self, addr=None, hostname=None, af=None ):
      if not ( host := self.getHost() ):
         return
      tcpCfg = host.tcpConfig
      self.setIpOrHostname( tcpCfg, addr=addr, hostname=hostname, af=af )

   def setIpOrHostnameAll( self, addr=None, hostname=None, af=None ):
      self.setIpOrHostnameIcmp( addr=addr, hostname=hostname, af=af )
      self.setIpOrHostnameTcp( addr=addr, hostname=hostname, af=af )

   def setTcpPort( self, useDport=False, dport=0 ):
      if not ( host := self.getHost() ):
         return
      tcpCfg = host.tcpConfig
      tcpCfg.useDport = useDport
      tcpCfg.dport = dport

   def setNameServerGroup( self, nameServerGroup=None ):
      if not ( host := self.getHost() ):
         return
      if nameServerGroup:
         if nameServerGroup != "default":
            host.nameServerGroup = "internal-ns-" + nameServerGroup
         else:
            host.nameServerGroup = nameServerGroup
      else:
         host.nameServerGroup = host.nameServerGroupDefault

   def setPingSize( self, size=None ):
      if not ( host := self.getHost() ):
         return
      icmpCfg = host.icmpConfig
      # default ping size is set to 56
      # making the size of the packet 64 bytes
      # i.e 56 bytes payload + 8 bytes header
      icmpCfg.pingSize = size if size else icmpCfg.pingSizeDefault

   def setIcmpDscp( self, dscpVal=None, dscpNameUsed=False ):
      if not ( host := self.getHost() ):
         return
      icmpCfg = host.icmpConfig
      if dscpVal:
         icmpCfg.icmpDscp = dscpVal
      else:
         # default DSCP bits is set to an invalid value
         # so that it is not taken into effect on TOS bit
         icmpCfg.icmpDscp = icmpCfg.icmpDscpDefault
      icmpCfg.icmpDscpNameUsed = dscpNameUsed

   def setPingCount( self, pingCount=None ):
      if not ( host := self.getHost() ):
         return
      icmpCfg = host.icmpConfig
      if pingCount:
         icmpCfg.pingCount = pingCount
      else:
         icmpCfg.pingCount = icmpCfg.pingCountDefault

   def setUrl( self, url=None ):
      if not ( host := self.getHost() ):
         return
      httpCfg = host.httpConfig
      if url:
         # Sanity check the URL.
         urlObj = urllib.parse.urlparse( url )
         if not urlObj.hostname:
            self.addError( '%s is an invalid url' % url )
            return

         # Add the URL.
         httpCfg.url = url
      else:
         httpCfg.url = ''

   def setDescription( self, text=None ):
      if not ( host := self.getHost() ):
         return

      if text:
         if len( text ) > 140:
            self.addError( 'Description text is too long' )
            return
         host.description = text
      else:
         host.description = ''

   def onExit( self ):
      if not ( host := self.getHost() ):
         return
      icmpCfg = host.icmpConfig
      httpCfg = host.httpConfig
      if ( not icmpCfg.ipAddr or icmpCfg.ipAddr.isAddrZero ) and \
         ( not icmpCfg.hostname ) and ( not httpCfg.url ):
         self.addWarning(
               'No ICMP destination endpoint or URL specified for host %s. '
               'Host will not be operational.' % host.key.hostName )

      BasicCli.ConfigModeBase.onExit( self )

ConnMonitorConfigHostMode.addCommandClass( LocalInterfacesHostConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Shutdown connectivity monitor',
   }

   @staticmethod
   def handler( mode, args ):
      disableConnectivityMonitor( mode )

   @staticmethod
   def noHandler( mode, args ):
      config.enabled = True

   defaultHandler = handler

ConnMonitorConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor connectivity
#--------------------------------------------------------------------------------
def disableConnectivityMonitor( mode ):
   config.enabled = False

def setInterval( mode, interval=None ):
   _config = getConfig( mode )
   _config.probeInterval = interval if interval else _config.probeIntervalDefault

def setPingCount( mode, pingCount=None ):
   _config = getConfig( mode )
   _config.pingCount = pingCount if pingCount else _config.pingCountDefault

def setLossThreshold( mode, lossThreshold=None ):
   _config = getConfig( mode )
   _config.probeLossThreshold = lossThreshold if lossThreshold else 0

def setNameServerGroup( mode, nameServerGroup=None ):
   _config = getConfig( mode )
   if nameServerGroup:
      if nameServerGroup != "default":
         _config.nameServerGroup = "internal-ns-" + nameServerGroup
      else:
         _config.nameServerGroup = nameServerGroup
   else:
      _config.nameServerGroup = _config.nameServerGroupDefault

def setIcmpDscp( mode, dscpVal=None, dscpNameUsed=False ):
   _config = getConfig( mode )
   _config.icmpDscp = dscpVal if dscpVal is not None else _config.icmpDscpDefault
   _config.icmpDscpNameUsed = dscpNameUsed

def delHost( mode, hostName ):
   del config.hostConfig[ hostName ]

class MonitorConnectivityCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor connectivity'
   noOrDefaultSyntax = syntax
   data = {
      'monitor' : CliToken.Monitor.monitorMatcher,
      'connectivity' : 'Connectivity monitor configuration',
   }

   @staticmethod
   def handler( mode, args ):
      config.vrf.newMember( IpLibConsts.DEFAULT_VRF )
      childMode = mode.childMode( ConnMonitorConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Restore defaults for all parameter.
      config.intfSet.clear()
      config.vrf.clear()
      setInterval( mode )
      setPingCount( mode )
      setLossThreshold( mode )
      setNameServerGroup( mode )
      setIcmpDscp( mode )
      for client in configDir:
         configDir[ client ].hostConfig.clear()
         configDir.deleteEntity( client )
      # Also clear config for default client
      defaultClientConfig.hostConfig.clear()
      disableConnectivityMonitor( mode )

BasicCli.GlobalConfigMode.addCommandClass( MonitorConnectivityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] loss-threshold THRESHOLD
#--------------------------------------------------------------------------------
class LossThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'loss-threshold THRESHOLD probes'
   noOrDefaultSyntax = 'loss-threshold ...'
   data = {
      'loss-threshold' : 'Configure the sequential lost probe count threshold',
      'THRESHOLD' : CliMatcher.IntegerMatcher( 1, 65535, helpdesc='Count' ),
      'probes' : 'Units in probes'
   }

   @staticmethod
   def handler( mode, args ):
      lossThreshold =  args[ 'THRESHOLD' ]
      setLossThreshold( mode, lossThreshold=lossThreshold )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setLossThreshold( mode )

ConnMonitorConfigMode.addCommandClass( LossThresholdCmd )
ConnMonitorConfigClientMode.addCommandClass( LossThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] interval INTERVAL
#--------------------------------------------------------------------------------
class IntervalIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'interval INTERVAL'
   noOrDefaultSyntax = 'interval ...'
   data = {
      'interval' : 'Configure probe interval in seconds (default is 10 seconds, ' +\
            'minimum is 5 seconds)',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 65535, helpdesc='Probe interval' ),
   }

   @staticmethod
   def handler( mode, args ):
      interval =  args[ 'INTERVAL' ]
      if interval < 5 :
         mode.addWarning( 'Using the minimum allowed interval of 5 seconds.' )
         interval = 5
      setInterval( mode, interval )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setInterval( mode )

ConnMonitorConfigMode.addCommandClass( IntervalIntervalCmd )
ConnMonitorConfigClientMode.addCommandClass( IntervalIntervalCmd )

# --------------------------------------------------------------------------------
# [ no | default ] icmp count COUNT
# --------------------------------------------------------------------------------
class PingCountCmd( CliCommand.CliCommandClass ):
   syntax = 'icmp count COUNT'
   noOrDefaultSyntax = 'icmp count ...'
   data = {
      'icmp': 'Configure ICMP ping',
      'count': 'Configure ping count (default is 5)',
      'COUNT': CliMatcher.IntegerMatcher( 1, 100, helpdesc='Ping count' ),
   }

   @staticmethod
   def handler( mode, args ):
      pingCount = args[ 'COUNT' ]
      if isinstance( mode, ConnMonitorConfigHostMode ):
         mode.setPingCount( pingCount )
         return
      setPingCount( mode, pingCount )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if isinstance( mode, ConnMonitorConfigHostMode ):
         mode.setPingCount()
         return
      setPingCount( mode )

if toggleCMIcmpPingCountEnabled():
   ConnMonitorConfigMode.addCommandClass( PingCountCmd )
   ConnMonitorConfigClientMode.addCommandClass( PingCountCmd )
   ConnMonitorConfigHostMode.addCommandClass( PingCountCmd )

# ----------------------------------------------------------------------------------
# (config-mon-connectivity)# no|default name-server group GROUP
# ----------------------------------------------------------------------------------
# ----------------------------------------------------------------------------------
# (mc-client-<clientname>)# no|default name-server group GROUP
# ----------------------------------------------------------------------------------
# ----------------------------------------------------------------------------------
# (config-mc-<host>)# no|default name-server group GROUP
# ----------------------------------------------------------------------------------
class NameServerGroupCmd( CliCommand.CliCommandClass ):
   '''Configures the name-server group which is the namespace that
      ConnectivityMonitor will use to do http probes'''
   syntax = "name-server group GROUP"
   noOrDefaultSyntax = 'name-server group ...'
   data = {
      'name-server': 'Configure the name server address',
      'group': 'Configure the group name',
      # Note that this matcher also accepts any word as NameServerGroup name,
      # in addition to dynamically-detected names from ip name-server group.
      'GROUP': CliMatcher.DynamicNameMatcher( allNameServerGroupNames,
                  helpdesc='NameServerGroup name', pattern='.+' )
      }

   @staticmethod
   def handler( mode, args ):
      nameServerGroup = args[ 'GROUP' ]
      if isinstance( mode, ConnMonitorConfigHostMode ):
         mode.setNameServerGroup( nameServerGroup=nameServerGroup )
         return
      setNameServerGroup( mode, nameServerGroup=nameServerGroup )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if isinstance( mode, ConnMonitorConfigHostMode ):
         mode.setNameServerGroup()
         return
      setNameServerGroup( mode )

ConnMonitorConfigMode.addCommandClass( NameServerGroupCmd )
ConnMonitorConfigClientMode.addCommandClass( NameServerGroupCmd )
ConnMonitorConfigHostMode.addCommandClass( NameServerGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] logging
#--------------------------------------------------------------------------------
class LoggingCmd( CliCommand.CliCommandClass ):
   syntax = 'logging'
   noOrDefaultSyntax = syntax
   data = {
      'logging' : 'Enable connectivity monitor logging',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      config.loggingEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.loggingEnabled = False

ConnMonitorConfigMode.addCommandClass( LoggingCmd )

def createHostNameVrfKey( hostName, vrf ):
   return Tac.newInstance( 'ConnectivityMonitor::HostNameVrfKey', hostName, vrf )

#--------------------------------------------------------------------------------
# [ no | default ] host HOSTNAME
#--------------------------------------------------------------------------------
class HostHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'host HOSTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'host' : 'Configure host parameters',
      'HOSTNAME' : matcherHostName,
   }

   @staticmethod
   def handler( mode, args ):
      hostName = args[ 'HOSTNAME' ]
      vrf = VrfNameType( getVrf( mode ) )
      key = createHostNameVrfKey( hostName, vrf )
      client = getClientName( mode )
      if client:
         host = configDir[ client ].hostConfig.newMember( key )
      else:
         host = defaultClientConfig.hostConfig.newMember( key )
      host.icmpConfig = ( "icmpConfig", )
      host.httpConfig = ( "httpConfig", )
      host.tcpConfig = ( "tcpConfig", )
      defaultKey = getDefaultIntfSetKey()
      if config.vrf[ vrf ].defaultIntfSet != defaultKey and \
         host.intfSetKey == defaultKey:
         host.intfSetKey = config.vrf[ vrf ].defaultIntfSet
         host.intfSetType = config.vrf[ vrf ].defaultIntfSetType

      # Transition into the new mode.
      childMode = mode.childMode( ConnMonitorConfigHostMode, clientName=client,
            hostKey=key )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostName = args[ 'HOSTNAME' ]
      vrf = VrfNameType( getVrf( mode ) )
      key = createHostNameVrfKey( hostName, vrf )
      _config = getConfig( mode )
      del _config.hostConfig[ key ]

ConnMonitorConfigMode.addCommandClass( HostHostnameCmd )
ConnMonitorConfigVrfMode.addCommandClass( HostHostnameCmd )
ConnMonitorConfigClientMode.addCommandClass( HostHostnameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] tcp port PORT
# --------------------------------------------------------------------------------
class TcpPortCmd( CliCommand.CliCommandClass ):
   _tcpPort = 'tcp port'
   syntax = _tcpPort + ' PORT'
   noOrDefaultSyntax = _tcpPort + ' ...'
   data = {
         'tcp': 'Configure TCP parameters',
         'port': 'Configure the TCP server port',
         'PORT': CliMatcher.IntegerMatcher( 0, 65535,
                                             helpdesc='TCP server port' ),
   }

   @staticmethod
   def handler( mode, args ):
      dport = args.get( 'PORT' )
      mode.setTcpPort( useDport=True, dport=dport )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # set port = 0. This is not a default value, as it is a reserved (but
      # technically valid) tcp port 0 is just an easy default value
      mode.setTcpPort( useDport=False )

if toggleCMTcpProbeEnabled():
   ConnMonitorConfigHostMode.addCommandClass( TcpPortCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ip ( ADDR | ( hostname HOSTNAME [ resolution AF ] ) )
# --------------------------------------------------------------------------------
class IpOrHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ( ADDR | ( hostname HOSTNAME [ resolution AF ] ) )'
   noOrDefaultSyntax = syntax
   data = {
      'ip': 'Configure endpoint for ICMP probing',
      'ADDR': IpGenAddrMatcher.IpGenAddrMatcher( 'IPv4 or IPv6 address',
                                                   helpdesc4='IPv4 address',
                                                   helpdesc6='IPv6 address' ),
      'hostname': 'Resolve IP address from a hostname',
      'HOSTNAME': HostnameCli.HostnameMatcher( helpname='WORD',
                                                helpdesc='Hostname destination' ),
      'resolution': 'Hostname resolution',
      'AF': CliMatcher.EnumMatcher( {
         'ipv4': 'Set IPv4 as preferred address family',
         'ipv6': 'Set IPv6 as preferred address family',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      if toggleCMTcpProbeEnabled():
         if hostname := args.get( 'HOSTNAME' ):
            af = args.get( 'AF' )
            mode.setIpOrHostnameAll( hostname=hostname, af=af )
         else:
            mode.setIpOrHostnameAll( addr=args[ 'ADDR' ] )
      else:
         if hostname := args.get( 'HOSTNAME' ):
            af = args.get( 'AF' )
            mode.setIpOrHostnameIcmp( hostname=hostname, af=af )
         else:
            mode.setIpOrHostnameIcmp( addr=args[ 'ADDR' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if toggleCMTcpProbeEnabled():
         if host := mode.getHost():
            if addr := args.get( 'ADDR' ):
               if host.icmpConfig.ipAddr == addr:
                  mode.setIpOrHostnameAll()
            if hostname := args.get( 'HOSTNAME' ):
               if host.icmpConfig.hostname == hostname:
                  mode.setIpOrHostnameAll()
      else:
         if host := mode.getHost():
            if addr := args.get( 'ADDR' ):
               if host.icmpConfig.ipAddr == addr:
                  mode.setIpOrHostnameIcmp()
            if hostname := args.get( 'HOSTNAME' ):
               if host.icmpConfig.hostname == hostname:
                  mode.setIpOrHostnameIcmp()

ConnMonitorConfigHostMode.addCommandClass( IpOrHostnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] icmp echo size SIZE
#--------------------------------------------------------------------------------
class PingCmd( CliCommand.CliCommandClass ):
   syntax = 'icmp echo size SIZE'
   noOrDefaultSyntax = syntax
   data = {
      'icmp' : 'Configure ICMP ping',
      'echo' : 'Configure ICMP ping',
      'size' : 'Configure ICMP ping packet size',
      'SIZE' : CliMatcher.IntegerMatcher( 36, 18024,
                                          helpdesc='Ping packet size in bytes' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.setPingSize( size=args[ 'SIZE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if host:= mode.getHost():
         size = args[ 'SIZE' ]
         if host.icmpConfig.pingSize == size:
            mode.setPingSize()

ConnMonitorConfigHostMode.addCommandClass( PingCmd )

# ----------------------------------------------------------------------------------
# (config-mon-connectivity)# no|default icmp echo qos dscp ( DSCP | DSCP_ACL )
# ----------------------------------------------------------------------------------
# ----------------------------------------------------------------------------------
# (mc-client-<clientname>)# no|default icmp echo qos dscp ( DSCP | DSCP_ACL )
# ----------------------------------------------------------------------------------
# ----------------------------------------------------------------------------------
# (config-mc-<host>)# no|default icmp echo qos dscp ( DSCP | DSCP_ACL )
# ----------------------------------------------------------------------------------
class IcmpDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'icmp echo qos dscp ( DSCP | DSCP_ACL )'
   noOrDefaultSyntax = 'icmp echo qos dscp ...'
   data = {
      'icmp': 'Configure ICMP ping',
      'echo': 'Configure ICMP ping',
      'qos': 'Configure QoS parameters',
      'dscp': 'Set DSCP value in IP header',
      'DSCP': CliMatcher.IntegerMatcher( *AclCliLib.dscpRange(),
                                          helpdesc='DSCP value' ),
      'DSCP_ACL': CliMatcher.DynamicKeywordMatcher( AclCliLib.getDscpAclNames )
   }

   @staticmethod
   def handler( mode, args ):
      dscpVal = args.get( 'DSCP' )
      dscpNameUsed = False
      if not dscpVal:
         dscpAclVal = args.get( 'DSCP_ACL' )
         dscpVal, _ = AclCliLib.dscpValueFromCli( mode, dscpAclVal )
         dscpNameUsed = True
      if isinstance( mode, ConnMonitorConfigHostMode ):
         mode.setIcmpDscp( dscpVal=dscpVal, dscpNameUsed=dscpNameUsed )
         return
      setIcmpDscp( mode, dscpVal=dscpVal, dscpNameUsed=dscpNameUsed )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if isinstance( mode, ConnMonitorConfigHostMode ):
         mode.setIcmpDscp()
         return
      setIcmpDscp( mode )

ConnMonitorConfigMode.addCommandClass( IcmpDscpCmd )
ConnMonitorConfigClientMode.addCommandClass( IcmpDscpCmd )
ConnMonitorConfigHostMode.addCommandClass( IcmpDscpCmd )

# --------------------------------------------------------------------------------
# [ no | default ] icmp echo disabled
# --------------------------------------------------------------------------------
class ShutdownIcmpCmd( CliCommand.CliCommandClass ):
   syntax = 'icmp echo disabled'
   noOrDefaultSyntax = syntax
   data = {
      'icmp': 'Configure ICMP ping',
      'echo': 'Configure ICMP ping',
      'disabled': 'Shut down ICMP probing'
   }

   @staticmethod
   def handler( mode, args ):
      mode.setIcmpEnabled()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setIcmpEnabled( False )

if toggleCMTcpProbeEnabled():
   ConnMonitorConfigHostMode.addCommandClass( ShutdownIcmpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] url URL
#--------------------------------------------------------------------------------
class UrlUrlCmd( CliCommand.CliCommandClass ):
   syntax = 'url URL'
   noOrDefaultSyntax = 'url ...'
   data = {
      'url' : 'Configure a URL for the host',
      'URL' : CliMatcher.PatternMatcher( pattern='.+',
         helpdesc='URL of the host', helpname='url' ),
   }

   @staticmethod
   def handler( mode, args ):
      url = args[ 'URL' ]
      mode.setUrl( url=url )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setUrl()

ConnMonitorConfigHostMode.addCommandClass( UrlUrlCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description
#--------------------------------------------------------------------------------
class HostDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description [DESCRIPTION]'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description' : 'Configure a brief description of the host',
      'DESCRIPTION': CliMatcher.StringMatcher( helpdesc='Description of the host',
                                               helpname='description' ),
   }

   @staticmethod
   def handler( mode, args ):
      text = args.get( 'DESCRIPTION' )
      if not text:
         text = BasicCliUtil.getSingleLineInput(
                mode, 'Enter host description (max 140 characters): ' )
      mode.setDescription( text=text )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setDescription()

ConnMonitorConfigHostMode.addCommandClass( HostDescriptionCmd )

def Plugin( entityManager ):
   global config, status, defaultClientConfig, defaultClientStatus, configDir, \
         statusDir, internalNamespace

   config = ConfigMount.mount( entityManager, 'connectivityMonitor/config',
                               'ConnectivityMonitor::Config', 'w' )
   status =  LazyMount.mount( entityManager, 'connectivityMonitor/status',
                              'ConnectivityMonitor::Status', 'r' )
   defaultClientConfig = ConfigMount.mount(
         entityManager, 'connectivityMonitor/clientConfigDir/default',
         'ConnectivityMonitor::ConfigDir', 'w' )
   defaultClientStatus = LazyMount.mount(
         entityManager, 'connectivityMonitor/clientStatusDir/default',
         'ConnectivityMonitor::StatusDir', 'r' )
   configDir = ConfigMount.mount(
         entityManager, 'connectivityMonitor/clientConfigDir/clients', 'Tac::Dir',
         'wi' )
   statusDir = LazyMount.mount(
         entityManager, 'connectivityMonitor/clientStatusDir/clients', 'Tac::Dir',
         'ri' )
   internalNamespace = LazyMount.mount(
         entityManager, 'sys/net/internalNamespace', 'Tac::Dir', 'ri' )
