# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import (
      ConnectivityMonitorCli,
      ConnectivityMonitorModel,
      TechSupportCli,
)
import BasicCli
import CliToken.Tracers
import LazyMount
import ShowCommand
import TacSigint
from Toggles.ConnectivityMonitorToggleLib import (
   toggleCMTcpProbeEnabled
)

# Globals
_status = None
_config = None
_configDir = None
_statusDir = None
_defaultClientConfig = None
_defaultClientStatus = None

#--------------------------------------------------------------------------------
# show monitor connectivity [ host HOSTNAME ]
#--------------------------------------------------------------------------------
class MonitorConnectivityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor connectivity [ host HOSTNAME ]'
   data = {
      'monitor' : CliToken.Tracers.monitorMatcherForShow,
      'connectivity' : 'Connectivity monitor information',
      'host' : 'Connectivity monitor information of a single host',
      'HOSTNAME' : ConnectivityMonitorCli.matcherHostName,
   }
   cliModel = ConnectivityMonitorModel.ConnMonitorVrfs

   @staticmethod
   def handler( mode, args ):
      host = args.get( 'HOSTNAME' )
      connMonVrfs = ConnectivityMonitorModel.ConnMonitorVrfs()
      hostStatus = []
      vrfModels = {}
      defaultClientProbeLossThreshold = _defaultClientConfig.probeLossThreshold
      for key in _defaultClientConfig.hostConfig:
         hostStatusVal = _defaultClientStatus.hostStatus.get( key )
         if hostStatusVal:
            if not host:
               hostStatus.append( ( '', hostStatusVal ) )
            elif host and key.hostName == host:
               hostStatus.append( ( '', hostStatusVal ) )
      for client in _configDir:
         for key in _configDir[ client ].hostConfig:
            if client in _statusDir:
               clientHostStatus = _statusDir[ client ].hostStatus.get( key )
               if clientHostStatus:
                  _client = client if client != 'default' else ''
                  if not host:
                     hostStatus.append( ( _client, clientHostStatus ) )
                  elif host and key.hostName == host:
                     hostStatus.append( ( _client, clientHostStatus ) )

      for client, status in hostStatus:
         if client:
            config = _configDir[ client ]
         else:
            config = _defaultClientConfig
         hostConfig = config.hostConfig[ status.key ]
         hostName = status.key.hostName
         vrfName = status.key.vrfName
         vrfConfig = _config.vrf[ vrfName ]
         vrfModel = ConnectivityMonitorModel.ConnMonitorVrf(
                                             description=vrfConfig.description )
         # pylint: disable=protected-access
         vrfModel._probeLossThreshold = defaultClientProbeLossThreshold
         if vrfName in vrfModels: # pylint: disable=consider-using-get
            vrfModel = vrfModels[ vrfName ]
         hostModel = None
         if client:
            clientModel = vrfModel.clients.get( client )
            if clientModel is None:
               clientModel = ConnectivityMonitorModel.ConnMonitorClient()
               # pylint: disable=protected-access
               clientModel._probeLossThreshold = config.probeLossThreshold
               vrfModel.clients[ client ] = clientModel
            hostModel = clientModel.hosts.get( hostName )
         else:
            hostModel = vrfModel.hosts.get( hostName )
         if hostModel is None:
            if ( not hostConfig.icmpConfig.ipAddr.isAddrZero ) or \
                 hostConfig.icmpConfig.hostname or \
                 ( not hostConfig.tcpConfig.ipAddr.isAddrZero ) or \
                 hostConfig.tcpConfig.hostname or \
                 hostConfig.httpConfig.url:
               hostModel = ConnectivityMonitorModel.ConnMonitorHost()
               hostModel.fromTacc( hostConfig )
               if hostConfig.icmpConfig.hostname:
                  hostModel.configError = status.configError
               if hostConfig.icmpConfig.icmpDscp == \
                  hostConfig.icmpConfig.icmpDscpDefault:
                  hostModel.icmpDscp = config.icmpDscp
               if hostConfig.icmpConfig.pingCount == \
                  hostConfig.icmpConfig.pingCountDefault:
                  hostModel.pingCount = config.pingCount
            else:
               continue

         icmpStatsKeyToStatModel = {}
         for icmpStatsKey, icmpStat in status.icmpStats.items():
            statModel = ConnectivityMonitorModel.IcmpStat()
            statModel.fromTacc( icmpStat )
            intfname = icmpStatsKey.interfaceName
            interface = intfname if intfname != '' else 'default'
            statsKey = ( icmpStatsKey.ipAddr, interface )
            icmpStatsKeyToStatModel[ statsKey ] = statModel

         httpStatsKeyToStatModel = {}
         for httpStatsKey, httpStat in status.httpStats.items():
            statModel = ConnectivityMonitorModel.HttpStat()
            statModel.fromTacc( httpStat )
            intfname = httpStatsKey.interfaceName
            interface = intfname if intfname != '' else 'default'
            statsKey = ( httpStatsKey.url, interface )
            httpStatsKeyToStatModel[ statsKey ] = statModel

         tcpStatsKeyToStatModel = {}
         if toggleCMTcpProbeEnabled():
            for tcpStatsKey, tcpStat in status.tcpStats.items():
               statModel = ConnectivityMonitorModel.TcpStat()
               statModel.fromTacc( tcpStat )
               intfname = tcpStatsKey.interfaceName
               interface = intfname if intfname != '' else 'default'
               statsKey = ( tcpStatsKey.ipAddr, interface, tcpStatsKey.dport )
               tcpStatsKeyToStatModel[ statsKey ] = statModel

         intfToReachabilityStats = {}
         statModel = ConnectivityMonitorModel.Stats()

         # Iterate over all of the interfaces in reachabilityStats
         for intf, reachabilityStats in status.reachabilityStats.items():
            intf = intf if intf != '' else 'default'
            statModel = ConnectivityMonitorModel.Stats()
            statModel.sequentialFailedProbeCount = \
                  reachabilityStats.sequentialFailedProbeCountIcmp
            statModel.hostUnreachable = reachabilityStats.hostUnreachable
            statModel.lastResponseAt = reachabilityStats.lastResponseAtIcmp
            intfToReachabilityStats[ intf ] = statModel

         interfaces = hostModel.interfaces

         for ( ipAddr, intf ), statModel in icmpStatsKeyToStatModel.items():
            if intf not in interfaces:
               if intf in intfToReachabilityStats:
                  interfaces[ intf ] = intfToReachabilityStats[ intf ]
               else:
                  interfaces[ intf ] = ConnectivityMonitorModel.Stats()
            interfaces[ intf ].icmpDestinations[ ipAddr ] = statModel

            if client:
               clientModel.hosts[ hostName ] = hostModel
            else:
               vrfModel.hosts[ hostName ] = hostModel

         for ( url, intf ), statModel in httpStatsKeyToStatModel.items():
            if intf not in interfaces:
               if intf in intfToReachabilityStats:
                  interfaces[ intf ] = intfToReachabilityStats[ intf ]
               else:
                  interfaces[ intf ] = ConnectivityMonitorModel.Stats()
            interfaces[ intf ].httpDestinations[ url ] = statModel

            if client:
               clientModel.hosts[ hostName ] = hostModel
            else:
               vrfModel.hosts[ hostName ] = hostModel

         if toggleCMTcpProbeEnabled():
            for ( ipAddr, intf, dport ), statModel in tcpStatsKeyToStatModel.items():
               if intf not in interfaces:
                  if intf in intfToReachabilityStats:
                     interfaces[ intf ] = intfToReachabilityStats[ intf ]
                  else:
                     interfaces[ intf ] = ConnectivityMonitorModel.Stats()
               addrPort = ipAddr.stringValue + ':' + str( dport )
               interfaces[ intf ].tcpDestinations[ addrPort ] = statModel

               if client:
                  clientModel.hosts[ hostName ] = hostModel
               else:
                  vrfModel.hosts[ hostName ] = hostModel

         # Configuration error is also considered as stats for ICMP probes
         if hostModel.configError:
            if client:
               clientModel.hosts[ hostName ] = hostModel
            else:
               vrfModel.hosts[ hostName ] = hostModel

         # Make sure the vrfModel is not empty before adding it to vrfModels. If
         # config under a nonexisit vrf is configured, probe won't be started but
         # the hostStatus for that hostNameVrfKey will be created.
         if icmpStatsKeyToStatModel or httpStatsKeyToStatModel or \
            tcpStatsKeyToStatModel or hostModel.configError:
            vrfModels[ vrfName ] = vrfModel
         elif ( ( hostModel.ipAddr and not hostModel.ipAddr.isAddrZero ) or
              hostModel.hostname ) and not hostModel.icmpEnabled:
            # If ICMP probe is disabled, icmpStatsKeyToStatModel will be empty
            # but we still want to show the configured endpoint if there is any
            if client:
               clientModel.hosts[ hostName ] = hostModel
            else:
               vrfModel.hosts[ hostName ] = hostModel
            vrfModels[ vrfName ] = vrfModel

         TacSigint.check()
      for name, vm in vrfModels.items():
         connMonVrfs.vrfs[ name ] = vm
      return connMonVrfs

BasicCli.addShowCommandClass( MonitorConnectivityCmd )

TechSupportCli.registerShowTechSupportCmd( '2024-04-29 18:18:19',
                                           cmds=[ 'show monitor connectivity' ] )

def Plugin( entityManager ):
   global _status, _config, _configDir, _statusDir, \
          _defaultClientConfig, _defaultClientStatus

   _status = LazyMount.mount( entityManager,
         'connectivityMonitor/status', 'ConnectivityMonitor::Status', 'r' )
   _config = LazyMount.mount( entityManager,
         'connectivityMonitor/config', 'ConnectivityMonitor::Config', 'r' )
   _configDir = LazyMount.mount(
         entityManager, 'connectivityMonitor/clientConfigDir/clients', 'Tac::Dir',
         'ri' )
   _statusDir = LazyMount.mount(
         entityManager, 'connectivityMonitor/clientStatusDir/clients', 'Tac::Dir',
         'ri' )
   _defaultClientConfig = LazyMount.mount(
         entityManager, 'connectivityMonitor/clientConfigDir/default',
         'ConnectivityMonitor::ConfigDir', 'ri' )
   _defaultClientStatus = LazyMount.mount(
         entityManager, 'connectivityMonitor/clientStatusDir/default',
         'ConnectivityMonitor::StatusDir', 'ri' )
