#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
import ConfigMount
from CliMode.ContainerMode import ContainerMode
from CliMode.ContainerPersistStorageMode import ContainerPersistStorageMode
from CliMode.ContainerProfileMode import ContainerProfileMode
from CliMode.ContainerProfilePersistStorageMode import (
      ContainerProfilePersistStorageMode )
from CliMode.ContainerProfileStartupMode import ContainerProfileStartupMode
from CliMode.ContainerStartupMode import ContainerStartupMode
from CliPlugin.ContainerMgrCliLib import (
      containerNameRe, pathRe, profileNameRe, imageNameRe )
import Tac
from Toggles.ContainerMgrToggleLib import toggleContainerStartupConditionEnabled
from TypeFuture import TacLazyType
import Url
# pkgdeps: import UrlPlugin.FlashUrl

containerConfigDir = None
profileConfigDir = None

ContainerParams = TacLazyType( 'ContainerMgr::ContainerParams' )
CpuShares = TacLazyType( 'ContainerMgr::CpuShares' )
CpuCores = TacLazyType( 'ContainerMgr::CpuCores' )
MemoryHardLimit = TacLazyType( 'ContainerMgr::MemoryHardLimit' )
MemorySoftLimit = TacLazyType( 'ContainerMgr::MemorySoftLimit' )
SecurityMode = TacLazyType( 'ContainerMgr::SecurityMode' )
NetworkingMode = TacLazyType( 'ContainerMgr::NetworkingMode' )
LoggingDriver = TacLazyType( 'ContainerMgr::LoggingDriver' )
RestartPolicy = TacLazyType( 'ContainerMgr::RestartPolicy' )
BindMountEntry = TacLazyType( 'ContainerMgr::BindMountEntry' )
EnabledState = TacLazyType( 'ContainerMgr::EnabledState' )
StartupConditionPollInterval = \
      TacLazyType( 'ContainerMgr::StartupConditionPollInterval' )

containerNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: containerConfigDir.container,
   pattern=containerNameRe,
   helpname='WORD',
   helpdesc='Name of the container' )

profileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: profileConfigDir.profile,
   pattern=profileNameRe,
   helpname='WORD',
   helpdesc='Name of the profile' )

imageNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: [ container.params.imageName for container in
                  containerConfigDir.container.values() ],
   pattern=imageNameRe,
   helpname='WORD',
   helpdesc='Image name of the container' )

def setConfigParam( containerOrProfileConfig, paramName, paramVal ):
   params = Tac.nonConst( containerOrProfileConfig.params )
   setattr( params, paramName, paramVal )
   containerOrProfileConfig.params = params

def isMemoryLimitInvalid( memory, minLimit ):
   memoryUnits = [ 'b', 'k', 'm', 'g' ]
   if memory.endswith( tuple( memoryUnits ) ):
      memoryUnit = memory[ len( memory ) - 1 ]
      index = memoryUnits.index( memoryUnit )
      memoryInBytes = int( memory[ 0 : len( memory ) - 1 ] ) * ( 1024 ** index )
   else:
      memoryInBytes = int( memory )
   return memoryInBytes < minLimit

class ContainerStartupConfigMode( ContainerStartupMode,
                                  BasicCli.ConfigModeBase ):
   name = "config-container-startup"

   def __init__( self, parent, session ):
      self.session_ = session
      ContainerStartupMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getConfig( self ):
      return self.parent_.getConfig()

class ContainerProfileStartupConfigMode( ContainerProfileStartupMode,
                                         BasicCli.ConfigModeBase ):
   name = "config-container-profile-startup"

   def __init__( self, parent, session ):
      self.session_ = session
      ContainerProfileStartupMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getConfig( self ):
      return self.parent_.getConfig()

class ContainerPersistStorageConfigMode( ContainerPersistStorageMode,
                                         BasicCli.ConfigModeBase ):
   name = "config-container-persist-storage"

   def __init__( self, parent, session ):
      self.session_ = session
      ContainerPersistStorageMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getConfig( self ):
      return self.parent_.getConfig()

class ContainerProfilePersistStorageConfigMode( ContainerProfilePersistStorageMode,
                                                BasicCli.ConfigModeBase ):
   name = "config-container-profile-persist-storage"

   def __init__( self, parent, session ):
      self.session_ = session
      ContainerProfilePersistStorageMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getConfig( self ):
      return self.parent_.getConfig()

class ContainerConfigMode( ContainerMode, BasicCli.ConfigModeBase ):
   name = "Container Manager Container Configuration"

   def __init__( self, parent, session, containerName ):
      self.containerName_ = containerName
      self.session_ = session
      ContainerMode.__init__( self, containerName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getConfig( self ):
      return containerConfigDir.container[ self.containerName_ ]

   def gotoStartupConfigMode( self ):
      childMode = self.childMode( ContainerStartupConfigMode )
      self.session_.gotoChildMode( childMode )

   def gotoPersistStorageConfigMode( self ):
      childMode = self.childMode( ContainerPersistStorageConfigMode )
      self.session_.gotoChildMode( childMode )

def addContainerConfig( mode, args ):
   containerName = args[ 'CONTAINER_NAME' ]
   childMode = mode.childMode( ContainerConfigMode,
                               containerName=containerName )
   mode.session_.gotoChildMode( childMode )
   containerConfigDir.newContainer( containerName )

def deleteContainerConfig( mode, args ):
   containerName = args[ 'CONTAINER_NAME' ]
   if containerName in containerConfigDir.container:
      # Remove the image from container. This will make container to stop and
      # deleted.
      setConfigParam( containerConfigDir.container[ containerName ],
                      'imageName', "" )
      del containerConfigDir.container[ containerName ]
   else:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Container %s does not exist" % containerName )

def deleteAllContainerConfigs( mode ):
   for containerName in containerConfigDir.container:
      deleteContainerConfig( mode,
            { 'CONTAINER_NAME': containerName } )

class ContainerProfileConfigMode( ContainerProfileMode, BasicCli.ConfigModeBase ):
   name = "config-container-profile"

   def __init__( self, parent, session, profileName ):
      self.profileName_ = profileName
      self.session_ = session
      self.cancelChanges = False

      # A local profile to store the capture commands issued in this mode instance
      self.tempProfile = Tac.newInstance( "ContainerMgr::ContainerProfileConfig",
                                          self.profileName_ )

      if self.profileName_ in profileConfigDir.profile:
         # Copy the configuration of original profile in temporary local profile
         copyProfile( self.tempProfile,
                      profileConfigDir.profile[ profileName ] )

      ContainerProfileMode.__init__( self, profileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitProfileConfig()
      BasicCli.ConfigModeBase.onExit( self )

   def commitProfileConfig( self ):
      if self.cancelChanges:
         return

      if self.profileName_ not in profileConfigDir.profile:
         profileConfig = profileConfigDir.newProfile( self.profileName_ )
         copyProfile( profileConfig, self.tempProfile )
         profileModified = True
      else:
         profileConfig = profileConfigDir.profile[ self.profileName_ ]
         profileModified = copyProfile( profileConfig, self.tempProfile )

      if profileModified:
         profileConfig.done += 1

   def getConfig( self ):
      return self.tempProfile

   def gotoStartupConfigMode( self ):
      childMode = self.childMode( ContainerProfileStartupConfigMode )
      self.session_.gotoChildMode( childMode )

   def gotoPersistStorageConfigMode( self ):
      childMode = self.childMode( ContainerProfilePersistStorageConfigMode )
      self.session_.gotoChildMode( childMode )

def copyProfile( dest, src ):
   assert dest is not None
   assert src is not None

   profileModified = False

   if dest.params != src.params:
      dest.params = src.params
      profileModified = True

   for ctrPath in dest.bindMount:
      if ctrPath not in src.bindMount:
         del dest.bindMount[ ctrPath ]
         profileModified = True
      elif dest.bindMount[ ctrPath ] != src.bindMount[ ctrPath ]:
         profileModified = True

   # Check if src.bindMount has anything not present in dest.bindMount.
   # The opposite case is covered in previous check.
   if len( dest.bindMount ) != len( src.bindMount ):
      profileModified = True

   for ctrPath in src.bindMount:
      hostPath = src.bindMount[ ctrPath ].hostPath
      dest.addBindMount( BindMountEntry( hostPath, ctrPath ) )

   return profileModified

def gotoContainerProfileConfigMode( mode, args ):
   profileName = args.get( 'PROFILE_NAME' )
   assert profileName
   childMode = mode.childMode( ContainerProfileConfigMode, profileName=profileName )
   mode.session_.gotoChildMode( childMode )

def deleteContainerProfile( mode, args ):
   profileName = args.get( 'PROFILE_NAME' )
   assert profileName
   del profileConfigDir.profile[ profileName ]

def deleteAllContainerProfiles( mode ):
   for profile in profileConfigDir.profile:
      deleteContainerProfile( mode, { 'PROFILE_NAME': profile } )

# --------------------------------------------------------------------------------
# [ no | default ] image IMAGE_NAME
# --------------------------------------------------------------------------------
class ImageImagenameCmd( CliCommand.CliCommandClass ):
   syntax = 'image IMAGE_NAME'
   noOrDefaultSyntax = 'image ...'
   data = {
      'image': 'Image name of the container',
      'IMAGE_NAME': imageNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.getConfig()
      imageName = args[ 'IMAGE_NAME' ]
      setConfigParam( config,
                      'imageName', imageName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'imageName', "" )

ContainerConfigMode.addCommandClass( ImageImagenameCmd )
ContainerProfileConfigMode.addCommandClass( ImageImagenameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] shutdown
# --------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown the container',
   }

   @staticmethod
   def handler( mode, args ):
      containerName = mode.containerName_
      containerConfigDir.container[ containerName ].enabledState = \
            EnabledState()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      containerConfigDir.container[ containerName ].enabledState = \
            EnabledState( True, 'on-boot' in args )

ContainerConfigMode.addCommandClass( ShutdownCmd )

# --------------------------------------------------------------------------------
# [ no | default ] on-boot
# --------------------------------------------------------------------------------
class OnBootCmd( CliCommand.CliCommandClass ):
   syntax = 'on-boot'
   noOrDefaultSyntax = syntax
   data = {
      'on-boot': 'Enable the container',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      ShutdownCmd.noOrDefaultHandler( mode, args )
      mode.addWarning( "The 'on-boot' command has been deprecated. "
                       "Please use 'no shutdown' instead." )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ShutdownCmd.handler( mode, args )
      mode.addWarning( "The 'no on-boot' command has been deprecated. "
                       "Please use 'shutdown' instead." )

ContainerConfigMode.addCommandClass( OnBootCmd )

# --------------------------------------------------------------------------------
# [ no | default ] cpu cores CPU_CORES
# --------------------------------------------------------------------------------
matcherCpu = CliMatcher.KeywordMatcher( 'cpu',
      helpdesc='CPU configuration for the container' )

class CpuCoresStringCmd( CliCommand.CliCommandClass ):
   syntax = 'cpu cores CPU_CORES'
   noOrDefaultSyntax = 'cpu cores ...'
   data = {
      'cpu': matcherCpu,
      'cores': 'Configure the CPU cores for the container',
      'CPU_CORES': CliMatcher.StringMatcher( helpdesc='Cpu Cores of the container',
         helpname='CPU_CORES' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'cpuCores', CpuCores( args[ 'CPU_CORES' ] ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'cpuCores', CpuCores() )

ContainerConfigMode.addCommandClass( CpuCoresStringCmd )
ContainerProfileConfigMode.addCommandClass( CpuCoresStringCmd )

# --------------------------------------------------------------------------------
# [ no | default ] cpu shares CPU_SHARES
# --------------------------------------------------------------------------------
class CpuSharesCpusharesCmd( CliCommand.CliCommandClass ):
   syntax = 'cpu shares CPU_SHARES'
   noOrDefaultSyntax = 'cpu shares ...'
   data = {
      'cpu': matcherCpu,
      'shares': 'Configure the relative CPU share for the container',
      'CPU_SHARES': CliMatcher.IntegerMatcher( 1, 2**32 - 1,
         helpdesc='CPU relative shares for the container' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'cpuShares', CpuShares( args[ 'CPU_SHARES' ] ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'cpuShares', CpuShares() )

ContainerConfigMode.addCommandClass( CpuSharesCpusharesCmd )
ContainerProfileConfigMode.addCommandClass( CpuSharesCpusharesCmd )

# --------------------------------------------------------------------------------
# [ no | default ] memory MEMORY
# --------------------------------------------------------------------------------
def memoryHardLimitHandler( mode, args ):
   memoryHardLimit = args[ 'MEMORY' ]
   if isMemoryLimitInvalid( memoryHardLimit, 9 * ( 1024**2 ) ):
      mode.addError( "Minimum memory hard-limit allowed is 9MB" )
      return
   config = mode.getConfig()
   setConfigParam( config,
                   'memoryHardLimit',
                   MemoryHardLimit( memoryHardLimit ) )

def memoryHardLimitNoOrDefaultHandler( mode, args ):
   config = mode.getConfig()
   setConfigParam( config,
                   'memoryHardLimit',
                   MemoryHardLimit() )

def memorySoftLimitHandler( mode, args ):
   memorySoftLimit = args[ 'MEMORY' ]
   if isMemoryLimitInvalid( memorySoftLimit, 6 * ( 1024**2 ) ):
      mode.addError( "Minimum memory soft-limit allowed is 6MB" )
      return
   config = mode.getConfig()
   setConfigParam( config,
                   'memorySoftLimit',
                   MemorySoftLimit( memorySoftLimit ) )

def memorySoftLimitNoOrDefaultHandler( mode, args ):
   config = mode.getConfig()
   setConfigParam( config,
                   'memorySoftLimit',
                   MemorySoftLimit() )

memoryHardLimitHelpStr = (
      'Configure the maximum memory to be used by the container. '
      'Default unit is bytes. Use b for bytes, k for kilobytes, m for megabytes '
      'and g for gigabytes' )
memorySoftLimitHelpStr = (
      'Configure the memory soft-limit to be used by the container. '
      'Default unit is bytes. Use b for bytes, k for kilobytes, m for megabytes '
      'and g for gigabytes' )
memoryLimitMatcher = CliMatcher.PatternMatcher( pattern='^[0-9]+[bkmg]?$',
      helpdesc='Memory Limit', helpname='Memory' )
memoryKw = CliMatcher.KeywordMatcher( 'memory',
      helpdesc='Memory configuration for the container' )

class MemoryMemoryCmd( CliCommand.CliCommandClass ):
   syntax = 'memory MEMORY'
   noOrDefaultSyntax = 'memory ...'
   data = {
      'memory': memoryHardLimitHelpStr,
      'MEMORY': memoryLimitMatcher,
   }

   handler = memoryHardLimitHandler
   noOrDefaultHandler = memoryHardLimitNoOrDefaultHandler

ContainerConfigMode.addCommandClass( MemoryMemoryCmd )

# --------------------------------------------------------------------------------
# [ no | default ] memory hard-limit MEMORY
# --------------------------------------------------------------------------------
class MemoryHardLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'memory hard-limit MEMORY'
   noOrDefaultSyntax = 'memory hard-limit ...'
   data = {
      'memory': memoryKw,
      'hard-limit': memoryHardLimitHelpStr,
      'MEMORY': memoryLimitMatcher,
   }

   handler = memoryHardLimitHandler
   noOrDefaultHandler = memoryHardLimitNoOrDefaultHandler

ContainerProfileConfigMode.addCommandClass( MemoryHardLimitCmd )

# --------------------------------------------------------------------------------
# [ no | default ] memory soft-limit MEMORY
# --------------------------------------------------------------------------------
class MemorySoftLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'memory soft-limit MEMORY'
   noOrDefaultSyntax = 'memory soft-limit ...'
   data = {
      'memory': memoryKw,
      'soft-limit': memorySoftLimitHelpStr,
      'MEMORY': memoryLimitMatcher,
   }

   handler = memorySoftLimitHandler
   noOrDefaultHandler = memorySoftLimitNoOrDefaultHandler

ContainerProfileConfigMode.addCommandClass( MemorySoftLimitCmd )

# --------------------------------------------------------------------------------
# [ no | default ] command COMMAND
# --------------------------------------------------------------------------------
class CommandStringCmd( CliCommand.CliCommandClass ):
   syntax = 'command COMMAND'
   noOrDefaultSyntax = 'command ...'
   data = {
      'command': 'Configure command for the container',
      'COMMAND': CliMatcher.StringMatcher( helpdesc='Container Command Arguments',
         helpname='Container Command' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'command', args[ 'COMMAND' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'command', "" )

ContainerConfigMode.addCommandClass( CommandStringCmd )
ContainerProfileConfigMode.addCommandClass( CommandStringCmd )

# --------------------------------------------------------------------------------
# [ no | default ] options OPTIONS
# --------------------------------------------------------------------------------
class OptionsStringCmd( CliCommand.CliCommandClass ):
   syntax = 'options OPTIONS'
   noOrDefaultSyntax = 'options ...'
   data = {
      # FIX : helpdesc needs to be more formal
      'options': 'Configure options for container should run with',
      'OPTIONS': CliMatcher.StringMatcher( helpdesc='Container run options',
         helpname='Container_run_options' ),
   }

   @staticmethod
   def handler( mode, args ):
      options = args[ 'OPTIONS' ]
      if '--rm' in options:
         mode.addError( "--rm option is not supported" )
         return
      config = mode.getConfig()
      setConfigParam( config,
                      'options', options )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'options', "" )

ContainerConfigMode.addCommandClass( OptionsStringCmd )
ContainerProfileConfigMode.addCommandClass( OptionsStringCmd )

# --------------------------------------------------------------------------------
# [ no | default ] environment ENVIRON_CONFIG
# --------------------------------------------------------------------------------
class EnvironmentCmd( CliCommand.CliCommandClass ):
   syntax = 'environment ENVIRON_CONFIG'
   noOrDefaultSyntax = 'environment ...'
   data = {
      'environment': CliMatcher.KeywordMatcher( 'environment',
         helpdesc="Configure container's OS environment" ),
      'ENVIRON_CONFIG':
         CliMatcher.PatternMatcher(
            pattern=r'\w+=[^:=]+(:\w+=[^:=]+)*$',
            helpdesc='Container environment configuration in the form of '
                     'VAR1=VAL1:VAR2=VAL2:...',
            helpname='EXPR' ),
   }

   @staticmethod
   def handler( mode, args ):
      envCfg = args[ 'ENVIRON_CONFIG' ]
      config = mode.getConfig()
      setConfigParam( config,
                      'environment', envCfg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'environment', "" )

ContainerProfileConfigMode.addCommandClass( EnvironmentCmd )

# --------------------------------------------------------------------------------
# [ no | default ] security mode privileged
# --------------------------------------------------------------------------------
class SecurityModeCmd( CliCommand.CliCommandClass ):
   syntax = 'security mode privileged'
   noOrDefaultSyntax = 'security mode ...'
   data = {
      'security': "Configure security for containers",
      'mode': "Configure security mode",
      'privileged': "Privileged container",
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'secMode', SecurityMode.secModePrivileged )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'secMode', SecurityMode.secModeUnconfigured )

ContainerProfileConfigMode.addCommandClass( SecurityModeCmd )

# --------------------------------------------------------------------------------
# [ no | default ] networking mode host
# --------------------------------------------------------------------------------
class NetworkingModeCmd( CliCommand.CliCommandClass ):
   syntax = 'networking mode host'
   noOrDefaultSyntax = 'networking mode ...'
   data = {
      'networking': "Configure networking for containers",
      'mode': "Configure networking mode",
      'host': "Use host networking",
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'netMode', NetworkingMode.netModeHost )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'netMode', NetworkingMode.netModeUnconfigured )

ContainerProfileConfigMode.addCommandClass( NetworkingModeCmd )

# --------------------------------------------------------------------------------
# [ no | default ] logging driver syslog
# --------------------------------------------------------------------------------
class LoggingDriverCmd( CliCommand.CliCommandClass ):
   syntax = 'logging driver syslog'
   noOrDefaultSyntax = 'logging driver ...'
   data = {
      'logging': "Configure logging for containers",
      'driver': "Configure logging driver",
      'syslog': "syslog driver",
   }

   @staticmethod
   def handler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'loggingDriver', LoggingDriver.driverSyslog )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'loggingDriver', LoggingDriver.driverUnconfigured )

ContainerProfileConfigMode.addCommandClass( LoggingDriverCmd )

# --------------------------------------------------------------------------------
# [ no | default ] on-exit restart policy ( never | ( status all ) )
# --------------------------------------------------------------------------------
class RestartPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'on-exit restart policy ( never | ( status all ) )'
   noOrDefaultSyntax = 'on-exit restart policy ...'
   data = {
      'on-exit': "Configure container-on exit parameters",
      'restart': "Configure restart parameters",
      'policy': "Configure restart policy",
      'never': "Don't restart container for any exit-status",
      'status': "Restart container for exit-status",
      'all': "All exit-statuses",
   }

   @staticmethod
   def handler( mode, args ):
      if 'never' in args:
         policy = RestartPolicy.restartPolicyNoRestart
      else:
         assert 'status' in args
         assert 'all' in args
         policy = RestartPolicy.restartPolicyAlways
      config = mode.getConfig()
      setConfigParam( config,
                      'restartPolicy', policy )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'restartPolicy', RestartPolicy.restartPolicyUnconfigured )

ContainerProfileConfigMode.addCommandClass( RestartPolicyCmd )

# --------------------------------------------------------------------------------
# [ no | default ] profile PROFILE_NAME
# --------------------------------------------------------------------------------
class ProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE_NAME'
   noOrDefaultSyntax = 'profile ...'
   data = {
      'profile': 'Configure a profile for the contaner',
      'PROFILE_NAME': profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      containerConfig = containerConfigDir.container[ mode.containerName_ ]
      containerConfig.profileName = args[ 'PROFILE_NAME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerConfig = containerConfigDir.container[ mode.containerName_ ]
      containerConfig.profileName = ""

ContainerConfigMode.addCommandClass( ProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] startup
# --------------------------------------------------------------------------------
class StartupCmd( CliCommand.CliCommandClass ):
   syntax = 'startup'
   noOrDefaultSyntax = syntax
   data = {
      'startup': 'Configure startup parameters',
   }

   @staticmethod
   def handler( mode, args ):
      mode.gotoStartupConfigMode()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      setConfigParam( config,
                      'startupConditionBash', "" )
      setConfigParam( config,
                      'startupConditionPollInterval',
                      StartupConditionPollInterval() )

if toggleContainerStartupConditionEnabled():
   ContainerConfigMode.addCommandClass( StartupCmd )
   ContainerProfileConfigMode.addCommandClass( StartupCmd )

# --------------------------------------------------------------------------------
# [ no | default ] trigger condition bash CMD
# --------------------------------------------------------------------------------
class RunConditionBashCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger condition bash [ CMD ]'
   noOrDefaultSyntax = 'trigger condition bash ...'
   data = {
      'trigger': 'Configure startup trigger parameters for container',
      'condition': 'Configure condition for startup',
      'bash': 'Define BASH command for evaluating startup condition',
      'CMD': CliMatcher.StringMatcher(
         helpdesc='BASH command. <cr> for multi-line',
         helpname='COMMAND' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'CMD' in args:
         bashCmd = args[ 'CMD' ]
      else:
         bashCmdWithModeIndentation = BasicCliUtil.getMultiLineInput(
            mode,
            cmd="trigger condition bash",
            prompt="Enter Multi-line bash script" )
         # We're 3 modes in with the command being the 4th nesting level
         # modes: global cmgr mode, container/profile mode, startup mode,
         # command: trigger condition bash
         bashCmd = BasicCliUtil.removeConfigIndentation( bashCmdWithModeIndentation,
                                                         nestingLevel=4 )

      setConfigParam( mode.getConfig(),
                      'startupConditionBash',
                      bashCmd )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setConfigParam( mode.getConfig(),
                      'startupConditionBash',
                      '' )

if toggleContainerStartupConditionEnabled():
   ContainerStartupConfigMode.addCommandClass( RunConditionBashCmd )
   ContainerProfileStartupConfigMode.addCommandClass( RunConditionBashCmd )

# --------------------------------------------------------------------------------
# [ no | default ] trigger condition poll interval INTERVAL
# --------------------------------------------------------------------------------
class RunConditionPollIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'trigger condition poll interval INTERVAL'
   noOrDefaultSyntax = 'trigger condition poll interval ...'
   data = {
      'trigger': 'Configure startup trigger parameters for container',
      'condition': 'Configure condition for startup',
      'poll': 'Configure polling parameters for startup condition',
      'interval': 'Configure polling interval for startup condition',
      'INTERVAL': CliMatcher.IntegerMatcher(
         StartupConditionPollInterval.min, StartupConditionPollInterval.max,
         helpdesc='Poll interval in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      setConfigParam( mode.getConfig(),
                      'startupConditionPollInterval',
                      StartupConditionPollInterval( args[ 'INTERVAL' ] ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setConfigParam( mode.getConfig(),
                      'startupConditionPollInterval',
                      StartupConditionPollInterval() )

if toggleContainerStartupConditionEnabled():
   ContainerStartupConfigMode.addCommandClass( RunConditionPollIntervalCmd )
   ContainerProfileStartupConfigMode.addCommandClass( RunConditionPollIntervalCmd )

# --------------------------------------------------------------------------------
# [ no | default ] persist storage
# --------------------------------------------------------------------------------
class PersistStorageCmd( CliCommand.CliCommandClass ):
   syntax = 'persist storage'
   noOrDefaultSyntax = syntax
   data = {
      'persist': 'Configure persistent storage',
      'storage': 'Configure persistent storage',
   }

   @staticmethod
   def handler( mode, args ):
      mode.gotoPersistStorageConfigMode()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = mode.getConfig()
      config.bindMount.clear()

ContainerConfigMode.addCommandClass( PersistStorageCmd )
ContainerProfileConfigMode.addCommandClass( PersistStorageCmd )

# --------------------------------------------------------------------------------
# abort
# --------------------------------------------------------------------------------
class AbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit without saving the configuations made in the current session',
   }

   @staticmethod
   def handler( mode, args ):
      mode.cancelChanges = True
      mode.session_.gotoParentMode()

ContainerProfileConfigMode.addCommandClass( AbortCmd )

# --------------------------------------------------------------------------------
# [ no | default ] mount src <HOSTPATH> dst <CTRPATH>
# --------------------------------------------------------------------------------
hostPathMatcher = Url.UrlMatcher(
      fsFunc=lambda fs: fs.scheme in [ 'flash:', 'file:' ],
      helpdesc="Host path URL in flash: or file: format" )
class MountSrcDstCmd( CliCommand.CliCommandClass ):
   syntax = 'mount src HOSTPATH dst CTRPATH'
   noOrDefaultSyntax = syntax
   data = {
      'mount': 'Bind mount',
      'src': 'Host path',
      'HOSTPATH': hostPathMatcher,
      'dst': 'Container path',
      'CTRPATH': CliMatcher.PatternMatcher( pattern=pathRe,
         helpdesc='path', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      hostPath = args[ 'HOSTPATH' ].url
      ctrPath = args[ 'CTRPATH' ]
      config = mode.getConfig()
      config.addBindMount( BindMountEntry( hostPath, ctrPath ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ctrPath = args[ 'CTRPATH' ]
      hostPath = args[ 'HOSTPATH' ].url
      config = mode.getConfig()
      entry = config.bindMount.get( ctrPath )
      if entry and entry.hostPath == hostPath:
         del config.bindMount[ ctrPath ]

ContainerPersistStorageConfigMode.addCommandClass( MountSrcDstCmd )
ContainerProfilePersistStorageConfigMode.addCommandClass( MountSrcDstCmd )

def Plugin( entityManager ):
   global containerConfigDir, profileConfigDir
   containerConfigDir = ConfigMount.mount( entityManager,
                                           'containerMgr/container/config',
                                           'ContainerMgr::ContainerConfigDir', 'w' )
   profileConfigDir = ConfigMount.mount(
         entityManager,
         'containerMgr/container/profileConfig',
         'ContainerMgr::ContainerProfileConfigDir', 'w' )
