#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import math
import Tac
import sys
import os
from time import time
import json
import dateutil.parser as dp
from datetime import timedelta, datetime
from http import HTTPStatus

containerNameRe = r'[a-zA-Z0-9][a-zA-Z0-9_.\-]+$'
profileNameRe = r'[a-zA-Z0-9][a-zA-Z0-9_.\-]+$'
imageNameRe = r'[A-Za-z0-9_\-\/\.]+(:[A-Za-z0-9_\-\.\/:]+)?$'
pathRe = r'[A-Za-z0-9_\-@:\.\/]+'
unixSocketOptions = [ '--unix-socket', '/var/run/docker.sock' ]
dockerRemoteAPICmd = 'localhost/'
authConfigFile = '/root/.docker/config.json'

'''
rev1:
containerState = ( 'created', 'dead', 'exited',
                   'paused', 'removing', 'restarting',
                   'running' )
'''
# rev2:
# new in rev2: incomplete, shutdown, unknown, waiting, failed
containerState = ( 'created', 'dead', 'exited',
                   'failed', 'incomplete',
                   'paused', 'removing',
                   'restarting', 'running',
                   'shutdown', 'unknown', 'waiting' )

def isContainerMgrDaemonRunning():
   dockerPid = '/var/run/docker.pid'
   dockerSock = '/var/run/docker.sock'
   if os.access( dockerPid, os.F_OK ) and os.access( dockerSock, os.F_OK ):
      return True
   return False

def parseHttpResponse( response ):
   '''
   parse HTTP response and return content
   if no error otherwise return empty string
   '''
   responseLines = response.splitlines()
   statusLine = responseLines[ 0 ]
   assert statusLine.startswith( "HTTP" )
   statusCode = int( statusLine.split()[ 1 ] )
   if statusCode == HTTPStatus.OK:
      content = responseLines[ -1 ]
   else:
      content = ''
   return content

def dockerAPI( apiEndpoint ):
   '''
   Query docker API and return result.
   Return empty string on any error in querying or
   if docker is down.

   As on Jan 30, 2024 on trunk
   % docker version -f json | jq '.Server | {Version,ApiVersion}'
   {
       "Version": "20.10.14",
       "ApiVersion": "1.41"
   }

   Docker API is backwards compatible
   Reference: https://docs.docker.com/engine/api/v1.44/
   The API uses standard HTTP status codes to indicate the success
   '''
   if not isContainerMgrDaemonRunning():
      return ''

   apiEndpointUrl = f'localHost/{apiEndpoint}'
   curlCmd = [ 'curl',
               '--silent',
               '-S',
               '--include',
               '--unix-socket', '/var/run/docker.sock',
               apiEndpointUrl ]
   response = Tac.run( curlCmd,
                       stdout=Tac.CAPTURE, stderr=sys.stderr,
                       asRoot=True,
                       ignoreReturnCode=True )

   return parseHttpResponse( response )

def isEnoughSpacePresent( path ):
   expectedFreeSpacePercent = 10
   sizeStat = os.statvfs( path )
   freeSpace = sizeStat.f_bavail * sizeStat.f_frsize
   totalSpace = sizeStat.f_blocks * sizeStat.f_frsize
   freeSpacePercent = ( freeSpace / float( totalSpace ) ) * 100
   if freeSpacePercent > expectedFreeSpacePercent:
      thresholdSpace = ( 1 - ( expectedFreeSpacePercent / 100.0 ) ) * totalSpace
      MaxImageSize = freeSpace - thresholdSpace
      return True, MaxImageSize
   return False, 0

def imagePresent( imageName ):
   cmd = [ 'curl', '-sS' ] + unixSocketOptions + \
         [ dockerRemoteAPICmd + 'images/' + imageName + '/json' ]
   output = ""
   try:
      output = Tac.run( cmd, stdout=Tac.CAPTURE,
                        stderr=sys.stderr, asRoot=True )
   except Tac.SystemCommandError:
      return False

   if output == 'No such image: %s\n' % imageName:
      return False
   return True

def imageSize( image ):
   cmd = [ 'curl', '-sS' ] + unixSocketOptions + \
         [ dockerRemoteAPICmd + 'images/' + image + '/json' ]
   output = ""
   try:
      output = Tac.run( cmd, stdout=Tac.CAPTURE,
                        stderr=sys.stderr, asRoot=True )
   except Tac.SystemCommandError as e:
      print( 'curl failed due to %s' % e.output )
   output = json.loads( output )
   return int ( output[ 'Size' ] // 8 )

def iso8601FormatToUnixTimestamp( timeInStr ):
   '''
   This returns the corresponding unix timestamp for an input
   IS08601 representation like this: 
   "2024-01-31T08:23:12.173289014Z"

   Note the 'Z' at the end. So it is already UTC, we can just parse
   into an aware datetime object.
   Since this is UTC, we don't need any other xform before populating
   CAPI model.
   '''
   assert isinstance( timeInStr, str )
   parsedDt = dp.isoparse( timeInStr )
   # CAPI model wants int, not float
   timestamp = int( datetime.timestamp( parsedDt ) )
   return timestamp

def unixTimestampToStr( timestamp ): # pylint: disable=inconsistent-return-statements
   currentTime = int( time() )
   diffTime = currentTime - timestamp
   sec = timedelta( seconds=diffTime )
   duration = datetime( 1, 1, 1 ) + sec
   if ( duration.day - 1 ) > 0:
      return '%s days ago' % ( duration.day - 1 )
   if duration.hour > 0:
      return '%s hours ago' % duration.hour
   if duration.minute > 0:
      return '%s minutes ago' % duration.minute
   if duration.second > 0:
      return '%s seconds ago' % duration.second

# Docker Remote API returns image/container size in bytes.
# This function will take bytes as input and convert it into a more
# readable string
def bytesInStr( size ):
   if ( size == 0 ): # pylint: disable=superfluous-parens
      return '0 B'
   sizeUnits = ( "B", "KB", "MB", "GB" )
   unitIndex = int ( math.floor( math.log( size, 1024 ) ) )
   unitSizeInKb = math.pow( 1024, unitIndex )
   size = round( size / unitSizeInKb, 2 )
   return f'{size} {sizeUnits[unitIndex]}'

def portsToString( ports ):
   listStr = []
   for port in ports:
      hostIp = port[ 'ip' ]
      hostPort = port[ 'publicPort' ]
      privatePort = port[ 'privatePort' ]
      portType = port[ 'portType' ]

      if hostIp is not None and hostPort is not None:
         hostStr = f'{hostIp}:{hostPort}->'
      else:
         hostStr = ''
      listStr.append( f'{hostStr}{privatePort}/{portType}' )
   return ', '.join( listStr )
