#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import ConfigMount
from CliMode.ContainerMgrMode import ContainerMgrMode
from CliMode.ContainerImagesMode import ImagesMode
from CliPlugin.ContainerMgrCliLib import pathRe
from CliPlugin.ContainerConfigCli import (
      deleteAllContainerConfigs, addContainerConfig, deleteContainerConfig,
      deleteAllContainerProfiles, gotoContainerProfileConfigMode,
      deleteContainerProfile, containerNameMatcher, profileNameMatcher )
from CliPlugin.ContainerRegistryConfigCli import (
      addRegistryConfig, deleteAllRegistryConfigs, deleteRegistryConfig,
      registryNameMatcher )
import Tac
import Url
# pkgdeps: import UrlPlugin.FlashUrl

containerMgrConfig = None
imageLoadConfigDir = None

class ContainerMgrConfigMode( ContainerMgrMode, BasicCli.ConfigModeBase ):
   name = "Container Manager Configuration"

   def __init__( self, parent, session ):
      self.session_ = session
      ContainerMgrMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ImagesConfigMode( ImagesMode, BasicCli.ConfigModeBase ):
   name = "Images configuration"

   def __init__( self, parent, session ):
      self.session_ = session
      ImagesMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# [ no | default ] container-manager
# --------------------------------------------------------------------------------
class ContainerManagerCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager'
   noOrDefaultSyntax = syntax
   data = {
      'container-manager': 'Configure Container Manager',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ContainerMgrConfigMode )
      containerMgrConfig.agentEnabled = True
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Delete all the container configs
      deleteAllContainerConfigs( mode )

      deleteAllRegistryConfigs()

      # Delete all container profiles
      deleteAllContainerProfiles( mode )

      # Delete all images
      imageLoadConfigDir.imagePathUrl.clear()

      # Delete all the daemon Arguments
      DaemonArgsStringCmd.noOrDefaultHandler( mode, {} )

      # Delete persistent path
      PersistentPathPersistpathCmd.noOrDefaultHandler( mode, {} )

      containerMgrConfig.agentEnabled = False

BasicCli.GlobalConfigMode.addCommandClass( ContainerManagerCmd )

# --------------------------------------------------------------------------------
# [ no | default ] registry REGISTRY_NAME
# --------------------------------------------------------------------------------
class RegistryRegistrynameCmd( CliCommand.CliCommandClass ):
   syntax = 'registry REGISTRY_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'registry': 'Configure Container Manager registry',
      'REGISTRY_NAME': registryNameMatcher,
   }

   handler = addRegistryConfig
   noOrDefaultHandler = deleteRegistryConfig

ContainerMgrConfigMode.addCommandClass( RegistryRegistrynameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] images
# --------------------------------------------------------------------------------
class ImagesCmd( CliCommand.CliCommandClass ):
   syntax = 'images'
   noOrDefaultSyntax = syntax
   data = {
      'images': 'Configure loading container images',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ImagesConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      imageLoadConfigDir.imagePathUrl.clear()

ContainerMgrConfigMode.addCommandClass( ImagesCmd )

# --------------------------------------------------------------------------------
# [ no | default ] load TARBALL_PATH
# --------------------------------------------------------------------------------
tarballPathMatcher = Url.UrlMatcher(
      fsFunc=lambda fs: fs.scheme in [ 'flash:', 'file:' ],
      helpdesc="Image tarball path URL in flash: or file: format" )

class LoadFromTarballCmd( CliCommand.CliCommandClass ):
   syntax = 'load TARBALL_PATH'
   noOrDefaultSyntax = syntax
   data = {
      'load': 'Load container image from tarball',
      'TARBALL_PATH': tarballPathMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      tarballPath = args[ 'TARBALL_PATH' ].url
      imageLoadConfigDir.imagePathUrl.add( tarballPath )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tarballPath = args[ 'TARBALL_PATH' ].url
      imageLoadConfigDir.imagePathUrl.remove( tarballPath )

ImagesConfigMode.addCommandClass( LoadFromTarballCmd )

# --------------------------------------------------------------------------------
# --------------------------------------------------------------------------------
# [ no | default ] container CONTAINER_NAME
# --------------------------------------------------------------------------------
class ContainerContainernameCmd( CliCommand.CliCommandClass ):
   syntax = 'container CONTAINER_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'container': 'Configure Container Manager container',
      'CONTAINER_NAME': containerNameMatcher,
   }

   handler = addContainerConfig
   noOrDefaultHandler = deleteContainerConfig

ContainerMgrConfigMode.addCommandClass( ContainerContainernameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] container-profile PROFILE_NAME
# --------------------------------------------------------------------------------
class ContainerProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'container-profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'container-profile': 'Configure container profile',
      'PROFILE_NAME': profileNameMatcher,
   }

   handler = gotoContainerProfileConfigMode
   noOrDefaultHandler = deleteContainerProfile

ContainerMgrConfigMode.addCommandClass( ContainerProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] daemon args ARGS
# --------------------------------------------------------------------------------
class DaemonArgsStringCmd( CliCommand.CliCommandClass ):
   syntax = 'daemon args ARGS'
   noOrDefaultSyntax = 'daemon args ...'
   data = {
      'daemon': 'Provide Container Manager Daemon Arguments',
      'args': 'Provide Container Manager Daemon Arguments',
      'ARGS': CliMatcher.StringMatcher(
         helpdesc='Container Manager Daemon Arguments',
         helpname='ARUGMENTS' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      containerMgrConfig.containerMgrArgs = args[ 'ARGS' ]
      mode.addWarning( "This command is deprecated" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerMgrConfig.containerMgrArgs = ""

ContainerMgrConfigMode.addCommandClass( DaemonArgsStringCmd )

# --------------------------------------------------------------------------------
# [ no | default ] persistent-path PERSIST_PATH
# --------------------------------------------------------------------------------
class PersistentPathPersistpathCmd( CliCommand.CliCommandClass ):
   syntax = 'persistent-path PERSIST_PATH'
   noOrDefaultSyntax = 'persistent-path ...'
   data = {
      'persistent-path': 'Persistent path for storing images',
      'PERSIST_PATH': CliMatcher.PatternMatcher( pattern=pathRe,
         helpdesc='path', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      containerMgrConfig.persistentPath = args[ 'PERSIST_PATH' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerMgrConfig.persistentPath = containerMgrConfig.defaultPersistentPath

ContainerMgrConfigMode.addCommandClass( PersistentPathPersistpathCmd )

def Plugin( entityManager ):
   global containerMgrConfig, imageLoadConfigDir
   containerMgrConfig = ConfigMount.mount( entityManager, 'containerMgr/config',
                                           'ContainerMgr::ContainerMgrConfig', 'w' )
   imageLoadConfigDir = ConfigMount.mount( entityManager,
                                           'containerMgr/image/config',
                                           'ContainerMgr::ImageLoadConfigDir', 'w' )
