#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Model, Str, Dict, Enum, Int, Bool, List
from ArnetModel import IpGenericAddress
from CliPlugin.ContainerMgrCliLib import bytesInStr, portsToString
from CliPlugin.ContainerMgrCliLib import unixTimestampToStr, containerState
import re
import TableOutput

class ContainerMgrImage( Model ):
   imageId = Str( help='Image id in sha256 format' )
   timeOfCreation = Int( help='Time of image creation' )
   imageSize = Int( help='Size of the image' )

class ContainerMgrPort( Model ):
   # rev2:
   #   made optional: ip, publicPort
   #                  because we can have just container port without
   #                  publishing to host.
   __revision__ = 2
   ip = IpGenericAddress( help='IP address of the port', optional=True )
   privatePort = Int( help='Private port number' )
   publicPort = Int( help='Public port number', optional=True )
   portType = Enum( values=( "tcp", "udp" ), help='Port type' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if dictRepr.get( 'ip' ) is None:
            dictRepr[ 'ip' ] = ''

         if dictRepr.get( 'publicPort' ) is None:
            dictRepr[ 'publicPort' ] = 0
      return dictRepr

class ContainerMgrContainer( Model ):
   __revision__ = 2
   # rev2:
   #   added: timeOfStart, managed, startAttempts, startAttemptsAllowed
   #   removed: onBoot
   #   made optional: containerId, imageName, imageId, command,
   #                  timeOfCreation, ports
   #       * this is because we add a new subcommand to also display
   #       * non-running containers.
   #       * (ie configured/managed but failed/waiting/image-not-configured)
   containerId = Str( help='Container id', optional=True )
   managed = Bool( help='If managed/configured through EOS' )
   imageName = Str( help='Image name with tag', optional=True )
   imageId = Str( help='Image id in sha256 format', optional=True )
   command = Str( help='Command for the container', optional=True )
   timeOfCreation = Int( help='Timestamp of container creation', optional=True )
   timeOfStart = Int( help='Timestamp of container start', optional=True )
   state = Enum( values=containerState, help='State of the container' )
   ports = List( valueType=ContainerMgrPort,
                 help='List of all the ports for this container',
                 optional=True )
   profileName = Str( help="Profile name configured for container", optional=True )
   startAttempts = Int( help="Number of attempts made to start a failing container",
                        optional=True )
   startAttemptsAllowed = Int( help="Maximum number of attempts thats will be "
                                    "made to start a failing container "
                                    "before giving up",
                               optional=True )
   _brief = Bool( help='Briefed output required, private' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # onBoot is a non-optional attr in rev 1
         # as per new model, all "no shutdown" containers can be
         # considered onBoot/enabled.
         dictRepr[ 'onBoot' ] = ( dictRepr[ 'state' ] != "shutdown" )
         optionalFieldsAfterV1 = [
               ( 'containerId', '' ),
               ( 'imageName', '' ),
               ( 'imageId', '' ),
               ( 'command', '' ),
               ( 'timeOfCreation', 0 ),
               ( 'ports', '' ),
         ]
         for optionalField, defaultVaulue in optionalFieldsAfterV1:
            dictRepr[ optionalField ] = dictRepr.get( optionalField, defaultVaulue )

         # Degrade for ContainerMgrPort submodel
         if dictRepr.get( 'ports' ):
            portsRepr = dictRepr[ 'ports' ]
            degradedPortsRepr = []
            for i in range( len( dictRepr[ 'ports' ] ) ):
               portRepr = portsRepr[ i ]
               degradedPortRepr = self.ports[ i ].degrade( portRepr, revision )
               degradedPortsRepr.append( degradedPortRepr )
            dictRepr[ 'ports' ] = degradedPortsRepr
      return dictRepr

class ContainerMgrRegistry( Model ):
   serverName = Str( help='Registry server name' )
   username = Str( help='Username for registry' )
   insecure = Bool( help='Insecure registry' )
   status = Enum( values=( "Success", "Failed" ), help='Registry status' )

class ContainerMgrImages( Model ):
   containerMgrImages = Dict( valueType=ContainerMgrImage,
         help="Container Manager images keyed by imageName:tag" )
   def render( self ):
      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( self.containerMgrImages ):
         return

      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left", minWidth=10 )
         f2 = TableOutput.Format( justify="left", minWidth=10 )
         f3 = TableOutput.Format( justify="left", minWidth=12 )
         f4 = TableOutput.Format( justify="left", minWidth=13 )
         f5 = TableOutput.Format( justify="left", minWidth=7 )
         table.formatColumns( f1, f2, f3, f4, f5 )
         return table

      tableHeadings = ( "Name", "Tag", "Id", "Created", "Size" )
      table = createTable( tableHeadings )
      for image in self.containerMgrImages:
         imageInfo = self.containerMgrImages[ image ]
         table.newRow( image.split( ':' )[ 0 ], image.split( ':' )[ 1 ],
                       imageInfo.imageId[ 7 : 19 ],
                       unixTimestampToStr( imageInfo.timeOfCreation ),
                       bytesInStr( imageInfo.imageSize ) )
      print( table.output() )

class ContainerMgrContainers( Model ):
   __revision__ = 2
   containerMgrContainers = Dict( valueType=ContainerMgrContainer,
                                  help="Container Manager containers keyed "
                                       "by containerName" )

   def degrade( self, dictRepr, revision ):
      for containerName in dictRepr[ 'containerMgrContainers' ]:
         ctrDictRepr = dictRepr[ 'containerMgrContainers' ][ containerName ]
         ctrModel = self.containerMgrContainers[ containerName ]
         degradedCtrDictRepr = ctrModel.degrade( ctrDictRepr, revision )
         dictRepr[ 'containerMgrContainers' ][ containerName ] = degradedCtrDictRepr
      return dictRepr

   def render( self ):
      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( self.containerMgrContainers ):
         return

      def renderState( cm, brief ):
         if cm.state == "failed":
            if brief:
               renderedState = cm.state
            else:
               retryState = f"{cm.startAttempts}/{cm.startAttemptsAllowed}"
               if cm.startAttempts < cm.startAttemptsAllowed:
                  renderedState = f"{cm.state} (retrying {retryState})"
               else:
                  renderedState = f"{cm.state} (retries exhausted {retryState})"
         elif cm.state == "waiting":
            suffix = "" if brief else " for startup condition"
            renderedState = f"{cm.state}{suffix}"
         else:
            renderedState = cm.state
         return renderedState

      def createTable( tableHeadings ):
         numFields = len( tableHeadings )
         table = TableOutput.createTable( tableHeadings, numFields )
         fmt = TableOutput.Format( justify='left' )
         fmt.noPadLeftIs( True )
         fmt.padLimitIs( True )
         table.formatColumns( *[ fmt for i in range( numFields ) ] )
         return table

      firstKey = next( iter( self.containerMgrContainers ) )
      # pylint: disable=W0212
      brief = self.containerMgrContainers[ firstKey ]._brief
      if brief:
         tableHeadings = ( "Container", "Managed", "Profile",
                           "Image", "State", "Started", "Command" )
         table = createTable( tableHeadings )
         for containerName in sorted( self.containerMgrContainers ):
            containerModel = self.containerMgrContainers[ containerName ]
            if containerModel.timeOfStart:
               timestampRepr = unixTimestampToStr( containerModel.timeOfStart )
            else:
               timestampRepr = None
            nullStr = '-'
            table.newRow( containerName,
                          containerModel.managed,
                          containerModel.profileName or nullStr,
                          containerModel.imageName or nullStr,
                          renderState( containerModel, True ),
                          timestampRepr or nullStr,
                          containerModel.command or nullStr )
         print( table.output() )
      else:
         for containerName in sorted( self.containerMgrContainers ):
            containerModel = self.containerMgrContainers[ containerName ]
            print( 'Container Name: %s' % containerName )
            print( 'State: %s' % renderState( containerModel, False ) )
            print( f'Managed: {containerModel.managed}' )
            if containerModel.containerId:
               print( 'Container Id: %s' % containerModel.containerId )
            if containerModel.profileName:
               print( 'Profile Name: %s' % containerModel.profileName )
            if containerModel.imageName:
               print( 'Image Name: %s' % containerModel.imageName )
            if containerModel.imageId:
               print( 'Image Id: %s' % containerModel.imageId )
            if containerModel.command:
               print( 'Command: %s' % containerModel.command )
            if containerModel.timeOfCreation:
               print( 'Created: %s'
                      % unixTimestampToStr( containerModel.timeOfCreation ) )
            if containerModel.timeOfStart:
               print( 'Started: %s'
                      % unixTimestampToStr( containerModel.timeOfStart ) )
            if containerModel.ports:
               print( 'Ports: %s' % portsToString( containerModel.ports ) )
            print()

class ContainerMgrLogs( Model ):
   containerMgrLogs = Str( help='Log of a container', default='' )
   def render( self ):
      if not self.containerMgrLogs:
         return
      
      for line in str(self.containerMgrLogs).splitlines():
         print( line )

class ContainerMgrRegistries( Model ):
   containerMgrRegistries = Dict( valueType=ContainerMgrRegistry,
                                  help="Container Manager registry keyed "
                                       "by registryName" )
   def render( self ):
      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( self.containerMgrRegistries ):
         return

      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left", minWidth=12 )
         f2 = TableOutput.Format( justify="left", minWidth=20 )
         f3 = TableOutput.Format( justify="left", minWidth=12 )
         f4 = TableOutput.Format( justify="left", minWidth=6 )
         f5 = TableOutput.Format( justify="left", minWidth=10 )
         table.formatColumns( f1, f2, f3, f4, f5 )
         return table
      tableHeadings = ( "Registry Name", "Server", "Username",
                        "Secure", "Status")
      table = createTable( tableHeadings )
      for registry in self.containerMgrRegistries:
         reg = self.containerMgrRegistries[ registry ]
         table.newRow( registry, reg.serverName, reg.username,
                       not reg.insecure, reg.status )
      print( table.output() )

class ContainerMgrInfo( Model ):
   containerNum = Int( help='Total number of Containers', default=0 )
   runningContainerNum = Int( help='Total number of running Containers', default=0 )
   pausedContainerNum = Int( help='Total number of paused Containers', default=0 )
   stoppedContainerNum = Int( help='Total number of stopped Containers', default=0 )
   imagesNum = Int( help='Total number of images', default=0 )
   storageDriver = Str( help='storage Driver', default='' )
   backingFilesystem = Str( help='backing Filesystem', default='' )
   loggingDriver = Str( help='logging Driver', default='' )
   cgroupDriver = Str( help='cgroup Driver', default='' )
   volumeType = List( valueType=str, help='Volume types', default=[] )
   networkType = List( valueType=str, help='Network types', default=[] )
   containerMgrEngineId = Str( help='Container-manager engine ID', default='' )
   hostName = Str( help='hostname', default='' )
   containerMgrRootDir = Str( help='Container-manager root dir', default='' )
   cpuNum = Int( help='Total number of CPUs', default=0 )
   memory = Int( help='Total Memory', default=0 )

   def render( self ):
      # Either all the attributes will be set or none of them.
      # So, just checking for one of them.
      if not self.storageDriver:
         return

      print( 'Total Number of Containers: %s' % self.containerNum )
      print( 'Total Number of paused Containers: %s' % self.pausedContainerNum )
      print( 'Total Number of stopped Containers: %s' % self.stoppedContainerNum )
      print( 'Total Number of Images: %s' % self.imagesNum )
      print( 'Storage Driver: %s' % self.storageDriver )
      print( '   Backing Filesystem: %s' % self.backingFilesystem )
      print( 'Logging Driver: %s' % self.loggingDriver )
      print( 'Cgroup Driver: %s' % self.cgroupDriver )
      print( 'Plugins:' )
      print( '   Volume: %s' % ", ".join( self.volumeType ) )
      print( '   Network: %s' % ", ".join( self.networkType ) )
      print( 'ID: %s' % self.containerMgrEngineId )
      print( 'ContainerMgr Root Dir: %s' % self.containerMgrRootDir )
      print( 'CPUs: %s' % self.cpuNum )
      print( 'Total Memory: %s' % bytesInStr( self.memory ) )
      print()

class ContainerMgrBackupFiles( Model ):
   backupFiles = List( valueType=str, help='Backup files' )

class ContainerMgrBackup( Model ):
   backup = Dict( valueType=ContainerMgrBackupFiles,
                  help="Container manager backup keyed by persistent directory" )
   def render( self ):
      if not len( self.backup ): # pylint: disable=use-implicit-booleaness-not-len
         return
      def createTable( tableHeadings ):
         table = TableOutput.createTable( tableHeadings )
         f1 = TableOutput.Format( justify="left", minWidth=22 )
         f2 = TableOutput.Format( justify="left", minWidth=22 )
         table.formatColumns( f1, f2 )
         return table
      tableHeadings = ( "Files", "Directory" )
      table = createTable( tableHeadings )
      for d in self.backup:
         directory = self.backup[ d ]
         for f in directory.backupFiles:
            table.newRow( re.sub( r'\.tar$', '', f ), d )
      print( table.output() )
