#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import LazyMount
import sys
import os
import BasicCli
import json
import CliMatcher
import CliPlugin.TechSupportCli
from CliPlugin.ContainerConfigCli import (
      containerNameMatcher,
      imageNameMatcher )
from CliPlugin.ContainerMgrCliLib import (
      authConfigFile, dockerAPI,
      iso8601FormatToUnixTimestamp,
      isContainerMgrDaemonRunning )
from CliPlugin.ContainerMgrModels import (
      ContainerMgrImages, ContainerMgrImage,
      ContainerMgrContainers, ContainerMgrContainer, ContainerMgrRegistries,
      ContainerMgrRegistry, ContainerMgrInfo, ContainerMgrBackup,
      ContainerMgrBackupFiles, ContainerMgrPort, ContainerMgrLogs )
from CliPlugin.ContainerRegistryConfigCli import registryNameMatcher
from TypeFuture import TacLazyType
import ShowCommand
import Tracing

t1 = Tracing.trace1

ContainerRunState = TacLazyType( 'ContainerMgr::ContainerRunState' )
ISO8601Min = '0001-01-01T00:00:00Z'

containerMgrConfig = None
containerConfigDir = None
profileConfigDir = None
containerStatusDir = None
registryConfigDir = None

matcherContainerManager = CliMatcher.KeywordMatcher( 'container-manager',
      helpdesc='Display container-manager configuration' )

# --------------------------------------------------------------------------------
# show container-manager images [ IMAGE_NAME ]
# --------------------------------------------------------------------------------
class ContainerManagerImagesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager images [ IMAGE_NAME ]'
   data = {
      'container-manager': matcherContainerManager,
      'images': 'Display all images information',
      'IMAGE_NAME': imageNameMatcher,
   }
   cliModel = ContainerMgrImages

   @staticmethod
   def handler( mode, args ):
      imageName = args.get( 'IMAGE_NAME' )
      allImages = {}
      if not isContainerMgrDaemonRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrImages( containerMgrImages={} )

      if not imageName:
         apiEndpoint = "images/json"
      else:
         apiEndpoint = f"images/{imageName}/json"

      apiOutputStr = dockerAPI( apiEndpoint )
      if not apiOutputStr:
         return ContainerMgrImages( containerMgrImages={} )

      try:
         apiOutput = json.loads( apiOutputStr )
      except json.decoder.JSONDecodeError:
         return ContainerMgrImages( containerMgrImages={} )

      # if imageNameName is specified, we get a dict,
      # otherwise it is a list
      if isinstance( apiOutput, dict ):
         # Make this a one-element list for simpler code
         apiOutput = [ apiOutput ]
      else:
         assert isinstance( apiOutput, list )

      for image in apiOutput:
         model = ContainerMgrImage()
         model.imageId = image[ 'Id' ]

         imageCreatedFromDockerAPI = image[ 'Created' ]
         # listing all images displays this as an integer timestamp,
         # while listing just one shows it as a string in IS08601 format.
         if isinstance( imageCreatedFromDockerAPI, str ):
            model.timeOfCreation = iso8601FormatToUnixTimestamp(
                  imageCreatedFromDockerAPI )
         else:
            model.timeOfCreation = imageCreatedFromDockerAPI
         model.imageSize = image[ 'Size' ]
         if image[ 'RepoTags' ]:
            img = image[ 'RepoTags' ][ 0 ]
         elif image[ 'RepoDigests' ]:
            img = image[ 'RepoDigests' ][ 0 ].split( '@' )[ 0 ] + ':<none>'
         else:
            img = image[ 'Id' ] + ':<none>'
         allImages[ img ] = model
      return ContainerMgrImages( containerMgrImages=allImages )

BasicCli.addShowCommandClass( ContainerManagerImagesCmd )

# --------------------------------------------------------------------------------
# show container-manager log CONTAINER_NAME
# --------------------------------------------------------------------------------
class ContainerManagerLogContainernameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager log CONTAINER_NAME'
   data = {
      'container-manager': matcherContainerManager,
      'log': 'Display log of a container',
      'CONTAINER_NAME': containerNameMatcher,
   }
   cliModel = ContainerMgrLogs

   @staticmethod
   def handler( mode, args ):
      containerName = args[ 'CONTAINER_NAME' ]
      if not isContainerMgrDaemonRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrLogs()

      cmd = [ 'docker', 'logs', containerName ]
      output = Tac.run( cmd, stdout=Tac.CAPTURE,
                        stderr=sys.stderr, asRoot=True,
                        text=False )
      output = output.decode( encoding='ascii', errors='ignore' )
      model = ContainerMgrLogs()
      model.containerMgrLogs = output

      return model

BasicCli.addShowCommandClass( ContainerManagerLogContainernameCmd )

# --------------------------------------------------------------------------------
# show container-manager containers [ CONTAINER_NAME ] [ all ][ brief ]
# --------------------------------------------------------------------------------
def containerModelFromDockerAPIOutput( ctrAPIOutput ):
   '''
   Construct ContainerMgrContainer model from apiOutput

   ctrAPIOutput example
   (truncated for relevant fields and line length limit)
   {
     "Name": "/foo",
     "Id": "0e3d072...",
     "Created": "2024-01-31T06:44:11.428667134Z",
     "Path": "sleep",
     "Args": [
       "10000"
     ],
     "State": {
       "Status": "running",
       "Running": true,
       "Paused": false,
       "Restarting": false,
       "OOMKilled": false,
       "Dead": false,
       "Pid": 23831,
       "ExitCode": 0,
       "Error": "",
       "StartedAt": "2024-01-31T06:44:11.660185321Z",
       "FinishedAt": "0001-01-01T00:00:00Z"
     },
     "Image": "sha256:d3cd072",
     "Config": {
       . . .
       "Cmd": [
         "sleep",
         "10000"
       ],
       "Image": "docker.corp.arista.io/busybox",
       . . .
     },
     "NetworkSettings": {
      . . .
      "Ports": {
        "3000/tcp": [
          {
            "HostIp": "0.0.0.0",
            "HostPort": "2000"
          },
          {
            "HostIp": "::",
            "HostPort": "2000"
          }
        ],
        "9000/tcp": null
      }
      . . .
     },
     . . .
   }
   '''
   model = ContainerMgrContainer()
   model.containerId = ctrAPIOutput[ 'Id' ]
   model.imageName = ctrAPIOutput[ 'Config' ][ 'Image' ]
   model.command = ' '.join( ctrAPIOutput[ 'Config' ][ 'Cmd' ] )
   model.imageId = ctrAPIOutput[ 'Image' ]
   model.state = ctrAPIOutput[ 'State' ][ 'Status' ]
   containerPorts = ctrAPIOutput[ 'NetworkSettings' ][ 'Ports' ]
   ports = []
   if containerPorts:
      for port in containerPorts:
         privatePort = int( port.split( '/' )[ 0 ] )
         portType = port.split( '/' )[ 1 ]
         if containerPorts[ port ]:
            for hostPort in containerPorts[ port ]:
               p = ContainerMgrPort()
               p.privatePort = privatePort
               p.portType = portType
               p.ip = hostPort[ 'HostIp' ]
               p.publicPort = int( hostPort[ 'HostPort' ] )
               ports.append( p )
         else:
            p = ContainerMgrPort()
            p.privatePort = privatePort
            p.portType = portType
            ports.append( p )
   if ports:
      model.ports = ports

   iso8601CreatedTime = ctrAPIOutput[ 'Created' ]
   iso8601StartedTime = ctrAPIOutput[ 'State' ][ 'StartedAt' ]
   if iso8601CreatedTime != ISO8601Min:
      model.timeOfCreation = iso8601FormatToUnixTimestamp( iso8601CreatedTime )
   if iso8601StartedTime != ISO8601Min:
      model.timeOfStart = iso8601FormatToUnixTimestamp( iso8601StartedTime )

   return model

def unstartedContainerModelFromSysdb( profileName, isUndefinedProfile,
                                      imageName, enabled, containerStatus ):
   model = ContainerMgrContainer()
   # managed is set from caller
   if containerStatus:
      if containerStatus.state == ContainerRunState.ctrStateInactive:
         if isUndefinedProfile or not imageName:
            state = 'incomplete'
         elif not enabled:
            state = 'shutdown'
         else:
            state = 'unknown'
      elif containerStatus.state == ContainerRunState.ctrStatePollStartupCondition:
         state = 'waiting'
      elif containerStatus.state in [ ContainerRunState.ctrStateFailureRetry,
                                      ContainerRunState.ctrStateFailed ]:
         state = 'failed'
         model.startAttempts = containerStatus.failureRetryCounter
         model.startAttemptsAllowed = containerStatus.maxFailureRetryCount
      else:
         state = 'unknown'
   else:
      # No status (unexpected)
      state = 'unknown'

   model.state = state
   if imageName:
      model.imageName = imageName
   if profileName:
      model.profileName = profileName
   return model

def showContainers( mode, args ):
   brief = 'brief' in args
   containerName = args.get( 'CONTAINER_NAME' )
   listOnlyStartedContainers = 'all' not in args

   if not isContainerMgrDaemonRunning():
      mode.addWarning( "ContainerMgr daemon is not running" )
      return ContainerMgrContainers( containerMgrContainers={} )

   # To handle show container-manager containers brief
   # We aren't handling if container name is 'brief'.
   if containerName == 'brief':
      brief = True
      containerName = None

   # To handle show container-manager containers all
   # We aren't handling if container name is 'all'.
   if containerName == 'all':
      listOnlyStartedContainers = False
      containerName = None

   if containerName:
      listSpecificContainer = True
      apiContainersToWalk = [ containerName ]
   else:
      listSpecificContainer = False

      # Get a list of all created containers
      allContainersAPIEndpoint = 'containers/json?all=1'
      listAllContainersAPIOutputStr = dockerAPI( allContainersAPIEndpoint )
      if not listAllContainersAPIOutputStr:
         apiContainersToWalk = []
      else:
         listAllContainersAPIOutput = json.loads( listAllContainersAPIOutputStr )
         if listAllContainersAPIOutput:
            assert isinstance( listAllContainersAPIOutput, list )

            # [
            #    {
            #     "Id": "0e3d070ba...",
            #     "Names": [
            #       "/foo"
            #     ],
            #     . . .
            #    },
            #    {
            #     . . .
            #    },
            #    . . .
            # }
            apiContainersToWalk = [ ci[ "Names" ][ 0 ].lstrip( "/" )
                                    for ci in listAllContainersAPIOutput ]
         else:
            apiContainersToWalk = []

   # Fetch detailed apiOutput for all containers
   # First go over all containers in API list alloutput,
   # and fetch Detailed Information.
   # This covers all containers that have been started,
   # provided they haven't been removed.
   allContainers = {}
   for thisContainerName in apiContainersToWalk:
      apiEndpoint = f'containers/{thisContainerName}/json'
      apiOutputStr = dockerAPI( apiEndpoint )

      try:
         apiOutput = json.loads( apiOutputStr )
      except json.decoder.JSONDecodeError:
         # for specific container case, this could happen if the
         # container hasn't yet started.
         # for list all containers case, this can happen due to a transient
         # condition.
         continue

      containerModel = containerModelFromDockerAPIOutput( apiOutput )
      containerConfig = containerConfigDir.container.get( thisContainerName )
      containerModel.managed = ( containerConfig is not None )
      if containerConfig and containerConfig.profileName:
         containerModel.profileName = containerConfig.profileName
      containerModel._brief = brief # pylint: disable=protected-access

      allContainers[ thisContainerName ] = containerModel

   if not listOnlyStartedContainers:
      # Add to the output, any containers that have not yet started,
      # but have been configured
      # ie stuff that we haven't walked over in the previous loop
      if listSpecificContainer:
         if containerName in containerConfigDir.container:
            configuredContainers = set( [ containerName ] )
         else:
            configuredContainers = set()
      else:
         configuredContainers = set( containerConfigDir.container.keys() )
      containersAlreadyWalked = set( allContainers.keys() )
      configuredContainersToWalk = configuredContainers - containersAlreadyWalked
      for thisContainerName in configuredContainersToWalk:
         containerConfig = containerConfigDir.container[ thisContainerName ]
         containerStatus = containerStatusDir.container.get( thisContainerName )
         profileName = containerConfig.profileName

         # Derive imageName and isUndefinedProfile from configuration
         isUndefinedProfile = ( profileName and
                                ( profileName not in profileConfigDir.profile ) )
         if containerConfig.params.imageName:
            # inline config takes precedence over profile config
            imageName = containerConfig.params.imageName
         elif profileName and not isUndefinedProfile:
            profileConfig = profileConfigDir.profile[ profileName ]
            imageName = profileConfig.params.imageName
         else:
            # no inline image configured and no profile image configured
            imageName = ''
         enabled = containerConfig.enabled
         containerModel = unstartedContainerModelFromSysdb(
               profileName, isUndefinedProfile, imageName, enabled,
               containerStatus )
         containerModel.managed = True
         containerModel._brief = brief # pylint: disable=protected-access
         allContainers[ thisContainerName ] = containerModel

   return ContainerMgrContainers( containerMgrContainers=allContainers )

class ContainerManagerContainersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager containers [ CONTAINER_NAME ] [ all ] [ brief ]'
   data = {
      'container-manager': matcherContainerManager,
      'containers': 'Display all containers information',
      'CONTAINER_NAME': containerNameMatcher,
      'all': 'Include configured containers that are yet to be created',
      'brief': 'Display tabular output of summary information',
   }
   cliModel = ContainerMgrContainers

   handler = showContainers

BasicCli.addShowCommandClass( ContainerManagerContainersCmd )

# --------------------------------------------------------------------------------
# show container-manager info
# --------------------------------------------------------------------------------
class ContainerManagerInfoCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager info'
   data = {
      'container-manager': matcherContainerManager,
      'info': 'Show container-manager information',
   }
   cliModel = ContainerMgrInfo

   @staticmethod
   def handler( mode, args ):
      if not isContainerMgrDaemonRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrInfo()

      try:
         output = json.loads( dockerAPI( 'info' ) )
      except json.decoder.JSONDecodeError:
         mode.addWarning( "ContainerMgr info not available" )
         return ContainerMgrInfo()

      model = ContainerMgrInfo()
      model.containerNum = output[ 'Containers' ]
      model.runningContainerNum = output[ 'ContainersRunning' ]
      model.pausedContainerNum = output[ 'ContainersPaused' ]
      model.stoppedContainerNum = output[ 'ContainersStopped' ]
      model.imagesNum = output[ 'Images' ]
      model.storageDriver = output[ 'Driver' ]
      if output[ 'DriverStatus' ][ 0 ][ 0 ] == 'Backing Filesystem':
         model.backingFilesystem = output[ 'DriverStatus' ][ 0 ][ 1 ]
      model.loggingDriver = output[ 'LoggingDriver' ]
      model.cgroupDriver = output[ 'CgroupDriver' ]
      model.volumeType = output[ 'Plugins' ][ 'Volume' ]
      model.networkType = output[ 'Plugins' ][ 'Network' ]
      model.containerMgrEngineId = output[ 'ID' ]
      model.hostName = output[ 'Name' ]
      model.containerMgrRootDir = output[ 'DockerRootDir' ]
      model.cpuNum = output[ 'NCPU' ]
      model.memory = output[ 'MemTotal' ]
      return model

BasicCli.addShowCommandClass( ContainerManagerInfoCmd )

# --------------------------------------------------------------------------------
# show container-manager registry [ REGISTRY_NAME ]
# --------------------------------------------------------------------------------
class ContainerManagerRegistryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager registry [ REGISTRY_NAME ]'
   data = {
      'container-manager': matcherContainerManager,
      'registry': 'Display all the configured registries',
      'REGISTRY_NAME': registryNameMatcher,
   }
   cliModel = ContainerMgrRegistries

   @staticmethod
   def handler( mode, args ):
      registryName = args.get( 'REGISTRY_NAME' )
      allRegistries = {}
      if not isContainerMgrDaemonRunning():
         mode.addWarning( "ContainerMgr daemon is not running" )
         return ContainerMgrRegistries( containerMgrRegistries={} )

      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( registryConfigDir.registry ):
         return ContainerMgrRegistries( containerMgrRegistries={} )

      data = Tac.run( [ 'cat', authConfigFile ], stdout=Tac.CAPTURE,
                      stderr=Tac.CAPTURE, asRoot=True, ignoreReturnCode=True )
      if 'No such file or directory' in data:
         loggedRegisteries = []
      else:
         auth = json.loads( data )[ 'auths' ]
         loggedRegisteries = list( auth )

      goFormatStr = "'{{json .RegistryConfig.IndexConfigs}}'"
      dockerCmd = [ 'docker', 'info', '--format', goFormatStr ]
      try:
         registryJson = Tac.run( dockerCmd, stdout=Tac.CAPTURE,
                                 asRoot=True )
      except Tac.SystemCommandError:
         mode.addWarning( "docker info failed!" )
         return ContainerMgrRegistries( containerMgrRegistries={} )
      # Strip the quotes from the output
      registryJson = registryJson.strip( " '\n" )
      registryDict = json.loads( registryJson )
      insecureRegistries = [ r for r in registryDict
                             if not registryDict[ r ][ 'Secure' ] ]

      for registry in registryConfigDir.registry:
         reg = registryConfigDir.registry[ registry ]
         if not reg.params.serverName:
            continue
         if registryName and registryName != registry:
            continue
         model = ContainerMgrRegistry()
         model.serverName = reg.params.serverName
         model.username = reg.params.userName
         model.insecure = reg.params.insecure
         serverName = reg.params.serverName.split( '://' )[ -1 ]

         if reg.params.insecure:
            if serverName in insecureRegistries:
               model.status = "Success"
            else:
               model.status = "Failed"
         else:
            if serverName in loggedRegisteries:
               model.status = "Success"
            else:
               model.status = "Failed"
         allRegistries[ registry ] = model
      return ContainerMgrRegistries( containerMgrRegistries=allRegistries )

BasicCli.addShowCommandClass( ContainerManagerRegistryCmd )

# --------------------------------------------------------------------------------
# show container-manager backup
# --------------------------------------------------------------------------------
class ContainerManagerBackupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show container-manager backup'
   data = {
      'container-manager': matcherContainerManager,
      'backup': 'Display all the backup files for container-manager',
   }
   cliModel = ContainerMgrBackup

   @staticmethod
   def handler( mode, args ):
      backup = {}
      persistPaths = [ containerMgrConfig.defaultPersistentPath ]
      if containerMgrConfig.persistentPath != \
            containerMgrConfig.defaultPersistentPath:
         persistPaths += [ containerMgrConfig.persistentPath ]

      for path in persistPaths:
         persistPath = path + '.containermgr'
         if not os.path.exists( persistPath ):
            continue
         result = ContainerMgrBackupFiles()
         result.backupFiles = os.listdir( persistPath )
         backup[ path ] = result
      return ContainerMgrBackup( backup=backup )

BasicCli.addShowCommandClass( ContainerManagerBackupCmd )

# --------------------------------------------------------------------------------
# Register ContainerMgr Show CLIs into "show Tech-Support"
# --------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2016-11-13 00:00:40',
   cmds=[ 'show container-manager images',
          'show container-manager containers all',
          'show container-manager registry',
          'show container-manager info',
          'show container-manager log',
          'show container-manager backup' ],
   cmdsGuard=lambda: containerMgrConfig.agentEnabled )

def Plugin( entityManager ):
   global containerMgrConfig, \
         containerConfigDir, profileConfigDir, \
         containerStatusDir, registryConfigDir
   containerMgrConfig = LazyMount.mount( entityManager, 'containerMgr/config',
                                         'ContainerMgr::ContainerMgrConfig', 'r' )
   containerConfigDir = LazyMount.mount( entityManager,
                                         'containerMgr/container/config',
                                         'ContainerMgr::ContainerConfigDir', 'r' )
   profileConfigDir = LazyMount.mount(
         entityManager,
         'containerMgr/container/profileConfig',
         'ContainerMgr::ContainerProfileConfigDir', 'r' )
   containerStatusDir = LazyMount.mount( entityManager,
                                        'containerMgr/container/status',
                                        'ContainerMgr::ContainerStatusDir', 'r' )
   registryConfigDir = LazyMount.mount( entityManager,
                                       'containerMgr/registry/config',
                                       'ContainerMgr::RegistryConfigDir', 'r' )
