# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import CliMode.Cvx
import CliParser
import CliPlugin.TechSupportCli
from CliPlugin.ControllerdbLib import oobStatus
import CliPlugin.MacAddr as MacAddr # pylint: disable=consider-using-from-import
import ConfigMount
import LazyMount
import HostnameCli
import Plugins
import Tac
import Tracing
import Toggles.ControllerdbToggleLib as cvxToggle

partialToggle = cvxToggle.toggleCvxPartialServiceCompatibilityEnabled()

__defaultTraceHandle__ = Tracing.Handle( 'ControllerCli' )
t8 = Tracing.trace8

clusterConfigDir = None
fileRepStatusDir = None
controllerdbConfig = None
controllerdbStatus = None
controllerdbMgr = None
serviceStatus = None
serviceConfig = None
mgmtSecConfig = None
controllerMountStatus = None
controllerMountConfig = None
proxyControllerMountStatus = None
proxyControllerMountConfig = None
ipConfig = None
constants = Tac.Value( "Controller::Constants" )
heartbeatIntervalType = Tac.Type( "Controller::Heartbeat::Interval" )
heartbeatTimeoutType =  Tac.Type( "Controller::Heartbeat::Timeout" )
peerTimeoutType = Tac.Type( "ControllerCluster::PeerTimeout" )

# Services that use CVX must use this token to create commands that configure
# the service or enter a specialized config mode.
serviceKwMatcher = CliMatcher.KeywordMatcher( 'service',
                                              helpdesc='Configure a CVX service' )

# the following is used by 'show service' command (not sure why it's different from
# 'show cvx service' or 'show management cvx service').
serviceAfterShowKwMatcher = CliMatcher.KeywordMatcher( 'service',
                                                       helpdesc='Show CVX services' )

refreshServiceKwMatcher = CliMatcher.KeywordMatcher( 'service',
                                              helpdesc='Refresh a CVX service' )

showCvxKwMatcher = CliMatcher.KeywordMatcher( 'cvx',
      helpdesc='Details of CVX operation' )

clientMatcher = CliMatcher.KeywordMatcher( 'client',
      helpdesc='specify one client using switch ID or hostname' )

systemIdMatcher = MacAddr.macAddrMatcher
# Setting PRIO_LOW here for cases where this matcher is or-ed with systemIdMatcher
# above.
hostnameMatcher = HostnameCli.IpAddrOrHostnameMatcher( priority=CliParser.PRIO_LOW )

# Services can register a cleanup function to be executed everytime the command
# 'no cvx' is executed. That way, running 'no cvx' will disable all configurable
# services. Each registered function will be called with the cli mode as argument.
_noCvxHook = []
def addNoCvxCallback( fn ):
   assert callable( fn )
   _noCvxHook.append( fn )

def runNoCvxCallbacks( mode ):
   for fn in _noCvxHook:
      fn( mode )

# Services can also register a cleanup function to be executed everytime
# [ no|default ] shutdown is executed in the cvx mode.  Each registered function
# will be called with the mode and with a boolean enabled argument, which will be
# True if "no shutdown" or "default shutdown" was executed, and False if "shutdown"
# was executed.
_cvxShutdownHook = []
def addCvxShutdownCallback( fn ):
   assert callable( fn )
   _cvxShutdownHook.append( fn )

def runCvxShutdownCallbacks( mode, enabled ):
   for fn in _cvxShutdownHook:
      fn( mode, enabled )

def clusterConfig():
   # The default config is created in DefaultConfigPlugin. It should always exist
   return clusterConfigDir.config[ constants.clusterDefaultName ]

def oobConfig():
   return clusterConfig().oobConfig

def inactive():
   status = oobStatus()
   return status.inactive if status else None

class CvxConfigMode( CliMode.Cvx.CvxConfigMode, BasicCli.ConfigModeBase ):
   name = 'CVX Configuration'

   def __init__( self, parent, session, clusterName ):
      # pylint: disable-next=consider-using-f-string
      t8( "initialize cvx config mode for cluster %s" % clusterName )
      CliMode.Cvx.CvxConfigMode.__init__( self, clusterName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      clusterConfig().clusterName = clusterName
      controllerdbConfig.primaryClusterName = clusterName

#------------------------------------------------------------------------------
# show tech-support commands
#------------------------------------------------------------------------------
showTechCmdList = [
   'show cvx',
   'show cvx connections all',
   'show cvx service',
   'show cvx file-replication',
   'show cvx mounts',
]

if partialToggle:
   showTechCmdList.append( 'show cvx service peer' )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-01-13 11:54:20',
   cmds=showTechCmdList,
   cmdsGuard=lambda: controllerdbConfig and controllerdbConfig.enabled )

@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global serviceConfig
   global mgmtSecConfig
   global clusterConfigDir
   global fileRepStatusDir
   global controllerdbConfig
   global controllerdbStatus
   global controllerMountStatus
   global controllerMountConfig
   global proxyControllerMountStatus
   global proxyControllerMountConfig
   global ipConfig

   serviceConfig = ConfigMount.mount( entityManager, "controller/service/config",
                                      "Controller::ServiceConfigDir", "w" )
   mgmtSecConfig = ConfigMount.mount( entityManager, "mgmt/security/ssl/config",
                                      "Mgmt::Security::Ssl::Config", "w" )
   clusterConfigDir = ConfigMount.mount(
      entityManager, "controller/cluster/configDir",
      "ControllerCluster::ClusterConfigDir", "w" )
   fileRepStatusDir = LazyMount.mount( entityManager,
                                       "controller/cluster/fileReplication/status",
                                       "Tac::Dir", "ri" )

   controllerdbConfig = ConfigMount.mount( entityManager, "controller/config",
                                           "Controllerdb::Config", "w" ) 
   LazyMount.mount(
      entityManager, "controller/publish/status/service", "Tac::Dir", "ri" )
   controllerMountStatus = LazyMount.mount(
      entityManager, "controller/mount/status/service", "Tac::Dir", "ri" )
   controllerMountConfig = LazyMount.mount(
      entityManager, "controller/mount/config/service", "Tac::Dir", "ri" )
   proxyControllerMountStatus = LazyMount.mount(
      entityManager, "controller/mount/status/proxy/service", "Tac::Dir", "ri" )
   proxyControllerMountConfig = LazyMount.mount(
      entityManager, "controller/mount/config/proxy/service", "Tac::Dir", "ri" )

   controllerdbStatus = LazyMount.mount( entityManager, "controller/status",
                                         "Controllerdb::Status", "r" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
