# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import TableOutput
import Tac
import Ark
from CliModel import Model, Str, Enum, Int, List, Submodel, Dict, Bool, Float
from ArnetModel import Ip4Address
from ControllerModel import ServiceStatus, HeartbeatStatus, MountState
from IpLibConsts import DEFAULT_VRF
from collections import Counter

VersionCompatibilityState = Tac.Type( "Controller::VersionCompatibilityState" )
SslDisableReason = Tac.Type( "Controller::SslProfileStatus::DisableReason" )

def _createCvxTable( headings, **kwargs ):
   table = TableOutput.createTable( headings, **kwargs )
   f = TableOutput.Format( justify='left' )
   f.noPadLeftIs( True )
   f.padLimitIs( True )
   table.formatColumns( *( [ f ] * len( headings ) ) )
   return table

class MgmtCvxMounts( Model ):
   mountStates = List( valueType=MountState,
                          help='State of CVX published mounts' )
   
   def render( self ):
      hasSecondary = False
      headings = ( 'MountPath', 'MountType', 'MountState' )
      table = _createCvxTable( headings, indent=2, tableWidth=120 )
      for state in self.mountStates:
         mountPath = state.path
         mountType = state.type
         mountState = state.state
         if mountPath[-1] == '*':
            hasSecondary = True
         table.newRow( mountPath, mountType, mountState )
      print( table.output() )
      if hasSecondary:
         print( '* mount from secondary cvx' )

class SslProfileStatus( Model ):
   sslProfileName = Str( help="SSL profile name" )
   enabled = Bool( help="Whether SSL profile is enabled" )
   disableReason = Enum( help="Reason for disabled SSL profile" ,
                         values=SslDisableReason.attributes, optional=True )

   def disableReasonStr( self ):
      return { 'profileNotExist': 'SSL profile does not exist',
               'profileInvalid': 'Invalid SSL profile',
               'certNotConfigured': 'Certificate not configured in SSL profile',
               'trustedCertNotConfigured': 
               'Trusted certificates not configured in SSL profile',
               'dhparamsNotReady': 
               'Diffie-Hellman parameters not yet ready' } [ self.disableReason ]
   
   def render( self ):
      print( f'{" "*2}SSL profile: {self.sslProfileName}' )
      print( f'{" "*4}Status: {"Enabled" if self.enabled else "Disabled"}' )
      if not self.enabled: 
         print( f'{" "*4}Disable Reason: {self.disableReasonStr()}' ) 

class ControllerStatus( Model ):
   oobConnectionStatus = Enum( values=( 'connecting',
                                        'negotiating',
                                        'established',
                                        'disconnecting',
                                        'shutdown' ),
                               help='Out-of-band connection status' )
   leader = Bool( help="Leader of the CVX cluster", optional=True )
   lastLeaderChangeTs = Float( help="Last leader switchover UTC timestamp",
                               optional=True )
   connectionTime = Float( help="Connection establishment UTC timestamp",
                           optional=True )
   versionCompatibility = Enum( values=VersionCompatibilityState.attributes,
                             help=( "Compatibility of switch with controller" ) )
   negotiatedVersion = Int( help='Negotiated version' )
   heartbeatStatus = Submodel( valueType=HeartbeatStatus,
                               help='Heartbeat status of the controller' )
   hostname = Str( help='Controller hostname' )
   controllerUUID = Str( help='Controller UUID' )
   oobConnectionSecured = Bool( help="Whether out-of-band connection is SSL secured",
                                optional=True )
   ibConnectionSecured = Bool( help="Whether in-band connection is SSL secured",
                               optional=True )
   duplicatedConnection = Bool( help="Multiple host entries for the same server",
                                optional=True )

   def versionStr( self ):
      d = { 'unnegotiated': 'Not negotiated',
            'inNegotiation': 'In negotiation',
            'incompatible': 'Not compatible' }
      return d.get( self.versionCompatibility, str( self.negotiatedVersion ) )

   def connectionSecuredStr( self, secured ):
      retStr = "SSL secured" if secured else "Not secured"
      return retStr
   
   def display( self, indent=' ' ):
      if self.leader:
         print( f'{indent}Master since '
                f'{Ark.timestampToStr(self.lastLeaderChangeTs, now=Tac.utcNow() )}'
                )
      duplicatedConnectionStr = ( ' (duplicated connection)' if
                                  self.duplicatedConnection else '' )
      print( f'{indent}Connection status: {self.oobConnectionStatus}'
             f'{duplicatedConnectionStr}' )
      if self.connectionTime:
         ts = Ark.timestampToStr( self.connectionTime, now=Tac.utcNow() )
         print( f'{indent}Connection timestamp: {ts}' )

      if self.oobConnectionSecured is not None:
         print( f'{indent + " "*2}Out-of-band connection: '
                f'{self.connectionSecuredStr( self.oobConnectionSecured )}' )
         if self.ibConnectionSecured is not None:
            print( f'{indent+" "*2}In-band connection: '
                   f'{self.connectionSecuredStr( self.ibConnectionSecured )}' )
      print( f'{indent}Negotiated version: {self.versionStr()}' )
      print( f'{indent}Controller UUID: {self.controllerUUID}' )
      if self.heartbeatStatus:
         self.heartbeatStatus.display( indent=indent )

class MgmtCvxClusterStatus( Model ):
   enabled = Bool( help='CVX client state' )
   cvxClusterName = Str( help='Name of the CVX cluster' )
   controllerStatuses = Dict( help="Controller connection status", keyType=str,
                              valueType=ControllerStatus )
   heartbeatInterval = Float( 'Heartbeat interval in seconds' )
   heartbeatTimeout = Float( 'Heartbeat timeout in seconds' )
   sslProfileStatus = Submodel( help="SSL profile status", 
                                valueType=SslProfileStatus, optional=True )
   sourceIntf = Str( help='source-interface name' )
   sourceIntfAddr = Ip4Address( help="source-interface address", optional=True )
   vrfName = Str( help='VRF used to connect to CVX', optional=True ) 
   
   def statusStr( self ):
      return 'Enabled' if self.enabled else 'Disabled'

   def sourceIntfStr( self ):
      if not self.sourceIntf:
         return 'Inactive (Not configured)'
      if self.sourceIntfAddr is None:
         return ( f'Inactive ({self.sourceIntf} configured but it is not a layer '
                  '3 interface)' )
      if str( self.sourceIntfAddr ) == '0.0.0.0':
         return ( f'Inactive ({self.sourceIntf} configured but it is not assigned '
                  'an IP address)' )
      return f'{self.sourceIntf} ({self.sourceIntfAddr})'

   def render( self ):
      print( 'CVX Client' )
      indent = ' ' * 2
      print( f'{indent}Status: {self.statusStr()}' )
      print( f'{indent}Source interface: {self.sourceIntfStr()}' )
      print( f'{indent}VRF: {self.vrfName if self.vrfName else DEFAULT_VRF}' )
      print( f'{indent}Heartbeat interval: {self.heartbeatInterval}' )
      print( f'{indent}Heartbeat timeout: {self.heartbeatTimeout}' )
      if self.cvxClusterName:
         print( f'{indent}Controller cluster name: {self.cvxClusterName}' )
         indent += ' ' * 2
      if self.enabled:
         # Build dict to check if multiple IP addresses map to the same hostname
         hostnames = ( c.hostname for c in self.controllerStatuses.values() )
         hostToIpCount = Counter( hostnames )

         for name, controllerStatus in self.controllerStatuses.items():
            hostname = controllerStatus.hostname
            if hostname and hostToIpCount[ hostname ] > 1:
               print( f'{indent}Controller status for {hostname} ({name})' )
            else:
               print( f'{indent}Controller status for {hostname or name}' )
            controllerStatus.display( indent=indent + ' ' * 2 )

         if self.sslProfileStatus:
            self.sslProfileStatus.render()

   # pylint: disable-next=inconsistent-return-statements
   def degrade( self, dictRepr, revision ):
      if revision == 1:
         controllerStatuses = dictRepr[ 'controllerStatuses' ]
         if not controllerStatuses:
            controllerName = ''
            controllerStatus = ControllerStatus(
               oobConnectionStatus='connecting',
               versionCompatibility='unnegotiated',
               negotiatedVersion=0, controllerUUID='' ).toDict()
            controllerStatus[ 'heartbeatStatus' ] = {}
            controllerStatus[ 'heartbeatStatus' ][ 'lastHeartbeatSent' ] = 0.0
            controllerStatus[ 'heartbeatStatus' ][ 'lastHeartbeatReceived'] = 0.0
         else:
            # In revision 1, only one controller is supported. Choose one (the 1st)
            # from the controllerStatuses to display
            controllerName = next( iter( controllerStatuses ) )
            controllerStatus = controllerStatuses[ controllerName ]

         connectionStatusXform = { 'connecting': 'connectionStateUnknown',
                                   'negotiating': 'connecting',
                                   'established': 'connected',
                                   'disconnecting': 'disconnected',
                                   'shutdown' : 'disconnected' }
         dictRepr[ 'connectionStatus' ] = connectionStatusXform[ controllerStatus[
               'oobConnectionStatus' ] ]
         dictRepr[ 'ctrlName' ] = controllerName
         for attrName in [ 'versionCompatibility', 'negotiatedVersion',
                           'controllerUUID', 'oobConnectionSecured',
                           'ibConnectionSecured' ]:
            if attrName in controllerStatus:
               dictRepr[ attrName ] = controllerStatus.get( attrName )

         # In rev1, the 'heartbeatStatus' is a dict mapping from controller systemId
         # to HeartbeatStatus. In rev2, it is a single HeartbeatStatus residing in
         # the controllerStatus. The info of controller systemId is no longer
         # available in rev2. Fake one here.
         systemIdStr = Tac.Value( 'Controller::SystemId',
                                  '00:00:00:00:00:00' ).stringValue
         heartbeatStatus = { systemIdStr : {} }
         hbStatus = controllerStatus[ 'heartbeatStatus' ]
         for attrName in [ 'lastHeartbeatSent', 'lastHeartbeatReceived',
                           'offset' ]:
            heartbeatStatus[ systemIdStr ][ attrName ] = hbStatus.get( attrName, 0 )
         dictRepr[ 'heartbeatStatus' ] = heartbeatStatus

         # 'lastConnectedTime' is no longer recorded in revision 2. set it to 0.
         dictRepr[ 'lastConnectedTime' ] = 0.0
         
         # remove new fields from dictRepr
         del dictRepr[ 'controllerStatuses' ]
         del dictRepr[ 'cvxClusterName' ]
         del dictRepr[ 'sourceIntf' ]
         return dictRepr

class MgmtCvxStatus( Model ):
   __revision__ = 3
   clusterStatus = Submodel( valueType=MgmtCvxClusterStatus,
         help='Management CVX Cluster status' )
   secondaryClusterStatus = Submodel( valueType=MgmtCvxClusterStatus,
         help='Management Secondary CVX Cluster status', optional=True )

   def render( self ):
      self.clusterStatus.render()
      if self.secondaryClusterStatus:
         print( 'Secondary', end=' ' )
         self.secondaryClusterStatus.render()

   # pylint: disable-next=inconsistent-return-statements
   def degrade( self, dictRepr, revision ):
      if revision == 1:
         return self.clusterStatus.degrade( dictRepr[ 'clusterStatus' ], revision )
      elif revision == 2:
         return dictRepr[ 'clusterStatus' ]


class MgmtCvxService( Model ):
   name = Str( help='Service name' )
   status = Submodel( valueType=ServiceStatus, help='Service status' )

   def render( self ):
      print( self.name )
      self.status.display( indent='  ' )

class MgmtCvxAllServices( Model ):
   services = List( valueType=MgmtCvxService,
                    help='Status of all services on the switch' )

   def render( self ):
      for service in self.services:
         service.render()
