# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import re
import datetime

import Tac
import Tracing
import BasicCli
import CliCommand
import CliMatcher
from CliPlugin.ControllerCli import CvxConfigMode, serviceKwMatcher, addNoCvxCallback
from CliPlugin.ControllerClient import CvxMgmtConfigMode, addNoMgmtCvxCallback
from CliPlugin.ControllerdbLib import registerNotifiee
from CliPlugin.ControllerDebugModel import Dirs, DirStatus
from CliPlugin.ControllerDebugModel import EntityStatus, PingPongStatus
from CliPlugin.DebugServiceCli import CvxDebugServiceMode, MgmtCvxDebugServiceMode
import ConfigMount
import LazyMount
import Plugins
import ShowCommand

# pkgdeps: rpm ControllerDebug-lib

ctrlDebugDir = 'controller/debug'
mgmtCtrlDebugDir = 'mgmt/controller/debug'

__defaultTraceHandle__ = Tracing.Handle( 'ControllerDebugCli' )
t0 = Tracing.trace0
t8 = Tracing.trace8

serverConfig = None
serverServiceConfig = None
clientConfig = None
clientServiceConfig = None
clientStatus = None
serverStatus = None
serverLocalTs = None
serverMountDir = None
clientLocalTs = None
clientPublishedDir = None
convergenceStatus = None
cvxDebugMPConfig = None
cvxMgmtDebugMPConfig = None
cvxDebugMPStatus = None
cvxMgmtDebugMPStatus = None
overrideServiceConfig = None
mgmtOverrideServiceConfig = None

_dateFmt = '%Y-%m-%d %H:%M:%S'

wordMatcher = CliMatcher.PatternMatcher( '.+', helpname='WORD',
      helpdesc='A single word' )

def debugServerGuard( mode, token ):
   if serverConfig.enabled:
      return None
   return 'service debug must be enabled first'

def debugClientGuard( mode, token ):
   if clientConfig.enabled:
      return None
   return 'service debug must be enabled first'

def printTs( timestamp ):
   td = datetime.datetime.fromtimestamp( timestamp.value )
   print( 'Last update: %s' % td.strftime( _dateFmt ) )
   print( 'In seconds: %d' % timestamp.value )

#--------------------------------------------------------------------------------
# [ no | default ] service debug
#--------------------------------------------------------------------------------
def resetConfig( config ):
   config.enabled = False
   config.updateInterval = 1
   config.maxRunTime = 0
   config.pingEnabled = False
   config.pingSize = 6
   config.maxPings = 0
   config.pingRunIdx = 0
   config.tableSize = 131072
   config.preserve = False
   config.versionWhiteList.clear()
   config.switchWhiteList.clear()

def disableServer( mode, args=None ):
   resetConfig( serverConfig )
   cvxDebugMPConfig.cdsDataDirConfig.subdirName.clear()
   cvxDebugMPConfig.cdsDataDirConfig.entityConfig.clear()
   cvxDebugMPConfig.cdsMountConfig.clear()
   cvxDebugMPConfig.cdsPublishConfig.clear()
   if 'ControllerDebug' in overrideServiceConfig.overrideService:
      del overrideServiceConfig.overrideService[ 'ControllerDebug' ]
   serverConfig.enabled = False
   serverServiceConfig.service[ 'ControllerDebug' ].enabled = False

def enableServer( mode, args ):
   serverConfig.enabled = True
   serverServiceConfig.service[ 'ControllerDebug' ].enabled = True

def enableClient( mode, args ):
   clientConfig.enabled = True
   clientServiceConfig.service[ 'ControllerDebug' ].enabled = True

def disableClient( mode, args=None ):
   clientConfig.enabled = False
   clientServiceConfig.service[ 'ControllerDebug' ].enabled = False

# When 'no cvx' gets called 'no service debug' should be called as well.
addNoCvxCallback( disableServer )
addNoMgmtCvxCallback( disableClient )

class ServiceDebugCmd( CliCommand.CliCommandClass ):
   syntax = 'service debug'
   noOrDefaultSyntax = syntax
   data = {
      'service': serviceKwMatcher,
      'debug': 'ControllerDebug service',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CvxDebugServiceMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = disableServer

CvxConfigMode.addCommandClass( ServiceDebugCmd )

class ServiceDebugCxvMgmtCmd( CliCommand.CliCommandClass ):
   syntax = 'service debug'
   noOrDefaultSyntax = syntax
   data = {
      'service': serviceKwMatcher,
      'debug': 'ControllerDebug service',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MgmtCvxDebugServiceMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = disableClient

CvxMgmtConfigMode.addCommandClass( ServiceDebugCxvMgmtCmd )

#-------------------------------------------------------------------------------
# [ no ] shutdown 
#-------------------------------------------------------------------------------
matcherShutdown = CliMatcher.KeywordMatcher( 'shutdown',
      helpdesc='Disable debug service' )

class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': matcherShutdown,
   }

   handler = disableServer
   noHandler = enableServer
   defaultHandler = disableServer

CvxDebugServiceMode.addCommandClass( ShutdownCmd )

class ShutdownCvxDebugCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': matcherShutdown,
   }

   handler = disableClient
   noHandler = enableClient
   defaultHandler = disableClient

MgmtCvxDebugServiceMode.addCommandClass( ShutdownCvxDebugCmd )

#------------------------------------------------------------------------------
# [ no | default ] interval INTERVAL
#------------------------------------------------------------------------------
intervalKwMatcher = CliMatcher.KeywordMatcher( 'interval',
                                       helpdesc='Set update interval' )
intervalMatcher = CliMatcher.PatternMatcher( r'\d+(\.\d+)?',
                                       helpname='INTERVAL (Seconds)',
                                       helpdesc='Interval' )

class SetIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'interval INTERVAL'
   noOrDefaultSyntax = 'interval [ INTERVAL ]'
   data = {
            'interval': intervalKwMatcher,
            'INTERVAL': intervalMatcher,
          }

   @staticmethod
   def _getConfig( mode ):
      if isinstance( mode, CvxDebugServiceMode ):
         return serverConfig
      if isinstance( mode, MgmtCvxDebugServiceMode ):
         return clientConfig
      assert False, 'Unknown mode'
      return None

   @staticmethod
   def handler( mode, args ):
      config = SetIntervalCmd._getConfig( mode )
      try:
         config.updateInterval = float( args[ 'INTERVAL' ] )
      except ValueError:
         mode.addError( 'Invalid interval value: %s' % args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = SetIntervalCmd._getConfig( mode )
      config.updateInterval = 1

CvxDebugServiceMode.addCommandClass( SetIntervalCmd )
MgmtCvxDebugServiceMode.addCommandClass( SetIntervalCmd )

#------------------------------------------------------------------------------
# [ no | default ] stats interval INTERVAL
#------------------------------------------------------------------------------
class StatsIntervalHandler( CliCommand.CliCommandClass ):
   syntax = 'stats interval INTERVAL'
   noOrDefaultSyntax = 'stats interval'

   data = { 'stats': 'Statistics collecting options',
            'interval': 'Statistics collecting interval',
            'INTERVAL': intervalMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      try:
         serverConfig.statsCollectInterval = float( args[ 'INTERVAL' ] )
      except ValueError:
         mode.addError( 'Invalid interval value: %s' % args[ 'INTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serverConfig.statsCollectInterval = 1.0

CvxDebugServiceMode.addCommandClass( StatsIntervalHandler )
 
#--------------------------------------------------------------------------------
# [ no | default ] preserve
#--------------------------------------------------------------------------------
class PreserveCmd( CliCommand.CliCommandClass ):
   syntax = 'preserve'
   noOrDefaultSyntax = syntax
   data = {
      'preserve': 'Do state-preserving unmounts on disconnect',
   }

   @staticmethod
   def handler( mode, args ):
      serverConfig.preserve = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serverConfig.preserve = False

CvxDebugServiceMode.addCommandClass( PreserveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] pingpong [ { ( ping-size PING_SIZE ) |
#                               ( max-pings MAX_PING_SIZE ) |
#                               ( run-time RUNTIME ) |
#                               ( table-size TABLESIZE ) } ]
#--------------------------------------------------------------------------------
class PingpongCvxDebugCmd( CliCommand.CliCommandClass ):
   syntax = ( 'pingpong [ { ( ping-size PING_SIZE ) | ( max-pings MAX_PING_SIZE ) | '
              '( run-time RUNTIME ) | ( table-size TABLESIZE ) } ]' )
   noOrDefaultSyntax = syntax
   data = {
      'pingpong': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'pingpong', helpdesc='PingPong test' ),
         guard=debugClientGuard ),
      'ping-size': CliCommand.singleKeyword( 'ping-size',
         helpdesc='How many pings to send' ),
      'PING_SIZE': CliMatcher.IntegerMatcher( 1, 2048, helpdesc='Ping size' ),
      'max-pings': CliCommand.singleKeyword( 'max-pings',
         helpdesc='Max pings to send' ),
      'MAX_PING_SIZE': CliMatcher.PatternMatcher( pattern=r'\d+',
         helpdesc='Ping size', helpname='Ping size (integer)' ),
      'run-time': CliCommand.singleKeyword( 'run-time',
         helpdesc='How long to run the test' ),
      'RUNTIME': CliMatcher.PatternMatcher( pattern=r'\d+',
         helpdesc='Run time', helpname='Run time (seconds)' ),
      'table-size': CliCommand.singleKeyword( 'table-size',
         helpdesc='Max pings on the wire' ),
      'TABLESIZE': CliMatcher.PatternMatcher( pattern=r'\d+',
         helpdesc='Max pings on the wire', helpname='Max pings on the wire' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'PING_SIZE' in args:
         clientConfig.pingSize = args[ 'PING_SIZE' ][ 0 ]
      if 'MAX_PING_SIZE' in args:
         clientConfig.maxPings = int( args[ 'MAX_PING_SIZE' ][ 0 ] )
      if 'RUNTIME' in args:
         clientConfig.maxRunTime = int( args[ 'RUNTIME' ][ 0 ] )
      if 'TABLESIZE' in args:
         clientConfig.tableSize = int( args[ 'TABLESIZE' ][ 0 ] )

      clientConfig.pingEnabled = True
      clientConfig.pingRunIdx = clientConfig.pingRunIdx + 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clientConfig.pingEnabled = False

MgmtCvxDebugServiceMode.addCommandClass( PingpongCvxDebugCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] pingpong
#-----------------------------------------------------------------------------------
class PingpongCmd( CliCommand.CliCommandClass ):
   syntax = 'pingpong'
   noOrDefaultSyntax = syntax
   data = {
      'pingpong': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'pingpong', helpdesc='PingPong test' ),
         guard=debugServerGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      serverConfig.pingEnabled = True

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      serverConfig.pingEnabled = False

CvxDebugServiceMode.addCommandClass( PingpongCmd )

#----------------------------------------------------------------------------------
# [ no | default ] version VERSION
#----------------------------------------------------------------------------------
def setVersion( config, mode, version, no=None ):
   if no:
      del config.overrideService[ 'ControllerDebug' ]
      return
   try:
      ver = int(version)
      if 'ControllerDebug' in config.overrideService:
         debugService = config.overrideService[ 'ControllerDebug' ]
      else :
         debugService = config.overrideService.newMember( 'ControllerDebug', )
      debugService.enabled = True
      debugService.versionConfig = ('debug-version',)
      serviceVersion = Tac.newInstance( 'Controller::ServiceVersionV1',  ver )
      debugService.versionConfig.addVersion( serviceVersion )
   except ValueError:
      mode.addError( 'Invalid version value: %s' % version )
      return

versionKwMatcher = CliMatcher.KeywordMatcher( 'version',
      helpdesc='Set overriding version' )
versionMatcher = CliMatcher.PatternMatcher( r'\d+(\.\d+)?',
      helpname='VERSION (num)', helpdesc='Version' )

class VersionCmd( CliCommand.CliCommandClass ):
   syntax = 'version VERSION'
   noOrDefaultSyntax = 'version [ VERSION ]'
   data = {
      'version': versionKwMatcher,
      'VERSION': versionMatcher
   }
   handler = lambda mode, args: setVersion( overrideServiceConfig, mode,
         args[ 'VERSION' ], None )
   noOrDefaultHandler = lambda mode, args: setVersion( overrideServiceConfig,
         mode, None, True )

CvxDebugServiceMode.addCommandClass( VersionCmd )
 
class VersionMgmtCvxCmd( CliCommand.CliCommandClass ):
   syntax = 'version VERSION'
   noOrDefaultSyntax = 'version [ VERSION ]'
   data = {
      'version': versionKwMatcher,
      'VERSION': versionMatcher
   }
   handler = lambda mode, args: setVersion( mgmtOverrideServiceConfig, mode,
         args[ 'VERSION' ], None )
   noOrDefaultHandler = lambda mode, args: setVersion( mgmtOverrideServiceConfig,
         mode, None, True )

MgmtCvxDebugServiceMode.addCommandClass( VersionMgmtCvxCmd )

def makePingPongStatus( status, enabled, server=True ):
   pps = PingPongStatus()

   pps.enabled = enabled

   pps.runTime = status.totalRunSeconds
   pps.totalPings = status.totalPings
   pps.addsPerSecond = ( int( status.totalPings / pps.runTime )
                          if pps.runTime > 0 else 0 )
   pps.addsPerSecondSnapshot = status.addsPerSecond

   if server:
      pps.delsPerSecond = status.delsPerSecond
      pps.downtimeCount = status.downtimeCount
      pps.longestDowntime = status.longestDowntime
      pps.totalDowntime = status.totalDowntime
      pps.dirChangeCount = status.dirChangeCount
      pps.totalDataChangeCount = status.totalDataChangeCount

   return pps

#------------------------------------------------------------------------------
# show management cvx debug pingpong
#------------------------------------------------------------------------------
def showMgmtCvxDebugPingPong( mode, args ):
   enabled = clientConfig.pingEnabled and \
      ( clientConfig.pingRunIdx != clientStatus.lastRunIdx )
   return makePingPongStatus( clientStatus, enabled, server=False )

#------------------------------------------------------------------------------
# show cvx debug pingpong
#------------------------------------------------------------------------------
def showCvxDebugPingPong( mode, args ):
   enabled = serverConfig.pingEnabled
   return makePingPongStatus( clientStatus, enabled, server=True )

#------------------------------------------------------------------------------
# show cvx debug server convergencestatus  
#------------------------------------------------------------------------------
def showCvxDebugConvergenceStatus( mode, args ):
   print( 'convergenceInProgress is :', convergenceStatus.convergenceInProgress )
   print( 'doInitialSwitchMountsComplete Count is : %d' % (
      convergenceStatus.doInitialSwitchMountsCompleteCount, ) )

#------------------------------------------------------------------------------
# show cvx debug [ client | server ] ts
#------------------------------------------------------------------------------
def showServerMountedTs( mode, args ):
   fmt = '%-20s %-40s'
   print( fmt % ( 'Switch ID', 'Timestamp' ) )
   print( fmt % ( '-' * 20, '-' * 40 ) )
   for switchId in serverMountDir:
      timestamp = serverMountDir.get( switchId )
      if ( not timestamp or 
         # pylint: disable-next=unidiomatic-typecheck
         type( timestamp ) != Tac.Type( 'ControllerDebug::TimestampV1' ) ):
         continue

      td = datetime.datetime.fromtimestamp( timestamp.value )
      tsStr = f'{td.strftime( _dateFmt )} ({timestamp.value})'
      print( fmt % ( re.sub( '-', ':', switchId ), tsStr ) )

#-----------------------------------------------------------------------------
# show cvx debug [ client | server ] version
#------------------------------------------------------------------------------
def showClientVersion( mode, args ):
   if 'ControllerDebug' in mgmtOverrideServiceConfig.overrideService:
      clientVersionSet = mgmtOverrideServiceConfig.\
            overrideService[ 'ControllerDebug' ].\
            versionConfig.version
   else:
      print( 'No override version available' )
      return
   print( 'Supported versions ', end=' ' )
   for version in clientVersionSet:
      print( version, end=' ' )

def showServerVersion( mode, args ):
   if 'ControllerDebug' in overrideServiceConfig.overrideService:
      serverVersionSet = overrideServiceConfig.\
            overrideService[ 'ControllerDebug' ].\
            versionConfig.version
   else:
      print( 'No override version available' )
      return
   print( 'Supported versions ', end=' ' )
   for version in serverVersionSet:
      print( version, end=' ' )

#------------------------------------------------------------------------------
# show management cvx debug [ client | server ] ts
#------------------------------------------------------------------------------
def printPublishedTs( mode, args ):
   if 'ts' in clientPublishedDir:
      printTs( clientPublishedDir[ 'ts' ] )
   
#---------------------------------------------------------------------------------
# [ no | default ] mount src SRC target TARGET version WHITELIST switch WHITELIST
# [ no | default ] publish src SRC target TARGET version WHITELIST switch WHITELIST
#---------------------------------------------------------------------------------
def enableMount( mode, args ):
   src = args[ 'SRC' ]
   version = args.get( 'VERSION' )
   oldDebugMPConfig = None
   try :
      oldDebugMPConfig = cvxDebugMPConfig.cdsMountConfig[ src ]
   except KeyError:
      pass

   switchWhitelist = versionWhitelist = ''

   if oldDebugMPConfig:
      switchWhitelist = oldDebugMPConfig.switchWhitelist
      versionWhitelist = oldDebugMPConfig.versionWhitelist

   if 'TARGET' in args:
      target = args[ 'TARGET' ][ 0 ]
   else:
      target = src

   if version:
      versionWhitelist = version

   if 'SWITCH' in args:
      switchWhitelist = args[ 'SWITCH' ][ 0 ]

   debugMPConfig = Tac.Value( 'ControllerDebug::ControllerDebugCliMPConfig',
                              src, target, versionWhitelist, switchWhitelist )
   cvxDebugMPConfig.cdsMountConfig.addMember( debugMPConfig )

def disableMount( mode, args ):
   del cvxDebugMPConfig.cdsMountConfig[ args[ 'SRC' ] ]

def disableMountSwitchWl( mode, args ):
   args[ 'SWITCH' ] = [ '' ]
   src = args[ 'SRC' ]
   try :
      _ = cvxDebugMPConfig.cdsMountConfig[ src ]
      enableMount( mode, args )
   except KeyError:
      pass
      
def enablePublish( mode, args ):
   src = args[ 'SRC' ]
   version = args.get( 'VERSION' )
   oldDebugMPConfig = None
   try :
      oldDebugMPConfig = cvxDebugMPConfig.cdsPublishConfig[ src ]
   except KeyError:
      pass

   switchWhitelist = versionWhitelist = ''

   if oldDebugMPConfig:
      switchWhitelist = oldDebugMPConfig.switchWhitelist
      versionWhitelist = oldDebugMPConfig.versionWhitelist

   if 'TARGET' in args:
      target = args[ 'TARGET' ][ 0 ]
   else:
      target = src

   if version:
      versionWhitelist = version

   if 'SWITCH' in args:
      switchWhitelist = args[ 'SWITCH' ][ 0 ]

   debugMPConfig = Tac.Value( 'ControllerDebug::ControllerDebugCliMPConfig',
                              src, target, versionWhitelist, switchWhitelist )
   cvxDebugMPConfig.cdsPublishConfig.addMember( debugMPConfig )

def disablePublish( mode, args ):
   del cvxDebugMPConfig.cdsPublishConfig[ args[ 'SRC' ] ]

def disablePublishSwitchWl( mode, args ):
   src = args[ 'SRC' ]
   args[ 'SWITCH' ] = [ '' ]
   try :
      _ = cvxDebugMPConfig.cdsPublishConfig[ src ]
      enablePublish( mode, args )
   except KeyError:
      pass

matcherMount = CliMatcher.KeywordMatcher( 'mount', helpdesc='Mount a path' )
matcherName = CliMatcher.KeywordMatcher( 'name', helpdesc='Name of the whitelist' )
matcherPublish = CliMatcher.KeywordMatcher( 'publish', helpdesc='Publish a path' )
matcherSrc = CliMatcher.KeywordMatcher( 'src', helpdesc='Source path' )
matcherSwitch = CliMatcher.KeywordMatcher( 'switch',
      helpdesc='The switchWhiteList name to apply to this mount group' )
matcherTarget = CliMatcher.KeywordMatcher( 'target', helpdesc='Target path' )
nodeMount = CliCommand.Node( matcher=matcherMount, guard=debugServerGuard )
nodePublish = CliCommand.Node( matcher=matcherPublish, guard=debugServerGuard )

#--------------------------------------------------------------------------------
# mount src SRC version VERSION [ { ( target TARGET ) | ( switch SWITCH ) } ]
#--------------------------------------------------------------------------------
class MountSrcVersionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'mount src SRC version VERSION '
                                    '[ { ( target TARGET ) | ( switch SWITCH ) } ]' )
   noOrDefaultSyntax = 'mount src SRC'
   data = {
      'mount': nodeMount,
      'src': matcherSrc,
      'SRC': wordMatcher,
      'version': 'The versionWhiteList name to apply to this mount group',
      'VERSION': wordMatcher,
      'target': CliCommand.singleNode( matcherTarget ),
      'TARGET': wordMatcher,
      'switch': CliCommand.singleNode( matcherSwitch ),
      'SWITCH': wordMatcher,
   }
   handler = enableMount
   noOrDefaultHandler = disableMount

CvxDebugServiceMode.addCommandClass( MountSrcVersionCmd )

class MountSrcSwitchCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'mount src SRC switch'
   data = {
      'mount': nodeMount,
      'src': matcherSrc,
      'SRC': wordMatcher,
      'switch': matcherSwitch,
   }
   noOrDefaultHandler = disableMountSwitchWl

CvxDebugServiceMode.addCommandClass( MountSrcSwitchCmd )

#--------------------------------------------------------------------------------
# publish src SRC version VERSION [ { ( target TARGET ) | ( switch SWITCH ) } ]
#--------------------------------------------------------------------------------
class PublishSrcVersionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'publish src SRC version VERSION '
                                    '[ { ( target TARGET ) | ( switch SWITCH ) } ]' )
   noOrDefaultSyntax = 'publish src SRC'
   data = {
      'publish': nodePublish,
      'src': matcherSrc,
      'SRC': wordMatcher,
      'version': 'The versionWhiteList name to apply to this mount group',
      'VERSION': wordMatcher,
      'target': CliCommand.singleNode( matcherTarget ),
      'TARGET': wordMatcher,
      'switch': CliCommand.singleNode( matcherSwitch ),
      'SWITCH': wordMatcher,
   }
   handler = enablePublish
   noOrDefaultHandler = disablePublish

CvxDebugServiceMode.addCommandClass( PublishSrcVersionCmd )

class PublishSrcSwitchCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'publish src SRC switch'
   data = {
      'publish': nodePublish,
      'src': matcherSrc,
      'SRC': wordMatcher,
      'switch': matcherSwitch,
   }
   noOrDefaultHandler = disablePublishSwitchWl

CvxDebugServiceMode.addCommandClass( PublishSrcSwitchCmd )

#------------------------------------------------------------------------------
# [ no | default ] crdir path PATH
#------------------------------------------------------------------------------
class CrdirHandler( CliCommand.CliCommandClass ):
   syntax = 'crdir path PATH'
   noOrDefaultSyntax = 'crdir path PATH'

   data = { 'crdir': 'Create a dir',
            'path': 'Dir path',
            'PATH': wordMatcher,
          }

   @staticmethod
   def _addDir( isClient, path ):
      if isClient:
         cvxMgmtDebugMPConfig.cdcDataDirConfig.subdirName[ path ] = True   
      else:
         cvxDebugMPConfig.cdsDataDirConfig.subdirName[ path ] = True   
   
   @staticmethod
   def _deleteDir( isClient, path ):
      if isClient:
         if cvxMgmtDebugMPConfig.cdcDataDirConfig is not None:
            del cvxMgmtDebugMPConfig.cdcDataDirConfig.subdirName[ path ]   
      else:
         if cvxDebugMPConfig.cdsDataDirConfig is not None:
            del cvxDebugMPConfig.cdsDataDirConfig.subdirName[ path ] 

   @staticmethod
   def handler( mode, args ):
      CrdirHandler._addDir( mode.isClient, args[ 'PATH' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      CrdirHandler._deleteDir( mode.isClient, args[ 'PATH' ] )

CvxDebugServiceMode.addCommandClass( CrdirHandler )
MgmtCvxDebugServiceMode.addCommandClass( CrdirHandler )

def printEntityStatus( nSpaces, entityStatus ):
   print( ' '*nSpaces, 'stringBufCount:', entityStatus.stringBufCount )
   print( ' '*nSpaces, 'stringBufChurnCount:', entityStatus.stringBufChurnCount )

#------------------------------------------------------------------------------
# show [ management ] cvx debug dirs 
#------------------------------------------------------------------------------
def printDirs( isClient ):
   dirModel = Dirs()
   if isClient:
      dataDirStatus = cvxMgmtDebugMPStatus.cdcDataDirStatus
   else:
      dataDirStatus = cvxDebugMPStatus.cdsDataDirStatus
   if dataDirStatus:
      dirModel.entityChurnCount = dataDirStatus.entityChurnCount
      dirModel.dirChurnCount = dataDirStatus.dirChurnCount
      for d in dataDirStatus.dirStatus:
         dirStatusModel = DirStatus()
         dirStatus = dataDirStatus.dirStatus[ d ]
         dirStatusModel.entryCount = dirStatus.entryCount
         dirStatusModel.entityChurnCount = dirStatus.entityChurnCount
         for es in dirStatus.entityStatus:
            entityStatusModel = EntityStatus()
            entityStatusModel.stringBufCount = es.stringBufCount
            entityStatusModel.stringBufChurnCount = es.stringBufChurnCount
            dirStatusModel[ es ] = entityStatusModel
         dirModel[ d ] = dirStatusModel
   return dirModel

class DirShowHandler( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ management ] cvx debug dirs'
   data = { 
            'management': '',
            'cvx': '',
            'debug': 'Debug service information',
            'dirs': 'display dirs'
          }
   hidden = True
   cliModel = Dirs

   @staticmethod
   def handler( mode, args ):
      isClient = 'management' in args
      return printDirs( isClient )

BasicCli.addShowCommandClass( DirShowHandler )

#------------------------------------------------------------------------------
# [ no | default ] entity create STRING
#------------------------------------------------------------------------------
def addEntity( isClient, name, index=None, value=None ):
   # pylint: disable-msg=E1103
   sb = None
   if isClient:
      if not name in cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig:
         cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig.newMember(name)
      if index is not None:
         i = int( index )
         sb = cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig[ name ].stringBuf   
         sb[ i ] = value
   else:
      if not name in cvxDebugMPConfig.cdsDataDirConfig.entityConfig:
         cvxDebugMPConfig.cdsDataDirConfig.entityConfig.newMember( name )
      if index is not None:
         i = int( index )
         sb = cvxDebugMPConfig.cdsDataDirConfig.entityConfig[ name ].stringBuf   
         sb[ i ] = value
   
def deleteEntity( isClient, name, index=None, value=None ):
   # pylint: disable-msg=E1103
   if index is None:
      if isClient:
         del cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig[ name ]   
      else:
         del cvxDebugMPConfig.cdsDataDirConfig.entityConfig[ name ] 
   else:
      sb = None  
      i = int( index )
      if isClient:
         if name in cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig:
            sb = cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig[ name ].stringBuf
            del sb[ i ]
      else:
         if name in cvxDebugMPConfig.cdsDataDirConfig.entityConfig:
            sb = cvxDebugMPConfig.cdsDataDirConfig.entityConfig[ name ].stringBuf   
            del sb[ i ]

class EntityCreateHandler( CliCommand.CliCommandClass ):
   syntax = 'entity create NAME'
   noOrDefaultSyntax = 'entity create NAME'

   data = { 'entity': 'modify an entity',
            'create': 'Create an entity',
            'NAME': wordMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      addEntity( mode.isClient, args[ 'NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      deleteEntity( mode.isClient, args[ 'NAME' ] )

CvxDebugServiceMode.addCommandClass( EntityCreateHandler )
MgmtCvxDebugServiceMode.addCommandClass( EntityCreateHandler )
         
#------------------------------------------------------------------------------
# [ no | default ] entity name STRING index INDEX value VALUE
#------------------------------------------------------------------------------
class EntityNameHandler( CliCommand.CliCommandClass ):
   syntax = 'entity name NAME index INDEX value VALUE'
   noOrDefaultSyntax = 'entity name NAME index INDEX value VALUE'
   data = { 'entity': 'modify an entity',
            'name': 'Publish a static entity',
            'NAME': wordMatcher,
            'index': 'value index',
            'INDEX': CliMatcher.PatternMatcher( r'\d+',
               helpname='Value index (integer)', helpdesc='Value index' ),
            'value': 'entity value',
            'VALUE': wordMatcher 
          }

   @staticmethod
   def handler( mode, args ):
      addEntity( mode.isClient, args[ 'NAME' ], args[ 'INDEX' ], args[ 'VALUE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      deleteEntity( mode.isClient, args[ 'NAME' ], args[ 'INDEX' ], args[ 'VALUE' ] )

CvxDebugServiceMode.addCommandClass( EntityNameHandler )
MgmtCvxDebugServiceMode.addCommandClass( EntityNameHandler )

#------------------------------------------------------------------------------
# show [management] cvx debug entities  
#------------------------------------------------------------------------------
def printStringBuf( sb ):
   if not sb:
      return
   for strng in sb:
      print( strng, ':', sb[ strng ] )

def printEntities( isClient ):
   # pylint: disable-msg=E1103
   if isClient:
      entityNames = cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig
      dataDirStatus = cvxMgmtDebugMPStatus.cdcDataDirStatus
   else:
      entityNames = cvxDebugMPConfig.cdsDataDirConfig.entityConfig
      dataDirStatus = cvxDebugMPStatus.cdsDataDirStatus
   if dataDirStatus:
      for entityName in dataDirStatus.entityStatus:
         print( entityName )
         if entityName in entityNames:
            entityData = entityNames[ entityName ]
            printStringBuf( entityData.stringBuf )
         printEntityStatus( 2, dataDirStatus.entityStatus[ entityName ] )

class EntityShowHandler( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ management ] cvx debug entities'
   data = { 
            'management': '',
            'cvx': '',
            'debug': 'Debug service information',
            'entities': 'display entities'
          }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      isClient = 'management' in args
      printEntities( isClient )

BasicCli.addShowCommandClass( EntityShowHandler )

#------------------------------------------------------------------------------
# [ no | default ] entity churn STRING entries ENTRIES size SIZE rate RATE
#------------------------------------------------------------------------------
class EntityChurnHandler( CliCommand.CliCommandClass ):
   syntax = 'entity churn NAME entries ENTRIES size SIZE rate RATE'
   noOrDefaultSyntax = \
            'entity churn NAME entries ENTRIES size SIZE rate RATE'

   data = { 'entity': 'modify an entity',
            'churn': 'Publish an entity that constantly changes value',
            'NAME': wordMatcher,
            'entries': 'num entries',
            'ENTRIES': CliMatcher.PatternMatcher( r'\d+',
               helpname='number of entries (integer)',
               helpdesc='number of entries' ),
            'size': 'size of entries',
            'SIZE': CliMatcher.PatternMatcher( r'\d+',
               helpname='entry size (integer)', helpdesc='size of entries' ),
            'rate': 'churn rate (seconds)',
            'RATE': CliMatcher.PatternMatcher( r'\d+',
               helpname='churn rate seconds (integer)', helpdesc='churn rate' ),
          }

   @staticmethod
   def _handleEntityChurn( isClient, name, entries, size, rate ):
      # pylint: disable-msg=E1103
      cfg = None
      if isClient:
         if not name in cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig:
            cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig.newMember(name)
         cfg = cvxMgmtDebugMPConfig.cdcDataDirConfig.entityConfig[ name ]
      else:
         if not name in cvxDebugMPConfig.cdsDataDirConfig.entityConfig:
            cvxDebugMPConfig.cdsDataDirConfig.entityConfig.newMember(name)
         cfg = cvxDebugMPConfig.cdsDataDirConfig.entityConfig[ name ]
      assert cfg
      cfg.autoChurnEntries = int( entries )  
      cfg.autoChurnEntrySize = int( size )  
      cfg.autoChurnRate = int( rate )  
   
   @staticmethod
   def handler( mode, args ):
      EntityChurnHandler._handleEntityChurn( mode.isClient, args[ 'NAME' ],
            args[ 'ENTRIES' ], args[ 'SIZE' ], args[ 'RATE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      EntityChurnHandler._handleEntityChurn( mode.isClient, args[ 'NAME' ],
            args[ 'ENTRIES' ], args[ 'SIZE' ], 0 ) # pass rate=0 to disable churn

CvxDebugServiceMode.addCommandClass( EntityChurnHandler )
MgmtCvxDebugServiceMode.addCommandClass( EntityChurnHandler )   

#--------------------------------------------------------------------------------
# [ no | default ] whitelist switch name NAME
#--------------------------------------------------------------------------------
matcherName = CliMatcher.KeywordMatcher( 'name', helpdesc='Name of the whitelist' )
matcherWhitelist = CliMatcher.KeywordMatcher( 'whitelist',
      helpdesc='Edit/Create version or switch whitelist' )

class WhitelistSwitchNameCmd( CliCommand.CliCommandClass ):
   syntax = 'whitelist switch name NAME'
   noOrDefaultSyntax = syntax
   data = {
      'whitelist': matcherWhitelist,
      'switch': 'Create a switch whitelist',
      'name': matcherName,
      'NAME': wordMatcher
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      switchWhitelist = serverConfig.switchWhiteList
      if name in switchWhitelist:
         print( '%s switchlist already exist' % name )
      else:
         switchWhitelist.newMember( name )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      name = args[ 'NAME' ]
      switchWhitelist = serverConfig.switchWhiteList
      if name in switchWhitelist:
         del switchWhitelist[ name ]
      else:
         mode.addError( 'Cannot delete non exisitng [%s] switch whitelist' % name )

CvxDebugServiceMode.addCommandClass( WhitelistSwitchNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] whitelist version name NAME
#--------------------------------------------------------------------------------
class WhitelistVersionNameCmd( CliCommand.CliCommandClass ):
   syntax = 'whitelist version name NAME'
   noOrDefaultSyntax = syntax
   data = {
      'whitelist': matcherWhitelist,
      'version': 'Create a version whitelist',
      'name': matcherName,
      'NAME': wordMatcher
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'NAME' ]
      versionWhitelist = serverConfig.versionWhiteList
      if name in versionWhitelist:
         print( '%s versionlist already exist' % name )
      else:
         versionWhitelist.newMember( name )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      name = args[ 'NAME' ]
      versionWhitelist = serverConfig.versionWhiteList
      if name in versionWhitelist:
         del versionWhitelist[name]
      else:
         mode.addError( 'Cannot delete non exisitng [%s] version whitelist' % name )

CvxDebugServiceMode.addCommandClass( WhitelistVersionNameCmd )

#-----------------------------------------------------------------------------
# whitelist name NAME [ add | delete ] <switch1,switch2..>|<version1,version2..>
#------------------------------------------------------------------------------
class WhiteListHandler( CliCommand.CliCommandClass ):
   syntax = 'whitelist name NAME (add|delete) VALUE'
   noOrDefaultSyntax = 'whitelist name NAME (add|delete) VALUE'

   data = { 'whitelist': 'Edit/Create version or switch whitelist',
            'name': 'name of the whitelist',
            'NAME': wordMatcher,
            'add': 'Add elements to the whitelist',
            'delete': 'Delete elements from the whitelist',
            'VALUE': wordMatcher
          }

   @staticmethod
   def _handleAddToWhiteList( mode, name, add ):
      switchWhitelist = serverConfig.switchWhiteList
      versionWhitelist = serverConfig.versionWhiteList
      whitelist = add.split( ',' )
      
      if whitelist is None:
         print( 'Error while parsing whitelist values' )
         return

      if name in switchWhitelist:
         config = switchWhitelist[name]
         for switch in whitelist:
            config.allowedMembers[switch] = True
         return
      elif name in versionWhitelist:
         config = versionWhitelist[name]
         for version in whitelist:
            try:
               int(version)
            except ValueError:
               print( ' %s not an integer ' % version )
               continue
            config.allowedMembers[version] = True
         return
      print( ' %s, no such named whitelist' % name )
  
   @staticmethod
   def _handleDeleteFromWhiteList( mode, name, delete ):
      versionWhitelist = serverConfig.versionWhiteList
      switchWhitelist = serverConfig.switchWhiteList
      whitelist = delete.split( ',' )

      if whitelist is None:
         print( 'Error while parsing whitelist values' )
         return

      if name in switchWhitelist:
         config = switchWhitelist[name]
         for switch in whitelist:
            del config.allowedMembers[switch]
         return
      elif name in versionWhitelist:
         config = versionWhitelist[name]
         for version in whitelist:
            try:
               int(version)
            except ValueError:
               print( '%s not an integer ' % version )
               continue
            del config.allowedMembers[version]
         return
      print( ' %s, no such named whitelist' % name )
    
   @staticmethod
   def handler( mode, args ):
      if 'add' in args:
         WhiteListHandler._handleAddToWhiteList( mode, args[ 'NAME' ],
               args[ 'VALUE' ] )
      elif 'delete' in args:
         WhiteListHandler._handleDeleteFromWhiteList( mode, args[ 'NAME' ],
               args[ 'VALUE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return

CvxDebugServiceMode.addCommandClass( WhiteListHandler )

def doControllerMounts( controllerdbEm ):
   global serverLocalTs, serverMountDir
   t8( 'ControllerDebugServer.doControllerMounts():',
       'enable' if controllerdbEm else 'disable' )
   if controllerdbEm:
      serverLocalTs = LazyMount.mount( controllerdbEm, 'controller/debug/v1/ts',
                                       'ControllerDebug::TimestampV1', 'r' )
      serverMountDir = LazyMount.mount( controllerdbEm, mgmtCtrlDebugDir + '/v1/ts',
                                       'Tac::Dir', 'ri' )
   else:
      serverLocalTs = None
      serverMountDir = None


@Plugins.plugin( requires=( 'ControllerdbMgr', ) )
def Plugin( em ):
   global clientLocalTs, serverConfig, clientConfig, clientStatus
   global clientServiceConfig, serverServiceConfig
   global clientPublishedDir
   global serverStatus
   global convergenceStatus
   global mgmtOverrideServiceConfig, overrideServiceConfig
   global cvxDebugMPConfig, cvxMgmtDebugMPConfig
   global cvxDebugMPStatus, cvxMgmtDebugMPStatus

   t0( 'Loading ControllerDebugCli plugin' )
   registerNotifiee( doControllerMounts )
   clientLocalTs = LazyMount.mount( 
      em, mgmtCtrlDebugDir + '/v1/ts',
      'ControllerDebug::TimestampV1', 'r' )

   # We mount the directory instead of the timestamp entity directly,
   # because we might run the show command before Sysdb mounts it from
   # Controllerdb.
   clientPublishedDir = LazyMount.mount( 
      em, ctrlDebugDir + '/v1',
      'Tac::Dir', 'ri' )
   serverConfig = ConfigMount.mount( 
      em, ctrlDebugDir + '/config',
      'ControllerDebug::Config', 'w' )
   clientConfig = ConfigMount.mount( 
      em, mgmtCtrlDebugDir + '/config',
      'ControllerDebug::Config', 'w' )
   clientStatus = LazyMount.mount( 
      em, mgmtCtrlDebugDir + '/status',
      'ControllerDebug::PingPongStats', 'r' )
   serverStatus = LazyMount.mount( 
      em, ctrlDebugDir + '/status',
      'ControllerDebug::PingPongStats', 'r' )
   convergenceStatus = LazyMount.mount(
      em, ctrlDebugDir + '/local/convergencestatus',
      'ControllerDebug::ConvergenceStatus', 'r' )
   cvxDebugMPConfig = ConfigMount.mount(
      em, ctrlDebugDir + '/mpconfig', 
      'ControllerDebug::ControllerDebugServerCliConfig', 'w' )
   cvxMgmtDebugMPConfig = ConfigMount.mount(
      em, mgmtCtrlDebugDir + '/mpconfig', 
      'ControllerDebug::ControllerDebugClientCliConfig', 'w' )
   cvxDebugMPStatus = LazyMount.mount(
      em, ctrlDebugDir + '/mpstatus', 
      'ControllerDebug::ControllerDebugServerCliStatus', 'r' )
   cvxMgmtDebugMPStatus = LazyMount.mount(
      em, mgmtCtrlDebugDir + '/mpstatus', 
      'ControllerDebug::ControllerDebugClientCliStatus', 'r' )
   serverServiceConfig = ConfigMount.mount(
      em, 'controller/service/config',
      'Controller::ServiceConfigDir', 'w' )
   clientServiceConfig = ConfigMount.mount(
      em, 'mgmt/controller/service/config',
      'Controller::ServiceConfigDir', 'w' )
   overrideServiceConfig = ConfigMount.mount( 
      em, ctrlDebugDir + '/overrideService',
      'Controller::OverrideServiceConfigDir', 'w' )
   mgmtOverrideServiceConfig = ConfigMount.mount(
      em, mgmtCtrlDebugDir + '/overrideService',
      'Controller::OverrideServiceConfigDir', 'w' )
