#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.ControllerCli as ControllerCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.ControllerdbLib as ControllerdbLib

#--------------------------------------------------------------------------------
# [ no | default ] cvx [ CVX ]
#--------------------------------------------------------------------------------
cvxKeywordMatcher = CliMatcher.KeywordMatcher(
      'cvx', helpdesc='Configure controller services' )
cvxMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [ cfg.clusterName for cfg in
                     ControllerCli.clusterConfigDir.config.values() ],
      'CVX cluster name' )

class CvxCmd( CliCommand.CliCommandClass ):
   syntax = 'cvx [ CVX ]'
   data = {
      'cvx':
         CliCommand.Node(
            matcher=cvxKeywordMatcher,
            guard=ControllerdbLib.controllerOnSwitchGuard ),
      'CVX': cvxMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      clusterName = args.get( 'CVX' ) or ControllerCli.constants.clusterDefaultName
      clusterCfg = ControllerCli.clusterConfig()
      if clusterCfg.clusterName and clusterCfg.clusterName != clusterName:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "CVX already configured for cluster %s. "
                        "Multiple clusters not supported." % clusterCfg.clusterName )
         return
      childMode = mode.childMode(
            ControllerCli.CvxConfigMode, clusterName=clusterName )
      mode.session_.gotoChildMode( childMode )

BasicCliModes.GlobalConfigMode.addCommandClass( CvxCmd )

class NoOrDefaultCvxCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'cvx [ CVX ]'
   data = {
      'cvx': cvxKeywordMatcher,
      'CVX': cvxMatcher,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clusterName = args.get( 'CVX' )
      clusterCfg = ControllerCli.clusterConfig()
      if clusterName and clusterCfg.clusterName != clusterName:
         # pylint: disable-next=consider-using-f-string
         mode.addWarning( "CVX cluster %s does not exist" % clusterName )
         return
      clusterCfg.clusterName = ""
      ControllerCli.controllerdbConfig.primaryClusterName = ""
      clusterCfg.enabled = False
      ControllerCli.controllerdbConfig.enabled = False
      clusterCfg.peerConfig.clear()
      clusterCfg.peerTimeout = ControllerCli.peerTimeoutType.defaultVal
      clusterCfg.port = ControllerCli.constants.controllerdbDefaultPort
      oobCfg = clusterCfg.oobConfig
      oobCfg.heartbeatConfig.punchInterval = \
            ControllerCli.heartbeatIntervalType.defaultVal
      oobCfg.heartbeatConfig.timeout = ControllerCli.heartbeatTimeoutType.defaultVal
      oobCfg.sslProfileConfig.sslProfileName = ""
      oobCfg.connectionPreserve = False
      clusterCfg.sourceIntf = ""

      # Run any registered callbacks.
      ControllerCli.runNoCvxCallbacks( mode )

BasicCliModes.GlobalConfigMode.addCommandClass( NoOrDefaultCvxCmd )
