#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import CliToken.Platform
import ShowCommand
import Tac
import Tracing
from CliCommand import hiddenKeyword
from CliModel import Model, Str, Submodel
from CpuFabricCliSandHandler import Sand
from CpuFabricCliStrataHandler import Strata
import CliPlugin.TechSupportCli
import kDeserialiser
__defaultTraceHandle__ = Tracing.Handle( 'CpuFabricCli' )

#--------------------------------------------------------------------------------
# show platform pkt [ fancy ]
#--------------------------------------------------------------------------------

def fabDump():
   try:
      return Tac.run( [ '/usr/bin/fab', 'dump' ], asRoot=True, stdout=Tac.CAPTURE,
              stderr=Tac.CAPTURE )
   except IOError:
      return ""
   except Tac.SystemCommandError as e:
      # This exception is raised if dma driver is not loaded
      return "unsupported command: error code %s" % e.error


class RootC( Model ):
   __public__ = False
   sand = Submodel( valueType=Sand, help="Sand fab information", optional=True )
   strata = Submodel( valueType=Strata, help="Strata fab information",
                      optional=True )
   _raw = Str( help="Old fab dump output" )

   def render( self ):
      if self._raw:
         print( self._raw )
      elif self.sand:
         self.sand.render()
      elif self.strata:
         self.strata.render()
      else:
         print( "Not supported" )


class PlatformPktCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform pkt [ fancy ]'
   data = {
      'platform': CliToken.Platform.platformMatcherForShow,
      'pkt': 'CPU packet info',
      'fancy': hiddenKeyword( 'fancy' ) # Display tabular output of CPU packet info
   }
   privileged = True
   cliModel = RootC

   @staticmethod
   def handler( mode, args ):
      try:
         d = kDeserialiser.Debug()
      except IOError:
         r = RootC()
      else:
         r = RootC.fromDict( d.data, ignoreMissingAttrs=True )
      # pylint:disable=protected-access
      r._raw = None if 'fancy' in args else fabDump()
      return r


BasicCli.addShowCommandClass( PlatformPktCmd )

# Register CpuFabric Cli command with 'show tech support'.
# the older timestamp of the original show platform trident pkt
# command is retained, as this show command does not do any 
# dumping of a more recent released feature.
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2011-05-01 14:00:00',
   cmds=[ 'show platform pkt' ] )
