#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=too-many-nested-blocks

import struct
import socket
import threading
import Ark
import Tac
import BasicCli
import CliMatcher
import ShowCommand
import SmashLazyMount
import LazyMount
import Toggles.gatedToggleLib
from natsort import natsorted
from collections import defaultdict
from CliPlugin.CspfShowDbModel import ( TeDbModel,
                                        ReservablePriorityBandwidth,
                                        InterfaceAddress,
                                        SharedRiskLinkGroup,
                                        NgbTeInfoModel,
                                        TeLinksModel,
                                        TeSourceProtocolDbModel,
                                        RouterInfoModel,
                                        TeRouterDbModel,
                                        TeAfDbModel,
                                        TeVrfDbModel,
                                        IgpInstanceInfoModel,
                                        IgpModel,
                                        TeInterfaceIpModel,
                                        InterfaceIpInfoModel,
                                    )
from CliPlugin.TeCli import getSrlgIdToNameMap, adminGroupDictToDecimalList
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from IpLibConsts import DEFAULT_VRF
from Toggles import TeToggleLib

matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
                            helpdesc='Traffic Engineering related information' )
AddressFamily = Tac.Type( "Arnet::AddressFamily" )
readerInfo = SmashLazyMount.mountInfo( 'reader' )
mountLock = threading.RLock()
topoDbExport = None
em = None

def splitSmashPath( topoDbPath ):
   if not isinstance( topoDbPath, str ):
      topoDbPath = topoDbPath.fullName
   tSplit = topoDbPath.split( '/' )
   if 'isis' in topoDbPath:
      proto = 'isis'
      igpId = int( tSplit[ -2 ] )
      instId = int( tSplit[ -3 ] )
      addrFamily = tSplit[ -4 ]
   else:
      assert 'ospf' in topoDbPath
      proto = 'ospf'
      # Convert area ID in integer to an IP address
      igpId = socket.inet_ntoa( struct.pack( '!I', int( tSplit[ -2 ] ) ) )
      addrFamily = tSplit[ -4 ]
      instId = int( tSplit[ -3 ] )
   return proto, addrFamily, instId, igpId

def setSourceDbModel( proto, instId, igpId, teRouter ):
   sourceDbModel = TeSourceProtocolDbModel()
   igpInstModel = teRouter.igps[ proto ].instances[ instId ]
   if proto == 'isis':
      igpInstModel.levels[ igpId ] = sourceDbModel
   else:
      igpInstModel.areas[ igpId ] = sourceDbModel
   return sourceDbModel

def getSourceDbModel( proto, instId, igpId, teRouter ):
   igpInstModel = teRouter.igps[ proto ].instances[ instId ]
   if proto == 'isis':
      sourceDbModel = igpInstModel.levels[ igpId ]
   else:
      sourceDbModel = igpInstModel.areas[ igpId ]
   return sourceDbModel

def setLanNeighborId( proto, ngbKey ):
   neighborId = ngbKey.networkId.drRouterId.stringValue
   if proto == 'isis':
      neighborId = neighborId + "." + \
            str( format( ngbKey.networkId.drNetworkId, 'x' ) )
   return neighborId

@Ark.synchronized( mountLock )
def showTeDatabase( mode, args ):
   topoDbList = []
   teRouterId = args.get( 'IP', None )

   if Toggles.gatedToggleLib.toggleOspfTopoDbExportEnabled():
      # Although the code is under toggle OspfTopoDbExport the
      # topoDbExport contains both ISIS and OSPF topoDb paths
      for key in topoDbExport:
         path = key.replace( '-', '/' )
         topoDb = SmashLazyMount.mount( em, path, 'Routing::Topology::TopoDb',
                                        readerInfo, autoUnmount=True,
                                        unmountTimeout=600 )
         topoDbList.append( topoDb )

   # List of TE Router IDs_TopoDbs seen so far
   # teTopoDbSeen = [ TE1_IPv4L1, TE1_IPv4L2, TE2_IPv4L2, ..]
   teTopoDbSeen = []
   nonTeRouterId = '0.0.0.0'
   teDatabase = TeDbModel()
   vrfModel = TeVrfDbModel()
   teDatabase.vrfs[ DEFAULT_VRF ] = vrfModel
   # SRLG Id to name Mapping.
   srlgIdToNameMap = getSrlgIdToNameMap()
   for topoDb in topoDbList:
      proto, addrFamily, instId, igpId = splitSmashPath( topoDb )

      # skip non default ISIS instance topodb
      if proto == 'isis' and instId != 0:
         continue
      for routerId, router in topoDb.router.items():
         if ( teRouterId is None or
               router.teRouterId.stringValue == teRouterId.stringValue ):
            if addrFamily == 'ip':
               if vrfModel.ipv4 is None:
                  vrfModel.ipv4 = TeAfDbModel()
               af = AddressFamily.ipv4
               afModel = vrfModel.ipv4
            elif addrFamily == 'ip6':
               if vrfModel.ipv6 is None:
                  vrfModel.ipv6 = TeAfDbModel()
               af = AddressFamily.ipv6
               afModel = vrfModel.ipv6
            # Check if the TE-router ID has already been seen
            existingTeRouter = afModel.teRouterIds.get( router.teRouterId.stringValue
                                                      ,None )
            if existingTeRouter != None: # pylint: disable=singleton-comparison
               teRouter = existingTeRouter
            # Omit the inclusion of the exported non-TE nodes
            # A non-TE node has 0.0.0.0 as its router ID
            elif router.teRouterId.stringValue == nonTeRouterId:
               continue
            else:
               teRouter = TeRouterDbModel()
               afModel.teRouterIds[ router.teRouterId.stringValue ] = teRouter
            # TeSourceProtocolDbModel() is created when :
            # 1. A TE router ID has not been seen before
            # 2. A TE router ID has been seen but belongs to a different topology
            # The model is not created when :
            # 1. Two differnt routers in the same topology have the same TE router ID
            teTopoDb = router.teRouterId.stringValue + topoDb.fullName
            if teTopoDb not in teTopoDbSeen:
               if proto not in teRouter.igps:
                  igpInstanceInfo = IgpInstanceInfoModel()
                  igpInstanceInfo.instances[ instId ] = IgpModel()
                  teRouter.igps[ proto ] = igpInstanceInfo

               sourceDbModel = setSourceDbModel( proto, instId, igpId, teRouter )
               teTopoDbSeen.append( teTopoDb )
            else:
               sourceDbModel = getSourceDbModel( proto, instId, igpId, teRouter )

            rtrInfoModel = RouterInfoModel()
            rtrInfoModel.numLinks = 0
            sourceDbModel.routers[ routerId.stringValue ] = rtrInfoModel
            for ngbKey in router.neighbor.values():
               ngb = topoDb.neighbor.get( ngbKey )
               if ngb is None:
                  continue
               # Omit the inclusion of non-TE links
               # A non-TE link is one which does not have TE enabled on its interface
               # Thus its corresponding local interface address sub-TLV remains empty
               if not ngb.interfaceAddr:
                  continue
               ngbTeInfo = NgbTeInfoModel()
               rtrInfoModel.numLinks += 1
               # Create a TE links model for either P2P or LAN
               # If it is a P2P link
               if ngbKey.neighborType == 'neighborTypeRouter':
                  if rtrInfoModel.p2p != None: # pylint: disable=singleton-comparison
                     rtrInfoModel.p2p.links.append( ngbTeInfo )
                  else:
                     linkType = TeLinksModel()
                     linkType.links.append( ngbTeInfo )
                     rtrInfoModel.p2p = linkType
                  ngbTeInfo.neighborId = ngbKey.neighborId.routerId.stringValue
               # If it is a LAN link
               elif ngbKey.neighborType == 'neighborTypeNetwork':
                  if rtrInfoModel.lan != None: # pylint: disable=singleton-comparison
                     rtrInfoModel.lan.links.append( ngbTeInfo )
                  else:
                     linkType = TeLinksModel()
                     linkType.links.append( ngbTeInfo )
                     rtrInfoModel.lan = linkType
                  ngbTeInfo.neighborId = setLanNeighborId( proto, ngbKey )

               # Fill in the neighbor TE information
               if ngb.adminGroupPresent:
                  if ngb.adminGroup[ 0 ]:
                     ngbTeInfo.administrativeGroup = ngb.adminGroup[ 0 ]
                  if TeToggleLib.toggleExtendedAdminGroupEnabled():
                     ngbTeInfo.administrativeGroupsExtended = \
                           adminGroupDictToDecimalList( ngb.adminGroup )
                  else:
                     ngbTeInfo.administrativeGroupsExtended = None
               else:
                  ngbTeInfo.administrativeGroupsExtended = None
               if ngb.teMetricPresent:
                  ngbTeInfo.metric = ngb.teMetric
               for addr in ngb.interfaceAddr.values():
                  intfAddr = InterfaceAddress()
                  if af == AddressFamily.ipv4:
                     intfAddr.ipv4Address = addr.stringValue
                  elif af == 'ipv6':
                     intfAddr.ipv6Address = addr.stringValue
                  ngbTeInfo.interfaceAddresses.append( intfAddr )
               for addr in ngb.neighborAddr.values():
                  intfAddr = InterfaceAddress()
                  if af == AddressFamily.ipv4:
                     intfAddr.ipv4Address = addr.stringValue
                  elif af == AddressFamily.ipv6:
                     intfAddr.ipv6Address = addr.stringValue
                  ngbTeInfo.neighborAddresses.append( intfAddr )
               # BW values in Smash are in Bytes per second and are stored as float.
               # Bytes per second are converted to bits per second for show commands.
               # This consversion from float to int would not cause an overflow issue
               # since Python will implictly convert the value to long and long type
               # can have unlimited length.
               if ngb.maxBwPresent:
                  ngbTeInfo.maxLinkBw = int( ngb.maxBw * 8 )
               if ngb.maxReservableBwPresent:
                  ngbTeInfo.maxReservableBw = int( ngb.maxReservableBw * 8 )
               if ngb.unreservedBwPresent:
                  NUM_BW_CLASSES = 8
                  unreservedBw = ReservablePriorityBandwidth()
                  ngbTeInfo.unreservedBw = unreservedBw
                  unreservedBwValues = list( ngb.legacyTeAttr.unreservedBw.values() )
                  for i in range( NUM_BW_CLASSES ):
                     val = unreservedBwValues[ i ]
                     if val is not None:
                        setattr(
                           unreservedBw, "priority" + str( i ), int( val * 8 ) )
               if ngb.srlgCount != 0:
                  for srlgId in ngb.legacyTeAttr.srlg.values():
                     srlg = SharedRiskLinkGroup()
                     srlg.groupId = srlgId
                     srlg.groupName = srlgIdToNameMap.get( srlgId, None )
                     ngbTeInfo.sharedRiskLinkGroups.append( srlg )

   return teDatabase

@Ark.synchronized( mountLock )
def showTeDuplicateInterface( mode, args ):
   topoDbList = []

   if Toggles.gatedToggleLib.toggleOspfTopoDbExportEnabled():
      # Although the code is under toggle OspfTopoDbExport the
      # topoDbExport contains both ISIS and OSPF topoDb paths
      for key in topoDbExport:
         path = key.replace( '-', '/' )
         topoDb = SmashLazyMount.mount( em, path, 'Routing::Topology::TopoDb',
                                        readerInfo, autoUnmount=True,
                                        unmountTimeout=600 )
         topoDbList.append( topoDb )

   teInterfaceIpModel = TeInterfaceIpModel()
   teIntfToRouters = defaultdict( set )
   # Below nested loop iterates over all the interface addresses in the topologies,
   # and creates a Dictionary 'teIntfRouters' where key is interfaceAddress and
   # and value is set of teRouterId's where interfaceAddress is present.
   for topoDb in topoDbList:
      for ngb in topoDb.neighbor.values():
         if ngb is None or not ngb.interfaceAddr:
            continue
         for addr in ngb.interfaceAddr.values():
            routerId = ngb.neighborKey.neighborOf
            teRouterId = topoDb.router[ routerId ].teRouterId
            teIntfToRouters[ addr ].add( teRouterId )

   # Below loops iterates over dictionary 'teIntfRouters' and inserts duplicate
   # interfaces in Model classes InterfaceIpInfoModel & TeInterfaceIpModel. Also,
   # interfaceIp & teRouterIds are inserted in sorted order in Model classes.
   for intfAddr in natsorted( teIntfToRouters ):
      if len( teIntfToRouters[ intfAddr ] ) <= 1:
         continue
      intfIpInfo = InterfaceIpInfoModel()
      intfIpInfo.teRouterIds = natsorted( teIntfToRouters[ intfAddr ] )
      teInterfaceIpModel.intfIps[ intfAddr ] = intfIpInfo

   return teInterfaceIpModel

#--------------------------------------------------------------------------------
# show traffic-engineering database
#--------------------------------------------------------------------------------
class TrafficEngineeringDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering database'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'database' : 'Show TE database information',
   }
   handler = showTeDatabase
   cliModel = TeDbModel

BasicCli.addShowCommandClass( TrafficEngineeringDatabaseCmd )

#--------------------------------------------------------------------------------
# show traffic-engineering database for particular router
#--------------------------------------------------------------------------------
class TrafficEngineeringDatabaseCmdForRouter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering database router-id IP'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'database' : 'Show TE database information',
      'router-id' : 'router ID',
      'IP' : IpGenAddrMatcher( helpdesc='Router IP address' ),
   }
   handler = showTeDatabase
   cliModel = TeDbModel

BasicCli.addShowCommandClass( TrafficEngineeringDatabaseCmdForRouter )

# --------------------------------------------------------------------------------
# show traffic-engineering database duplicate interface ip's
# --------------------------------------------------------------------------------
class TrafficEngineeringDatabaseCmdForDuplicateInterfaceIP(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering database duplicate interface-address'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'database' : 'Show TE database information',
      'duplicate' : 'Show duplicate objects',
      'interface-address' : 'Show duplicate interface address',
   }
   handler = showTeDuplicateInterface
   cliModel = TeInterfaceIpModel

BasicCli.addShowCommandClass( TrafficEngineeringDatabaseCmdForDuplicateInterfaceIP )

def Plugin( entityManager ):
   global topoDbExport
   global em

   if Toggles.gatedToggleLib.toggleOspfTopoDbExportEnabled():
      topoDbExport = LazyMount.mount( entityManager,
                                      'routing/topoDbExport', 'Tac::Dir',
                                      'ri' )
      em = entityManager
