#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import ShowCommand
import CliPlugin.CspfShowPathCli
from CliPlugin.RoutingIsisCli import getHostName, isisKw
from CliPlugin.IsisCliModels import getRouterId
from CliPlugin.CspfShowFlexAlgoPathModel import ( FlexAlgoPathModel,
                                                  FlexAlgoPathVrfModel,
                                                  FlexAlgoPathAfModel,
                                                  FlexAlgoPathTopoIdModel,
                                                  FlexAlgoPathDestModel,
                                                  FlexAlgoPathEntryModel,
                                                  FlexAlgoViaModel,
                                                  FlexAlgoPathEntryDetailModel,
                                                  FlexAlgoPathConstraintModel, )
from CliPlugin.CspfShowPathCli import ( AddressFamily,
                                        getClientConfigStatus,
                                        getSrSmashStatus,
                                        dstIdMatcher,
                                        TopoId,
                                        matcherDetail,
                                        populateAdminGroupModel )
from CliPlugin.TeCli import getSrlgIdToNameMap
from TypeFuture import TacLazyType
import Tac

FlexAlgoConstraint = TacLazyType( "Smash::SegmentRouting::FlexAlgoConstraint" )
LevelAlgoInstNumKey = TacLazyType( "Smash::SegmentRouting::LevelAlgoInstNumKey" )
MetricType = TacLazyType( "Cspf::MetricType" )
TopoDbRouteProto = TacLazyType( "Cspf::TopoDbRouteProto" )

def getFlexAlgoDetailModel( pathConfig, pathConstraint, pathStatus, metric ):
   detailModel = FlexAlgoPathEntryDetailModel()
   detailModel.pathId = pathConfig.key.id
   constraintModel = FlexAlgoPathConstraintModel()

   constraintModel.metricType = 'igp'
   if pathConstraint.metricType == MetricType.minDelayMetric:
      constraintModel.metricType = 'minDelay'
   if pathConstraint.metricType == MetricType.teMetric:
      constraintModel.metricType = 'te'

   if pathConstraint.excludeExplicitSrlg:
      # pylint: disable-msg=protected-access
      constraintModel._srlgIdToNameMap = getSrlgIdToNameMap()
      for s in pathConstraint.excludeExplicitSrlg.values():
         constraintModel.excludeSrlg.append( s )
   else:
      constraintModel.excludeSrlg = None

   populateAdminGroupModel( constraintModel, pathConstraint )

   detailModel.constraint = constraintModel
   detailModel.refreshReqSeq = pathConfig.refreshReqSeq
   detailModel.refreshRespSeq = pathStatus.refreshRespSeq
   detailModel.changeCount = pathStatus.changeCount
   detailModel.lastUpdatedTime = Tac.utcNow() - (
      Tac.now() - pathStatus.lastUpdatedTime )
   detailModel.metric = metric

   return detailModel

def showClientFlexAlgoPathInfo( mode, args ):
   pathModel = FlexAlgoPathModel()

   reqDstId = args.get( "DST" )
   detail = "detail" in args
   pathModel._detailCmd = detail # pylint: disable-msg=protected-access
   clientName = 'flexAlgoIsis'
   clientDir = CliPlugin.CspfShowPathCli.clientDir
   flexAlgoConfig = CliPlugin.CspfShowPathCli.flexAlgoConfig

   client = clientDir.entityPtr.get( clientName )
   if not client or clientName not in clientDir.entryState:
      return pathModel

   for vrf in client.vrf:
      vrfModel = FlexAlgoPathVrfModel()
      pathModel.vrfs[ vrf ] = vrfModel
      for af in client.addressFamily:
         afModel = FlexAlgoPathAfModel()
         if af == AddressFamily.ipv4:
            vrfModel.v4Info = afModel
         else:
            vrfModel.v6Info = afModel

         topoIdModelL1 = FlexAlgoPathTopoIdModel()
         topoIdModelL2 = FlexAlgoPathTopoIdModel()
         addL1Model = addL2Model = False

         ( config, status ) = getClientConfigStatus( clientName, vrf, af )
         for key, pathStatus in status.pathStatus.items():
            pathConfig = config.pathConfig.get( key )
            if not pathConfig:
               continue

            dstId = key.dst.stringValue
            pathConstraint = pathConfig.constraint
            topoId = pathConstraint.topoId

            dstHostname = getHostName( getRouterId( dstId, topoId.instance ), vrf )
            if reqDstId and reqDstId not in ( dstId, dstHostname ):
               continue

            if pathConstraint.topoId.protoId == 1:
               topoIdModel = topoIdModelL1
               addL1Model = True
            else:
               topoIdModel = topoIdModelL2
               addL2Model = True

            if dstId not in topoIdModel.destinations:
               dstModel = FlexAlgoPathDestModel()
               topoIdModel.destinations[ dstId ] = dstModel
               dstModel.hostname = dstHostname
            dstModel = topoIdModel.destinations[ dstId ]
            pathEntryModel = FlexAlgoPathEntryModel()
            algoId = pathConstraint.algoId
            dstModel.paths[ algoId ] = pathEntryModel
            if algoId in flexAlgoConfig.definition:
               pathEntryModel.algoName = flexAlgoConfig.flexAlgoName( algoId )

            viaInfoIdx = pathStatus.viaInfoIndex
            metric = None
            viaInfo = status.viaInfo.get( viaInfoIdx )
            if viaInfo:
               if viaInfo.via:
                  metric = viaInfo.metric
               for via in viaInfo.via.values():
                  viaModel = FlexAlgoViaModel()
                  viaModel.nexthop = via.ipGenAddr
                  viaModel.intf = via.intfId
                  pathEntryModel.vias.append( viaModel )

            if detail:
               igpString = "isis" if pathConstraint.topoId.igpSource ==\
                  TopoDbRouteProto.topoDbRouteProtoIsis else "ospf"
               srSmashStatus = getSrSmashStatus( igpString, vrf )
               faConstKey = LevelAlgoInstNumKey( pathConstraint.topoId.protoId,
                                                 algoId,
                                                 pathConstraint.topoId.instance )
               faConst = srSmashStatus.flexAlgoConstraint.get(
                  faConstKey, FlexAlgoConstraint() )
               pathEntryModel.details = getFlexAlgoDetailModel(
                  pathConfig, faConst.pathConstraint, pathStatus, metric )

         if addL1Model:
            afModel.topologies[ TopoId.isisL1.protoId ] = topoIdModelL1
         if addL2Model:
            afModel.topologies[ TopoId.isisL2.protoId ] = topoIdModelL2

   return pathModel

#--------------------------------------------------------------------------------
# show isis flex-algo path [ DST ] [ detail ]
#--------------------------------------------------------------------------------
class IsisFlexAlgoPathCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show isis flex-algo path [ DST ] [ detail ]'
   data = {
      'isis' : isisKw,
      'flex-algo' : 'Flexible algorithm',
      'path' : 'Flex algo path to destination',
      'DST' : dstIdMatcher,
      'detail' : matcherDetail
   }
   handler = showClientFlexAlgoPathInfo
   cliModel = FlexAlgoPathModel

BasicCli.addShowCommandClass( IsisFlexAlgoPathCmd )
