#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import threading

import Ark
import BasicCli
import CliMatcher
from CliPlugin.CspfShowIntfModel import ( TeIntfListModel, TeIntfModel, TeDelay,
                                       TeMaxResvBw, IntfSrlg, AdjSid, NodeSid,
                                       TeInterfaceBwStatsModel, TeInterfaceBwModel,
                                       AdvertisedBwOfPriorityClasses,
                                       BandwidthAdByIgp )
from CliPlugin import IntfCli
from CliPlugin.TeCli import adminGroupDictToDecimalList
import LazyMount
import ShowCommand
import SmashLazyMount
import Arnet
import Tac
from TypeFuture import TacLazyType
from Toggles import TeToggleLib

isisConfig = None
srConfig = None
teConfig = None
entityMgr = None
isisStatusDir = None

MonitorResultKey = Tac.Type( "Twamp::Light::MonitorResultKey" )
SRConstants = TacLazyType( "Routing::SegmentRoutingCli::Constants" )
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
                            helpdesc='Traffic Engineering related information' )
mountLock = threading.RLock()

def _getIsisIntfConfig( intfName ):
   return isisConfig.intfConfig.get( intfName )

def _getSrIntfConfig( intfName ):
   return srConfig.intfConfig.get( intfName )

def _getTeIntfConfig( intfName ):
   return teConfig.intfConfig.get( intfName )

def _getTeBwUpdateThreshold():
   if teConfig.teBwFloodThreshold:
      return teConfig.teBwFloodThreshold
   return teConfig.teBwFloodThresholdDefault

def _getIsisIntfVrfName( instName ):
   instConfKey = Tac.Type( "Routing::Isis::InstanceName" )( instName )
   instConf = isisConfig.instanceConfig.get( instConfKey )
   if instConf is None:
      return None
   return instConf.vrfName

def _getIsisIntfInstanceName( intfName ):
   intfConf = _getIsisIntfConfig( intfName )
   instName = None
   if intfConf:
      instName = intfConf.instanceName
   return instName

def _getIsisInstAndIntfInfo( instName, intfName, vrfName ):
   isisStatus = isisStatusDir.get( vrfName )
   if not isisStatus:
      return [ None, None ]
   instNameKey = Tac.Type( "Routing::Isis::InstanceName" )( instName )

   # Retrieve instance id information
   instStatus = isisStatus.instanceStatus.get( instNameKey )
   instanceId = None
   if instStatus:
      instanceId = instStatus.instanceId

   # Retrieve the level information.
   instKey = Tac.Type( "Routing::Isis::InstanceKey" )( intfName, instName )
   intfStatus = isisStatus.intfStatus.get( instKey )
   activeCircuitLev = None
   if intfStatus:
      activeCircuitLev = intfStatus.activeCircuitType

   return [ instanceId, activeCircuitLev ]

def _getTeIntfBwDetails( intfName, teIntf ):
   # Get the IGP Instance Name from the interfcae name
   # Using the instance name get the VRF name associated with the IGP
   # instnace and fetch information from interface BW status maintained
   # by the BW Manager.
   # To support with OSPF we need to fetch the VRF information either
   # from OSPF configuration path or fetch from l3/intf/status/intfStatus/
   # that will be the independent of IGPs
   vrfName = None
   isisInstName = _getIsisIntfInstanceName( intfName )
   if isisInstName is None:
      return -1

   vrfName = _getIsisIntfVrfName( isisInstName )
   if vrfName is None:
      return -1

   tmp = SmashLazyMount.mountInfo( 'reader' )
   smashPath = 'te/bwm/status/vrf/' + vrfName
   bwStatus = SmashLazyMount.mount( entityMgr, smashPath,
                                 'TrafficEngineering::BandwidthStatus', tmp,
                                 autoUnmount=True, unmountTimeout=600 )
   if bwStatus is None:
      return -1
   ifaceBw = bwStatus.interfaceBandwidth.get( Arnet.IntfId( intfName ) )
   if ifaceBw is None:
      return -1

   # Bandwidth values are stored in number of bytes per second.
   # Convert the bandwidth value to bits per second so that we
   # can display in bits per second format (Gbps, Mbps, Kbps)
   teIntf.maxBandwidth = int( ifaceBw.maxBandwidth * 8 )
   teIntf.maxReservableBandwidth = int( ifaceBw.maxReservableBandwidth * 8 )

   for className, unResvbw in ifaceBw.unreservedBandwidth.items():
      teIntf.unreservedBandwidths[ className ] = int( unResvbw * 8 )

   return 0

def _getTeIsisIntfAdvBwByLevel( instanceId, vrfName,
                              intfName, level ):
   # ISIS Path for advertised TE BW te/isis/instnce_id/
   igpAdvBw = AdvertisedBwOfPriorityClasses()

   # Mount TE BW path
   teIntfBwPath = 'te/isis/'
   teIntfBwPath += f"{instanceId}/"
   teIntfBwPath += f"{level}/"
   teIntfBwPath += vrfName

   tmp = SmashLazyMount.mountInfo( 'reader' )
   isisAdvTeBw = SmashLazyMount.mount( entityMgr, teIntfBwPath,
                              'TrafficEngineering::BandwidthStatus', tmp,
                              autoUnmount=True, unmountTimeout=600 )
   if isisAdvTeBw is None:
      return None

   intfBW = isisAdvTeBw.interfaceBandwidth.get( Arnet.IntfId( intfName ) )
   if intfBW is None:
      return None

   # Bandwidth values are stored in number of bytes per second.
   # Convert the bandwidth value to bits per second so that we
   # can display in bits per second format (Gbps, Mbps, Kbps)
   for className, advUnResvbw in intfBW.unreservedBandwidth.items():
      igpAdvBw.bandwidths[ className ] = int( advUnResvbw * 8 )

   return igpAdvBw

def _getTeIsisIntfAdvBwDetails( intfName, teIntf, isisInstName ):
   vrfName = _getIsisIntfVrfName( isisInstName )
   instanceId, cirLevel = _getIsisInstAndIntfInfo( isisInstName, intfName,
                                                   vrfName )

   # Based on the ISIS level active on the the link retrieve the information
   LevelTypes = Tac.Type( "Routing::Isis::Level" )
   advBwByIsis = BandwidthAdByIgp()
   advBwByIsis.instanceName = isisInstName
   # pylint: disable-msg=W0212
   advBwByIsis._igpname = "IS-IS"

   if cirLevel in ( LevelTypes.level1, LevelTypes.level1_2 ):
      igpAdvBw = _getTeIsisIntfAdvBwByLevel( instanceId,
                                          vrfName, intfName, 1 )
      if igpAdvBw is None:
         return -1
      advBwByIsis.bandwidthsInEachLevelOrArea[ 1 ] = igpAdvBw

   if cirLevel in ( LevelTypes.level2, LevelTypes.level1_2 ):
      igpAdvBw = _getTeIsisIntfAdvBwByLevel( instanceId,
                                          vrfName, intfName, 2 )
      if igpAdvBw is None:
         return -1
      advBwByIsis.bandwidthsInEachLevelOrArea[ 2 ] = igpAdvBw

   teIntf.advertisedUnreservedBandwidths[ 'IS-IS' ] = advBwByIsis
   return 0

def _getTeIntfAdvBwDetails( intfName, teIntf ):
   isisInstName = _getIsisIntfInstanceName( intfName )
   if isisInstName is not None:
      ret = _getTeIsisIntfAdvBwDetails( intfName, teIntf, isisInstName )
      if ret < 0:
         return -1
   return 0

def _mountTwampMonitorTargetsAndResults( vrf_igp_inst ):
   mountInfo = SmashLazyMount.mountInfo( 'reader' )
   targetSmashPath = f'monitor/twamp/twampLight/input/{vrf_igp_inst}'
   resultSmashPath = f'monitor/twamp/twampLight/result/{vrf_igp_inst}'
   clientMonitorTargets = SmashLazyMount.mount( entityMgr, targetSmashPath,
                                                'Twamp::Light::ClientMonitorTargets',
                                                mountInfo, autoUnmount=True,
                                                unmountTimeout=600 )
   result = SmashLazyMount.mount( entityMgr, resultSmashPath,
                                  'Twamp::Light::Result', mountInfo,
                                  autoUnmount=True, unmountTimeout=600 )
   return ( clientMonitorTargets, result )

# Helper method that returns all the dynamic min-delay values configured on
# inerfaces
def _getDynamicMinDelayValues( ifList ):
   visitedMonitorTargets = {}
   dynaicMinDelayDict = {}
   for intf in ifList:
      intfName = intf.name
      instName = _getIsisIntfInstanceName( intfName )
      if not instName:
         continue
      vrfName = _getIsisIntfVrfName( instName )
      instanceId, _ = _getIsisInstAndIntfInfo( instName, intfName, vrfName )
      if instanceId is None:
         continue

      vrf_igp_inst = f'{vrfName}-isis-{instanceId}'

      # Skip the already visited mount path
      if vrf_igp_inst in visitedMonitorTargets:
         continue

      visitedMonitorTargets[ vrf_igp_inst ] = True

      # Mount ClientMonitorTargets and Result and then process each entry in Result
      clientMonitorTargets, result = _mountTwampMonitorTargetsAndResults(
         vrf_igp_inst )
      if ( not clientMonitorTargets ) or ( not result ):
         continue

      # Save dynamic min-delay details after iterating each Result collection entries
      for key, mt in clientMonitorTargets.monitorTarget.items():
         monitorResultKey = MonitorResultKey( key.address )
         mr = result.monitorResult.get( monitorResultKey, None )
         if not mr or mt.senderId <= 0:
            continue
         if mr.valid:
            dynaicMinDelayDict[ mt.intfId ] = {
               'dynamicDelayType' : 'measured',
               'value' : mr.oneWayMinDelay,
               'units' : 'microseconds'
            }
         else:
            teIntfConfig = _getTeIntfConfig( mt.intfId )
            if not teIntfConfig:
               continue
            dynaicMinDelayDict[ mt.intfId ] = {
               'dynamicDelayType' : 'fallback',
               'value' : teIntfConfig.dynMinDelayFallback,
               'units' : teIntfConfig.dynMinDelayFallbackUnit
            }
   return dynaicMinDelayDict

@Ark.synchronized( mountLock )
def showTeIntf( mode, args ):
   intf = args.get( 'INTF' )
   ifList = IntfCli.Intf.getAll( mode, intf=intf )
   intfList = TeIntfListModel()
   single = len( ifList ) == 1
   teIfList = []
   dynaicMinDelayDict = _getDynamicMinDelayValues( ifList )

   for intf in ifList:
      teIntf = TeIntfModel()
      teIntf.intf = intf.name
      teIntf.maxResvBw = TeMaxResvBw()
      teIntf.maxResvBw.value = 0
      teIntf.maxResvBw.bwUnitType = 'percent'
      teIntf.adjSids = []
      teIntf.nodeSids = []
      teIntf.adminGroup = 0
      teIntf.adminGroupsExtended = []
      teIntf.delay = TeDelay()
      teIntf.delay.value = 0
      teIntf.delay.units = 'milliseconds'
      teIntf.delay.style = 'static'
      teIntf.srlg = []
      teIntf.metric = 0

      intfName = teIntf.intf.stringValue
      isisIntfConfig = _getIsisIntfConfig( intfName )
      srIntfConfig = _getSrIntfConfig( intfName )
      teIntfConfig = _getTeIntfConfig( intfName )
      ofInterest = False

      if isisIntfConfig:
         keys = ( list( isisIntfConfig.srSingleAdjacencySegment ) +
                  list( isisIntfConfig.srMultipleAdjacencySegment ) )
         for key in keys:
            adjSid = AdjSid()
            adjSid.index = key.sid.index
            adjSid.isValue = key.sid.isValue
            adjSid.isIpV6 = key.isIpV6
            ofInterest = True
            teIntf.adjSids.append( adjSid )

      if srIntfConfig:

         if srIntfConfig.srNodeSegmentIndex != SRConstants.srSidInvalid:
            ofInterest = True
            nodeSid = NodeSid()
            nodeSid.index = srIntfConfig.srNodeSegmentIndex
            nodeSid.isIpV6 = False
            nodeSid.flexAlgoName = ''
            teIntf.nodeSids.append( nodeSid )

         if srIntfConfig.srV6NodeSegmentIndex != SRConstants.srSidInvalid:
            ofInterest = True
            nodeSid = NodeSid()
            nodeSid.index = srIntfConfig.srV6NodeSegmentIndex
            nodeSid.isIpV6 = True
            nodeSid.flexAlgoName = ''
            teIntf.nodeSids.append( nodeSid )

         for ( sidList, isV6 ) in [ ( srIntfConfig.srV4NodeSegment, False ),
                                    ( srIntfConfig.srV6NodeSegment, True ) ]:
            for sid in sidList.values():
               nodeSid = NodeSid()
               nodeSid.index = sid.index
               nodeSid.isIpV6 = isV6
               nodeSid.flexAlgoName = sid.flexAlgoName
               teIntf.nodeSids.append( nodeSid )

      if teIntfConfig:

         if teIntfConfig.maxReservableBw:
            ofInterest = True
            teIntf.maxResvBw.value = teIntfConfig.maxReservableBw
            teIntf.maxResvBw.bwUnitType = teIntfConfig.maxReservableBwUnit

         if teIntfConfig.metric:
            ofInterest = True
            teIntf.metric = teIntfConfig.metric

         if teIntfConfig.adminGroup:
            ofInterest = True
            if 0 in teIntfConfig.adminGroup:
               teIntf.adminGroup = teIntfConfig.adminGroup[ 0 ]
            if TeToggleLib.toggleExtendedAdminGroupEnabled():
               teIntf.adminGroupsExtended = adminGroupDictToDecimalList(
                     teIntfConfig.adminGroup )

         if teIntfConfig.adminGroupName:
            teIntf.adminGroupNames = list( teIntfConfig.adminGroupName )
            ofInterest = True

         if teIntfConfig.srlgIdList:
            ofInterest = True
            for srlg in teIntfConfig.srlgIdList:
               teSrlg = IntfSrlg()
               teSrlg.gid = srlg
               teSrlg.name = ""
               teIntf.srlg.append( teSrlg )

         if teIntfConfig.srlgNameList:
            ofInterest = True
            for srlg in teIntfConfig.srlgNameList:
               teSrlg = IntfSrlg()
               teSrlg.name = srlg
               teSrlg.gid = 0
               teIntf.srlg.append( teSrlg )

         if teIntfConfig.minDelay:
            ofInterest = True
            teIntf.delay.value = teIntfConfig.minDelay
            teIntf.delay.units = teIntfConfig.minDelayUnit
            teIntf.delay.dynamicDelayType = None

         if intfName in dynaicMinDelayDict:
            ofInterest = True
            teIntf.delay.style = 'dynamic'
            teIntf.delay.value = dynaicMinDelayDict[ intfName ][ 'value' ]
            teIntf.delay.units = dynaicMinDelayDict[ intfName ][ 'units' ]
            teIntf.delay.dynamicDelayType = \
               dynaicMinDelayDict[ intfName ][ 'dynamicDelayType' ]

      if single or ofInterest:
         teIfList.append( teIntf )
   intfList.interfaces = teIfList
   return intfList

@Ark.synchronized( mountLock )
def showTeIntfBw( mode, args ):
   teIntfsBwStats = TeInterfaceBwStatsModel()
   teIntfsBwStats.threshold = _getTeBwUpdateThreshold()

   # Get statistics for each of the interface
   teIntfBwStats = {}

   intf = args.get( 'INTF' )
   if intf:
      if intf.name not in teConfig.intfConfig:
         return teIntfsBwStats

      teIntf = TeInterfaceBwModel()

      if _getTeIntfBwDetails( intf.name, teIntf ) < 0:
         return teIntfsBwStats
      if _getTeIntfAdvBwDetails( intf.name, teIntf ) < 0:
         return teIntfsBwStats

      teIntfBwStats[ intf.name ] = teIntf
   else:
      for iconfig in teConfig.intfConfig.values():
         teIntf = TeInterfaceBwModel()

         if _getTeIntfBwDetails( iconfig.name, teIntf ) < 0:
            # We will display information for other interfaces
            # those are available
            continue
         if _getTeIntfAdvBwDetails( iconfig.name, teIntf ) < 0:
            continue
         teIntfBwStats[ iconfig.name ] = teIntf

   teIntfsBwStats.interfaces = teIntfBwStats
   return teIntfsBwStats

#--------------------------------------------------------------------------------
# show traffic-engineering interfaces [<name>]
#--------------------------------------------------------------------------------
class TrafficEngineeringIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering interfaces [ INTF ]'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'interfaces' : 'Show TE interface information',
      'INTF' : IntfCli.Intf.matcher,
   }
   handler = showTeIntf
   cliModel = TeIntfListModel

BasicCli.addShowCommandClass( TrafficEngineeringIntfCmd )

# --------------------------------------------------------------------------------
# show traffic-engineering interfaces [<name>] bandwidth
# --------------------------------------------------------------------------------
class TrafficEngineeringIntfBwCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering interfaces [ INTF ] bandwidth'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'interfaces' : 'Show TE interface information',
      'INTF' : IntfCli.Intf.matcher,
      'bandwidth' : 'Show interface bandwidth statistics',
   }
   handler = showTeIntfBw
   cliModel = TeInterfaceBwStatsModel

BasicCli.addShowCommandClass( TrafficEngineeringIntfBwCmd )

def Plugin( entityManager ):
   global isisConfig
   global srConfig
   global teConfig
   global entityMgr
   global isisStatusDir

   isisConfig = LazyMount.mount( entityManager, 'routing/isis/config',
                                 'Routing::Isis::Config', 'ri' )
   srConfig = LazyMount.mount( entityManager, 'routing/sr/config',
                               'Routing::SegmentRoutingCli::Config', 'ri' )
   teConfig = LazyMount.mount( entityManager, 'te/config',
                               'TrafficEngineering::Config', 'ri' )
   isisStatusDir = LazyMount.mount( entityManager, 'routing/isis/status/',
                                    'Tac::Dir', 'ri' )
   entityMgr = entityManager
