# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# -----------------------------------------------------------------------------------
# This module implements a CLI command that provides the user with a Python shell in
# the context of the current CLI process.
# -----------------------------------------------------------------------------------
import sys
import code

import BasicCli
import CliCommand
import Tracing

t0 = Tracing.trace0

def handleExit( status=None ):
   raise SystemExit( status )

def doPythonShell( mode ):
   def readfunc( prompt ):
      return mode.session_.cliInput.simpleReadline( prompt )

   # Convenience function exported into the shell to facilitate the debugging
   # of cli commands executed under gdb control, like so:
   # mySwitch# python-shell
   # > > > pdbCli("show aliases")
   # (Pdb) break CliPlugin/CliCliModel.py:32
   # (Pdb) continue
   def pdbCli( cmd ):
      # Disable pagination: this prevents people from having to wonder where the
      # propmt has gone, where their typing goes and why there is no output...
      oldDisableAutoMore_ = mode.session.disableAutoMore_
      mode.session.disableAutoMore_ = True
      # For some reason, in spite of stdout being a terminal, flushes only
      # happen at \n, so reopen the file with 0 buffer size so that prompt is seen.
      # pylint: disable-next=import-outside-toplevel
      import pdb # let's not import before actually needed...
      sys.stdout.flush()
      pdb.runcall( mode.session_.runCmd, cmd )
      mode.session.disableAutoMore_ = oldDisableAutoMore_
   # Might want to use this api to get into the relevant mode first (python-shell
   # lives in exec mode).

   def runCli( cmd ):
      mode.session_.runCmd( cmd )

   # Make sure people don't exit the whole ConfigAgent by mistake.
   # There is only one good way to exit the python-debug: type control-d.
   # exit() would close the whole ConfigAgent, not just this thread: prevent it
   myExit = handleExit
   # quit() gets you back to the Cli prompt, but will later prevent you from exiting
   # the cli (operation on closed fd exception), so prevent that as well.
   myQuit = handleExit

   try:
      code.interact( banner=( 'Welcome to the Python shell.  '
                              'Press Ctrl-D to exit.' ),
                     readfunc=readfunc,
                     local={
                             'pdbCli': pdbCli,
                             'runCli': runCli,
                             'quit': myQuit,
                             'exit': myExit,
                           }
                   )
   except SystemExit as e:
      t0( "exit() from python shell with status:", e )

class PythonShellCmd( CliCommand.CliCommandClass ):
   syntax = "python-shell"
   data = { 'python-shell': 'Break into a Python shell' }

   @staticmethod
   def handler( mode, args ):
      doPythonShell( mode )

BasicCli.EnableMode.addCommandClass( PythonShellCmd )
