# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements decap groups configuration.
#-------------------------------------------------------------------------------

'''Configuration commands supported for Decap Groups'''
from __future__ import absolute_import, division, print_function
import Arnet
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
from CliMode.DecapGroup import DecapGroupBaseMode
import CliParser
import CliToken.Ip
import ConfigMount
import DecapGroupModel
import LazyMount
import ShowCommand
import SmashLazyMount
import Tac

# pkgdeps: library FlexCounters
FapId = Tac.Type( 'FlexCounters::FapId' )
AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
AddressType = Tac.Type( 'Tunnel::Decap::AddressType' )
MgmtIntfId = Tac.Type( 'Arnet::MgmtIntfId' )
# pkgdeps: library EthIntf
InternalIntfId = Tac.Type( 'Arnet::InternalIntfId' )
VrfName = Tac.Type( 'L3::VrfName' )

ipAddrZero = Tac.Value( 'Arnet::IpAddr' ).ipAddrZero

cliDgConfig = None
readOnlyCliDgConfig = None
dgDynInputDir = None
mergedDgConfig = None
allDgStatus = None
dgConfigMergeSm = None

pluginEntityManager = None
v6Error = "%s tunnels are not supported with IPv6"

# Support for DecapGroup counters.
countersClearConfig = None
decapGroupCounterTable = None
decapGroupSnapshotTable = None
ipStatus = None
ip6Status = None

def decapGroupSupportedGuard( mode, token ):
   if allDgStatus.decapSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def decapGroupIpGuard( mode, token ):
   if 'ip' in allDgStatus.decapIpSrcSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def decapGroupIntfGuard( mode, token ):
   if 'intf' in allDgStatus.decapIpSrcSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def decapGroupPayloadTypeSupportedGuard( mode, token ):
   # the enum value for 'ip' is 'ipvx' in PayloadType
   if token == 'ip':
      token = 'ipvx'
   if allDgStatus.decapGroupPayloadTypeSupported.get( token ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def decapTunnelTypeUdpSupportedGuard( mode, token ):
   if allDgStatus.decapTunnelTypeSupported.get( 'udp', False ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def decapTunnelUdpPortSupportedGuard( mode, token ):
   if allDgStatus.perDecapGroupUdpDestPortSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def decapTunnelGlobalUdpPortSupportedGuard( mode, token ):
   if allDgStatus.globalUdpDestPortSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def forwardingVrfSupportedGuard( mode, token ):
   if allDgStatus.forwardingVrfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def decapPrefixInTcamSupportedGuard( mode, token ):
   if allDgStatus.decapIpv4PrefixSupportedInTcam:
      return None
   else:
      return CliParser.guardNotThisPlatform

 
def decapIp6SupportedGuard( mode, token ):
   if 'udp' in allDgStatus.decapTunnelIp6DestinationSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform 

def maxIterForDecapIp():
   if allDgStatus is not None and allDgStatus.multiDecapIpSupported:
      return 0
   else:
      return 1

def decapGroupTunnelDedicatedCounterSupported():
   return allDgStatus is not None and \
           allDgStatus.decapGroupTunnelDedicatedCounterSupported

def decapTunnelIp6DestinationSupported( mode, tunnelType ):
   # bypass the check if this is startup mode
   if mode.session_.startupConfig() or \
      ( allDgStatus is not None and
        tunnelType in allDgStatus.decapTunnelIp6DestinationSupported ):
      return True

   return False

tacTunnelType = Tac.Type( 'Tunnel::Decap::TunnelType' )
def tunnelTypeGuard( mode, token ):
   # Check decap-group tunnel types supported by platform
   if allDgStatus.decapTunnelTypeSupported.get( token.lower() ):
      return None
   return CliParser.guardNotThisPlatform

def greGuard( mode, token ):
   if ( not greKeyToForwardingVrfMappingGuard( mode, token ) or
        not qosTcToMplsTcDecapGuard( mode, token ) ):
      return None
   return CliParser.guardNotThisPlatform

def greKeyToForwardingVrfMappingGuard( mode, token ):
   # check gre key to vrf mapping is supported under decap group by platform
   if allDgStatus.greKeyToForwardingVrfMappingSupported:
      return None
   return CliParser.guardNotThisPlatform

tacTunnelPayloadType = Tac.Type( 'Tunnel::Decap::PayloadType' )
def decapPayloadTypeGuard( mode, token ):
   # Check payload types supported by platform. But currently there is only one type
   # so not adding the platform status.
   if token == tacTunnelPayloadType.mpls:
      return None
   return CliParser.guardNotThisPlatform

# Guard for MPLS-TC to TC mapping on GRE decap-groups
def qosTcToMplsTcDecapGuard( mode, token ):
   if allDgStatus.greOrUdpDecapGroupQosFromMplsTcSupported:
      return None
   return CliParser.guardNotThisPlatform

class DecapGroupMode( DecapGroupBaseMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Decap Group Configuration'

   def __init__( self, parent, session, dgName ):
      DecapGroupBaseMode.__init__( self, dgName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.group = self.dgConfig()

   def dgConfig( self ):
      group = cliDgConfig.decapGroup.get( self.dgName )
      if group is None:
         return cliDgConfig.decapGroup.newMember( self.dgName )
      else:
         return group

   def numIp6DgEntriesForDg( self, dgName ):
      count = 0
      dgConfig = cliDgConfig.decapGroup.get( dgName, None )
      if not dgConfig:
         return 0
      for ipAddr in dgConfig.decapIp:
         if ipAddr.v6Addr:
            count += 1
      for decapIntfKey in dgConfig.decapIntf:
         if decapIntfKey.addressFamily == 'ipv6':
            count += 1
      return count

   def numIp6DgEntries( self ):
      count = 0
      for dgName in cliDgConfig.decapGroup:
         count += self.numIp6DgEntriesForDg( dgName )
      return count

   def ip6DecapGroupPlatformWarning( self, add=True ):
      if allDgStatus.ip6DecapFeatureWarning:
         if self.numIp6DgEntries() == 1 and add:
            self.addWarning( "Please disable \"platform sand ipv6 host-route "
                             "exact-match\" and \"ipv6 hardware fib optimize "
                             "prefix-length 128\", if configured, to enable "
                             "IPv6 decap group support." )
         elif self.numIp6DgEntries() == 0 and not add:
            self.addWarning( "Please configure \"platform sand ipv6 host-route "
                             "exact-match\" to switch to default mode." )

   def handleTunnelDecapIp( self, decapIps=None ):
      if self.dgConfig().tunnelType not in [ 'gre', 'ipip', 'udp' ]:
         self.addError( "Must configure tunnel type first" )
         return
      elif not decapIps:
         return
      # verify all ip addresses are supported
      for ip in decapIps:
         ipAddress = Tac.Value( 'Arnet::IpGenPrefix', str( ip ) )
         if ipAddress.af == 'ipv6':
            if not decapTunnelIp6DestinationSupported( self,
                                                       self.dgConfig().tunnelType ):
               self.addError( v6Error % self.dgConfig().tunnelType.upper() )
               return

      # now apply the config
      for ip in decapIps:
         ipAddress = Tac.Value( 'Arnet::IpGenPrefix', str( ip ) )
         self.dgConfig().decapIp[ ipAddress ] = True
         if ipAddress.af == 'ipv6':
            self.ip6DecapGroupPlatformWarning()

   def delTunnelDecapIp( self, decapIps=None ):
      if decapIps:
         for ip in decapIps:
            ipAddress = Tac.Value( 'Arnet::IpGenPrefix', str( ip ) )
            if ipAddress in self.dgConfig().decapIp:
               del self.dgConfig().decapIp[ ipAddress ]
               if ipAddress.af == 'ipv6':
                  self.ip6DecapGroupPlatformWarning( False )

   def updateExcludeIntfs( self, decapIntfKey, excludeIntfRanges ):
      # Due to the syntax having '{ IntfRangeMatcher }',
      # we have an iterable of iterables of interfaces.
      intfs = [ intf for intfs in excludeIntfRanges
                     for intf in intfs ]

      # update exclude interface set
      if intfs:
         decapExcludeIntfSet = Tac.Value( 'Tunnel::Decap::DecapExcludeIntfSet',
                                          decapIntfKey )
         for intf in intfs:
            decapExcludeIntfSet.excludeIntf.add( Tac.Value( 'Arnet::IntfId', intf ) )
         self.dgConfig().decapExcludeIntfSet[ decapIntfKey ] = decapExcludeIntfSet
      else:
         del self.dgConfig().decapExcludeIntfSet[ decapIntfKey ]

   def handleTunnelDecapIntf( self, intfOption=None, addressOption=None,
                              excludeIntfOption=None, **kwargs ):
      if intfOption != "all":
         intfId = Tac.Value( 'Arnet::IntfId', intfOption.name )
         addrFamily = "ipv4"
         if addressOption != None: # pylint: disable=singleton-comparison
            addrType = addressOption
            oldConfig = False
         else:
            addrType = "primary"
            oldConfig = True
         decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                                   intfId, addrFamily )
         decapIntf = Tac.Value( 'Tunnel::Decap::DecapIntf', 
                                decapIntfKey, addrType, oldConfig )
         assert not excludeIntfOption
         self.dgConfig().decapIntf[ decapIntfKey ] = decapIntf
      else:
         intfId = Tac.Value( 'Arnet::IntfId', "" )
         intfId = intfId # pylint: disable=self-assigning-variable
         addrFamily = "ipv4"
         if addressOption != None: # pylint: disable=singleton-comparison
            addrType = addressOption
            oldConfig = False
         else:
            addrType = "primary"
            oldConfig = True
         decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                                   intfId, addrFamily )
         decapIntfAll = Tac.Value( 'Tunnel::Decap::DecapIntf', 
                                   decapIntfKey, addrType, oldConfig )
         self.updateExcludeIntfs( decapIntfKey, excludeIntfOption )
         self.dgConfig().decapIntf[ decapIntfKey ] = decapIntfAll

   def handleTunnelDecapIntfV6( self, intfOption=None, excludeIntfOption=None ):
      if not decapTunnelIp6DestinationSupported( self,
                                                 self.dgConfig().tunnelType ):
         self.addError( v6Error % self.dgConfig().tunnelType.upper() )
         return

      if intfOption != "all":
         intfId = Tac.Value( 'Arnet::IntfId', intfOption.name )
      else:
         intfId = Tac.Value( 'Arnet::IntfId', "" )
      addrFamily = "ipv6"
      addrType = "all"
      oldConfig = False
      decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                                intfId, addrFamily )
      decapIntf = Tac.Value( 'Tunnel::Decap::DecapIntf',
                             decapIntfKey, addrType, oldConfig )
      if intfOption == "all":
         # only permit exclude interface option for decap-interface all
         self.updateExcludeIntfs( decapIntfKey, excludeIntfOption )
      self.dgConfig().decapIntf[ decapIntfKey ] = decapIntf
      self.ip6DecapGroupPlatformWarning()

   def delTunnelDecapIntf( self, intfOption=None ):
      if intfOption != "all":
         intfId = Tac.Value( 'Arnet::IntfId', intfOption.name )
         addrFamily = "ipv4"
         decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                                   intfId, addrFamily )
         del self.dgConfig().decapIntf[ decapIntfKey ]
      else:
         intfId = Tac.Value( 'Arnet::IntfId', "" )
         intfId = intfId # pylint: disable=self-assigning-variable
         addrFamily = "ipv4"
         decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                                   intfId, addrFamily )
         del self.dgConfig().decapIntf[ decapIntfKey ]

   def delTunnelDecapIntfV6( self, intfOption=None ):
      if intfOption != "all":
         intfId = Tac.Value( 'Arnet::IntfId', intfOption.name )
      else:
         intfId = Tac.Value( 'Arnet::IntfId', "" )
      addrFamily = "ipv6"
      decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                                intfId, addrFamily )
      del self.dgConfig().decapIntf[ decapIntfKey ]
      self.ip6DecapGroupPlatformWarning( False )

   def handleTunnelType( self, tunnelType ):
      # check for v6 supported
      if self.numIp6DgEntriesForDg( self.dgConfig().name ) > 0 and \
           not decapTunnelIp6DestinationSupported( self, tunnelType ):
         self.addError( v6Error % tunnelType.upper() )
         return

      if tunnelType == 'gre':
         self.dgConfig().tunnelType = 'gre'
      elif tunnelType == 'ipip':
         self.dgConfig().tunnelType = 'ipip'
      elif tunnelType == 'udp':
         self.dgConfig().tunnelType = 'udp'
      else:
         assert False

      if tunnelType != 'gre':
         if self.dgConfig().greKeyToForwardingVrfMapping:
            self.addWarning( "Removing gre-key to forwarding-vrf mappings" )
            self.dgConfig().greKeyToForwardingVrfMapping.clear()

      if tunnelType not in [ 'gre', 'udp'] :
         if self.dgConfig().qosTcFromMplsTc:
            self.addWarning( "Removing MPLS-TC to TC QoS mapping" )
            self.dgConfig().qosTcFromMplsTc = False

   def handleTunnelGreOrUdpQosTcFromMplsTc( self, args ):
      if self.dgConfig().tunnelType not in [ 'gre', 'udp' ]:
         self.addErrorAndStop(
            "Setting MPLS-TC to TC QoS mapping only supported for tunnel type " 
            "gre or udp"
         )
      self.dgConfig().qosTcFromMplsTc = True

   def noTunnelGreOrUdpQosTcFromMplsTc( self, args ):
      self.dgConfig().qosTcFromMplsTc = False

   def handleTunnelDestinationPort( self, dport=6635, payloadType=None ):
      if self.dgConfig().tunnelType not in [ 'udp' ]:
         self.addError(
            "Setting tunnel destination port only supported for tunnel type udp" )
         return
      self.dgConfig().payloadType = payloadType
      self.dgConfig().destinationPort = dport

   def delTunnelDestinationPort( self ):
      self.dgConfig().payloadType = 'unknownPayload'
      self.dgConfig().destinationPort = 0

   def handleForwardingVrf( self, vrfName ):
      if self.dgConfig().tunnelType not in [ 'gre', 'ipip', 'udp' ]:
         self.addError( "Must configure tunnel type first" )
         return
      if self.dgConfig().greKeyToForwardingVrfMapping:
         self.addError( "Cannot configure both forwarding vrf as well as gre-key" +
                        " to forwarding vrf mapping" )
         return
      forwardingVrf = Tac.Value( 'L3::VrfName', vrfName )
      self.dgConfig().forwardingVrf = forwardingVrf

   def delForwardingVrf( self ):
      self.dgConfig().forwardingVrf = VrfName() 

   def handleGreKeyToForwardingVrfMapping( self, greKey, vrfName ):
      if self.dgConfig().tunnelType != 'gre':
         self.addError( "Must configure tunnel type as GRE first" )
         return
      if self.dgConfig().forwardingVrf:
         self.addError( "Cannot configure both forwarding vrf as well as gre-key" +
                        " to forwarding vrf mapping" )
         return
      vrf = Tac.Value( 'L3::VrfName', vrfName )
      self.dgConfig().greKeyToForwardingVrfMapping[ greKey ] = vrf

   def delGreKeyToForwardingVrfMapping( self, greKey, vrfName ):
      curVrfName = self.dgConfig().greKeyToForwardingVrfMapping.get( greKey )
      if vrfName and curVrfName:
         if vrfName != curVrfName:
            vrf = Tac.Value( 'L3::VrfName', vrfName )
            self.addError( "Vrf %s is not mapped to GRE key %d" % ( vrf,
                                                                    greKey ) )
            return
      del self.dgConfig().greKeyToForwardingVrfMapping[ greKey ]

def gotoDecapGroupMode( mode, args ):
   dgName = args[ 'DGNAME' ]
   childMode = mode.childMode( DecapGroupMode, dgName=dgName )
   mode.session_.gotoChildMode( childMode )

def deleteDg( mode, args ):
   dgName = args[ 'DGNAME' ]
   if cliDgConfig and cliDgConfig.decapGroup.get( dgName ):
      del cliDgConfig.decapGroup[ dgName ]

#--------------------------------------------
# show ip decap-group
#--------------------------------------------
# Hook implemented by platform package to provide the current state of the
# decap group counter feature.  There can only be one registered extension.
#
# Note: Cannot reference Ale::FlexCounter::FeatureConfigDir from the Tunnel package
# because the abuild will fail if any btests reference types in AleCounters package.
# Importing anything from AleCounters will cause a package dependency
# cycle: Ira->Tunnel->AleCounters->Qos->Acl->Ira.
decapGroupCountersSupportedHook = CliExtensions.CliHook()
decapGroupCountersEnabledHook = CliExtensions.CliHook()

class ShowIpDecapGroup( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip decap-group [ dynamic | DECAP_GROUP ]'''
   # NB: The default priority for a KeywordRule is PRIO_KEYWORD===PRIO_HIGH,
   # therefore "default" will never match a decap-group name as a filter.
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'decap-group': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'decap-group',
                  helpdesc='Decap group information' ),
               guard=decapGroupSupportedGuard ),
            'dynamic': 'Display only dynamic entries',
            'DECAP_GROUP': CliMatcher.DynamicNameMatcher(
               lambda _: mergedDgConfig.decapGroup, helpdesc='Decap group name' )
            }
   cliModel = DecapGroupModel.DecapGroups

   @staticmethod
   def _decapGroupSmashCounter( decapKey ):
      counter = decapGroupCounterTable.decapGroupCounter.get( decapKey )
      snapshot = decapGroupSnapshotTable.decapGroupCounter.get( decapKey )
      if counter is None:
         pkts = 0
         octets = 0
      elif snapshot is None:
         pkts = counter.pkts
         octets = counter.octets
      else:
         pkts = counter.pkts - snapshot.pkts
         octets = counter.octets - snapshot.octets
      return pkts, octets

   @staticmethod
   def _ipIntfCounters( ipIntf, addressType, tunnelType ):
      totalPackets = 0
      totalOctets = 0
      if not decapGroupTunnelDedicatedCounterSupported():
         tunnelType = tacTunnelType.unknown
      intfId = Tac.Value( 'Arnet::IntfId', ipIntf.intfId )
      if MgmtIntfId.isMgmtIntfId( intfId ) or \
            InternalIntfId.isInternalIntfId( intfId ) or \
            ipIntf.vrf != VrfName.defaultVrf:
         return totalPackets, totalOctets

      ipAddr = ipIntf.activeAddrWithMask.address
      decapIp = Arnet.IpGenPrefix( ipAddr )
      decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
         decapIp, tunnelType )
      packets, octets = ShowIpDecapGroup._decapGroupSmashCounter( decapKey )
      totalPackets += packets
      totalOctets += octets
      if addressType == AddressType.all:
         for ipAddr in ipIntf.activeSecondaryWithMask:
            decapIp = Arnet.IpGenPrefix( ipAddr.address )
            decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
               decapIp, tunnelType )
            packets, octets = ShowIpDecapGroup._decapGroupSmashCounter( decapKey )
            totalPackets += packets
            totalOctets += octets
         if ipIntf.activeVirtualAddrWithMask.address != ipAddrZero:
            decapIp = Arnet.IpGenPrefix(
               ipIntf.activeVirtualAddrWithMask.address )
            decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
               decapIp, tunnelType )
            packets, octets = ShowIpDecapGroup._decapGroupSmashCounter( decapKey )
            totalPackets += packets
            totalOctets += octets
      return totalPackets, totalOctets

   @staticmethod
   def _ip6IntfCounters( ip6Intf, tunnelType ):
      if not decapGroupTunnelDedicatedCounterSupported():
         tunnelType = tacTunnelType.unknown
      totalPackets = 0
      totalOctets = 0
      intfId = Tac.Value( 'Arnet::IntfId', ip6Intf.intfId )
      if MgmtIntfId.isMgmtIntfId( intfId ) or \
            InternalIntfId.isInternalIntfId( intfId ) or \
            ip6Intf.vrf != VrfName.defaultVrf:
         return totalPackets, totalOctets

      for addrInfo in ip6Intf.addr:
         if not addrInfo.address.isLinkLocal:
            decapIp = Arnet.IpGenPrefix( addrInfo.address.stringValue )
            decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
               decapIp, tunnelType )
            packets, octets = ShowIpDecapGroup._decapGroupSmashCounter( decapKey )
            totalPackets += packets
            totalOctets += octets
      return totalPackets, totalOctets

   @staticmethod
   def _decapGroupIntfAllCounters( intf, excludeIntfSet, tunnelType ):
      '''
      If the interface ID is the empty interface ID, get the counters for all
      configured L3 interfaces.
      '''
      assert intf.intfId == Tac.Value( 'Arnet::IntfId', '' )
      counter = DecapGroupModel.DecapGroupCounterEntry()
      counter.packets = 0
      counter.octets = 0
      if intf.addressFamily == AddressFamily.ipv4:
         for ipIntf in ipStatus.ipIntfStatus.values() :
            if excludeIntfSet and ipIntf.intfId in excludeIntfSet.excludeIntf:
               continue
            packets, octets = \
                  ShowIpDecapGroup._ipIntfCounters( ipIntf, intf.addressType,
                          tunnelType )
            counter.packets += packets
            counter.octets += octets
      elif intf.addressFamily == AddressFamily.ipv6:
         for ip6Intf in ip6Status.intf.values() :
            if excludeIntfSet and ip6Intf.intfId in excludeIntfSet.excludeIntf:
               continue
            packets, octets = ShowIpDecapGroup._ip6IntfCounters( ip6Intf,
                    tunnelType )
            counter.packets += packets
            counter.octets += octets
      return counter

   @staticmethod
   def _decapGroupIntfCounters( intf, tunnelType ):
      '''
      Get the counters for the decap group interface.  Always get the counters
      for the primary address on the interface.  If the address type is 'all',
      accumulate the counters for all secondary and virtual addresses on the
      interface into a single counter for the interface.
      '''
      assert intf.intfId != Tac.Value( 'Arnet::IntfId', '' )
      counter = DecapGroupModel.DecapGroupCounterEntry()
      if intf.addressFamily == AddressFamily.ipv4:
         if intf.intfId.stringValue in ipStatus.ipIntfStatus:
            ipIntf = ipStatus.ipIntfStatus[ intf.intfId.stringValue ]
            counter.packets, counter.octets = ShowIpDecapGroup._ipIntfCounters(
               ipIntf, intf.addressType, tunnelType )
         else:
            counter.packets = 0
            counter.octets = 0
      elif intf.addressFamily == AddressFamily.ipv6:
         if intf.intfId.stringValue in ip6Status.intf:
            ip6Intf = ip6Status.intf[ intf.intfId.stringValue ]
            counter.packets, counter.octets = \
                  ShowIpDecapGroup._ip6IntfCounters( ip6Intf, tunnelType )
         else:
            counter.packets = 0
            counter.octets = 0
      else:
         assert False, 'Unexpected address family {0}'.format( intf.addressFamily )
      return counter

   @staticmethod
   def _decapGroupCountersEnabled():
      countersEnabled = False
      for hook in decapGroupCountersEnabledHook.extensions():
         # There can only be one hook registered.
         assert len( decapGroupCountersEnabledHook.extensions() ) == 1
         countersEnabled = hook()
      return countersEnabled

   @staticmethod
   def _addCounters( dg, dgModel ):
      if dgModel.tunnelType == tacTunnelType.unknown:
         return

      if not ShowIpDecapGroup._decapGroupCountersEnabled():
         dgModel.decapIpCounters = None
         for intf in dgModel.decapIntf:
            intf.counter = None
         return

      for decapIp in dgModel.decapIp:
         tunnelType = dgModel.tunnelType
         if not decapGroupTunnelDedicatedCounterSupported():
            tunnelType = tacTunnelType.unknown
         decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
            decapIp, tunnelType )
         counter = DecapGroupModel.DecapGroupCounterEntry()
         counter.packets, counter.octets = \
               ShowIpDecapGroup._decapGroupSmashCounter( decapKey )
         dgModel.decapIpCounters[ decapIp ] = counter

      for intf in dgModel.decapIntf:
         if intf.intfId == Tac.Value( 'Arnet::IntfId', '' ):
            decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                                      intf.intfId, intf.addressFamily )
            excludeIntfSet = None
            if decapIntfKey in dg.decapExcludeIntfSet:
               excludeIntfSet = dg.decapExcludeIntfSet[ decapIntfKey ]
            intf.counter = \
                  ShowIpDecapGroup._decapGroupIntfAllCounters( intf, excludeIntfSet,
                          dgModel.tunnelType )
         else:
            intf.counter = ShowIpDecapGroup._decapGroupIntfCounters( intf,
                    dgModel.tunnelType )
   
   @staticmethod
   def handler( mode, args ):
      groups = {}
      dynamicOnly = 'dynamic' in args
      nameFilter = args.get( 'DECAP_GROUP' )

      for dgName, dg in mergedDgConfig.decapGroup.items():
         if nameFilter and dgName != nameFilter:
            continue
         if dynamicOnly and not dg.dynamic:
            continue
         if dg.tunnelType == tacTunnelType.gre:
            dgModel = DecapGroupModel.DecapGroupGreModel( decapIp=list( dg.decapIp
               ) )
            for intf in dg.decapIntf:
               decapIntf = DecapGroupModel.DecapIntfModel()
               decapIntf.intfId = intf.intfId
               decapIntf.addressFamily = intf.addressFamily
               decapIntf.addressType = dg.decapIntf[ intf ].addressType
               decapIntf.oldConfig = dg.decapIntf[ intf ].oldConfig
               oldIntf = decapIntf
               oldIntf.oldConfig = True
               if oldIntf not in dgModel.decapIntf:
                  dgModel.decapIntf.append( decapIntf )
         elif dg.tunnelType == tacTunnelType.ipip:
            dgModel = DecapGroupModel.DecapGroupIpIpModel()
            for intf in dg.decapIntf:
               decapIntf = DecapGroupModel.DecapIntfModel()
               decapIntf.intfId = intf.intfId
               decapIntf.addressFamily = intf.addressFamily
               decapIntf.addressType = dg.decapIntf[ intf ].addressType
               decapIntf.oldConfig = dg.decapIntf[ intf ].oldConfig
               oldIntf = decapIntf
               oldIntf.oldConfig = True
               if oldIntf not in dgModel.decapIntf:
                  dgModel.decapIntf.append( decapIntf )
            dgModel.decapIp.extend( dg.decapIp.keys() )
         elif dg.tunnelType == tacTunnelType.udp:
            dgModel = DecapGroupModel.DecapGroupUdpModel()
            for intf in dg.decapIntf:
               decapIntf = DecapGroupModel.DecapIntfModel()
               decapIntf.intfId = intf.intfId
               decapIntf.addressFamily = intf.addressFamily
               decapIntf.addressType = dg.decapIntf[ intf ].addressType
               decapIntf.oldConfig = dg.decapIntf[ intf ].oldConfig
               oldIntf = decapIntf
               oldIntf.oldConfig = True
               if oldIntf not in dgModel.decapIntf:
                  dgModel.decapIntf.append( decapIntf )
            dgModel.decapIp.extend( dg.decapIp.keys() )
            dgModel.destinationPort = dg.destinationPort
            dgModel.payloadType = dg.payloadType
         else:
            dgModel = DecapGroupModel.DecapGroupModelBase()
         dgModel.tunnelType = dg.tunnelType
         dgModel.persistent = not dg.dynamic
         if dg.forwardingVrf:
            dgModel.forwardingVrf = dg.forwardingVrf
         ShowIpDecapGroup._addCounters( dg, dgModel )
         groups[ dgName ] = dgModel

      destPortToPayloadType = {}
      for dport, ptype in mergedDgConfig.globalDestPortToPayloadType.items():
         destPortToPayloadType[ dport ] = ptype 

      groupsModel = DecapGroupModel.DecapGroups( decapGroups=groups,
                     globalUdpDestPortToPayloadType=destPortToPayloadType )
      return groupsModel

BasicCli.addShowCommandClass( ShowIpDecapGroup )

#--------------------------------------------
# show ip decap-group qos
#--------------------------------------------
# Shows the QoS parameters for the decap groups. Specifically for the GRE/UDP
# tunnel type decap-groups.

class ShowIpDecapGroupQos( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip decap-group [ dynamic | DECAP_GROUP ] qos'''
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'decap-group': CliCommand.guardedKeyword( 'decap-group',
                                                helpdesc='Decap group information',
                                                guard=decapGroupSupportedGuard ),
      'dynamic': 'Display only dynamic entries',
      'DECAP_GROUP': CliMatcher.DynamicNameMatcher(
         lambda _: mergedDgConfig.decapGroup, helpdesc='Decap group name' ),
      'qos': CliCommand.guardedKeyword( 'qos', helpdesc='QoS information',
                                        guard=qosTcToMplsTcDecapGuard )
   }
   cliModel = DecapGroupModel.DecapGroupQos

   @staticmethod
   def handler( mode, args ):
      groups = {}
      dynamicOnly = 'dynamic' in args
      nameFilter = args.get( 'DECAP_GROUP' )

      for dgName, dg in mergedDgConfig.decapGroup.items():
         if nameFilter and dgName != nameFilter:
            continue
         if dynamicOnly and not dg.dynamic:
            continue
         # Since this QoS information is only applicable for GRE/UDP tunnels, the
         # show CLI will only consider the GRE/UDP tunnels.
         if dg.tunnelType not in [ tacTunnelType.gre, tacTunnelType.udp ]:
            continue
         dgModel = DecapGroupModel.DecapGroupQosTcModel()
         dgModel.tunnelType = dg.tunnelType
         dgModel.persistent = not dg.dynamic
         if dg.forwardingVrf:
            dgModel.forwardingVrf = dg.forwardingVrf
         dgModel.qosTcFromMplsTc = dg.qosTcFromMplsTc
         groups[ dgName ] = dgModel

      groupsModel = DecapGroupModel.DecapGroupQos( groups=groups )
      return groupsModel

BasicCli.addShowCommandClass( ShowIpDecapGroupQos )

#--------------------------------------------
# show ip decap-group gre keys
#--------------------------------------------
# Shows the GRE keys and corresponding forwarding VRFs for GRE decap groups.

class ShowIpDecapGroupGreKeys( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip decap-group [ dynamic | DECAP_GROUP ] gre keys'''
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'decap-group': CliCommand.guardedKeyword( 'decap-group',
                                                helpdesc='Decap group information',
                                                guard=decapGroupSupportedGuard ),
      'dynamic': 'Display only dynamic entries',
      'DECAP_GROUP': CliMatcher.DynamicNameMatcher(
         lambda _: mergedDgConfig.decapGroup, helpdesc='Decap group name' ),
      'gre': CliCommand.guardedKeyword( 'gre',
         helpdesc='Tunnel type GRE', guard=greKeyToForwardingVrfMappingGuard ),
      'keys':  CliCommand.singleKeyword( 'keys',
         helpdesc='Show GRE keys and associated forwarding VRFs' )
   }
   cliModel = DecapGroupModel.DecapGroupsWithOnlyGreKeyToVrfMap

   @staticmethod
   def handler( mode, args ):
      groups = {}
      dynamicOnly = 'dynamic' in args
      nameFilter = args.get( 'DECAP_GROUP' )

      for dgName, dg in mergedDgConfig.decapGroup.items():
         if nameFilter and dgName != nameFilter:
            continue
         if dynamicOnly and not dg.dynamic:
            continue
         if dg.tunnelType != tacTunnelType.gre:
            continue
         dgModel = DecapGroupModel.DecapGroupWithOnlyGreKeyToVrfMapModel()
         greVrfMappingToCopy = dg.greKeyToForwardingVrfMapping
         dgModel.greKeyToForwardingVrfMapping.update( greVrfMappingToCopy )
         groups[ dgName ] = dgModel

      groupsModel = DecapGroupModel.DecapGroupsWithOnlyGreKeyToVrfMap(
         decapGroups=groups )
      return groupsModel

BasicCli.addShowCommandClass( ShowIpDecapGroupGreKeys )

def decapGroupCountersGuard( mode, token ):
   '''
   Guard the DecapGroup counters CLI commands.
   '''
   for hook in decapGroupCountersSupportedHook.extensions():
      # There can only be one hook registered.
      assert len( decapGroupCountersSupportedHook.extensions() ) == 1
      if not hook():
         return CliParser.guardNotThisPlatform

   for hook in decapGroupCountersEnabledHook.extensions():
      # There can only be one hook registered.
      assert len( decapGroupCountersEnabledHook.extensions() ) == 1
      if hook():
         return None

   return "hardware counter feature decap-group should be enabled first"

def decapGroupClearIntfAllCounters( dg, intf, clearReq ):
   '''
   If the interface ID is the empty interface ID, clear the counters for all
   configured L3 interfaces.
   '''
   assert intf.intfId == ''
   decapIntfKey = Tac.Value( 'Tunnel::Decap::DecapIntfKey',
                             intf.intfId, intf.addressFamily )
   excludeIntfSet = None
   tunnelType = dg.tunnelType
   if not decapGroupTunnelDedicatedCounterSupported():
      tunnelType = tacTunnelType.unknown
   if decapIntfKey in dg.decapExcludeIntfSet:
      excludeIntfSet = dg.decapExcludeIntfSet[ decapIntfKey ]
   if intf.addressFamily == AddressFamily.ipv4:
      addressType = dg.decapIntf[ intf ].addressType
      for ipIntf in ipStatus.ipIntfStatus.values() :
         if excludeIntfSet and ipIntf.intfId in excludeIntfSet.excludeIntf:
            continue
         if MgmtIntfId.isMgmtIntfId( ipIntf.intfId ) or \
            InternalIntfId.isInternalIntfId( ipIntf.intfId ) or \
            ipIntf.vrf != VrfName.defaultVrf:
            continue

         decapIp = Arnet.IpGenPrefix( ipIntf.activeAddrWithMask.address )
         decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
            decapIp, tunnelType )
         clearReq[ decapKey ] = True
         if addressType == AddressType.all:
            for ipAddr in ipIntf.activeSecondaryWithMask:
               decapIp = Arnet.IpGenPrefix( ipAddr.address )
               decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
                  decapIp, tunnelType )
               clearReq[ decapKey ] = True
            if ipIntf.activeVirtualAddrWithMask.address != ipAddrZero:
               decapIp = Arnet.IpGenPrefix(
                  ipIntf.activeVirtualAddrWithMask.address )
               decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
                  decapIp, tunnelType )
               clearReq[ decapKey ] = True
   elif intf.addressFamily == AddressFamily.ipv6:
      for ip6Intf in ip6Status.intf.values() :
         if excludeIntfSet and ip6Intf.intfId in excludeIntfSet.excludeIntf:
            continue
         if MgmtIntfId.isMgmtIntfId( ip6Intf.intfId ) or \
               InternalIntfId.isInternalIntfId( ip6Intf.intfId ) or \
               ip6Intf.vrf != VrfName.defaultVrf:
            continue

         for addrInfo in ip6Intf.addr:
            if not addrInfo.address.isLinkLocal:
               decapIp = Arnet.IpGenPrefix( addrInfo.address.stringValue )
               decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
                  decapIp, tunnelType )
               clearReq[ decapKey ] = True

def decapGroupClearIntfCounters( dg, clearReq ):
   tunnelType = dg.tunnelType
   if not decapGroupTunnelDedicatedCounterSupported():
      tunnelType = tacTunnelType.unknown
   for intf in dg.decapIntf:
      if intf.intfId == '':
         decapGroupClearIntfAllCounters( dg, intf, clearReq )
         continue

      if intf.addressFamily == AddressFamily.ipv4:
         if intf.intfId not in ipStatus.ipIntfStatus:
            continue
         ipIntf = ipStatus.ipIntfStatus[ intf.intfId ]
         if ipIntf is None:
            continue
         if ipIntf.activeAddrWithMask.address is not None:
            decapIp = Arnet.IpGenPrefix( ipIntf.activeAddrWithMask.address )
            decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
               decapIp, tunnelType )
            clearReq[ decapKey ] = True
         if dg.decapIntf[ intf ].addressType == AddressType.all:
            for ipAddr in ipIntf.activeSecondaryWithMask:
               decapIp = Arnet.IpGenPrefix( ipAddr.address )
               decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
                  decapIp, tunnelType )
               clearReq[ decapKey ] = True
               if ipIntf.activeVirtualAddrWithMask.address != ipAddrZero:
                  decapIp = Arnet.IpGenPrefix(
                     ipIntf.activeVirtualAddrWithMask.address )
                  decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
                     decapIp, tunnelType )
                  clearReq[ decapKey ] = True

      elif intf.addressFamily == AddressFamily.ipv6:
         if intf.intfId not in ip6Status.intf:
            continue
         ip6Intf = ip6Status.intf[ intf.intfId ]
         if ip6Intf is None:
            continue
         for addrInfo in ip6Intf.addr:
            if not addrInfo.address.isLinkLocal:
               decapIp = Arnet.IpGenPrefix( addrInfo.address.stringValue )
               decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
                  decapIp, tunnelType )
               clearReq[ decapKey ] = True

def clearIpDecapGroupCounters( mode, nameFilter=None ):
   if nameFilter and nameFilter == 'dynamic':
      dynamicOnly = True
      nameFilter = None
   else:
      dynamicOnly = False

   clearReq = countersClearConfig.clearCountersRequest
   clearReq.clear()
   for dgName, dg in mergedDgConfig.decapGroup.items():
      if nameFilter and dgName != nameFilter:
         continue
      if dynamicOnly and not dg.dynamic:
         continue
      if dg.tunnelType == tacTunnelType.unknown:
         continue

      tunnelType = dg.tunnelType
      if not decapGroupTunnelDedicatedCounterSupported():
         tunnelType = tacTunnelType.unknown
      for decapIp in dg.decapIp:
         decapKey = Tac.Value( 'Tunnel::Decap::DecapGroupCounterKey',
            decapIp, tunnelType )
         clearReq[ decapKey ] = True

      decapGroupClearIntfCounters( dg, clearReq )

def getPrintStrForPayloads( payloadType ):
   if payloadType == 'ipvx':
      return "IP"
   elif payloadType == 'ipv4':
      return "IPv4"
   elif payloadType == 'ipv6':
      return "IPv6"
   elif payloadType == 'mpls':
      return "MPLS"
   return None

def globalUdpDestPortHandler( mode, args ):
   destPort = int( args.get( 'DEST_PORT' ) )
   payloadType = args.get( 'PAYLOAD_TYPE' )
   payloadType = 'ipvx' if payloadType == 'ip' else payloadType

   for port, ptype in cliDgConfig.globalDestPortToPayloadType.items():
      if port != destPort and ptype == payloadType:
         mode.addErrorAndStop( 
            'There can be only one UDP destination port per payload type' )
      if ( ( not allDgStatus.decapAllIpPayloadTypeSupported ) and \
         ( ( payloadType in [ 'ipv4', 'ipv6' ] and ptype == 'ipvx' ) or \
         ( payloadType == 'ipvx' and ptype in [ 'ipv4', 'ipv6' ] ) ) ):
         mode.addErrorAndStop(
         'There can be only one IP payload configuration, either ip or ipv4/ipv6' )
      if port == destPort and ptype != payloadType:
         oldPayloadStr = getPrintStrForPayloads( ptype ) 
         newPayloadStr = getPrintStrForPayloads( payloadType ) 

         warningMsg = \
         "WARNING! The UDP destination port {} is already associated " \
         "with payload type {} and is now being moved to payload type " \
         "{}\n".format( destPort, oldPayloadStr, newPayloadStr )
         suggestionMsg = \
         "If you want to use the same UDP destination port for " \
         "both IPv4 and IPv6 payloads, please use " \
         "\'ip decap-group type udp destination port <port value> payload ip\'"
         
         if "mpls" in [ ptype, payloadType ]:
            displayMsg = warningMsg
         else:
            displayMsg = warningMsg + suggestionMsg
         
         mode.addWarning( displayMsg )
         
         # Ensure the destPort associated with previous payload is removed from the
         # config/HW before programming the destPort for the new payload
         del cliDgConfig.globalDestPortToPayloadType[ destPort ]

   cliDgConfig.globalDestPortToPayloadType[ destPort ] = payloadType


def globalUdpDestPortDelHandler( mode, args ):
   destPort = int( args.get( 'DEST_PORT' ) )
   if destPort in cliDgConfig.globalDestPortToPayloadType:
      del cliDgConfig.globalDestPortToPayloadType[ destPort ]

def startDecapGroupConfigMergeSm():
   global mergedDgConfig
   global dgConfigMergeSm
   mergedDgConfig = Tac.newInstance( "Tunnel::Decap::Config" )
   dgConfigMergeSm = Tac.newInstance( "Tunnel::Decap::DecapGroupConfigMergeSm",
                                      mergedDgConfig,
                                      readOnlyCliDgConfig,
                                      dgDynInputDir,
                                      pluginEntityManager.cEntityManager() )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliDgConfig
   global readOnlyCliDgConfig
   global dgDynInputDir
   global allDgStatus

   global pluginEntityManager
   global decapGroupCounterTable
   global decapGroupSnapshotTable
   global countersClearConfig
   global ipStatus
   global ip6Status

   pluginEntityManager = entityManager

   cliDgConfig = ConfigMount.mount( entityManager, "routing/tunneldecap/input/cli",
                                    "Tunnel::Decap::ConfigInput", "wi" )

   mg = entityManager.mountGroup()
   # We cannot use the ConfigMount'd cliDgConfig in the DecapGroupConfigMergeSm
   # because we need the actual entity, and not a proxy, so we need to mount it
   # a second time directly through a mount group.
   # The variable is marked as read-only just as a reminder not to write
   # using it, even though it is mounted as 'w' because it needs to use the same
   # flags as the ConfigMount case.
   readOnlyCliDgConfig = mg.mount( "routing/tunneldecap/input/cli",
                                   "Tunnel::Decap::ConfigInput", "wi" )
   dgDynInputDir = mg.mount( "routing/tunneldecap/input/dynamic",
                             "Tac::Dir", "r" )
   mg.close( startDecapGroupConfigMergeSm )

   allDgStatus = LazyMount.mount( entityManager,
                                  "routing/hardware/tunneldecap/capability",
                                  "Tunnel::Decap::Hardware::Capability", "r" )

   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ip6Status = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )

   countersClearConfig = LazyMount.mount( entityManager,
                                         "hardware/counter/decapGroup/clear/config",
                                         "Tunnel::Decap::CountersClearConfig",
                                         "w" )

   info = SmashLazyMount.mountInfo( 'reader' )
   mountPath = 'flexCounters/counterTable/DecapGroup/%u' % ( FapId.allFapsId )
   decapGroupCounterTable = SmashLazyMount.mount(
      entityManager, mountPath, "Ale::FlexCounter::DecapGroupCounterTable", info )
   mountPath = 'flexCounters/snapshotTable/DecapGroup/%u' % ( FapId.allFapsId )
   decapGroupSnapshotTable = SmashLazyMount.mount(
      entityManager, mountPath, "Ale::FlexCounter::DecapGroupCounterTable", info )
