# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements IP DHCP Relay configuration.
# It contains interface-specific commands.
#-------------------------------------------------------------------------------

# Unused arguments 'mode' and 'entityManager'
# pylint: disable-msg=W0613

import BasicCliModes
import BasicCli
import CliCommand
import CliMatcher, LazyMount, Tac
import CliParser
from CliMode.DhcpRelay import DhcpRelayGlobalBaseMode, InformationOptionBaseMode
import ConfigMount
import HostnameCli
import Arnet
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.AclCli as AclCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IraIpIntfCli as IraIpIntfCli
import CliPlugin.VlanCli as VlanCli
from CliPlugin.VirtualIntfRule import VirtualIntfMatcher
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
from CliPlugin.VxlanCli import VxlanIntf
from CliToken.Clear import clearKwNode
from CliToken.Dhcp import dhcpMatcherForConfig
from CliToken.Ip import (
      ipMatcherForConfigIf, ipMatcherForClear, ipMatcherForConfig )
from CliToken.Ipv6 import (
      ipv6MatcherForConfigIf, ipv6MatcherForClear, ipv6MatcherForConfig )
from CliToken.Lanz import matcherAccessList
from CommonGuards import ssoStandbyGuard
import Toggles.DhcpRelayToggleLib as DhcpRelayToggleLib
import Toggles.DhcpLibToggleLib as DhcpLibToggleLib

# true if call given from forwarder CLI, used to bypass competing relay check
fromForwarder = False

dhcpRelayConfigDir = None
dhcpRelayCounterConfigDir = None
dhcpFwdConfig = None
dhcpRelayStatusDir = None
mlagDhcpRelayStatusDir = None
platformHardwareSliceDir = None
dhcpRelayReqFloodDisableHwStatusDir = None

dhcp6RouteTypeEnum = Tac.Type( "Ip::Helper::DhcpRelay::Dhcp6RouteType" )

def drConfig():
   return dhcpRelayConfigDir

def drCounterConfig():
   return dhcpRelayCounterConfigDir

def drStatus():
   return dhcpRelayStatusDir

def drMlagStatus():
   return mlagDhcpRelayStatusDir

def vrfHostnamePair( vrf, hostname ):
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfHostnamePair", vrf,
                     hostname )

def vrfHostnameSrcIntfSrcIp( vrf, hostname, srcIntf, srcIp ):
   intfId = Tac.Value( 'Arnet::IntfId', '' )
   if srcIntf:
      intfId = Tac.Value( 'Arnet::IntfId', srcIntf.name )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfHostnameSrcIntfSrcIp", vrf,
                     hostname, intfId, Arnet.IpGenAddr( str( srcIp ) ) )

def vrfIpSrcIntfSrcIp( vrf, ip, srcIntf, srcIp ):
   intfId = Tac.Value( 'Arnet::IntfId', '' )
   if srcIntf:
      intfId = Tac.Value( 'Arnet::IntfId', srcIntf.name )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfIpSrcIntfSrcIp", vrf,
                     Arnet.IpGenAddr( str( ip ) ), intfId,
                     Arnet.IpGenAddr( str( srcIp ) ) )

def optionFormatEncode( value, encodingFormat ):
   # if format is hex, add 0x prefix in the show command
   # to signify we send that value raw
   # if invalid hex we keep empty
   if encodingFormat == '%p':
      return value
   else:
      try:
         int( value, 16 )
      except ValueError:
         return ''
      return '0x' + value

# ip helper addresses are not exactly a routing protocol, but they apply to
# the same set of interfaces (e.g. not management or loopback)
modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

matcherHelper = CliMatcher.KeywordMatcher(
   'helper-address', helpdesc='Specify DHCP Server' )
matcherRelay = CliMatcher.KeywordMatcher(
   'relay', helpdesc='DHCP Relay' )
matcherDest = CliMatcher.KeywordMatcher(
   'destination', helpdesc='DHCP Relay destination address' )
nodeClear = CliCommand.guardedKeyword( 'clear', helpdesc='Reset functions', 
      guard=ssoStandbyGuard )
addressFamilyExprForClear = CliMatcher.EnumMatcher( {
   'ip' : ipMatcherForClear.helpdesc_,
   'ipv6' : ipv6MatcherForClear.helpdesc_,
} )
matcherCounters = CliMatcher.KeywordMatcher( 'counters', helpdesc='Counters' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan', helpdesc='VLAN' )

#-------------------------------------------------------------------------------
# "[no|default] ip helper-address" interface command.
#-------------------------------------------------------------------------------
class IpHelperAddressCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip helper-address'
   data = {
      'ip' : ipMatcherForConfigIf,
      'helper-address' : matcherHelper,
   }

   noOrDefaultHandler = "DhcpRelayHelperHandler.noDhcpRelay"

modelet.addCommandClass( IpHelperAddressCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ip helper-address <ip address/hostname> vrf <vrfName>
# [ source-interface <interface> | source-address <ip address> ]"
# interface command.
#-------------------------------------------------------------------------------
vrfExprFactory = VrfExprFactory(
      helpdesc='Specify VRF of the DHCP server' )
matcherSourceAddr = CliMatcher.KeywordMatcher(
      'source-address',
      helpdesc=( 'Specify the source address to communicate '
                 'with the DHCP server' ) )

class IpHelperHostnameVrfSourceCmd( CliCommand.CliCommandClass ):
   syntax = ( "ip helper-address HOST_OR_ADDR [ VRF ] "
              "[ ( source-interface INTF ) | ( source-address ADDR ) ]" )
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'helper-address' : matcherHelper,
      'HOST_OR_ADDR' : HostnameCli.IpAddrOrHostnameMatcher( ipv6=False,
         helpdesc='Hostname or IP address of DHCP server' ),
      'VRF' : vrfExprFactory,
      'source-interface' : ( 'Specify the source interface to communicate '
                             'with the DHCP server' ),
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
      'source-address' : matcherSourceAddr,
      'ADDR' : IpAddrMatcher.ipAddrMatcher,
   }

   handler = "DhcpRelayHelperHandler.setIntfDhcpRelayHandler"
   noOrDefaultHandler = handler

modelet.addCommandClass( IpHelperHostnameVrfSourceCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 helper-address" interface command.
#-------------------------------------------------------------------------------
class Ipv6DhcpRelayDestCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 dhcp relay destination'
   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'destination' : matcherDest,
   }
   
   noOrDefaultHandler = "DhcpRelayHelperHandler.noDhcpRelayv6"

class Ipv6HelperAddressCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 helper-address'
   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'helper-address' : matcherHelper,
   }

   noOrDefaultHandler = "DhcpRelayHelperHandler.noDhcpRelayv6"

modelet.addCommandClass( Ipv6DhcpRelayDestCmd )
modelet.addCommandClass( Ipv6HelperAddressCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp smart-relay' interface command.
#-------------------------------------------------------------------------------

class IpDhcpSmartRelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp smart-relay'
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp': dhcpMatcherForConfig,
      'smart-relay' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'smart-relay',
         helpdesc='DHCP Smart Relay Agent Forwarding on this interface' ),
         deprecatedByCmd='ip dhcp relay all-subnets default' ),
   }

   handler = "DhcpRelayHelperHandler.setIntfSmartRelay"
   noOrDefaultHandler = handler

class IpDhcpRelaySubnetsCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay all-subnets'
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCP Relay Forwarding on all the subnets',
   }

   handler = "DhcpRelayHelperHandler.setIntfSmartRelay"
   noOrDefaultHandler = handler

modelet.addCommandClass( IpDhcpSmartRelayCmd )
modelet.addCommandClass( IpDhcpRelaySubnetsCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 dhcp realy all-subnets' interface command.
#-------------------------------------------------------------------------------

class Ipv6DhcpRelaySubnetsCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay all-subnets'
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCPv6 relay forwarding on all the subnets',
   }

   handler = "DhcpRelayHelperHandler.setIntfAllSubnetsRelayV6"
   noOrDefaultHandler = handler

modelet.addCommandClass( Ipv6DhcpRelaySubnetsCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 helper-address <ipv6 address> vrf <vrf name > 
# [ source-interface <interface> | source-address <ipv6 address> ]
# [ link-address <ipv6 address> ]"
# interface command.
#-------------------------------------------------------------------------------
ip6AddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher( "DHCP Server's IPv6 address" )
matcherLocalIntf = CliMatcher.KeywordMatcher(
      'local-interface',
      helpdesc='Specify the local interface to communicate with the DHCP server' )
matcherLinkAddr = CliMatcher.KeywordMatcher(
      'link-address',
      helpdesc='Override the default link address specified in the relayed DHCP '
               'packet' )

class Ipv6HelperHostnameVrfSourceCmd( CliCommand.CliCommandClass ):
   syntax = ( "ipv6 helper-address IP6ADDR [ VRF ] "
              "[ ( local-interface INTF ) | ( source-address ADDR ) ] "
              "[ link-address LINKADDR ]" )
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'helper-address' : matcherHelper,
      'IP6ADDR' : ip6AddrMatcher,
      'VRF' : vrfExprFactory,
      'local-interface' : matcherLocalIntf,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
      'source-address' : matcherSourceAddr,
      'ADDR' : Ip6AddrMatcher.ip6AddrMatcher,
      'link-address' : matcherLinkAddr,
      'LINKADDR' : Ip6AddrMatcher.ip6AddrMatcher,
   }

   handler = "DhcpRelayHelperHandler.setIntfDhcp6Relay"
   noOrDefaultHandler = handler

class Ipv6DhcpRelayDestinationVrfSourceCmd( CliCommand.CliCommandClass ):
   syntax = ( "ipv6 dhcp relay destination IP6ADDR [ VRF ] "
              "[ ( local-interface INTF ) | ( source-address ADDR ) ] "
              "[ link-address LINKADDR ]" )
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'destination' : matcherDest, 
      'IP6ADDR' : ip6AddrMatcher,
      'VRF' : vrfExprFactory,
      'local-interface' : matcherLocalIntf,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
      'source-address' : matcherSourceAddr,
      'ADDR' : Ip6AddrMatcher.ip6AddrMatcher,
      'link-address' : matcherLinkAddr,
      'LINKADDR' : Ip6AddrMatcher.ip6AddrMatcher,
   }

   handler = "DhcpRelayHelperHandler.setIntfDhcp6Relay"
   noOrDefaultHandler = handler

modelet.addCommandClass( Ipv6DhcpRelayDestinationVrfSourceCmd )
modelet.addCommandClass( Ipv6HelperHostnameVrfSourceCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 dhcp relay install routes [prefix-delegation]" 
# interface command.
#-------------------------------------------------------------------------------

class Ipv6DhcpRelayInstallRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay install routes [ prefix-delegation ]'
   noOrDefaultSyntax = syntax

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'install' : 'Install routes for DHCP server assigned addresses',
      'routes' : 'Install routes for DHCP server assigned addresses',
      'prefix-delegation' : 'Install only DHCPv6 prefix delegation routes',
   }

   handler = "DhcpRelayHelperHandler.setIntfDhcp6RelayRtInstall"
   noOrDefaultHandler = handler

modelet.addCommandClass( Ipv6DhcpRelayInstallRoutesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 dhcp relay add vendor-option ccap-core <ipv6 address>" 
# interface command
#-------------------------------------------------------------------------------
class Ipv6DhcpRelayAddVendorCcapCore( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay add vendor-option ccap-core CORE'
   noOrDefaultSyntax = 'ipv6 dhcp relay add vendor-option ccap-core [ CORE ]'

   data = {
      'ipv6' : ipv6MatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'add' : 'Add sub-options under DHCP Options',
      'vendor-option' : 'Vendor Specific Information Option (17)',
      'ccap-core' : 'Sub-option 61 containing CCAP Core Ipv6 addresses',
      'CORE' : Ip6AddrMatcher.ip6AddrMatcher,
   }

   handler = "DhcpRelayHelperHandler.setDhcp6CcapCores"
   noOrDefaultHandler = "DhcpRelayHelperHandler.noDhcp6CcapCores"

modelet.addCommandClass( Ipv6DhcpRelayAddVendorCcapCore )

#-------------------------------------------------------------------------------
# The "ip dhcp relay information option circuit-id <circuit-id>" interface command.
#-------------------------------------------------------------------------------

circuitIdValueMatcher = CliMatcher.PatternMatcher(
   pattern=r'[0-9a-zA-Z._:/\-]{1,50}',
   helpdesc='Up to 50 alphanumeric characters describing this interface',
   helpname='Circuit ID'
)

remoteIdValueMatcher = CliMatcher.PatternMatcher(
   pattern=r'[0-9a-zA-Z._:/\-]{1,50}',
   helpdesc='Up to 50 alphanumeric characters',
   helpname='Remote ID'
)

class IpDhcpRelayInformationOptionCircuitIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay information option circuit-id ID'
   noOrDefaultSyntax = 'ip dhcp relay information option circuit-id ...'

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'information' : 'Option 82 (Information Option)',
      'option' : 'DHCP option',
      'circuit-id' : 'Relay Agent (Option 82) Circuit ID for this interface',
      'ID' : circuitIdValueMatcher,
   }

   handler = "DhcpRelayHelperHandler.setIntfCircuitId"
   noOrDefaultHandler = "DhcpRelayHelperHandler.noIntfCircuitId"

modelet.addCommandClass( IpDhcpRelayInformationOptionCircuitIdCmd )

class IpDhcpRelayInformationOptionRemoteIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay information option remote-id ID'
   noOrDefaultSyntax = 'ip dhcp relay information option remote-id ...'

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'information' : 'Option 82 (Information Option)',
      'option' : 'DHCP option',
      'remote-id' : 'Relay Agent (Option 82) Remote ID for this interface',
      'ID' : remoteIdValueMatcher,
   }

   handler = "DhcpRelayHelperHandler.setIntfRemoteId"
   noOrDefaultHandler = "DhcpRelayHelperHandler.noIntfRemoteId"

if DhcpRelayToggleLib.toggleDhcpRelayOpt82RemoteIdV4Enabled():
   modelet.addCommandClass( IpDhcpRelayInformationOptionRemoteIdCmd )

dhcpHelp = 'DHCP server, client and relay interface configuration'
ipv4DhcpRelayHelpDesc = 'IPv4 DHCP relay configuration'
ipv6DhcpRelayHelpDesc = 'IPv6 DHCP relay configuration'
dhcpRelayDisabledHelpDesc = 'Disable DHCP relay configurations'

#-------------------------------------------------------------------------------
# "[no|default] dhcp relay ( ipv4 | ipv6 ) disabled" 
# interface command
#-------------------------------------------------------------------------------
class IpDhcpRelayDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'dhcp relay ( ipv4 | ipv6 ) disabled'
   noOrDefaultSyntax = syntax

   data = {
      'dhcp' : dhcpHelp,   
      'relay' : matcherRelay,
      'ipv4' : ipv4DhcpRelayHelpDesc,
      'ipv6' : ipv6DhcpRelayHelpDesc,
      'disabled' : dhcpRelayDisabledHelpDesc,
      }

   handler = "DhcpRelayHelperHandler.setDhcpRelayDisabledCmd"
   noOrDefaultHandler = handler

modelet.addCommandClass( IpDhcpRelayDisabledCmd )

#-------------------------------------------------------------------------------
# The DhcpRelayIntf class is used to remove the DhcpRelay IntfConfig object when an
# interface is deleted. The Intf class will create a new instance of DhcpRelayIntf
# and call destroy when the interface is deleted
#-------------------------------------------------------------------------------
class DhcpRelayIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the dhcprelay IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      del drConfig().intfConfig[ self.intf_.name ]

#--------------------------------------------------------------------------------
# clear ( ip | ipv6 ) dhcp relay counters INTF
#--------------------------------------------------------------------------------
vxlanIntfMatcher = VirtualIntfMatcher( 'Vxlan', 1, 1,
      value=lambda mode, intf: VxlanIntf( intf, mode ),
      helpdesc="VXLAN Tunnel Interface" )
routedOrVxlanIntfMatcher = IntfCli.Intf.matcherWithRoutingProtoSupport
routedOrVxlanIntfMatcher |= vxlanIntfMatcher

class ClearDhcpRelayCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ADDR_FAMILY dhcp relay counters [ interface INTF ]'
   data = {
      'clear' : nodeClear,
      'ADDR_FAMILY' : addressFamilyExprForClear,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : 'DHCP Relay',
      'counters' : matcherCounters,
      'interface' : 'Specify an interface as a target for the command',
      'INTF' : routedOrVxlanIntfMatcher
   }

   handler = "DhcpRelayHelperHandler.resetDhcpRelayCounters"

BasicCliModes.EnableMode.addCommandClass( ClearDhcpRelayCountersCmd )

#--------------------------------------------------------------------------------
# clear ip dhcp relay counters access-list
#--------------------------------------------------------------------------------
class ClearIpDhcpRelayCountersAccessListCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ADDR_FAMILY dhcp relay counters access-list'
   data = {
      'clear' : nodeClear,
      'ADDR_FAMILY' : addressFamilyExprForClear,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : 'DHCP Relay',
      'counters' : matcherCounters,
      'access-list' : matcherAccessList,
   }

   # Moving this function into this file leads to a failure in CliTest004.py
   handler = "DhcpRelayHelperHandler.clearAclCounters"

BasicCliModes.EnableMode.addCommandClass( ClearIpDhcpRelayCountersAccessListCmd )

#############################################################
# clear ipv6 dhcp relay routes [ ( VRF | INTF | PREFIX_V6 ) ]
# in mode: enable
##############################################################
class ClearIpv6DhcpRelayRoutesCmd( CliCommand.CliCommandClass ):
   syntax = '''clear ipv6 dhcp relay routes [ ( [ VRF ] [ PREFIX_V6 ] ) |
                                                INTF                    ]'''
   data = {
         'clear' : clearKwNode,
         'ipv6' : ipv6MatcherForClear,
         'dhcp' : dhcpMatcherForConfig,
         'relay' : matcherRelay,
         'routes' : 'Install routes for DHCP server assigned addresses',
         'VRF': VrfExprFactory( helpdesc='Clear DHCP Relay routes in a VRF',
                                inclAllVrf=True, inclDefaultVrf=True ),
         'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
         'PREFIX_V6' : Ip6AddrMatcher.Ip6PrefixMatcher( 'DHCP route to clear' )
         }
   
   handler = "DhcpRelayHelperHandler.clearDhcpRelayRoutes"

BasicCliModes.EnableMode.addCommandClass( ClearIpv6DhcpRelayRoutesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ( ip | ipv6 ) dhcp relay always-on
#--------------------------------------------------------------------------------
class DhcpRelayAlwaysOnCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) dhcp relay always-on'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'always-on' : 'Enable Agent Always-On Option',
   }

   handler = "DhcpRelayHelperHandler.setAlwaysOn"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( DhcpRelayAlwaysOnCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ( ip | ipv6 ) dhcp relay qos dscp DSCP
#--------------------------------------------------------------------------------
class DhcpRelayQosDscpCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) dhcp relay qos dscp DSCP'
   noOrDefaultSyntax = '( ip | ipv6 ) dhcp relay qos dscp ...'
   data = {
      'ip' : ipMatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'qos' : 'Configure QoS parameters',
      'dscp' : 'Set DSCP value in IP header',
      'DSCP' : CliMatcher.IntegerMatcher( 
         0, 63, helpdesc='DSCP value' ),
   }

   handler = "DhcpRelayHelperHandler.setDscp"
   noOrDefaultHandler = "DhcpRelayHelperHandler.noDscp"

BasicCliModes.GlobalConfigMode.addCommandClass( DhcpRelayQosDscpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay access-group ACLNAME [ vrf VRFNAME ] [ in ]
#--------------------------------------------------------------------------------
class IpDhcpRelayAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) dhcp relay access-group ACL [ VRF ] [ in ]'
   noOrDefaultSyntax = '( ip | ipv6 ) dhcp relay access-group [ ACL ] [ VRF ] [ in ]'
   data = {
      'ip' : ipMatcherForConfig,
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'access-group' : AclCli.accessGroupKwMatcher,
      'ACL' : AclCli.ipAclNameMatcher,
      'VRF' : AclCli.vrfKwAndNameExprFactory,
      'in' : AclCli.inKwMatcher,
   }

   handler = "DhcpRelayHelperHandler.setAcl"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayAccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay all-subnets default
#--------------------------------------------------------------------------------
class IpDhcpRelayAllSubnetsDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay all-subnets default'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCP Relay Forwarding on all the subnets',
      'default' : 'Enable DHCP Relay Forwarding across all interfaces',
   }

   handler = "DhcpRelayHelperHandler.setSmartRelay"
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayAllSubnetsDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp smart-relay global
#--------------------------------------------------------------------------------
class IpDhcpSmartRelayGlobalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp smart-relay global'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'smart-relay' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 
         'smart-relay',
         helpdesc='DHCP Smart Relay Agent Forwarding' ),
         deprecatedByCmd='ip dhcp relay all-subnets default' ),
      'global' : 'Enable DHCP Smart Relay Agent Forwarding across all interfaces',
   }

   handler = "DhcpRelayHelperHandler.setSmartRelay"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpSmartRelayGlobalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay information option
#--------------------------------------------------------------------------------
class IpDhcpRelayInformationOptionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay information option'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'information' : 'Option 82 (Information Option)',
      'option' : 'Enable Option 82',
   }

   handler = "DhcpRelayHelperHandler.setIntfDhcpRelayInfo"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayInformationOptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp relay option link-layer address
#--------------------------------------------------------------------------------
class Ipv6DhcpRelayOptionLinkLayerCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay option link-layer address'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'option' : 'DHCP option',
      'link-layer' : 'Option 79 (Link Layer Address Option)',
      'address' : 'Address keyword',
   }

   handler = "DhcpRelayHelperHandler.setDhcpRelayv6LinkLayerAddrOpt"
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpRelayOptionLinkLayerCmd )

#--------------------------------------------------------------------------------
# "[no|default] ipv6 dhcp relay option remote-id format ( %m:%i | %m:%p )"
#-------------------------------------------------------------------------------
class Ipv6DhcpRelayRemoteIdFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay option remote-id format ( %m:%i | %m:%p )'
   noOrDefaultSyntax = 'ipv6 dhcp relay option remote-id format ...'
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'option' : 'DHCP option',
      'remote-id' : 'RemoteID option 37',
      'format': 'Encoding format',
      '%m:%i' : 'MAC address and interface ID',
      '%m:%p' :  'MAC address and interface name',
   }
   handler = "DhcpRelayHelperHandler.setDhcpRelayv6RemoteIdOptFormat"
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpRelayRemoteIdFormatCmd )

#--------------------------------------------------------------------------------
# [ no | default } ipv6 dhcp relay all-subnets default
#--------------------------------------------------------------------------------
class Ipv6DhcpRelayAllSubnetsDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 dhcp relay all-subnets default'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : ipv6MatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'all-subnets' : 'DHCPv6 relay forwarding on all the subnets',
      'default' : 'Enable DHCPv6 all subnet relaying across all interfaces',
   }

   handler = "DhcpRelayHelperHandler.setSmartRelayV6"
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6DhcpRelayAllSubnetsDefaultCmd )

#--------------------------------------------------------------------------------
# dhcp relay
# in mode: config
#--------------------------------------------------------------------------------
class DhcpRelayGlobalMode( DhcpRelayGlobalBaseMode, BasicCli.ConfigModeBase ):
   name = "DHCP Relay"

   def __init__( self, parent, session ):
      DhcpRelayGlobalBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterDhcpRelayGlobalModeCmd( CliCommand.CliCommandClass ):
   syntax = '''dhcp relay'''
   noOrDefaultSyntax = syntax

   data = {
      'dhcp': 'DHCP configuration',
      'relay': 'DHCP relay configuration',
      }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( DhcpRelayGlobalMode )
      mode.session_.gotoChildMode( childMode )
   
   noOrDefaultHandler = "DhcpRelayHelperHandler.setDhcpRelayGlobalDefaultValues"

BasicCli.GlobalConfigMode.addCommandClass( EnterDhcpRelayGlobalModeCmd )

#--------------------------------------------------------------------------------
# information option
# in mode: dhcp relay
#--------------------------------------------------------------------------------
class InformationOptionMode( InformationOptionBaseMode, BasicCli.ConfigModeBase ):
   name = "Information Option"

   def __init__( self, parent, session ):
      InformationOptionBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class InformationOptionModeCmd( CliCommand.CliCommandClass ):
   syntax = '''information option'''
   noOrDefaultSyntax = syntax

   data = {
      'information' : 'Option 82 (Information Option)',
      'option' : 'DHCP option',
      }

   handler = "DhcpRelayHelperHandler.setInformationOptionMode"
   noOrDefaultHandler = "DhcpRelayHelperHandler.setInformationOptionDefault"

if DhcpRelayToggleLib.toggleDhcpRelayOpt82RemoteIdV4Enabled():
   DhcpRelayGlobalMode.addCommandClass( InformationOptionModeCmd )

#--------------------------------------------------------------------------------
# server ( IP_GENADDR | HOSTNAME )
# in mode: dhcp relay
# Equivalent to helper-address in interface mode
#--------------------------------------------------------------------------------
class DhcpRelayGlobalServerCmd( CliCommand.CliCommandClass ):
   syntax = '''server ( HOST_OR_ADDR | IPV6_ADDR )'''
   noOrDefaultSyntax = syntax 
   
   # Using IpAddrOrHostnameMatcher, since having two separate matchers
   # (HostnameCli.HostnameMatcher and IpAddrMatcher.IpAddrMatcher) causes
   # both nodes to match simultaneously for inputs like 1.0.0.0
   data = {
      'server' : 'Specify DHCP Server',
      'HOST_OR_ADDR' : HostnameCli.IpAddrOrHostnameMatcher( 
                           helpdesc='IPv4 address or Hostname of DHCP server' ),
      'IPV6_ADDR' : Ip6AddrMatcher.Ip6AddrMatcher(
                           helpdesc='IPv6 address of DHCP server' )
      }

   @staticmethod
   def adapter( mode, args , argsList ):
      # default value to create key
      vrfName = DEFAULT_VRF
      ipAddr = '0.0.0.0'
      srcIntf = ''
      srcIp = '0.0.0.0'
      
      # set ipAddr or hostname
      ipAddrOrHostname = args.get( 'HOST_OR_ADDR' )
      ipAddr, hostname = None, None
      if ipAddrOrHostname:
         try:
            ipAddr = Arnet.IpAddress( ipAddrOrHostname )
         except ValueError:
            hostname = ipAddrOrHostname
      if hostname:
         args[ 'HOSTNAME' ] = hostname
         key = vrfHostnameSrcIntfSrcIp( vrfName, hostname, srcIntf, srcIp )
      else:
         args[ 'IP_ADDR' ] = ipAddr or args.pop( 'IPV6_ADDR' )
         key = vrfIpSrcIntfSrcIp( vrfName, args[ 'IP_ADDR' ], srcIntf, srcIp )
      # save the key to enter the right collection
      args[ 'key' ] = key

   handler = "DhcpRelayHelperHandler.setDhcpRelayGlobalServerCmd"
   noOrDefaultHandler = handler

DhcpRelayGlobalMode.addCommandClass( DhcpRelayGlobalServerCmd )

matcherEncoding = CliMatcher.KeywordMatcher( 'encoding', helpdesc= 'Encoding format')
matcherHexKw = CliMatcher.KeywordMatcher( '%x', helpdesc= 'Hex')
matcherStrKw = CliMatcher.KeywordMatcher( '%p', helpdesc= 'String')

#--------------------------------------------------------------------------------
# (config-information-option) [ no | default ]
#      circuit-id encoding ( %x | %p )
#--------------------------------------------------------------------------------
class Ipv4DhcpRelayOption82CircuitIdFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'circuit-id encoding ( %x | %p )'
   noOrDefaultSyntax = 'circuit-id encoding ...'
   data = {
      'circuit-id' : 'CircuitID (Option 82)',
      'encoding' : matcherEncoding,
      '%x' : matcherHexKw,
      '%p' : matcherStrKw,
   }
   handler = "DhcpRelayHelperHandler.setIpv4DhcpRelayOption82CircuitIdFormat"
   noOrDefaultHandler = \
         "DhcpRelayHelperHandler.noIpv4DhcpRelayOption82CircuitIdFormat"

if DhcpRelayToggleLib.toggleDhcpRelayOpt82RemoteIdV4Enabled():
   InformationOptionMode.addCommandClass( 
         Ipv4DhcpRelayOption82CircuitIdFormatCmd )

#--------------------------------------------------------------------------------
# (config-information-option) [ no | default ]
#      remote-id encoding ( %x | %p )
#--------------------------------------------------------------------------------
class Ipv4DhcpRelayOption82RemoteIdFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'remote-id encoding ( %x | %p )'
   noOrDefaultSyntax = 'remote-id encoding ...'
   data = {
      'remote-id' : 'RemoteID (Option 82)',
      'encoding' : matcherEncoding,
      '%x' : matcherHexKw,
      '%p' : matcherStrKw,
   }
   handler = "DhcpRelayHelperHandler.setIpv4DhcpRelayOption82RemoteIdFormat"
   noOrDefaultHandler = \
         "DhcpRelayHelperHandler.noIpv4DhcpRelayOption82RemoteIdFormat"

if DhcpRelayToggleLib.toggleDhcpRelayOpt82RemoteIdV4Enabled():
   InformationOptionMode.addCommandClass(
         Ipv4DhcpRelayOption82RemoteIdFormatCmd )

#--------------------------------------------------------------------------------
# (config-information-option) [ no | default ] vendor-option
#--------------------------------------------------------------------------------
class Ipv4DhcpRelayOption82VendorSpecificSubOptCmd( CliCommand.CliCommandClass ):
   syntax = 'vendor-option'
   noOrDefaultSyntax = syntax
   data = {
      'vendor-option' : 'Enable vendor specific suboption(Option 82)',
   }
   handler = "DhcpRelayHelperHandler.setIpv4DhcpRelayOption82VendorSpecificSubOpt"
   noOrDefaultHandler = \
         "DhcpRelayHelperHandler.noIpv4DhcpRelayOption82VendorSpecificSubOpt"

if DhcpRelayToggleLib.toggleDhcpRelayOpt82RemoteIdV4Enabled() and\
   DhcpLibToggleLib.toggleDhcpRelayUnnumberedIntfEnabled():
   InformationOptionMode.addCommandClass(
         Ipv4DhcpRelayOption82VendorSpecificSubOptCmd )

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] tunnel requests disabled
#--------------------------------------------------------------------------------
class DhcpRelayGlobalTunnelRequestsDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel requests disabled'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel' : 'Configure tunnel setting for DHCP relay',
      'requests' : 'DHCP requests',
      'disabled' : 'Disable processing DHCP request received from tunnel'
   }

   handler = "DhcpRelayHelperHandler.setTunnelRequestsDisabled"
   noOrDefaultHandler = handler

DhcpRelayGlobalMode.addCommandClass( DhcpRelayGlobalTunnelRequestsDisabledCmd )

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] mlag peer-link requests disabled
#--------------------------------------------------------------------------------
class MlagPeerLinkRequestsDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'mlag peer-link requests disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mlag' : 'Configure MLAG setting for DHCP Relay',
      'peer-link' : 'Configure MLAG peer-link setting for DHCP Relay',
      'requests' : 'DHCP requests',
      'disabled' : 'Disable processing DHCP request received over a MLAG peer-link'
   }

   handler = "DhcpRelayHelperHandler.setMlagPeerLinkRequestsDisabled"
   noOrDefaultHandler = handler
DhcpRelayGlobalMode.addCommandClass( MlagPeerLinkRequestsDisabledCmd )

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [no|default] client requests flooding suppression
# vlan <vlan on vlanset>
#--------------------------------------------------------------------------------
def dhcpRelayReqFloodDisableHwStatusReady( sliceInfo, hwStatus ):
   if sliceInfo and ( sliceInfo.generationId == hwStatus.genId ) and \
         hwStatus.dhcpReqFloodingDisableSupported:
      return True
   elif not sliceInfo and hwStatus.dhcpReqFloodingDisableSupported and \
         ( hwStatus.genId == 0xffffffff ):
      # For a system that creates only one instance of HwStatus, the
      # genId should be set to the max value.
      return True
   else:
      return False

def dhcpRelayReqFloodDisableSupported():
   if not dhcpRelayReqFloodDisableHwStatusDir:
      return False
   for sliceId, hwStatus in dhcpRelayReqFloodDisableHwStatusDir.items():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if dhcpRelayReqFloodDisableHwStatusReady( sliceInfo, hwStatus ):
         return True
   return False

def dhcpRelayReqFloodDisableGuard( mode, token ):
   if dhcpRelayReqFloodDisableSupported():  
      return None
   return CliParser.guardNotThisPlatform

class ReqFloodDisableVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'client requests flooding suppression vlan VLANSET'
   noOrDefaultSyntax = 'client requests flooding suppression vlan [ VLANSET ]'
   data = {
         'client' : "Configure DHCP client request settings",
         'requests' : 'DHCP requests',
         'flooding' : 'DHCP requests flooding',
         'suppression' : CliCommand.guardedKeyword( 'suppression',
               helpdesc='Disable DHCP request flooding to other interfaces in VLAN',
               guard=dhcpRelayReqFloodDisableGuard ),
         'vlan' : matcherVlan,
         'VLANSET' : VlanCli.vlanSetMatcher
   }

   handler = "DhcpRelayHelperHandler.setReqFloodDisableVlan"
   noOrDefaultHandler = "DhcpRelayHelperHandler.resetReqFloodDisableVlan"

DhcpRelayGlobalMode.addCommandClass( ReqFloodDisableVlanCmd )

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] ipv6 routes persistence disabled
#--------------------------------------------------------------------------------
class PdRoutesPersistenceDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 routes persistence disabled'
   noOrDefaultSyntax = syntax
   data = {
         'ipv6' : ipv6MatcherForConfig,
         'routes' : 'DHCPv6 Routes',
         'persistence' : 'DHCPv6 Routes Persistence',
         'disabled' : 'Disable DHCP Routes Persistence'
   }

   handler = "DhcpRelayHelperHandler.setPersistenceConfig"
   noOrDefaultHandler = "DhcpRelayHelperHandler.unsetPersistenceConfig"

DhcpRelayGlobalMode.addCommandClass( PdRoutesPersistenceDisabledCmd )

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] vss control disabled
#--------------------------------------------------------------------------------
class DhcpRelayGlobalVssControlDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'vss control disabled'
   noOrDefaultSyntax = syntax
   data = {
         'vss' : 'Configure virtual subnet selection for DHCP relay',
         'control' : 'Control subOption 152',
         'disabled' : 'Disable adding control suboption 152 to DHCP packet',
   }

   handler = "DhcpRelayHelperHandler.setVssControlDisabled"
   noOrDefaultHandler = "DhcpRelayHelperHandler.unsetVssControlDisabled"

DhcpRelayGlobalMode.addCommandClass( DhcpRelayGlobalVssControlDisabledCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   # Global variable undefined so far
   global dhcpRelayConfigDir
   global dhcpRelayCounterConfigDir
   global dhcpFwdConfig
   global dhcpRelayStatusDir
   global mlagDhcpRelayStatusDir
   global platformHardwareSliceDir
   global dhcpRelayReqFloodDisableHwStatusDir

   # pkgdeps: rpmwith %{_libdir}/SysdbMountProfiles/DhcpRelay
   dhcpRelayConfigDir = ConfigMount.mount( entityManager,
                                           "ip/helper/dhcprelay/config",
                                           "Ip::Helper::DhcpRelay::Config", "w" )
   dhcpRelayCounterConfigDir = LazyMount.mount( entityManager,
                                    "ip/helper/dhcprelay/counterConfig",
                                    "Ip::Helper::DhcpRelay::CounterConfig", "w" )
   # check the config for the incompatible dhcp-forwarder
   dhcpFwdConfig = LazyMount.mount( entityManager, "ip/dhcp/relay/config",
                                    "Ip::Dhcp::Relay::Config", "r" )
   dhcpRelayStatusDir = LazyMount.mount( entityManager, "ip/helper/dhcprelay/status",
                                         "Ip::Helper::DhcpRelay::Status", "r" )
   mlagDhcpRelayStatusDir = LazyMount.mount( entityManager,
                                             "ip/helper/dhcprelay/mlagStatus",
                                             "MlagDhcpRelay::Status", "r" )
   IntfCli.Intf.registerDependentClass( DhcpRelayIntf )
   platformHardwareSliceDir = LazyMount.mount(
         entityManager, "platform/hardware/slice",
         "Hardware::PlatformSliceDir", "r" )
   dhcpRelayReqFloodDisableHwStatusDir = LazyMount.mount(
         entityManager, "ip/helper/dhcprelay/hardware/status",
         "Tac::Dir", "ri" )
