#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import EthAddr
from ArnetModel import Ip4Address, Ip6Address
from ArnetModel import IpGenericPrefix
from ArnetModel import MacAddress
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from IntfModels import Interface
from EosDhcpServerLib import (
      aristaOuis,
      convertLeaseSeconds,
      defaultVendorId,
      filterClientClasses,
      tftpServerOptions,
      vendorSubOptionType,
      featureEchoClientId,
      )
from Ethernet import convertMacAddrToDisplay
import TableOutput
import Tac
import codecs
import time

# all characters in string.printable except '\n\r\x0b\x0c'
printableSet = set( '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!'
                    '"#$%&\'()*+,-./:;<=>?@[\\]^_`{|}~ \t' )

leaseErrorString = "Unable to fetch DHCPv{} leases, please re-run the show command"

def dhcpTimestampStr( seconds ):
   return time.strftime( "%Y/%m/%d %H:%M:%S UTC", time.gmtime( seconds ) )

class DhcpServerClientClassModel( Model ):
   # This model currently only contains fields we wish to render
   inactiveReason = Str( help='Inactive Reason' )

class DhcpServerBaseModel( Model ):
   clientClasses = Dict(
         optional=True, keyType=str,
         valueType=DhcpServerClientClassModel,
         help="Dictionary of Client Class names to inactive reasons" )

   def renderClientClasses( self, configType, activeStr, inactiveStr ):
      print( "Active", configType, "client classes:", activeStr )
      print( "Inactive", configType, "client classes:", inactiveStr )

class DhcpServerRangeBaseModel( DhcpServerBaseModel ):
   def render( self ):
      toPrint = [ "Range:", self.startRangeAddress, "to", self.endRangeAddress ]
      if self.assignedLeases is not None:
         count = f"({self.assignedLeases}/{self.totalLeases}"
         end = [ count, "addresses", "leased)" ]
         toPrint += end
      print( *toPrint )
      # Range Client Classes
      clientClasses = self.clientClasses
      if clientClasses:
         activeClassesStr, inactiveClassesStr = filterClientClasses( clientClasses )
         self.renderClientClasses( "range", activeClassesStr, inactiveClassesStr )

class DhcpServerIpv4Range( DhcpServerRangeBaseModel ):
   startRangeAddress = Ip4Address( help="Start address" )
   endRangeAddress = Ip4Address( help="End address" )
   assignedLeases = Int( help="Number of assigned leases in range", optional=True )
   totalLeases = Int( help="Number of total leases in range", optional=True )

class DhcpServerIpv6Range( DhcpServerRangeBaseModel ):
   startRangeAddress = Ip6Address( help="Start address" )
   endRangeAddress = Ip6Address( help="End address" )
   assignedLeases = Int( help="Number of assigned leases in range", optional=True )
   totalLeases = Int( help="Number of total leases in range", optional=True )

# vendor-option
class DhcpServerSubOptionModel( Model ):
   optionCode = Int( help="Sub-option code" )
   optionType = Enum( values=( vendorSubOptionType.string,
                      vendorSubOptionType.ipAddress ),
                      help="Sub-option type" )
   optionDataIpAddresses = List( optional=True, valueType=Ip4Address,
                           help="IPv4 addresses for sub-option type 'ipv4-address'" )
   optionDataString = Str( optional=True,
                           help="String for sub-option type 'string'" )

class DhcpServerIpv4VendorOptionModel( Model ):
   subOptions = List( valueType=DhcpServerSubOptionModel,
                      help="Sub-options configured" )

   def render( self ):
      headings = ( "Sub-options", "Data" )
      fmt_center = TableOutput.Format( justify="center" )
      table = TableOutput.createTable( headings )
      table.formatColumns( fmt_center, fmt_center )
      for subOption in self.subOptions:
         if subOption.optionType == vendorSubOptionType.string:
            # add quotes to display
            data = f'"{subOption.optionDataString}"'
         elif subOption.optionType == vendorSubOptionType.ipAddress:
            data = ', '.join( subOption.optionDataIpAddresses )
         else:
            assert False, 'Unknown sub-option type'

         table.newRow( subOption.optionCode, data )

      print( table.output() )

# reservations mac-address
class DhcpServerIpv4ReservationsMacAddressModel( Model ):
   macAddress = MacAddress( help="MAC address of client" )
   ipv4Address = Ip4Address( optional=True,
                             help="Reserved IPv4 address" )
   hostname = Str( optional=True,
                   help="Reserved hostname" )

   def render( self ):
      print( f"MAC address: {self.macAddress.displayString}" )
      if self.ipv4Address:
         print( f"IPv4 address: {str( self.ipv4Address )}" )
      if self.hostname:
         print( f"Hostname: {str( self.hostname )}" )

class DhcpServerIpv6ReservationsMacAddressModel( Model ):
   macAddress = MacAddress( help="MAC address of client" )
   ipv6Address = Ip6Address( optional=True,
                             help="Reserved IPv6 address" )
   hostname = Str( optional=True,
                   help="Reserved hostname" )

   def render( self ):
      print( f"MAC address: {self.macAddress.displayString}" )
      if self.ipv6Address:
         print( f"IPv6 address: {str( self.ipv6Address )}" )
      if self.hostname:
         print( f"Hostname: {str( self.hostname )}" )

class DhcpServerSubnetBaseModel( DhcpServerBaseModel ):
   subnet = IpGenericPrefix( help="Subnet" )
   activeLeases = Int( help="Number of active leases for this subnet" )
   name = Str( help='Subnet name' )
   leaseDuration = Float( optional=True,
                          help="Duration of leases (seconds)" )
   overlappingSubnets = List( valueType=IpGenericPrefix, optional=True,
                              help="Overlapping subnets" )
   unknownSubnet = Bool( optional=True, help="Subnet is unknown" )

   def doAfSpecificRender( self ):
      pass

   def renderReservationsMacAddress( self ):
      # The code to print is identical for IPv4 and IPv6, but this function is called
      # by doAfSpecificRender() instead of _render() to keep the show command
      # consistent with the original order of the show command.
      if self.reservationsMacAddress:
         print( "Reservations:" )
         for macAddr in sorted( self.reservationsMacAddress ):
            self.reservationsMacAddress[ macAddr ].render()
            print()

   def _render( self, ipVersion ):
      # Set subnet message and status
      subnetMsg = f"Subnet: {self.subnet}"

      # Unknown Subnets
      subnetStatusMsg = ''
      if self.unknownSubnet:
         subnetStatusMsg = 'Unknown'

      # Disabled Message
      if self.disabledMessage:
         subnetStatusMsg = self.disabledMessage

      if subnetStatusMsg:
         subnetMsg = subnetMsg + f" ({subnetStatusMsg})"

      print( subnetMsg )
      print( f'Subnet name: {self.name}' )

      # Subnet Client Classes
      clientClasses = self.clientClasses
      if clientClasses:
         activeClassesStr, inactiveClassesStr = filterClientClasses( clientClasses )
         self.renderClientClasses( "subnet", activeClassesStr, inactiveClassesStr )

      for r in self.ranges:
         r.render()

      print( "DNS server(s): {}".format( " ".join( list( map(
         str, self.dnsServers ) ) ) ) )

      if self.leaseDuration:
         print( 'Lease duration: {} days {} hours {} minutes'.format(
            *convertLeaseSeconds( self.leaseDuration ) ) )

      self.doAfSpecificRender()

      if self.disabledMessage:
         # print IPv4 and IPv6 shared disabledMessages
         print( "Disabled reason(s):" )

         if self.duplicateReservedIp:
            # duplicate reserved ip address
            disabledReason = "Duplicate IPv{} address reservation: {}"
            print( disabledReason.format( ipVersion, self.duplicateReservedIp ) )

         if self.invalidRange:
            # invalid ranges
            rangeStart = self.invalidRange.startRangeAddress
            rangeEnd = self.invalidRange.endRangeAddress
            invalidStr = f"{rangeStart}-{rangeEnd}"
            print( f"Invalid range: {invalidStr}" )

         if self.overlappingRange:
            # overlapping ranges
            rangeStart = self.overlappingRange.startRangeAddress
            rangeEnd = self.overlappingRange.endRangeAddress
            overlappedStr = f"{rangeStart}-{rangeEnd}"
            print( f"Overlapping range: {overlappedStr}" )

         if self.overlappingSubnets:
            # Overlapping subnet(s)
            overlappedList = list( map( str, self.overlappingSubnets ) )
            overlappedStr = " ".join( sorted( overlappedList ) )
            print( f"Overlapping subnets: {overlappedStr}" )

class DhcpServerIpv4SubnetModel( DhcpServerSubnetBaseModel ):
   ranges = List( valueType=DhcpServerIpv4Range,
                  help="Ranges configured" )
   disabledMessage = Str( optional=True,
                          help="Why IPv4 subnet is disabled if configured" )
   dnsServers = List( valueType=Ip4Address,
                      help="DNS servers" )
   defaultGateway = Ip4Address( help="Default gateway address" )
   tftpServerOption66 = Str( optional=True, help="TFTP server option 66" )
   tftpServerOption150 = List( optional=True, valueType=Ip4Address,
                               help="TFTP server option 150" )
   tftpBootFile = Str( optional=True, help="TFTP boot file" )
   reservationsMacAddress = Dict(
         optional=True, keyType=MacAddress,
         valueType=DhcpServerIpv4ReservationsMacAddressModel,
         help="Dictionary of MAC addresses to reserved IPv4 addresses" )
   duplicateReservedIp = Ip4Address( optional=True,
                                     help="Duplicate reserved IPv4 address" )
   invalidRange = Submodel( valueType=DhcpServerIpv4Range, optional=True,
                             help="Invalid ranges" )
   overlappingRange = Submodel( valueType=DhcpServerIpv4Range, optional=True,
                             help="Overlapping ranges" )

   def render( self ):
      self._render( 4 )

   def doAfSpecificRender( self ):
      print( f'Default gateway address: {self.defaultGateway}' )

      serverOptions = tftpServerOptions( self.tftpServerOption66,
                                         self.tftpServerOption150 )
      if serverOptions:
         print( serverOptions )

      if self.tftpBootFile:
         print( f"TFTP boot file: {self.tftpBootFile}" )

      print( f"Active leases: {self.activeLeases}" )

      self.renderReservationsMacAddress()

class DhcpServerIpv6SubnetModel( DhcpServerSubnetBaseModel ):
   ranges = List( valueType=DhcpServerIpv6Range,
                  help="Ranges configured" )
   disabledMessage = Str( optional=True,
                          help="Why IPv6 subnet is disabled if configured" )
   dnsServers = List( valueType=Ip6Address,
                      help="DNS servers" )
   directActive = Bool( help="IPv6 server responds to direct requests" )
   directActiveDetail = Str( optional=True, help="Details of direct status" )
   relayActive = Bool( help="IPv6 server responds to relay requests" )
   relayActiveDetail = Str( optional=True, help="Details of relay status" )
   reservationsMacAddress = Dict(
         optional=True, keyType=MacAddress,
         valueType=DhcpServerIpv6ReservationsMacAddressModel,
         help="Dictionary of MAC addresses to reserved IPv6 addresses" )
   duplicateReservedIp = Ip6Address( optional=True,
                                     help="Duplicate reserved IPv6 address" )
   invalidRange = Submodel( valueType=DhcpServerIpv6Range, optional=True,
                             help="Invalid ranges" )
   overlappingRange = Submodel( valueType=DhcpServerIpv6Range, optional=True,
                             help="Overlapping ranges" )

   def render( self ):
      self._render( 6 )

   def doAfSpecificRender( self ):
      directStr = "Active" if self.directActive else "Inactive"
      if self.directActiveDetail:
         directStr = f"{directStr} ({self.directActiveDetail})"

      relayStr = "Active" if self.relayActive else "Inactive"
      if self.relayActiveDetail:
         relayStr = f"{relayStr} ({self.relayActiveDetail})"

      print( f"Direct: {directStr}" )
      print( f"Relay: {relayStr}" )
      print( f"Active leases: {self.activeLeases}" )

      self.renderReservationsMacAddress()

class DhcpServerInterfaceStatusShowModel( Model ):
   active = Bool( help="This interface is active" )
   detail = Str( optional=True,
                 help="Details of interface activity" )

class DhcpServerShowBaseModel( DhcpServerBaseModel ):
   debugLog = Bool( optional=True, help="Debug log is enabled" )

   def _render( self, ipVersion, serverActive, activeLeases, interfaces,
                disabledMessage, subnets, dnsServers, domainName, leaseDuration,
                tftpOptions=None, vendorOptions=None, clientClasses=None,
                echoClientId=True ):
      serverActiveStr = 'active' if serverActive else 'inactive'
      msg = f'IPv{ipVersion} DHCP server is {serverActiveStr}'

      if disabledMessage:
         msg = msg + f' ({disabledMessage})'

      print( msg )
      if self.debugLog:
         print( "Debug log is enabled" )
      print( 'DNS server(s): {}'.format( ' '.join( list( map(
         str, dnsServers ) ) ) ) )
      print( f'DNS domain name: {domainName}' )
      if ipVersion == 4 and featureEchoClientId():
         echoClientIdStr = 'enabled' if echoClientId else 'disabled'
         msg = f'Client ID echo: {echoClientIdStr}'
         print( msg )
      print( 'Lease duration: {} days {} hours {} minutes'.format(
         *convertLeaseSeconds( leaseDuration ) ) )

      if tftpOptions:
         tftpServerOption66 = tftpOptions.get( 'serverOption66' )
         tftpServerOption150 = tftpOptions.get( 'serverOption150' )
         serverOptions = tftpServerOptions( tftpServerOption66,
                                            tftpServerOption150 )
         if serverOptions:
            print( serverOptions )

         tftpBootFile = tftpOptions.get( 'file' )
         if tftpBootFile:
            print( f"TFTP file: {tftpBootFile}" )

      print( f'Active leases: {activeLeases}' )
      print( f"IPv{ipVersion} DHCP interface status:" )

      headings = ( "Interface", "Status" )
      table = TableOutput.createTable( headings )
      fmt_left = TableOutput.Format( justify="left" )
      table.formatColumns( fmt_left, fmt_left )
      for intf, status in sorted( interfaces.items() ):
         statusMsg = "Active" if status.active else "Inactive"
         detail = ""
         if status.detail:
            detail = f" ({status.detail})"
         rowMsg = f"{statusMsg}{detail}"
         table.newRow( intf, rowMsg )

      print( table.output() )

      # Global Client Classes
      if clientClasses:
         activeClassesStr, inactiveClassesStr = filterClientClasses( clientClasses )
         self.renderClientClasses( "global", activeClassesStr, inactiveClassesStr )

      # Vendor Options
      if vendorOptions:
         print( "Vendor information:" )
         defaultSubOptions = vendorOptions.pop( defaultVendorId, None )
         if defaultSubOptions:
            # print default vendor-option first
            print( f"Vendor ID: {defaultVendorId}" )
            defaultSubOptions.render()

         for vendorId in sorted( vendorOptions ):
            print( f"Vendor ID: {vendorId}" )
            vendorOptions[ vendorId ].render()

      for subnet in sorted( subnets.values(),
                            key=lambda subnetModel: subnetModel.subnet ):
         print()
         subnet.render()
      print()

class DhcpServerIpv4ShowModel( DhcpServerShowBaseModel ):
   ipv4ServerActive = Bool( default=False,
                            help="IPv4 DHCP server is active" )
   ipv4ActiveLeases = Int( default=0,
                           help="Number of active leases" )
   ipv4Interfaces = Dict( keyType=Interface,
                          valueType=DhcpServerInterfaceStatusShowModel,
                          help="A mapping of IPv4 DHCP interfaces to their status" )
   ipv4DisabledMessage = Str( optional=True,
                              help="Why IPv4 DHCP server is disabled if configured" )
   ipv4Subnets = Dict( keyType=IpGenericPrefix,
                       valueType=DhcpServerIpv4SubnetModel,
                       help="Dictionary of IPv4 subnets configured" )
   ipv4DnsServers = List( valueType=Ip4Address,
                          help="DNS servers" )
   ipv4DomainName = Str( default='', help='DNS domain name' )
   ipv4LeaseDuration = Float( default=7200.0,
                              help="Duration of leases (seconds)" )
   ipv4TftpServerOption66 = Str( optional=True, help="TFTP Server option 66" )
   ipv4TftpServerOption150 = List( valueType=Ip4Address, optional=True,
                                  help="TFTP Server option 150" )
   ipv4TftpBootFile = Str( optional=True, help="TFTP boot file" )

   ipv4VendorOptions = Dict( optional=True,
                             keyType=str,
                             valueType=DhcpServerIpv4VendorOptionModel,
                             help="Dictionary of IPv4 vendor-options configured, \
                                   mapping vendor-option to sub-options" )

   if featureEchoClientId():
      ipv4EchoClientId = Bool( default=True,
                            help="IPv4 DHCP server will send back client-id" )

   def render( self ):
      tftpOptions = {}
      if self.ipv4TftpServerOption66:
         tftpOptions[ 'serverOption66' ] = self.ipv4TftpServerOption66

      if self.ipv4TftpServerOption150:
         tftpOptions[ 'serverOption150' ] = self.ipv4TftpServerOption150

      if self.ipv4TftpBootFile:
         tftpOptions[ 'file' ] = self.ipv4TftpBootFile

      if featureEchoClientId():
         self._render( 4, self.ipv4ServerActive, self.ipv4ActiveLeases,
                      self.ipv4Interfaces, self.ipv4DisabledMessage,
                      self.ipv4Subnets, self.ipv4DnsServers, self.ipv4DomainName,
                      self.ipv4LeaseDuration, tftpOptions, self.ipv4VendorOptions,
                      self.clientClasses, self.ipv4EchoClientId )
      else:
         self._render( 4, self.ipv4ServerActive, self.ipv4ActiveLeases,
                      self.ipv4Interfaces, self.ipv4DisabledMessage,
                      self.ipv4Subnets, self.ipv4DnsServers, self.ipv4DomainName,
                      self.ipv4LeaseDuration, tftpOptions, self.ipv4VendorOptions,
                      self.clientClasses )

class DhcpServerIpv6ShowModel( DhcpServerShowBaseModel ):
   ipv6ServerActive = Bool( default=False,
                            help="IPv6 DHCP server is active" )
   ipv6ActiveLeases = Int( default=0,
                           help="Number of active leases" )
   ipv6Interfaces = Dict( keyType=Interface,
                          valueType=DhcpServerInterfaceStatusShowModel,
                          help="A mapping of IPv4 DHCP interfaces to their status" )
   ipv6DisabledMessage = Str( optional=True,
                          help="Why IPv6 DHCP server is disabled if configured" )
   ipv6Subnets = Dict( keyType=IpGenericPrefix,
                       valueType=DhcpServerIpv6SubnetModel,
                       help="Dictionary of IPv6 subnets configured" )
   ipv6DnsServers = List( valueType=Ip6Address,
                      help="DNS servers" )
   ipv6DomainName = Str( default='', help='DNS domain name' )
   ipv6LeaseDuration = Float( default=7200.0,
                              help="Duration of leases (seconds)" )

   def render( self ):
      self._render( 6, self.ipv6ServerActive, self.ipv6ActiveLeases,
                    self.ipv6Interfaces, self.ipv6DisabledMessage, self.ipv6Subnets,
                    self.ipv6DnsServers, self.ipv6DomainName,
                    self.ipv6LeaseDuration, clientClasses=self.clientClasses )

class DhcpServerShowModel( Model ):
   ipv4Server = Submodel( valueType=DhcpServerIpv4ShowModel,
                          help="IPv4 DHCP server information" )
   ipv6Server = Submodel( valueType=DhcpServerIpv6ShowModel,
                          help="IPv6 DHCP server information" )
   addressFamily = Str( optional=True, help='IP version of the server called' )

   def render( self ):
      servers = { 'ipv4': self.ipv4Server,
                  'ipv6': self.ipv6Server }
      if self.addressFamily:
         servers[ self.addressFamily ].render()
      else:
         self.ipv4Server.render()
         self.ipv6Server.render()

class DhcpServerShowVrfModel( Model ):
   vrfs = Dict( keyType=str, valueType=DhcpServerShowModel,
                help="DHCP server information for all vrfs" )
   _all = Bool( help="Whether all VRFs were requested" )

   def render( self ):
      for vrf, model in sorted( self.vrfs.items() ):
         if self._all:
            print( "VRF", vrf )
            print( "----------------" )
         model.render()

class BaseLeases( Model ):
   endLeaseTime = Float( help="End time of lease" )
   lastTransaction = Float( help="Last transaction time with client" )
   macAddress = MacAddress( help="MAC address of client" )

   def render( self ):
      print( str( self.ipAddress ) )
      print( "End: {}".format(
         dhcpTimestampStr( self.endLeaseTime ) ) )
      print( "Last transaction: {}".format(
         dhcpTimestampStr( self.lastTransaction ) ) )
      print( f"MAC address: {self.macAddress.displayString}" )

def hexToAscii( hexVal, offset=2 ):
   # Second decode() is needed in python 3 to convert from bytes to str. In python 2,
   # the second decode() will convert from str to unicode, which is still fine for
   # printing.
   return codecs.decode( hexVal[ offset : ], 'hex' ).decode()

def printAsciiOrHex( header, hexString ):
   '''
   Prints the hexString as ASCII if all characters are printable as ASCII and prints
   it as hex if not.
   '''
   if not hexString:
      # If this is an empty string, just print the header
      print( f'{header}:' )
      return
   try:
      asciiString = hexToAscii( hexString )
      asciiPrintable = set( asciiString ).issubset( printableSet )
   except UnicodeDecodeError:
      asciiPrintable = False
   if asciiPrintable:
      print( "{} (ASCII): {} ({})".format( header, asciiString,
                                           hexString ) )
   else:
      print( f"{header}: {hexString}" )

class Option82SubOption( Model ):
   typeCode = Int( help="Unknown sub-option type code" )
   value = Str( help="Unknown sub-option value in hex" )

   def render( self ):
      printAsciiOrHex( f"Information option sub-option {self.typeCode}",
                       self.value )

class Option82( Model ):
   circuitIds = List( valueType=str, help="Circuit ID in hex", optional=True )
   remoteIds = List( valueType=str, help="Remote ID in hex", optional=True )
   # Note that even though keyType is set to an int, in the json output, the key will
   # be outputted as a string.
   unknownSubOpts = List( valueType=Option82SubOption,
                          help="Unknown sub-options", optional=True )

   def render( self ):
      if ( self.circuitIds and self.remoteIds and not self.unknownSubOpts and
           len( self.circuitIds ) == 1 and len( self.remoteIds ) == 1 and
           len( self.remoteIds[ 0 ] ) == 40 ):
         # Check if this is an arista remote ID. First 2 characters is 0x. Next 4 is
         # the length byte. After that is the MAC address.
         try:
            remoteId = self.remoteIds[ 0 ]
            macStr = hexToAscii( remoteId, offset=6 )
            if hex( EthAddr( macStr ).oui() ) in aristaOuis:
               circuitId = self.circuitIds[ 0 ]
               print( "Circuit ID (Arista): {} ({})".format( hexToAscii( circuitId ),
                                                             circuitId ) )
               print( "Remote ID (Arista): {} ({})".format(
                         convertMacAddrToDisplay( macStr ), remoteId ) )
               return
         except UnicodeDecodeError:
            pass

      if self.circuitIds:
         for circuitId in self.circuitIds:
            printAsciiOrHex( "Circuit ID", circuitId )

      if self.remoteIds:
         for remoteId in self.remoteIds:
            printAsciiOrHex( "Remote ID", remoteId )

      if self.unknownSubOpts:
         for unknownSubOpt in self.unknownSubOpts:
            unknownSubOpt.render()

class Ipv4Leases( BaseLeases ):
   ipAddress = Ip4Address( help="IP address" )
   option82 = Submodel( valueType=Option82, help="Information Option",
                        optional=True )

   def render( self ):
      BaseLeases.render( self )
      if self.option82:
         self.option82.render()

class Ipv6Leases( BaseLeases ):
   ipAddress = Ip6Address( help="Ipv6 Address" )

class DhcpServerShowLeasesModel( Model ):
   ipv4ActiveLeases = List( valueType=Ipv4Leases,
                            help="Active IPv4 Leases" )
   ipv6ActiveLeases = List( valueType=Ipv6Leases,
                            help="Active IPv6 Leases" )
   ipv4LeaseError = Bool( default=False,
                          help="There was an error grabbing IPv4 leases" )
   ipv6LeaseError = Bool( default=False,
                          help="There was an error grabbing IPv4 leases" )

   def render( self ):
      if self.ipv4LeaseError:
         print( leaseErrorString.format( '4' ) )
         print()
      else:
         for lease in self.ipv4ActiveLeases:
            lease.render()
            print()
      if self.ipv6LeaseError:
         print( leaseErrorString.format( '6' ) )
         print()
      else:
         for lease in self.ipv6ActiveLeases:
            lease.render()
            print()

class DhcpServerShowVrfLeasesModel( Model ):
   vrfs = Dict( keyType=str, valueType=DhcpServerShowLeasesModel,
                help="DHCP server lease information for all vrfs" )
   _all = Bool( help="Whether all VRFs were requested" )

   def render( self ):
      for vrf, leases in sorted( self.vrfs.items() ):
         if self._all:
            print( "Leases in vrf", vrf )
            print( "----------------" )
         leases.render()
