# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Ip
import ConfigMount
import CliExtensions
from CliPlugin.ClassificationCliLib import (
      CommitAbortModelet,
      FieldSetIpPrefixBaseConfigCmd,
      FieldSetIpPrefixConfigBase,
      protectedFieldSetNamesRegex,
)
from CliMode.DirectedBroadcast import (
      DirectedBroadcastConfigMode,
      FieldSetIpPrefixConfigMode,
)

ipDirectedBroadcastHostsConfig = None
ipMatcherForConfig = CliToken.Ip.ipMatcherForConfig
configMode = BasicCli.GlobalConfigMode

# -------------------------------------------------------------------------------
# CLI guard
# -------------------------------------------------------------------------------
hardwareIpDbAclModeUnsupportedHook = CliExtensions.CliHook()

def hardwareIpDbAclModeUnsupportedGuard( mode, token ):
   for hook in hardwareIpDbAclModeUnsupportedHook.extensions():
      if hook( mode, token ):
         return CliParser.guardNotThisPlatform
   return None

# -------------------------------------------------------------------------------
# Keyword tokens and help msg
# -------------------------------------------------------------------------------
directedbcastConfigToken = CliCommand.guardedKeyword( 'directed-broadcast',
      helpdesc='Configure Global Directed Broadcast Rules',
      guard=hardwareIpDbAclModeUnsupportedGuard )

# -------------------------------------------------------------------------------
# The "ip directed-broadcast" to configure global directed-broadcast rules.
# -------------------------------------------------------------------------------
class IpDirectedBroadcastConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ip directed-broadcast'
   noOrDefaultSyntax = syntax
   data = { 'ip': ipMatcherForConfig,
            'directed-broadcast': directedbcastConfigToken
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( DirectedBroadcastConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipDirectedBroadcastHostsConfig.fieldSetIpPrefix.clear()
      ipDirectedBroadcastHostsConfig.currentFieldSetName = ""

# -------------------------------------------------------------------------------
# The "field-set ipv4 prefix FIELD_NAME" command to configure hosts.
# -------------------------------------------------------------------------------
def getIpPrefixFieldSetNames( mode ):
   return ipDirectedBroadcastHostsConfig.fieldSetIpPrefix

ipPrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   "IP prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIpPrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   syntax = 'field-set ipv4 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "ipdb"
   data = {
      'FIELD_SET_NAME': ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName': fieldSetIpPrefixName,
         'fieldSetConfig': ipDirectedBroadcastHostsConfig,
         'setType': setType,
         'childMode': FieldSetIpPrefixConfigMode,
         'featureName': "ipdb",
         # Enables config rollback in the IpPrefixFieldSetContext.
         'rollbackSupported': True,
      }

# -------------------------------------------------------------------------------
# The "source host field-set FIELD_SET_NAME action forward" command .
# This command installs the given field-set config in hardware.
# -------------------------------------------------------------------------------
class SourceFieldSetCmd( CliCommand.CliCommandClass ):
   syntax = 'source host field-set FIELD_SET_NAME action forward'
   noOrDefaultSyntax = 'source host field-set [ FIELD_SET_NAME [ action forward ] ]'
   data = {
      'source': 'Source',
      'host': 'Configure Hosts for Directed Broadcast',
      'field-set': 'Field Set',
      'FIELD_SET_NAME': ipPrefixFieldSetNameMatcher,
      'action': 'Set the action for the selected field-set',
      'forward': 'Permit hosts from selected field-set ',
   }

   @staticmethod
   def handler( mode, args ):
      fieldSetName = args[ 'FIELD_SET_NAME' ]
      ipDirectedBroadcastHostsConfig.currentFieldSetName = fieldSetName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      fieldSetName = args.get( 'FIELD_SET_NAME' )
      if ( not fieldSetName or
            fieldSetName == ipDirectedBroadcastHostsConfig.currentFieldSetName ):
         ipDirectedBroadcastHostsConfig.currentFieldSetName = ""

class FieldSetIpPrefixConfigNoAuthCmds( FieldSetIpPrefixConfigBase ):
   data = FieldSetIpPrefixConfigBase.data.copy()
   authz = False

   @staticmethod
   def handler( mode, args ):
      # Display the warning only if an IP address is added to the field set
      add = 'remove' not in args
      if add:
         mode.addWarning( "IP address having 32 length "
                          "prefix will only be installed" )
      FieldSetIpPrefixConfigBase.handler( mode, args )

configMode.addCommandClass( IpDirectedBroadcastConfigCmd )
DirectedBroadcastConfigMode.addCommandClass( FieldSetIpPrefixConfigCmd )
DirectedBroadcastConfigMode.addCommandClass( SourceFieldSetCmd )
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpPrefixConfigNoAuthCmds )

DirectedBroadcastConfigMode.addModelet( CommitAbortModelet )
FieldSetIpPrefixConfigMode.addModelet( CommitAbortModelet )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ipDirectedBroadcastHostsConfig

   hostsConfigType = "DirectedBroadcast::HostsConfig"
   hostsConfigPath = "directed-broadcast/config/hosts"

   ipDirectedBroadcastHostsConfig = ConfigMount.mount( entityManager,
                                                       hostsConfigPath,
                                                       hostsConfigType, "w" )
