#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliGuards
import CliMatcher
import CliToken.Clear
# pylint: disable-next=consider-using-from-import
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import ConfigMount
import LazyMount
import Tac

dmfGlobal = CliGlobal.CliGlobal( dict( config=None, tapAggHwStatus=None,
                                       policyConfig=None, policyStatus=None,
                                       flowStatus=None ) )

#----------------------------------------------------------------
# management dmf
#----------------------------------------------------------------
def dmfGuard( mode, token ):
   if dmfGlobal.tapAggHwStatus.dmfSupported:
      return None
   return CliGuards.guardNotThisPlatform

dmfMatcher = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'dmf',
                                      helpdesc='DANZ Monitoring Fabric' ),
   guard=dmfGuard )
dmfCtrlrMatcher = CliMatcher.KeywordMatcher( 'controller',
                                      helpdesc='Display DANZ Monitoring Fabric '
                                      'controller information' )

class DmfConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = "DMF configuration"

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "dmf" )

   @staticmethod
   def enableDmf( mode, args ):
      dmfGlobal.config.enabled = True

   @staticmethod
   def disableDmf( mode, args ):
      dmfGlobal.config.enabled = False

   @staticmethod
   def controllerIs( mode, args ):
      # Figure out which one(s) to remove and which one(s) to add,
      # so we don't remove/add the same address.
      oldAddrs = set( dmfGlobal.config.controller )
      newAddrs = { x for x in ( args.get( 'ADDR1' ),
                                   args.get( 'ADDR2' ) ) if x }

      for addr in oldAddrs - newAddrs:
         del dmfGlobal.config.controller[ addr ]

      for addr in newAddrs - oldAddrs:
         dmfGlobal.config.controller[ addr ] = True

####################################################################
#               management dmf
####################################################################
def gotoMgmtDmf( mode, args ):
   childMode = mode.childMode( DmfConfigMode )
   mode.session_.gotoChildMode( childMode )

class MgmtDmfCmd( CliCommand.CliCommandClass ):
   syntax = 'management dmf'
   noOrDefaultSyntax = syntax
   data = {
      'management': ConfigMgmtMode.managementKwMatcher,
      'dmf': dmfMatcher
      }
   handler = gotoMgmtDmf
   noOrDefaultHandler = DmfConfigMode.clear

BasicCli.GlobalConfigMode.addCommandClass( MgmtDmfCmd )

####################################################################
# management dmf
#      - [ no | default ] disabled
####################################################################
class DmfDisabledCmd( CliCommand.CliCommandClass ):
   syntax = "disabled"
   noSyntax = defaultSyntax = syntax
   data = {
      "disabled": "Disable DANZ Monitoring Fabric"
      }
   handler = defaultHandler = DmfConfigMode.disableDmf
   noHandler = DmfConfigMode.enableDmf

DmfConfigMode.addCommandClass( DmfDisabledCmd )

####################################################################
# management dmf
#      - [ no | default ] controller address ADDR1 [ ADDR2]
####################################################################
class DmfControllerCmd( CliCommand.CliCommandClass ):
   syntax = "controller address ADDR1 [ ADDR2 ]"
   noOrDefaultSyntax = "controller address ..."
   data = {
      "controller": "Configure DMF controller",
      "address": "Configure DMF controller address",
      "ADDR1": IpGenAddrMatcher( "Configure first controller address" ),
      "ADDR2": IpGenAddrMatcher( "Configure second controller address" )
      }
   handler = DmfConfigMode.controllerIs
   noOrDefaultHandler = handler

DmfConfigMode.addCommandClass( DmfControllerCmd )

###########################################################################
# clear management dmf flow (ingress|egress) [ TABLE ]
##########################################################################
def clearDmfFlowTableCounters( mode, table=None, egress=False ):
   if not table:
      if not egress:
         clearDmfFlowTableCounters( mode, 1 )
         clearDmfFlowTableCounters( mode, 2 )
      else:
         clearDmfFlowTableCounters( mode, 1, egress=True )
      return

   # pylint: disable-next=consider-using-f-string
   if not egress:
      counterAttr = f"clearIngFlow{table}CounterRequestTime"
   else:
      if table == 2:
         mode.addError( "Egress Flow 2 table does not exist" )
         return
      counterAttr = f"clearEgrFlow{table}CounterRequestTime"
   setattr( dmfGlobal.config, counterAttr, Tac.now() )

class ClearDmfFlowTableCounterCmd( CliCommand.CliCommandClass ):
   syntax = "clear management dmf flow (ingress|egress) [ TABLE ] counters"
   data = {
      "clear": CliToken.Clear.clearKwNode,
      "management": ConfigMgmtMode.managementClearKwMatcher,
      "dmf": dmfMatcher,
      "flow": "DMF Flow",
      "ingress": "Ingress Flow Table",
      "egress": "Egress Flow Table",
      "TABLE": CliMatcher.IntegerMatcher( 1, 2,
                                          helpdesc="Ingress Flow Table Number" ),
      "counters": "Clear DMF Flow packet counters"
   }

   @staticmethod
   def handler( mode, args ):
      table = args.get( "TABLE" )
      clearDmfFlowTableCounters( mode, table, egress="egress" in args )

BasicCli.EnableMode.addCommandClass( ClearDmfFlowTableCounterCmd )

def Plugin( entityManager ):
   dmfGlobal.config = ConfigMount.mount( entityManager,
                                         "dmf/cli/config",
                                         "Dmf::Cli::Config", "w" )
   dmfGlobal.tapAggHwStatus = LazyMount.mount( entityManager,
                                               "tapagg/hwstatus",
                                               "TapAgg::HwStatus", "r" )
   dmfGlobal.policyConfig = LazyMount.mount( entityManager,
                                             "dmf/policy/config",
                                             "Dmf::Policy::Config", "r" )
   dmfGlobal.policyStatus = LazyMount.mount( entityManager,
                                             "dmf/policy/status",
                                             "Dmf::Policy::Status", "r" )
   dmfGlobal.flowStatus = LazyMount.mount( entityManager,
                                           "dmf/indigo/flowstatus",
                                           "Dmf::Indigo::Gentable::FlowStatus", "r" )
