# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=ungrouped-imports
import BasicCli
import ShowCommand
from CliPlugin import IntfCli
from CliPlugin import ConfigMgmtMode
from CliPlugin import DmfCli
from CliPlugin import DmfIndigoCli
from CliPlugin.DmfIndigoCliModel import ( DmfIndigoModel, ControllerModel,
      DmfIndigoPacketsModel, CpuQueueModel,
      DmfInterfacesSummaryModel, DmfInterfacesModel, InterfaceModel,
      ConnectionEventModel, DmfIndigoConnectionHistoryModel )
from CliPlugin.DmfShowCli import registerDmfShowTech
from CliPlugin.IntfCli import interfacesShowKw
from CliPlugin.PtpCliModel import portStateToName
from MultiRangeRule import multiRangeFromCanonicalString
from TypeFuture import TacLazyType
import LazyMount

UnidirectionalLinkMode = TacLazyType( 'Interface::UnidirectionalLinkMode' )
SwitchportMode = TacLazyType( 'Bridging::SwitchportMode' )
EthFecEncoding = TacLazyType( 'Interface::EthFecEncoding' )
LoopbackMode = TacLazyType( 'Interface::LoopbackMode' )
PortChannelIntfId = TacLazyType( "Arnet::PortChannelIntfId" )
AutonegState = TacLazyType( "Interface::AutonegState" )
ShapeRateVal = TacLazyType( "Qos::ShapeRateVal" )

config = None
bridgingInputConfig = None
indigoStatus = None
ethPhyIntfStatus = None
qosStatus = None
lagIntfStatusDir = None
indigoCoreStatus = None
ptpStatus = None
bridgingInputDynVlanAllowedVlans = None
vxlanIntf = None
hwPhyStatusDataCoherentSlice = None

def showDmfIndigo( mode, args ):
   model = DmfIndigoModel()
   model.enabled = config.enabled
   model.active = indigoStatus.active
   model.tcamProfileStatus = indigoStatus.tcamProfile
   model.flexCounterStatus = indigoStatus.flexCounter
   model.forwardingChipDiscovery = (
      'success' if indigoStatus.forwardingChipDiscovery else 'pending' )

   for controller in indigoStatus.controller.values():
      controllerModel = ControllerModel()
      controllerModel.ipAddr = controller.ipAddr
      controllerModel.connectionStatus = controller.state
      controllerModel.connectionRole = controller.role
      model.controllers[ controller.id ] = controllerModel

   return model

def showDmfIndigoPackets( mode, args ):
   model = DmfIndigoPacketsModel()

   packetManager = indigoStatus.packetManager

   model.packetInCount = packetManager.packetInCount
   model.packetOutCount = packetManager.packetOutCount
   model.lldpInCount = packetManager.lldpInCount

   for queue in packetManager.cpuQueue.values():
      queueModel = CpuQueueModel()
      queueModel.mirrorSession = queue.mirrorSession
      queueModel.mirrorIntf = queue.mirrorIntf
      queueModel.packetInCount = queue.packetInCount
      model.queues[ queue.queueId ] = queueModel

   return model

def showDmfIndigoConnectionHistory( mode, args ):
   model = DmfIndigoConnectionHistoryModel()
   if not indigoStatus.indigoCore:
      return model
   for cxn in reversed( list( indigoStatus.indigoCore.cxnLog.values() ) ):
      event = ConnectionEventModel()
      event.time = cxn.timestamp
      event.controllerIpAddr = cxn.ipAddr
      event.protocol = cxn.protocol
      event.port = cxn.port
      event.event = cxn.event
      event.auxId = cxn.auxId
      model.connectionEvents.append( event )

   return model

def showDmfInterfacesSummary( mode, args ):
   model = DmfInterfacesSummaryModel()
   ports = set()
   if indigoStatus.portCountersDir:
      ports = iter( indigoStatus.portCountersDir.portCounters )
   for port in ports:
      interfaceModel = InterfaceModel()

      # indigoStatus -> portCounters, dmfConfigured, intfId
      portCounters = indigoStatus.portCountersDir.portCounters[ port ]
      intfId = portCounters.intfId

      interfaceModel.intf = intfId
      interfaceModel.dmfConfigured = \
            indigoStatus.portDmfConfigured[ port ]
      interfaceModel.packetOutCount = portCounters.packetOutCount
      interfaceModel.packetInCount = portCounters.packetInCount

      model.portIds[ port ] = interfaceModel
   return model

def getStripVlanString( removeVlansString ):
   if removeVlansString == "":
      return "disabled"
   elif removeVlansString == "1":
      return "outer"
   elif removeVlansString == "1-2":
      return "inner and outer"
   return "unknown"

def showDmfInterfaces( mode, args ):
   model = DmfInterfacesModel()
   intRange = args.get( "INTS" )
   if intRange:
      intfs = IntfCli.Intf.getAll( mode, intRange )
      intfs = { x.ethIntfConfig.intfId for x in intfs }

   # initialized here because portDS indexed by PortDSKey, not IntfId
   portPtpStates = {}
   for portDSKey in ptpStatus.intfStatus:
      portPtpStates[ portDSKey.intf ] = portStateToName[
            ptpStatus.portDS.get( portDSKey ).portState ]
   ports = set()
   if indigoStatus.portCountersDir:
      ports = indigoStatus.portCountersDir.portCounters.items()
   #pylint: disable=too-many-nested-blocks
   for portId, portCounters in ports:
      intfId = portCounters.intfId

      if intRange and intfId not in intfs:
         continue

      interfaceModel = InterfaceModel()

      # indigoStatus -> portCounters, dmfConfigured, intfId
      interfaceModel.intf = intfId
      interfaceModel.dmfConfigured = \
            indigoStatus.portDmfConfigured[ portId ]
      interfaceModel.packetOutCount = portCounters.packetOutCount
      interfaceModel.packetInCount = portCounters.packetInCount
      interfaceModel.lagMembers = None
      interfaceModel.dynVlans = None

      if interfaceModel.dmfConfigured:

         # switchConfig -> switchportMode, stripVlans, disableXmit
         switchConfig = bridgingInputConfig.switchIntfConfig.get( intfId )

         if switchConfig:
            if not switchConfig.enabled:
               switchportMode = SwitchportMode.routed
            else:
               switchportMode = switchConfig.switchportMode
            interfaceModel.switchportMode = switchportMode
            interfaceModel.stripVlans = getStripVlanString(
               switchConfig.toolDot1qRemoveVlans )
            interfaceModel.xmit = \
                  not switchConfig.switchportMode == SwitchportMode.tap
            interfaceModel.stripVxlan = switchConfig.tapVxlanStrip
            if interfaceModel.stripVxlan:
               vti = vxlanIntf.vtiConfig.get( "Vxlan1" )
               if vti:
                  interfaceModel.udpPort = vti.udpPort
         else:
            interfaceModel.switchportMode = SwitchportMode.access
            interfaceModel.stripVlans = ''
            interfaceModel.xmit = True

         if PortChannelIntfId.isPortChannelIntfId( intfId ):
            # Populate LAG port details
            interfaceModel.speed = 0
            elis = lagIntfStatusDir.intfStatus.get( intfId, None )
            if elis:
               interfaceModel.speed = elis.getSpeed()
               if elis.member:
                  interfaceModel.lagMembers = list( elis.member )

            # Following are not applicable in case of LAG.
            interfaceModel.autoneg = None
            interfaceModel.fec = None
            interfaceModel.loopbackMode = None
            interfaceModel.forceLinkUp = None
            interfaceModel.rateLimit = None
         else:
            # ethIntfStatus -> speed, autoneg, fec, loopbackMode, forceLinkUp
            ethIntfStatus = ethPhyIntfStatus.intfStatus.get( intfId )

            if ethIntfStatus:
               interfaceModel.speed = ethIntfStatus.getSpeed()
               autonegState = ethIntfStatus.autonegStatus.state
               if autonegState in ( AutonegState.anegStateUnknown,
                                    AutonegState.anegStateOff ):
                  interfaceModel.autoneg = False
               else:
                  interfaceModel.autoneg = True

               for coherentStatusDir in hwPhyStatusDataCoherentSlice.values():
                  coherentStatus = coherentStatusDir.phyCoherentStatus.get( intfId )
                  if coherentStatus:
                     interfaceModel.fec = coherentStatus.fecStatus
                     break
               else:
                  interfaceModel.fec = ethIntfStatus.fecStatus
               interfaceModel.loopbackMode = ethIntfStatus.loopbackMode
               interfaceModel.forceLinkUp = \
                  ethIntfStatus.unidirectionalLinkMode == \
                  UnidirectionalLinkMode.uniLinkModeSendReceive
            else:
               interfaceModel.speed = 0
               interfaceModel.autoneg = False
               interfaceModel.fec = EthFecEncoding.fecEncodingUnknown
               interfaceModel.loopbackMode = LoopbackMode.loopbackNone
               interfaceModel.forceLinkUp = False

            # qosIntfStaus -> rateLimit
            qosIntfStatus = qosStatus.intfStatus.get( intfId )
            if qosIntfStatus and \
               qosIntfStatus.shapeRate.rate != ShapeRateVal.invalid:
               interfaceModel.rateLimit = qosIntfStatus.shapeRate.rate

            # ptpStatus -> ptpState
            interfaceModel.ptpState = portPtpStates.get( intfId )

            # AllowedVlans -> dynVlans
            dynVlansStr = bridgingInputDynVlanAllowedVlans.vlans.get( intfId )
            if dynVlansStr is not None:
               interfaceModel.dynVlans = multiRangeFromCanonicalString(
                     dynVlansStr,
                     asList=True )
      model.portIds[ portId ] = interfaceModel
   return model

###################################################################
# show management dmf indigo
##################################################################
class DmfIndigoShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show management dmf indigo"
   data = {
         "management": ConfigMgmtMode.managementShowKwMatcher,
         "dmf": DmfCli.dmfMatcher,
         "indigo": DmfIndigoCli.indigoMatcher,
   }

   handler = showDmfIndigo
   cliModel = DmfIndigoModel

BasicCli.addShowCommandClass( DmfIndigoShowCmd )

###################################################################
# show management dmf indigo packets
##################################################################
class DmfIndigoPacketsShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show management dmf indigo packets"
   data = {
         "management": ConfigMgmtMode.managementShowKwMatcher,
         "dmf": DmfCli.dmfMatcher,
         "indigo": DmfIndigoCli.indigoMatcher,
         "packets": "Information about packets trapped or injected"
                    " to/by the controller",
   }

   handler = showDmfIndigoPackets
   cliModel = DmfIndigoPacketsModel

BasicCli.addShowCommandClass( DmfIndigoPacketsShowCmd )

###################################################################
# show management dmf controller history
##################################################################
class DmfIndigoConnectionHistoryShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show management dmf controller history"
   data = {
         "management": ConfigMgmtMode.managementShowKwMatcher,
         "dmf": DmfCli.dmfMatcher,
         "controller": DmfCli.dmfCtrlrMatcher,
         "history": "Information about past events involving controller",
   }

   handler = showDmfIndigoConnectionHistory
   cliModel = DmfIndigoConnectionHistoryModel

BasicCli.addShowCommandClass( DmfIndigoConnectionHistoryShowCmd )

###################################################################
# show management dmf interfaces summary
##################################################################
class DmfInterfacesSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show management dmf interfaces summary"
   data = {
         "management": ConfigMgmtMode.managementShowKwMatcher,
         "dmf": DmfCli.dmfMatcher,
         "interfaces": interfacesShowKw,
         "summary": "DMF interfaces summary",
   }

   handler = showDmfInterfacesSummary
   cliModel = DmfInterfacesSummaryModel

BasicCli.addShowCommandClass( DmfInterfacesSummaryCmd )

###################################################################
# show management dmf interfaces [ INTS ]
#################################################################
class DmfInterfacesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show management dmf interfaces [ INTS ]"
   data = {
         "management": ConfigMgmtMode.managementShowKwMatcher,
         "dmf": DmfCli.dmfMatcher,
         "interfaces": interfacesShowKw,
         "INTS": DmfIndigoCli.dmfIntfRangeMatcher,
   }

   handler = showDmfInterfaces
   cliModel = DmfInterfacesModel

BasicCli.addShowCommandClass( DmfInterfacesCmd )

def Plugin( entityManager ):
   global config
   global bridgingInputConfig
   global bridgingInputDynVlanAllowedVlans
   global indigoStatus
   global ethPhyIntfStatus
   global qosStatus
   global lagIntfStatusDir
   global ptpStatus
   global vxlanIntf
   global hwPhyStatusDataCoherentSlice

   config = LazyMount.mount( entityManager,
                             "dmf/cli/config",
                             "Dmf::Cli::Config", "r" )
   bridgingInputConfig = LazyMount.mount( entityManager,
                                          "bridging/input/config/dmf",
                                          "Bridging::Input::Config", "r" )
   bridgingInputDynVlanAllowedVlans = LazyMount.mount( entityManager,
                                          "bridging/input/dynvlan/allowedvlan/dmf",
                                          "Bridging::Input::AllowedVlans", "r" )
   indigoStatus = LazyMount.mount( entityManager,
                                   "dmf/indigo/status",
                                   "Dmf::Indigo::Status", "r" )
   ethPhyIntfStatus = LazyMount.mount( entityManager,
                                       "interface/status/eth/phy/all",
                                       "Interface::AllEthPhyIntfStatusDir", "r" )
   qosStatus = LazyMount.mount( entityManager,
                                "qos/status",
                                "Qos::Status", "r" )
   lagIntfStatusDir = LazyMount.mount( entityManager,
                                       "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   ptpStatus = LazyMount.mount( entityManager, "ptp/status", "Ptp::Status", "r" )
   vxlanIntf = LazyMount.mount( entityManager, "interface/config/eth/vxlan",
                               "Vxlan::VtiConfigDir", "r" )
   hwPhyStatusDataCoherentSlice = LazyMount.mount( entityManager,
         "hardware/phy/status/data/coherent/slice", "Tac::Dir", "r" )

   registerDmfShowTech( '2021-01-18 16:00:01',
                        [ 'show management dmf indigo',
                          'show management dmf indigo packets' ] )
