# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import ShowCommand
# pylint: disable-msg=ungrouped-imports
# pylint: disable-next=consider-using-from-import
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
from CliPlugin.DmfCli import dmfMatcher, dmfGlobal
from CliPlugin.DmfCliModel import ( FlowModel, ActionModel,
                                    TableStatsModel, DmfFlowTableSummaryModel,
                                    DmfFlowTableModel, FlowErrorModel )
import CliPlugin.FileCli as FileCli # pylint: disable=consider-using-from-import
from CliPlugin.TechSupportCli import registerShowTechSupportCmd
import ShowRunOutputModel
from CliMatcher import IntegerMatcher
from MultiRangeRule import MultiRangeMatcher
import TrafficPolicyLib

FlowMatchType = \
      Tac.Type( "Dmf::Policy::FlowMatchType" )

def getStructuredFilterString( matchType, flow ):
   filterStr = ""
   if flow.structuredFilter is not None:
      if flow.structuredFilter.innerMatch is not None:
         innerMatch = flow.structuredFilter.innerMatch
         cmds = TrafficPolicyLib.structuredFilterToCmds(
            innerMatch, None, innerMatch.af )
      else:
         cmds = TrafficPolicyLib.structuredFilterToCmds(
            flow.structuredFilter, None, matchType )
      for field in TrafficPolicyLib.structuredFilterFieldsList:
         cmd = cmds.get( field, '' )
         if isinstance( cmd, list ):
            filterStr += " ".join( cmd )
         else:
            filterStr += cmd
         if cmd:
            filterStr += " "

   return filterStr.strip()

def getFilterString( flow ):
   if hasattr( flow, "structuredFilter" ): # ing flow 2
      return getStructuredFilterString( flow.matchType, flow )
   if flow.matchType == FlowMatchType.ipv4:
      return flow.ipFilter.toStrep()
   elif flow.matchType == FlowMatchType.ipv6:
      return flow.ipv6Filter.toStrep()
   else: # mac
      return flow.macFilter.toStrep()

def populateFlowModel( flowModel, flowConfig, flowStatus ):
   flowModel.flowId = flowConfig.id
   flowModel.priority = flowConfig.priority
   # pylint: disable-msg=protected-access
   flowModel._type = flowConfig.type
   flowModel.matchType = flowConfig.matchType
   flowModel.filterString = getFilterString( flowConfig )
   flowModel.ports = (
      list( getattr( flowStatus, 'port', [] ) or flowConfig.port ) )
   for action in flowConfig.action:
      actionModel = ActionModel()
      actionModel.actionType = action.actionType
      actionModel.val = action.val
      actionModel.valStr = action.valStr
      actionModel.intf = action.intf
      flowModel.actions.append( actionModel )
   flowModel.state = flowStatus.state
   flowModel.packetCount = (
      flowStatus.packetCount - getattr( flowStatus, 'packetCountCheckpoint', 0 ) )
   if hasattr( flowStatus, 'byteCount' ) and flowStatus.byteCount is not None:
      flowModel.byteCount = (
         flowStatus.byteCount - getattr( flowStatus, 'byteCountCheckpoint', 0 ) )
   if hasattr( flowStatus, 'countLastUpdateTime' ):
      flowModel.lastUpdateTime = flowStatus.countLastUpdateTime
   if not flowConfig.port:
      flowModel._wildcarded = True # pylint: disable-msg=protected-access

def populateSummaryModel( summaryModel, programmingCounters, flowStats=None ):
   summaryModel.programmed = programmingCounters[ 'programmed' ]
   summaryModel.pending = programmingCounters[ 'pendingProgram' ] \
         + programmingCounters[ 'pendingDelete' ]
   summaryModel.failed = programmingCounters[ 'failed' ]
   summaryModel.total = \
         summaryModel.programmed + summaryModel.pending + summaryModel.failed
   if flowStats:
      summaryModel.requestedAdds = flowStats.requestedAdds
      summaryModel.requestedMods = flowStats.requestedMods
      summaryModel.requestedDels = flowStats.requestedDels
      summaryModel.installed = flowStats.installed
      summaryModel.pendingAdds = flowStats.pendingAdds
      summaryModel.pendingMods = flowStats.pendingMods
      summaryModel.pendingDels = flowStats.pendingDels
      summaryModel.pendingCleanups = flowStats.pendingCleanups
      summaryModel.errorsFull = flowStats.errorsFull
      summaryModel.errorsParam = flowStats.errorsParam
      summaryModel.errorsPolicy = flowStats.errorsPolicy
      summaryModel.errorsMisc = flowStats.errorsMisc

def populateErrorModel( errorModel, error, entry=None ):
   errorModel.message = error.message
   if error.hwError:
      errorModel.hwError = error.hwError
   errorModel.time = error.timestamp
   if entry:
      flowModel = FlowModel()
      populateFlowModel( flowModel, entry, error )
      errorModel.flow = flowModel

def showDmfFlowTableSummary( mode, args ):
   model = DmfFlowTableSummaryModel()

   ing1StatsModel = TableStatsModel()

   programmingCounters = { "programmed": 0, "pendingProgram": 0,
         "pendingDelete": 0, "failed": 0 }
   for flow in dmfGlobal.policyStatus.if1Table.entry.values():
      programmingCounters[ flow.state ] += 1
   populateSummaryModel( ing1StatsModel, programmingCounters )

   ing2StatsModel = TableStatsModel()

   programmingCounters = { "programmed": 0, "pendingProgram": 0,
         "pendingDelete": 0, "failed": 0 }
   for flow in dmfGlobal.policyStatus.if2Table.entry.values():
      programmingCounters[ flow.state ] += 1
   populateSummaryModel( ing2StatsModel, programmingCounters )

   egr1StatsModel = TableStatsModel()

   programmingCounters = { "programmed": 0, "pendingProgram": 0,
         "pendingDelete": 0, "failed": 0 }
   for flow in dmfGlobal.policyStatus.ef1Table.entry.values():
      programmingCounters[ flow.state ] += 1
   populateSummaryModel( egr1StatsModel, programmingCounters )

   model.ingress1 = ing1StatsModel
   model.ingress2 = ing2StatsModel
   model.egress1 = egr1StatsModel

   return model

def showDmfFlowTable( mode, args ):
   model = DmfFlowTableModel()
   table = args[ "TABLE" ]
   flowIds = args.get( "IDS" )
   egress = "egress" in args
   brief = "brief" in args
   errors = "errors" in args
   summaryModel = None

   if int( table ) == 1:
      if not egress:
         statusEntries = dmfGlobal.policyStatus.if1Table.entry
         configEntries = dmfGlobal.policyConfig.if1Table.entry
         configSavedEntries = dmfGlobal.policyConfig.if1Table.savedEntry
         flowStats = dmfGlobal.flowStatus.ingTable1Stats
         flowErrors = dmfGlobal.flowStatus.ingTable1ErrorLog
      else:
         statusEntries = dmfGlobal.policyStatus.ef1Table.entry
         configEntries = dmfGlobal.policyConfig.ef1Table.entry
         configSavedEntries = dmfGlobal.policyConfig.ef1Table.savedEntry
         flowStats = dmfGlobal.flowStatus.egrTable1Stats
         flowErrors = dmfGlobal.flowStatus.egrTable1ErrorLog
   else:
      if not egress:
         statusEntries = dmfGlobal.policyStatus.if2Table.entry
         configEntries = dmfGlobal.policyConfig.if2Table.entry
         configSavedEntries = dmfGlobal.policyConfig.if2Table.savedEntry
         flowStats = dmfGlobal.flowStatus.ingTable2Stats
         flowErrors = dmfGlobal.flowStatus.ingTable2ErrorLog
      else:
         mode.addError( "Egress Flow 2 table does not exist" )
         return None

   if errors:
      for error in flowErrors.error.values():
         entry = configSavedEntries.get( error.uniqueId )
         errorModel = FlowErrorModel()
         populateErrorModel( errorModel, error, entry=entry )
         model.flowErrors.append( errorModel )

   if flowIds:
      sEntriesTmp = {}
      cEntriesTmp = {}
      for flowId in flowIds.values():
         if statusEntries and flowId in statusEntries:
            sEntriesTmp[ flowId ] = statusEntries[ flowId ]
         if configEntries and flowId in configEntries:
            cEntriesTmp[ flowId ] = configEntries[ flowId ]
      statusEntries = sEntriesTmp
      configEntries = cEntriesTmp
   else: # do not print table summary if we are explicitly passing flow ids
      summaryModel = TableStatsModel()

   programmingCounters = { "programmed": 0, "pendingProgram": 0,
         "pendingDelete": 0, "failed": 0 }
   model._showSysTime = False # pylint: disable-msg=protected-access
   for flowId, flowStatus in statusEntries.items():
      flowConfig = configEntries.get( flowId )
      # FIXME: the entry is being deleted, should we show anything?
      if not flowConfig:
         continue
      if summaryModel:
         programmingCounters[ flowStatus.state ] += 1
      if errors:
         continue
      flowModel = FlowModel()
      populateFlowModel( flowModel, flowConfig, flowStatus )
      model.flows[ flowId ] = flowModel
      model._showSysTime = True # pylint: disable-msg=protected-access
   if summaryModel:
      # populate summary model only if we are interested in entire table
      if brief:
         flowStats = None
      populateSummaryModel( summaryModel, programmingCounters, flowStats=flowStats )
      model.summary = summaryModel
   model._brief = brief # pylint: disable-msg=protected-access
   model._showErrors = errors # pylint: disable-msg=protected-access
   return model

###########################################################################
# show management dmf flow summary
##########################################################################
class DmfFlowTableSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show management dmf flow summary" )
   data = {
      "management": ConfigMgmtMode.managementShowKwMatcher,
      "dmf": dmfMatcher,
      "flow": "DMF Flow",
      "summary": "Flow tables summary",
   }

   handler = showDmfFlowTableSummary
   cliModel = DmfFlowTableSummaryModel

BasicCli.addShowCommandClass( DmfFlowTableSummaryCmd )

###########################################################################
# show management dmf flow ( ingress | egress ) TABLE [ entry IDS ]
##########################################################################
class DmfFlowTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show management dmf flow ( ingress | egress ) TABLE \
              [ ( [ entry IDS ] [ brief ] ) | errors ]" )
   data = {
      "management": ConfigMgmtMode.managementShowKwMatcher,
      "dmf": dmfMatcher,
      "flow": "DMF Flow",
      "ingress": "Ingress Flow Table",
      "egress": "Egress Flow Table",
      "TABLE": IntegerMatcher( 1, 2, helpdesc="Flow Table Number" ),
      "entry": "DMF Flow Entry",
      "IDS": MultiRangeMatcher( lambda: ( 1, 2**16 - 1 ), False,
                                helpdesc="Flow ID number" ),
      "brief": "Display brief flow state",
      "errors": "Display recent errors"
   }

   handler = showDmfFlowTable
   cliModel = DmfFlowTableModel

BasicCli.addShowCommandClass( DmfFlowTableCmd )

###########################################################################
# show management dmf dynamic configuration
##########################################################################
class DmfShowDynamicConfigCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show management dmf dynamic configuration"
   data = {
      "management": ConfigMgmtMode.managementShowKwMatcher,
      "dmf": dmfMatcher,
      "dynamic": "Display dynamic information",
      "configuration": "Display dynamic configuration applied by DMF"
   }
   privileged = True
   cliModel = ShowRunOutputModel.Mode

   @staticmethod
   def handler( mode, args ):
      return FileCli.showRunningConfig( mode, showFilteredRoot=True )

BasicCli.addShowCommandClass( DmfShowDynamicConfigCmd )

def dmfShowTechGuard():
   # run DMF commands in show-tech if DMF is active
   return dmfGlobal.policyStatus.active

def registerDmfShowTech( timestamp, cmds ):
   registerShowTechSupportCmd(
      timestamp,
      cmds=cmds,
      cmdsGuard=dmfShowTechGuard )

def Plugin( entityManager ):
   registerDmfShowTech( '2021-01-18 16:00:00',
                        [ 'show management dmf flow summary',
                          'show management dmf flow ingress 1 brief',
                          'show management dmf flow ingress 1',
                          'show management dmf flow ingress 1 errors',
                          'show management dmf flow ingress 2 brief',
                          'show management dmf flow ingress 2',
                          'show management dmf flow ingress 2 errors',
                          'show management dmf flow egress 1 brief',
                          'show management dmf flow egress 1',
                          'show management dmf flow egress 1 errors',
                          'show management dmf interfaces summary',
                          'show management dmf interfaces',
                          'show management dmf dynamic configuration',
                          'show management dmf controller zerotouch',
                          'show management dmf version' ] )
