# Copyright (c) 2009-2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from collections import defaultdict
import BasicCliModes
import CliParser
import BasicCli
import LazyMount
import CliMatcher
import CliCommand
import ShowCommand
import Tac
import Cell
import AaaCliLib
from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import Dot1xModel
from CliPlugin import DpAclCli
from CliPlugin import IntfCli
from CliPlugin import MacAddr
from CliPlugin import TechSupportCli
from CliPlugin import Ssl
from CliPlugin.BridgingCli import bridgingCheckStaticMacHook, \
      warnMacTableUnsupportedUFTModeHook
from CliPlugin.Dot1xModeCli import Dot1xMode
from CliPlugin.Dot1xModeCli import Dot1xCacheMode
from CliPlugin.Dot1xSupplicantCli import noDot1xSupplicant, matcherSupplicant
from CliPlugin.Dot1xSupplicantCli import profileExpression, setDot1xSupplicant
from CliPlugin.EthIntfCli import EthIntfModelet, EthPhyIntf
from CliPlugin.IntfCli import Intf
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.SubIntfCli import SubIntfModelet
from CliPlugin.VlanCli import vlanIdMatcher, vlanSetMatcher
from Intf.IntfRange import IntfRangeMatcher
from IntfRangePlugin.EthIntf import EthPhyAutoIntfType
from IntfRangePlugin.SwitchIntf import SwitchAutoIntfType
import CliToken.Clear
from Dot1xLib import portControlStr2Enum, hostModeStr2Enum
from Dot1xLib import Dot1xConsts, Dot1xAclInputPriority
import ConfigMount
import Tracing
import Toggles.Dot1xToggleLib as Dot1xToggle
import Toggles.DhcpLibToggleLib as DhcpLibToggle
import Url
import UrlPlugin.NetworkUrl as _
from TypeFuture import TacLazyType
from MultiRangeRule import multiRangeToCanonicalString
from Ethernet import convertMacAddrToDisplay
import SmashLazyMount
import Ark

t0 = Tracing.trace0
t8 = Tracing.trace8

config = None
radiusConfig = None
cliConfig = None
bridgingConfig = None
configReq = None
status = None
identityDot1xStatus = None
dot1xInputHostTable = None
mergedHostTable = None
hwstatus = None
radiusDeadTime = None
webAgentStatus = None
subIntfCapabilityDir = None
ipStatus = None
blockedMacTable = None
allIntfDot1xDroppedCounter = None
dot1xAclInputConfig = None
ethIntfStatusDir = None
netStatus = None
nbrClassificationStatus = None
mgmtGnmiClientConfig = None

# Tac Type
AaaUnresponsiveTrafficAllow = TacLazyType(
   "Dot1x::Dot1xAaaUnresponsiveTrafficAllow" )
DroppedCtrConsts = TacLazyType( "Interface::IntfDroppedCounterConstants" )
EthIntfId = TacLazyType( "Arnet::EthIntfId" )
IntfId = TacLazyType( "Arnet::IntfId" )
SessionReplaceDetection = TacLazyType( "Dot1x::SessionReplaceDetection" )
CachedResultsTimeoutUnit = TacLazyType( "Dot1x::CachedResultsTimeoutUnit" )
CachedResultsTimeout = TacLazyType( "Dot1x::CachedResultsTimeout" )
TristateBool = TacLazyType( "Ark::TristateBoolean" )
ProfilingStateEnum = TacLazyType( "Dot1x::Dot1xDeviceProfilingAttrState" )
Dot1xProfilingDhcpOptionType = TacLazyType( "Dot1x::Dot1xProfilingDhcpOptionType" )
Dot1xIntfGuestVlan = TacLazyType( "Dot1x::GuestVlan" )

# Other global variables
possibleMgmtIntfs = [ "Management0",
                      "Management1",
                      "Management1/1", "Management1/2",
                      "Management2/1", "Management2/2" ]

intfRangeMatcher = IntfRangeMatcher( explicitIntfTypes=( EthPhyAutoIntfType,
                                                         SwitchAutoIntfType ) )

def makeOrGetDot1xIntfConfig( intfName ):
   """
   Make a Dot1xIntfConfig and Dot1xConfigReq
   via the idempotent new operator.
   Returns the Dot1xIntfConfig
   """
   dot1xIntfConfig = config.newDot1xIntfConfig( intfName )
   configReq.newDot1xIntfConfigReq( intfName )
   return dot1xIntfConfig

def populateApplyCachedResultsTimeout( c, args, phone ):
   unit = CachedResultsTimeoutUnit.timeoutUnitNone
   timeout = Tac.endOfTime
   if 'cached-results' in args and 'timeout' in args:
      timeout_days = args.get( 'TIMEOUT_DAY' )
      timeout_hrs = args.get( 'TIMEOUT_HRS' )
      timeout_mins = args.get( 'TIMEOUT_MIN' )
      timeout_secs = args.get( 'TIMEOUT_SEC' )
      if timeout_days:
         unit = CachedResultsTimeoutUnit.timeoutUnitDays
         timeout = timeout_days * 86400
      elif timeout_hrs:
         unit = CachedResultsTimeoutUnit.timeoutUnitHours
         timeout = timeout_hrs * 3600
      elif timeout_mins:
         unit = CachedResultsTimeoutUnit.timeoutUnitMinutes
         timeout = timeout_mins * 60
      elif timeout_secs:
         unit = CachedResultsTimeoutUnit.timeoutUnitSeconds
         timeout = timeout_secs

   if phone:
      c.aaaUnresponsivePhoneApplyCachedResultsTimeout = \
            CachedResultsTimeout( timeout, unit )
   else:
      c.aaaUnresponsiveApplyCachedResultsTimeout = \
            CachedResultsTimeout( timeout, unit )

#------------------------------------------------------------
# [no|default] dot1x system-auth-control
#------------------------------------------------------------
def setDot1xSysAuthControl( mode, args ):
   config.dot1xEnabled = True
   # Hook is populated on Trident4 platforms to warn of conflicting configuration
   warnMacTableUnsupportedUFTModeHook.notifyExtensions( mode )

def noDot1xSysAuthControl( mode, args ):
   config.dot1xEnabled = False

def dot1xSupportedGuard( mode, token ):
   if hwstatus.dot1xSupported:
      return None
   return CliParser.guardNotThisPlatform

def dot1xPerMacAclSupportedGuard( mode, token ):
   if hwstatus.dot1xPerMacAclSupported:
      return None
   return CliParser.guardNotThisPlatform

def dot1xMbaSupportedGuard( mode, token ):
   if hwstatus.dot1xMbaSupported:
      return None
   return CliParser.guardNotThisPlatform

def dot1xOnSubIntfSupportedGuard( mode, token ):
   if not EthIntfId.isEthIntfId( mode.intf.name ):
      return None
   for sliceStatus in subIntfCapabilityDir.values():
      parentIntfName = EthIntfId.parentIntfIdFromSubIntf( mode.intf.name )
      if sliceStatus and parentIntfName in sliceStatus.dot1xCapable:
         return None

   return CliParser.guardNotThisPlatform 

nodeDot1x = CliCommand.guardedKeyword( 'dot1x',
      helpdesc='IEEE 802.1X port authentication',
      guard=dot1xSupportedGuard )

class Dot1XSystemAuthControlCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x system-auth-control'
   noOrDefaultSyntax = 'dot1x system-auth-control ...'
   data = {
      'dot1x': nodeDot1x,
      'system-auth-control': 'Enable or disable SysAuthControl',
   }
   handler = setDot1xSysAuthControl
   noOrDefaultHandler = noDot1xSysAuthControl

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XSystemAuthControlCmd )

#------------------------------------------------------------
# [no|default] dot1x dynamic-authorization
#------------------------------------------------------------
def setDot1xDynamicAuthorization( mode, args ):
   config.dot1xDynAuthEnabled = True

def noDot1xDynamicAuthorization( mode, args ):
   config.dot1xDynAuthEnabled = False

class Dot1XDynamicAuthorizationCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x dynamic-authorization'
   noOrDefaultSyntax = 'dot1x dynamic-authorization'
   data = {
      'dot1x': nodeDot1x,
      'dynamic-authorization': 'Enable dynamic authorization',
   }
   handler = setDot1xDynamicAuthorization
   noOrDefaultHandler = noDot1xDynamicAuthorization

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XDynamicAuthorizationCmd )

#------------------------------------------------------------
# [no|default] radius av-pair service-type
#------------------------------------------------------------

matcherRadius = CliMatcher.KeywordMatcher( 'radius',
               helpdesc='Keyword to set RADIUS parameters' )
matcherAVPair = CliMatcher.KeywordMatcher( 'av-pair',
               helpdesc='Attribute-Value Pair' )

def setDot1xRadiusServiceTypeAttribute( mode, args ):
   config.serviceType = True

def noDot1xRadiusServiceTypeAttribute( mode, args ):
   config.serviceType = False

class Dot1XRadiusServiceTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'radius av-pair service-type'
   noOrDefaultSyntax = 'radius av-pair service-type'
   data = {
      'radius': matcherRadius,
      'av-pair': matcherAVPair,
      'service-type': 'service-type av-pair',
   }
   handler = setDot1xRadiusServiceTypeAttribute
   noOrDefaultHandler = noDot1xRadiusServiceTypeAttribute

Dot1xMode.addCommandClass( Dot1XRadiusServiceTypeCmd )

#------------------------------------------------------------
# [no|default] radius av-pair filter-id ( multiple | ( delimiter period ) |
#                                         ( ipv4 ipv6 required ) )
# ------------------------------------------------------------

def setDot1xRadiusFilterIdAttribute( mode, args ):
   if 'multiple' in args:
      config.multipleFilterId = True
   elif 'delimiter' in args:
      config.filterIdDelim = True
   elif 'required' in args:
      config.applyIpv4v6Acl = True

def noDot1xRadiusFilterIdAttribute( mode, args ):
   if 'multiple' in args:
      config.multipleFilterId = False
   if 'delimiter' in args:
      config.filterIdDelim = False
   if 'required' in args:
      config.applyIpv4v6Acl = False

class Dot1xRadiusFilterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'radius av-pair filter-id ( multiple | ( delimiter period ) |'\
                                      '( ipv4 ipv6 required ) )'
   noOrDefaultSyntax = syntax
   data = {
      'radius': matcherRadius,
      'av-pair': matcherAVPair,
      'filter-id': 'Filter-id av-pair',
      'multiple': 'Multiple attribute',
      'delimiter': 'Delimiter to use',
      'period': 'Period as delimiter',
      'ipv4': 'IPv4 filter-id',
      'ipv6': 'IPv6 filter-id',
      'required': 'Filter-id are required',
   }
   handler = setDot1xRadiusFilterIdAttribute
   noOrDefaultHandler = noDot1xRadiusFilterIdAttribute

Dot1xMode.addCommandClass( Dot1xRadiusFilterIdCmd )

# ------------------------------------------------------------
# [no|default] radius av-pair framed-mtu <length>
#------------------------------------------------------------

matcherFramedMtu = CliMatcher.IntegerMatcher( Dot1xConsts.minFramedMtu, 
                                              Dot1xConsts.maxFramedMtu,
      helpdesc='RADIUS framed MTU size in bytes' )

def setDot1xRadiusFramedMtuAttribute( mode, args ):
   config.framedMtu = args[ 'LENGTH' ]

def noDot1xRadiusFramedMtuAttribute( mode, args ):
   config.framedMtu = config.defaultFramedMtu

class Dot1XRadiusFramedMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'radius av-pair framed-mtu LENGTH'
   noOrDefaultSyntax = 'radius av-pair framed-mtu ...'
   data = {
      'radius': matcherRadius,
      'av-pair': matcherAVPair,
      'framed-mtu': 'framed-mtu size av-pair',
      'LENGTH': matcherFramedMtu,
   }
   handler = setDot1xRadiusFramedMtuAttribute
   noOrDefaultHandler = noDot1xRadiusFramedMtuAttribute

Dot1xMode.addCommandClass( Dot1XRadiusFramedMtuCmd )

# ------------------------------------------------------------
# [no|default] radius av-pair lldp ( system-name | system-description ) [auth-only]
#------------------------------------------------------------
matcherAuthOnly = CliMatcher.KeywordMatcher( 'auth-only',
                  helpdesc='Stop sending updates on the av-pair' )

def getDot1xProfilingState( args ):
   if 'auth-only' in args:
      return ProfilingStateEnum.authOnlyProfiling
   else:
      return ProfilingStateEnum.updateProfiling

def setDot1xLldpTlvProfilingState( mode, args ):
   profilingState = getDot1xProfilingState( args )
   if 'system-name' in args:
      config.lldpSysNameProfilingState = profilingState
   elif 'system-description' in args:
      config.lldpSysDescProfilingState = profilingState

def noDot1xLldpTlvProfilingState( mode, args ):
   profilingState = ProfilingStateEnum.disableProfiling
   if 'system-name' in args:
      config.lldpSysNameProfilingState = profilingState
   elif 'system-description' in args:
      config.lldpSysDescProfilingState = profilingState

class Dot1XLldpTlvProfilingCmd( CliCommand.CliCommandClass ):
   syntax = 'radius av-pair lldp ( system-name | system-description ) [auth-only]'
   noOrDefaultSyntax = 'radius av-pair lldp ( system-name | system-description ) ...'
   data = {
      'radius': matcherRadius,
      'av-pair': matcherAVPair,
      'lldp': 'Select LLDP av-pair',
      'system-name': 'LLDP system name (LLDP TLV 5) av-pair',
      'system-description': 'LLDP system description (LLDP TLV 6) av-pair',
      'auth-only': matcherAuthOnly
   }
   handler = setDot1xLldpTlvProfilingState
   noOrDefaultHandler = noDot1xLldpTlvProfilingState

Dot1xMode.addCommandClass( Dot1XLldpTlvProfilingCmd )

# ----------------------------------------------------------------------
# [no|default] radius av-pair dhcp ( hostname | parameter-request-list |
#                                    vendor-class-id ) [auth-only]
# -----------------------------------------------------------------------
def getDot1xProfilingDhcpOptionType( args ):
   if 'hostname' in args:
      return Dot1xProfilingDhcpOptionType.hostname
   elif 'parameter-request-list' in args:
      return Dot1xProfilingDhcpOptionType.paramReqList
   else:
      return Dot1xProfilingDhcpOptionType.vendorClassId

def setDhcpOptionsProfilingState( mode, args ):
   profilingState = getDot1xProfilingState( args )
   dhcpOption = getDot1xProfilingDhcpOptionType( args )
   dot1xProfilingDhcpOption = Tac.Value(
         "Dot1x::Dot1xProfilingDhcpOption", dhcpOption, profilingState )
   config.addDot1xProfilingDhcpOption( dot1xProfilingDhcpOption )

def noDhcpOptionsProfilingState( mode, args ):
   dhcpOption = getDot1xProfilingDhcpOptionType( args )
   del config.dot1xProfilingDhcpOption[ dhcpOption ]

class Dot1XProfilingDhcpOptionsCmd( CliCommand.CliCommandClass ):
   syntax = '''radius av-pair dhcp ( hostname | parameter-request-list | 
            vendor-class-id ) [auth-only]'''
   noOrDefaultSyntax = '''radius av-pair dhcp ( hostname | parameter-request-list | 
                       vendor-class-id ) ...'''
   data = {
      'radius': matcherRadius,
      'av-pair': matcherAVPair,
      'dhcp': 'Select DHCP av-pair',
      'hostname': 'Hostname (DHCP Option 12)',
      'parameter-request-list': 'Configuration parameters requested by host '
                                '(DHCP Option 55)',
      'vendor-class-id': 'Vendor class identifier (DHCP Option 60)',
      'auth-only': matcherAuthOnly
   }
   handler = setDhcpOptionsProfilingState
   noOrDefaultHandler = noDhcpOptionsProfilingState

if DhcpLibToggle.toggleDhcpOptionsProfilingEnabled():
   Dot1xMode.addCommandClass( Dot1XProfilingDhcpOptionsCmd )

# ------------------------------------------------------------
# [no|default] radius av-pair framed-ip source l3-neighbor group GROUP [auth-only]
# ------------------------------------------------------------
mgmtGnmiClientGroupNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: mgmtGnmiClientConfig.gnmiServerGroup,
      helpdesc='gNMI server-group name' )

def setFramedIpSourceL3ProfilingState( mode, args ):
   profilingState = getDot1xProfilingState( args )
   config.framedIpSourceL3ProfilingState = profilingState
   config.framedIpSourceL3ServerGroup = args[ 'GROUP' ]

def noFramedIpSourceL3ProfilingState( mode, args ):
   config.framedIpSourceL3ProfilingState = ProfilingStateEnum.disableProfiling
   config.framedIpSourceL3ServerGroup = ''

class Dot1XProfilingFramedIpSourceL3Cmd( CliCommand.CliCommandClass ):
   syntax = ( 'radius av-pair framed-ip source l3-neighbor group GROUP' )
   noOrDefaultSyntax = 'radius av-pair framed-ip source l3-neighbor ...'
   data = {
      'radius': matcherRadius,
      'av-pair': matcherAVPair,
      'framed-ip': 'Select Framed-IP-Address av-pair',
      'source': 'Specify source for the IP address',
      'l3-neighbor': 'Use L3 neighbor as the IP address source',
      'group': 'Configure L3 neighbor server-group',
      'GROUP': mgmtGnmiClientGroupNameMatcher
   }
   handler = setFramedIpSourceL3ProfilingState
   noOrDefaultHandler = noFramedIpSourceL3ProfilingState

if Dot1xToggle.toggleDot1xFramedIpSourceL3Enabled():
   Dot1xMode.addCommandClass( Dot1XProfilingFramedIpSourceL3Cmd )

#------------------------------------------------------------
# [no|default] vlan assignment group <name> members <vlan_list>
#------------------------------------------------------------

groupNameMatcher = CliMatcher.PatternMatcher( pattern='.+',
                      helpname='WORD',
                      helpdesc='The ASCII name for the VLAN group',
                      value=lambda mode, match: match[ :32 ] )

def setDot1xVlanGroup( mode, args ):
   entry = Tac.Value( "Dot1x::VlanGroup", args[ 'GROUPNAME' ] )
   vlanIdSet = args[ 'VLAN_SET' ]
   for vlanId in vlanIdSet:
      entry.vlanId.add( vlanId )
   config.vlanGroup.addMember( entry )

def noDot1xVlanGroup( mode, args ):
   if group := args.get( 'GROUPNAME' ):
      del config.vlanGroup[ group ]
   else:
      # We got called from `Dot1xMode.clear`; no args.
      config.vlanGroup.clear()

class Dot1XVlanGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan assignment group GROUPNAME members VLAN_SET'
   noOrDefaultSyntax = 'vlan assignment group GROUPNAME ...'
   data = {
         'vlan': 'VLAN group',
         'assignment': 'Assign VLAN to group',
         'group': 'Group name',
         'GROUPNAME': groupNameMatcher,
         'members': 'Members of group',
         'VLAN_SET': vlanSetMatcher,
   }
   handler = setDot1xVlanGroup
   noOrDefaultHandler = noDot1xVlanGroup

Dot1xMode.addCommandClass( Dot1XVlanGroupCmd )

#------------------------------------------------------------
# [no|default] mac-based-auth radius av-pair <attribute-name> 
#                  delimiter [none|colon|hyphen|period] [uppercase|lowercase]
#------------------------------------------------------------

# XXX Remove the CLI guard once BUG182344 is fixed.
nodeMBA = CliCommand.Node( CliMatcher.KeywordMatcher( 'mac-based-auth',
                  helpdesc='Dot1x Mac Based Auth' ),
                  guard=dot1xMbaSupportedGuard )

def setDot1xMbaAttribute( mode, args ):
   delimiterMap = {
         'none': '',
         'colon': ':',
         'hyphen': '-',
         'period': '.'
   }
   config.mbaUserName = Tac.Value( "Dot1x::MbaUserName",
                                    userNameDelim = \
                                          delimiterMap[ args[ 'DELIMITER' ] ],
                                    userNameIsUpperCase = \
                                          ( args[ 'CASE' ] == 'uppercase' ),
                                    userNameGroupSize = \
                                          4 if args[ 'DELIMITER' ] == \
                                             'period' else 2 )

def noDot1xMbaAttribute( mode, args ):
   config.mbaUserName = Tac.Value( "Dot1x::MbaUserName" )

class Dot1XMbaAttributeFormatCmd( CliCommand.CliCommandClass ):
   syntax = 'mac-based-auth radius av-pair user-name delimiter DELIMITER CASE'
   noOrDefaultSyntax = 'mac-based-auth radius av-pair user-name ...'
   data = {
      'mac-based-auth': nodeMBA,
      'radius': 'Keyword to set RADIUS parameters',
      'av-pair': 'Attribute-Value Pair',
      'user-name': 'RADIUS user-name attribute',
      'delimiter': 'Delimiter to use',
      'DELIMITER': CliMatcher.EnumMatcher( {
         'none': 'No delimiter in MAC address string',
         'colon': 'Set colon as delimiter in MAC address string',
         'hyphen': 'Set hyphen as delimiter in MAC address string',
         'period': 'Set period as delimiter in MAC address string',
      } ),
      'CASE': CliMatcher.EnumMatcher( {
         'uppercase': 'MAC address string in uppercase',
         'lowercase': 'MAC address string in lowercase',
      } ),
   }
   handler = setDot1xMbaAttribute
   noOrDefaultHandler = noDot1xMbaAttribute

Dot1xMode.addCommandClass( Dot1XMbaAttributeFormatCmd )

#-----------------------------------------------------------------------------
# (dot1x)# [no|default] captive-portal [ url <URL> ] [ ssl profile <profile> ]
#-----------------------------------------------------------------------------

def setCaptivePortal( mode, args ):
   urlValue = args.get( 'URL', "" )
   config.captivePortal = Tac.Value( "Dot1x::CaptivePortal", 
                                     enabled=True, 
                                     url=urlValue and urlValue.url,
                                     cpHostName=urlValue and urlValue.urlhostname )
   sslProfile = args.get( 'PROFILE_NAME', "" )
   config.captivePortalSslProfileName = sslProfile

def noCaptivePortal( mode, args ):
   config.captivePortal = Tac.Value( "Dot1x::CaptivePortal" )
   config.captivePortalSslProfileName = ""

class Dot1xCaptivePortalCmd( CliCommand.CliCommandClass ):
   syntax = 'captive-portal [ url URL ] [ ssl profile PROFILE_NAME ]'
   noOrDefaultSyntax = 'captive-portal ...'

   data = {
      'captive-portal' : 'Configure captive portal parameters',
      'url' : 'Configure captive portal URL',
      'URL' :  Url.UrlMatcher( lambda fs: fs.scheme in [ 'http:', 'https:' ], 
         acceptSimpleFile=False,
         helpdesc="URL of form http[s]://<hostname>[:<port>]" ),
      'ssl': Ssl.sslMatcher,
      'profile': Ssl.profileMatcher,
      'PROFILE_NAME': Ssl.profileNameMatcher,
   }
   handler = setCaptivePortal
   noOrDefaultHandler = noCaptivePortal
      
Dot1xMode.addCommandClass( Dot1xCaptivePortalCmd )

#------------------------------------------------------------------
# (dot1x)# [no|default] captive-portal access-list ipv4 <acl-name>
#------------------------------------------------------------------

aclNameMatcher = AclCli.AclNameMatcher()

def setCaptivePortalACL( mode, args ):
   aclName = args.get( 'ACL_NAME', "" )
   config.captivePortalIpv4Acl = aclName

class Dot1xCaptivePortalACLCmd( CliCommand.CliCommandClass ):
   syntax = 'captive-portal access-list ipv4 ACL_NAME'
   noOrDefaultSyntax = 'captive-portal access-list ipv4 ...'

   data = {
         'captive-portal': 'Configure captive portal parameters',
         'access-list': 'Configure access control list',
         'ipv4': 'Type of access control list',
         'ACL_NAME': CliCommand.Node( matcher=aclNameMatcher )
   }
   handler = setCaptivePortalACL
   noOrDefaultHandler = setCaptivePortalACL

Dot1xMode.addCommandClass( Dot1xCaptivePortalACLCmd )

#------------------------------------------------------------------
# (dot1x)# [no|default] captive-portal start limit infinite
#------------------------------------------------------------------

def setCaptivePortalStartLimitInfinite( mode, args ):
   config.captivePortalStartLimitInfinite = True

def noCaptivePortalStartLimitInfinite( mode, args ):
   config.captivePortalStartLimitInfinite = False

class Dot1xCaptivePortalStartLimitInfinite( CliCommand.CliCommandClass ):
   syntax = 'captive-portal start limit infinite'
   noOrDefaultSyntax = syntax

   data = {
         'captive-portal': 'Configure captive portal parameters',
         'start': 'Configure captive-portal start parameters',
         'limit': 'Set limit',
         'infinite': "Infinite number of times",
   }
   handler = setCaptivePortalStartLimitInfinite
   noOrDefaultHandler = noCaptivePortalStartLimitInfinite

Dot1xMode.addCommandClass( Dot1xCaptivePortalStartLimitInfinite )

def setAclInputPriority( mode, priority ):
   config.dot1xAclInputPriority = priority
   DpAclCli.tryWaitForHwStatus( mode )

def setDot1xAclInputPriority( mode, args ):
   setAclInputPriority( mode, Dot1xAclInputPriority.dot1xAclInputPriorityLow )

def noDot1xAclInputPriority( mode, args ):
   setAclInputPriority( mode, config.dot1xAclInputPriorityDefault )

class Dot1xAclInputPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'access-list dot1x priority low'
   noOrDefaultSyntax = syntax

   data = {
      'access-list': 'Configure dot1x access control list',
      'priority': 'Configure the priority of dot1x access control list',
      'dot1x': nodeDot1x,
      'low': 'Set dot1x configured access control list priority to low'
   }
   handler = setDot1xAclInputPriority
   noOrDefaultHandler = noDot1xAclInputPriority

Dot1xMode.addCommandClass( Dot1xAclInputPriorityCmd )

# -----------------------------------------------------------------------------
# (dot1x)# [no|default] captive-portal bypass [ FQDN with wildcard ]
# -----------------------------------------------------------------------------

fqdnWildcardRegex = ''.join( [
   # optional initial wildcard followed by a dot:
   r'(\*\.)?',
   # Components are formed by letters+digits+dash, except that they can't
   # start or end in a dash.
   # Components followed by a dot:
   r'(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])\.)*',
   # Last component, no dot at the end:
   r'([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])',
   ] )

fqdnWildcardMatcher = CliMatcher.PatternMatcher(
   pattern=fqdnWildcardRegex,
   helpname='WORD',
   helpdesc='FQDN with optional wildcard' )

def addCaptivePortalBypass( mode, args ):
   config.captivePortalAllowlist[ args[ 'WILDCARD_FQDN' ] ] = True

def noCaptivePortalBypass( mode, args ):
   fqdn = args.get( 'WILDCARD_FQDN' )
   if fqdn:
      del config.captivePortalAllowlist[ fqdn ]
   else:
      config.captivePortalAllowlist.clear()

class Dot1xCaptivePortalBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'captive-portal bypass WILDCARD_FQDN'
   noOrDefaultSyntax = 'captive-portal bypass [ WILDCARD_FQDN ]'
   data = {
      'captive-portal': 'Configure captive portal parameters',
      'bypass': 'Bypass captive-portal redirection',
      'WILDCARD_FQDN': fqdnWildcardMatcher,
   }
   handler = addCaptivePortalBypass
   noOrDefaultHandler = noCaptivePortalBypass

if Dot1xToggle.toggleDot1xWebFqdnAllowlistEnabled():
   Dot1xMode.addCommandClass( Dot1xCaptivePortalBypassCmd )

#------------------------------------------------------------
# (dot1x)# [no|default] address tracking ipv4
#------------------------------------------------------------

def setIpTracking( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if no:
      config.ipTrackingEnabled = False
   else:
      config.ipTrackingEnabled = True
   
class Dot1xIpTrackingCmd( CliCommand.CliCommandClass ):
   syntax = 'address tracking ipv4'
   noOrDefaultSyntax = syntax

   data = {
      'address' : 'Enable IP tracking',
      'tracking' : 'Enable IP tracking',
      'ipv4' : 'Enable tracking for IPv4 addresses'
   }
   handler = setIpTracking
   noOrDefaultHandler = handler
      
if Dot1xToggle.toggleDot1xIpTrackingEnabled():
   Dot1xMode.addCommandClass( Dot1xIpTrackingCmd )

#------------------------------------------------------------
# [no|default] dot1x protocol lldp bypass
#------------------------------------------------------------
def setLldpBypass( mode, args ):
   config.lldpBypass = True

def noLldpBypass( mode, args ):
   config.lldpBypass = False

class Dot1XProtocolLldpBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x protocol lldp bypass'
   noOrDefaultSyntax = 'dot1x protocol lldp bypass'
   data = {
      'dot1x': nodeDot1x,
      'protocol': 'Set protocol processing',
      'lldp': 'LLDP frame processing',
      'bypass': 'Transmit/Receive without protection',
   }
   handler = setLldpBypass
   noOrDefaultHandler = noLldpBypass

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XProtocolLldpBypassCmd )

#------------------------------------------------------------
# [no|default] dot1x protocol bpdu bypass
#------------------------------------------------------------
def setBpduBypass( mode, args ):
   config.bpduBypass = True

def noBpduBypass( mode, args ):
   config.bpduBypass = False

class Dot1XProtocolBpduBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x protocol bpdu bypass'
   noOrDefaultSyntax = 'dot1x protocol bpdu bypass'
   data = {
      'dot1x': nodeDot1x,
      'protocol': 'Set protocol processing',
      'bpdu': 'BPDU frame processing',
      'bypass': 'Transmit/Receive without protection',
   }
   handler = setBpduBypass
   noOrDefaultHandler = noBpduBypass

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XProtocolBpduBypassCmd )

#------------------------------------------------------------
# [no|default] dot1x protocol lacp bypass
#------------------------------------------------------------
def setLacpBypass( mode, args ):
   config.lacpBypass = True

def noLacpBypass( mode, args ):
   config.lacpBypass = False

class Dot1XProtocolLacpBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x protocol lacp bypass'
   noOrDefaultSyntax = 'dot1x protocol lacp bypass'
   data = {
      'dot1x': nodeDot1x,
      'protocol': 'Set protocol processing',
      'lacp': 'LACP frame processing',
      'bypass': 'Transmit/Receive without protection',
   }
   handler = setLacpBypass
   noOrDefaultHandler = noLacpBypass

if Dot1xToggle.toggleDot1xLacpBypassEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XProtocolLacpBypassCmd )


# ------------------------------------------------------------
# (dot1x-result-caching)# [no|default] eapol log off ignore
# ------------------------------------------------------------
def ignoreEapolLogoff( mode, args ):
   config.eapolLogoffIgnore = True

def noIgnoreEapolLogoff( mode, args ):
   config.eapolLogoffIgnore = False

class Dot1xEapolLogoffIgnore( CliCommand.CliCommandClass ):
   syntax = 'eapol logoff ignore'
   noOrDefaultSyntax = syntax
   data = {
         'eapol': 'Configure Dot1x EAPOL caching result attribute',
         'logoff': 'Configure caching based on the eapol-logoff message',
         'ignore': 'Ignore eapol logoff',
         }
   handler = ignoreEapolLogoff
   noOrDefaultHandler = noIgnoreEapolLogoff

Dot1xCacheMode.addCommandClass( Dot1xEapolLogoffIgnore )
#------------------------------------------------------------
# (dot1x)# [no|default] aaa unresponsive action apply cached-results else traffic
#                                                             allow [ vlan <vlanID> ]
# (dot1x)# [no|default] aaa unresponsive action traffic allow [ vlan <vlanID> ]
# (dot1x)# [no|default] aaa unresponsive action apply cached-results
#------------------------------------------------------------
matcherDot1xAaa = CliMatcher.KeywordMatcher(
   'aaa', helpdesc='Configure AAA parameters' )
matcherDot1xAaaUnresp = CliMatcher.KeywordMatcher(
   'unresponsive', helpdesc='Configure AAA timeout options' )
matcherDot1xAaaUnrespAction = CliMatcher.KeywordMatcher(
   'action', helpdesc='Set action for supplicant when AAA times out' )
matcherDot1xAaaUnrespActionTraffic = CliMatcher.KeywordMatcher(
   'traffic', helpdesc='Set action for supplicant traffic when AAA times out' )
matcherDot1xAaaUnrespActionTrafficAllow = CliMatcher.KeywordMatcher(
   'allow', helpdesc='Allow traffic when AAA times out' )
matcherDot1xAaaUnrespActionApply = CliMatcher.KeywordMatcher(
   'apply', helpdesc='Apply an action when AAA times out' )
matcherDot1xAaaUnrespCachedResults = CliMatcher.KeywordMatcher(
   'cached-results', helpdesc='Use results from a previous AAA response' )
matcherDot1xAaaCachedTimeout = CliMatcher.KeywordMatcher(
   'timeout', helpdesc='Enable caching for a specific time duration' )
# 864 * ( 10**6 ) is the number of seconds in 10000 days
# 144 * ( 10**5 ) is the number of minutes in 10000 days
# 24 * ( 10**4 ) is the number of hours in 10000 days
matcherDot1xAaaCachedTimeoutSec = CliMatcher.IntegerMatcher(
   1, 864 * ( 10**6 ), helpdesc='Enable caching for a specific duration in seconds' )
matcherDot1xAaaCachedTimeoutMin = CliMatcher.IntegerMatcher(
   1, 144 * ( 10**5 ), helpdesc='Enable caching for a specific duration in minutes' )
matcherDot1xAaaCachedTimeoutHrs = CliMatcher.IntegerMatcher(
   1, 24 * ( 10**4 ), helpdesc='Enable caching for a specific duration in hours' )
matcherDot1xAaaCachedTimeoutDay = CliMatcher.IntegerMatcher(
   1, 10**4, helpdesc='Enable caching for a specific duration in days' )
matcherDot1xTimeoutSeconds = CliMatcher.KeywordMatcher(
   'seconds', helpdesc='Time duration unit in seconds' )
matcherDot1xTimeoutMinutes = CliMatcher.KeywordMatcher(
   'minutes', helpdesc='Time duration unit in minutes' )
matcherDot1xTimeoutHours = CliMatcher.KeywordMatcher(
   'hours', helpdesc='Time duration unit in hours' )
matcherDot1xTimeoutDays = CliMatcher.KeywordMatcher(
   'days', helpdesc='Time duration unit in days' )
matcherDot1xElse = CliMatcher.KeywordMatcher(
   'else', helpdesc='Apply alternate action if primary action fails' )

cacheTimeoutArg = ' [ timeout ( ( TIMEOUT_SEC seconds ) | \
      ( TIMEOUT_MIN minutes ) | (TIMEOUT_HRS hours ) | ( TIMEOUT_DAY days ) ) ]'

def setAaaUnresponsiveTrafficAllow( mode, args ):
   populateApplyCachedResultsTimeout( config, args, phone=False )

   if not 'allow' in args:
      config.aaaUnresponsiveTrafficAllow = AaaUnresponsiveTrafficAllow(
         False, False, 0 )
   else:
      vlan = args.get( 'VLAN', 0 )
      config.aaaUnresponsiveTrafficAllow = AaaUnresponsiveTrafficAllow(
         True, True, Tac.Value( 'Bridging::VlanIdOrNone', vlan and vlan.id ) )
   config.aaaUnresponsiveApplyCachedResults = 'cached-results' in args

def noAaaUnresponsiveTrafficAllow( mode, args ):
   config.aaaUnresponsiveApplyCachedResultsTimeout = \
         CachedResultsTimeout( Tac.endOfTime,
               CachedResultsTimeoutUnit.timeoutUnitNone )
   config.aaaUnresponsiveTrafficAllow = AaaUnresponsiveTrafficAllow(
      False, False, 0 )
   config.aaaUnresponsiveApplyCachedResults = False

class Dot1XAaaUnresponsiveTrafficActionAllowCmd( CliCommand.CliCommandClass ):
   syntax = f'''aaa unresponsive action
   ( ( apply cached-results{cacheTimeoutArg} [ else traffic allow [ vlan VLAN ] ] )
   | ( traffic allow [ vlan VLAN ] ) )'''
   noOrDefaultSyntax = 'aaa unresponsive action ...'
   data = {
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'action': matcherDot1xAaaUnrespAction,
      'apply' : matcherDot1xAaaUnrespActionApply,
      'cached-results' : matcherDot1xAaaUnrespCachedResults,
      'timeout': matcherDot1xAaaCachedTimeout,
      'TIMEOUT_SEC': matcherDot1xAaaCachedTimeoutSec,
      'TIMEOUT_MIN': matcherDot1xAaaCachedTimeoutMin,
      'TIMEOUT_HRS': matcherDot1xAaaCachedTimeoutHrs,
      'TIMEOUT_DAY': matcherDot1xAaaCachedTimeoutDay,
      'seconds': matcherDot1xTimeoutSeconds,
      'minutes': matcherDot1xTimeoutMinutes,
      'hours': matcherDot1xTimeoutHours,
      'days': matcherDot1xTimeoutDays,
      'else' : matcherDot1xElse,
      'traffic': matcherDot1xAaaUnrespActionTraffic,
      'allow': matcherDot1xAaaUnrespActionTrafficAllow,
      'vlan': 'Allow traffic in VLAN when AAA times out',
      'VLAN': vlanIdMatcher,
   }
   handler = setAaaUnresponsiveTrafficAllow
   noOrDefaultHandler = noAaaUnresponsiveTrafficAllow

Dot1xMode.addCommandClass( Dot1XAaaUnresponsiveTrafficActionAllowCmd )

#------------------------------------------------------------
# (dot1x)# [no|default] aaa unresponsive phone action apply cached-results else
#                                                                       traffic allow
# (dot1x)# [no|default] aaa unresponsive phone action traffic allow
# (dot1x)# [no|default] aaa unresponsive phone action apply cached-results
#------------------------------------------------------------
def setAaaUnresponsivePhoneAllow( mode, args ):
   populateApplyCachedResultsTimeout( config, args, phone=True )

   config.aaaUnresponsivePhoneAllow = 'allow' in args
   if 'cached-results' in args:
      config.aaaUnresponsivePhoneApplyCachedResults = 'aaaUnresponsiveApplyCached'
   else:
      config.aaaUnresponsivePhoneApplyCachedResults = \
         'aaaUnresponsiveDoNotApplyCached'

def noAaaUnresponsivePhoneAllow( mode, args ):
   config.aaaUnresponsivePhoneApplyCachedResultsTimeout = \
         CachedResultsTimeout( Tac.endOfTime,
               CachedResultsTimeoutUnit.timeoutUnitNone )
   config.aaaUnresponsivePhoneAllow = False
   config.aaaUnresponsivePhoneApplyCachedResults = 'aaaUnresponsiveCacheInheritRule'

class Dot1XAaaUnresponsivePhoneActionAllowCmd( CliCommand.CliCommandClass ):
   syntax = f'''aaa unresponsive phone action
   ( ( apply cached-results{cacheTimeoutArg} [ else traffic allow ] ) | \
( traffic allow ) )'''

   noOrDefaultSyntax = 'aaa unresponsive phone action ...'
   data = {
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'phone': 'Configure AAA timeout options for phone',
      'action': matcherDot1xAaaUnrespAction,
      'apply' : matcherDot1xAaaUnrespActionApply,
      'cached-results' : matcherDot1xAaaUnrespCachedResults,
      'timeout': matcherDot1xAaaCachedTimeout,
      'TIMEOUT_SEC': matcherDot1xAaaCachedTimeoutSec,
      'TIMEOUT_MIN': matcherDot1xAaaCachedTimeoutMin,
      'TIMEOUT_HRS': matcherDot1xAaaCachedTimeoutHrs,
      'TIMEOUT_DAY': matcherDot1xAaaCachedTimeoutDay,
      'minutes': matcherDot1xTimeoutMinutes,
      'hours': matcherDot1xTimeoutHours,
      'days': matcherDot1xTimeoutDays,
      'seconds': matcherDot1xTimeoutSeconds,
      'else' : matcherDot1xElse,
      'traffic': matcherDot1xAaaUnrespActionTraffic,
      'allow': 'Allow phone traffic in phone VLAN when AAA times out',
   }
   handler = setAaaUnresponsivePhoneAllow
   noOrDefaultHandler = noAaaUnresponsivePhoneAllow

Dot1xMode.addCommandClass( Dot1XAaaUnresponsivePhoneActionAllowCmd )

#------------------------------------------------------------
# (dot1x)# [no|default] aaa unresponsive eap response [ success | disabled ]
#------------------------------------------------------------

def setAaaUnresponsiveEapResponse( mode, args ):
   if 'disabled' in args:
      config.aaaUnresponsiveEapResponse = 'aaaUnresponsiveEapResponseDisabled'
   else:
      config.aaaUnresponsiveEapResponse = 'aaaUnresponsiveEapResponseSuccess'

class Dot1xAaaUnresponsiveEapResponse( CliCommand.CliCommandClass ):
   syntax = '''aaa unresponsive eap response [ success | disabled ]'''
   noOrDefaultSyntax = 'aaa unresponsive eap response ...'
   data = {
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'eap': 'EAP options',
      'response': 'EAP response to send',
      'disabled': 'Send no response',
      'success': 'Send EAP success (default)',
   }
   handler = setAaaUnresponsiveEapResponse
   noOrDefaultHandler = setAaaUnresponsiveEapResponse

Dot1xMode.addCommandClass( Dot1xAaaUnresponsiveEapResponse )

# ------------------------------------------------------------
# (dot1x)# [no|default] aaa unresponsive recovery action reauthenticate
# ------------------------------------------------------------
def setUnresponsiveRecoveryAction( mode, args ):
   config.aaaUnresponsiveRecoveryReauth = True

def noUnresponsiveRecoveryAction( mode, args ):
   config.aaaUnresponsiveRecoveryReauth = False

class Dot1xAaaUnresponsiveRecoveryAction( CliCommand.CliCommandClass ):
   syntax = 'aaa unresponsive recovery action reauthenticate'
   noOrDefaultSyntax = syntax
   data = {
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'recovery': 'Configure unresponsive recovery',
      'action': 'Configure recovery action',
      'reauthenticate': 'Re-authenticate supplicant',
   }
   handler = setUnresponsiveRecoveryAction
   noOrDefaultHandler = noUnresponsiveRecoveryAction

Dot1xMode.addCommandClass( Dot1xAaaUnresponsiveRecoveryAction )

#--------------------------------------------------------------------------
# (dot1x)# [no|default] aaa accounting update interval [ INTERVAL ] seconds
#--------------------------------------------------------------------------
def setDot1xAccountingUpdateInterval( mode, args ):
   config.accountingUpdateInterval = args[ 'INTERVAL' ]

def noDot1xAccountingUpdateInterval( mode, args ):
   config.accountingUpdateInterval = config.defaultAccountingUpdateInterval

matcherIntervalPeriod = CliMatcher.IntegerMatcher( 5, 65535, 
      helpdesc='Interval period in seconds' )

class Dot1XAccountingUpdateIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'aaa accounting update interval INTERVAL seconds'
   noOrDefaultSyntax = 'aaa accounting update interval ...'
   data = {
      'aaa': matcherDot1xAaa,
      'accounting': 'Configure accounting',
      'update' : 'Configure update messages',
      'interval': 'Configure interval',
      'INTERVAL' : matcherIntervalPeriod,
      'seconds' : 'Unit in seconds'
   }
   handler = setDot1xAccountingUpdateInterval
   noOrDefaultHandler = noDot1xAccountingUpdateInterval

Dot1xMode.addCommandClass( Dot1XAccountingUpdateIntervalCmd )

#---------------------------------------------------------------------------
# (dot1x)# [no|default] statistics packets dropped
#---------------------------------------------------------------------------

def droppedCounterSupportedGuard( self, token ):
   if hwstatus.dot1xDroppedCounterSupported:
      return None
   return CliParser.guardNotThisPlatform

statisticsNode = CliCommand.guardedKeyword( 'statistics',
                                            '802.1X port authentication statistics',
                                            guard=droppedCounterSupportedGuard )

def setDroppedCounter( mode, args ):
   config.dropCounterEnabled = True

def noDroppedCounter( mode, args ):
   config.dropCounterEnabled = False

class StatisticsPacketDropCmd( CliCommand.CliCommandClass ):
   syntax = 'statistics packets dropped'
   noOrDefaultSyntax = syntax
   data = {
      'statistics': statisticsNode,
      'packets': '802.1X port authentication data packet statistics',
      'dropped': '802.1X port authentication dropped data packet statistics',
   }

   handler = setDroppedCounter
   noOrDefaultHandler = noDroppedCounter

Dot1xMode.addCommandClass( StatisticsPacketDropCmd )

#------------------------------------------------------------
# [no|default] dot1x pae authenticator
#------------------------------------------------------------
def setDot1xPaeAuthenticator( mode, args ):
   # Disable supplicant configuration on the interface in case it was configured 
   # earlier.
   noDot1xPaeSupplicant( mode, args )
   
   # Configure the interface to be the authenticator.
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.dot1xEnabled = True

def noDot1xPaeAuthenticator( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.dot1xEnabled = False

matcherIntfDot1x = CliMatcher.KeywordMatcher( 'dot1x',
                                       helpdesc='IEEE 802.1X port authentication' )
nodeIntfDot1x = CliCommand.Node( matcherIntfDot1x, guard=dot1xSupportedGuard )
nodeSubIntfDot1x = CliCommand.Node( matcherIntfDot1x,
                                    guard=dot1xOnSubIntfSupportedGuard )
matcherPae = CliMatcher.KeywordMatcher( 'pae',
         helpdesc='Set 802.1X interface Port Access Entity type' )

class Dot1XPaeAuthenticatorCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x pae authenticator'
   noOrDefaultSyntax = 'dot1x pae authenticator ...'
   handler = setDot1xPaeAuthenticator
   noOrDefaultHandler = noDot1xPaeAuthenticator

class Dot1XPaeAuthenticatorOnIntfCmd( Dot1XPaeAuthenticatorCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'pae': matcherPae,
      'authenticator': 'Set Port Access Entity type as Authenticator',
   }

EthIntfModelet.addCommandClass( Dot1XPaeAuthenticatorOnIntfCmd )

class Dot1XPaeAuthenticatorOnSubIntfCmd( Dot1XPaeAuthenticatorCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'pae': matcherPae,
      'authenticator': 'Set Port Access Entity type as Authenticator',
   }

SubIntfModelet.addCommandClass( Dot1XPaeAuthenticatorOnSubIntfCmd )
#------------------------------------------------------------
# [no|default] dot1x pae supplicant <profileName>
#------------------------------------------------------------
def setDot1xPaeSupplicant( mode, args ):
   profileName = args[ 'PROFILE' ]
   # Disable authenticator on the interface if it was configured earlier.
   noDot1xPaeAuthenticator( mode, args )
   # Configure the interface to be the supplicant.
   setDot1xSupplicant( mode, profileName )

def noDot1xPaeSupplicant( mode, args ):
   noDot1xSupplicant( mode )

class Dot1XPaeSupplicantCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x pae supplicant PROFILE'
   noOrDefaultSyntax = 'dot1x pae supplicant ...'
   handler = setDot1xPaeSupplicant
   noOrDefaultHandler = noDot1xPaeSupplicant

class Dot1XPaeSupplicantOnIntfCmd( Dot1XPaeSupplicantCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'pae': matcherPae,
      'supplicant': matcherSupplicant,
      'PROFILE': profileExpression,
   }

EthIntfModelet.addCommandClass( Dot1XPaeSupplicantOnIntfCmd )

class Dot1XPaeSupplicantOnSubIntfCmd( Dot1XPaeSupplicantCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'pae': matcherPae,
      'supplicant': matcherSupplicant,
      'PROFILE': profileExpression,
   }

SubIntfModelet.addCommandClass( Dot1XPaeSupplicantOnSubIntfCmd )
#------------------------------------------------------------
# [no|default dot1x 
#------------------------------------------------------------
def gotoDot1xMode( mode, args ):
   childMode = mode.childMode( Dot1xMode )
   mode.session_.gotoChildMode( childMode )

def noDot1xMode( mode, args ):
   Dot1xMode.clear( mode, args ) # Also clears profiles.
   # 'radius av-pair service-type'
   config.serviceType = False
   # 'radius av-pair filter-id ...'
   config.multipleFilterId = False
   config.filterIdDelim = False
   config.applyIpv4v6Acl = False
   # 'radius av-pair framed-mtu LENGTH'
   config.framedMtu = config.defaultFramedMtu
   # 'radius av-pair lldp ...'
   profilingState = ProfilingStateEnum.disableProfiling
   config.lldpSysNameProfilingState = profilingState
   config.lldpSysDescProfilingState = profilingState
   # 'radius av-pair dhcp ...'
   config.dot1xProfilingDhcpOption.clear()

class Dot1XCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x'
   noOrDefaultSyntax = 'dot1x'
   data = {
      'dot1x': nodeDot1x,
   }
   handler = gotoDot1xMode
   noOrDefaultHandler = noDot1xMode

BasicCliModes.GlobalConfigMode.addCommandClass( Dot1XCmd )

# ------------------------------------------------------------
# [no|default] result-caching
# ------------------------------------------------------------
def gotoResultCachingMode( mode, args ):
   childMode = mode.childMode( Dot1xCacheMode )
   mode.session_.gotoChildMode( childMode )

def noResultCachingMode( mode, args ):
   config.eapolLogoffIgnore = False

class Dot1xResultCachingCmd( CliCommand.CliCommandClass ):
   syntax = 'result-caching'
   noOrDefaultSyntax = 'result-caching'
   data = {
         'result-caching': 'Configure cache results',
         }
   handler = gotoResultCachingMode
   noOrDefaultHandler = noResultCachingMode

Dot1xMode.addCommandClass( Dot1xResultCachingCmd )

# -------------------------------------------------------------------
# (config-dot1x) event mac new action send request-identity unicast
# -------------------------------------------------------------------

def setSendIdReqMacNew( mode, args ):
   config.sendIdentityReqNewMac = 'macNewSendIdReq'

def disableSendIdReqMacNew( mode, args ):
   config.sendIdentityReqNewMac = 'doNotSendIdReq'

class Dot1xSendIdReqMacNew( CliCommand.CliCommandClass ):
   syntax = 'event mac new action send request-identity unicast'
   noOrDefaultSyntax = syntax
   data = {
         'event': 'Configure an action for an event',
         'mac': 'Configure an action for a MAC related event',
         'new': 'Configure an action for a new MAC',
         'action': 'Set action for an event',
         'send': 'Send 802.1X packet',
         'request-identity': 'Send EAP identity request packet',
         'unicast': 'Send a unicast identity request packet'
         }
   handler = setSendIdReqMacNew
   noOrDefaultHandler = disableSendIdReqMacNew

Dot1xMode.addCommandClass( Dot1xSendIdReqMacNew )

# -------------------------------------------------------------------
# (config-dot1x) event mac logged-off action send request-identity unicast
# -------------------------------------------------------------------

def setSendIdReqMacDisconnect( mode, args ):
   config.sendIdentityReqDisconnectMac = 'macDisconnectSendIdReq'

def disableSendIdReqMacDisconnect( mode, args ):
   config.sendIdentityReqDisconnectMac = 'doNotSendIdReq'

class Dot1xSendIdReqMacDisconnect( CliCommand.CliCommandClass ):
   syntax = 'event mac logged-off action send request-identity unicast'
   noOrDefaultSyntax = syntax
   data = {
         'event': 'Configure an action for an event',
         'mac': 'Configure an action for a MAC related event',
         'logged-off': 'Configure an action when a MAC is logged off',
         'action': 'Set action for an event',
         'send': 'Send 802.1X packet',
         'request-identity': 'Send EAP identity request packet',
         'unicast': 'Send a unicast identity request packet'
         }
   handler = setSendIdReqMacDisconnect
   noOrDefaultHandler = disableSendIdReqMacDisconnect

Dot1xMode.addCommandClass( Dot1xSendIdReqMacDisconnect )

#------------------------------------------------------------
# dot1x send request-identity <INTFS>
# ------------------------------------------------------------
def setAuthenticateIntf( mode, args ):
   intfList = args[ 'INTFS' ]
   for intfName in intfList:
      dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( intfName )
      if dot1xIntfConfigReq:
         reauthCause = Tac.Type( 'Dot1x::Dot1xManualAuthRequestCause' )
         dot1xManualAuthReq = Tac.Value( 'Dot1x::Dot1xManualAuthRequest',
               cause=reauthCause.intfListMultiCast,
               time=Tac.now() )
         dot1xIntfConfigReq.manualAuthRequest = dot1xManualAuthReq

class Dot1XAuthenticateCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x send request-identity INTFS'
   data = {
      'dot1x': nodeDot1x,
      'send': '802.1X Packet',
      'request-identity': 'Request Identity Packet',
      'INTFS': intfRangeMatcher,
   }
   handler = setAuthenticateIntf

BasicCliModes.EnableMode.addCommandClass( Dot1XAuthenticateCmd )

# ----------------------------------------------------------------
# dot1x send request-identity <INTF> host <MAC> [ vlan <VLAN> ]
# ---------------------------------------------------------------
def setAuthenticateIntfHost( mode, args ):
   intf = args[ 'INTF' ]
   dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( intf.name )
   if dot1xIntfConfigReq:
      vlan = args.get( 'VLAN', 0 )
      reauthCause = Tac.Type( 'Dot1x::Dot1xManualAuthRequestCause' )
      manualAuthVlanId = Tac.Value( 'Bridging::VlanIdOrNone',
                                                          vlan and vlan.id )
      manualAuthHost = args[ 'MAC' ]
      dot1xManualAuthReq = Tac.Value( 'Dot1x::Dot1xManualAuthRequest',
            cause=reauthCause.perHostAndVlan,
            time=Tac.now(),
            host=manualAuthHost,
            vlanId=manualAuthVlanId )
      dot1xIntfConfigReq.manualAuthRequest = dot1xManualAuthReq

class Dot1XAuthenticateHostCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x send request-identity INTF host MAC [ vlan VLAN ] '
   data = {
      'dot1x': nodeDot1x,
      'send': '802.1X Packet',
      'request-identity': 'Request Identity Packet',
      'INTF': EthPhyIntf.ethMatcher,
      'host': '802.1X host MAC',
      'MAC': MacAddr.macAddrMatcher,
      'vlan': 'VLAN tag',
      'VLAN': vlanIdMatcher,
   }
   handler = setAuthenticateIntfHost

BasicCliModes.EnableMode.addCommandClass( Dot1XAuthenticateHostCmd )

# ---------------------------------------------------------------------------
# ------------------------------------------------------------
# [no|default] dot1x re-authenticate
#------------------------------------------------------------
def setReauthenticateIntf( mode, args ):
   intf = args[ 'IPINTF' ]
   dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( intf.name )
   if dot1xIntfConfigReq:
      reauthCause = Tac.Type( 'Dot1x::Dot1xManualAuthRequestCause' )
      dot1xManualAuthReq = Tac.Value( 'Dot1x::Dot1xManualAuthRequest',
            cause=reauthCause.perIntfUnicast,
            time=Tac.now() )
      dot1xIntfConfigReq.manualAuthRequest = dot1xManualAuthReq

class Dot1XReAuthenticateIpintfCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x re-authenticate IPINTF'
   data = {
      'dot1x': nodeDot1x,
      're-authenticate': 'Manually re-authenticate 802.1X supplicant',
      'IPINTF': Intf.matcherWithIpSupport,
   }
   handler = setReauthenticateIntf

BasicCliModes.EnableMode.addCommandClass( Dot1XReAuthenticateIpintfCmd )

#---------------------------------------------------------------------------
# [no|default] dot1x port-control auto|force-authorized|force-unauthorized
#---------------------------------------------------------------------------
def setPortControl( mode, args ):
   portControl = args[ 'PORT_STATE' ]
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( mode.intf.name )
   assert dot1xIntfConfigReq, \
          "dot1xIntfConfig and Req should have been made together"
   dot1xIntfConfigReq.lastPortCtrlSetting = dot1xIntfConfig.portCtrlSetting
   dot1xIntfConfig.portCtrlSetting = portControlStr2Enum[ portControl ]

def noPortControl( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( mode.intf.name )
      dot1xIntfConfigReq.lastPortCtrlSetting = dot1xIntfConfig.portCtrlSetting
      dot1xIntfConfig.portCtrlSetting = 'forceAuth'

matcherPortControl = CliMatcher.KeywordMatcher( 'port-control',
                              helpdesc='Set port control state' )
matcherPortState = CliMatcher.EnumMatcher( {
   'auto': 'Set port state to automatic',
   'force-authorized': 'Set port state to authorized',
   'force-unauthorized': 'Set port state to unauthorized',
} )

class Dot1XPortControlCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x port-control PORT_STATE'
   noOrDefaultSyntax = 'dot1x port-control [ PORT_STATE ]'
   handler = setPortControl
   noOrDefaultHandler = noPortControl

class Dot1XPortControlOnIntfCmd( Dot1XPortControlCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'port-control': matcherPortControl,
      'PORT_STATE': matcherPortState,
   }

EthIntfModelet.addCommandClass( Dot1XPortControlOnIntfCmd )

class Dot1XPortControlOnSubIntfCmd( Dot1XPortControlCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'port-control': matcherPortControl,
      'PORT_STATE': matcherPortState,
   }

SubIntfModelet.addCommandClass( Dot1XPortControlOnSubIntfCmd )

#---------------------------------------------------------------------------
# [no|default] dot1x port-control force-authorized phone
#---------------------------------------------------------------------------
class Dot1xForceAuthorizedPhoneOnIntfBase( CliCommand.CliCommandClass ):
   syntax = 'dot1x port-control force-authorized phone'
   noOrDefaultSyntax = syntax

   data = {
      'dot1x': nodeIntfDot1x,
      'port-control': matcherPortControl,
      'force-authorized': 'Set port state to authorized',
      'phone': 'Set port state to authorized for phone devices',
   }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
      dot1xIntfConfig.forceAuthPhone = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
      dot1xIntfConfig.forceAuthPhone = False

class Dot1xForceAuthorizedPhoneOnIntfCmd( Dot1xForceAuthorizedPhoneOnIntfBase ):
   data = Dot1xForceAuthorizedPhoneOnIntfBase.data.copy()

EthIntfModelet.addCommandClass( Dot1xForceAuthorizedPhoneOnIntfCmd )

class Dot1xForceAuthorizedPhoneOnSubintfCmd( Dot1xForceAuthorizedPhoneOnIntfBase ):
   data = Dot1xForceAuthorizedPhoneOnIntfBase.data | { 'dot1x': nodeSubIntfDot1x }
   
SubIntfModelet.addCommandClass( Dot1xForceAuthorizedPhoneOnSubintfCmd )

#-----------------------------------------------------------
# [no|default] dot1x timeout quiet-period|tx-period
#-----------------------------------------------------------
def noQuietPeriod( mode ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.quietPeriod = dot1xIntfConfig.defaultQuietPeriod

def noTxPeriod( mode ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.txPeriod = dot1xIntfConfig.defaultTxPeriod

matcherTimeout = CliMatcher.KeywordMatcher( 'timeout', 
      helpdesc='Configure timeout setting' )
matcherTimeoutPeriod = CliMatcher.IntegerMatcher( 1, 65535,
      helpdesc='Timeout period in seconds' )
# helpname alters the default help text for the new range
maxReauth = ( 2 ** 32 ) - 1
matcherReauthPeriod = CliMatcher.IntegerMatcher( 1, maxReauth,
                                                 helpname=f'<60-{maxReauth}>',
                                                 helpdesc='Timeout period in '
                                                          'seconds' )

class Dot1XTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x timeout PERIOD TIMEOUT'
   noOrDefaultSyntax = 'dot1x timeout PERIOD ...'
   data = {
      'timeout': matcherTimeout,
      'PERIOD': CliMatcher.EnumMatcher( {
         'quiet-period': 'Timeout for quiet period after a failed authentication',
         'tx-period': 'Timeout for supplicant retries',
      } ),
      'TIMEOUT': matcherTimeoutPeriod,
   }
   
   @staticmethod
   def handler( mode, args ):
      attr = 'quietPeriod' if args[ 'PERIOD' ] == 'quiet-period' else 'txPeriod'
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
      setattr( dot1xIntfConfig, attr, args[ 'TIMEOUT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'quiet-period' in args[ 'PERIOD' ]:
         noQuietPeriod( mode )
      else:
         noTxPeriod( mode )

class Dot1XTimeoutCmdOnIntf( Dot1XTimeoutCmd ):
   data = Dot1XTimeoutCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XTimeoutCmdOnIntf )

class Dot1XTimeoutCmdOnSubIntf( Dot1XTimeoutCmd ):
   data = Dot1XTimeoutCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XTimeoutCmdOnSubIntf )
#-----------------------------------------------------------------
# [no|default] dot1x reauthentication
# [no|default] dot1x timeout reauth-period [<60-65535>|server]
#-----------------------------------------------------------------
def setReauth( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   if not dot1xIntfConfig.reauth:
      dot1xIntfConfig.reauth = True

def noReauth( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.reauth = False

class Dot1XReauthenticationCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x reauthentication'
   noOrDefaultSyntax = 'dot1x reauthentication ...'
   data = {
      'reauthentication': 'Enable or disable reauthentication',
   }
   handler = setReauth
   noOrDefaultHandler = noReauth

class Dot1XReauthenticationCmdOnIntf( Dot1XReauthenticationCmd ):
   data = Dot1XReauthenticationCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XReauthenticationCmdOnIntf )

class Dot1XReauthenticationCmdOnSubIntf( Dot1XReauthenticationCmd ):
   data = Dot1XReauthenticationCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XReauthenticationCmdOnSubIntf )

class Dot1XReauthPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x timeout reauth-period ( TIMEOUT | server )'
   noOrDefaultSyntax = 'dot1x timeout reauth-period ...'
   data = {
      'timeout': matcherTimeout,
      'reauth-period': 'Timeout for periodically re-authenticate supplicant',
      'TIMEOUT': matcherReauthPeriod,
      'server': 'Obtain re-authentication timeout value from the server',
   }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
      if 'server' in args:
         dot1xIntfConfig.reauthOpts = 'useAuthSrvrSettings'
      else:
         # the help text for period has been changed to suggest only 60 seconds or
         # above is supported.  However for upgrade purposes, we will detect smaller
         # values and use a minimum 60 second value.  If interactive, the warning
         # will be printed.
         period = args[ 'TIMEOUT' ]
         if period < 60:
            mode.addWarning(
               "A reauth period of less than 60 seconds is no longer supported" )
            period = 60
         dot1xIntfConfig.reauthPeriod = period
         dot1xIntfConfig.reauthOpts = 'useSetTimeout'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
      if dot1xIntfConfig:
         dot1xIntfConfig.reauthPeriod = dot1xIntfConfig.defaultReauthPeriod
         dot1xIntfConfig.reauthOpts = 'useSetTimeout'

class Dot1XReauthPeriodCmdOnIntf( Dot1XReauthPeriodCmd ):
   data = Dot1XReauthPeriodCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XReauthPeriodCmdOnIntf )

class Dot1XReauthPeriodCmdOnSubIntf( Dot1XReauthPeriodCmd ):
   data = Dot1XReauthPeriodCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XReauthPeriodCmdOnSubIntf )

#------------------------------------------------------------ 
# [no|default] dot1x timeout reauth-timeout-ignore always
#------------------------------------------------------------ 
def setDot1xReauthTimeoutIgnore( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name ) 
   dot1xIntfConfig.reauthTimeoutIgnore = True 

def noDot1xReauthTimeoutIgnore( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name ) 
   if dot1xIntfConfig:
      dot1xIntfConfig.reauthTimeoutIgnore = False 

matcherTimeout = CliMatcher.KeywordMatcher( 'timeout',
         helpdesc='Configure timeout setting' )
matcherReauthTimeoutIgnore = CliMatcher.KeywordMatcher( 'reauth-timeout-ignore',
         helpdesc='Retain current port auth status on reauth server timeouts' )
matcherAlways = CliMatcher.KeywordMatcher( 'always',
         helpdesc='Retain port auth status indefinitely on reauth server timeouts' )

class Dot1XReauthTimeoutIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x timeout reauth-timeout-ignore always'
   noOrDefaultSyntax = 'dot1x timeout reauth-timeout-ignore ...'
   handler = setDot1xReauthTimeoutIgnore
   noOrDefaultHandler = noDot1xReauthTimeoutIgnore

class Dot1XReauthTimeoutIgnoreOnIntfCmd( Dot1XReauthTimeoutIgnoreCmd ):
   data = {
      'dot1x': nodeIntfDot1x,
      'timeout': matcherTimeout,
      'reauth-timeout-ignore': matcherReauthTimeoutIgnore,
      'always': matcherAlways,
   }

EthIntfModelet.addCommandClass( Dot1XReauthTimeoutIgnoreOnIntfCmd )

class Dot1XReauthTimeoutIgnoreOnSubIntfCmd( Dot1XReauthTimeoutIgnoreCmd ):
   data = {
      'dot1x': nodeSubIntfDot1x,
      'timeout': matcherTimeout,
      'reauth-timeout-ignore': matcherReauthTimeoutIgnore,
      'always': matcherAlways,
   }

SubIntfModelet.addCommandClass( Dot1XReauthTimeoutIgnoreOnSubIntfCmd )

class Dot1XTimeoutIdleHost( CliCommand.CliCommandClass ):
   syntax = 'dot1x timeout idle-host TIMEOUT seconds'
   noOrDefaultSyntax = 'dot1x timeout idle-host ... '
   data = {
         'timeout': matcherTimeout,
         'idle-host': 'Timeout for idle host',
         'TIMEOUT': CliMatcher.IntegerMatcher( 10, 65535,
                     helpdesc='Idle timeout value' ),
         'seconds': 'Unit in seconds'
         }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
      dot1xIntfConfig.idleTimeout = args[ 'TIMEOUT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
      if not dot1xIntfConfig:
         return
      dot1xIntfConfig.idleTimeout = dot1xIntfConfig.defaultIdleTimeout

class Dot1XTimeoutIdleHostOnIntf( Dot1XTimeoutIdleHost ):
   data = Dot1XTimeoutIdleHost.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XTimeoutIdleHostOnIntf )

class Dot1XTimeoutIdleHostOnSubIntf( Dot1XTimeoutIdleHost ):
   data = Dot1XTimeoutIdleHost.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XTimeoutIdleHostOnSubIntf )

#-----------------------------------------------------------------
# [ no | default ] dot1x reauthorization request limit
#
# legacy:
# [ no | default ] dot1x max-reauth-req
#-----------------------------------------------------------------
def setMaxReauthReq( mode, args ):
   req = args[ 'LIMIT' ]
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.maxReauthReq = req

def noMaxReauthReq( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.maxReauthReq = dot1xIntfConfig.defaultMaxReauthReq

# [ no | default ] dot1x max-reauth-req - deprecated 

matcherReqLimit = CliMatcher.IntegerMatcher( 1, 10,
      helpdesc='Max number of reauth request allowed' )

class Dot1XMaxReauthReqCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x max-reauth-req LIMIT'
   noOrDefaultSyntax = 'dot1x max-reauth-req ...'
   data = {
      'max-reauth-req': CliCommand.Node( 
         CliMatcher.KeywordMatcher( 'max-reauth-req',
         helpdesc='Maximum number of reauthentication attempts' ),
         deprecatedByCmd='dot1x reauthorization request limit' ),
      'LIMIT': matcherReqLimit,
   }
   handler = setMaxReauthReq
   noOrDefaultHandler = noMaxReauthReq
   
class Dot1XMaxReauthReqCmdOnIntf( Dot1XMaxReauthReqCmd ):
   data = Dot1XMaxReauthReqCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x
   
EthIntfModelet.addCommandClass( Dot1XMaxReauthReqCmdOnIntf )

class Dot1XMaxReauthReqCmdOnSubIntf( Dot1XMaxReauthReqCmd ):
   data = Dot1XMaxReauthReqCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XMaxReauthReqCmdOnSubIntf )

# [ no | default ] dot1x reauthorization request limit

class Dot1XReauthReqLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x reauthorization request limit LIMIT'
   noOrDefaultSyntax = 'dot1x reauthorization request limit ...'
   data = {
      'reauthorization': 'Maximum number of reauthentication attempts',
      'request': 'Maximum number of reauthentication attempts',
      'limit': 'Maximum number of reauthentication attempts',
      'LIMIT': matcherReqLimit,
   }
   handler = setMaxReauthReq
   noOrDefaultHandler = noMaxReauthReq

class Dot1XReauthReqLimitCmdOnIntf( Dot1XReauthReqLimitCmd ):
   data = Dot1XReauthReqLimitCmd.data.copy()
   data[ 'dot1x' ] = nodeIntfDot1x

EthIntfModelet.addCommandClass( Dot1XReauthReqLimitCmdOnIntf )

class Dot1XReauthReqLimitCmdOnSubIntf( Dot1XReauthReqLimitCmd ):
   data = Dot1XReauthReqLimitCmd.data.copy()
   data[ 'dot1x' ] = nodeSubIntfDot1x

SubIntfModelet.addCommandClass( Dot1XReauthReqLimitCmdOnSubIntf )

#------------------------------------------------------------------------------
# [no|default] dot1x host-mode ( ( single-host [ session replaceable ] ) |
#                                ( multi-host [authenticated] ) )
#------------------------------------------------------------------------------
def singleHostModeGuard( mode, token ):
   if hwstatus.singleHostModeSupported:
      return None
   return CliParser.guardNotThisPlatform

matcherHostMode = CliMatcher.KeywordMatcher( 'host-mode',
                  helpdesc='Set the Host mode for authentication on this interface' )
matcherSingleHost = CliMatcher.KeywordMatcher( 'single-host',
                                               helpdesc='Single host mode' )
nodeSingleHost = CliCommand.Node( matcherSingleHost, guard=singleHostModeGuard )
matcherMultiHost = CliMatcher.KeywordMatcher( 'multi-host',
                                              helpdesc='Multiple host mode' )
matcherAuthenticated = CliMatcher.KeywordMatcher( 'authenticated',
                        helpdesc='Require each host to authenticate individually' )
nodeAuthenticated = CliCommand.Node( matcherAuthenticated,
                                     guard=singleHostModeGuard )
matcherSession = CliMatcher.KeywordMatcher(
      'session', helpdesc='Configure single-host session parameters' )
matcherReplaceable = CliMatcher.KeywordMatcher(
      'replaceable', helpdesc='Enable replacing Dot1x session in single-host mode' )
nodeReplaceable = CliCommand.Node( matcherReplaceable )

class Dot1XHostModeCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x host-mode ( ( single-host [ session replaceable ] ) | \
         ( multi-host [ authenticated ] ) )'
   noOrDefaultSyntax = 'dot1x host-mode ...'
   data = {
      'dot1x' : nodeIntfDot1x,
      'host-mode' : matcherHostMode,
      'single-host' : nodeSingleHost,
      'session': matcherSession,
      'replaceable': nodeReplaceable,
      'multi-host' : matcherMultiHost,
      'authenticated' : nodeAuthenticated,
   }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
      if dot1xIntfConfig:
         # For all other host-modes, sessionReplaceEnabled should be false
         sessionReplaceEnabled = False
         if 'single-host' in args:
            dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'single-host' ]
            sessionReplaceEnabled = ( 'session' and 'replaceable' in args )
         elif 'multi-host' in args and 'authenticated' in args:
            dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'multi-host authenticated' ]
         elif 'multi-host' in args:
            dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'multi-host' ]
         dot1xIntfConfig.sessionReplaceEnabled = sessionReplaceEnabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
      if dot1xIntfConfig:
         dot1xIntfConfig.hostMode = hostModeStr2Enum[ 'multi-host' ]
         dot1xIntfConfig.sessionReplaceEnabled = False

EthIntfModelet.addCommandClass( Dot1XHostModeCmd )

# ------------------------------------------------------------------
# [no|default] dot1x eapol disabled 
#-----------------------------------------------------------------
def setEapolDisabled( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.eapolDisabled = True

def noEapolDisabled( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.eapolDisabled = dot1xIntfConfig.eapolDisabledDefault

class Dot1xEapolDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x eapol disabled'
   noOrDefaultSyntax = syntax
   data = {
      'dot1x' : nodeIntfDot1x, 
      'eapol' : 'Configure Dot1x EAPOL attribute',
      'disabled' : 'Disable EAPOL processing for this interface'
   }
   handler = setEapolDisabled
   noOrDefaultHandler = noEapolDisabled

EthIntfModelet.addCommandClass( Dot1xEapolDisabledCmd )

#-----------------------------------------------------------------
# [no|default] dot1x eapol authentication failure fallback mba  [ timeout <value> ]
#-----------------------------------------------------------------
def setEapolAuthFailFallback( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   if dot1xIntfConfig.mbaAuthAlways:
      mode.addWarning(
            "'dot1x mac based authentication always' will supersede fallback." )
   dot1xIntfConfig.mbaFallback = True
   dot1xIntfConfig.mbaTimeout = args.get( 'TIMEOUT', 
         Dot1xConsts.mbaTimeoutDefault )

def noEapolAuthFailFallback( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      dot1xIntfConfig.mbaFallback = False
      dot1xIntfConfig.mbaTimeout = Dot1xConsts.mbaTimeoutDefault

class Dot1XEapolAuthFailFallbackCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x eapol authentication failure fallback mba [ timeout TIMEOUT ]'
   noOrDefaultSyntax = 'dot1x eapol authentication failure fallback mba ...'
   data = {
      'dot1x': nodeIntfDot1x,
      'eapol': 'Configure Dot1x EAPOL attribute',
      'authentication': 'Eapol authentication',
      'failure': 'Set authentication failure action',
      'fallback': 'Fallback to auth method',
      'mba': 'Use MAC authentication method for this interface',
      'timeout': 'Timeout for starting MAC authentication',
      'TIMEOUT': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='Timeout period in seconds' ),
   }
   handler = setEapolAuthFailFallback
   noOrDefaultHandler = noEapolAuthFailFallback

EthIntfModelet.addCommandClass( Dot1XEapolAuthFailFallbackCmd )

#-----------------------------------------------------------------
# [no|default] dot1x mac based authentication [ host-mode common ] [ always ]
#
# legacy:
# [no|default] dot1x mac authentication bypass
#-----------------------------------------------------------------
def setMacBasedAuth( mode, args, deprecatedCmd=False ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   switchIntfConfig = cliConfig.switchIntfConfig.get( mode.intf.name )
   if switchIntfConfig and not switchIntfConfig.enabled:
      mode.addWarning(
            "MAC based authentication is only supported on switchport" )
   dot1xIntfConfig.mbaEnabled = True
   if 'common' in args:
      dot1xIntfConfig.mbaHostMode = True
   elif 'always' in args:
      dot1xIntfConfig.mbaAuthAlways = True
      if dot1xIntfConfig.mbaFallback:
         mode.addWarning(
            "'dot1x mac based authentication always' will supersede fallback." )
   else:
      dot1xIntfConfig.mbaAuthAlways = False
      dot1xIntfConfig.mbaHostMode = False

def noMacBasedAuth( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if dot1xIntfConfig:
      if 'common' in args:
         dot1xIntfConfig.mbaHostMode = False
      elif 'always' in args:
         dot1xIntfConfig.mbaAuthAlways = False
      else:
         dot1xIntfConfig.mbaHostMode = False
         dot1xIntfConfig.mbaEnabled = False
         dot1xIntfConfig.mbaAuthAlways = False

# XXX Remove the CLI guard once BUG182344 is fixed.
nodeMac = CliCommand.Node( CliMatcher.KeywordMatcher( 'mac',
                  helpdesc='Configure Dot1x MAC attributes' ),
                  guard=dot1xMbaSupportedGuard )

# [no|default] dot1x mac authentication bypass - deprecated

nodeAuth = CliCommand.Node( CliMatcher.KeywordMatcher( 'authentication',
               helpdesc='Configure MAC authentication method for this interface' ),
                  deprecatedByCmd='dot1x mac based authentication' )

class Dot1XMacAuthBypassCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x mac authentication bypass'
   noOrDefaultSyntax = syntax
   data = {
      'dot1x': nodeIntfDot1x,
      'mac': nodeMac,
      'authentication': nodeAuth,
      'bypass': 'Configure MAC authentication bypass on this interface',
   }
   handler = setMacBasedAuth
   noOrDefaultHandler = noMacBasedAuth

EthIntfModelet.addCommandClass( Dot1XMacAuthBypassCmd )

# [no|default] dot1x mac based authentication
class Dot1XMacBasedAuthCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x mac based authentication [ host-mode common ] [ always ]'
   noOrDefaultSyntax = syntax
   data = {
      'dot1x': nodeIntfDot1x,
      'mac': nodeMac,
      'based': 'Configure MAC attributes to be used for this interface',
      'authentication': 'Configure MAC authentication method for this interface',
      'host-mode': 'Include MAC authentication method hosts',
      'common': 'Dot1x host-mode',
      'always': 'Retain MAC authentication session when EAPOL is in progress',
   }
   handler = setMacBasedAuth
   noOrDefaultHandler = noMacBasedAuth

EthIntfModelet.addCommandClass( Dot1XMacBasedAuthCmd )

#-----------------------------------------------------------------
# [no|default] dot1x mac based access-list
#-----------------------------------------------------------------
def setMacBasedAccessList( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.aclMode = 'modePerMac'

def noMacBasedAccessList( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.aclMode = 'modeInterface'

nodeAccessList = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'access-list',
      helpdesc='Enable interface to operate in per-mac access-list mode' ),
   guard=dot1xPerMacAclSupportedGuard )

class Dot1XMacBasedAccessListCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x mac based access-list'
   noOrDefaultSyntax = syntax
   data = {
      'dot1x': nodeIntfDot1x,
      'mac': nodeMac,
      'based': 'Configure MAC attributes to be used for this interface',
      'access-list': nodeAccessList,
   }
   handler = setMacBasedAccessList
   noOrDefaultHandler = noMacBasedAccessList

EthIntfModelet.addCommandClass( Dot1XMacBasedAccessListCmd )

#--------------------------------------------------------------------------
# (dot1x)# [no|default] mac based authentication delay DELAY seconds
#--------------------------------------------------------------------------
def setDot1xMbaDelay( mode, args ):
   config.mbaDelay = args[ 'DELAY' ]

def noDot1xMbaDelay( mode, args ):
   config.mbaDelay = config.defaultMbaDelay

delaySeconds = CliMatcher.IntegerMatcher( 0, 300, helpdesc='Delay in seconds' )

class Dot1xMbaDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'mac based authentication delay DELAY seconds'
   noOrDefaultSyntax = 'mac based authentication delay ...'
   data = {
      'mac': nodeMac,
      'based': 'Configure MAC attributes to be used for this interface',
      'authentication': 'Configure MAC authentication method for this interface',
      'delay': 'Configure delay',
      'DELAY': delaySeconds,
      'seconds': 'Unit in seconds'
   }
   handler = setDot1xMbaDelay
   noOrDefaultHandler = noDot1xMbaDelay

Dot1xMode.addCommandClass( Dot1xMbaDelayCmd )

#--------------------------------------------------------------------------
# (dot1x)# [no|default] mac based authentication hold period HOLDPERIOD seconds
#--------------------------------------------------------------------------
def setDot1xMbaHoldPeriod( mode, args ):
   config.mbaHoldPeriod = args[ 'HOLDPERIOD' ]

def noDot1xMbaHoldPeriod( mode, args ):
   config.mbaHoldPeriod = config.defaultMbaHoldPeriod

holdPeriodSeconds = CliMatcher.IntegerMatcher(
   1, 300, helpdesc='Hold period in seconds' )

class Dot1xMbaHoldPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'mac based authentication hold period HOLDPERIOD seconds'
   noOrDefaultSyntax = 'mac based authentication hold ...'
   data = {
      'mac': nodeMac,
      'based': 'Configure MAC attributes to be used for this interface',
      'authentication': 'Configure MAC authentication method for this interface',
      'hold': 'Configure wait time before MAC can authenticate again after '
              'failure or timeout',
      'period': 'Configure period',
      'HOLDPERIOD': holdPeriodSeconds,
      'seconds': 'Unit in seconds',
   }
   handler = setDot1xMbaHoldPeriod
   noOrDefaultHandler = noDot1xMbaHoldPeriod

Dot1xMode.addCommandClass( Dot1xMbaHoldPeriodCmd )

# -----------------------------------------------------------------
# (dot1x)# [no|default] session move allow eapol
# -----------------------------------------------------------------
class Dot1xSessionMoveCmd( CliCommand.CliCommandClass ):
   syntax = 'session move allow eapol'
   noOrDefaultSyntax = 'session move allow eapol'
   data = {
      'session': 'Configure session parameters',
      'move': 'Configure Dot1x session move',
      'allow': 'Allow Dot1x session move',
      'eapol': 'Configure Dot1x EAPOL session move',
   }

   @staticmethod
   def handler( mode, args ):
      config.sessionMoveEapol = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.sessionMoveEapol = False

Dot1xMode.addCommandClass( Dot1xSessionMoveCmd )

# ---------------------------------------------------------------------------
# (dot1x)# [no|default] single-host session replace detection [ errdisable ]
# ---------------------------------------------------------------------------
class Dot1xSessionReplaceDetectionCmd( CliCommand.CliCommandClass ):
   syntax = 'single-host session replace detection [ errdisable ]'
   noOrDefaultSyntax = 'single-host session replace detection ...'
   data = {
      'single-host': 'Single host mode parameters',
      'session': 'Session parameters in single host mode',
      'replace': 'Session replace parameters in single host mode',
      'detection': 'Log frequent session replacements on an interface',
      'errdisable': 'Error disable interface on frequent session replacements',
   }

   @staticmethod
   def handler( mode, args ):
      if 'errdisable' in args:
         config.sessionReplaceDetection = SessionReplaceDetection.errdisable
      else:
         config.sessionReplaceDetection = SessionReplaceDetection.logOnly

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.sessionReplaceDetection = SessionReplaceDetection.detectionDisabled

Dot1xMode.addCommandClass( Dot1xSessionReplaceDetectionCmd )

# -----------------------------------------------------------------
# [no|default] dot1x authentication failure action traffic
#                 [drop | allow { [vlan <v>] [access-list <l>] } ]
# -----------------------------------------------------------------
nodeAuthFailAccessList = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'access-list',
      helpdesc='Apply access-list to unauthenticated traffic' ),
   guard=dot1xPerMacAclSupportedGuard, maxMatches=1 )

nodeAuthFailVlan = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'vlan', helpdesc='Set authentication failure VLAN' ),
   maxMatches=1 )

class VlanExpr( CliCommand.CliExpression ):
   expression = ( '( vlan VLAN )' )
   data = {
      'vlan': nodeAuthFailVlan,
      'VLAN': vlanIdMatcher
   }

class AclExpr( CliCommand.CliExpression ):
   expression = ( '( access-list ACL_NAME )' )
   data = {
      'access-list': nodeAuthFailAccessList,
      'ACL_NAME': CliCommand.Node( matcher=aclNameMatcher )
   }

def noAuthFailVlanOrAcl( dot1xIntfConfig, vlan=None, acl=None ):
   if ( not vlan ) or (
         vlan.id and ( vlan.id == dot1xIntfConfig.authFailVlan ) ):
      # Clear afVlan if not provided or if vlanId matches the configured afVlanId
      dot1xIntfConfig.authFailVlan = Tac.Value( 'Bridging::VlanIdOrNone', 0 )
      dot1xIntfConfig.afDropConfigured = False
   if ( acl is None ) or ( acl and ( acl == dot1xIntfConfig.authFailAcl ) ):
      # Clear AF acl if not provided or if acl matches the configured AF acl
      dot1xIntfConfig.authFailAcl = ""
      dot1xIntfConfig.afDropConfigured = False

def setAuthFailParams( mode, args ):
   if 'allow' in args:
      dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
      vlan = args.get( 'VLAN', 0 )
      if vlan:
         vlan = vlan[ 0 ].id
      dot1xIntfConfig.authFailVlan = Tac.Value( 'Bridging::VlanIdOrNone', vlan )
      acl = args.get( 'ACL_NAME', '' )
      if acl:
         acl = acl[ 0 ]
      dot1xIntfConfig.authFailAcl = acl
      if vlan or acl:
         dot1xIntfConfig.afDropConfigured = False
   elif 'drop' in args:
      dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
      noAuthFailVlanOrAcl( dot1xIntfConfig )   # Clear AFVLAN and AF ACL
      dot1xIntfConfig.afDropConfigured = True  # Set flag to indicate drop-cmd config

def noAuthFailParams( mode, args ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
   if 'allow' in args:
      if dot1xIntfConfig:
         vlan = args.get( 'VLAN' )
         acl = args.get( 'ACL_NAME' )
         noAuthFailVlanOrAcl( dot1xIntfConfig, vlan, acl )
   elif 'drop' in args:
      if dot1xIntfConfig:
         dot1xIntfConfig.afDropConfigured = False

class Dot1XAuthFailureActionTrafficCmd( CliCommand.CliCommandClass ):
   syntax = '''dot1x authentication failure action traffic
               ( drop
               | ( allow { VLAN_EXPR | ACL_EXPR } ) )'''
   noOrDefaultSyntax = '''dot1x authentication failure action traffic
                          ( drop
                          | ( allow [ ( vlan | VLAN_EXPR )
                                    | ( access-list | ACL_EXPR ) ] ) ) ...'''
   data = {
      'dot1x': nodeIntfDot1x,
      'authentication': 'Set dot1x authentication failure action',
      'failure': 'Set dot1x authentication failure action',
      'action': 'Set dot1x authentication failure action',
      'traffic': 'Set dot1x authentication failure action',
      'drop': 'Drop all unauthenticated traffic',
      'allow':
         'Allow unauthenticated traffic with authentication failure parameters',
      'VLAN_EXPR': VlanExpr,
      'ACL_EXPR': AclExpr,
   }
   handler = setAuthFailParams
   noOrDefaultHandler = noAuthFailParams
   
EthIntfModelet.addCommandClass( Dot1XAuthFailureActionTrafficCmd )

#-------------------------------------------------------------------------------
# [no|default] eapol vlan change logoff disabled
# -------------------------------------------------------------------------------
def setEapolLogoffDisable( mode, args ):
   config.disableVlanChangeLogoff = True

def noEapolLogoffDisable( mode, args ):
   config.disableVlanChangeLogoff = False

class Dot1xEapolLogoffDisableCmd( CliCommand.CliCommandClass ):
   syntax = 'eapol vlan change logoff disabled'
   noOrDefaultSyntax = syntax
   data = {
      'eapol': 'Configure Dot1x EAPOL attribute',
      'vlan': 'Virtual LAN',
      'change': 'VLAN change',
      'logoff': 'Log the host off',
      'disabled': 'Disable host log off',
   }
   handler = setEapolLogoffDisable
   noOrDefaultHandler = noEapolLogoffDisable

Dot1xMode.addCommandClass( Dot1xEapolLogoffDisableCmd )

# -------------------------------------------------------------------------------
# [no|default] eapol unresponsive action traffic allow vlan <v>]
#-------------------------------------------------------------------------------
class Dot1xEapolUnresponsiveActionTrafficCmd( CliCommand.CliCommandClass ):
   _baseSyntax = 'eapol unresponsive action traffic allow'
   syntax = _baseSyntax + ' vlan VLAN'
   noOrDefaultSyntax = _baseSyntax + '[ vlan VLAN ]'
   
   data = {
      'eapol' : 'Configure Dot1x EAPOL attribute',
      'unresponsive': 'Configure EAPOL unresponsive hosts',
      'action': 'Action in case of EAPOL unresponsive hosts',
      'traffic': 'Traffic action in case of EAPOL unresponsive hosts',
      'allow': 'Allow traffic in case of EAPOL unresponsive hosts',
      'vlan': 'Vlan to allow traffic in case of EAPOL unresponsive hosts',
      'VLAN': vlanIdMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      setGuestVlan( mode, args[ 'VLAN' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlan = args.get( 'VLAN' )
      if vlan is None or vlan.id == config.guestVlan:
         setGuestVlan( mode, 0 )

def setGuestVlan( mode, vlan ):
   config.guestVlan = Tac.Value( 'Bridging::VlanIdOrNone',
                                  vlan and vlan.id )

Dot1xMode.addCommandClass( Dot1xEapolUnresponsiveActionTrafficCmd )

# ---------------------------------------------------------------------------
# Per-interface dot1x eapol unresponsive action
# ---------------------------------------------------------------------------

# ---------------------------------------------------------------------------
# (intf)# [no|default] dot1x eapol unresponsive action traffic allow vlan <v>
# (intf)# [no|default] dot1x eapol unresponsive action traffic disabled
# ---------------------------------------------------------------------------

def setIntfGuestVlan( mode, vlan, disabled ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   vlan = Tac.Value( 'Bridging::VlanIdOrNone',
                                         vlan and vlan.id )
   dot1xIntfConfig.guestVlan = Dot1xIntfGuestVlan( vlan, disabled )

class Dot1xEapolUnresponsiveActionTrafficIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x eapol unresponsive action traffic ( allow vlan VLAN ) | disabled '
   noOrDefaultSyntax = 'dot1x eapol unresponsive action traffic ...'

   data = {
      'dot1x': nodeIntfDot1x,
      'eapol': 'Configure Dot1x EAPOL attribute',
      'unresponsive': 'Configure EAPOL unresponsive hosts',
      'action': 'Action in case of EAPOL unresponsive hosts',
      'traffic': 'Traffic action in case of EAPOL unresponsive hosts',
      'allow': 'Allow traffic in case of EAPOL unresponsive hosts',
      'vlan': 'Vlan to allow traffic in case of EAPOL unresponsive hosts',
      'VLAN': vlanIdMatcher,
      'disabled': 'Disable setting',
   }

   @staticmethod
   def handler( mode, args ):
      if 'vlan' in args:
         vlan = args[ 'VLAN' ]
         setIntfGuestVlan( mode, vlan, False )
      elif 'disabled' in args:
         setIntfGuestVlan( mode, 0, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setIntfGuestVlan( mode, 0, False )

EthIntfModelet.addCommandClass( Dot1xEapolUnresponsiveActionTrafficIntfCmd )

# ---------------------------------------------------------------------------
# (config-if-Et<>)# dot1x aaa server group <GroupName> { group <GroupName> }
# ---------------------------------------------------------------------------

class PortDot1xMethodExpression( CliCommand.CliExpression ):
   expression = "( group <GROUPTYPE> | <GROUPNAME> )"
   data = {
      'group': AaaCliLib.groupKwMatcher,
      '<GROUPTYPE>': AaaCliLib.authnDot1xMethodListGroupMatcher,
      '<GROUPNAME>': AaaCliLib.radiusServerGroupNameMatcher
   }

matcherDot1xAaaServer = CliMatcher.KeywordMatcher(
   'server', helpdesc='RADIUS Server group' )

mlParam = '<methodList>'

# parses the arg list to create a list of server groups
def portGroupListAdapter( mode, args, argsList ):
   # this converts <GROUPTYPE>, <GROUPNAME> to <methodList>
   for replaced in ( '<GROUPTYPE>', '<GROUPNAME>', 'group' ):
      args.pop( replaced, None )
   args[ mlParam ] = []
   for arg in argsList:
      if arg[ 0 ] == '<GROUPTYPE>' or arg[ 0 ] == '<GROUPNAME>':
         methodName = arg[ 1 ]
         args[ mlParam ].append( methodName )

def setDot1xPortServerGroup( mode, args ):
   makeOrGetDot1xIntfConfig( mode.intf.name )
   groups = args[ mlParam ]
   configParam = config.portServerGroupList.newMember( mode.intf.name )
   if configParam:
      groupList = configParam.method
      groupList.clear()
      groupList.update( enumerate( groups ) )

def noSetDot1xPortServerGroup( mode, args ):
   configParam = config.portServerGroupList.newMember( mode.intf.name )
   if configParam:
      groupList = configParam.method
      groupList.clear()

class Dot1xAaaPerPortRadiusServerGroup( CliCommand.CliCommandClass ):
   syntax = "dot1x aaa server { { <methodExpr> } }"
   noOrDefaultSyntax = "dot1x aaa server ..."
   data = {
      'dot1x': nodeDot1x,
      'aaa': matcherDot1xAaa,
      'server': matcherDot1xAaaServer,
      '<methodExpr>': PortDot1xMethodExpression
      }
   adapter = portGroupListAdapter
   handler = setDot1xPortServerGroup
   noOrDefaultHandler = noSetDot1xPortServerGroup

if Dot1xToggle.toggleDot1xPortRadiusServerGroupEnabled():
   EthIntfModelet.addCommandClass( Dot1xAaaPerPortRadiusServerGroup )

#---------------------------------------------------------------------------
# Per-interface dot1x aaa unresponsive *
#---------------------------------------------------------------------------

#---------------------------------------------------------------------------
# (intf)# [no|default] dot1x aaa unresponsive action apply cached-results
#                                     else traffic allow [ vlan <vlanID> ]
# (intf)# [no|default] dot1x aaa unresponsive action apply cached-results
# (intf)# [no|default] dot1x aaa unresponsive action traffic allow
#                                             { [vlan <vlanId> ] [access-list <l>] }
# (intf)# [no|default] dot1x aaa unresponsive action disabled
#---------------------------------------------------------------------------

def setAaaUnresponsiveTrafficAllowIntf( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   populateApplyCachedResultsTimeout( dot1xIntfConfig, args, phone=False )

   if not 'allow' in args:
      dot1xIntfConfig.aaaUnresponsiveTrafficAllow = AaaUnresponsiveTrafficAllow(
         False, True, 0 )
   else:
      vlan = args.get( 'VLAN', 0 )
      if vlan:
         vlan = vlan[ 0 ].id
      dot1xIntfConfig.aaaUnresponsiveTrafficAllow = AaaUnresponsiveTrafficAllow(
         True, True, Tac.Value( 'Bridging::VlanIdOrNone', vlan ) )
   dot1xIntfConfig.aaaUnresponsiveApplyCachedResults = TristateBool.valueSet(
      'cached-results' in args )
   acl = args.get( 'ACL_NAME', "" )
   if acl:
      acl = acl[ 0 ]
   dot1xIntfConfig.aaaUnresponsiveAcl = acl

def noAaaUnresponsiveTrafficAllowIntf( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.aaaUnresponsiveApplyCachedResultsTimeout = \
         CachedResultsTimeout( Tac.endOfTime,
               CachedResultsTimeoutUnit.timeoutUnitNone )
   dot1xIntfConfig.aaaUnresponsiveTrafficAllow = AaaUnresponsiveTrafficAllow(
      False, False, 0 )
   dot1xIntfConfig.aaaUnresponsiveApplyCachedResults = TristateBool.valueInvalid()
   dot1xIntfConfig.aaaUnresponsiveAcl = ""

nodeAaaUnresponsiveVlan = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Allow traffic in VLAN when AAA times out' ),
   maxMatches=1 )

nodeAaaUnresponsiveAccessList = CliCommand.Node(
    CliMatcher.KeywordMatcher( 'access-list',
       helpdesc='Apply access-list when AAA times out' ),
    guard=dot1xPerMacAclSupportedGuard, maxMatches=1 )

class AaaUnresponsiveVlanExpr( CliCommand.CliExpression ):
   expression = ( '( vlan VLAN )' )
   data = {
      'vlan': nodeAaaUnresponsiveVlan,
      'VLAN': vlanIdMatcher
   }

class AaaUnresponsiveAclExpr( CliCommand.CliExpression ):
   expression = ( '( access-list ACL_NAME )' )
   data = {
      'access-list': nodeAaaUnresponsiveAccessList,
      'ACL_NAME': CliCommand.Node( matcher=aclNameMatcher )
   }

class Dot1xAaaUnresponsiveTrafficActionAllowIntfCmd( CliCommand.CliCommandClass ):
   syntax = f'''dot1x aaa unresponsive action
   ( ( apply cached-results{cacheTimeoutArg} [ else traffic allow [ {{ VLAN_EXPR | \
ACL_EXPR }} ] ] )
   | ( traffic allow [ {{ VLAN_EXPR | ACL_EXPR }} ] )
   | disabled )'''
   noOrDefaultSyntax = 'dot1x aaa unresponsive action ...'
   data = {
      'dot1x': nodeDot1x,
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'action': matcherDot1xAaaUnrespAction,
      'apply': matcherDot1xAaaUnrespActionApply,
      'cached-results': matcherDot1xAaaUnrespCachedResults,
      'timeout': matcherDot1xAaaCachedTimeout,
      'TIMEOUT_SEC': matcherDot1xAaaCachedTimeoutSec,
      'TIMEOUT_MIN': matcherDot1xAaaCachedTimeoutMin,
      'TIMEOUT_HRS': matcherDot1xAaaCachedTimeoutHrs,
      'TIMEOUT_DAY': matcherDot1xAaaCachedTimeoutDay,
      'minutes': matcherDot1xTimeoutMinutes,
      'hours': matcherDot1xTimeoutHours,
      'days': matcherDot1xTimeoutDays,
      'seconds': matcherDot1xTimeoutSeconds,
      'else': matcherDot1xElse,
      'traffic': matcherDot1xAaaUnrespActionTraffic,
      'allow': matcherDot1xAaaUnrespActionTrafficAllow,
      'VLAN_EXPR': AaaUnresponsiveVlanExpr,
      'ACL_EXPR': AaaUnresponsiveAclExpr,
      'disabled': 'Disable setting',
   }
   handler = setAaaUnresponsiveTrafficAllowIntf
   noOrDefaultHandler = noAaaUnresponsiveTrafficAllowIntf

EthIntfModelet.addCommandClass( Dot1xAaaUnresponsiveTrafficActionAllowIntfCmd )
SubIntfModelet.addCommandClass( Dot1xAaaUnresponsiveTrafficActionAllowIntfCmd )

#------------------------------------------------------------
# (intf)# [no|default] dot1x aaa unresponsive phone action apply cached-results
#                                                          else traffic allow
# (intf)# [no|default] dot1x aaa unresponsive phone action traffic allow
# (intf)# [no|default] dot1x aaa unresponsive phone action apply cached-results
#------------------------------------------------------------

def setAaaUnresponsivePhoneAllowIntf( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   populateApplyCachedResultsTimeout( dot1xIntfConfig, args, phone=True )
   dot1xIntfConfig.aaaUnresponsivePhoneAllow = TristateBool.valueSet(
      'allow' in args )
   if 'cached-results' in args:
      dot1xIntfConfig.aaaUnresponsivePhoneApplyCachedResults = \
         'aaaUnresponsiveApplyCached'
   else:
      dot1xIntfConfig.aaaUnresponsivePhoneApplyCachedResults = \
         'aaaUnresponsiveDoNotApplyCached'

def noAaaUnresponsivePhoneAllowIntf( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.aaaUnresponsivePhoneApplyCachedResultsTimeout = \
         CachedResultsTimeout( Tac.endOfTime,
               CachedResultsTimeoutUnit.timeoutUnitNone )
   dot1xIntfConfig.aaaUnresponsivePhoneAllow = TristateBool.valueInvalid()
   dot1xIntfConfig.aaaUnresponsivePhoneApplyCachedResults = \
      'aaaUnresponsiveApplyCachedUnset'

class Dot1xAaaUnresponsivePhoneActionAllowIntfCmd( CliCommand.CliCommandClass ):
   syntax = f'''dot1x aaa unresponsive phone action
   ( ( apply cached-results{cacheTimeoutArg} [ else traffic allow ] )
   | ( traffic allow )
   | disabled )'''
   noOrDefaultSyntax = 'dot1x aaa unresponsive phone action ...'
   data = {
      'dot1x': nodeDot1x,
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'phone': 'Configure AAA timeout options for phone',
      'action': matcherDot1xAaaUnrespAction,
      'apply': matcherDot1xAaaUnrespActionApply,
      'cached-results': matcherDot1xAaaUnrespCachedResults,
      'timeout': matcherDot1xAaaCachedTimeout,
      'TIMEOUT_SEC': matcherDot1xAaaCachedTimeoutSec,
      'TIMEOUT_MIN': matcherDot1xAaaCachedTimeoutMin,
      'TIMEOUT_HRS': matcherDot1xAaaCachedTimeoutHrs,
      'TIMEOUT_DAY': matcherDot1xAaaCachedTimeoutDay,
      'minutes': matcherDot1xTimeoutMinutes,
      'hours': matcherDot1xTimeoutHours,
      'days': matcherDot1xTimeoutDays,
      'seconds': matcherDot1xTimeoutSeconds,
      'else': matcherDot1xElse,
      'traffic': matcherDot1xAaaUnrespActionTraffic,
      'allow': 'Allow phone traffic in phone VLAN when AAA times out',
      'disabled': 'Disable setting',
   }
   handler = setAaaUnresponsivePhoneAllowIntf
   noOrDefaultHandler = noAaaUnresponsivePhoneAllowIntf

EthIntfModelet.addCommandClass( Dot1xAaaUnresponsivePhoneActionAllowIntfCmd )
SubIntfModelet.addCommandClass( Dot1xAaaUnresponsivePhoneActionAllowIntfCmd )

#------------------------------------------------------------
# (intf)# [no|default] dot1x aaa unresponsive eap response [ success | disabled ]
#------------------------------------------------------------

def setAaaUnresponsiveEapResponseIntf( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   if 'disabled' in args:
      dot1xIntfConfig.aaaUnresponsiveEapResponse = \
         'aaaUnresponsiveEapResponseDisabled'
   else:
      dot1xIntfConfig.aaaUnresponsiveEapResponse = \
         'aaaUnresponsiveEapResponseSuccess'

def noSetAaaUnresponsiveEapResponseIntf( mode, args ):
   dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name )
   dot1xIntfConfig.aaaUnresponsiveEapResponse = \
      'aaaUnresponsiveEapResponseUnset'

class Dot1xAaaUnresponsiveEapResponseIntf( CliCommand.CliCommandClass ):
   syntax = '''dot1x aaa unresponsive eap response [ success | disabled ]'''
   noOrDefaultSyntax = 'dot1x aaa unresponsive eap response ...'
   data = {
      'dot1x': nodeDot1x,
      'aaa': matcherDot1xAaa,
      'unresponsive': matcherDot1xAaaUnresp,
      'eap': 'EAP options',
      'response': 'EAP response to send',
      'disabled': 'Send no response',
      'success': 'Send EAP success (default)',
   }
   handler = setAaaUnresponsiveEapResponseIntf
   noOrDefaultHandler = noSetAaaUnresponsiveEapResponseIntf

EthIntfModelet.addCommandClass( Dot1xAaaUnresponsiveEapResponseIntf )
SubIntfModelet.addCommandClass( Dot1xAaaUnresponsiveEapResponseIntf )

#--------------------------------------------------------------------------- 
# [no|default] dot1x unauthorized [ access | native ] vlan membership egress
#--------------------------------------------------------------------------- 
class UnauthorizedVlanMembershipCmd( CliCommand.CliCommandClass ):
   syntax = 'dot1x unauthorized ( access | native ) vlan membership egress'
   noOrDefaultSyntax = 'dot1x unauthorized ( access | native ) vlan membership ...'
   data = {
      'dot1x' : nodeIntfDot1x,
      'unauthorized' : 'Configuration applicable when port is unauthorized ',
      'access' : 'Applied to access VLAN of access port',
      'native' : 'Applied to native VLAN of phone trunk port',
      'vlan' : 'VLAN configuration ',
      'membership' : 'VLAN membership configuration ',
      'egress' : 'Grant VLAN membership in egress direction '
   }

   @staticmethod
   def handler( mode, args ):
      dot1xIntfConfig = makeOrGetDot1xIntfConfig( mode.intf.name ) 
      if 'access' in args:
         dot1xIntfConfig.unauthorizedAccessVlanEgress = True 
      elif 'native' in args:
         dot1xIntfConfig.unauthorizedNativeVlanEgress = True 

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name ) 
      if dot1xIntfConfig:
         if 'access' in args:
            dot1xIntfConfig.unauthorizedAccessVlanEgress = False 
         elif 'native' in args:
            dot1xIntfConfig.unauthorizedNativeVlanEgress = False 

EthIntfModelet.addCommandClass( UnauthorizedVlanMembershipCmd )
#-----------------------------------------------------------------
# [no|default] dot1x guest-vlan
#-----------------------------------------------------------------
#def setGuestVlan( mode, vlan ):
#   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
#   dot1xIntfConfig.guestVlan = vlan
#
#def noGuestVlan( mode ):
#   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
#   if dot1xIntfConfig:
#      dot1xIntfConfig.guestVlan = 0
#
#tokenGuestVlan = CliParser.KeywordRule( 'guest-vlan',
#               helpdesc='Configure a Guest VLAN' )
#vlanIdRule = CliParser.RangeRule( 1, 4094, name='vlan', helpdesc='Enter a VLAN id' )
# we don't support guest vlan and restricted now
#EthIntfModelet.addCommand(
#      ( tokenIntfDot1x, tokenGuestVlan, vlanIdRule, setGuestVlan ) )
#EthIntfModelet.addCommand(
#      ( BasicCli.noOrDefault, tokenIntfDot1x, tokenGuestVlan,
#         BasicCli.trailingGarbage, noGuestVlan ) )

#-----------------------------------------------------------------
# [no|default] dot1x auth-fail vlan
#-----------------------------------------------------------------
#def setAuthFailVlan( mode, vlan ):
#   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
#   dot1xIntfConfig.authFailVlan = vlan
#
#def noAuthFailVlan( mode ):
#   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
#   if dot1xIntfConfig:
#      dot1xIntfConfig.authFailVlan = 0
#
#tokenAuthFail = CliParser.KeywordRule( 'auth-fail',
#               helpdesc='Configure authentication failure behavior' )
#tokenVlan = CliParser.KeywordRule( 'vlan',
#               helpdesc='Configure an authentication failure VLAN' )
#EthIntfModelet.addCommand(
#      ( tokenIntfDot1x, tokenAuthFail, tokenVlan, vlanIdRule, setAuthFailVlan ) )
#EthIntfModelet.addCommand(
#      ( BasicCli.noOrDefault, tokenIntfDot1x, tokenAuthFail, tokenVlan,
#         BasicCli.trailingGarbage, noAuthFailVlan ) )

#-----------------------------------------------------------------
# [no|default] dot1x auth-fail max-attempts
#-----------------------------------------------------------------
#def setAuthFailMaxAttempt( mode, attempt ):
#   dot1xIntfConfig = config.newDot1xIntfConfig( mode.intf.name )
#   dot1xIntfConfig.authFailMaxAttempt = attempt
#
#def noAuthFailMaxAttempt( mode ):
#   dot1xIntfConfig = config.dot1xIntfConfig.get( mode.intf.name )
#   if dot1xIntfConfig:
#      dot1xIntfConfig.authFailMaxAttempt = dot1xIntfConfig.defaultAuthFailMaxAttempt
#
#tokenMaxAttempt = CliParser.KeywordRule( 'max-attempts',
#               helpdesc='Maximum number of authentication attempts' )
#maxAttempRule = CliParser.RangeRule( 1, 10, name='attempt',
#                     helpdesc='Enter a value between 1 and 10' )
#EthIntfModelet.addCommand(
#      ( tokenIntfDot1x, tokenAuthFail, tokenMaxAttempt,
#         maxAttempRule, setAuthFailMaxAttempt ) )
#EthIntfModelet.addCommand(
#      ( BasicCli.noOrDefault, tokenIntfDot1x, tokenAuthFail, tokenMaxAttempt,
#         BasicCli.trailingGarbage, noAuthFailMaxAttempt ) )

#-----------------------------------------------------------------
# show dot1x interface [details|statistics]
#-----------------------------------------------------------------
def setDot1xIntfInfo( info, dot1xIntfConfig, dot1xIntfStatus ):
   info.portControl = dot1xIntfConfig.portCtrlSetting
   info.forceAuthPhone = dot1xIntfConfig.forceAuthPhone
   info.sessionReplaceEnabled = dot1xIntfConfig.sessionReplaceEnabled
   info.quietPeriod = dot1xIntfConfig.quietPeriod
   info.txPeriod = dot1xIntfConfig.txPeriod
   if dot1xIntfConfig.idleTimeout != Tac.endOfTime:
      info.idleTimeout = int( dot1xIntfConfig.idleTimeout )
   info.reauthTimeoutIgnore = dot1xIntfConfig.reauthTimeoutIgnore
   info.unauthorizedAccessVlanEgress = dot1xIntfConfig.unauthorizedAccessVlanEgress
   info.unauthorizedNativeVlanEgress = dot1xIntfConfig.unauthorizedNativeVlanEgress
   info.mbaEnabled = dot1xIntfConfig.mbaEnabled
   if dot1xIntfStatus:
      info.hostMode = dot1xIntfStatus.hostMode
      info.authFailVlan = dot1xIntfStatus.authFailVlan
      info.eapolDisabled = dot1xIntfStatus.eapolDisabled
      info.mbaHostMode = dot1xIntfStatus.mbaHostModeConfigured
      info.mbaAuthAlways = dot1xIntfStatus.mbaAuthAlwaysConfigured
   else:
      info.hostMode = dot1xIntfConfig.hostMode
      info.authFailVlan = dot1xIntfConfig.authFailVlan
      info.eapolDisabled = False
      info.mbaHostMode = dot1xIntfConfig.mbaHostMode
      info.mbaAuthAlways = dot1xIntfConfig.mbaAuthAlways
   info.maxReauthReq = dot1xIntfConfig.maxReauthReq
   info.mbaTimeout = dot1xIntfConfig.mbaTimeout
   info.mbaFallback = dot1xIntfConfig.mbaFallback
   if dot1xIntfStatus and dot1xIntfStatus.mbaAuthAlwaysConfigured:
      info.mbaFallback = False
   #info.authFailMaxAttempt = dot1xIntfConfig.authFailMaxAttempt
   #info.guestVlan = dot1xIntfConfig.guestVlan
   return info

def _parseFilterId( multipleFilterId, aclType ):
   """ Returns list of ACLs in case of FilterId like ACL-1.in.ip, ACL-1.out.ipv6 """
   filterIds = multipleFilterId.split( '\x00' )
   aclNames = set()
   for filterId in filterIds:
      filterIdParts = filterId.split( "." )
      # In case of IP type present in filterIdParts, verify that it is same as the
      # aclType provided in args so that the ACLs having the same name in other IP
      # types are not added to the result set.
      if len( filterIdParts ) <= 2 or ( len( filterIdParts ) > 2 and
                                        aclType == filterIdParts[ 2 ] ):
         # Parsing the ACL name
         aclNames.add( filterIdParts[ 0 ] )
   return list( aclNames )

def getDot1xSpec( name, aclType ):
   modePerMacIntfs = []
   modeinterfaceIntfs = []
   for interface, dot1xIntf in status.dot1xIntfStatus.items():
      aclMode = dot1xIntf.aclMode
      supplicants = []
      for _, supplicant in dot1xIntf.supplicant.items():
         macAddr = supplicant.mac
         username = supplicant.identity
         if supplicant.nasFilterRule and name == supplicant.aclName:
            supplicants.append( AclCliModel.Dot1xSupplicantInfo(
               macAddr=macAddr, username=username ) )
         else:
            aclNames = _parseFilterId( supplicant.aclName, aclType )
            if name in aclNames:
               supplicants.append( AclCliModel.Dot1xSupplicantInfo(
                  macAddr=macAddr, username=username ) )
      if supplicants and aclMode == 'modePerMac':
         modePerMacIntfs.append( AclCliModel.Dot1xInterface(
            name=interface, supplicants=supplicants ) )
      elif supplicants and aclMode == 'modeInterface':
         modeinterfaceIntfs.append( AclCliModel.Dot1xInterface(
            name=interface, supplicants=supplicants ) )

   return AclCliModel.Dot1x( perMacAclMode=modePerMacIntfs,
                             interfaceAclMode=modeinterfaceIntfs )

def populateDot1xInfo( aclSummaryModel, aclName, aclType ):
   dot1xModel = getDot1xSpec( aclName, aclType )
   aclSummaryModel.dot1x = dot1xModel
   return aclSummaryModel

AclCli.dot1xStatusHook.addExtension( populateDot1xInfo )

def showIntfRangeInfo( mode, args ):
   intfs = args[ 'INTFS' ]
   model = Dot1xModel.Dot1xInterfaceRangeInformation()
   getCliIntf = intfs.type().getCliIntf
   for intf in intfs:
      if submodel := showIntfInfo( mode, getCliIntf( mode, intf ) ):
         model.interfaces[ intf ] = submodel
   return model

def showIntfInfo( mode, intf ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( intf.name )
   dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
   if dot1xIntfConfig and dot1xIntfConfig.dot1xEnabled:
      info = Dot1xModel.Dot1xInterfaceInformation()
      info = setDot1xIntfInfo( info, dot1xIntfConfig, dot1xIntfStatus )
      return info
   else:
      return None

def showIntfRangeDetails( mode, args ):
   intfs = args[ 'INTFS' ]
   model = Dot1xModel.Dot1xInterfaceRangeDetails()
   getCliIntf = intfs.type().getCliIntf
   for intf in intfs:
      if submodel := showIntfDetails( mode, getCliIntf( mode, intf ) ):
         model.interfaces[ intf ] = submodel
   return model

def getPortShutdownStatus( intfId ) -> str:
   if intfId in status.portShutdownForCoa.intfRecoveryTimeStamp:
      endTs = status.portShutdownForCoa.intfRecoveryTimeStamp[ intfId ]
      if endTs == Tac.endOfTime:
         return 'Disabled indefinitely'
      return 'Yes, recovery ' + Ark.timestampToStr( endTs, True, Tac.utcNow() )
   else:
      return 'no'

def showIntfDetails( mode, intf ):
   dot1xIntfConfig = config.dot1xIntfConfig.get( intf.name )
   dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
   if dot1xIntfConfig and dot1xIntfConfig.dot1xEnabled:
      details = Dot1xModel.Dot1xInterfaceDetails()
      details = setDot1xIntfInfo( details, 
                                  dot1xIntfConfig, dot1xIntfStatus )
      dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
      if dot1xIntfStatus:
         # For every port check if it was disabled by AAA and after how
         # long the port will come back up and populate model accordingly
         if Dot1xToggle.toggleDot1xCoaPortShutdownEnabled():
            details.portShutdownStatus = getPortShutdownStatus( intf.name )
         details.portAuthorizationState = dot1xIntfStatus.portAuthState
         # Iterate over list of supplicants and display mac of
         # all the authenticated supplicants
         authStagesToShow = { 'successfulAuth', 'webAuthStart' }
         for supplicant in dot1xIntfStatus.supplicant.values():
            if supplicant.authStage in authStagesToShow:
               if dot1xIntfConfig.reauthOpts == 'useAuthSrvrSettings':
                  details.supplicantMacs[ supplicant.mac ] = \
                              supplicant.sessionTimeoutSrv
               else:
                  details.supplicantMacs[ supplicant.mac ] = \
                               supplicant.sessionTimeout
      return details
   else:
      return None

def showIntfRangeStats( mode, args ):
   intfs = args[ 'INTFS' ]
   model = Dot1xModel.Dot1xInterfaceRangeStats()
   getCliIntf = intfs.type().getCliIntf
   for intf in intfs:
      if submodel := showIntfStats( mode, getCliIntf( mode, intf ) ):
         model.interfaces[ intf ] = submodel
      if dot1xIntfStatus := status.dot1xIntfStatus.get( intf ):
         if dot1xIntfStatus.dropCounterConfigured:
            if droppedStats := showIntfDroppedCounterStats( mode, intf ):
               model.dropCounters[ intf ] = droppedStats
   return model

def showIntfStats( mode, intf ):
   dot1xIntfStatus = status.dot1xIntfStatus.get( intf.name )
   if dot1xIntfStatus:
      stats = Dot1xModel.Dot1xInterfaceStats()
      stats.rxStart = dot1xIntfStatus.stats.rxStart
      stats.rxLogoff = dot1xIntfStatus.stats.rxLogoff
      stats.rxRespId = dot1xIntfStatus.stats.rxRespId
      stats.rxResp = dot1xIntfStatus.stats.rxResp
      stats.rxInvalid = dot1xIntfStatus.stats.rxInvalid
      stats.rxTotal = stats.rxStart + stats.rxLogoff + stats.rxRespId \
                        + stats.rxResp + stats.rxInvalid
      stats.txReqId = dot1xIntfStatus.stats.txReqId
      stats.txReq = dot1xIntfStatus.stats.txReq
      stats.txTotal = stats.txReqId + stats.txReq
      stats.rxVersion = dot1xIntfStatus.stats.rxVersion
      stats.lastRxSrcMac = dot1xIntfStatus.stats.lastRxSrcMac
      return stats
   else:
      return None

def showIntfDroppedCounterStats( mode, intf ):
   droppedCounterStatus = \
         allIntfDot1xDroppedCounter.dot1xEapolDroppedCounter.get( intf )
   if droppedCounterStatus:
      droppedStats = Dot1xModel.Dot1xDroppedCounterStats()
      droppedStats.unauthEapolPort = \
         droppedCounterStatus.unAuthorizedPortPktsDropped
      droppedStats.unauthEapolHost = \
         droppedCounterStatus.unAuthorizedEapoLHostPktsDropped
      droppedStats.unauthMbaHost = \
         droppedCounterStatus.unAuthorizedMbaHostPktsDropped
      return droppedStats
   else:
      return None

matcherDetails = CliMatcher.KeywordMatcher( 'details', 
                        helpdesc='Show 802.1X port authentication detail' )
matcherStatistics = CliMatcher.KeywordMatcher( 'statistics', 
                        helpdesc='Show 802.1X port authentication statistics' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface', 
                        helpdesc='Choose an interface' )

# show dot1x interface IPINTF
class Dot1XInterfaceIpintfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x interface INTFS'
   data = {
      'dot1x': nodeDot1x,
      'interface': matcherInterface,
      'INTFS': Intf.rangeMatcher,
   }
   
   handler = showIntfRangeInfo
   cliModel = Dot1xModel.Dot1xInterfaceRangeInformation

BasicCli.addShowCommandClass( Dot1XInterfaceIpintfCmd )

# show dot1x interface IPINTF details
class Dot1XInterfaceIpintfDetailsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x interface INTFS details'
   data = {
      'dot1x': nodeDot1x,
      'interface': matcherInterface,
      'INTFS': Intf.rangeMatcher,
      'details': matcherDetails,
   }
   
   handler = showIntfRangeDetails
   cliModel = Dot1xModel.Dot1xInterfaceRangeDetails

BasicCli.addShowCommandClass( Dot1XInterfaceIpintfDetailsCmd )

# show dot1x interface IPINTF statistics
class Dot1XInterfaceIpintfStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x interface INTFS statistics'
   data = {
      'dot1x': nodeDot1x,
      'interface': matcherInterface,
      'INTFS': Intf.rangeMatcher,
      'statistics': matcherStatistics,
   }
   
   handler = showIntfRangeStats
   cliModel = Dot1xModel.Dot1xInterfaceRangeStats

BasicCli.addShowCommandClass( Dot1XInterfaceIpintfStatisticsCmd )

#-----------------------------------------------------------------
# show dot1x all [brief|details|statistics]
# legacy:
# show dot1x all [summary|details|statistics]
#-----------------------------------------------------------------
def showAllInfo( mode, args ):
   allInfo = Dot1xModel.Dot1xAllInformation()
   allInfo.systemAuthControl = config.dot1xEnabled
   allInfo.lldpBypass = identityDot1xStatus.lldpBypass
   allInfo.bpduBypass = status.bpduBypass
   allInfo.lacpBypass = status.lacpBypass
   allInfo.dynAuth = identityDot1xStatus.dot1xDynAuthEnabled
   allInfo.version = Dot1xConsts.version2004
   allInfo.sendIdentityReqNewMac = False
   allInfo.sendIdentityReqLoggedOffMac = False
   if config.sendIdentityReqNewMac == 'macNewSendIdReq':
      allInfo.sendIdentityReqNewMac = True
   if config.sendIdentityReqDisconnectMac == 'macDisconnectSendIdReq':
      allInfo.sendIdentityReqLoggedOffMac = True
   allInfo.sessionMoveEapol = config.sessionMoveEapol
   allInfo.sessionReplaceDetectionEnabled = (
      config.sessionReplaceDetection != SessionReplaceDetection.detectionDisabled )
   allInfo.sessionReplaceDetectionAction = config.sessionReplaceDetection
   for intfName in config.dot1xIntfConfig:
      intf = EthPhyIntf( intfName, mode )
      info = showIntfInfo( mode, intf )
      if info:
         allInfo.interfaces[ intfName ] = info
   return allInfo

def showAllSummary( mode, args ):
   allSummary = Dot1xModel.Dot1xAllSummary()
   allSummary.systemAuthControl = config.dot1xEnabled
   allSummary.lldpBypass = identityDot1xStatus.lldpBypass
   allSummary.bpduBypass = status.bpduBypass
   allSummary.lacpBypass = status.lacpBypass
   allSummary.dynAuth = identityDot1xStatus.dot1xDynAuthEnabled
   allSummary.version = Dot1xConsts.version2004

   # Following attributes are not used in summary, so they
   # need not be updated. However, they need to be initialized
   allSummary.sendIdentityReqNewMac = False
   allSummary.sendIdentityReqLoggedOffMac = False
   allSummary.sessionMoveEapol = False
   allSummary.sessionReplaceDetectionEnabled = False
   allSummary.sessionReplaceDetectionAction = (
      SessionReplaceDetection.detectionDisabled )
   for intfName in config.dot1xIntfConfig:
      intf = EthPhyIntf( intfName, mode )
      detail = showIntfDetails( mode, intf )
      if detail:
         allSummary.interfaces[ intfName ] = detail
   return allSummary

def showAllDetails( mode, args ):
   allDetails = Dot1xModel.Dot1xAllDetails()
   allDetails.systemAuthControl = config.dot1xEnabled
   allDetails.lldpBypass = identityDot1xStatus.lldpBypass
   allDetails.bpduBypass = status.bpduBypass
   allDetails.lacpBypass = status.lacpBypass
   allDetails.dynAuth = identityDot1xStatus.dot1xDynAuthEnabled
   allDetails.version = Dot1xConsts.version2004
   allDetails.sendIdentityReqNewMac = False
   allDetails.sendIdentityReqLoggedOffMac = False
   if config.sendIdentityReqNewMac == 'macNewSendIdReq':
      allDetails.sendIdentityReqNewMac = True
   if config.sendIdentityReqDisconnectMac == 'macDisconnectSendIdReq':
      allDetails.sendIdentityReqLoggedOffMac = True
   allDetails.sessionMoveEapol = config.sessionMoveEapol
   allDetails.sessionReplaceDetectionEnabled = (
      config.sessionReplaceDetection != SessionReplaceDetection.detectionDisabled )
   allDetails.sessionReplaceDetectionAction = config.sessionReplaceDetection
   for intfName in config.dot1xIntfConfig:
      intf = EthPhyIntf( intfName, mode )
      detail = showIntfDetails( mode, intf )
      if detail:
         allDetails.interfaces[ intfName ] = detail
   return allDetails

def showAllStats( mode, args ):
   model = Dot1xModel.Dot1xInterfaceRangeStats()
   for intf in status.dot1xIntfStatus:
      model.interfaces[ intf ] = showIntfStats( mode, EthPhyIntf( intf, mode ) )
      if dot1xIntfStatus := status.dot1xIntfStatus.get( intf ):
         if dot1xIntfStatus.dropCounterConfigured:
            if submodel := showIntfDroppedCounterStats( mode, intf ):
               model.dropCounters[ intf ] = submodel
   return model

matcherAll = CliMatcher.KeywordMatcher( 'all', 
      helpdesc='Show 802.1X port authentication information for all interfaces' )

# show dot1x all
class Dot1XAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
   }
   handler = showAllInfo
   cliModel = Dot1xModel.Dot1xAllInformation

BasicCli.addShowCommandClass( Dot1XAllCmd )

# show dot1x all brief
class Dot1XAllBriefCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all brief'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'brief': 'Show 802.1X port authentication brief',
   }
   handler = showAllSummary
   cliModel = Dot1xModel.Dot1xAllSummary

BasicCli.addShowCommandClass( Dot1XAllBriefCmd )

# show dot1x all summary - Deprecated Command
class Dot1XAllSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all summary'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'summary': CliCommand.Node( CliMatcher.KeywordMatcher( 'summary', 
                  helpdesc='Show 802.1X port authentication summary' ),
                  deprecatedByCmd='show dot1x all brief' ),
   }
   handler = showAllSummary
   cliModel = Dot1xModel.Dot1xAllSummary

BasicCli.addShowCommandClass( Dot1XAllSummaryCmd )

# show dot1x all details
class Dot1XAllDetailsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all details'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'details': matcherDetails,
   }
   handler = showAllDetails
   cliModel = Dot1xModel.Dot1xAllDetails

BasicCli.addShowCommandClass( Dot1XAllDetailsCmd )

# show dot1x all statistics
class Dot1XAllStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x all statistics'
   data = {
      'dot1x': nodeDot1x,
      'all': matcherAll,
      'statistics': matcherStatistics,
   }
   handler = showAllStats
   cliModel = Dot1xModel.Dot1xInterfaceRangeStats

BasicCli.addShowCommandClass( Dot1XAllStatisticsCmd )

#----------------------------------------------------------------------------
# show dot1x statistics hosts [ ( interface INTFS ) | ( vlan VLAN ) ]
#----------------------------------------------------------------------------
def showDot1xHostStatistics( mode, args ):
   model = Dot1xModel.Dot1xAllHostStats()
   model.fromTacc( status, vlans=args.get( 'VLANS' ), intfs=args.get( 'INTFS' ) )
   return model

class Dot1xHostStatisticsShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x statistics hosts [ ( interface INTFS ) | ( vlan VLANS ) ]'
   data = {
         'dot1x': nodeDot1x,
         'statistics': 'Show 802.1X host statistics',
         'hosts': 'Show information for all the supplicants',
         'interface': matcherInterface,
         'INTFS': intfRangeMatcher,
         'vlan': 'Show 802.1X statistics for specific VLANs',
         'VLANS': vlanSetMatcher,
   }
   handler = showDot1xHostStatistics
   cliModel = Dot1xModel.Dot1xAllHostStats

BasicCli.addShowCommandClass( Dot1xHostStatisticsShowCmd )

# -----------------------------------------------------------------
# show dot1x radius
# -----------------------------------------------------------------
def showRadiusDeadTimer( mode, args ):
   deadTimers = Dot1xModel.Dot1xDeadTimers()
   deadTimers.radiusConfigDeadTime = radiusConfig.deadtime
   for hostspec, time in radiusDeadTime.deadTime.items():
      deadTimers.hosts[ hostspec.hostname ] = float( time )
   return deadTimers

class Dot1XDeadTimerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x radius'
   data = {
      'dot1x': nodeDot1x,
      'radius': 'Show status of RADIUS servers',
   }
   handler = showRadiusDeadTimer
   cliModel = Dot1xModel.Dot1xDeadTimers

BasicCli.addShowCommandClass( Dot1XDeadTimerCmd )

#-----------------------------------------------------------------
# show dot1x hosts [ <intfName> ]
#-----------------------------------------------------------------
def showHost( supplicantStatus ):
   host = Dot1xModel.Dot1xHost()
   host.fromTacc( supplicantStatus )
   vlanConfig = bridgingConfig.vlanConfig
   if not supplicantStatus.vlan:
      if hostEntry := mergedHostTable.hostEntry.get( supplicantStatus.mac ):
         host.vlanId = str( hostEntry.vlanId )
         if hostEntry.vlanId in vlanConfig and vlanConfig[
               hostEntry.vlanId ].internal:
            host.internalVlan = True
         else:
            host.staticVlan = True
   if host.vlanId:
      if vlanConfig := cliConfig.vlanConfig.get( int( host.vlanId ) ):
         host.vlanName = vlanConfig.configuredName
   return host

def showHosts( intfName ):
   hosts = Dot1xModel.Dot1xHosts()
   if dot1xIntfStatus := status.dot1xIntfStatus.get( intfName ):
      for supplicantMac, supplicant in dot1xIntfStatus.supplicant.items():
         if supplicant.authStage != 'pendingDeletion':
            hosts.supplicants[ supplicantMac ] = showHost( supplicant )
   return hosts

def showAllHosts( mode, args ):
   intfList = args.get( 'INTFS' )
   allHosts = Dot1xModel.Dot1xAllHosts()
   if intfList:
      for ifName in intfList:
         intf = intfList.type().getCliIntf( mode, ifName )
         if not intf.lookup() or not intf.status().deviceName:
            continue
         if ifName in status.dot1xIntfStatus:
            allHosts.intfSupplicantsDict[ ifName ] = showHosts( ifName )
   else:
      for intfName in status.dot1xIntfStatus:
         allHosts.intfSupplicantsDict[ intfName ] = showHosts( intfName )

   return allHosts

class Dot1XHostsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x hosts [ INTFS ]'
   data = {
      'dot1x': nodeDot1x,
      'hosts': 'Show information for all the supplicants',
      'INTFS': intfRangeMatcher,
   }
   handler = showAllHosts
   cliModel = Dot1xModel.Dot1xAllHosts

BasicCli.addShowCommandClass( Dot1XHostsCmd )

#-----------------------------------------------------------------
# show dot1x hosts blocked
#-----------------------------------------------------------------
def showBlockedMacEntry():
   entries = {}
   for macAddr, macEntry in blockedMacTable.blockedMacEntry.items():
      entry = Dot1xModel.Dot1xBlockedMacEntry()
      entry.interfaces.update( macEntry.intfVlan )
      entries[ macAddr ] = entry
   return entries

def showBlockedMacTable( mode, args ):
   macTable = Dot1xModel.Dot1xBlockedMacTable()
   macTable.macAddresses.update( showBlockedMacEntry() )
   return macTable

class Dot1XBlockedMacTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x hosts blocked'
   data = {
       'dot1x': nodeDot1x,
       'hosts': 'Show information for all the supplicants',
       'blocked': 'Show MAC addresses blocked via dynamic authorization',
   }
   handler = showBlockedMacTable
   cliModel = Dot1xModel.Dot1xBlockedMacTable

BasicCli.addShowCommandClass( Dot1XBlockedMacTableCmd )

#-----------------------------------------------------------------
# show dot1x hosts mac <mac-addr> [ detail ]
#-----------------------------------------------------------------
def getSupplicantAclInfo( supplicantModel, supplicantStatus, detail ):
   def aclIsDynamic( aclName ):
      aclConfig = dot1xAclInputConfig.config.get( 'ip' )
      return aclConfig and aclName in aclConfig.acl

   # The default value of a CliModel.List is [].
   # Since supplicantModel.nasFilterRules is optional and
   # we do not want show it when detail is False, we need to explicitly set to None
   if not detail:
      supplicantModel.nasFilterRules = None
      return

   filterId = ''
   daclRules = []
   supplicantAclName = supplicantStatus.aclName
   # If the session is soft-cached, we wont be able to use aclIsDynamic, rather
   # check if nasFilterRule is present, if yes use it, otherwise use aclName.
   if supplicantStatus.sessionCached:
      if supplicantStatus.nasFilterRule:
         daclRules = supplicantStatus.nasFilterRule.strip( '\0' ).split( '\0' )
      elif supplicantAclName != config.captivePortalIpv4Acl:
         filterId = supplicantAclName
   elif supplicantAclName:
      if aclIsDynamic( supplicantAclName ):
         # dynamic ACL rules from AAA server
         # supplicantStatus.nasFilterRule is a string of dynamic ACL rules
         # delimited by '\0'
         daclRules = supplicantStatus.nasFilterRule.strip( '\0' ).split( '\0' )
      elif supplicantAclName != config.captivePortalIpv4Acl:
         # static ACL name from AAA server
         filterId = supplicantAclName
   # Multiple filter-Id's will be delimited by '\0'
   supplicantModel.filterId = filterId.replace( '\0', ',' ).strip( ',' )
   supplicantModel.nasFilterRules = daclRules

def dot1XSupplicantAttrShowCmd( mode, args ):
   hostMac = args.get( 'MAC' )
   supplicantAttrModel = Dot1xModel.Dot1XSupplicantAttr()
   detail = 'detail' in args
   for intfName, dot1xIntfStatus in status.dot1xIntfStatus.items():
      for supplicantMac, supplicant in dot1xIntfStatus.supplicant.items():
         if supplicant.authStage != 'pendingDeletion':
            if convertMacAddrToDisplay( supplicantMac ) == hostMac or \
                  supplicantMac.upper() == hostMac.upper():
               supplicantAttrModel.interface = intfName
               supplicantAttrModel.fromTacc( supplicant, config, detail, status )
               getSupplicantAclInfo( supplicantAttrModel, supplicant, detail )
               supplicantAttrModel.detail_ = detail
   return supplicantAttrModel

class Dot1XSupplicantAttrShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x hosts mac MAC [ detail ]'
   data = {
         'dot1x': nodeDot1x,
         'hosts': 'Show information for all the supplicants',
         'mac': 'Show 802.1X supplicant specific information based on MAC',
         'MAC': MacAddr.macAddrMatcher,
         'detail': 'Show 802.1X supplicant detailed attributes',
   }
   handler = dot1XSupplicantAttrShowCmd
   cliModel = Dot1xModel.Dot1XSupplicantAttr

BasicCli.addShowCommandClass( Dot1XSupplicantAttrShowCmd )

#-----------------------------------------------------------------
# show dot1x hosts mac <mac-addr> accounting
#-----------------------------------------------------------------
def dot1XSupplicantAcctAttrShowCmd( mode, args ):
   hostMac = args.get( 'MAC' )
   supplicantAcctModel = Dot1xModel.Dot1XSupplicantAcctAttr()
   for intfName, dot1xIntfStatus in status.dot1xIntfStatus.items():
      for supplicantMac, supplicant in dot1xIntfStatus.supplicant.items():
         if supplicant.authStage != 'pendingDeletion':
            if ( convertMacAddrToDisplay( supplicantMac ) == hostMac or
                 supplicantMac.upper() == hostMac.upper() ):
               supplicantAcctModel.interface = intfName
               supplicantAcctModel.fromTacc( supplicant, config,
                                             dot1xIntfStatus.intfId, radiusConfig,
                                             ethIntfStatusDir, netStatus, status )
   return supplicantAcctModel

class Dot1XSupplicantAcctAttrShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x hosts mac MAC accounting'
   data = {
         'dot1x': nodeDot1x,
         'hosts': 'Show information for all the supplicants',
         'mac': 'Show 802.1X supplicant specific information based on MAC',
         'MAC': MacAddr.macAddrMatcher,
         'accounting': 'Show 802.1X supplicant accounting attributes',
   }
   handler = dot1XSupplicantAcctAttrShowCmd
   cliModel = Dot1xModel.Dot1XSupplicantAcctAttr

BasicCli.addShowCommandClass( Dot1XSupplicantAcctAttrShowCmd )

#-----------------------------------------------------------------
# show dot1x hosts interface INTFS detail
# -----------------------------------------------------------------
def isPhone( supplicantStatus ):
   if supplicantStatus.deviceType != "noType":
      return supplicantStatus.deviceType == "phone"

   if nbr := nbrClassificationStatus.neighbor.get( supplicantStatus.mac ):
      return nbr.classification.telephone
   
   return False

def getHostDetails( hosts, intfName ):
   if dot1xIntfStatus := status.dot1xIntfStatus.get( intfName ):
      for mac, supplicant in dot1xIntfStatus.supplicant.items():
         if supplicant.authStage != 'pendingDeletion':
            suppModel = Dot1xModel.Dot1XSupplicantExtendedAttr()
            suppModel.fromTacc( supplicant, config, cliConfig, bridgingConfig,
                                mergedHostTable, intfName, status )
            getSupplicantAclInfo( suppModel.supplicantAttr, supplicant, True )
            if isPhone( supplicant ):
               suppModel.deviceDomain = "phone"
            hosts.supplicants[ mac ] = suppModel

def dot1xIntfHostsDetailShowCmd( mode, args ):
   model = Dot1xModel.Dot1xIntfHostsDetail()
   intfs = args.get( "INTFS" )
   for intfName in intfs:
      hosts = Dot1xModel.Dot1xIntfHosts()
      getHostDetails( hosts, intfName )
      model.interfaces[ intfName ] = hosts
   return model

class Dot1xIntfHostsDetailShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x hosts interface INTFS detail'
   data = {
         'dot1x': nodeDot1x,
         'hosts': 'Show information for all the supplicants',
         'interface': matcherInterface,
         'INTFS': intfRangeMatcher,
         'detail': 'Show 802.1X supplicant detailed attributes',
   }
   handler = dot1xIntfHostsDetailShowCmd
   cliModel = Dot1xModel.Dot1xIntfHostsDetail

BasicCli.addShowCommandClass( Dot1xIntfHostsDetailShowCmd )

# -----------------------------------------------------------------
# show dot1x vlan assignment group [ <groupName> ]
#-----------------------------------------------------------------
def getGroupNames( mode ):
   return config.vlanGroup

matcherGroupName = CliMatcher.DynamicNameMatcher( getGroupNames,
                                                  'Assigned group name' )

def showDot1xVlanAssignmentGroupNames( mode, args ):
   groupName = args.get( 'GROUP' )
   vlanAssignedGroup = Dot1xModel.Dot1xVlanAssignmentGroupNames()
   groups = []
   if groupName:
      if groupName in config.vlanGroup:
         groups = [ groupName ]
   else:
      groups = config.vlanGroup

   for group in groups:
      vlans = config.vlanGroup[ group ].vlanId
      canonicalString = multiRangeToCanonicalString( vlans )
      vlanAssignedGroup.vlanGroupNameDict[ group ] = canonicalString
   return vlanAssignedGroup

class Dot1XVlanAssignmentGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x vlan assignment group [ GROUP ]'
   data = {
      'dot1x': nodeDot1x,
      'vlan': 'VLAN group',
      'assignment' : 'Assigned VLAN to group',
      'group' : 'Group name',
      'GROUP' : matcherGroupName,
   }
   handler = showDot1xVlanAssignmentGroupNames
   cliModel = Dot1xModel.Dot1xVlanAssignmentGroupNames

BasicCli.addShowCommandClass( Dot1XVlanAssignmentGroupCmd )

#-----------------------------------------------------------------
# clear dot1x interface
#-----------------------------------------------------------------
def clearIntfStats( mode, args ):
   intf = args[ 'IPINTF' ]
   dot1xIntfConfigReq = configReq.dot1xIntfConfigReq.get( intf.name )
   if dot1xIntfConfigReq:
      dot1xIntfConfigReq.clearStatsRequestTime = Tac.now()

matcherStatistics = CliMatcher.KeywordMatcher( 'statistics', 
               helpdesc='Clear 802.1X port authentication statistics' )
matcherHost = CliMatcher.KeywordMatcher( 'host',
               helpdesc='Clear host commands' )
matcherBlocked = CliMatcher.KeywordMatcher( 'blocked',
               helpdesc='Clear blocked hosts' )
matcherHostInterface = CliMatcher.KeywordMatcher( 'interface',
               helpdesc='Clear hosts on interface' )
matcherHostMac = CliMatcher.KeywordMatcher( 'mac',
               helpdesc='Clear hosts based on mac' )
matcherSoftCached = CliMatcher.KeywordMatcher( 'software-cached',
               helpdesc='Clear software-cached hosts' )
matcherClearAll = CliMatcher.KeywordMatcher( 'all',
               helpdesc='Clear hosts on all interfaces' )

tokenClearNode = CliToken.Clear.clearKwNode

class ClearDot1XStatisticsInterfaceIpintfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x statistics interface IPINTF'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'statistics': matcherStatistics,
      'interface': 'Choose an interface',
      'IPINTF': Intf.matcherWithIpSupport,
   }
   handler = clearIntfStats

BasicCliModes.EnableMode.addCommandClass( ClearDot1XStatisticsInterfaceIpintfCmd )

#-----------------------------------------------------------------
# clear dot1x all
#-----------------------------------------------------------------
def clearAllStats( mode, args ):
   for dot1xIntfConfigReq in configReq.dot1xIntfConfigReq.values():
      dot1xIntfConfigReq.clearStatsRequestTime = Tac.now()

class ClearDot1XStatisticsAllCmd( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x statistics all'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'statistics': matcherStatistics,
      'all': 'Clear 802.1X port authentication information for all interfaces',
   }
   handler = clearAllStats

BasicCliModes.EnableMode.addCommandClass( ClearDot1XStatisticsAllCmd )

def dot1xUnauthorizedExplanation( intfName, mode ):
   dot1xIntfStatus = status.dot1xIntfStatus.get( intfName )
   if dot1xIntfStatus and dot1xIntfStatus.portAuthState == 'blocked':
      return ( 'N/A', 'Unauthorized interface', 'dot1x configuration' )
   return ( None, None, None )

IntfCli.unauthorizedExplanationHook.addExtension( dot1xUnauthorizedExplanation )

def dot1xCheckStaticMac( mode, macAddr, vlanId, intfsOrDrop=None ):
   hostEntry = dot1xInputHostTable.hostEntry.get( macAddr )
   if hostEntry:
      hostEntry = mergedHostTable.hostEntry.get( macAddr )
      if hostEntry and vlanId.id == hostEntry.vlanId:
         mode.addError( 'MAC address already in use by dot1x' )
         return False
   return True

bridgingCheckStaticMacHook.addExtension( dot1xCheckStaticMac )

#-------------------------------------------------------------------------------
# Cleanup per-interface configuration
#-------------------------------------------------------------------------------
class IntfDot1xConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface Dot1x config
   when the interface is deleted."""
   def setDefault( self ):
      del config.dot1xIntfConfig[ self.intf_.name ]

#-------------------------------------------------------------------------------
# clear dot1x host mac <mac-address>
#-------------------------------------------------------------------------------
def clearHost( mode, args ):
   hostMac = args.get( 'MAC' )
   configReq.hostToBeCleared = hostMac
   configReq.hostToBeClearedTime = Tac.now()

class ClearDot1XHostMac( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host mac MAC'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'mac': matcherHostMac,
      'MAC' : MacAddr.macAddrMatcher,
   }
   handler = clearHost

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostMac )

#-------------------------------------------------------------------------------
# clear dot1x host [software-cached] all
#-------------------------------------------------------------------------------
def clearAllHost( mode, args ):
   configReq.clearSoftCachedOnly = 'software-cached' in args
   configReq.allHostClearTime = Tac.now()

class ClearDot1XAllHost( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host [ software-cached ] all'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'software-cached': matcherSoftCached,
      'all': matcherClearAll,
   }
   handler = clearAllHost

BasicCliModes.EnableMode.addCommandClass( ClearDot1XAllHost )

#-------------------------------------------------------------------------------
# clear dot1x host [software-cached] interface <interface-name>
#-------------------------------------------------------------------------------
def clearInterfaceHost( mode, args ):
   configReq.clearSoftCachedOnly = 'software-cached' in args
   intf = args.get( 'ETHINTF' )
   intfId = IntfId( intf.name )
   configReq.intfToBeCleared = intfId
   configReq.intfToBeClearedTime = Tac.now()

class ClearDot1XInterfaceHost( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host [ software-cached ] interface ETHINTF'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'software-cached': matcherSoftCached,
      'interface': matcherHostInterface,
      'ETHINTF': EthPhyIntf.ethMatcher,
   }
   handler = clearInterfaceHost

BasicCliModes.EnableMode.addCommandClass( ClearDot1XInterfaceHost )

#-------------------------------------------------------------------------------
# clear dot1x host blocked mac <mac-address>
#-------------------------------------------------------------------------------
def clearHostBlocked( mode, args ):
   hostMac = args.get( 'MAC' )
   configReq.hostToBeCleared = hostMac
   configReq.hostBlockedToBeClearedTime = Tac.now()

class ClearDot1XHostBlockedMac( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host blocked mac MAC'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'blocked': matcherBlocked,
      'mac': matcherHostMac,
      'MAC': MacAddr.macAddrMatcher,
   }
   handler = clearHostBlocked

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostBlockedMac )

#-------------------------------------------------------------------------------
# clear dot1x host blocked interface <interface-name>
#-------------------------------------------------------------------------------
def clearInterfaceHostBlocked( mode, args ):
   intf = args.get( 'ETHINTF' )
   intfId = IntfId( intf.name )
   configReq.intfToBeCleared = intfId
   configReq.intfHostBlockedToBeClearedTime = Tac.now()

class ClearDot1XHostBlockedInterface( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host blocked interface ETHINTF'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'blocked': matcherBlocked,
      'interface': matcherHostInterface,
      'ETHINTF': EthPhyIntf.ethMatcher,
   }
   handler = clearInterfaceHostBlocked

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostBlockedInterface )

#-------------------------------------------------------------------------------
# clear dot1x host blocked all
#-------------------------------------------------------------------------------
def clearAllHostBlocked( mode, args ):
   configReq.allHostBlockedClearTime = Tac.now()

class ClearDot1XHostBlockedAll( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x host blocked all'
   data = {
      'clear': tokenClearNode,
      'dot1x': nodeDot1x,
      'host': matcherHost,
      'blocked': matcherBlocked,
      'all': matcherClearAll,
   }
   handler = clearAllHostBlocked

BasicCliModes.EnableMode.addCommandClass( ClearDot1XHostBlockedAll )

# --------------------------------------------------------------------------------
# show dot1x captive-portal bypass
# --------------------------------------------------------------------------------

matcherCaptivePortal = CliMatcher.KeywordMatcher( 'captive-portal',
               helpdesc='Captive Portal for Web Authentication' )

matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Filter by IPv4 or IPv6 address' )

def showDot1xCaptivePortalBypass( mode, args ):
   dot1xCaptivePortalBypass = Dot1xModel.Dot1xCaptivePortalBypass()
   data = defaultdict( set )
   addrArg = args.get( 'ADDR' )
   if addrArg is not None:
      fqdn = webAgentStatus.allowedIpMatch.get( addrArg )
      if fqdn is not None:
         data[ fqdn ].add( addrArg )
   else:
      for ip, fqdn in webAgentStatus.allowedIpMatch.items():
         data[ fqdn ].add( ip )
   for fqdn, ips in data.items():
      dot1xCaptivePortalBypass.fqdns[ fqdn ] = \
         Dot1xModel.Dot1xCaptivePortalBypassIpMatch()
      dot1xCaptivePortalBypass.fqdns[ fqdn ].addresses = \
         [ str( ip ) for ip in sorted( ips ) ]
   return dot1xCaptivePortalBypass

class Dot1XCaptivePortalBypassCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x captive-portal bypass [ address ADDR ]'
   data = {
      'dot1x': nodeDot1x,
      'captive-portal': matcherCaptivePortal,
      'bypass': 'Show IP addresses bypassing captive-portal redirection',
      'address': matcherAddress,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
   }
   handler = showDot1xCaptivePortalBypass
   cliModel = Dot1xModel.Dot1xCaptivePortalBypass

if Dot1xToggle.toggleDot1xWebFqdnAllowlistEnabled():
   BasicCli.addShowCommandClass( Dot1XCaptivePortalBypassCmd )

# --------------------------------------------------------------------------------
# show dot1x captive-portal resolutions
# --------------------------------------------------------------------------------

def showDot1xCaptivePortalResolutions( mode, args ):
   dot1xCaptivePortalResolutions = Dot1xModel.Dot1xCaptivePortalResolutions()
   if not webAgentStatus.reverseDnsEntries:
      return dot1xCaptivePortalResolutions
   addrArg = args.get( 'ADDR' )
   if addrArg:
      reverseDnsEntry = webAgentStatus.reverseDnsEntries.entry.get( addrArg )
      if reverseDnsEntry is not None:
         modelEntry = Dot1xModel.Dot1xCaptivePortalResolutionsEntry()
         modelEntry.fromTacc( reverseDnsEntry )
         dot1xCaptivePortalResolutions.addresses[ addrArg ] = modelEntry
   else:
      dot1xCaptivePortalResolutions.fromTacc( webAgentStatus.reverseDnsEntries )
   return dot1xCaptivePortalResolutions

class Dot1XCaptivePortalResolutionsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x captive-portal resolutions [ address ADDR ]'
   data = {
      'dot1x': nodeDot1x,
      'captive-portal': matcherCaptivePortal,
      'resolutions': 'Show reverse DNS resolutions',
      'address': matcherAddress,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
   }
   handler = showDot1xCaptivePortalResolutions
   cliModel = Dot1xModel.Dot1xCaptivePortalResolutions

if Dot1xToggle.toggleDot1xWebFqdnAllowlistEnabled():
   BasicCli.addShowCommandClass( Dot1XCaptivePortalResolutionsCmd )

# --------------------------------------------------------------------------------
# The "show dot1x captive-portal counters [ interface <INTFS> ]" command.
# --------------------------------------------------------------------------------

matcherDot1xCounters = CliMatcher.KeywordMatcher( 'counters',
               helpdesc='Request 802.1X counters' )

def getDot1xCaptivePortalCountersInterface( intfCaptivePortalCounters ):
   interfaceCounter = Dot1xModel.CaptivePortalCountersInterface()
   interfaceCounter.fromTacc( intfCaptivePortalCounters )
   return interfaceCounter

def showDot1xCaptivePortalCounters( mode, args ):
   dot1xCaptivePortalCounters = Dot1xModel.Dot1xCaptivePortalCounters()
   intfList = args.get( 'INTFS', webAgentStatus.captivePortalCounters )
   for intf in intfList:
      intfCaptivePortalCounters = webAgentStatus.captivePortalCounters.get( intf )
      if intfCaptivePortalCounters:
         dot1xCaptivePortalCounters.interfaces[ intf ] = \
            getDot1xCaptivePortalCountersInterface( intfCaptivePortalCounters )
   return dot1xCaptivePortalCounters

class Dot1XCaptivePortalCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1x captive-portal counters [ interface INTFS ]'
   data = {
      'dot1x': nodeDot1x,
      'counters': matcherDot1xCounters,
      'captive-portal': matcherCaptivePortal,
      'interface': matcherInterface,
      'INTFS': intfRangeMatcher,
   }
   handler = showDot1xCaptivePortalCounters
   cliModel = Dot1xModel.Dot1xCaptivePortalCounters

BasicCli.addShowCommandClass( Dot1XCaptivePortalCountersCmd )

#----------------------------------------------------------------------------------
# The "clear dot1x captive-portal counters [ interface INTF ]" command.
#----------------------------------------------------------------------------------

def clearCaptivePortalCounters( mode, args ):
   intf = args.get( 'ETHINTF' )
   if intf:
      intfId = IntfId( intf.name )
   else:
      intfId = IntfId()
   clearCounter = Tac.Value( 'Dot1x::ClearCaptivePortalCounter',
                             clearTime=Tac.now(), intfId=intfId )
   configReq.clearCaptivePortalCounter = clearCounter

class ClearCaptivePortalCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x captive-portal counters [ interface ETHINTF ]'
   data = {
     'clear': tokenClearNode,
     'dot1x': nodeDot1x,
     'counters': matcherDot1xCounters,
     'captive-portal': matcherCaptivePortal,
     'interface': matcherInterface,
     'ETHINTF': EthPhyIntf.ethMatcher,
   }
   handler = clearCaptivePortalCounters

BasicCliModes.EnableMode.addCommandClass( ClearCaptivePortalCountersCmd )

# ----------------------------------------------------------------------------------
# The "clear dot1x captive-portal resolutions" command.
# ----------------------------------------------------------------------------------

def clearCaptivePortalResolutions( mode, args ):
   clearResolutions = Tac.Value( 'Dot1x::ClearCaptivePortalResolutions',
                                 clearTime=Tac.now() )
   configReq.clearCaptivePortalResolutions = clearResolutions

class ClearCaptivePortalResolutionsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear dot1x captive-portal resolutions'
   data = {
     'clear': tokenClearNode,
     'dot1x': nodeDot1x,
     'captive-portal': matcherCaptivePortal,
     'resolutions': 'Clear reverse DNS resolutions used by captive-portal bypass',
   }
   handler = clearCaptivePortalResolutions

if Dot1xToggle.toggleDot1xWebFqdnAllowlistEnabled():
   BasicCliModes.EnableMode.addCommandClass( ClearCaptivePortalResolutionsCmd )

#-------------------------------------------------------------------------------
# Register to show tech-support
#-------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2019-07-17 13:48:02',
   cmds=[ 'show dot1x hosts',
          'show dot1x all details',
          'show dot1x all statistics',
          'show dot1x hosts blocked',
          'show dot1x supplicant' ],
   cmdsGuard=lambda: status and status.dot1xEnabled )

#-------------------------------------------------------------------------------
# Register to show tech-support for Dot1xWeb
#-------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2024-07-29 13:48:02',
   cmds=[ 'show dot1x captive-portal counters',
          'show dot1x captive-portal resolutions',
          'show dot1x captive-portal bypass' ],
   cmdsGuard=lambda: webAgentStatus and webAgentStatus.running )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config, configReq, status, identityDot1xStatus, dot1xInputHostTable
   global hwstatus, bridgingConfig, cliConfig, subIntfCapabilityDir, mergedHostTable
   global ipStatus, blockedMacTable, allIntfDot1xDroppedCounter
   global dot1xAclInputConfig, radiusDeadTime, webAgentStatus
   global radiusConfig, ethIntfStatusDir, netStatus, nbrClassificationStatus
   global mgmtGnmiClientConfig

   config = ConfigMount.mount( entityManager, "dot1x/config",
                             "Dot1x::Config", "w" )
   radiusConfig = LazyMount.mount( entityManager, "security/aaa/radius/config",
         "Radius::Config", "r" )
   configReq = LazyMount.mount( entityManager, "dot1x/configReq",
                             "Dot1x::ConfigReq", "w" )
   status = LazyMount.mount( entityManager, "dot1x/status",
                             "Dot1x::Status", "r" )
   webAgentStatus = LazyMount.mount( entityManager, "dot1x/webAgentStatus",
                                     "Dot1x::WebAgentStatus", "r" )
   identityDot1xStatus = LazyMount.mount( entityManager, "identity/dot1x/status",
                                          "Identity::Dot1x::Status", "r" )
   dot1xInputHostTable = LazyMount.mount( entityManager,
                                          "bridging/input/hostTable/Dot1x",
                                          "Dot1x::HostTable", "r" )
   mergedHostTable = LazyMount.mount( entityManager, "dot1x/hostTable",
                                      "Dot1x::HostTable", "r" )
   hwstatus = LazyMount.mount( entityManager, "dot1x/hwstatus",
                             "Dot1x::HwStatus", "r" )
   cliConfig = LazyMount.mount( entityManager, "bridging/input/config/cli",
                                "Bridging::Input::CliConfig", "r" )
   bridgingConfig = LazyMount.mount(
         entityManager, "bridging/config", "Bridging::Config", "r" )
   subIntfCapabilityDir = LazyMount.mount( entityManager, 
                                     'hardware/status/dot1x/slice',
                                     'Tac::Dir', 'ri' )
   ipStatus = LazyMount.mount( entityManager, "ip/status",
                               "Ip::Status", "r" )
   blockedMacTable = LazyMount.mount( entityManager, "dot1x/blockedMacTable",
                                      "Dot1x::BlockedMacTable", "r" )
   allIntfDot1xDroppedCounter = SmashLazyMount.mount( entityManager,
                                 DroppedCtrConsts.smashPath,
                                 "Interface::AllIntfDot1xDroppedCounter",
                                 SmashLazyMount.mountInfo( 'reader' ) )
   dot1xAclInputConfig = LazyMount.mount( entityManager,
                                          "acl/config/input/dot1x",
                                          "Acl::Input::Config", "r" )
   radiusDeadTime = LazyMount.mount( entityManager, "dot1x/radiusDeadTime",
                                     "Dot1x::RadiusDeadTime", "r" )
   ethIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
   netStatus = LazyMount.mount( entityManager, Cell.path( "sys/net/status" ),
                          "System::NetStatus", "r" )
   nbrClassificationStatus = LazyMount.mount( entityManager,
         "identity/nbrClassification/status", "Identity::NbrClassification::Status",
         "r" )
   mgmtGnmiClientConfig = LazyMount.mount( entityManager, "mgmtGnmiClient/config",
                                           "MgmtGnmiClient::Config", "r" )

# register interface-deletion handler
   IntfCli.Intf.registerDependentClass( IntfDot1xConfigCleaner, priority=10 )

