#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
from CliPlugin.DpllCommon import updatePllResetConfig, clockNames
import CliPlugin.FruCli as FruCli
import CliPlugin.TechSupportCli as TechSupportCli
import CliToken.Reset
import CliToken.Hardware
import LazyMount
import ShowCommand

gv = CliGlobal.CliGlobal(
   allPllConfig=None,
   resetConfig=None,
)

# --------------------------------------------------------------------------------
# show hardware pll [ module MODULE ]
# --------------------------------------------------------------------------------
class HardwarePllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show hardware pll [ module MODULE ]"
   data = {
      "hardware": CliToken.Hardware.hardwareMatcherForShow,
      "pll": "Show PLL status",
      "module": CliCommand.guardedKeyword( "module",
         helpdesc="Show PLL status for a specific module",
         guard=FruCli.modularSystemGuard ),
      "MODULE": FruCli.SlotExpressionFactory(),
   }
   cliModel = "DpllCliModel.PlatformModel"
   handler = "DpllCliHandler.showHardwarePll"

BasicCli.addShowCommandClass( HardwarePllCmd )

# --------------------------------------------------------------------------------
# reset hardware pll CLOCK_NAME
# --------------------------------------------------------------------------------
def resetHardwarePllGuard( mode, token ):
   pllName = token
   updatePllResetConfig( gv.allPllConfig, gv.resetConfig )
   for pllModel in gv.allPllConfig:
      pllType = gv.allPllConfig.entity[ pllModel ]
      if pllName in pllType.pllConfig:
         pllConfig = pllType.pllConfig[ pllName ]
         if not pllConfig.hwResetGpoName:
            return CliParser.guardNotThisPlatform
         else:
            return None
   return None

class ResetHardwarePllClocknameCmd( CliCommand.CliCommandClass ):
   syntax = "reset hardware pll CLOCK_NAME"
   data = {
      "reset": CliToken.Reset.resetMatcher,
      "hardware": CliToken.Hardware.hardwareMatcherForReset,
      "pll": "Reset a PLL",
      "CLOCK_NAME": CliCommand.Node(
         matcher=CliMatcher.DynamicNameMatcher(
            lambda mode: clockNames( gv.allPllConfig ),
            helpdesc="PLL Name", pattern="[A-Za-z0-9]+/[0-9]+" ),
         guard=resetHardwarePllGuard )
   }
   handler = "DpllCliHandler.resetHardwarePll"

BasicCli.EnableMode.addCommandClass( ResetHardwarePllClocknameCmd )

TechSupportCli.registerShowTechSupportCmd(
   "2018-03-26 12:18:56",
   cmds=[ "show hardware pll" ],
   # guard the show hardware pll to only run when we have at least 1 pll
   # in the system
   cmdsGuard=lambda: len( gv.allPllConfig ) == 0,
)

def Plugin( entityManager ):
   gv.allPllConfig = LazyMount.mount( entityManager,
      "hardware/pllClockMux/config",
      "Tac::Dir",
      "ri" )
   gv.resetConfig = LazyMount.mount( entityManager,
      "hardware/pllClockMux/resetConfig",
      "Tac::Dir",
      "w" )
