#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliToken.Clear
from CliPlugin import DpsCliModel
from CliPlugin import IpAddrMatcher
from CliPlugin import TechSupportCli
from CliPlugin.WanTEShowCli import dpsSupported, pathSelectionSupportedGuard
import LazyMount
import re
import AgentCommandRequest
import json
from io import StringIO
import ShowCommand
import SmashLazyMount
import SharkLazyMount
import Tac
from TypeFuture import TacLazyType

constants = TacLazyType( 'Dps::DpsConstants' )
PathType = Tac.Type( 'Dps::PathType' )
UnsecuredPath = PathType.unsecuredPath
SecuredPath = PathType.securedPath
PathState = Tac.Type( 'Dps::PathState' )
UnresolvedPath = PathState.unresolved
ResolvedPath = PathState.resolved
PathRouteState = Tac.Type( 'Dps::PathRouteState' )
RemoteViaType = Tac.Type( 'Dps::RemoteViaType' )
StaticVia = RemoteViaType.StaticVia
ItsSessKey = Tac.Type( 'Its::ItsSessKey' )
VrfAppKey = Tac.Type( 'Dps::VrfAppKey' )

dpsCountersSnapShotReq = None
dpsInput = None
dpsPathStatusDir = None
dpsPeerStatus = None
dpsVrfStatus = None
itsSessStatus = None
itsSessStateStatus = None
dpsBgpInput = None
dpsBgpOutput = None
entityMgr = None

def getPathIndex( pathName ):
   p = r'^[pP]ath(\d+)'
   m = re.search( p, pathName )
   if m is None:
      return None
   return int( m.group( 1 ) )

DpsLbProfileCounterEntry = Tac.Type( 'Dps::DpsLbProfileCounterEntry' )
DpsPathCounterEntry = Tac.Type( 'Dps::DpsPathCounterEntry' )

pathSelectionMatcher = CliMatcher.KeywordMatcher( 'path-selection',
                          helpdesc='Show dynamic path selection information' )
pathSelectionMatcherForClear = CliMatcher.KeywordMatcher( 'path-selection',
                               helpdesc='Clear dynamic path selection information' )
pathSelectionNode = CliCommand.Node( matcher=pathSelectionMatcher,
                                     guard=pathSelectionSupportedGuard )
pathSelectionNodeForClear = CliCommand.Node( matcher=pathSelectionMatcherForClear,
                                     guard=pathSelectionSupportedGuard )
clearMatcher = CliToken.Clear.clearKwNode

peerIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Peer address' )
vrfMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                        helpdesc='Match the VRF',
                                        helpname='WORD' )
appProfileMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                               helpname='WORD',
                                               helpdesc='Application profile' )
dstIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Destination address' )
srcIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Source address' )
peerIpv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='Peer address' )
trafficClassMatcher = CliMatcher.IntegerMatcher( 0, 7,
                                             helpdesc='Internal class of traffic' )
pathGroupMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                              helpname='WORD',
                                              helpdesc='Name of the path group' )
pathNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                             helpname='WORD',
                                             helpdesc='Name of the path' )
peerNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                             helpname='WORD',
                                             helpdesc='Name of the peer' )

# NOTE: This should match with ItsPmtu.h
pathMtuStateToStrMap = {
   0: 'init',
   1: 'probing',
   2: 'done',
   3: 'error',
}

# NOTE: This should match with ItsPmtu.h
errStateReasonToStrMap = {
   0: 'none',
   1: 'response message timeout',
   2: 'path down event',
   3: 'mtu outside defined range',
}

def mountDpsCounters():
   dpsCounters = SharkLazyMount.mount( entityMgr, 'dps/shark/counters',
                                       'DpsShark::DpsCounters',
                                       SharkLazyMount.mountInfo( 'shadow' ),
                                       True )
   return dpsCounters

def getDpsGroups():
   groups = {}
   for pathStatus in dpsPathStatusDir.pathStatus.values():
      for pathEntry in pathStatus.pathStatusEntry.values():
         # mapping from path index to group name
         groups[ pathEntry.pathIndex ] = pathEntry.pgName
   return groups

def getDpsCounters( mode, args ):
   dpsCounters = mountDpsCounters()
   lbProfileDict = {}
   groups = getDpsGroups()

   vrf = args.get( 'VRF' )
   peerIp = args.get( 'PEER' )
   appProfile = args.get( 'APPPROF' )

   for lbKey, profile in dpsCounters.lbProfileCounter.items():
      lbId = lbKey.lbId
      lbProfile = DpsCliModel.DpsLbProfiles()

      profileKeyInfo = profile.lbKeyInfo
      vrfAppKey = VrfAppKey( profileKeyInfo.vrfId, profileKeyInfo.appId )
      vrfApp = dpsVrfStatus.vrfApp.get( vrfAppKey )
      if vrfApp is not None:
         lbProfile.vrf = vrfApp.vrfName
         lbProfile.appProfile = vrfApp.appName
      else:
         lbProfile.vrf = "vrf" + str( profileKeyInfo.vrfId )
         lbProfile.appProfile = "app" + str( profileKeyInfo.appId )
      lbProfile.peerIp = profileKeyInfo.vtepIp
      lbProfile.peerName = ''
      peerStatus = dpsPeerStatus.peerStatusEntry.get( profileKeyInfo.vtepIp )
      if peerStatus is not None:
         lbProfile.peerName = peerStatus.peerName

      if ( ( vrf and vrf != lbProfile.vrf ) or
           ( peerIp and peerIp != str( lbProfile.peerIp ) ) or
           ( appProfile and appProfile != lbProfile.appProfile ) ):
         continue

      outBytesSnapShot = 0
      outPktsSnapShot = 0
      lbCounterSnapShot = dpsCounters.lbProfileCounterSnapShot.get( lbKey )
      if lbCounterSnapShot and lbCounterSnapShot.timestamp > profile.timestamp:
         outBytesSnapShot = lbCounterSnapShot.outBytes
         outPktsSnapShot = lbCounterSnapShot.outPkts
      lbProfile.outBytes = profile.outBytes - outBytesSnapShot
      lbProfile.outPkts = profile.outPkts - outPktsSnapShot
      lbProfile.flows = profile.flows
      lbProfile.throughput = profile.throughput
      lbProfile.pathGroups = {}

      for pathId, path in profile.pathCounter.items():
         if path.pathId == 0:
            continue
         groupName = groups.get( path.pathId, '' )
         if groupName not in lbProfile.pathGroups:
            pG = DpsCliModel.DpsLbProfiles.PathGroups()
            pG.paths = {}
            lbProfile.pathGroups[ groupName ] = pG
         p = DpsCliModel.DpsLbProfiles.PathGroups.Paths()
         outBytesSnapShot = 0
         outPktsSnapShot = 0
         if lbCounterSnapShot:
            pCounterSnapShot = lbCounterSnapShot.pathCounter.get( pathId )
         else:
            pCounterSnapShot = None
         if pCounterSnapShot and pCounterSnapShot.timestamp > path.timestamp:
            outBytesSnapShot = pCounterSnapShot.outBytes
            outPktsSnapShot = pCounterSnapShot.outPkts
         p.outBytes = path.outBytes - outBytesSnapShot
         p.outPkts = path.outPkts - outPktsSnapShot
         p.flows = path.flows
         p.throughput = path.throughput
         lbProfile.pathGroups[ groupName ].paths[ pathId ] = p

      lbProfileDict[ lbId ] = lbProfile
   return lbProfileDict

#-----------------------------------------------------------------------------------
# show path-selection load-balance counters [ detail ]
#    [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]
#-----------------------------------------------------------------------------------
class ShowPathSelectionLbCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection load-balance counters [ detail ]
               [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]'''
   data = {
             'path-selection': pathSelectionNode,
             'load-balance' : 'Show load-balance information',
             'counters' : 'Show path-selection counters',
             'detail': 'Show path-selection counters in detail',
             'vrf': 'Show path-selection counters filtered by vrf name',
             'VRF': vrfMatcher,
             'peer': 'Show path-selection counters filtered by peer IP',
             'PEER': peerIpv4AddrMatcher,
             'application-profile':
                   'Show path-selection counters filtered by application-profile',
             'APPPROF': appProfileMatcher,
          }
   cliModel = DpsCliModel.DpsLbCounters

   @staticmethod
   def handler( mode, args ):
      lbCounters = DpsCliModel.DpsLbCounters()
      lbCounters.lbProfiles = getDpsCounters( mode, args )
      lbCounters._detail = 'detail' in args # pylint: disable=protected-access
      return lbCounters

BasicCli.addShowCommandClass( ShowPathSelectionLbCounters )

#-----------------------------------------------------------------------------------
# show path-selection application counters
#    [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]
#-----------------------------------------------------------------------------------
class ShowPathSelectionAppCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection application counters
               [ vrf VRF ] [ peer PEER ] [ application-profile APPPROF ]'''
   data = {
             'path-selection': pathSelectionNode,
             'application' : 'Show application information',
             'counters' : 'Show path-selection counters',
             'vrf': 'Show path-selection counters filtered by vrf name',
             'VRF': vrfMatcher,
             'peer': 'Show path-selection counters filtered by peer IP',
             'PEER': peerIpv4AddrMatcher,
             'application-profile':
                   'Show path-selection counters filtered by application-profile',
             'APPPROF': appProfileMatcher,
          }
   cliModel = DpsCliModel.DpsAppCounters

   @staticmethod
   def handler( mode, args ):
      appCounters = DpsCliModel.DpsAppCounters()
      appCounters.lbProfiles = getDpsCounters( mode, args )
      return appCounters

BasicCli.addShowCommandClass( ShowPathSelectionAppCounters )

def setDefaultPmtuValues( path ):
   path.mtu = 0
   path.mtuConfigType = 'auto'
   path.mtuState = 'notApplicable'
   path.mtuErrStateReason = 'none'
   path.mtuDiscDueSecs = 0
   path.mtuDiscTimeTakenMsecs = 0.0
   path.mtuDiscIntervalSecs = 300
   path.mtuNumProbesSent = 0
   path.mtuNumReportReqSent = 0
   path.mtuNumReportRespRcvd = 0
   path.mtuNumRespProcErr = 0
   path.mtuNumProbesRcvd = 0
   path.mtuNumReportReqRcvd = 0
   path.mtuNumReportRespSent = 0
   path.mtuNumReqProcErr = 0

def getDpsPaths( mode, args, detail ):
   # 'show path-selection paths' cmd
   displayMode = 'mtu only'
   if 'mtu' in args: # 'show path-selection mtu' cmd
      displayMode = 'mtu detail' if detail else 'mtu summary'

   model = DpsCliModel.DpsPaths( _detail=detail,
                                 _mtuInfoDisplayMode=displayMode )

   itsSessionState = itsSessStateStatus.itsSessionState
   itsCharacteristics = itsSessStatus.itsSessCharacData
   itsPmtuStatus = itsSessStatus.itsSessPmtuStatus
   itsPmtuStats = itsSessStatus.itsSessPmtuStats

   filtPeer = args.get( 'PEERIP' )
   filtPeerName = args.get( 'PEERNAME' )
   filtGroup = args.get( 'GROUPNAME' )
   filtPath = args.get( 'PATHNAME' )
   filtSrc = args.get( 'SRCIP' )
   filtDst = args.get( 'DSTIP' )
   filtTc = args.get( 'TC' )

   for pathStatus in dpsPathStatusDir.pathStatus.values():
      for pathId, pathEntry in pathStatus.pathStatusEntry.items():
         pathKey = pathEntry.pathKey
         peerIp = pathEntry.peerIp
         label = pathEntry.label
         peerName = ''
         peerStatusEntry = dpsPeerStatus.peerStatusEntry.get( peerIp )
         if peerStatusEntry is not None:
            peerName = peerStatusEntry.peerName
         groupName = pathEntry.pgName
         pathViaPair = pathEntry.pathViaPair
         pathName = "path" + str( pathId )
         srcIp = pathKey.srcAddr.stringValue
         dstIp = pathKey.dstAddr.stringValue
         vrfId = pathKey.vrfId

         # Check filters
         if ( ( filtPeer and filtPeer != peerIp.stringValue ) or
              ( filtPeerName and filtPeerName != peerName ) or
              ( filtGroup and filtGroup != groupName ) or
              ( filtPath and getPathIndex( filtPath ) != pathId ) or
              ( filtSrc and filtSrc != srcIp ) or
              ( filtDst and filtDst != dstIp ) ):
            continue

         if pathEntry.viaType == StaticVia:
            pathType = "static"
         else:
            # TODO: Should distinguish paths discovered from BGP and Stun
            pathType = "dynamic"

         peer = model.dpsPeers.get( peerIp )
         if peer is None:
            peer = DpsCliModel.DpsPeer()
            peer.peerName = peerName
            if detail and ( peerIp in dpsBgpInput.remoteNodeInfo ):
               # Fill in the peer's AVT region/zone/site names
               peerNodeInfo = dpsBgpInput.remoteNodeInfo[ peerIp ]
               peer.avtRegionId = peerNodeInfo.regionId
               peer.avtZoneId = peerNodeInfo.zoneId
               peer.avtSiteId = peerNodeInfo.siteId

            model.dpsPeers[ peerIp ] = peer

         group = peer.dpsGroups.get( groupName )
         if group is None:
            group = DpsCliModel.DpsGroup()
            peer.dpsGroups[ groupName ] = group

         path = group.dpsPaths.get( pathName )
         if path is None:
            path = DpsCliModel.DpsPath()
            path.source = srcIp
            path.destination = dstIp
            if detail:
               path.sourcePort = pathEntry.localEndPoint.endpointUdpPort
               path.sourceWanId = pathEntry.localEndPoint.wanId
               path.destinationPort = pathEntry.remoteEndPoint.endpointUdpPort
               path.destinationWanId = pathEntry.remoteEndPoint.wanId
               path.localIntf = pathEntry.localIntf
               path.sourceWanIdPathGroup = pathViaPair.localPg
               path.destinationWanIdPathGroup = pathViaPair.remotePg
            path.state = pathEntry.pathRouteState
            path.pathType = pathType
            group.dpsPaths[ pathName ] = path
            path.label = label

         # Read session status from ITS smash table
         for tc in range( 0, 8 ):
            if filtTc and filtTc != tc:
               continue

            itsSessKey = ItsSessKey( vrfId, pathId, tc )
            session = None
            itsSess = itsCharacteristics.get( itsSessKey )
            if itsSess is not None:
               itsState = itsSessionState.get( itsSessKey )
               session = DpsCliModel.DpsSession()
               if itsState and itsState.active:
                  session.active = True
                  session.seconds = int( round( itsSess.activeTime / 1e9 ) )
               else:
                  session.active = False
                  session.seconds = 0
            elif tc == 0:
               # Always show tc 0
               session = DpsCliModel.DpsSession()
               session.active = False
               session.seconds = 0
            if session:
               path.dpsSessions[ tc ] = session

         # Set PMTU related details
         setDefaultPmtuValues( path )
         itsSessKey = ItsSessKey( vrfId, pathId, 0 )
         pmtuStatus = itsPmtuStatus.get( itsSessKey )
         pmtuStats = itsPmtuStats.get( itsSessKey )

         # read the config from pathStatusEntry
         if pathEntry.mtuConfig == constants.pmtuDisabled:
            path.mtuConfigType = 'disabled'
         elif pathEntry.mtuConfig > 0:
            path.mtuConfigType = 'static'

         if pmtuStatus is not None:
            path.mtu = pmtuStatus.mtu
            path.mtuState = 'notApplicable'
            if path.mtuConfigType == 'auto':
               path.mtuState = pathMtuStateToStrMap[ pmtuStatus.state ]
               path.mtuErrStateReason = \
                       errStateReasonToStrMap[ pmtuStatus.err_state_reason ]
            path.mtuDiscTimeTakenMsecs = pmtuStatus.discovery_time_taken_us \
                                            / 1000.0
            path.mtuDiscDueSecs = pmtuStatus.discovery_due_in_secs
            path.mtuDiscIntervalSecs = pathEntry.mtuDiscIntervalConfig
         if pmtuStats is not None:
            path.mtuNumProbesSent = pmtuStats.num_probes_sent
            path.mtuNumReportReqSent = pmtuStats.num_report_req_sent
            path.mtuNumReportRespRcvd = pmtuStats.num_report_resp_rcvd
            path.mtuNumRespProcErr = pmtuStats.num_resp_processing_err
            path.mtuNumProbesRcvd = pmtuStats.num_probes_rcvd
            path.mtuNumReportReqRcvd = pmtuStats.num_report_req_rcvd
            path.mtuNumReportRespSent = pmtuStats.num_report_resp_sent
            path.mtuNumReqProcErr = pmtuStats.num_req_processing_err

   return model

def showUnprogrammed( mode, params ):
   buff = StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager,
         'Sfe', 'unprogrammed-paths', params,
          stringBuff=buff )
   model = DpsCliModel.DpsPaths()
   try:
      output = json.loads( buff.getvalue() )
      model.setAttrsFromDict( output )
   except ValueError:
      mode.addError( buff.getvalue() )
   return model

#-----------------------------------------------------------------------------------
# show path-selection paths [ detail ]
# [ peer PEER ] [ peer-name PEERNAME ]
# [ path-group GROUPNAME ] [ path-name PATHNAME ]
# [ source SRCIP ] [ destination DSTIP ] [ traffic-class TC ]
#-----------------------------------------------------------------------------------
class ShowPathSelectionPaths( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection paths [ (unprogrammed
               [ source SRCIP ] [ destination DSTIP ]
               [ path-group GROUPNAME ] [ peer PEERIP ]) |
               ([ detail ] [ peer PEERIP ] [ peer-name PEERNAME ]
               [ path-group GROUPNAME ] [ path-name PATHNAME ]
               [ source SRCIP ] [ destination DSTIP ] [ traffic-class TC ])]'''
   data = {
             'path-selection': pathSelectionNode,
             'paths': 'Show path status information',
             'unprogrammed': 'Show unprogrammed path information',
             'detail': 'Show path status detailed information',
             'peer': 'Show path status filtered by peer IP',
             'PEERIP': peerIpv4AddrMatcher,
             'peer-name': 'Show path status filtered by peer name',
             'PEERNAME': peerNameMatcher,
             'path-group': 'Show path status filtered by path group name',
             'GROUPNAME': pathGroupMatcher,
             'path-name': 'Show path status filtered by path name',
             'PATHNAME': pathNameMatcher,
             'source': 'Show path status filtered by source IP',
             'SRCIP': srcIpv4AddrMatcher,
             'destination': 'Show path status filtered by destination IP',
             'DSTIP': dstIpv4AddrMatcher,
             'traffic-class': 'Show path status filtered by traffic class',
             'TC': trafficClassMatcher,
          }
   cliModel = DpsCliModel.DpsPaths

   @staticmethod
   def handler( mode, args ):
      if 'unprogrammed' in args:
         params = 'path'
         srcAddress = args.get( 'SRCIP' )
         dstAddress = args.get( 'DSTIP' )
         pathGroup = args.get( 'GROUPNAME' )
         peerIp = args.get( 'PEERIP' )
         if srcAddress:
            params += '\tsrc=%s' % srcAddress
         if dstAddress:
            params += '\tdst=%s' % dstAddress
         if pathGroup:
            params += '\tpathGroup=%s' % pathGroup
         if peerIp:
            params += '\tpeer=%s' % peerIp
         return showUnprogrammed( mode, params )

      return getDpsPaths( mode, args, detail='detail' in args )

BasicCli.addShowCommandClass( ShowPathSelectionPaths )

# -----------------------------------------------------------------------------------
# show path-selection mtu [ detail ]
# [ peer PEER ] [ peer-name PEERNAME ]
# [ path-group GROUPNAME ] [ path-name PATHNAME ]
# [ source SRCIP ] [ destination DSTIP ]
# -----------------------------------------------------------------------------------
class ShowPathSelectionMtu( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection mtu [ detail ]
               [ peer PEERIP ] [ peer-name PEERNAME ]
               [ path-group GROUPNAME ] [ path-name PATHNAME ]
               [ source SRCIP ] [ destination DSTIP ]'''
   data = {
             'path-selection': pathSelectionNode,
             'mtu': 'Show MTU information',
             'detail': 'Show MTU detailed information',
             'peer': 'Show MTU information filtered by peer IP',
             'PEERIP': peerIpv4AddrMatcher,
             'peer-name': 'Show MTU information filtered by peer name',
             'PEERNAME': peerNameMatcher,
             'path-group': 'Show MTU information filtered by path group name',
             'GROUPNAME': pathGroupMatcher,
             'path-name': 'Show MTU information filtered by path name',
             'PATHNAME': pathNameMatcher,
             'source': 'Show MTU information filtered by source IP',
             'SRCIP': srcIpv4AddrMatcher,
             'destination': 'Show MTU information filtered by destination IP',
             'DSTIP': dstIpv4AddrMatcher,
          }
   cliModel = DpsCliModel.DpsPaths

   @staticmethod
   def handler( mode, args ):
      return getDpsPaths( mode, args, detail='detail' in args )

BasicCli.addShowCommandClass( ShowPathSelectionMtu )

def getDpsPathsExport( mode, args ):
   def populateWanIdExportInfo( export, info ):
      endpoint = info.endpoint
      export.ipAddress = endpoint.endpointIp
      export.port = endpoint.endpointUdpPort
      # dropping last two charcters of ipSource, to convert
      # eg, InterfaceIp to interface
      export.ipSource = info.ipSource.lower()[ : -2 ]
      export.intf = info.localIntf

   model = DpsCliModel.DpsPathsExport()
   for pathGroup, dpsBgpPathGroup in dpsBgpOutput.localPathGroup.items():
      for peer, dpsDynamicViaConfig in dpsBgpPathGroup.router.items():
         model.vtep = peer
         for wanId, info in dpsDynamicViaConfig.via.items():
            if wanId not in model.wans:
               model.wans[ wanId ] = DpsCliModel.DpsExport()
            if not info.imported:
               # If the wanId is configured under this pathGroup, it should be
               # populated both in the dict for wans and the dict for pathGroups
               # to maintain backwards compatability
               model.wans[ wanId ].pathGroup = pathGroup
               populateWanIdExportInfo( model.wans[ wanId ], info )

            pgExport = DpsCliModel.DpsPathGroupExport()
            populateWanIdExportInfo( pgExport, info )
            model.wans[ wanId ].pathGroups[ pathGroup ] = pgExport
   return model

# -----------------------------------------------------------------------------------
# show path-selection paths export
# -----------------------------------------------------------------------------------
class ShowPathSelectionPathsExport( ShowCommand.ShowCliCommandClass ):
   syntax = '''show path-selection paths export'''
   data = {
      'path-selection': pathSelectionNode,
      'paths': 'Show path status information',
      'export': 'Show IPs exported to BGP'
   }
   cliModel = DpsCliModel.DpsPathsExport

   handler = getDpsPathsExport

BasicCli.addShowCommandClass( ShowPathSelectionPathsExport )

#-----------------------------------------------------------------------------------
# clear path-selection counters
#-----------------------------------------------------------------------------------
class ClearPathSelectionCounters( CliCommand.CliCommandClass ):
   syntax = 'clear path-selection counters'
   data = {
             'clear': clearMatcher,
             'path-selection': pathSelectionNodeForClear,
             'counters': 'Clear path-selection counters',
          }

   @staticmethod
   def handler( mode, args ):
      # trigger snapshot copy
      dpsCountersSnapShotReq.lbProfileCounterRequestTime = Tac.now()

BasicCli.EnableMode.addCommandClass( ClearPathSelectionCounters )

SHOW_TECH_CMDS = [
   'show path-selection load-balance counters',
   'show path-selection load-balance counters detail',
   'show path-selection application counters',
   'show path-selection paths',
   'bash bessctl show dps bestpath',
   'bash bessctl show dps rules',
   'bash bessctl show dps-flows',
   'bash bessctl show dpi-flow-info',
   'bash bessctl show dps icmp statistics'
]

SHOW_TECH_CMDS.append( 'show path-selection mtu detail' )
SHOW_TECH_CMDS.append( 'bash bessctl show dps vtepmtutable' )

timeStamp = '2019-07-09 13:58:09'
TechSupportCli.registerShowTechSupportCmd(
   timeStamp,
   cmds=SHOW_TECH_CMDS,
   cmdsGuard=dpsSupported )
TechSupportCli.registerShowTechSupportCmd(
   timeStamp,
   cmds=SHOW_TECH_CMDS,
   cmdsGuard=dpsSupported,
   extended='sfe' )

def Plugin( em ):
   global dpsInput
   global dpsPathStatusDir
   global dpsPeerStatus
   global dpsVrfStatus
   global itsSessStatus
   global itsSessStateStatus
   global dpsBgpInput
   global dpsBgpOutput
   global entityMgr
   global dpsCountersSnapShotReq

   entityMgr = em

   dpsCountersSnapShotReq = LazyMount.mount( em, 'dps/cli/counters/snapshot/request',
                                             'Dps::DpsCounterSnapShotRequest', 'w' )
   dpsInput = LazyMount.mount( em, 'dps/input/cli',
                                      'Dps::DpsCliConfig', 'r' )
   dpsPathStatusDir = LazyMount.mount( em, 'dps/path/status',
                                      'Dps::PathStatusDir', 'r' )
   dpsPeerStatus = LazyMount.mount( em, 'dps/peer/status',
                                    'Dps::PeerStatus', 'r' )
   dpsVrfStatus = LazyMount.mount( em, 'dps/vrf/status',
                                   'Dps::VrfStatus', 'r' )
   dpsBgpInput = LazyMount.mount( em, 'dps/input/bgp',
                                  'Dps::DpsBgpInput', 'r' )
   dpsBgpOutput = LazyMount.mount( em, 'dps/output/bgp',
                                   'Dps::DpsBgpOutput', 'r' )
   itsSessStatus = SmashLazyMount.mount( em, 'its/sessStatus',
                                     'ItsSmash::ItsSessStatus',
                                     SmashLazyMount.mountInfo( 'reader' ) )
   itsSessStateStatus = SmashLazyMount.mount( em, 'its/sessStateStatus',
                                              'ItsSmash::ItsSessStateStatus',
                                              SmashLazyMount.mountInfo( 'reader' ) )
