#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing, Tac
import ConfigMount
import BasicCli
import LazyMount
import CliPlugin.TechSupportCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.DynamicPrefixListHelper import (
      dynPfxListNameMatcher,
      sessionBgpNameMatcher,
)
from CliPlugin.RouteMapCli import (
   CommonTokens,
   RouteMapMode,
)
from CliPlugin.VrfCli import getAllPlusReservedVrfNames
import CliCommand
import CliMatcher
from CliMode.DynamicPrefixList import (
      DynamicPrefixListMode,
      DynPfxSessionBgpMode,
)
from RouteMapLib import isMatchAttrEnabled
import ShowCommand

traceHandle = Tracing.Handle( 'DynamicPrefixList' )
t5 = traceHandle.trace5 # Info

dynPfxListConfigDir = None # Dynamic Prefix List
routeMapConfig = None

#----------------------------------------------------------------------------------
# Config Handler Functions.
#----------------------------------------------------------------------------------

# Delete Dynamic Prefix List.
def deleteDynPfxList( dynPfx ):
   del dynPfxListConfigDir.dynamicPrefixList[ dynPfx ]

# Get Dynamic Prefix List.
def getDynPfxList( dynPfx ):
   return dynPfxListConfigDir.dynamicPrefixList.get( dynPfx )

def getOrCreateDynPfxList ( dynPfx ):
   dynPfxList = getDynPfxList( dynPfx )
   if not dynPfxList:
      dynPfxList = dynPfxListConfigDir.dynamicPrefixList.newMember( dynPfx )

   return dynPfxList

# Delete Session Bgp Dynamic Prefix List
def deleteSessionBgp( sessionBgp ):
   del dynPfxListConfigDir.sessionBgpDplList[ sessionBgp ]

# Get Session Bgp Dynamic Prefix List
def getSessionBgp( sessionBgp ):
   return dynPfxListConfigDir.sessionBgpDplList.get( sessionBgp )

# Get or Create Session Bgp Dynamic Prefix List
def getOrCreateSessionBgp( sessionBgp ):
   return dynPfxListConfigDir.sessionBgpDplList.newMember( sessionBgp )

# ----------------------------------------------------------------------------------
# dynamic-prefix bgp session mode
# ----------------------------------------------------------------------------------
class DynPfxBgpSessionConfigMode( DynPfxSessionBgpMode, BasicCli.ConfigModeBase ):
   name = 'Dynamic Prefix Session BGP Configuration'
   # -------------------------------------------------------------------------------
   # Construct a new Session Bgp Dynamic-Prefix Config Mode
   # -------------------------------------------------------------------------------

   def __init__( self, parent, session, sessionBgpDplList ):
      self.sessionBgpDplList = sessionBgpDplList
      DynPfxSessionBgpMode.__init__( self, self.sessionBgpDplList )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.sessionBgpDplEntry = Tac.newInstance(
            "Routing::DynamicPrefixList::SessionBgpDplEntry", sessionBgpDplList )
      existingEntry = getSessionBgp( sessionBgpDplList )
      self.copyEntry( self.sessionBgpDplEntry, existingEntry )
      self.abort_ = False

   def copyEntry( self, dstEntry, srcEntry ):
      if not dstEntry or not srcEntry:
         return
      dstEntry.sessionBgpAddr.clear()
      for addr in srcEntry.sessionBgpAddr:
         dstEntry.sessionBgpAddr[ addr ] = True
      dstEntry.originate = srcEntry.originate

   def commitContext( self ):
      if self.abort_:
         return
      entry = getOrCreateSessionBgp( self.sessionBgpDplList )
      self.copyEntry( entry, self.sessionBgpDplEntry )

      # Increment version.
      entry.version += 1
      t5( 'Commiting changes' )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

#----------------------------------------------------------------------------------
# dynamic-prefix mode
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigMode( DynamicPrefixListMode, BasicCli.ConfigModeBase ):
   name = 'Dynamic Prefix Configuration'
   #-------------------------------------------------------------------------------
   # Construct a new Dynamic-Prefix Config Mode
   #-------------------------------------------------------------------------------
   def __init__( self, parent, session, dynPfx ):
      self.dynPfx = dynPfx
      DynamicPrefixListMode.__init__( self, self.dynPfx )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.dynPfxEntry = Tac.newInstance(
         "Routing::DynamicPrefixList::DynamicPrefixListEntry", dynPfx )
      existingEntry = getDynPfxList( dynPfx )
      self.copyEntry( self.dynPfxEntry, existingEntry )
      self.abort_ = False

   def copyEntry( self, dstEntry, srcEntry ):
      if not dstEntry or not srcEntry:
         return
      dstEntry.matchRcf = srcEntry.matchRcf
      dstEntry.matchMap = srcEntry.matchMap
      dstEntry.ipv4PrefixList = srcEntry.ipv4PrefixList
      dstEntry.ipv6PrefixList = srcEntry.ipv6PrefixList

   def commitContext( self ):
      if self.abort_:
         return
      entry = getOrCreateDynPfxList( self.dynPfx )
      self.copyEntry( entry, self.dynPfxEntry )

      # Increment version.
      entry.version += 1
      t5( 'Commiting changes' )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

#----------------------------------------------------------------------------------
# (config)# [ no ] dynamic prefix-list <P1>
#----------------------------------------------------------------------------------
class DynPfxSessionBgpConfig( CliCommand.CliCommandClass ):
   syntax = 'dynamic prefix-list NAME session bgp'
   noOrDefaultSyntax = syntax
   data = {
         'dynamic': 'Configure dynamic prefix-list',
         'prefix-list': 'Prefix list',
         'NAME': sessionBgpNameMatcher,
         'session': 'Configure session bgp',
         'bgp': 'Configure session bgp'
   }
   handler = "DynamicPrefixListCliHandler.handlerDynPfxSessionBgpConfig"
   noOrDefaultHandler = \
      "DynamicPrefixListCliHandler.noOrDefaultHandlerDynPfxSessionBgpConfig"

BasicCli.GlobalConfigMode.addCommandClass( DynPfxSessionBgpConfig )

class DynamicPrefixListConfig( CliCommand.CliCommandClass ):
   syntax = 'dynamic prefix-list NAME'
   noOrDefaultSyntax = syntax
   data = {
      'dynamic': 'Configure dynamic prefix-list',
      'prefix-list': 'Prefix list',
      'NAME': dynPfxListNameMatcher,
   }
   handler = "DynamicPrefixListCliHandler.handlerDynamicPrefixListConfig"
   noOrDefaultHandler = \
      "DynamicPrefixListCliHandler.noOrDefaultHandlerDynamicPrefixListConfig"

BasicCli.GlobalConfigMode.addCommandClass( DynamicPrefixListConfig )

#----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# [ no ] match-map <route-map-name>
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigModeMatchMap( CliCommand.CliCommandClass ):
   syntax = 'match-map MAPNAME'
   noOrDefaultSyntax = syntax
   data = {
      'match-map': 'Configure route map to determine the state of the dynamic '
                   'prefix list',
      'MAPNAME': CommonTokens.routeMapName,
   }
   handler = "DynamicPrefixListCliHandler.handlerDynamicPrefixListConfigModeMatchMap"
   noOrDefaultHandler = \
      "DynamicPrefixListCliHandler." + \
      "noOrDefaultHandlerDynamicPrefixListConfigModeMatchMap"

DynamicPrefixListConfigMode.addCommandClass( DynamicPrefixListConfigModeMatchMap )

# ----------------------------------------------------------------------------------
# (config-dyn-pfx-bgp-session-S1)# [ no ] match
#                                   session <ADDRESS> originate peer-address
# ----------------------------------------------------------------------------------
class DynPfxSessionBgpConfigModeMatchSession( CliCommand.CliCommandClass ):
   syntax = 'match session ADDRESS originate peer-address'
   noOrDefaultSyntax = syntax
   data = {
         'match': 'Session to Match',
         'session': 'Session Bgp',
         'ADDRESS': IpGenAddrMatcher( helpdesc='IPv4/IPv6 Peer address' ),
         'originate': 'Originate',
         'peer-address': 'Peer-address',
   }
   handler = \
      "DynamicPrefixListCliHandler.handlerDynPfxSessionBgpConfigModeMatchSession"
   noOrDefaultHandler = \
      "DynamicPrefixListCliHandler." + \
      "noOrDefaultHandlerDynPfxSessionBgpConfigModeMatchSession"

DynPfxBgpSessionConfigMode.addCommandClass( DynPfxSessionBgpConfigModeMatchSession )

#----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# [ no ] prefix-list ( ipv4 | ipv6 ) <prefix-list-name>
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigModePrefixList( CliCommand.CliCommandClass ):
   syntax = 'prefix-list ( ipv4 | ipv6 ) LISTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'prefix-list': 'Prefix list',
      'ipv4': 'IPv4 specific information',
      'ipv6': 'IPv6 specific information',
      'LISTNAME': dynPfxListNameMatcher,
   }
   handler = \
      "DynamicPrefixListCliHandler.handlerDynamicPrefixListConfigModePrefixList"
   noOrDefaultHandler = \
      "DynamicPrefixListCliHandler." + \
      "noOrDefaultHandlerDynamicPrefixListConfigModePrefixList"

DynamicPrefixListConfigMode.addCommandClass( DynamicPrefixListConfigModePrefixList )

#----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# abort
#----------------------------------------------------------------------------------
class DynamicPrefixListConfigModeAbort( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit without committing changes',
   }
   handler = "DynamicPrefixListCliHandler.handlerDynamicPrefixListConfigModeAbort"

DynamicPrefixListConfigMode.addCommandClass( DynamicPrefixListConfigModeAbort )

# ----------------------------------------------------------------------------------
# (config-dyn-pfx-bgp-session-S1)# abort
# ----------------------------------------------------------------------------------
class SessionBgpDplConfigModeAbort( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit without committing changes',
   }
   handler = "DynamicPrefixListCliHandler.handlerSessionBgpDplConfigModeAbort"

DynPfxBgpSessionConfigMode.addCommandClass( SessionBgpDplConfigModeAbort )

#-------------------------------------------------------------------------------
# Show commands
#-------------------------------------------------------------------------------

routeMapShowCmdDict = {}

allVrfWithNameMatcher = CliMatcher.DynamicNameMatcher(
   getAllPlusReservedVrfNames, helpdesc='VRF name' )

class ShowDynamicPrefixListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dynamic prefix-list [ LISTNAME ] [ vrf VRFNAME ]'
   data = {
      'dynamic': 'Show dynamic policy',
      'prefix-list': 'Prefix list',
      'LISTNAME': dynPfxListNameMatcher,
      'vrf': 'VRF name',
      'VRFNAME': allVrfWithNameMatcher,
   }
   cliModel = "DynamicPrefixListCliHandler.dynamicPrefixListVrfModel"
   handler = "DynamicPrefixListCliHandler.handlerShowDynamicPrefixListCmd"

BasicCli.addShowCommandClass( ShowDynamicPrefixListCmd )

#-------------------------------------------------------------------------------
# Register dynamic prefix-list Show Commands Into "Show tech-Support"
#-------------------------------------------------------------------------------
def showDynamicPrefixListGuard():
   return dynPfxListConfigDir.dynamicPrefixList

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2018-07-02 20:28:13',
   cmds=[ 'show dynamic prefix-List vrf all' ],
   cmdsGuard=showDynamicPrefixListGuard
)

def Plugin( entityManager ):
   global dynPfxListConfigDir
   global routeMapConfig

   dynPfxListConfigDir = ConfigMount.mount(
      entityManager, 'routing/dynPfxList/config',
      'Routing::DynamicPrefixList::Config', 'w' )
   routeMapConfig = LazyMount.mount( entityManager,
                                     "routing/routemap/config",
                                     "Routing::RouteMap::Config",
                                     "r" )

# Add matches for dynamic prefix-list

# match ip address dynamic prefix-list
class MatchIpAddrDynamicPrefixListCmd( CliCommand.CliCommandClass ):
   syntax = 'match ip address dynamic prefix-list LISTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Route map match rule',
      'ip': 'Match IP specific information',
      'address': 'Match ip address',
      'dynamic': 'Configure dynamic prefix-list',
      'prefix-list': 'Prefix list',
      'LISTNAME': dynPfxListNameMatcher,
   }
   handler = "DynamicPrefixListCliHandler.handlerMatchIpAddrDynamicPrefixListCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchIpAddrDynamicPrefixList' ):
   RouteMapMode.addCommandClass( MatchIpAddrDynamicPrefixListCmd )

# match ipv6 address dynamic prefix-list
class MatchIpv6AddrDynamicPrefixListCmd( CliCommand.CliCommandClass ):
   syntax = 'match ipv6 address dynamic prefix-list LISTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Route map match rule',
      'ipv6': 'Match IPv6 specific information',
      'address': 'Match ip address',
      'dynamic': 'Configure dynamic prefix-list',
      'prefix-list': 'Prefix list',
      'LISTNAME': dynPfxListNameMatcher,
   }
   handler = "DynamicPrefixListCliHandler.handlerMatchIpv6AddrDynamicPrefixListCmd"
   noOrDefaultHandler = handler

if isMatchAttrEnabled( 'matchIpv6AddrDynamicPrefixList' ):
   RouteMapMode.addCommandClass( MatchIpv6AddrDynamicPrefixListCmd )
