# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in

import CliParser
import CliCommand
from CliPlugin.EbraEthIntfCliModel import(
   InterfacesStatus, interfaceStatusHook, Encapsulation )
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin.VlanCli import SwitchportModelet
from CliPlugin.VlanCli import getSwitchIntfConfigEvenIfDisabled
from CliPlugin.PhysicalIntfRule import PhysicalIntfMatcher
from CliPlugin.IntfStatusCli import ShowIntfStatusCommand
from CliPlugin.EthIntfCli import (
   EthPhyIntf, EthPhyAutoIntfType, UnconnectedEthPhyAutoIntfType )
from CliPlugin.SwitchIntfCli import SwitchIntf, SwitchAutoIntfType
import LazyMount
import BasicCli, ConfigMount
import CliToken.Switch
from EthIntfUtil import switchPortIntfTypes # pylint: disable=unused-import
from EthIntfUtil import registerSwitchPortIntfType
from TypeFuture import TacLazyType

ethIntfStatusDir = None
bridgingHwCapabilities = None
bridgingConfig = None
cliConfig = None

EncapConfigMode = TacLazyType( "Interface::SubIntfDot1qEncapConfigMode" )

switchPortMatcher = IntfMatcher()
switchPortMatcher |= PhysicalIntfMatcher(
   'Ethernet',
   value=lambda mode, intf: EthPhyIntf( intf, mode ) )
switchPortMatcher |= SwitchIntf.matcher

registerSwitchPortIntfType( EthPhyAutoIntfType )
registerSwitchPortIntfType( UnconnectedEthPhyAutoIntfType )
registerSwitchPortIntfType( SwitchAutoIntfType )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] status" command, in enable mode.
#  show interfaces [<name>] status
#  show interfaces status module <modnum>
#  show interfaces module <modnum> status
#-------------------------------------------------------------------------------
# returns a tuple. The first parameter is the mode (routed, inactive, bridged
# or trunk). The second parameter returns the VLAN id if applicable otherwise None
def _generateBridgedOrRoutedVlanInfo( intf ):
   switchIntfConfig = bridgingConfig.switchIntfConfig.get( intf.name ) 
   if( not switchIntfConfig or switchIntfConfig.switchportMode == 'routed' ): 
      return ( 'routed', None )
   elif switchIntfConfig.switchportMode == 'access' or \
        switchIntfConfig.switchportMode == 'dot1qTunnel':
      if switchIntfConfig.nativeVlan == 0:
         return ( 'inactive', None )
      else:
         return ( 'bridged', switchIntfConfig.nativeVlan )
   elif switchIntfConfig.switchportMode in ( 'trunk', 'tap', 'tool', 'tapTool' ):
      return ( switchIntfConfig.switchportMode, None )
   else:
      assert False, "unexpected switchportMode"

def getVlanInfo( intf, mode ):
   # If intf isn't bridged then vlan isn't relevant.  Overwrite that field
   # with explanation for why not bridged.
   eis = ethIntfStatusDir.intfStatus.get( intf.name )
   vlanInfo = InterfacesStatus.InterfaceStatusEntry.VlanInformation()
   if intf.hardware() == 'fabric':
      vlanInfo.interfaceMode, vlanInfo.vlanId = 'fabric', None
   elif intf.isSubIntf():
      subIntfStatus = intf.status()
      if subIntfStatus.isL2SubIntf():
         if subIntfStatus.forwardingVlanId > 0:
            vlanInfo.vlanId = subIntfStatus.forwardingVlanId
            vlanInfo.interfaceMode = "encap"
         else:
            vlanInfo.interfaceMode = "inactive"
      else:
         subIntfConfig = intf.config()
         # We populate encap outer tag into vlanId for single tag L3 subintf
         # to be backwards compatible.
         if not subIntfConfig.dot1qEncap.innerTag:
            vlanInfo.vlanId = subIntfConfig.dot1qEncap.outerTag
         vlanInfo.interfaceMode = "encap"
   elif eis:
      if eis.forwardingModel == "intfForwardingModelLinkQualification":
         ( vlanInfo.vlanExplanation, _, _, _ ) = \
               intf.linkQualificationExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelLayer1Patch":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.layer1PatchExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelLayer1PatchAndLldp":
         ( vlanInfo.vlanExplanation, _, _, _ ) = \
               intf.layer1PatchExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelRecirculation":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.recirculationExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelDataLink":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.dataLinkExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelQuietDataLink":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.quietDataLinkExplanation( mode, intf.status() )
      elif eis.forwardingModel == "intfForwardingModelUnauthorized":
         ( vlanInfo.vlanExplanation , _, _, _ ) = \
               intf.unauthorizedExplanation( mode, intf.status() )
      else:
         ( vlanInfo.interfaceMode, 
           vlanInfo.vlanId ) = _generateBridgedOrRoutedVlanInfo( intf )

   vlanInfo.interfaceForwardingModel = intf.forwardingModel()
   return vlanInfo

def getEncapsulation( intf ):
   encapsulation = None
   if intf.isSubIntf():
      subIntfConfig = intf.config()
      if subIntfConfig.dot1qEncap.outerTag:
         encapsulation = Encapsulation()
         encapsulation.encapsulationType = "dot1q"
         if ( subIntfConfig.dot1qEncapConfigMode ==
              EncapConfigMode.subIntfDot1qEncapConfigLegacy ):
            tags = [ subIntfConfig.dot1qEncap.outerTag ]
            if subIntfConfig.dot1qEncap.innerTag:
               tags.append( subIntfConfig.dot1qEncap.innerTag )
            encapsulation.dot1qVlanTags = tags
         elif ( subIntfConfig.dot1qEncapConfigMode ==
              EncapConfigMode.subIntfDot1qEncapConfigFlexEncap ):
            # Flex Encap information cannot be filled into 'show interface status'
            # due to limited space in the existing encapsulation column.
            # Instead, a * will be printed in the encapsulation column, along
            # with a warning message, directing the operator to use the alternate
            # show command.
            # pylint: disable=protected-access
            encapsulation._incompleteInformation = True
   return encapsulation

def showIntfStatusIncompleteEncapWarningMsg():
   return ( "* Incomplete encapsulation information. "
            "Use 'show interface encapsulation vlan' instead" )

def showInterfacesStatus( mode, intfs ):
   ret = InterfacesStatus()
   showIncompleteEncapWarning = False

   for x in intfs:
      try:
         eis = ethIntfStatusDir.intfStatus.get( x.name )
         encapsulation = getEncapsulation( x )
         interfaceStatusEntry = InterfacesStatus.InterfaceStatusEntry(
            description=x.description(),
            linkStatus=x.linkStatus(),
            vlanInformation=getVlanInfo( x, mode ),
            encapsulation=encapsulation,
            bandwidth=x.bandwidth() * 1000,
            autoNegotigateActive=x.autonegActive(),
            autoNegotiateActive=x.autonegActive(),
            duplex=x.getIntfDuplexStr(),
            interfaceType=x.xcvrTypeStr(),
            lineProtocolStatus=x.getStatus()[ 0 ],
            interfaceDamped=eis.linkDamped,
         )
         for hook in interfaceStatusHook.extensions():
            hook( x.name, interfaceStatusEntry )
         ret.interfaceStatuses[ x.name ] = interfaceStatusEntry
         # pylint: disable=protected-access
         if encapsulation._incompleteInformation:
            showIncompleteEncapWarning = True
      except AttributeError:
         # Some interfaces have been deleted during the process
         continue
   if showIncompleteEncapWarning:
      warning = showIntfStatusIncompleteEncapWarningMsg()
      mode.addWarning( warning )
   return ret

def showInterfacesStatusHandler( mode, args ):
   intfs = ShowIntfStatusCommand.getIntfs( mode, args )
   if not intfs:
      return InterfacesStatus()

   return showInterfacesStatus( mode, intfs )

class ShowIntfStatus( ShowIntfStatusCommand ):
   syntax = 'show interfaces status '
   data = {}
   cliModel = InterfacesStatus
   handler = showInterfacesStatusHandler

BasicCli.addShowCommandClass( ShowIntfStatus )

#-------------------------------------------------------------------------------
# The "[no|default] switchport source-interface tx" command, in "config-if" mode.
#
# The full syntax of this command is:
#
#   switchport source-interface tx
#   {no|default} switchport source-interface tx
#-------------------------------------------------------------------------------

def sourceportFilterSupportedGuard( mode, token ):
   if bridgingHwCapabilities.sourceportFilterSupported:
      return None
   return CliParser.guardNotThisPlatform

def sourceportFilterMulticastSupportedGuard( mode, token ):
   if bridgingHwCapabilities.sourceportFilterMulticastSupported:
      return None
   return CliParser.guardNotThisPlatform

def disableSourceportFilterMode( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   
   mode = 'sourceportFilterDisabled'
   if 'multicast' in args:
      mode = 'sourceportFilterMulticastDisabled'

   sic.sourceportFilterMode = mode

def defaultSourceportFilterMode( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   sic.sourceportFilterMode = 'sourceportFilterEnabled'

class SourceportFilterModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport source-interface tx [ multicast ]'
   noOrDefaultSyntax = syntax
   data = {
         'switchport' : 'Set switching mode characteristics',
         'source-interface' : CliCommand.guardedKeyword( 'source-interface', 
                                 helpdesc='Configure source interface properties',
                                 guard=sourceportFilterSupportedGuard ),
         'tx' : CliCommand.guardedKeyword( 'tx',
                   helpdesc='Allow bridged packets to go out on the '
                            'source interface',
                   guard=sourceportFilterSupportedGuard ),
         'multicast' : CliCommand.guardedKeyword( 'multicast',
                   helpdesc='Allow multicast packets only to go out on '
                            'the source inteface',
                   guard=sourceportFilterMulticastSupportedGuard )
         }
   handler = disableSourceportFilterMode
   noOrDefaultHandler = defaultSourceportFilterMode

SwitchportModelet.addCommandClass( SourceportFilterModeCmd )

#-------------------------------------------------------------------------------
# The "[no|default] forwarding l3 source-interface drop" command
#-------------------------------------------------------------------------------
def l3SourceportDropSupportedGuard( mode, token ):
   if not bridgingHwCapabilities.l3SourceportDropSupported:
      return CliParser.guardNotThisPlatform
   # FIXME: remove this guard if we support it on LAG and all
   # other switchports
   if not isinstance( mode.intf, EthPhyIntf ):
      return "Not supported for this interface"
   return None

class ForwardingL3SourceIntfDrop( CliCommand.CliCommandClass ):
   syntax = "forwarding l3 source-interface drop"
   noOrDefaultSyntax = syntax
   data = {
      "forwarding" : CliToken.Switch.matcherForwarding,
      "l3" : 'Configure L3 forwarding features',
      "source-interface" : CliCommand.guardedKeyword(
         'source-interface',
         'Configure source interface properties',
         l3SourceportDropSupportedGuard ),
      "drop" : 'Drop packets'
   }

   @staticmethod
   def handler( mode, args ):
      sic = getSwitchIntfConfigEvenIfDisabled( mode )
      sic.l3SourceportDrop = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sic = getSwitchIntfConfigEvenIfDisabled( mode )
      sic.l3SourceportDrop = False

SwitchportModelet.addCommandClass( ForwardingL3SourceIntfDrop )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ethIntfStatusDir, bridgingConfig
   global bridgingHwCapabilities, cliConfig
   ethIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   cliConfig = ConfigMount.mount( entityManager, "bridging/input/config/cli", 
                                  "Bridging::Input::CliConfig", "w" )
