# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
from CliMode.EosSdkRpc import MgmtEosSdkRpcMode, EosSdkRpcTransportMode
import ShowCommand

import Toggles.EosSdkRpcToggleLib as EosSdkRpcToggle

# Tell package dependency generator that we depend on our SysdbMountProfile
# pkgdeps: rpmwith %{_libdir}/SysdbMountProfiles/ConfigAgent-DaemonCli

from CliPlugin.AclCli import (
      accessGroupKwMatcher,
      inKwMatcher,
      ipKwForServiceAclMatcher,
      standardIpAclNameMatcher )
from CliPlugin.ConfigMgmtMode import (
      apiKwMatcher,
      managementKwMatcher,
      managementShowKwMatcher )
from CliPlugin.LauncherDaemonCli import DaemonConfigModeBase
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin import VrfCli
from CliPlugin import (
      EthIntfCli,
      LoopbackIntfCli,
      ManagementActiveIntfCli
)
import CliMatcher
import ConfigMount
import LazyMount
import Tac
from TypeFuture import TacLazyType
from Url import UrlMatcher

AGENT_TYPE = TacLazyType( 'GenericAgent::AgentTypeEnum' )

controlConfig = None
serverConfigDir = None
sslConfig = None

executeable = "/usr/bin/EosSdkRpcAgent"

matcherProfile = CliMatcher.KeywordMatcher( 'profile',
      helpdesc='Configure SSL profile' )
matcherSsl = CliMatcher.KeywordMatcher( 'ssl',
      helpdesc='Configure SSL related options' )
sslProfileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: sslConfig.profileConfig,
      'Profile name' )
matcherApiForShow = CliMatcher.KeywordMatcher( 'api',
      helpdesc='Show management APIs' )

localhostHelpDesc = 'Configure the local endpoints'
# -----------------------------------------------------------------------------------
# This module implements the following configuration.

# management api eos-sdk-rpc
#       [no] transport grpc <name>
#       [no] disabled

# -----------------------------------------------------------------------------------
# EosSdkRpc mode
# -----------------------------------------------------------------------------------
class MgmtEosSdkRpcConfigMode( MgmtEosSdkRpcMode, BasicCli.ConfigModeBase ):
   name = "EosSdkRpc configuration"

   def __init__( self, parent, session ):
      MgmtEosSdkRpcMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )


class MgmtApiEosSdkRpcCmd( CliCommand.CliCommandClass ):
   syntax = 'management api eos-sdk-rpc'
   noOrDefaultSyntax = syntax
   data = {
         'management': managementKwMatcher,
         'api': apiKwMatcher,
         'eos-sdk-rpc': 'EosSdk gRPC server configuration',
   }

   handler = "EosSdkRpcCliHandler.MgmtApiEosSdkRpcCmd_handler"
   noOrDefaultHandler = "EosSdkRpcCliHandler.MgmtApiEosSdkRpcCmd_noOrDefaultHandler"

BasicCliModes.GlobalConfigMode.addCommandClass( MgmtApiEosSdkRpcCmd )

# -----------------------------------------------------------------------------------
# [ no ] transport grpc SERVER_NAME
# -----------------------------------------------------------------------------------
class EosSdkRpcTransportConfigMode( EosSdkRpcTransportMode,
                                    DaemonConfigModeBase ):
   name = "EosSdkRpc server configuration"

   def __init__( self, parent, session, name ):
      self.daemonName = name
      self.callModeConstructorsInOrder( parent, session, self.daemonName,
            EosSdkRpcTransportMode )
      self.serverConfig = serverConfigDir.get( self.daemonName )
      if not self.serverConfig:
         self.serverConfig = serverConfigDir.newEntity( 'EosSdkRpc::Server::Config',
                                                     self.daemonName )
      if not self.launcherCfg.exe:
         self.doSetDaemonExecutable(
               { "EXECUTABLE": executeable,
                 "ARGV": f"--daemon-name {self.daemonName}" } )
      # Ensure EosSdkRpcControl agent is enabled
      controlConfig.enabled = True

NAME_MAX_LEN = Tac.Type( "GenericAgent::AgentName" ).maxLength

class TransportModeCmd( CliCommand.CliCommandClass ):
   syntax = 'transport grpc TRANSPORT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'transport': 'Configure a transport',
         'grpc': 'Configure gRPC transport for EosSdkRpc',
         'TRANSPORT_NAME': CliMatcher.PatternMatcher(
            pattern=f'[A-Za-z0-9_-]{{1,{NAME_MAX_LEN}}}',
            helpdesc='Transport name',
            helpname='WORD' ),
   }

   handler = "EosSdkRpcCliHandler.TransportModeCmd_handler"
   noOrDefaultHandler = "EosSdkRpcCliHandler.TransportModeCmd_noHandler"

MgmtEosSdkRpcConfigMode.addCommandClass( TransportModeCmd )

# --------------------------------------------------------------------------------
# [ no | default ] accounting requests all logging
# --------------------------------------------------------------------------------
class AccountingCmd( CliCommand.CliCommandClass ):
   syntax = 'accounting requests all logging'
   noOrDefaultSyntax = syntax
   data = {
      'accounting': 'Configure accounting',
      'requests': 'Configure RPC requests accounting',
      'all': 'All RPC request types',
      'logging': 'System log each RPC request'
   }

   handler = "EosSdkRpcCliHandler.AccountingRequests_handler"
   noOrDefaultHandler = "EosSdkRpcCliHandler.AccountingRequests_noOrDefaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( AccountingCmd )

# -----------------------------------------------------------------------------------
# [ no ] disabled
# -----------------------------------------------------------------------------------
class DisableCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Disable the EosSdk gRPC server',
   }
   handler = "EosSdkRpcCliHandler.DisableCmd_handler"
   noOrDefaultHandler = "EosSdkRpcCliHandler.DisableCmd_noOrDefaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( DisableCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] local interface INTF [ port PORT ]
# -----------------------------------------------------------------------------------
intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.matcher
intfMatcher |= LoopbackIntfCli.LoopbackIntf.matcher
# Include matcher for Ma0 on modular duts
intfMatcher |= ManagementActiveIntfCli.ManagementActiveIntf.matcher

class LocalIntfPortCmd( CliCommand.CliCommandClass ):
   syntax = 'local interface INTERFACE [ port PORT ]'
   noOrDefaultSyntax = syntax
   data = {
         'local': 'Configure the local interface and port',
         'interface': 'Configure EosSdkRpc server to listen on an interface',
         'INTERFACE': intfMatcher,
         'port': 'Configure listening port for EosSdkRpc (default: 9543)',
         'PORT': CliMatcher.IntegerMatcher( 1, 65535,
                                 helpdesc='EosSdkRpc server listening port' ),
   }
   handler = "EosSdkRpcCliHandler.LocalIntfPortCmd_handler"
   noOrDefaultHandler = "EosSdkRpcCliHandler.LocalIntfPortCmd_noOrDefaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( LocalIntfPortCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] localhost loopback [ port PORT ] [ vrf VRF ]
# -----------------------------------------------------------------------------------

class LocalhostLoopbackCmd( CliCommand.CliCommandClass ):
   syntax = 'localhost loopback [ port PORT ] [ vrf VRF ]'
   noOrDefaultSyntax = syntax
   data = {
         'localhost': localhostHelpDesc,
         'loopback': 'Configure loopback address',
         'port': 'Configure listening port for EosSdkRpc (default: 9543)',
         'PORT': CliMatcher.IntegerMatcher( 1, 65535,
                                 helpdesc='EosSdkRpc server listening port' ),
         'vrf': 'Configure VRF',
         'VRF': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }
   handler = "EosSdkRpcCliHandler.LocalhostLoopbackCmd_handler"
   noOrDefaultHandler = "EosSdkRpcCliHandler.LocalhostLoopbackCmd_noOrDefaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( LocalhostLoopbackCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] localhost unix-socket [ file path ]
# -----------------------------------------------------------------------------------

class LocalhostUnixSocketCmd( CliCommand.CliCommandClass ):
   syntax = 'localhost unix-socket FILE'
   noOrDefaultSyntax = syntax
   data = {
         'localhost': localhostHelpDesc,
         'unix-socket': 'Configure Unix Domain Socket',
         'FILE': UrlMatcher( fsFunc=lambda fs: fs.scheme == 'file:',
                             helpdesc="Location of unix-socket" ),
   }
   handler = "EosSdkRpcCliHandler.LocalhostUnixSocketCmd_handler"
   noOrDefaultHandler = \
         "EosSdkRpcCliHandler.LocalhostUnixSocketCmd_noOrDefaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( LocalhostUnixSocketCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] service [ all | { SERVICES } ]
# -----------------------------------------------------------------------------------
sdkServices = {
      'AclService': 'RPCs specific to ACL',
      'AgentService': 'RPCs specific to this agent',
      'BgpPathService': 'RPCs specific to BGP path',
      'BgpService': 'RPCs specific to BGP',
      'ClassMapService': 'RPCs specific to class maps',
      'EapiService': 'RPCs specific to EAPI',
      'EthLagIntfService': 'RPCs specific to Ethernet LAG interfaces',
      'EthPhyIntfService': 'RPCs specific to Ethernet physical interfaces',
      'EthPhyIntfCountersService': 'RPCs specific to Ethernet physical interface '
         'counters',
      'IntfService': 'RPCs specific to interfaces',
      'IntfCounterService': 'RPCs specific to interface counters',
      'IpRouteService': 'RPCs specific to IP route',
      'IpIntfService': 'RPCs specific to IP interfaces',
      'MacsecService': 'RPCs specific to MACsec',
      'MplsRouteService': 'RPCs specific to MPLS',
      'MplsVrfLabelService': 'RPCs specific to MPLS VRF labels',
      'NexthopGroupService': 'RPCs specific to nexthop groups',
      'PolicyMapCountersService': 'RPCs sepcific to policy map counters',
      'PolicyMapService': 'RPCs specific to policy maps',
}

class EnableServicesCmd( CliCommand.CliCommandClass ):
   syntax = 'service ( all | { SERVICES } )'
   noOrDefaultSyntax = 'service [ all | { SERVICES } ]'
   data = {
         'service': 'Enable defined gRPC services',
         'all': 'Enable all defined gRPC services',
         'SERVICES': CliCommand.SetEnumMatcher( sdkServices )
   }
   handler = "EosSdkRpcCliHandler.EnableServicesCmd_handler"
   noHandler = "EosSdkRpcCliHandler.EnableServicesCmd_noHandler"
   defaultHandler = "EosSdkRpcCliHandler.EnableServicesCmd_defaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( EnableServicesCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ip access-group ACLNAME [ in ]
# --------------------------------------------------------------------------------
class IpAccessGroupAclNameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACLNAME [ in ]'
   noOrDefaultSyntax = 'ip access-group ...'
   data = {
      'ip': ipKwForServiceAclMatcher,
      'access-group': accessGroupKwMatcher,
      'ACLNAME': standardIpAclNameMatcher,
      'in': inKwMatcher,
   }
   handler = "EosSdkRpcCliHandler.IpAccessGroupAclNameCmd_handler"
   noOrDefaultHandler = \
      "EosSdkRpcCliHandler.IpAccessGroupAclNameCmd_noOrDefaultHandler"

if EosSdkRpcToggle.toggleEosSdkRpcAclEnabled():
   EosSdkRpcTransportConfigMode.addCommandClass( IpAccessGroupAclNameCmd )

# --------------------------------------------------------------------------------
# vrf [ VRFNAME ]
# --------------------------------------------------------------------------------
class VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRFNAME'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure VRF',
      'VRFNAME': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }
   handler = "EosSdkRpcCliHandler.VrfCmd_handler"
   noOrDefaultHandler = handler

if EosSdkRpcToggle.toggleEosSdkRpcAclVrfEnabled():
   EosSdkRpcTransportConfigMode.addCommandClass( VrfCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ssl profile PROFILENAME
# --------------------------------------------------------------------------------
class SslProfileProfileNameCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': matcherSsl,
      'profile': matcherProfile,
      'PROFILENAME': sslProfileNameMatcher,
   }
   handler = "EosSdkRpcCliHandler.SslProfileProfileNameCmd_handler"
   noOrDefaultHandler = \
      "EosSdkRpcCliHandler.SslProfileProfileNameCmd_noOrDefaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( SslProfileProfileNameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] metadata username authentication channel secure
# --------------------------------------------------------------------------------
class UserAuthCmd( CliCommand.CliCommandClass ):
   syntax = 'metadata username authentication channel secure'
   noOrDefaultSyntax = 'metadata username authentication ...'
   data = {
      'metadata': 'Configure gRPC metadata options',
      'username': 'Configure username and password options',
      'authentication': 'Configure authentication options',
      'channel': 'Configure the gRPC channel type',
      'secure': 'Configure for secure channels (TLS/mTLS) only'
   }
   handler = "EosSdkRpcCliHandler.UserAuthCmd_handler"
   noOrDefaultHandler = "EosSdkRpcCliHandler.UserAuthCmd_noOrDefaultHandler"

EosSdkRpcTransportConfigMode.addCommandClass( UserAuthCmd )

# --------------------------------------------------------------------------------
# show management api eos-sdk-rpc
# --------------------------------------------------------------------------------
class ManagementApiEosSdkRpcShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api eos-sdk-rpc'
   data = {
      'management': managementShowKwMatcher,
      'api': matcherApiForShow,
      'eos-sdk-rpc': 'Show status of EosSdkRpc endpoints',
   }
   handler = "EosSdkRpcShowCmdHandler.showStatusEosSdkRpc"
   cliModel = "EosSdkRpcModel.EosSdkRpcAggregateStatus"

BasicCli.addShowCommandClass( ManagementApiEosSdkRpcShowCmd )

def Plugin( entityManager ):
   global controlConfig, serverConfigDir
   global sslConfig

   serverConfigDir = ConfigMount.mount( entityManager,
                                        'eossdkrpc/server/config',
                                        'Tac::Dir', 'wi' )
   controlConfig = ConfigMount.mount( entityManager,
                                      'eossdkrpc/control/config',
                                      'EosSdkRpc::Control::Config', 'w' )
   sslConfig = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/config",
                                "Mgmt::Security::Ssl::Config",
                                "r" )
