# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements errdisable configuration.  
# -  the "[no] errdisable detect cause { cause1 | cause2 ... }" command
# -  the "[no] errdisable recovery cause { cause1 | cause2 ... }
#    [ interval <secs> ]" command
# -  the "[no] errdisable recovery interval <secs>" command
# -  the "show errdisable { recovery | detect | description }" command
# -  the "show interfaces status errdisabled" command
#
# The cause names are dynamically discovered from the sysdb directory:
# interface/errdisable/causegroup/
#-------------------------------------------------------------------------------

import textwrap

import Arnet
import BasicCli
import CliCommand
import CliMatcher
from CliModel import (
   Dict,
   List,
   Model,
   Str
)
import CliParserCommon
from CliPlugin import IntfCli
from IntfModels import Interface
import ConfigMount
import ErrdisableCliLib
from ErrdisableCliLib import causeNameWithoutSliceId
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import LazyMount
import ShowCommand
from TableOutput import createTable, Format
import Tac

errDisableConfig = None
allIntfConfigDir = None

#-------------------------------------------------------------------------------
# The "[no|default] errdisable detect cause { <cause1> | <cause2> ... }" command.
#-------------------------------------------------------------------------------
def detectCauseHelpDict( mode, hidden ):
   d = {}
   causeGroupDir = mode.sysdbRoot.entity[ "interface/errdisable/causegroup" ]
   for name in ErrdisableCliLib.getCauseGroups( causeGroupDir ):
      cg = causeGroupDir[ name ]
      if cg.installCauseEnableCliCmd and ( cg.hiddenCause == hidden ):
         strippedName = causeNameWithoutSliceId( name )
         d[ strippedName ] = "The %s cause" % ( strippedName )
   # this is here to prevent multiple matches in Cli save tests
   d[ 'link-change' ] = 'The link-change cause'
   d.pop( 'link-flap', None )
   return d
 
class ErrdisableDetectCause( CliCommand.CliCommandClass ):
   syntax = ( 'errdisable detect cause '
          '( link-flap | CAUSE_NAME1 | CAUSE_NAME2 )' )
   noOrDefaultSyntax = syntax
   data = {
      'errdisable': 'Configure error disable functionality',
      'detect': 'Configure error disable detection',
      'cause': 'Specify the errdisable cause',
      'link-flap': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'link-flap',
            priority=CliParserCommon.PRIO_LOW,
            helpdesc='The link-flap cause'),
         alias='CAUSE',
         deprecatedByCmd='errdisable detect cause link-change' ),
      'CAUSE_NAME1': CliCommand.Node(
         matcher=CliMatcher.DynamicKeywordMatcher(
            lambda mode: detectCauseHelpDict( mode, False ),
            alwaysMatchInStartupConfig=True ),
         alias='CAUSE'),
      'CAUSE_NAME2': CliCommand.Node( 
         CliMatcher.DynamicKeywordMatcher(
            lambda mode: detectCauseHelpDict( mode, True ),
         priority=CliParserCommon.PRIO_NORMAL ),
         alias='CAUSE',
         hidden=True )
   }
   
   @staticmethod
   def handler( mode, args ):
      causeGroup = args[ 'CAUSE' ]
      if causeGroup == 'link-change':
         causeGroup = 'link-flap'
      del errDisableConfig.disabled[ causeGroup ]
      errDisableConfig.enabled[ causeGroup ] = True
   
   @staticmethod
   def noHandler( mode, args ):
      causeGroup = args[ 'CAUSE' ]
      if causeGroup == 'link-change':
         causeGroup = 'link-flap'
      errDisableConfig.disabled[ causeGroup ] = True
      del errDisableConfig.enabled[ causeGroup ]

   @staticmethod
   def defaultHandler( mode, args ):
      causeGroup = args[ 'CAUSE' ]
      if causeGroup == 'link-change':
         causeGroup = 'link-flap'
      del errDisableConfig.disabled[ causeGroup ]
      del errDisableConfig.enabled[ causeGroup ]

BasicCli.GlobalConfigMode.addCommandClass( ErrdisableDetectCause )

#---------------------------------------------------------------------------------
# The "[no|default] errdisable recovery cause { <cause1> | <cause2> ... }
# [ interval <secs> ]" command.
#--------------------------------------------------------------------------------
def recoveryCauseHelpDict( mode, hidden, enableOrDisable ):
   d = {}
   causeGroupDir = mode.sysdbRoot.entity[ "interface/errdisable/causegroup" ]
   for name in ErrdisableCliLib.getCauseGroups( causeGroupDir ):
      cg = causeGroupDir[ name ]
      if cg.installCauseRecoveryCliCmd and ( cg.hiddenCause == hidden ):
         strippedName = causeNameWithoutSliceId( name )
         d[ strippedName ] = "%s the %s cause" % ( enableOrDisable, strippedName )
   return d

class ErrdisableRecoveryCauseInterval ( CliCommand.CliCommandClass ):
   syntax = ( 'errdisable recovery cause ( CAUSE_NAME1 | CAUSE_NAME2 )' \
              ' [ interval INTERVAL_VAL ]' )
   noOrDefaultSyntax = ( 'errdisable recovery cause ( CAUSE_NAME1 | CAUSE_NAME2 )'
                         ' [ interval ... ]' )
   data = {
      'errdisable': 'Configure error disable functionality',
      'recovery': 'Configure errdisable recovery',
      'cause': 'Specify the errdisable cause',
      'CAUSE_NAME1': CliMatcher.DynamicKeywordMatcher(
         lambda mode: recoveryCauseHelpDict( mode, False, "Enable" ),
         alwaysMatchInStartupConfig=True ),
      'CAUSE_NAME2': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher(
            lambda mode: recoveryCauseHelpDict( mode, True, "Enable" ) ),
         hidden=True ),
      'interval': 'Configure the cause-specific recovery timer interval',
      'INTERVAL_VAL': CliMatcher.IntegerMatcher( 30,
                                             86400,
                                             helpdesc='Recovery time in seconds' ),
   }

   @staticmethod
   def handler ( mode, args ):
      if 'CAUSE_NAME1' in args:
         causeName = args[ 'CAUSE_NAME1' ]
      else:
         causeName = args[ 'CAUSE_NAME2' ]
      # handles setting recovery timer for the individual cause
      if 'interval' in args:
         errDisableConfig.causeRecoveryInterval[ causeName ] = args[ 'INTERVAL_VAL' ]
      # handles enabling recovery for the cause
      else:
         errDisableConfig.timerRecoveryEnabled[ causeName ] = True

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      if 'CAUSE_NAME1' in args:
         causeName = args[ 'CAUSE_NAME1' ]
      else:
         causeName = args[ 'CAUSE_NAME2' ]
      if 'interval' in args:
         del errDisableConfig.causeRecoveryInterval[ causeName ]
      else:
         del errDisableConfig.timerRecoveryEnabled[ causeName ]

BasicCli.GlobalConfigMode.addCommandClass( ErrdisableRecoveryCauseInterval )

#-------------------------------------------------------------------------------
# The "[no|default] errdisable recovery interval <secs>" command.
#-------------------------------------------------------------------------------
class ErrdisableRecoveryInterval( CliCommand.CliCommandClass ):
   syntax = 'errdisable recovery interval SECONDS'
   noOrDefaultSyntax = 'errdisable recovery interval ...'
   data = {
      'errdisable': 'Configure error disable functionality',
      'recovery': 'Configure errdisable recovery',
      'interval': 'Configure recovery timer interval',
      'SECONDS': CliMatcher.IntegerMatcher( 30, 86400,
                                            helpdesc='Recovery time in seconds' )
   }

   @staticmethod
   def handler( mode, args ):
      errDisableConfig.recoveryInterval = args[ 'SECONDS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      errDisableConfig.recoveryInterval = errDisableConfig.defaultRecoveryInterval

BasicCli.GlobalConfigMode.addCommandClass( ErrdisableRecoveryInterval )
#-------------------------------------------------------------------------------
# The "show errdisable recovery" command.
#-------------------------------------------------------------------------------
def getErrdisabledIntfs( mode, intfName=None,
                         includeInactive=True,
                         includeUnconnected=True ):
   intfCauseSet = { }
   intfErrdisableExpirytime = { }

   sysdbRoot = mode.sysdbRoot
   causeGroupDir = sysdbRoot.entity[ "interface/errdisable/causegroup" ]

   # pylint: disable-next=too-many-nested-blocks
   for causeGroupName, causeGroup in causeGroupDir.items():
      if causeGroup.type != 'errdisable':
         continue
      causeGroupName = causeNameWithoutSliceId( causeGroupName )
      timerEnabled = causeGroupName in errDisableConfig.timerRecoveryEnabled
      for causeData in causeGroup.causeStatus.values():
         intfsToCheck = causeData.intfStatus

         if intfName is not None:
            if intfName in intfsToCheck:
               intfsToCheck = [ intfName ]
            else:
               intfsToCheck = [ None ]

         for intf in intfsToCheck:
            if intf:
               intfType, _ = IntfRange.intfTypeFromName( intf )
               cliIntf = intfType.getCliIntf( mode, intf )
               if not cliIntf.userVisible():
                  continue
            # don't show inactive intfs if they are not exposed in Cli
            if not includeInactive:
               intfConfig = allIntfConfigDir.intfConfig.get( intf )
               if intfConfig and intfConfig.enabledStateReason == 'inactive':
                  continue
            # don't show unconnected intfs if they are not exposed in Cli            
            if not includeUnconnected:
               if intf.startswith( ( 'Un', 'Ue' ) ):
                  continue
            if intfName is None or intf == intfName:
               if timerEnabled:
                  expiryTime = causeData.intfStatus[ intf ].errdisabledTime
                  expiryTime += errDisableConfig.recoveryInterval
               else:
                  expiryTime = Tac.endOfTime

               if intf in intfCauseSet:
                  intfCauseSet[ intf ].add( causeGroupName )
                  if intfErrdisableExpirytime[ intf ] < expiryTime:
                     intfErrdisableExpirytime[ intf ] = expiryTime
               else:
                  intfCauseSet[ intf ] = { causeGroupName }
                  intfErrdisableExpirytime[ intf ] = expiryTime            

   return intfCauseSet, intfErrdisableExpirytime 

def multiLineCauseFormat( width, causeList ):
   """Concatenates the causes, into multiple comma separated strings.
   Each string will be less than 'width' in len if the cause names are
   less than 'width' in length. If a cause name is more than width, then
   the string containing that cause will not contain any other cause"""

   currAvailableWidth = width - 1  # to keep room for ','
   lines = [ ]
   line = [ ]
   for cause in causeList:
      if line and len( cause ) > currAvailableWidth:
         lines.append( line )
         currAvailableWidth = width - 1  # to keep room for ','
         line = [ ]
      line.append( cause )
      currAvailableWidth -= len( cause ) + 1   # the cause string len + ','
   lines.append( line )

   output = [ ]
   if not lines:
      return output
   line = lines.pop( 0 )
   s = ','.join( line )
   for line in lines:
      s += ','
      output.append( s )
      s = ','.join( line )
   output.append( s )
   return output


class ShowErrdisableRecovery( ShowCommand.ShowCliCommandClass ):
   syntax = 'show errdisable recovery'
   data = {
      'errdisable': 'Show errdisable information',
      'recovery': 'Show errdisable recovery information'
   }

   @staticmethod
   def handler( mode, args ):
      # Use Table formatting support.
      t1 = createTable( ( "Errdisable Reason", "Timer Status",
                          "Timer Interval" ) )
      f1 = Format( justify="left", noBreak=True, terminateRow=False )
      f2 = Format( justify="left" )
      f2.noTrailingSpaceIs( True )
      t1.formatColumns( f1, f2 )

      causesSeen = set()
      causeGroupDir = mode.sysdbRoot.entity[ "interface/errdisable/causegroup" ]
      for causeGroup in sorted( ErrdisableCliLib.getCauseGroups( causeGroupDir ) ):

         if causeGroupDir[ causeGroup ].hiddenCause:
            continue
         strippedCauseName = causeNameWithoutSliceId( causeGroup )
         if strippedCauseName in causesSeen:
            continue
         causesSeen.add( strippedCauseName )
         if strippedCauseName in errDisableConfig.timerRecoveryEnabled:
            status = "Enabled"
         else:
            status = "Disabled"

         if strippedCauseName in errDisableConfig.causeRecoveryInterval:
            causeInterval = '%d' % \
                  errDisableConfig.causeRecoveryInterval[ strippedCauseName ]
         elif causeGroupDir[ causeGroup ].installCauseRecoveryCliCmd:
            causeInterval = '%d' % errDisableConfig.recoveryInterval
         else:
            causeInterval = 'N/A'
         t1.newRow( strippedCauseName, status, causeInterval )

      print( t1.output() )

      ( intfCauseSet, intfErrdisableExpirytime ) = getErrdisabledIntfs(
         mode )

      if intfCauseSet:
         print( "Interfaces that will be enabled at the next timeout:" )
         t2 = createTable( ( "Interface", "Errdisable reason", "Time left(sec)" ) )
         f1 = Format( justify="left", noBreak=True, terminateRow=False )
         f2 = Format( justify="left" )
         f3 = Format( justify="left" )
         t2.formatColumns( f1, f2, f3 )
         currTime = Tac.now( )
         for intf in intfCauseSet: # pylint: disable=consider-using-dict-items
            expiryTime = intfErrdisableExpirytime[ intf ]
            if expiryTime != Tac.endOfTime:
               time = expiryTime - currTime
               if time < 0: # pylint: disable=consider-using-max-builtin
                  time = 0
               #
               # If the causes do not all fit in the column, then they are
               # split over multiple lines. Each line's cause column is packed
               # as much as possible
               #
               causeStrList = multiLineCauseFormat( 22, sorted(
                     intfCauseSet[ intf ] ) )
               causeStr = causeStrList[ 0 ]
               t2.newRow( intf, causeStr, int( time ) )
               for causeStr in causeStrList[ 1: ]:
                  t2.newRow( '', causeStr )
         print( t2.output() )

BasicCli.addShowCommandClass( ShowErrdisableRecovery )

#-------------------------------------------------------------------------------
# The "show interfaces status errdisabled" command.
#-------------------------------------------------------------------------------

class InterfaceCauseModel( Model ):
   description = Str( help='Interface description', optional=True )
   status = Str( help='Interface enabled state reason', optional=True )
   causes = List( valueType=str, help='errdisabled causes for an interface',
                  optional=True )

   def renderInterfaceCauseModel( self, shortname, t ):
      #
      # If the causes do not all fit in the column, then they are
      # split over multiple lines. Each line's cause column is packed
      # as much as possible
      #
      causeStrList = multiLineCauseFormat( 22, sorted( self.causes ) )
      causeStr = causeStrList[ 0 ]
      t.newRow( shortname, self.description, self.status, causeStr )

      for causeStr in causeStrList[ 1: ]:
         t.newRow( '', '', '', causeStr )


class ShowStatusErrdisableModel( Model ):
   interfaceStatuses = Dict( keyType=Interface, valueType=InterfaceCauseModel,
                             help="Map interfaces to errdisabled status",
                             optional=False )

   def render( self ):

      if self.interfaceStatuses:
         t = createTable( ( 'Port', 'Name', 'Status', 'Reason' ) )
         f1 = Format( justify="left", noBreak=True, terminateRow=False )
         f2 = Format( justify="left", minWidth=10, maxWidth=20 )
         f3 = Format( justify="left" )
         f4 = Format( justify="left" )
         t.formatColumns( f1, f2, f3, f4 )

         for intf in sorted( self.interfaceStatuses, key=Arnet.intfNameKey ):
            self.interfaceStatuses[ intf ].renderInterfaceCauseModel(
               IntfCli.Intf.getShortname ( intf ), t )

         print( t.output() )

def showStatusErrdisable( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   if intf is None and mod is None:
      globalIntfConfig = IntfCli.globalIntfConfig
      includeInactiveIntfs = ( globalIntfConfig and
                               globalIntfConfig.exposeInactiveLanes )
      includeUnconnIntfs = ( globalIntfConfig and 
                             globalIntfConfig.exposeUnconnectedLanes )
      intfCauseSet = getErrdisabledIntfs( 
         mode, None,
         includeInactive=includeInactiveIntfs,
         includeUnconnected=includeUnconnIntfs )[ 0 ]
   else:
      intfCauseSet = {}
      intfs = IntfCli.Intf.getAll( mode, intf, mod )
      if not intfs:
         return ShowStatusErrdisableModel( interfaceStatuses=None
                                           if mode.session.hasError() else {} )
      for i in intfs:
         causeSet = getErrdisabledIntfs( 
            mode, i.name )[ 0 ]
         intfCauseSet.update( causeSet )

   if not intfCauseSet:
      return ShowStatusErrdisableModel( interfaceStatuses=None
                                        if mode.session.hasError() else {} )

   interfaces = {}
   for intf in intfCauseSet: # pylint: disable=consider-using-dict-items
      if intf in allIntfConfigDir.intfConfig:
         intfConfig = allIntfConfigDir.intfConfig[ intf ]
      else:
         intfConfig = None

      if intfConfig is not None:
         if not intfConfig.adminEnabled:
            continue
         description = intfConfig.description
         status = intfConfig.enabledStateReason
      else:
         description = ''
         status = ''

      causes = InterfaceCauseModel( description=description,
                                    status=status,
                                    causes=list( intfCauseSet[ intf ] ) )
      interfaces [ intf ] = causes

   return ShowStatusErrdisableModel( interfaceStatuses=interfaces )


statusErrdisableKw = CliMatcher.KeywordMatcher(
   'errdisabled',
   alternates=[ 'err-disabled' ],
   helpdesc="Show errdisabled interface information" )

class ShowIntfErrdisable( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces status errdisabled'
   data = dict( status='Details on the state of interfaces',
                errdisabled=statusErrdisableKw )
   handler = showStatusErrdisable
   cliModel = ShowStatusErrdisableModel
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfErrdisable )


#-------------------------------------------------------------------------------
# The "show errdisable detect" command.
#-------------------------------------------------------------------------------
class ShowErrdisableDetect( ShowCommand.ShowCliCommandClass ):
   syntax = 'show errdisable detect'
   data = {
         'errdisable': 'Show errdisable information',
         'detect': 'Show errdisable recovery information'
   }

   @staticmethod
   def handler( mode, args ):
      t1 = createTable( ( "Errdisable Reason", "Detection Status" ) )
      f1= Format( justify="left", noBreak=True, terminateRow=False )
      f2 = Format( justify="left" )
      t1.formatColumns( f1, f2 )

      causesSeen = set()
      causeGroupDir = mode.sysdbRoot.entity[ "interface/errdisable/causegroup" ]
      for causeGroup in sorted( ErrdisableCliLib.getCauseGroups( causeGroupDir ) ):
         if causeGroupDir[ causeGroup ].hiddenCause:
            continue
         strippedCauseName = causeNameWithoutSliceId( causeGroup )
         if strippedCauseName in causesSeen:
            continue
         causesSeen.add( strippedCauseName )
         if strippedCauseName in errDisableConfig.disabled:
            status = "Disabled"
         elif strippedCauseName in errDisableConfig.enabled or \
              causeGroupDir[ causeGroup ].causeDetectDefault:
            status = "Enabled"
         else:
            status = "Disabled"
         t1.newRow( strippedCauseName, status )
      print( t1.output() )

BasicCli.addShowCommandClass( ShowErrdisableDetect )

#-------------------------------------------------------------------------------
# The "show errdisable description" command.
#-------------------------------------------------------------------------------
def showCauseHelpDict( mode, hidden ):
   d = {}
   causeGroupDir = mode.sysdbRoot.entity[ "interface/errdisable/causegroup" ]
   for name in ErrdisableCliLib.getCauseGroups( causeGroupDir ):
      cg = causeGroupDir[ name ]
      if cg.hiddenCause == hidden:
         d[ name ] = "Describe the %s cause" % name
   return d

class ShowErrdisableDescription ( ShowCommand.ShowCliCommandClass ):
   syntax = 'show errdisable description [ CAUSE1 | CAUSE2 ]'
   data = {
      'errdisable': 'Show errdisable information',
      'description': 'Describe errdisable causes',
      'CAUSE1': CliMatcher.DynamicKeywordMatcher(
         lambda mode: showCauseHelpDict( mode, False ) ),
      'CAUSE2': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher(
            lambda mode: showCauseHelpDict( mode, True ) ),
         hidden=True )
   }

   @staticmethod
   def handler ( mode, args ):  
      headers = ( "Errdisable Reason", "Description" )
      t1 = createTable( headers )
      descWidth = 45
      f1 = Format( justify="left", noBreak=True, terminateRow=False )
      f2 = Format( justify="left", wrap=True, maxWidth=descWidth )
      t1.formatColumns( f1, f2 )

      firstRow = True
      causesSeen = set()
      causeGroupDir = mode.sysdbRoot.entity[ "interface/errdisable/causegroup" ]
      for causeGroupName in sorted( 
            ErrdisableCliLib.getCauseGroups( causeGroupDir ) ):
         causeGroup = causeGroupDir[ causeGroupName ]

         # do filtering
         cause = args.get( 'CAUSE1' ) or args.get( 'CAUSE2' )
         if ( cause == causeGroupName or
               ( cause is None and not causeGroup.hiddenCause ) ):
            if firstRow:
               firstRow = False
            else:
               t1.newRow( '', '' )
            strippedCauseName = causeNameWithoutSliceId( causeGroupName )
            if strippedCauseName in causesSeen:
               continue
            causesSeen.add( strippedCauseName )
            if causeGroup.description:
               t1.newRow( strippedCauseName, textwrap.fill( causeGroup.description,
                                                            descWidth ) )
            else:
               t1.newRow( strippedCauseName, 'No description available.' )

      print( t1.output() )

BasicCli.addShowCommandClass( ShowErrdisableDescription )

#-------------------------------------------------------------------------------
# The "errdisable test interface <intf>" command.
#-------------------------------------------------------------------------------
class ErrdisableTestInterface( CliCommand.CliCommandClass ):
   syntax = 'errdisable test interface INTERFACE_NAME'
   data = {
      'errdisable': 'Errdisable',
      'test': 'Errdisable using test cause',
      'interface': 'Interface to test',
      'INTERFACE_NAME': IntfCli.Intf.matcherWOSubIntf
   }
   hidden = True
   
   @staticmethod
   def handler( mode, args ):    
      causeDir = mode.sysdbRoot[ 'interface' ][ 'errdisable' ][ 'cause' ]
      intf = args[ 'INTERFACE_NAME' ]
      if 'test' in causeDir:
         causeStatus = causeDir[ 'test' ]
         causeStatus.newIntfStatus( intf.name, Tac.now( ) )

BasicCli.EnableMode.addCommandClass( ErrdisableTestInterface )

#-------------------------------------------------------------------------------
# Mount the needed sysdb state
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global errDisableConfig, allIntfConfigDir
   errDisableConfig = ConfigMount.mount( entityManager,
                                         "interface/errdisable/config",
                                         "Errdisable::Config", "w" )
   allIntfConfigDir = LazyMount.mount( entityManager, "interface/config/all",
                                       "Interface::AllIntfConfigDir", "r" )
   #
   # 'interface/errdisable/cause' directory is mounted with immediate
   # option 'i' to mount the cause dir, and its subdirectories in one step.
   # The subdirectories are CauseStatus entities which themselves are
   # mount points, and so they are not mounted automatically without
   # the immediate option
   # This needs to happen in a single step, because the CauseStatus entities
   # are needed to find out the cause names, which are needed to install the
   # Cli commands.
   #
   mg1 = entityManager.mountGroup( )
   _ = mg1.mount( "interface/errdisable/cause", "Tac::Dir", "ri" )
   _ = mg1.mount( "interface/errdisable/causegroup", "Tac::Dir", "ri" )

   def _finish1( ):
      pass

   mg1.close( _finish1 )
