# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Int, Model, Str, Bool, List, Submodel
from TableOutput import createTable, Format
from ArnetModel import IpGenericPrefix, MacAddress
import socket
import Tac

IpProtocolNumberType = Tac.Type( "Arnet::IpProtocolNumber" )

class HandlerCounter( Model ):
   '''Model for routing handler counter.'''
   ignored = Int( help="Number of ignored packets" )
   routed = Int( help="Number of routed packets" )

class HandlerCounters( Model ):
   '''Model for show platform etba counters'''
   ebraCounters = Dict( keyType=str,
                        valueType=HandlerCounter, optional=True,
                        help="Maps EbraTestBridge routing handler names "
                             "to respective counters" )
   ebraIgnored = Int( help="Number of packets that were ignored by "
                           "all EbraTestBridge routing handlers" )
   ebraRouted = Int( help="Number of packets that were routed by at least "
                          "one EbraTestBridge routing handler" )
   fwdIntfCounters = Dict( keyType=str,
                           valueType=HandlerCounter, optional=True,
                           help="Maps FwdIntfDevice routing handler names "
                                "to respective counters" )
   fwdIntfIgnored = Int( help="Number of packets that were ignored by "
                              "all FwdIntfDevice routing handlers" )
   fwdIntfRouted = Int( help="Number of packets that were routed by "
                             "a FwdIntfDevice routing handler" )

   def render( self ):
      ignoredMsg = 'Ignored by all handlers: '
      fwdIntfRoutingHandlerStr = 'FwdIntfDevice:RoutingHandler' 
      ebraRoutingHandlerStr = 'EbraTestBridge:RoutingHandler' 

      formats = {}
      formats[ fwdIntfRoutingHandlerStr ] = Format( justify='left' )
      formats[ ebraRoutingHandlerStr ] = Format( justify='left' )
      formats[ 'Ignored' ] = Format( justify='left' )
      formats[ 'Routed' ] = Format( justify='left' )
      for columnFormat in formats.values():
         columnFormat.noPadLeftIs( True )
         columnFormat.padLimitIs( True )

      if self.fwdIntfCounters:
         fwdIntfCounterTable = createTable( ( fwdIntfRoutingHandlerStr,
                                              'Ignored', 'Routed' ),
                                            tableWidth=100 )
         fwdIntfCounterTable.formatColumns( formats[ fwdIntfRoutingHandlerStr ],
                                            formats[ 'Ignored' ],
                                            formats[ 'Routed' ], )
         for name, counter in sorted( self.fwdIntfCounters.items() ):
            fwdIntfCounterTable.newRow( name, counter.ignored, counter.routed )
         print( fwdIntfCounterTable.output() )
      else:
         print( fwdIntfRoutingHandlerStr + '\n' +
               '-' * len( fwdIntfRoutingHandlerStr ) )

      print( ignoredMsg + str( self.fwdIntfIgnored ) )
      print( 'Routed by a handler: ' + str( self.fwdIntfRouted ) + '\n\n' )

      if self.ebraCounters:
         ebraCounterTable = createTable( ( ebraRoutingHandlerStr,
                                           'Ignored', 'Routed' ),
                                         tableWidth=100 )
         ebraCounterTable.formatColumns( formats[ ebraRoutingHandlerStr ],
                                         formats[ 'Ignored' ],
                                         formats[ 'Routed' ], )
         for name, counter in sorted( self.ebraCounters.items() ):
            ebraCounterTable.newRow( name, counter.ignored, counter.routed )
         print( ebraCounterTable.output() )
      else:
         print( ebraRoutingHandlerStr + '\n' + '-' * len( ebraRoutingHandlerStr ) )

      print( ignoredMsg + str( self.ebraIgnored ) )
      print( 'Routed by at least one handler: ' + str( self.ebraRouted ) )

class NamedCounterSubGroup( Model ):
   counters = Dict( keyType=str,
                    valueType=int,
                    help="Counter values indexed by counter name" )
   __public__ = False

class NamedCounterGroup( Model ):
   subgroups = Dict( keyType=str,
                     valueType=NamedCounterSubGroup,
                     help="Counter subgroups indexed by sub-group name" )
   __public__ = False

   def render( self ):
      for subGroupName, counters in sorted( self.subgroups.items() ):
         for counterName, value in sorted( counters.counters.items() ):
            # pylint: disable-next=consider-using-f-string
            print( "%s: %s: %d" % ( subGroupName, counterName, value ) )

class ArfaCounterGroups( Model ):
   '''Model for Arfa counters'''
   groups = Dict( keyType=str,
                  valueType=NamedCounterGroup,
                  help="Counter groups indexed by group name" )
   __public__ = False

   def render( self ):
      for groupName, group in sorted( self.groups.items() ):
         print( "%s:" % groupName ) # pylint: disable=consider-using-f-string
         group.render()
         print()

class PamCollectionEntry( Model ):
   inDrops = Int( help="The number of drops on the rx queue" )
   inErrors = Int( help="The number of rx failures" )
   inOctets = Int( help="The number of bytes received" )
   inPkts = Int( help="The number of packets received" )
   outDrops = Int( help="The number of tx failures because the socket buf was full" )
   outErrors = Int( help="The number of tx failures for any other reason" )
   outOctets = Int( help="The number of bytes sent" )
   outPkts = Int( help="The number of packets sent" )
   lastOutError = Str( help="The errno returned while attempting to send the packet",
                      optional=True )

class PamCollection( Model ):
   pams = Dict( keyType=str,
                valueType=PamCollectionEntry,
                help="PAMs keyed by their name" )

class PipelineStep( Model ):
   name = Str( help="Name of step" )
   canTerminatePipeline = Bool( help="Can cause pipeline to terminate" )
   __public__ = False

class GenericPipeline( Model ):
   steps = Dict( keyType=int,
                 valueType=PipelineStep,
                 help="Steps keyed by their priority" )
   __public__ = False

   def render( self ):
      for sPrio, step in sorted( self.steps.items() ):
         canTerm = " (can terminate pipeline)" if step.canTerminatePipeline else ""
         print( f"{sPrio}: {step.name}{canTerm}" )

class AllGenericPipelines( Model ):
   pipelines = Dict( keyType=str,
                     valueType=GenericPipeline,
                     help="Pipelines keyed by their name" )
   __public__ = False

   def render( self ):
      for pName, pipeline in sorted( self.pipelines.items() ):
         print( "Pipeline:", pName )
         pipeline.render()
         print()

class GenericLookup( Model ):
   id = Str( help="Id of lookup" )
   key = Str( help="Key(EnumName) of lookup" )
   handler = Str( help="Handler of lookup" )

class GenericLookups( Model ):
   lookups = List( help="Lookups specified by this feature",
         valueType=GenericLookup )

class LookupsTable( Model ):
   lookupsTable = Dict( keyType=str,
                     valueType=GenericLookups,
                     help="LookupsTable keyed by their name" )

class AllArfaPlugins( Model ):
   plugins = List( help="Plugins used by TFA", valueType=str )
   __public__ = False

   def render( self ):
      if self.plugins:
         print( "Plugins:" )
         for plugin in sorted( self.plugins ):
            print( plugin )

class DemuxListenTuple( Model ):
   ipPrefix = IpGenericPrefix( help="IP Prefix" )
   protocol = Int( help="IP protocol" )
   port = Int( help="TCP or UDP port" )
   numLookups = Int( help="Number of times this tuple was used for a lookup" )
   __public__ = False

def _genTupleKey( toople ):
   return ( toople[ "ipPrefix" ], toople[ "protocol" ], toople[ "port" ] )

class DemuxListenFeature( Model ):
   endpoints = List( help="Endpoints specified by this feature",
                  valueType=DemuxListenTuple )
   __public__ = False

class DemuxEndpoints( Model ):
   features = Dict( keyType=str, valueType=DemuxListenFeature,
                    help="Features using the demux infrastructure" )
   __public__ = False

   def render( self ):
      headings = ( "Feature", "IP Prefix", "Protocol", "Port", "NumLookups" )
      table = createTable( headings )
      justifyRight = Format( justify='right' )
      justifyLeft = Format( justify='left' )
      table.formatColumns(
         justifyLeft, justifyLeft, justifyRight, justifyRight, justifyRight )

      for featureName, feature in sorted( self.features.items() ):
         for toople in sorted( feature.endpoints, key=_genTupleKey ):
            # Try to find the named port if it exists
            portFmt = "%s"
            try:
               portName = socket.getservbyport( toople.port )
               portFmt = portName + " (%s)"
            except OSError:
               pass

            protoFmt = "%s"
            try:
               protoName = Tac.enumName( IpProtocolNumberType, toople.protocol )
               # strip the ipProto
               protoFmt = protoName[ len( "ipProto" ) : ] + " (%s)"
            except AttributeError as e:
               # pylint: disable-next=unsupported-membership-test
               if "with value" not in e:
                  raise

            table.newRow(
               featureName, toople.ipPrefix, protoFmt % toople.protocol,
               portFmt % toople.port, toople.numLookups )

      print( table.output() )

class AgingTableFids( Model ):
   lastSeenMacs = Dict( keyType=MacAddress, valueType=float,
                help="The time that traffic was last seen from each MAC address" )
   __public__ = False

   def render( self ):
      headings = ( "MAC Address", "Age (seconds)" )
      table = createTable( headings )
      justifyRight = Format( justify='right' )
      justifyLeft = Format( justify='left' )
      table.formatColumns( justifyLeft, justifyRight )
      utcNow = Tac.utcNow()
      for mac, lastSeen in sorted( self.lastSeenMacs.items() ):
         table.newRow( mac, f"{utcNow - lastSeen:.02f}" )
      print( table.output() )

class AgingTables( Model ):
   fids = Dict( keyType=int, valueType=AgingTableFids,
                help="Per-FID MAC aging table" )
   __public__ = False

   def render( self ):
      for fid, fidTable in sorted( self.fids.items() ):
         print( "FID %d:" % fid ) # pylint: disable=consider-using-f-string
         fidTable.render()

class PktCounter( Model ):
   __public__ = False
   packets = Int( help="Number of packets counted in this bucket" )
   octets = Int( help="Number of octets counted in this bucket" )

class PktCounters( Model ):
   __public__ = False
   keyType = Str( help="Description of the name used in the counters dict" )
   counters = Dict( keyType=str, valueType=PktCounter,
                    help="A mapping of counter description to counter values" )

   def render( self ):
      headings = ( self.keyType, "Packets", "Bytes" )
      table = createTable( headings )
      justifyRight = Format( justify='right' )
      justifyLeft = Format( justify='left' )
      table.formatColumns( justifyLeft, justifyRight, justifyRight )
      for counterName, counter in sorted( self.counters.items() ):
         table.newRow( counterName, counter.packets, counter.octets )
      print( table.output() )

class PktCounterTables( Model ):
   __public__ = False
   features = Dict( keyType=str, valueType=PktCounters,
                    help="A mapping of feature to counters associated with "
                         "that feature" )

   def render( self ):
      for tableName, counterTable in sorted( self.features.items() ):
         print( "Feature: %s" % tableName ) # pylint: disable=consider-using-f-string
         counterTable.render()

class ArfaRateCounter( Model ):
   __public__ = False
   packets = Int( help="Number of packets counted in this bucket" )
   octets = Int( help="Number of octets counted in this bucket" )
   packetsRate = Int( help="Number of packets per second" )
   bitsRate = Int( help="Number of bits per second" )

class ArfaRateCounterDirections( Model ):
   __public__ = False
   input = Submodel( valueType=ArfaRateCounter, optional=True,
                     help="Input counters" )
   output = Submodel( valueType=ArfaRateCounter, optional=True,
                      help="Output counters" )

class ArfaRateCounterCollection( Model ):
   __public__ = False
   counters = Dict( keyType=str, valueType=ArfaRateCounterDirections,
                    help="Collections keyed by counter's name" )
