# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import
# pylint: disable=protected-access
# pylint: disable=consider-using-in
# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements the Ethernet interface type, and associated commands.
# In particular, it provides:
# -  the EthIntf class
# -  the "mac-address <mac_addr>" command
# -  the "no mac-address [<mac_addr>]" command.
# -  the "[no] logging event congestion-drops ( interface config level )
# -  the "[no] logging event congestion-drops interval ( global config level )
#-------------------------------------------------------------------------------
'''Commands supported on an Ethernet-like interface'''

import math
import os
import re
from collections import OrderedDict, namedtuple
from functools import partial

import Ark
import BasicCli
import BasicCliUtil
import CliCommand
import CliExtensions
from CliGlobal import CliGlobal
import CliMatcher
from CliMode.EthIntf import InterfaceDefaultsEthernetMode
import CliParser
import CliPlugin.EthIntfModel as EthIntfModel
from CliPlugin.EthIntfModel import EthInterfaceCounters, EthInterfaceStatistics
import CliPlugin.IntfCli as IntfCli
from CliPlugin.IntfCli import LoggingEventGlobalCmd, LoggingEventIntfCmd
from CliPlugin.IntfModel import (
   ErrorCounters,
   HalfDuplexErrorCounters,
   InterfaceCountersRate
)
import CliPlugin.IntfRangeCli as IntfRangeCli
import CliPlugin.FruCli as FruCli
import CliPlugin.MacAddr as MacAddr
import CliPlugin.ModuleIntfCli as ModuleIntfCli
import CliPlugin.PhysicalIntfRule as PhysicalIntfRule
import CliPlugin.SpeedGroupCli as SpeedGroupCli
from CliPlugin.VirtualIntfRule import IntfMatcher
import CliPlugin.XcvrEthIntfDir as XcvrEthIntfDir
import CliPlugin.XcvrCapabilitiesModel as XcvrCapabilitiesModel
import Cell
import ConfigMount
import Ethernet
import EthIntfLib
import Intf.IntfRange as IntfRange
from IntfRangePlugin.EthIntf import (
   EthPhyAutoIntfType,
   MgmtAutoIntfType,
   MgmtPhyAutoIntfType,
   UnconnectedEthPhyAutoIntfType
)
import LazyMount
import QuickTrace
import SharedMem
import ShowCommand
import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "EthIntfCli" )
t0 = __defaultTraceHandle__.trace0
qv = QuickTrace.Var
qt8 = QuickTrace.trace8

ZeroEthIntfCounter = TacLazyType( 'Interface::ZeroEthIntfCounter' )

gv = CliGlobal( l1ProductConfig=None, l2MtuValidIntfType=None )

# l2MtuValidIntfType tracks all interface types that support L2 mtu.
gv.l2MtuValidIntfType = () # It's either this or pylint complains (no-member).

ethPhyIntfConfigSliceDir = None
ethIntfStatusDir = None
ethIntfGenerationSliceDir = None
ethIntfModeSliceDir = None
entmib = None
intfHwStatus = None
ethPhyIntfStatusDir = None
ethPhyIntfDefaultConfigDir = None
ethIntfSliceDefaultConfigDir = None
ethPhyIntfConfigReqDir = None
ethPhyDefaultCapsSliceDir = None
ethIntfCounterWriterStatusDir = None
checkpointCounterDir = None
ethIntfCounterReaderSm = None
ethPhyIntfCounterDir_ = None
intfXcvrStatusDir = None
internalSissSliceDir = None
entityMib = None
ethFlowControlGlobalConfig = None
linkFlapDampingConfig = None
linkFlapDampingIntfStatusDir = None
resourceMgrStatusSliceDir = None
l1CapsSlotDir = None
defaultL1CapsSlotDir = None

systemName = None
entityManager = None
bridgingHwCapabilities = None

LOOPBACK_MAC = 1
LOOPBACK_PHY = 2
LOOPBACK_PHYREMOTE = 3

# Annotations for interfaces in show vlan configured port, when bridging
# config is overridden.
# NOTE: If you change these chars, you must change them in XxxCliTests.py
vlanDataLinkAnnotations = ( '*', '^', '+', '$' )
# Annotations used for explanation of quietDataLink
vlanQuietDataLinkAnnotations = ( '#', '%' )
# Annotations used for explanation of unauthorized
vlanUnauthorizedAnnotations = ( '~', '-' )
# Annotations used for explanation of recirculation
vlanRecirculationAnnotations = ( '!' )
# Annotations used for explanation of layer1 patch
layer1PatchAnnotations = ( '&' )
# Annotations used for explanation of link qualification
linkQualificationAnnotations = ( '@', '@@' )

# A canPromptForAbortHook accepts three arguments: the mode, and the intfList
# which contains the single interface, or the interfaces in the interface range,
# and the requested linkMode.
# If any of the hooks return true, the request is ignored. As these functions
# can issue prompts for user input we only call them in interactive mode.
# This hook can be called for signle interface mode or for interface range mode.
canPromptForAbortHook = CliExtensions.CliHook()

AutonegMode = TacLazyType( "Interface::AutonegMode" )
EthDuplex = TacLazyType( "Interface::EthDuplex" )
EthIntfId = TacLazyType( 'Arnet::EthIntfId' )
EthLaneCount = TacLazyType( "Interface::EthLaneCount" )
EthLinkMode = TacLazyType( 'Interface::EthLinkMode' )
EthLinkModeSet = TacLazyType( 'Interface::EthLinkModeSet' )
EthSpeed = TacLazyType( "Interface::EthSpeed" )
EthSpeedApi = TacLazyType( "Interface::EthSpeedApi" )
EthTypesApi = TacLazyType( "Interface::EthTypesApi" )
LoopbackMode = TacLazyType( 'Interface::LoopbackMode' )
L1MountConstants = TacLazyType( 'L1::MountConstants' )
L1PolicyVersion = TacLazyType( 'L1::PolicyVersion' )
L1ProductConfig = TacLazyType( 'L1::ProductConfig' )
MgmtIntfId = TacLazyType( 'Arnet::MgmtIntfId' )
XcvrMode = TacLazyType( 'Interface::XcvrMode' )

def ethPhyIntfCounterDir():
   return ethPhyIntfCounterDir_

# Method to indicate an interface type supports L2 mtu.
# intfType is a class that defines a particular interface class.
def registerL2MtuValidIntfType( intfType ):
   gv.l2MtuValidIntfType += ( intfType, )

def isVEos():
   return entityMib.root and entityMib.root.modelName == 'vEOS'

def isModular():
   # Breadth tests related to mgmtactiveintf setup entmib to a give notion of
   # modular system. So we use modularSystemGuard for modular system check,
   # although celltype alone will be sufficient on a real dut.
   return ( ( not FruCli.modularSystemGuard( None, None ) ) or
            ( Cell.cellType() == "supervisor" ) )

# ----------------------------------------------------------------------------
# Utility functions used by "showInterfacesCapabilities()"
# ----------------------------------------------------------------------------
def _xcvrSlot( intfName, card, port ):
   """
    Find the xcvrSlot on the given card for this interface.  We have to
    match the tag and label rather than trying to just index by the port
    number, because management and regular ethernet ports all live together
    in the xcvrSlot collection of a fixed config system.
   """
   m = re.match( r'^([^\d]+)', intfName )
   if not m:
      return None
   tag = m.group( 1 )
   for slot in card.xcvrSlot.values():
      for p in slot.port.values():
         if ( p.tag == tag ) and ( p.label == port ):
            return slot
   return None

def modelStr( intfName ):
   """
    Return a model string appropriate for the 'show interface capabilities'
    command.  This is a bit weird.  It returns the model number of the
    card or fixed config system where the port sits, concatenated with
    the transceiver slot type if there is one for the port.  To get this
    information, we rummage around in the entity mib.
   """
   ( mod, port ) = IntfCli.moduleAndPort( intfName )
   if mod is None and port is None:
      return "Unknown"
   if not entmib.root:
      # On vEOS, we are not populating the entity mib at all. This
      # needs further rationalization - BUG9410
      return "Unknown"
   elif entmib.fixedSystem:
      # Fixed system.
      card = entmib.root
   else:
      # Modular system.
      cardSlot = entmib.root.cardSlot.get( mod, None )
      if not cardSlot:
         return "Unknown"
      card = cardSlot.card
   if not card:
      return "Unknown"
   model = card.modelName
   xcvrSlot = _xcvrSlot( intfName, card, port )
   if not xcvrSlot:
      return model
   xcvrSlotType = xcvrSlot.slotType
   if not xcvrSlotType:
      return model
   return model + "-" + xcvrSlotType.upper()

NON_EXISTENT_INTF_WARN = "Interface does not exist. The configuration will " \
   "not take effect until the module is inserted."

class L1Intf( IntfCli.Intf ):
   """
   A semi-concrete child of the Intf class for Ethernet / LAG interfaces.
   This class contains PHY related helpers (for speed, autoneg, etc.) that are
   used to populate some CLI outputs.
   Unimplemented abstract method warning; this is intended as a semi-concrete child
   """
   def __init__( self, name, mode ):
      t0( "L1Intf::__init__", name )
      super().__init__( name, mode )
      self.intfXcvrStatusDir = intfXcvrStatusDir
      self.ethIntfStatuses = ethIntfStatusDir.intfStatus

   # ----------------------------------------------------------------------------
   # Interface Lifespan Controls
   #
   # L1 Interface configuration entities are special in that they can either be
   # created by FRU or the CLI through the use of a "request" mechanism which a
   # special "configuration creator" listens to and thus handles the lifespan of
   # the interface.
   #
   # The one exception to this is when the CLI is operating on a configuration
   # session where there is no such "creator" and thus the CLI has to create it's
   # own entites.
   # ----------------------------------------------------------------------------

   def noInterface( self ):
      """
      L1 interfaces can only be destroyed if the underlying physical hardware has
      been removed.
      """
      if not self.lookupPhysical():
         self.destroy()
         return

      self.mode_.addWarning(
            f"Removal of physical interfaces is not permitted: {self.name}" )
      self.setDefault()

   def createPhysical( self, startupConfig=False ):
      t0( 'Creating physical L1 Interface:', self.name )

      if not self.lookupPhysical() and not startupConfig:
         self.mode_.addWarning( NON_EXISTENT_INTF_WARN )

      # since configuration entities can either be created by the CLI or by FRU,
      # neither contexts is explicitly allowed to directly create the entities.
      # Instead, they both request the entity be created and a separate creator
      # arbitrates the requests, creates the interface configuration entity, and
      # manages it's lifespan.
      self.createInterfaceConfigurationRequest()

      # When in a configuration session context the CLI modifies a copy of the Sysdb
      # root which the in interface configuration creator does not know about. As
      # such, the CLI has to bypass it and create the interface configuration entity
      # directly.
      #
      # During the commit of the session, a special entity copy handler will
      # reconcile the entities in the configuration session with the ones in Sysdb.
      if self.mode_.session.inConfigSession():
         self.createInterfaceConfiguration()
         return

      # Wait for the interface configuration creator to create the interface
      # configuration entities.
      Tac.waitFor( lambda: self.config() is not None,
                   sleep=not Tac.activityManager.inExecTime.isZero,
                   timeout=60.0, warnAfter=None )

   def destroyPhysical( self ):
      """
      since configuration entities can either be created by the CLI or by FRU,
      neither contexts is explicitly allowed to directly delete the entities.
      Instead, they both can request that the interface configuration be created
      and the creator will enforce that. Deletions thus can only happen if there
      are no more requestors for the interface configuration entity.
      """
      t0( 'Deleting physical L1 Interface:', self.name )
      self.deleteInterfaceConfigurationRequest()

      if ( self.mode_.session.entityManager.isLocalEm or
           self.mode_.session.inConfigSession() ):
         self.deleteInterfaceConfiguration()
         return

      # The CLI only allows this method to be called once it is sure that FRU is not
      # keeping the entity alive. As such this implementation can safely wait until
      # the configuration entity has been deleted.
      Tac.waitFor( lambda: self.config() is None,
                   sleep=not Tac.activityManager.inExecTime.isZero,
                   timeout=60.0, warnAfter=None )

   def createInterfaceConfigurationRequest( self ):
      '''Creates an interface configuration entity creation request for a particular
      interface.
      '''
      raise NotImplementedError()

   def deleteInterfaceConfigurationRequest( self ):
      '''Deletes an interface configuration entity creation request for a particular
      interface.
      '''
      raise NotImplementedError()

   def createInterfaceConfiguration( self ):
      '''Creates an interface configuration entity.

      This method should only be called from configurations session contexts where
      the standard interface configuration creation mechanisms are unavailable.

      Returns:
         The create interface configuration entity
      '''
      raise NotImplementedError()

   def deleteInterfaceConfiguration( self ):
      '''Deletes an interface configuration entity.

      This method should only be called from configurations session contexts where
      the standard interface lifecycle management mechanisms are unavailable.
      '''
      raise NotImplementedError()

   def ethPhyDefaultCaps( self ):
      """
      Returns the default capabilities (link mode, autoneg, fec, etc.)
      for this interface.
      """
      raise NotImplementedError()

   def ethIntfMode( self ):
      """
      Returns the EthIntfMode object (speed, duplex, FEC) for this interface.
      """
      raise NotImplementedError()

   @Tac.memoize
   def intfXcvrStatus( self ):
      if self.intfXcvrStatusDir:
         return self.intfXcvrStatusDir.get( self.name )
      return None

   def intfXcvrPresent( self ) -> bool:
      intfXcvrStatus = self.intfXcvrStatus()
      return ( intfXcvrStatus is not None and
               intfXcvrStatus.xcvrPresence == 'xcvrPresent' )

   #----------------------------------------------------------------------------
   # Return the EntityMibPort for the interface
   #----------------------------------------------------------------------------
   @Tac.memoize
   def entityMibPort( self ):
      if self.slotId() == 'FixedSystem' and entmib and entmib.fixedSystem:
         portDir = entmib.fixedSystem.port
      else:
         # BUG862836: Find a way to map slotId here to entityMib slot idx.
         # We can return NONE, since Modular systems do not have extFabric
         return None
      for port in portDir:
         if portDir[ port ].intfId == self.name:
            return portDir[ port ]
      return None

   def internal( self ):
      portMib = self.entityMibPort()
      if "Fabric" in self.name:
         return not ( portMib and portMib.role == 'FrontPanelFabric' )
      return False

   def ethIntfModeDir( self ):
      if Cell.cellType() == "fixed":
         slotId = "FixedSystem"
      else:
         slotId = self.sliceName()
      ethIntfModeDirs = ethIntfModeSliceDir.get( slotId )
      if ethIntfModeDirs:
         ethIntfGeneration = ethIntfGenerationSliceDir.get( slotId )
         for ethIntfModeDir in ethIntfModeDirs.values():
            if ethIntfModeDir.generation == ethIntfGeneration.generation:
               ethIntfMode = ethIntfModeDir.ethIntfMode.get( self.name )
               if ethIntfMode:
                  return ethIntfModeDir
      return None

   def resourceManagerStatus( self ):
      resourceManagerStatusList = []
      for slot, slotDir in resourceMgrStatusSliceDir.items():
         for resourceManagerStatus in slotDir.values():
            intfState = resourceManagerStatus.intfStateDir.intfState.get( self.name )
            if intfState:
               resourceManagerStatusList.append( ( slot, resourceManagerStatus ) )

      # On a Glacier like system, there may be more than 1 resourceManagerStatus for
      # one interface. For example, one resourceManagerStatus for the Linecard where
      # the interface is on, and one resourceManagerStatus for the whole Switchcard.
      # As IPM is running inside the scope of forwarding agent, we return the one
      # for Switchcard if there are multiple resourceManagerStatus.
      if len( resourceManagerStatusList ) == 1:
         return resourceManagerStatusList[ 0 ][ 1 ]
      elif len( resourceManagerStatusList ) == 2:
         for slot, resourceManagerStatus in resourceManagerStatusList:
            if slot.startswith( "Switchcard" ):
               return resourceManagerStatus

      return None

   @Tac.memoize
   def sliceName( self, fixedSystemCardSlotId=False ):
      '''Retrieves the card slot ID associated with the interface.

      Args:
         fixedSystemCardSlotId ( bool ): If set then the "FixedSystem" string will
                                         be used instead of the cell ID on fixed
                                         systems.
      '''
      return EthIntfLib.sliceName( self.name, fixedSystemCardSlotId )

   @Tac.memoize
   def subdomain( self ):
      return self.status().subdomain

   def l1PolicyVersion( self ):
      '''Returns the L1 configuration policy used to manage the interface.

      The policy version affects how the system will treat manual user configurations
      and the output of some CLI commands.


      Returns:
         The L1::PolicyVersion associated with an interface or a default constructed
         value if L1 policy agent is not in use.
      '''

      # Some special interfaces go through the L1 CLI and do not have a card slot ID
      # associated with them. L1 Policy agent does not manage these interface so a
      # default constructed policy version is returned.
      #
      # TODO BUG963901: Turn these in to proper interface ID checks once the
      #                 interface ID definitions move to EthIntf.
      if ( ( 'PeerEthernet' in self.name ) or
           ( 'PeerUnconnectedEthernet' in self.name ) or
           ( 'PeerSwitch' in self.name ) ):
         return L1PolicyVersion()


      cardSlotId = self.sliceName( fixedSystemCardSlotId=True )
      if not cardSlotId:
         return L1PolicyVersion()

      l1ProductCardSlotConfig = gv.l1ProductConfig.cardSlotConfig.get( cardSlotId )
      if l1ProductCardSlotConfig is None:
         return L1PolicyVersion()

      l1ProductSubdomainConfig = l1ProductCardSlotConfig.subdomainConfig.get(
         self.subdomain() )
      if l1ProductSubdomainConfig is None:
         return L1PolicyVersion()

      return l1ProductSubdomainConfig.policyVersion

   def speedGroupStatus( self ):
      for linecard in SpeedGroupCli.speedGroupStatusSliceDir.values():
         for speedGroupStatusDir in linecard.values():
            group = speedGroupStatusDir.intfSpeedGroup.get( self.name )
            if group:
               return speedGroupStatusDir.group.get( group )
      return None

   def setAdvertisedModes( self, advertisedModes=None ):
      """
      Sets the modes that are advertised when autonegotiation is turnd on.
      """
      if advertisedModes is None:
         advertisedModes = EthLinkModeSet() # Need to match the TACC type
      self.config().doSetAdvertisedModes( advertisedModes )

   def setLinkMode( self, linkMode, advertisedModes=None,
                    legacySpeedCmdSyntax=False ):
      """
      Sets the linkMode (speed and duplex) of the interface to a specified value
      or the default value.

      When the linkMode is set to auto-negotiate, advertisedModes can be used to
      limit the modes that are advertised to the link partner.
      """
      if linkMode != 'linkModeAutoneg' and linkMode != 'linkModeAuto40GbpsFull':
         # advertisedModes is only valid when the port is set to autonegotiate
         assert advertisedModes is None

      if not self.isLinkModeSupported( linkMode, advertisedModes ):
         return

      self.setAdvertisedModes( advertisedModes )
      self.config().linkModeLocal = linkMode
      self.config().legacySpeedCmdSyntax = legacySpeedCmdSyntax

   def getLinkMode( self ):
      '''Retrieves the user configured link mode of the interface.'''
      config = self.config()
      if not config:
         return EthLinkMode.linkModeUnknown

      return config.linkModeLocal

   def isAdvertisedModesSupported( self, advertisedModes ):
      """
      When linkMode is set to auto-negotiate, verify support for the desired
      speeds and duplex that are advertised to the link partner.
      """
      if advertisedModes is not None:
         # confirm that the advertised modes are supported for autoneg
         return self.status().advertisedModesSupported( advertisedModes )
      if self.status().phyAutonegCapabilities.mode == 'anegModeClause73':
         # clause 73 needs a speed to advertise
         return False
      # if advertisedModes is not explicitly configured then we advertise what we can
      return True

   def maybeAllowUnsupportedConfig( self, bypass, mode, msg ):
      """
      Helper function that asks for confirmation to allow an unsupported config
      """
      if bypass:
         mode.addWarning( msg )
         prompt = "Do you wish to proceed with this command? [y/N]"
         if BasicCliUtil.confirm( mode, prompt, answerForReturn=False ):
            return True
      else:
         mode.addError( msg )
      return False

   def linkModeUnsupportedErrorMsg( self ):
      return ( 'Speed and duplex settings are not compatible '
               'with transceiver for interface %s.' % self.name )

   def isLinkModeSupported( self, desiredLinkMode, advertisedModes=None ):
      """
      Check if the desired linkMode is supported for the interface.
      """
      supported = True
      if desiredLinkMode != 'linkModeUnknown':
         if self.lookupPhysical():
            if not self.intfXcvrPresent():
               # we optimistically claim support after a warning label
               self.mode_.addWarning(
                  'Transceiver for interface %s is not present. '
                  'Cannot verify compatibility of speed and duplex settings.' %
                  self.shortname_ )
            else:
               errorMsg = ""
               if not self.status().linkModeSupported[ desiredLinkMode ]:
                  # desired linkMode is not supported
                  errorMsg = self.linkModeUnsupportedErrorMsg()
                  supported = False
               elif desiredLinkMode == 'linkModeAutoneg':
                  supported = self.isAdvertisedModesSupported( advertisedModes )
                  if not supported:
                     errorMsg = ( 'Cannot advertise desired speeds and '
                                  'duplex settings for interface %s.'
                                   % self.name )
               if not supported:
                  supported = self.maybeAllowUnsupportedConfig(
                     ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.speed,
                     self.mode_, errorMsg )
         else:
            # we optimistically claim support after a warning label
            self.mode_.addWarning(
               'Transceiver is not present. '
               'Cannot verify compatibility of speed and duplex settings.' )

      return supported

   speedStringsFormat1 = { 'speed10Mbps': '10Mb/s',
                           'speed100Mbps': '100Mb/s',
                           'speed1Gbps': '1Gb/s',
                           'speed2p5Gbps': '2.5Gb/s',
                           'speed5Gbps': '5Gb/s',
                           'speed10Gbps': '10Gb/s',
                           'speed25Gbps': '25Gb/s',
                           'speed40Gbps': '40Gb/s',
                           'speed50Gbps': '50Gb/s',
                           'speed100Gbps': '100Gb/s',
                           'speed200Gbps': '200Gb/s',
                           'speed400Gbps': '400Gb/s',
                           'speed800Gbps': '800Gb/s',
                           'speedUnknown': 'Unconfigured' }

   speedStringsFormat2 = { 'speed10Mbps': '10M',
                           'speed100Mbps': '100M',
                           'speed1Gbps': '1G',
                           'speed2p5Gbps': '2.5G',
                           'speed5Gbps': '5G',
                           'speed10Gbps': '10G',
                           'speed25Gbps': '25G',
                           'speed40Gbps': '40G',
                           'speed50Gbps': '50G',
                           'speed100Gbps': '100G',
                           'speed200Gbps': '200G',
                           'speed400Gbps': '400G',
                           'speed800Gbps': '800G',
                           'speedUnknown': 'unconfigured' }

   speedStringsFormat3 = { 'speed10Mbps': 'a-10M',
                           'speed100Mbps': 'a-100M',
                           'speed1Gbps': 'a-1G',
                           'speed2p5Gbps': 'a-2.5G',
                           'speed5Gbps': 'a-5G',
                           'speed10Gbps': 'a-10G',
                           'speed25Gbps': 'a-25G',
                           'speed40Gbps': 'a-40G',
                           'speed50Gbps': 'a-50G',
                           'speed100Gbps': 'a-100G',
                           'speedUnknown': 'unconfigured' }

   speedStrings = { 'Format1': speedStringsFormat1,
                    'Format2': speedStringsFormat2,
                    'Format3': speedStringsFormat3 }

   autonegSpeedStrings = { 'Format1': 'Auto-speed',
                           'Format2': 'auto',
                           'Format3': 'auto' }

   @staticmethod
   def _speedStr( value, stringFormat ):
      """
      Helper function that returns the string representation of the interface's
      speed in one of a number of specifiable formats.
      """
      s = L1Intf.speedStrings.get( stringFormat )
      return s[ value ]

   def operSpeed( self, stringFormat ):
      speed = self.status().speed
      autonegActive = self.autonegActive()
      if autonegActive and speed == EthSpeed.speedUnknown:
         return self.autonegSpeedStrings[ stringFormat ]

      laneCount = self.getLaneCount()
      ethIntfMode = self.ethIntfMode()
      if ethIntfMode and ethIntfMode.speed != EthSpeed.speedUnknown and \
         ethIntfMode.laneCount != EthLaneCount.laneCountUnknown:
         speed = ethIntfMode.speed
         laneCount = ethIntfMode.laneCount

      return L1Intf._speedLanesStr( speed, laneCount, self.showLanes(),
                                        stringFormat )

   def setOperSpeedInProperties( self ):
      '''
      Method to return interface speed and lane count.
      Returns
      ------
      speed : Interfaces::EthSpeed. If the interface speed is 10G,
              the speed will be speed10Gbps.
      laneCount : Interfaces:EthLaneCount. If the interface is installed
                  a x-lane transceiver, the laneCount will be laneCountx.
      '''
      speed = self.status().speed
      laneCount = self.getLaneCount()
      ethIntfMode = self.ethIntfMode()
      if( ethIntfMode and ethIntfMode.speed != EthSpeed.speedUnknown and
          ethIntfMode.laneCount != EthLaneCount.laneCountUnknown ):
         speed = ethIntfMode.speed
         laneCount = ethIntfMode.laneCount
      return speed, laneCount


   def bandwidth( self ):
      '''Returns the integer value of the interface's bandwidth in kbits/sec.'''
      bw = EthSpeedApi.speedMbps( self.status().speed ) * 1000

      # V3.0 and newer systems always show "Unconfigured" for inactive interfaces
      # unless they have been manually configured.
      if ( self.l1PolicyVersion() >= L1PolicyVersion( 3, 0 ) and
           ( not self.active() ) and
           ( self.getLinkMode() == EthLinkMode.linkModeUnknown ) ):
         bw = 0

      return bw

   def getLaneCount( self ):
      if self.autonegActive():
         return self.status().autonegStatus.laneCount
      ethIntfMode = self.ethIntfMode()
      if ethIntfMode:
         return ethIntfMode.laneCount
      else:
         return EthLaneCount.laneCountUnknown

   @staticmethod
   def _speedLanesStr( speed, lanes, showLanes, stringFormat ):
      """
      Helper function that returns the string representation of the interface's
      speed and lane count.
      """
      if( speed == EthSpeed.speedUnknown or
          lanes == EthLaneCount.laneCountUnknown or
          not showLanes ):
         return L1Intf._speedStr( speed, stringFormat )
      else:
         return EthIntfLib.speedLanestoSpeedLanesStr[ speed, lanes ]

   @staticmethod
   def _showLanes( ethPhyDefaultCaps ):
      if ethPhyDefaultCaps is None:
         return False
      caps = ethPhyDefaultCaps.linkModeCapabilities
      return ( caps.mode50GbpsFull1Lane or
               caps.mode100GbpsFull1Lane or
               caps.mode100GbpsFull2Lane or
               caps.mode200GbpsFull2Lane or
               caps.mode200GbpsFull4Lane or
               caps.mode200GbpsFull8Lane or
               caps.mode400GbpsFull4Lane or
               caps.mode400GbpsFull8Lane or
               caps.mode800GbpsFull8Lane )

   def showLanes( self ):
      return L1Intf._showLanes( self.ethPhyDefaultCaps() )

   def _speedSetStr( self, speedSet, detail=False ):
      """
      Helper function that returns a string containing active speeds in an
      an EthLinkModeSet
      """
      setStr = ''
      sep = ''
      if speedSet.mode10MbpsHalf or speedSet.mode10MbpsFull:
         setStr += sep + '10M'
         sep = '/'
      if speedSet.mode100MbpsHalf or speedSet.mode100MbpsFull:
         setStr += sep + '100M'
         sep = '/'
      if speedSet.mode1GbpsHalf or speedSet.mode1GbpsFull:
         setStr += sep + '1G'
         sep = '/'
      if speedSet.mode2p5GbpsFull:
         setStr += sep + '2.5G'
         sep = '/'
      if speedSet.mode5GbpsFull:
         setStr += sep + '5G'
         sep = '/'
      if speedSet.mode10GbpsFull:
         setStr += sep + '10G'
         sep = '/'
      if speedSet.mode25GbpsFull:
         setStr += sep + '25G'
         sep = '/'
      if speedSet.mode40GbpsFull:
         setStr += sep + '40G'
         sep = '/'
      if speedSet.mode50GbpsFull:
         setStr += sep + '50G'
         sep = '/'
      if speedSet.mode50GbpsFull1Lane:
         setStr += sep + '50G'
         sep = '/'
      if speedSet.mode100GbpsFull or speedSet.mode100GbpsFull2Lane or \
            speedSet.mode100GbpsFull1Lane:
         setStr += sep + '100G'
         sep = '/'
      if speedSet.mode200GbpsFull4Lane or speedSet.mode200GbpsFull2Lane:
         setStr += sep + '200G'
         sep = '/'
      if speedSet.mode400GbpsFull8Lane or speedSet.mode400GbpsFull4Lane:
         setStr += sep + '400G'
         sep = '/'
      if speedSet.mode800GbpsFull8Lane:
         setStr += sep + '800G'
         sep = '/'
      if setStr == '':
         setStr = 'None' if detail else '-'
      return setStr

   def localSpeedSet( self, detail=False ):
      advert = self.status().autonegStatus.advertisement
      return self._speedSetStr( advert.linkModes, detail )

   def partnerSpeedSet( self, detail=False ):
      advert = self.status().autonegStatus.partnerAdvertisement
      return self._speedSetStr( advert.linkModes, detail )

   def _duplexSetStr( self, speedSet, detail=False ):
      """
      Helper function that returns a string containing active duplexes in an
      an EthLinkModeSet
      """
      setStr = ''
      sep = ''
      setStr = ''
      sep = ''
      if ( speedSet.mode10MbpsHalf or speedSet.mode100MbpsHalf or
           speedSet.mode1GbpsHalf ):
         setStr += sep + 'half'
         sep = '/'
      if ( speedSet.mode10MbpsFull or speedSet.mode100MbpsFull or
           speedSet.mode1GbpsFull or speedSet.mode10GbpsFull or
           speedSet.mode25GbpsFull or speedSet.mode40GbpsFull or
           speedSet.mode50GbpsFull or speedSet.mode50GbpsFull1Lane or
           speedSet.mode100GbpsFull or speedSet.mode100GbpsFull2Lane or
           speedSet.mode100GbpsFull1Lane or speedSet.mode200GbpsFull4Lane or
           speedSet.mode200GbpsFull2Lane or speedSet.mode400GbpsFull8Lane or
           speedSet.mode400GbpsFull4Lane or speedSet.mode800GbpsFull8Lane ):
         setStr += sep + 'full'
      if setStr == '':
         setStr = 'None' if detail else '-'
      return setStr

   def localDuplexSet( self, detail=False ):
      advert = self.status().autonegStatus.advertisement
      return self._duplexSetStr( advert.linkModes, detail )

   def partnerDuplexSet( self, detail=False ):
      advert = self.status().autonegStatus.partnerAdvertisement
      return self._duplexSetStr( advert.linkModes, detail )

   def autonegActive( self ):
      """
      Helper function that returns the string representation of the interface's
      duplex in one of a number of specifiable formats.
      """
      return self.status().autonegActive

   def operDuplex( self, stringFormat ):
      return EthIntfLib.generateOperDuplex( stringFormat,
                                           self.autonegActive(),
                                           self.status().duplex )

   def adminDuplex( self, stringFormat ):
      if self.autonegActive():
         return EthIntfLib.autonegDuplexStrings[ stringFormat ]
      ( _, _, duplex ) = \
            EthIntfLib.linkModeToSpeedLanesDuplex[ self.config().linkMode ]
      return EthIntfLib._duplexStr( duplex, stringFormat )

   _anegPauseStrMap = { 'pauseUnknown': 'None',
                        'pauseDisabled': 'Disabled',
                        'pauseTxOnly': 'TxOnly',
                        'pauseSymmetric': 'Symmetric',
                        'pauseRxOrSymmetric': 'RxOrSymmetric'
                        }

   def localAnegPause( self ):
      """
      Helper function that returns a string describing advertised Pause
      capabilities in an AutonegCapabilities object
      """
      advert = self.status().autonegStatus.advertisement
      return self._anegPauseStrMap[ advert.pause ]

   def partnerAnegPause( self ):
      advert = self.status().autonegStatus.partnerAdvertisement
      return self._anegPauseStrMap[ advert.pause ]

   def _autonegState( self ):
      anegStatus = self.status().autonegStatus
      state = ( anegStatus.state if self.config().enabled else 'anegStateOff' )
      return state

   _anegStateShortStrMap = { 'anegStateUnknown': 'unknown',
                             'anegStateOff': 'off',
                             'anegStateNegotiating': 'nego',
                             'anegStateSuccessful': 'success',
                             'anegStateFailed': 'failed'
                             }

   _anegStateLongStrMap = { 'anegStateUnknown': 'Unknown',
                            'anegStateOff': 'Off',
                            'anegStateNegotiating': 'Negotiating',
                            'anegStateSuccessful': 'Success',
                            'anegStateFailed': 'Failed'
                            }

   def autonegStateStr( self, longStr=False ):
      """
      Helper function that returns the string representation of the AutonegState
      in the autonegStatus member of an interface
      """
      if longStr:
         return self._anegStateLongStrMap[ self._autonegState() ]
      else:
         return self._anegStateShortStrMap[ self._autonegState() ]

   anegFecEncodingStrMap = { 'fecEncodingUnknown': 'N/A',
                             'fecEncodingDisabled': 'Off',
                             'fecEncodingReedSolomon544': 'RS',
                             'fecEncodingReedSolomon': 'RS',
                             'fecEncodingFireCode': 'FC'
                             }
   downshiftStatusStrMap = { 'downshiftNotApplicable': 'Not Applicable',
                             'downshiftDisabled': 'Disabled',
                             'downshifted': 'Downshifted',
                             'downshiftEnabled': 'Enabled'
                              }


   def speedDownshiftStatus( self ):
      downshift = 'downshiftNotApplicable'
      autonegStatus = self.status().autonegStatus
      if ( self.config().enabled and
           autonegStatus.state not in [ 'anegStateUnknown', 'anegStateOff' ] ):
         downshift = self.status().autonegStatus.downshiftStatus
      return self.downshiftStatusStrMap[ downshift ]

   _consortiumTechTypes = [ 'techAbilityEthCon25gCr',
                            'techAbilityEthCon25gKr',
                            'techAbilityEthCon50gCr2',
                            'techAbilityEthCon50gKr2',
                            'techAbilityEthCon400gCr8' ]

   def autonegConsortiumMode( self ):
      """
      Helper function to check if autonegotiation mode is in 'Consortium' mode.
      """
      autonegStatus = self.status().autonegStatus
      techType = autonegStatus.technologyType
      mode = autonegStatus.mode
      return mode == 'anegModeClause73' and techType in self._consortiumTechTypes

   _backplaneTechTypes = [ 'techAbility10gKr',
                           'techAbilityEthCon25gKr',
                           'techAbilityEthCon50gKr2', ]

   def autonegBackplaneMode( self ):
      """
      Helper function to check if autonegotiation mode is in 'KR' mode.
      """
      autonegStatus = self.status().autonegStatus
      techType = autonegStatus.technologyType
      mode = autonegStatus.mode
      return mode == 'anegModeClause73' and techType in self._backplaneTechTypes

   _anegModeStrMap = { 'anegModeUnknown': 'None',
                       'anegModeDisabled': 'None',
                       'anegModeClause28': 'BASE-T (IEEE Clause 28)',
                       'anegModeClause37': '1000BASE-X (IEEE Clause 37)',
                       'anegModeSgmii': 'SGMII' }

   def autonegMode( self ):
      """
      Helper function to return a mode string for an autonegotiation mode
      """
      mode = self.status().autonegStatus.mode
      if mode == 'anegModeClause73':
         mediumStr = 'CR'
         if self.autonegBackplaneMode():
            mediumStr = 'KR'
         modeStr = 'IEEE Clause 73'
         if self.autonegConsortiumMode():
            modeStr = 'Ethernet Consortium'
         return 'BASE-{mediumStr} ({modeStr})'.format(
               mediumStr=mediumStr, modeStr=modeStr )
      else:
         return self._anegModeStrMap[ mode ]

   # ----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesXcvrProperties()"
   # ----------------------------------------------------------------------------
   def xcvrTypeStr( self ):
      intfXcvrStatus = self.intfXcvrStatus()
      if intfXcvrStatus is None:
         return "N/A"
      xcvrPresence = intfXcvrStatus.xcvrPresence
      if xcvrPresence == 'xcvrPresenceUnknown':
         return 'Unknown'
      elif xcvrPresence == 'xcvrNotPresent':
         return 'Not Present'
      elif xcvrPresence == 'xcvrNotApplicable':
         return "N/A"
      else:
         assert xcvrPresence == 'xcvrPresent', \
            f"Unhandled xcvrPresence value '{ xcvrPresence }'"
         return intfXcvrStatus.xcvrType

   def adminSpeed( self, stringFormat ):
      if self.autonegActive():
         return self.autonegSpeedStrings[ stringFormat ]
      speed, lanes, _ = EthIntfLib.linkModeToSpeedLanesDuplex[
         self.config().linkMode ]
      return speed, lanes

   def adminSpeedStr( self ):
      return self.adminSpeed( 'Format2' )

   def adminDuplexStr( self ):
      return self.adminDuplex( 'Format2' )

   def _autonegSuffix( self ):
      if not self.status().linkStatus == 'linkUp':
         return ""
      if self.status().autonegStatus.mode == 'anegModeUnknown':
         # If auto-negotiation is not enabled and the link mode is the
         # default, the speed/duplex setting are the default for the
         # media type of the Xcvr, else they are forced by configuration
         if self.config().linkMode != 'linkModeUnknown':
            return 'forced'
         else:
            return ""
      else:
         return 'autoNegotiated'

   # ----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesFlowcontrol()"
   # ----------------------------------------------------------------------------
   flowcontrolCfgStringsFormat1 = { 'flowControlConfigOn': 'on',
                                    'flowControlConfigOff': 'off',
                                    'flowControlConfigDesired': 'desired',
                                    'flowControlConfigUnknown': 'unknown' }

   flowcontrolCfgStrings = { 'Format1': flowcontrolCfgStringsFormat1 }

   flowcontrolStatusStringsFormat1 = { 'flowControlStatusOn': 'on',
                                       'flowControlStatusOff': 'off',
                                       'flowControlStatusDisagree': 'disagree',
                                       'flowControlStatusUnknown': 'unknown' }

   flowcontrolStatusStrings = { 'Format1': flowcontrolStatusStringsFormat1 }

   def flowcontrolStateStr( self, capability, dictionary, key ):
      if capability == 'flowControlNotCapable':
         return 'Unsupp.'
      return dictionary[ key ]

   def flowcontrolTxAdminState( self ):
      return self.flowcontrolStateStr( self.status().txFlowcontrolCapabilities,
                                       self.flowcontrolCfgStringsFormat1,
                                       self.config().txFlowcontrol )

   def flowcontrolTxOperState( self ):
      return self.flowcontrolStateStr( self.status().txFlowcontrolCapabilities,
                                       self.flowcontrolStatusStringsFormat1,
                                       self.status().txFlowcontrol )

   def flowcontrolRxAdminState( self ):
      return self.flowcontrolStateStr( self.status().rxFlowcontrolCapabilities,
                                       self.flowcontrolCfgStringsFormat1,
                                       self.config().rxFlowcontrol )

   def flowcontrolRxOperState( self ):
      return self.flowcontrolStateStr( self.status().rxFlowcontrolCapabilities,
                                       self.flowcontrolStatusStringsFormat1,
                                       self.status().rxFlowcontrol )
   # ----------------------------------------------------------------------------
   # Loopback Configuration Handler
   # ----------------------------------------------------------------------------

   def setNoLoopbackMode( self ):
      cfg = self.config()
      cfg.loopbackMode = 'loopbackNone'

   def setLoopbackMode( self, loopbackSources, loopbackDevice ):
      cfg = self.config()
      if loopbackSources == 'network' and loopbackDevice == 'phy':
         cfg.loopbackMode = LOOPBACK_PHYREMOTE
      elif loopbackSources == 'system' and loopbackDevice == 'mac':
         cfg.loopbackMode = LOOPBACK_MAC
      elif loopbackSources == 'system' and loopbackDevice == 'phy':
         cfg.loopbackMode = LOOPBACK_PHY

   def loopbackModeSupported( self, lbMode='any' ):
      status = self.status()
      if not status:
         return False

      if lbMode == 'any':
         return any( self.status().loopbackModeSupported.values() )

      return ( ( lbMode in self.status().loopbackModeSupported ) and
               self.status().loopbackModeSupported[ lbMode ] )

   def showLoopbackMode( self, intfStatusModel ):
      status = self.status()
      if not status:
         return None

      intfStatusModel.loopbackMode = status.loopbackMode
      return intfStatusModel.loopbackMode

   def setDefault( self ):
      """
      Default handler
      """
      super().setDefault()
      self.setNoLoopbackMode()

   def linkModePriorityLists( self ):
      if isModular():
         sliceName = self.sliceName()
      else:
         sliceName = "FixedSystem"
      pList = Tac.newInstance( "Interface::PriorityListFactory" )
      resolveHelper = Tac.newInstance( "Interface::XcvrModeResolveHelper" )
      resolveHelper.sliceDefaultConfig = \
            ethIntfSliceDefaultConfigDir.defaultConfig.get( sliceName )
      resolveHelper.globalDefaultConfig = ethPhyIntfDefaultConfigDir
      # NOTE: intfXcvrStatus will be None for non external interfaces.
      resolveHelper.ethIntfXcvrStatus = self.intfXcvrStatus()
      resolveHelper.resolveXcvrMode()
      resolveHelper.resolveDefaultConfigs()
      pList.xcvrMode = resolveHelper.xcvrMode

      fdlDefaultLinkMode = 'linkModeUnknown'
      if self.config().ethDefaultConfig:
         fdlDefaultLinkMode = \
               self.config().ethDefaultConfig.linkModeLocal
      defaultLmPList = Tac.newInstance(
         "Interface::DefaultLinkModePriorityList",
         resolveHelper.moduleL1DefaultConfig.linkMode,
         resolveHelper.globalL1DefaultConfig.linkMode,
         self.status().overrideDefaultConfigLinkModeLocal,
         fdlDefaultLinkMode )
      return defaultLmPList, pList

   def linkModeCapabilitiesToList( self ):
      """
      return : List of CapList using L1PolicyAgent CapsResolver
      """
      lmCaps = []
      if not self.intfXcvrPresent():
         return lmCaps
      autonegCapable = (
         ( self.status().phyAutonegCapabilities.mode != 'anegModeUnknown' and
           self.status().phyAutonegCapabilities.mode != 'anegModeUnsupported' ) or
         ( self.status().autonegCapabilities.mode != 'anegModeUnknown' and
           self.status().autonegCapabilities.mode != 'anegModeUnsupported' ) )
      imcompatElms = False
      CapList = EthIntfLib.CapList
      l1CapsSliceDir = l1CapsSlotDir[ self.slotId() ]
      l1CapsSubdomainDir = l1CapsSliceDir[ self.subdomain() ]
      intfCaps = l1CapsSubdomainDir.intfCapabilities[ self.name ]
      for mode in intfCaps.modeCapabilities:
         defaultMode = self.status().defaultLinkMode( *self.linkModePriorityLists() )
         default = mode.forcedLinkMode == defaultMode
         lmCaps.append( CapList( mode.speed, mode.laneCount, self.showLanes(),
                                 mode.duplex, default, imcompatElms ) )
      if autonegCapable:
         lmCaps.append(
               CapList( 'auto', EthLaneCount.laneCountUnknown, False,
                        'full', False, False ) )
      return lmCaps

   def coherentErrorCorrectionSupported( self ):
      return None

   def cmisCoherentErrorCorrectionSupported( self ):
      return None

   def autonegCapabilitiesMode( self ):
      return None

   def autonegCapabilitiesToList( self ):
      return None

   def errorCorrectionSupported( self ):
      """If the interface supports any encoding value this will return
      True.  If no encodings at any speed or the EthPhyIntfStatus has
      not yet been created it will return False. We use this to
      suppress output in show commands on interfaces that just don't
      suppport FEC at all or we don't yet know if they support FEC."""

      if self.lookupPhysical():
         for fecSet in self.status().phyFecCapabilities.values():
            for attrList in EthIntfLib.tokenToFecEncodingAttrs.values():
               for attr in attrList:
                  # If fecEncodingDisabled is the only supported encoding,
                  # don't count it
                  if attr == 'fecEncodingDisabled':
                     continue
                  if getattr( fecSet, attr ):
                     return True
      return False

   def errorCorrectionEncodingSupported( self, encoding, ignoreOutputErrors=True ):
      """
      Support for error-correction configuration
      """
      if encoding and encoding not in EthIntfLib.tokenToFecEncodingAttrs:
         self.mode_.addError( 'Invalid error-correction encoding %s' % encoding )
         return False
      supported = False
      if self.lookupPhysical():
         for fecSet in self.status().phyFecCapabilities.values():
            for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
               supported |= getattr( fecSet, attr )
         if not supported and not ignoreOutputErrors:
            errorMsg = (
               'Error-correction encoding %s is unsupported for interface %s.' %
               ( encoding, self.name ) )
            supported = self.maybeAllowUnsupportedConfig(
               ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.errorCorrection,
               self.mode_, errorMsg )
         return supported
      else:
         return True

   def errorCorrectionCapabilitiesToList( self, encoding ):
      if not self.intfXcvrPresent():
         return []
      intfXcvrStatus = self.intfXcvrStatus()
      # Use a Dict to store supported link modes for given FEC. If the FEC is
      # the default, set the value to True.
      linkModes = {}
      for linkMode, fecSet in self.status().phyFecCapabilities.items():
         # Don't display linkModes for FEC that are not in the linkMode caps for the
         # interface. This can happen when the Xcvr inserted does not have all
         # linkModes which support FEC available.
         if not self.status().linkModeSupported[ linkMode ]:
            continue
         for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
            if getattr( fecSet, attr ):
               # FEC is supported in this linkMode
               linkModes[ linkMode ] = [ False, attr ]
               if( linkMode in intfXcvrStatus.mediaFecStandard and
                   intfXcvrStatus.mediaFecStandard[ linkMode ] == attr ):
                  # Mark this FEC as default FEC for this linkMode.
                  linkModes[ linkMode ] = [ True, attr ]
                  break

         if ( linkMode in linkModes
              and linkMode in intfXcvrStatus.mediaFecStandard
              and intfXcvrStatus.mediaFecStandard[ linkMode ]
              == 'fecEncodingReedSolomon'
              and not getattr( fecSet, 'fecEncodingReedSolomon' )
              and getattr( fecSet, 'fecEncodingFireCode' ) ):
            # In the case where there's a mismatch between the FEC capability of the
            # phy and the xcvr (ie. RS FEC unsupported by phy) and the FC FEC is
            # supported, use FireCode as the default.
            linkModes[ linkMode ][ 0 ] = True

      # Sort based on the order of keys in linkModeToSpeedLanesDuplex
      linkModeCap = []
      showLanes = self.showLanes()
      for linkMode in EthIntfLib.linkModeToSpeedLanesDuplex:
         if linkMode in linkModes:
            speed, lanes, duplex = EthIntfLib.linkModeToSpeedLanesDuplex[ linkMode ]
            duplex = duplex[ len( 'duplex' ) : ].lower()
            linkModeCap.append(
                     EthIntfLib.FecCapList( linkModes[ linkMode ][ 1 ], speed,
                                 lanes, showLanes,
                                 linkModes[ linkMode ][ 0 ],
                                 duplex ) )

      return linkModeCap

   # ----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesDebounceTime()"
   # ----------------------------------------------------------------------------
   def linkUpDebouncePeriod( self ):
      return self.config().linkUpDebouncePeriod

   def linkDownDebouncePeriod( self ):
      return self.config().linkDownDebouncePeriod

   # ------------------------------------------------------------------------------
   # Sets the link debounce times for the interface to the specified values.
   # ------------------------------------------------------------------------------
   def setLinkDebounceTimes( self, linkUpDebounceTime, linkDownDebounceTime ):
      self.config().linkUpDebouncePeriod = linkUpDebounceTime
      self.config().linkDownDebouncePeriod = linkDownDebounceTime

   def setErrorCorrectionEncoding( self, encoding=None, noOrDefault=False ):
      """
      1. if encoding is specified we need to go down one of the client/coherent
         paths, configure the encoding and set the other path to default.
      2. if no/default is specified go down both the paths and configure no/default.
         Note: no/default with or without encoding will exercise both paths
               with no/default
      """
      coherentEncoding = nonCoherentEncoding = encoding
      coherentNoOrDefault = nonCoherentNoOrDefault = noOrDefault
      if 'coherent' in errorCorrectionCfgFns:
         if ( encoding and
              EthIntfLib.encodingType( encoding ) == "non-coherent" ):
            coherentEncoding = None
            coherentNoOrDefault = "default"
         self.config().isCoherentFecConfigured = \
            errorCorrectionCfgFns[ 'coherent' ].configFn( self, coherentEncoding,
                                                          coherentNoOrDefault )
      if ( encoding and
           EthIntfLib.encodingType( encoding ) == "coherent" ):
         nonCoherentEncoding = None
         if noOrDefault != True: # pylint: disable=singleton-comparison
            nonCoherentNoOrDefault = "default"
      self.setNonCoherentFec( nonCoherentEncoding, nonCoherentNoOrDefault )

   def setNonCoherentFec( self, encoding=None, noOrDefault=False ):
      isNo = noOrDefault is True
      isDefault = str( noOrDefault ).lower() == 'default'

      if encoding and not self.errorCorrectionEncodingSupported( encoding, False ):
         return

      assert encoding != 'disabled'

      fecEncoding = Tac.nonConst( self.config().fecEncodingConfig )

      # Get the current encoding and clear the 'none' value as we are
      # about to configure a valid encoding.
      # If no form with no encoding then that completely disables FEC
      # no and default form with an encoding just turns off that encoding.
      if encoding:
         for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
            setattr( fecEncoding,
                     attr,
                     not bool( noOrDefault ) )
         # If we enabled an encoding then fecEncodingDisabled must be False.
         if not bool( noOrDefault ):
            fecEncoding.fecEncodingDisabled = False

      # Without an encoding no and default differ.
      # No disables FEC by setting fecEncodingDisabled to True
      # Default runs FEC when appropriate based on the xcvr media
      elif isNo or isDefault:
         # This disables FEC
         fecEncoding.fecEncodingDisabled = isNo
         # Turn off each of the individual FEC types.
         for attrList in EthIntfLib.tokenToFecEncodingAttrs.values():
            for attr in attrList:
               if attr == 'fecEncodingDisabled':
                  continue
               setattr( fecEncoding, attr, False )
      else:
         raise ValueError( "Invalid input parameter combination: %s and %s" %
                           encoding, noOrDefault )

      self.config().fecEncodingConfig = fecEncoding



# -------------------------------------------------------------------------------
# A subclass of the L1Intf class for Ethernet / LAG interfaces.
# Ethernet and Lag further subclass this class.
#
# There are classes that doubly inherit from EthIntf or derivatives which
# results in a diamond-shaped inheritance structure. To ensure base-class methods,
# such as setDefault, are called only once, methods overriden in derived classes
# should use the super() construct for calling base-class methods to ensure the
# mro is followed and the base-class method called only once.
# -------------------------------------------------------------------------------
class EthIntf( IntfCli.Intf ):
   """
   A subclass of the L1Intf class for Ethernet / LAG interfaces.
   Ethernet and Lag further subclass this class.
   """
   def __init__( self, name, mode ):
      """
      Creates a new EthIntf instance of the specified name.
      """
      super().__init__( name, mode )
      t0( "EthIntf::__init__", name )
      self.ethIntfConfigDir = None
      self.ethIntfStatuses = ethIntfStatusDir.intfStatus
      self.ethIntfConfig = None

   def lookupPhysical( self ):
      """
      Determines whether the Interface::EthIntfStatus object for this interface
      exists.
      """
      ethIntfStatus = self.status()
      return ethIntfStatus is not None

   def config( self ):
      """
      Returns the EthIntfConfig object for this interface.
      EthIntfConfig is cached to prevent an update from the EntityLog thread
      in the middle of processing a "show interfaces" command from causing an
      inconsistent snapshot to be displayed.
      The caching is done only if not in a config session, since the above
      scenario cannot happen when in a config session.
      """
      config = self.ethIntfConfigDir.intfConfig.get( self.name )
      if not self.mode_.session_.inConfigSession():
         self.ethIntfConfig = config
      return config

   @Tac.memoize
   def status( self ):
      """
      Returns the EthIntfStatus object for this interface.
      This method is Tac.memoized to prevent an update from the EntityLog thread
      in the middle of processing a "show interfaces" command from causing an
      inconsistent snapshot to be displayed.
      """
      return self.ethIntfStatuses.get( self.name )

   def getIntfCounterDir( self ):
      """
      Returns the EthIntfCounterDir object for this interface.
      """
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return ethPhyIntfCounterDir_

   def dataLinkExplanation( self, mode, ethIntfStatus ):
      """
      Returns a tuple for the explanation of a datalink
      ( Reason, Long Reason, Legend, Vlan annotation )
      """
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.dataLinkExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanDataLinkAnnotations )
               return ( reason, longReason,
                           legend, vlanDataLinkAnnotations[ index ] )
      return "Not bridged", None, None, None

   def quietDataLinkExplanation( self, mode, ethIntfStatus ):
      """
      Returns a tuple for the explanation of a quietDataLink port
      ( Reason, Long Reason, Legend, Vlan annotation )
      """
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.quietDataLinkExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanQuietDataLinkAnnotations )
               return ( reason, longReason,
                           legend, vlanQuietDataLinkAnnotations[ index ] )
      return "Not bridged", None, None, None

   def unauthorizedExplanation( self, mode, ethIntfStatus ):
      """
      Returns a tuple for the explanation of an unauthorized port
      ( Reason, Long Reason, Legend, Vlan annotation )
      """
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.unauthorizedExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanUnauthorizedAnnotations )
               return ( reason, longReason,
                           legend, vlanUnauthorizedAnnotations[ index ] )
      return "Not bridged", None, None, None

   def recirculationExplanation( self, mode, ethIntfStatus ):
      """
      Returns a tuple for the explanation of an unauthorized port
      ( Reason, Long Reason, Legend, Vlan annotation )
      """
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.recirculationExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( vlanRecirculationAnnotations )
               return ( reason, longReason,
                           legend, vlanRecirculationAnnotations[ index ] )
      return "Not bridged", None, None, None

   def linkQualificationExplanation( self, mode, ethIntfStatus ):
      """
      Returns a tuple for the explanation of a port in link qualification mode
      ( Reason, Long Reason, Legend, Vlan annotation )
      """
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
            enumerate( IntfCli.linkQualificationExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( linkQualificationAnnotations )
               return ( reason, longReason,
                           legend, linkQualificationAnnotations[ index ] )
      return "Not bridged", None, None, None

   def layer1PatchExplanation( self, mode, ethIntfStatus ):
      """
      Returns a tuple for the explanation of an layer1 patch port
      ( Reason, Long Reason, Legend, Vlan annotation )
      """
      reason = None
      legend = None
      longReason = None
      for ( index, ext ) in \
         enumerate( IntfCli.layer1PatchExplanationHook.extensions() ):
         if ethIntfStatus:
            ( reason, longReason, legend ) = ext( ethIntfStatus.intfId, mode )
            if reason is not None:
               assert index < len( layer1PatchAnnotations )
               return ( reason, longReason,
                           legend, layer1PatchAnnotations[ index ] )
      return "Not bridged", None, None, None

   def getIntfMembership( self, mode ):
      """
      Prints the interface membership (for e.g. if part of
      monitor destination or a Port-Channel)
      """
      specialModels = [
         "intfForwardingModelDataLink",
         "intfForwardingModelLinkQualification",
         "intfForwardingModelLayer1Patch",
         "intfForwardingModelLayer1PatchAndLldp",
         "intfForwardingModelRecirculation",
      ]

      if self.status().forwardingModel not in specialModels:
         return None
      elif self.status().forwardingModel == "intfForwardingModelLinkQualification":
         ( _, longExpl, _, _ ) = \
              self.linkQualificationExplanation( mode, self.status() )
         return longExpl
      elif self.status().forwardingModel == "intfForwardingModelDataLink":
         ( _, longExpl, _, _ ) = \
              self.dataLinkExplanation( mode, self.status() )
         return longExpl
      elif self.status().forwardingModel == "intfForwardingModelRecirculation":
         ( _, longExpl, _, _ ) = \
              self.recirculationExplanation( mode, self.status() )
         return longExpl
      # intfForwardingModelLayer1Patch or intfForwardingModelLayer1PatchAndLldp
      else:
         ( _, longExpl, _, _ ) = \
              self.layer1PatchExplanation( mode, self.status() )
         return longExpl

   def showPhysical( self, mode, intfStatusModel ):
      """
      Outputs information about this interface in an interface type-specific
      manner.
      """
      intfStatusModel.interfaceMembership = self.getIntfMembership( mode )
      self.showLinkTypeSpecific( intfStatusModel )
      self.showLoopbackMode( intfStatusModel )
      self.showL2Mru( intfStatusModel )
      if self.countersSupported():
         intfStatusModel.interfaceStatistics = self.getInterfaceStatistics()
         intfStatusModel.interfaceCounters = self.getInterfaceCounters()

   def bandwidth( self ):
      raise NotImplementedError

   def autonegActive( self ):
      raise NotImplementedError

   def flowcontrolRxPause( self ):
      raise NotImplementedError

   def flowcontrolTxPause( self ):
      raise NotImplementedError

   def showLoopbackMode( self, intfStatusModel ):
      raise NotImplementedError

   def getIntfStatusModel( self ):
      return EthIntfModel.EthInterfaceStatus( name=self.status().intfId )

   def ratePct( self, bps, pps ):
      """ Uses the link bandwidth to determinte the % of theoretical
      bitrate achieved including the 12 byte inter-frame gap and the
      8 byte preable.

      Returns None of bandwidth is not set"""

      maxBw = self.bandwidth() * 1000.0
      per = None
      if maxBw != 0:
         per = ( bps + 160 * pps ) * 100 / maxBw
      return per

   def rateValue( self, attr ):
      """
      Calculate the rate counter value by exponentially decay the
      previously saved value and subtract it from the current rate
      value.
      """
      assert self.counter()
      counter = self.counter()
      ckpt = self.getLatestCounterCheckpoint()
      loadInterval = IntfCli.getActualLoadIntervalValue(
         self.config().loadInterval )

      sysdbValue = getattr( counter, attr )
      if ckpt:
         ckptValue = getattr( ckpt.rates, attr, 0 )
         ckptStatsUpdateTime = getattr( ckpt.rates, "statsUpdateTime", 0 )
      else:
         ckptValue = 0
         ckptStatsUpdateTime = 0

      # If loadInterval = 0, return rate is the rate at the most recent snapshot
      # without being decayed.
      if loadInterval == 0:
         return sysdbValue

      # If an interface is error disabled, its counters would never be updated. For
      # an error disabled interface, If clear command is issued, the last time when
      # counters are updated will be less than the last time counter are cleared.
      # So Addin a guard for this case!
      if counter.rates.statsUpdateTime < ckptStatsUpdateTime:
         return 0.0

      # If the interface is not enabled, then just return the rate counter as 0
      if not ( self.ethIntfConfig and self.ethIntfConfig.enabled ):
         return 0.0

      expFactor = - ( ( counter.rates.statsUpdateTime - ckptStatsUpdateTime ) /
                      loadInterval )
      decayedCkptValue = ckptValue * math.exp( expFactor )
      return max( 0.0, sysdbValue - decayedCkptValue )

   def getInterfaceStatistics( self ):
      if self.counter() is None:
         return None

      interfaceStatistics = EthInterfaceStatistics()
      interfaceStatistics.updateInterval = \
                     IntfCli.getActualLoadIntervalValue( self.config().loadInterval )
      interfaceStatistics.inBitsRate = self.rateValue( "inBitsRate" )
      interfaceStatistics.outBitsRate = self.rateValue( "outBitsRate" )
      interfaceStatistics.inPktsRate = self.rateValue( "inPktsRate" )
      interfaceStatistics.outPktsRate = self.rateValue( "outPktsRate" )
      return interfaceStatistics

   def updateInterfaceCountersRateModel( self ):
      if self.counter() is None:
         return None
      intfCountersRateModel = InterfaceCountersRate( _name=self.name )
      intfCountersRateModel.description = self.description()
      intfCountersRateModel.interval = int(
            IntfCli.getActualLoadIntervalValue( self.config().loadInterval ) + 0.5 )
      intfCountersRateModel.inBpsRate = self.rateValue( "inBitsRate" )
      intfCountersRateModel.inPpsRate = self.rateValue( "inPktsRate" )
      intfCountersRateModel.inPktsRate = \
                     self.ratePct( intfCountersRateModel.inBpsRate,
                     intfCountersRateModel.inPpsRate )
      intfCountersRateModel.outBpsRate = self.rateValue( "outBitsRate" )
      intfCountersRateModel.outPpsRate = self.rateValue( "outPktsRate" )
      intfCountersRateModel.outPktsRate = \
                     self.ratePct( intfCountersRateModel.outBpsRate,
                     intfCountersRateModel.outPpsRate )
      # lastUpdateTimestamp should not be a "normalized" delta -- it should be the
      # last update UTC value
      intfCountersRateModel.lastUpdateTimestamp = (
            Ark.switchTimeToUtc( self.counter().rates.statsUpdateTime ) )

      return intfCountersRateModel

   def getLastUpdateTimeStamp( self ):
      ckpt = self.getLatestCounterCheckpoint()
      if ckpt:
         statsUpdateTime = getattr( ckpt.rates, 'statsUpdateTime', None )
      else:
         statsUpdateTime = None
      if statsUpdateTime is not None:
         return statsUpdateTime + Tac.utcNow() - Tac.now()
      else:
         return None

   def getInterfaceCounters( self, notFoundString=None, checkpoint=None ):
      counter = self.counter()
      if counter is None:
         return None
      if checkpoint is None:
         ckpt = self.getLatestCounterCheckpoint()
      else:
         ckpt = checkpoint
      def stat( attr ):
         sysdbValue = getattr( counter.statistics, attr, None )
         if sysdbValue is None:
            ethStats = getattr( counter, 'ethStatistics', None )
            if ethStats is not None:
               sysdbValue = getattr( ethStats, attr, None )
         if sysdbValue is None:
            return notFoundString
         checkpointValue = 0
         if ckpt:
            checkpointValue = getattr( ckpt.statistics, attr, None )
            if checkpointValue is None:
               checkpointValue = getattr( ckpt.ethStatistics, attr, 0 )
         return sysdbValue - checkpointValue

      interfaceCounters = EthInterfaceCounters()
      interfaceCounters.lastClear = self.getLastUpdateTimeStamp()

      # Handle interface types with and without the
      # physical layer counters.  The counters that
      # aren"t present are returned as "None", so be
      # careful doing math with them (like collisions)!
      interfaceCounters._name = self.name
      interfaceCounters.inUcastPkts = stat( "inUcastPkts" )
      interfaceCounters.inMulticastPkts = stat( "inMulticastPkts" )
      interfaceCounters.inBroadcastPkts = stat( "inBroadcastPkts" )
      interfaceCounters.inOctets = stat( "inOctets" )
      interfaceCounters.inTotalPkts = stat( "inTotalPkts" )
      if interfaceCounters.inTotalPkts == 0:
         interfaceCounters.inTotalPkts = interfaceCounters.inUcastPkts + \
                                         interfaceCounters.inMulticastPkts + \
                                         interfaceCounters.inBroadcastPkts
      interfaceCounters.inDiscards = stat( "inDiscards" )
      interfaceCounters.totalInErrors = stat( "inErrors" )
      runtFrames = stat( "frameTooShorts" )
      if runtFrames is not None:
         inputErrors = EthInterfaceCounters.PhysicalInputErrors()
         inputErrors.runtFrames = runtFrames
         inputErrors.giantFrames = stat( "frameTooLongs" )
         inputErrors.fcsErrors = stat( "fcsErrors" )
         inputErrors.alignmentErrors = stat( "alignmentErrors" )
         inputErrors.symbolErrors = stat( "symbolErrors" )
         # Use the same function as "show interfaces flowcontrol" to eliminate
         # duplicate code that is out of sync.
         inputErrors.rxPause = self.flowcontrolRxPause( )
         interfaceCounters.inputErrorsDetail = inputErrors

      interfaceCounters.outUcastPkts = stat( "outUcastPkts" )
      interfaceCounters.outMulticastPkts = stat( "outMulticastPkts" )
      interfaceCounters.outBroadcastPkts = stat( "outBroadcastPkts" )
      interfaceCounters.outOctets = stat( "outOctets" )
      interfaceCounters.outTotalPkts = stat( "outTotalPkts" )
      if interfaceCounters.outTotalPkts is None:
         interfaceCounters.outTotalPkts = interfaceCounters.outUcastPkts + \
                                         interfaceCounters.outMulticastPkts + \
                                         interfaceCounters.outBroadcastPkts
      interfaceCounters.outDiscards = stat( "outDiscards" )

      interfaceCounters.totalOutErrors = stat( "outErrors" )
      interfaceCounters.counterRefreshTime = Tac.utcNow()
      collisions = stat( "singleCollisionFrames" )
      if collisions is not None:
         outputErrors = EthInterfaceCounters.PhysicalOutputErrors()
         outputErrors.collisions = collisions + stat( "multipleCollisionFrames" )
         outputErrors.lateCollisions = stat( "lateCollisions" )
         outputErrors.deferredTransmissions = stat( "deferredTransmissions" )
         # Use the same function as "show interfaces flowcontrol" to eliminate
         # duplicate code that is out of sync.
         outputErrors.txPause = self.flowcontrolTxPause( )
         interfaceCounters.outputErrorsDetail = outputErrors

      # Get the number of link status changes since last
      # 'clear counters [session]'.
      interfaceCounters.linkStatusChanges = self.status().operStatusChange.count - \
                                             getattr( ckpt, "linkStatusChanges", 0 )

      return interfaceCounters

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().addr )

   def routedAddr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().routedAddr )

   def hardware( self ):
      raise NotImplementedError()

   def addrStr( self ):
      raise NotImplementedError()

   def showLinkTypeSpecific( self, intfStatusModel ):
      raise NotImplementedError()

   def showL2Mru( self, intfStatusModel ):
      if self.status().l2Mru:
         intfStatusModel.l2Mru = self.status().l2Mru

   linkStatusStringsFormat1 = { 'linkUp' : 'up',
                                'linkDown' : 'down',
                                'linkUnknown' : 'unknown' }

   linkStatusStringsFormat2 = { 'linkUp' : 'connected',
                                'linkDown' : 'notconnect',
                                'linkUnknown' : 'unknown' }

   pmaInterfaceLinkStatusStrings = { 'linkUp': 'signal',
                                     'linkDown': 'noSignal',
                                     'linkUnknown': 'unknown' }

   linkStatusStrings = { 'Format1' : linkStatusStringsFormat1,
                         'Format2': linkStatusStringsFormat2,
                         'pmaIntf': pmaInterfaceLinkStatusStrings }

   def _linkStatusStr( self, stringFormat ):
      """
      Helper function that returns the string representation of the interface's
      link status in one of a number of specifiable formats.
      """
      l = self.linkStatusStrings.get( stringFormat )
      return l[ self.status().linkStatus ]

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesStatus()"
   #----------------------------------------------------------------------------
   def linkStatus( self ):
      if not self.config().enabled:
         # If the adminEnabled state, and enabled state are
         # different, then the interface must be errdisabled.
         # the enabledStateReason contains the status string
         if ( not self.config().adminEnabled ) and self.status().active:
            return 'disabled'
         else:
            # As we enter this block, there is a small chance that the interface
            # becomes active and the enabledStateReason is just cleared.
            reason = self.config().enabledStateReason
            return reason if reason else "unknown"
      linkStatusFormat = 'pmaIntf' if self.isPmaInterface() else 'Format2'
      return self._linkStatusStr( linkStatusFormat )

   def isPmaInterface( self ):
      # isPmaInterface only applies to EthPhyIntfs so we return False here
      return False

   #----------------------------------------------------------------------------
   # Utility function used by switchport CLI modelet
   #----------------------------------------------------------------------------
   def switchportEligible( self ):
      raise NotImplementedError( )

   #----------------------------------------------------------------------------
   # A hack to lazy mount counters
   #----------------------------------------------------------------------------
   def counter( self ):
      return IntfCli.Intf.counter( self )

   def setL2Mru( self, value ):
      """
      Sets the L2 Mru value
      """
      cfg = self.config()
      # MRU range for an interface may change dynamically, thus a value cannot be
      # used now may be configured. In this case, CLI will accept the value, and
      # agent code should handle it.
      if self.lookupPhysical() and value != 0:
         status = self.status()
         if value < status.l2MruMin:
            self.mode_.addWarning( 'Min MRU for %s is %d bytes' %
                                   ( self.name, status.l2MruMin ) )
         elif value > status.l2MruMax:
            self.mode_.addWarning( 'Max MRU for %s is %d bytes' %
                                   ( self.name, status.l2MruMax ) )
      cfg.l2Mru = value

   def setTimestampMode( self, timestampMode ):
      """
      Sets the timestamp mode for an interface.
      """
      self.config().timestampMode = timestampMode

class EthPhyIntf( L1Intf, EthIntf ):
   """
   A subclass of the EthIntf class for physical Ethernet interfaces.
   """

   def __init__( self, name, mode ):
      """
      Creates a new EthPhyIntf instance of the specified name.
      """
      t0( "EthPhyIntf::__init__", name )
      super().__init__( name, mode )

   def ethPhyDefaultCaps( self ):
      """
      Returns the default capabilities (link mode, autoneg, fec, etc.)
      for this interface.
      """
      if isModular():
         sliceName = self.sliceName()
      else:
         sliceName = "FixedSystem"
      if sliceName:
         ethPhyDefaultCapsDirs = ethPhyDefaultCapsSliceDir.get( sliceName )
         if ethPhyDefaultCapsDirs:
            for ethPhyDefaultCapsDir in ethPhyDefaultCapsDirs.values():
               ethPhyDefaultCaps = ethPhyDefaultCapsDir.portCaps.get( self.name )
               if ethPhyDefaultCaps:
                  return ethPhyDefaultCaps

      return None

   def ethIntfMode( self ):
      """
      Returns the EthIntfMode object (speed, duplex, FEC) for this interface.
      """
      if isModular():
         sliceName = self.sliceName()
      else:
         sliceName = "FixedSystem"
      if sliceName:
         ethIntfModeDirs = ethIntfModeSliceDir.get( sliceName )
         if ethIntfModeDirs:
            ethIntfGeneration = ethIntfGenerationSliceDir.get( sliceName )
            for ethIntfModeDir in ethIntfModeDirs.values():
               if ethIntfModeDir.generation == ethIntfGeneration.generation:
                  ethIntfMode = ethIntfModeDir.ethIntfMode.get( self.name )
                  if ethIntfMode:
                     return ethIntfMode

      return None

   def ethIntfModeDir( self ):
      if Cell.cellType() == "fixed":
         slotId = "FixedSystem"
      else:
         slotId = self.sliceName()
      ethIntfModeDirs = ethIntfModeSliceDir.get( slotId )
      if ethIntfModeDirs:
         ethIntfGeneration = ethIntfGenerationSliceDir.get( slotId )
         for ethIntfModeDir in ethIntfModeDirs.values():
            if ethIntfModeDir.generation == ethIntfGeneration.generation:
               ethIntfMode = ethIntfModeDir.ethIntfMode.get( self.name )
               if ethIntfMode:
                  return ethIntfModeDir
      return None

   def resourceManagerStatus( self ):
      resourceManagerStatusList = []
      for slot, slotDir in resourceMgrStatusSliceDir.items():
         for resourceManagerStatus in slotDir.values():
            intfState = resourceManagerStatus.intfStateDir.intfState.get( self.name )
            if intfState:
               resourceManagerStatusList.append( ( slot, resourceManagerStatus ) )

      # On a Glacier like system, there may be more than 1 resourceManagerStatus for
      # one interface. For example, one resourceManagerStatus for the Linecard where
      # the interface is on, and one resourceManagerStatus for the whole Switchcard.
      # As IPM is running inside the scope of forwarding agent, we return the one
      # for Switchcard if there are multiple resourceManagerStatus.
      if len( resourceManagerStatusList ) == 1:
         return resourceManagerStatusList[ 0 ][ 1 ]
      elif len( resourceManagerStatusList ) == 2:
         for slot, resourceManagerStatus in resourceManagerStatusList:
            if slot.startswith( "Switchcard" ):
               return resourceManagerStatus

      return None

   def config( self ):
      """
      Returns the EthIntfConfig object for this interface.
      Overrides the parent method
      """
      def getEthIntfConfig():
         if self.sliceName():
            ethIntfConfigDir = ethPhyIntfConfigSliceDir.get( self.sliceName() )
            if ethIntfConfigDir:
               return ethIntfConfigDir.intfConfig.get( self.name )

         return None

      if self.mode_.session_.inConfigSession():
         return getEthIntfConfig()
      else:
         if not self.ethIntfConfig:
            self.ethIntfConfig = getEthIntfConfig()
         return self.ethIntfConfig

   def getStatus( self ):
      lineProtocolStatus, interfaceStatus = EthIntf.getStatus( self )
      if self.isPmaInterface():
         if interfaceStatus == "connected":
            interfaceStatus = "signal"
         elif interfaceStatus == "notconnect":
            interfaceStatus = "noSignal"
      return ( lineProtocolStatus, interfaceStatus )

   @Tac.memoize
   def slotId( self ):
      # TODO BUG604500: Management interfaces are not yet supported since the exact
      #                 impact of adding them is unknown. This is due to the fact
      #                 that management interfaces do not generally use slot IDs.
      #                 Instead, they tend to always use the cell ID as the dir key.
      if not EthIntfId.isEthIntfId( self.name ):
         return None

      # TODO BUG574130: This hack is needed until EthIntfCli returns the correct
      #                 slot ID on fixed systems.
      return self.sliceName() if Cell.cellType() == 'supervisor' else 'FixedSystem'


   #----------------------------------------------------------------------------
   # The rule for matching Ethernet interface names.  When this pattern matches,
   # it returns an instance of the EthPhyIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   # pylint: disable=undefined-variable
   valueFunc_ = lambda mode, intf: EthPhyIntf( intf, mode )
   matcher = IntfMatcher()
   ethOrMgmtMatcher = matcher
   ethMatcher = PhysicalIntfRule.PhysicalIntfMatcher( 'Ethernet',
                                                      value=valueFunc_ )
   mgmtMatcher = PhysicalIntfRule.PhysicalIntfMatcher( 'Management',
                                                       value=valueFunc_ )
   matcher |= ethMatcher
   matcher |= mgmtMatcher

   def vrfSupported( self ):
      ''' Support VRF on Ethernet and Management interfaces '''
      return ( self.name.startswith( 'Ethernet' ) or
            self.name.startswith( 'Management' ) )

   def intfCreationCurrentlySupported( self, mode ):
      return True

   def createInterfaceConfigurationRequest( self ):
      """
      Creates the Interface::EthPhyIntfConfig object for this interface if it does
      not already exist.
      """
      # It's important to create the slice dir before creating the config request.
      # This way when the Sysdb reactor for the config request fires the slice dir
      # will already exist
      ethPhyIntfConfigDir = ethPhyIntfConfigSliceDir.newEntity(
         "Interface::EthPhyIntfConfigDir",
         self.sliceName() )

      IntfCli.cliCreateIntfConfigRequest( ethPhyIntfConfigDir, self.name )

   def createInterfaceConfiguration( self, startupConfig=False ):
      ethPhyIntfConfigDir = ethPhyIntfConfigSliceDir[ self.sliceName() ]
      intfConfig = ethPhyIntfConfigDir.intfConfig.newMember( self.name )

      if not EthIntfId.isEthernet( self.name ):
         return

      # handle global ethernet shutdown to make sure adminEnabledStateLocal
      # is initialized properly
      if intfConfig.adminEnabledStateLocal == 'unknownEnabledState':
         if IntfCli.globalIntfConfig.defaultEthernetShutdown:
            # similar to SysdbEthIntf's handleIntfConfigRequest(), initialize
            # the shutdown state if global ethernet shutdown is configured
            # and our shutdown state is 'unknown' (new interfaces).
            intfConfig.adminEnabledStateLocal = 'shutdown'
         else:
            sysdbEthPhyIntfConfigDir = ethPhyIntfConfigSliceDir.sysdbObj().get(
               self.sliceName() )
            if sysdbEthPhyIntfConfigDir:
               sysdbIntfConfig = sysdbEthPhyIntfConfigDir.intfConfig.get( self.name )
               # If the sysdb object has a valid adminEnabledStateLocal, and
               # we don't, we should not override it. This is a bit hackish, but
               # the only time we need to do this is when
               # 1) the interface has no status (no defaultConfig) and
               # 2) default ethernet is "no shutdown" (default)
               if ( sysdbIntfConfig and
                    sysdbIntfConfig.adminEnabledStateLocal
                    != 'unknownEnabledState' and
                    not sysdbIntfConfig.defaultConfig ):
                  intfConfig.adminEnabledStateLocal = 'enabled'

   def deleteInterfaceConfigurationRequest( self ):
      """
      Destroys the Interface::EthPhyIntfConfig object for this interface if it
      already exists.
      """
      ethPhyIntfConfigDir = ethPhyIntfConfigSliceDir[ self.sliceName() ]
      assert not self.status()
      assert 'Sysdb' not in IntfCli.intfConfigRequestors( ethPhyIntfConfigDir,
                                                          self.name )
      IntfCli.cliDeleteIntfConfigRequest( ethPhyIntfConfigDir, self.name )

   def deleteInterfaceConfiguration( self ):
      ethPhyIntfConfigDir = ethPhyIntfConfigSliceDir[ self.sliceName() ]
      del ethPhyIntfConfigDir.intfConfig[ self.name ]

   def counter( self ):
      counterDir = self.getIntfCounterDir()
      if counterDir and counterDir.intfCounterDir.get( self.name ):
         return counterDir.intfCounterDir[ self.name ].intfCounter[ 'current' ]
      qt8( "EthPhyIntf.counter:", qv( self.name ), "is None" )
      # Read counter from intfStatus.counter
      counter = IntfCli.Intf.counter( self )
      if not counter:
         qt8( "EthPhyIntf.counter:", qv( self.name ),
              "No counter found in status, returning default counter" )
         counter = ZeroEthIntfCounter( self.name )
      return counter

   def getIntfCounterDir( self ):
      """
      Returns the EthIntfCounterDir object for this interface.
      """
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return ethPhyIntfCounterDir_

   #----------------------------------------------------------------------------
   # Utility functions used by showPhysical().
   #----------------------------------------------------------------------------
   def addrStr( self ):
      if self.status().forwardingModel == 'intfForwardingModelRouted':
         addr = self.routedAddr()
         if self.routedAddr() != self.addr():
            # If an interface has a different address, it is probably an
            # l2 port, so the l2 and bia are just information overload.
            # If the addresses are the same, it is probably a management
            # port, so the bia is relevant.
            return "address is %s" % addr
      else:
         addr = self.addr()
      return "address is %s (bia %s)" % ( addr, self.burnedInAddr() )

   def getIntfStatusModel( self ):
      return EthIntfModel.EthPhyInterfaceStatus( name=self.status().intfId )

   def hardware( self ):
      return "ethernet"

   def burnedInAddr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay(
         self.status().burnedInAddr )

   def showLinkTypeSpecific( self, intfStatusModel ):
      intfStatusModel.duplex = self.getIntfDuplexStr()
      intfStatusModel.autoNegotiate = \
                                         self.autonegStateStr( longStr=True ).lower()
      # pylint: disable-next=protected-access
      intfStatusModel._autoNegotiateActive = self.autonegActive()
      if self.showLanes():
         laneCount = self.getLaneCount()
      else:
         # If interface only supports legacy speeds, do not show lane count.
         laneCount = EthLaneCount.laneCountUnknown
      intfStatusModel.lanes = EthTypesApi.laneCountNumber( laneCount )
      if self.unidirectionalLinkSupported():
         # pylint: disable-next=protected-access
         intfStatusModel._uniLinkMode = self.showUnidirectionalLinkMode()
      else:
         # pylint: disable-next=protected-access
         intfStatusModel._uniLinkMode = 'n/a'

      # pylint: disable-msg=W0212
      return ( intfStatusModel.duplex, intfStatusModel.autoNegotiate,
               intfStatusModel._autoNegotiateActive )
      # pylint: enable-msg=W0212

   def getIntfDuplexStr( self ):
      '''Returns the configured EthDuplex for the interface.'''

      duplex = self.status().duplex
      # V3.0 and newer systems always show "Unconfigured" for inactive interfaces.
      if ( self.l1PolicyVersion() >= L1PolicyVersion( 3, 0 ) and
           ( not self.active() ) and
           ( self.getLinkMode() == EthLinkMode.linkModeUnknown ) ):
         duplex = EthDuplex.duplexUnknown

      return duplex

   #----------------------------------------------------------------------------
   # Utility functions used by "showInterfacesFlowcontrol()"
   #----------------------------------------------------------------------------
   def flowcontrolRxPause( self ):
      counter = self.counter()
      if counter is None:
         qt8( "EthPhyIntf-flowcontrolRxPause:", qv( self.name ), "is None" )
         return 0
      sysdbValue = getattr( counter.ethStatistics, 'inPauseFrames', None )
      if sysdbValue is None:
         return 0
      ckpt = self.getLatestCounterCheckpoint( )
      if ckpt:
         checkpointValue = getattr( ckpt.ethStatistics, 'inPauseFrames', 0 )
      else:
         checkpointValue = 0
      return sysdbValue - checkpointValue

   def flowcontrolTxPause( self ):
      counter = self.counter()
      if counter is None:
         qt8( "EthPhyIntf-flowcontrolTxPause:", qv( self.name ), "is None" )
         return 0
      sysdbValue = getattr( counter.ethStatistics, 'outPauseFrames', None )
      if sysdbValue is None:
         return 0
      ckpt = self.getLatestCounterCheckpoint( )
      if ckpt:
         checkpointValue = getattr( ckpt.ethStatistics, 'outPauseFrames', 0 )
      else:
         checkpointValue = 0
      return sysdbValue - checkpointValue

   def internalSis( self ):
      if isModular():
         sliceName = self.sliceName()
      else:
         sliceName = "FixedSystem"
      internalSissDir = internalSissSliceDir.get( sliceName )
      if internalSissDir:
         for subdomain in internalSissDir.values():
            internalSiss = subdomain.fetchSlot( self.name )
            if internalSiss:
               return internalSiss.subdomainIntfStatus.get( self.name )
      return None

   def speedGroupConfigDir( self ):
      return SpeedGroupCli.speedGroupConfigSliceDir[ self.slotId() ]

   def speedGroupStatusDir( self ):
      speedGroupStatusSlice = SpeedGroupCli.speedGroupStatusSliceDir.get(
            self.slotId() )
      if speedGroupStatusSlice:
         for speedGroupStatusDir in speedGroupStatusSlice.values():
            group = speedGroupStatusDir.intfSpeedGroup.get( self.name )
            if group:
               return speedGroupStatusDir
      return None

   def speedGroupStatus( self ):
      for linecard in SpeedGroupCli.speedGroupStatusSliceDir.values():
         for speedGroupStatusDir in linecard.values():
            group = speedGroupStatusDir.intfSpeedGroup.get( self.name )
            if group:
               return speedGroupStatusDir.group.get( group )
      return None

   def linkModeCapabilitiesToList( self ):
      ''' return : List of CapList '''
      lmCaps = []
      if not self.intfXcvrPresent():
         return lmCaps
      # The Capabilities::ResolverSm writes phyAutonegCapabilities which AutonegSm
      # will copy into autonegCapabilities. So, the two are generally equivalent.
      # Some btests set phyAutonegCapabilities and the do not run the
      # AutonegSm, others set autonegCapabilities.
      # For that reason, we need to check both here.
      # Eventually we can remove phyAutonegCapabilities and switch
      # all usages to autonegCapabilities.  phyAutonegCapabilities defaults to
      # anegModeUnknown. After the resolver runs it will update it to the supported
      # anegMode or anegModeUnsupported.  In keeping with past behavior we will
      # consider both anegModeUnknown and anegModeUnsupported as indicating the
      # interface is not capable of autoneg and therefore not display auto in
      # the capabilities.
      autonegCapable = (
         ( self.status().phyAutonegCapabilities.mode != 'anegModeUnknown' and
           self.status().phyAutonegCapabilities.mode != 'anegModeUnsupported' ) or
         ( self.status().autonegCapabilities.mode != 'anegModeUnknown' and
           self.status().autonegCapabilities.mode != 'anegModeUnsupported' ) )
      caps = self.status().linkModeCapabilities

      # For non-hotswappable BASE-T transceivers, xcvr is always present even
      # though capabilities may not yet be available. The condition makes sure that
      # we return 'Unknown' in these cases.
      # All fixed BASE-T ports support autoneg so we can just check
      # if autonegCapable is False here and then conclude the BASE-T PHY
      # has not yet registered capabilities.
      if ( not autonegCapable and self.intfXcvrStatus().fixed and
           'BASE-T' in self.intfXcvrStatus().xcvrType ):
         lmCaps.append( EthIntfLib.CapList ( 'speedUnknown', 'laneCountUnknown',
                                             False, 'duplexUnknown', False,
                                             False ) )
         return lmCaps

      # This should be consistent with how autonegLib determines when autoneg is
      # active with a default configuration.
      anegMode = self.status().autonegCapabilities.mode
      autonegByDefault = ( ( anegMode == AutonegMode.anegModeClause28 or
                             ( anegMode == AutonegMode.anegModeClause37 and
                               caps.exclusive1GbpsFull ) or
                             ( anegMode == AutonegMode.anegModeClause73 and
                               self.internalSis() and
                               self.internalSis().medium == "backplane" ) ) and
                           self.status().speedAutoAdj )
      groupStatus = self.speedGroupStatus()
      imcompatElms = None
      if groupStatus and groupStatus.setting:
         speedCompat = TacLazyType( 'Interface::SpeedCompatSetting' )
         cl28Cl37Compat = groupStatus.setting.get( speedCompat.compatibility10g )
         if autonegByDefault and cl28Cl37Compat:
            # Default speed is shown as 'auto' if interface supports autoneg at 1g
            # and serdes speed is set to 10g ( 1g and 10g are compatible ).
            defaultMode = 'auto'
         else:
            # For SFP28 port, when serdes speed is set, determine the default
            # linkMode based on serdes speed settings and linkModeCapabilities
            if groupStatus.masterIntf:
               defaultMode = groupStatus.defaultLinkMode( caps )
            else:
               defaultMode = self.status().defaultLinkMode(
                     *self.linkModePriorityLists() )
         incompatMode = set( groupStatus.supportedModes ) - \
                        set( groupStatus.setting )
         imcompatElms = EthLinkModeSet()
         for mode in incompatMode:
            imcompatElms |= groupStatus.supportedModes[ mode ]
      else:
         if autonegByDefault:
            defaultMode = 'auto'
         else:
            defaultMode = self.status().defaultLinkMode(
                  *self.linkModePriorityLists() )
         # Handle SFP28 case where master interface drives serdes setting.
         if groupStatus and groupStatus.masterIntf and \
            self.name != groupStatus.masterIntf:
            if self.ethIntfStatuses.get( groupStatus.masterIntf ).speed == \
                  EthSpeed.speed25Gbps:
               imcompatElms = EthLinkModeSet( mode1GbpsFull=True,
                                              mode10GbpsFull=True )
            else:
               imcompatElms = EthLinkModeSet( mode25GbpsFull=True )

      # TODO: BUG955581, need to implement default mode for new policy
      if self.l1PolicyVersion() >= L1PolicyVersion( 3, 0 ):
         defaultMode = None

      return EthIntfLib.linkModeCapabilitiesToList( caps,
                                                    autonegCapable=autonegCapable,
                                                    defaultMode=defaultMode,
                                                    showLanes=self.showLanes(),
                                                    imcompatElms=imcompatElms )

   def autonegCapabilitiesMode( self ):
      return self.status().autonegCapabilities.mode

   def autonegCapabilitiesToList( self ):
      ''' return : List of CapList '''
      anegCapsList = EthIntfLib.linkModeCapabilitiesToList(
               self.status().autonegCapabilities.linkModes,
               showLanes=self.showLanes() )
      if not anegCapsList:
         anegCapsList = [ EthIntfLib.CapList( 'speedUnknown', 'laneCountUnknown',
                                              False, 'duplexUnknown', False,
                                              False ) ]
      return anegCapsList

   #----------------------------------------------------------------------------
   # Special case to allow users to configure the link mode *only* on
   # 1000BASE-T SFPs, into one of three values:
   #
   # o autoneg, 1000/full  (Default)
   # o autoneg, 100/full
   # o force link at 100/full (No autoneg)
   #
   # This really ought to be unified with "setLinkMode" above, but
   # since these settings affect all our platforms (all PHYs and all
   # MACs), and this is going into a release with fairly short notice,
   # defer that work (and testing!) until later.
   # ----------------------------------------------------------------------------
   def setLinkModeSfp100BaseTAuto100Full( self ):
      """
      Special case to allow users to configure the link mode *only* on
      1000BASE-T SFPs, into one of three values:

      o autoneg, 1000/full  (Default)
      o autoneg, 100/full
      o force link at 100/full (No autoneg)

      This really ought to be unified with "setLinkMode" above, but
      since these settings affect all our platforms (all PHYs and all
      MACs), and this is going into a release with fairly short notice,
      defer that work (and testing!) until later.
      """
      desiredLinkMode = 'linkModeAutoneg'
      advertisedMode = EthLinkModeSet()
      advertisedMode.mode100MbpsFull = True
      if not self.isLinkModeSupported( desiredLinkMode, advertisedMode ):
         return

      if self.lookupPhysical() and self.intfXcvrPresent():
         if self.intfXcvrStatus().xcvrType != '1000BASE-T':
            # Actually, this test won't catch the case where this
            # command is entered on native RJ45 1000BASE-T ports,
            # but that is a misconfig, and the failure mode is that
            # those ports will autoneg to their max capabilities.
            errorMsg = self.linkModeUnsupportedErrorMsg()
            if ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.speed:
               self.mode_.addWarning( errorMsg )
               prompt = "Do you wish to proceed with this command? [y/N]"
               if not BasicCliUtil.confirm( self.mode_, prompt,
                                            answerForReturn=False ):
                  return
            else:
               self.mode_.addError( errorMsg )
               return

      # Disable auto-100/full or forced-100/full, and just go back to
      # the SFP's default capabilities -- auto-1000/full. (In fact,
      # note that we don't even verify that there is config state for
      # this interface, or an xcvr is present, etc.)
      cfg = self.config()

      elms = EthLinkModeSet()
      elms.mode100MbpsFull = True
      self.setAdvertisedModes( elms )
      cfg.advertisedModesConfiguredForSfp1000BaseT = True

      cfg.linkModeLocal = desiredLinkMode

   def setAutonegStandard( self, aneg25gModes=None ):
      """
      Sets the autoneg 25g mode of the interface to a specified value or the
      default value.
      """
      negotiation25gMode = Tac.Value( "Interface::Negotiation25gMode" )

      if aneg25gModes:
         # SetRule returns a tuple of names and keywords,
         # the names in this case are names of TAC attributes.
         for attr in aneg25gModes:
            setattr( negotiation25gMode, attr, True )
      else:
         # This configures all Negotiation Modes to default setting,
         # which is consortium extension and ieee standard both 'On'
         negotiation25gMode.consortium25g = True
         negotiation25gMode.ieee25g = True

      self.config().negotiation25gModeConfig = negotiation25gMode

   def autonegStandardSupported( self, aneg25gMode ):
      return getattr( self.status().aneg25gModeCaps, aneg25gMode )

   def setFlowcontrol( self, direction, value ):
      """
      Sets the flowcontrol of the interface to a specified value or the default
      value.
      """
      cfg = self.config()
      if self.lookupPhysical():
         status = self.status()
         caps  = status.autonegCapabilities
         flowControlValid = {
            ( 'send', 'flowControlConfigUnknown' ) : True,
            ( 'send', 'flowControlConfigOff' ) : True,
                  ( 'send', 'flowControlConfigOn' ) :
            status.txFlowcontrolCapabilities != 'flowControlNotCapable' ,
            ( 'send', 'flowControlConfigDesired' ) :
                  caps.txPause and
                  status.txFlowcontrolCapabilities != 'flowControlNotCapable',
            ( 'receive', 'flowControlConfigUnknown' ) : True,
            ( 'receive', 'flowControlConfigOff' ) : True,
            ( 'receive', 'flowControlConfigOn' ) :
                  status.rxFlowcontrolCapabilities != 'flowControlNotCapable' ,
            ( 'receive', 'flowControlConfigDesired' ) :
                  caps.rxPause and
                  status.rxFlowcontrolCapabilities != 'flowControlNotCapable'
            }
         if not flowControlValid[ ( direction, value ) ]:
            self.mode_.addError( 'Flow control setting is not compatible with '
                                 'interface %s.' % self.name )
            return

      if direction == 'send':
         cfg.txFlowcontrolLocal = value
      else:
         assert direction == 'receive'
         cfg.rxFlowcontrolLocal = value

   def setCongestionDropsLoggingMode( self, logMode ):
      self.config().congestionDropsLoggingMode = logMode

   def setL2Mtu( self, value ):
      """
      Sets the L2 Mtu value
      """
      cfg = self.config()
      cfg.l2Mtu = value

   @staticmethod
   def getAllPhysical( mode ):
      """
      Returns an unsorted list of EthPhyIntf objects representing all the existing
      Interface::EthPhyIntfStatus objects.
      """
      intfs = []
      for name in ethPhyIntfStatusDir.intfStatus:
         skippedIntfs = (
            # Skip internal interfaces
            # BUG944 - need a more general way of doing this. Should 'internal' be
            # an attribute of the IntfConfig intead?
            'Internal',
            # Handled in UnconnectedEthPhyIntf.
            'UnconnectedEthernet',
            # Handled by the 'CpuIntf' class in CpuIntfCli
            'Cpu',
            # Handled by the 'FpgaManagement' class in the MakoIntf package.
            'FpgaManagement', 'FpgaFunction',
            # Handled by the 'TrafficGen' class in the MakoIntf package.
            'TrafficGen',
            # Handled by the 'SwitchIntf' class in SwitchIntfCli
            'Switch',
            # Handled by the corresponding CLI plugins in the MetaWatch package.
            'WatchIn', 'WatchOut', 'WatchPassthrough', 'Ptp', 'WhiteRabbit',
            # Handled by the 'MetaMuxInIntf' and 'MetaMuxOutIntf' classes in the
            # MetaMux package.
            'MuxIn', 'MuxOut',
            # Handled by the 'DownstreamIntf' and 'UpstreamIntf' classes in the
            # MultiAccess package.
            'Downstream', 'Upstream',
            # See BUG901479 : Handled by FabricIntfCli
            'Fabric' )
         if not name.startswith( skippedIntfs ):
            intf = EthPhyIntf( name, mode )
            if intf.lookupPhysical():
               intfs.append( intf )
      return intfs

   def clearCounters( self, sessionOnly=False ):
      ''' Hooks/support for "clear counters". '''
      ckpt = self.getCounterCheckpoint( className="Interface::EthIntfCounterBase",
                                        sessionOnly=sessionOnly )
      if ckpt is None or self.counter() is None:
         return
      ckpt.ethStatistics = self.counter().ethStatistics
      EthIntf.clearCounters( self, sessionOnly=sessionOnly )

   def switchportEligible( self ):
      ''' Is this port eligible for "switchport" commands? '''
      return self.name.startswith( "Ethernet" )

   def setDefault( self ):
      """
      First run through the registered hooks and confirm that we can go back
      to the default link mode.
      """
      if checkChangeCancelHooks( self.mode_, [ self ], 'linkModeUnknown', None ):
         return
      if checkChangeCancelOnSfp28( self.mode_, [ self ], 'linkModeUnknown', None ):
         return

      self.setFlowcontrol( 'send', 'flowControlConfigUnknown')
      self.setFlowcontrol( 'receive', 'flowControlConfigUnknown')
      self.setMacAddr( None )
      self.setL2Mtu( 0 )
      self.setTimestampMode( 'timestampModeDisabled' )
      self.setLinkMode( 'linkModeUnknown' )
      self.setLinkDebounceTimes( 0, 0 )
      self.setCongestionDropsLoggingMode( 'useGlobal' )
      self.setUnidirectionalLinkMode( 'uniLinkModeDisabled' )
      self.setL2Mtu( 0 )
      self.setL2Mru( 0 )
      self.setErrorCorrectionEncoding( noOrDefault='default' )
      self.noDefaultErrorCorrectionBypass()
      self.setAutonegStandard()
      super().setDefault()

   def defaultShutdownConfig( self ):
      if EthIntfId.isEthernet( self.name ):
         return IntfCli.globalIntfConfig.defaultEthernetShutdown
      else:
         return False

   #----------------------------------------------------------------------------
   # Does this port support UnidirectionalLinkMode?
   #----------------------------------------------------------------------------
   def unidirectionalLinkSupported( self ):
      ''' Does this port support UnidirectionalLinkMode? '''
      return self.status() and self.status().unidirectionalLinkSupported

   def setUnidirectionalLinkMode( self, unidirectionalLinkMode ):
      ''' Hooks/support for "unidirectional" config commands'''
      self.config().unidirectionalLinkMode = unidirectionalLinkMode

   uniLinkModeStrings = {
         'uniLinkModeSendOnly' : 'send-only',
         'uniLinkModeReceiveOnly' : 'receive-only',
         'uniLinkModeSendReceive' : 'send-receive',
         'uniLinkModeDisabled' : 'disabled',
   }

   def showUnidirectionalLinkMode( self ):
      ''' Hooks/support for "show interfaces [<name>] unidirectional" '''
      uniLinkMode = self.config().unidirectionalLinkMode
      return EthPhyIntf.uniLinkModeStrings[ uniLinkMode ]

   def coherentErrorCorrectionSupported( self ):
      if 'coherent' in errorCorrectionShCapFns:
         return errorCorrectionShCapFns[ 'coherent' ].checkFn( self.name )
      return None

   def cmisCoherentErrorCorrectionSupported( self ):
      if 'coherent' in errorCorrectionShCapFns:
         return errorCorrectionShCapFns[ 'coherent' ].checkFn( self.name,
                                                               checkCmisSlots=True )
      return None

   def coherentErrorCorrectionEncodingSupp( self, encoding ):
      allPhyCoherentStatus = {}
      # Get all coherent status' at once.
      if 'Modulation' in interfaceCapabilityFns:
         allPhyCoherentStatus = interfaceCapabilityFns[ 'Modulation' ].getAll()

      if ( 'coherent' in errorCorrectionShCapFns and
           allPhyCoherentStatus.get( self.name ) ):
         return errorCorrectionShCapFns[ 'coherent' ].checkFec(
                   allPhyCoherentStatus[ self.name ].fecCapabilities, encoding )
      return None

   def errorCorrectionBypassSupported( self, bypassValue, ignoreOutputErrors=True ):
      if not self.errorCorrectionEncodingSupported( 'reed-solomon',
                                                    ignoreOutputErrors ):
         return False

      if bypassValue and bypassValue not in EthIntfLib.tokenToBypassAttr:
         self.mode_.addError( "Invalid bypass value %s" % bypassValue )
         return False

      if self.lookupPhysical():
         supported = getattr( self.status().fecBypassCapabilities,
                              EthIntfLib.tokenToBypassAttr[ bypassValue ] )
         if not supported and not ignoreOutputErrors:
            errorMsg = (
               'Error-correction bypass is unsupported for interface %s.' %
               self.name )
            supported = self.maybeAllowUnsupportedConfig(
               ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.errorCorrection,
               self.mode_, errorMsg )
         return supported
      else:
         return True

   FecBypassMode = TacLazyType( 'Interface::EthFecBypassMode' )
   def setErrorCorrectionBypass( self, bypassValue ):
      if bypassValue and not self.errorCorrectionBypassSupported( bypassValue,
                                                                  False ):
         return
      self.config().fecBypass = EthIntfLib.tokenToBypassAttr[ bypassValue ]

   def noDefaultErrorCorrectionBypass( self ):
      self.config().fecBypass = EthPhyIntf.FecBypassMode.fecBypassDisabled

   def _populateCountersErrorsModel( self, errors, attributes ):
      def stat( counter, checkpoint, attribute ):
         if counter is None:
            # Unders some circumstances (e.g. a linecard being hotswapped),
            # the counter argument may be None. In theory, the right thing
            # to do in this case would be to return notFoundString.
            # This would however cause some non-optional counter attributes
            # in the interface counters model (e.g., inOctets, inUcastPkts, etc.)
            # to be missing. Because of that we for now simply return 0.
            # Mark as BUG159812
            return 0

         sysdbValue = getattr( counter.ethStatistics, attribute, None )
         if sysdbValue is None:
            sysdbValue = getattr( counter.statistics, attribute, None )
         if sysdbValue is None:
            return 'n/a'
         if checkpoint:
            checkpointValue = getattr( checkpoint.ethStatistics, attribute, None )
            if checkpointValue is None:
               checkpointValue = getattr( checkpoint.statistics, attribute, 0 )
         else:
            checkpointValue = 0
         return sysdbValue - checkpointValue
      checkpoint = self.getLatestCounterCheckpoint()
      counter = self.counter()
      for attribute in attributes:
         setattr( errors, attribute, stat( counter, checkpoint, attribute ) )

   def getCountersErrorsModel( self ):
      errors = ErrorCounters()
      attributes = ( 'fcsErrors', 'alignmentErrors', 'symbolErrors', 'inErrors',
                     'frameTooShorts', 'frameTooLongs', 'outErrors', )
      self._populateCountersErrorsModel( errors, attributes )
      return errors

   def getCountersHalfDuplexErrorsModel( self ):
      errors = HalfDuplexErrorCounters()
      attributes = ( 'singleCollisionFrames', 'multipleCollisionFrames',
                     'lateCollisions', 'excessiveCollisions',
                     'deferredTransmissions', )
      self._populateCountersErrorsModel( errors, attributes )
      return errors

   def countersSupported( self ):
      return self.status() and self.status().countersSupported

   def countersErrorsSupported( self ):
      return self.status() and self.status().countersErrorsSupported

   def isPmaInterface( self ):
      return self.status() and self.status().isPmaInterface

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as a type of physical interface.
# Routing protocol is not supported on management interfaces, and hence excluded
# in ruleWithRoutingSupport
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( EthPhyIntf,
                               ( EthPhyAutoIntfType, MgmtAutoIntfType ),
                               withIpSupport=True,
                               withRoutingProtoSupport=True,
                               matcherWithRoutingProtoSupport=EthPhyIntf.ethMatcher )

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as a type that supports L2 mtu.
#-------------------------------------------------------------------------------
registerL2MtuValidIntfType( EthPhyIntf )

class EthPhyDefaultCaps:
   ''' This class represents an EthPhyDefaultCapabilities entity. '''

   def __init__( self, name, parentDir ):
      t0( "EthPhyIntf::__init__", name )
      self.name = name
      self.ethPhyDefaultCapsDir = parentDir

   def showLanes( self ):
      return EthPhyIntf._showLanes( self.capabilities() )

   def linkModeCapabilitiesToList( self ):
      return EthIntfLib.linkModeCapabilitiesToList(
                     self.capabilities().linkModeCapabilities,
                     showLanes=self.showLanes() )

   def checkForAutoneg( self ) -> bool:
      anegCaps = self.capabilities().autonegCapabilities
      autonegModes = ( AutonegMode.anegModeClause28,
                       AutonegMode.anegModeClause37,
                       AutonegMode.anegModeClause73 )
      for anegMode in autonegModes:
         if ( anegMode in anegCaps and
              anegCaps[ anegMode ].mode == anegMode ):
            return True
      return False

   # return : List of CapList
   def autonegCapabilitiesToList( self, autonegMode ):
      ''' return : List of CapList '''
      anegCaps = self.capabilities().autonegCapabilities
      if autonegMode not in anegCaps:
         return []

      # we never resolve to anegModeUnknown
      assert anegCaps[ autonegMode ].mode != AutonegMode.anegModeUnknown

      anegLinkModes = anegCaps[ autonegMode ].linkModes
      return EthIntfLib.linkModeCapabilitiesToList( anegLinkModes,
                                                    showLanes=self.showLanes() )

   def anegCl73ModeCapabilitiesToList( self, negCl73Mode ):
      ''' return : List of CapList '''
      anegCl73Caps = []
      if AutonegMode.anegModeClause73 not in self.capabilities().autonegCapabilities:
         return []

      # we never resolve to anegModeUnknown
      assert self.capabilities().autonegCapabilities[ \
                  AutonegMode.anegModeClause73 ] != AutonegMode.anegModeUnknown

      clause73LinkModes = self.capabilities().autonegCapabilities[ \
            AutonegMode.anegModeClause73 ].linkModes

      if negCl73Mode == 'ieee':
         # Both 25G and 50G Clause 73 Auto-Negotiation are not always supported
         # by hardware, so we must check our aneg25gModeCaps before displaying
         # those speeds.
         IeeeSupport = self.capabilities().aneg25gModeCaps.ieee25g
         for mode, supported, speed, lanes in [
               ( clause73LinkModes.mode10GbpsFull, True, EthSpeed.speed10Gbps,
                 EthLaneCount.laneCount1 ),
               ( clause73LinkModes.mode25GbpsFull, IeeeSupport, EthSpeed.speed25Gbps,
                 EthLaneCount.laneCount1 ),
               ( clause73LinkModes.mode40GbpsFull, True, EthSpeed.speed40Gbps,
                 EthLaneCount.laneCount4 ),
               ( clause73LinkModes.mode50GbpsFull1Lane, IeeeSupport,
                 EthSpeed.speed50Gbps, EthLaneCount.laneCount1 ),
               ( clause73LinkModes.mode100GbpsFull, True, EthSpeed.speed100Gbps,
                 EthLaneCount.laneCount4 ),
               ( clause73LinkModes.mode100GbpsFull2Lane, True, EthSpeed.speed100Gbps,
                 EthLaneCount.laneCount2 ),
               ( clause73LinkModes.mode100GbpsFull1Lane, True, EthSpeed.speed100Gbps,
                 EthLaneCount.laneCount1 ),
               ( clause73LinkModes.mode200GbpsFull4Lane, True, EthSpeed.speed200Gbps,
                 EthLaneCount.laneCount4 ),
               ( clause73LinkModes.mode200GbpsFull2Lane, True, EthSpeed.speed200Gbps,
                 EthLaneCount.laneCount2 ),
               ( clause73LinkModes.mode400GbpsFull4Lane, True, EthSpeed.speed400Gbps,
                 EthLaneCount.laneCount4 ) ]:
            if mode and supported:
               anegCl73Caps.append( EthIntfLib.CapList( speed, lanes,
                                                        self.showLanes(), "full",
                                                        False, False ) )

      elif negCl73Mode == 'consortium':
         # These speeds can support consortium and IEEE
         if self.capabilities().aneg25gModeCaps.consortium25g:
            for mode, speed, lanes in [
                  ( clause73LinkModes.mode25GbpsFull, EthSpeed.speed25Gbps,
                    EthLaneCount.laneCount1 ),
                  ( clause73LinkModes.mode50GbpsFull, EthSpeed.speed50Gbps,
                    EthLaneCount.laneCount2 ) ]:
               if mode:
                  anegCl73Caps.append( EthIntfLib.CapList( speed, lanes,
                                                           self.showLanes(), "full",
                                                           False, False ) )
         # These speeds only currently support consortium, do not base on
         # consortium25g
         for mode, speed, lanes in [
               ( clause73LinkModes.mode400GbpsFull8Lane, EthSpeed.speed400Gbps,
                 EthLaneCount.laneCount8 ),
               ( clause73LinkModes.mode800GbpsFull8Lane, EthSpeed.speed800Gbps,
                 EthLaneCount.laneCount8 ) ]:
            if mode:
               anegCl73Caps.append( EthIntfLib.CapList( speed, lanes,
                                                        self.showLanes(), "full",
                                                        False, False ) )

      else:
         # If we hit this case, perhaps some new consortium Mode was added
         assert False, "Unrecognized Clause 73 Mode " + negCl73Mode

      return anegCl73Caps

   def errorCorrectionEncodings( self ):
      capableEncodingTokens = []
      for token, attrs in EthIntfLib.tokenToFecEncodingAttrs.items():
         for attr in attrs:
            for fecCapabilities in self.capabilities().fecCapabilities.values():
               if( ( getattr( fecCapabilities, attr ) ) and
                   ( token not in capableEncodingTokens ) ):
                  capableEncodingTokens.append( token )
      return capableEncodingTokens

   def errorCorrectionLinkModes( self, encoding ):
      """
      input: encoding, shall be value of errorCorrectionEncodingToLbls
      return: List of FecCapList
      The CapabilitiesResolver checks that the linkModes in FEC capabilities are
      actually supported in linkModeCapabilities. Therefore we do not need
      to cross check the two again.
      """
      linkModeToEncodings = {}
      for linkMode, fecCapsSet in self.capabilities().fecCapabilities.items():
         for attr in EthIntfLib.tokenToFecEncodingAttrs[ encoding ]:
            if getattr( fecCapsSet, attr ):
               if linkMode not in linkModeToEncodings:
                  linkModeToEncodings.update( { linkMode : attr } )

      # Sort based on the order of keys in linkModeToSpeedLanesDuplex
      linkModeCap = []
      showLanes = self.showLanes()
      for linkMode in EthIntfLib.linkModeToSpeedLanesDuplex:
         if linkMode in linkModeToEncodings:
            speed, lanes, duplex = EthIntfLib.linkModeToSpeedLanesDuplex[ linkMode ]
            duplex = duplex[ len( 'duplex' ): ].lower()
            linkModeCap.append( EthIntfLib.FecCapList(
                                                linkModeToEncodings[ linkMode ],
                                                speed, lanes, showLanes,
                                                False, duplex ) )
      return linkModeCap

   @Tac.memoize
   def capabilities( self ):
      return self.ethPhyDefaultCapsDir.portCaps[ self.name ]

   @staticmethod
   def getAllIntfCapsInfo():
      allEthPhyDefaultCaps = {}
      for subDirs in ethPhyDefaultCapsSliceDir.values():
         for ethPhyDefaultCapsDir in subDirs.values():
            for intfName in ethPhyDefaultCapsDir.portCaps:
               allEthPhyDefaultCaps[ intfName ] = \
                     EthPhyDefaultCaps( intfName, ethPhyDefaultCapsDir )
      return allEthPhyDefaultCaps

intfCapsInfoClasses_ = set()
def registerEthPhyIntfCapsInfoClass( cls ):
   intfCapsInfoClasses_.add( cls )

def getIntfCapsInfo( intfs ) -> list[ tuple[ EthPhyIntf, EthPhyDefaultCaps ] ]:
   allEthPhyDefaultCaps = {}
   for cls in intfCapsInfoClasses_:
      allEthPhyDefaultCaps.update( cls.getAllIntfCapsInfo() )

   intfCapsInfos = []
   for intf in intfs:
      # There should never be a case where an EthPhyDefaultCapabilities object
      # is missing from Sysdb. It should be invariant that there is a one-to-one
      # matching of EthPhyIntfStatus to EthPhyDefaultCapabilities; however, to
      # be absolutely safe we check anyways
      if intf.name in allEthPhyDefaultCaps:
         intfCapsInfos.append( ( intf, allEthPhyDefaultCaps[ intf.name ] ) )

   return intfCapsInfos

#-------------------------------------------------------------------------------
# Register the EthPhyDefaultCaps class as a type that provides
# getAllIntfCapsInfo.
#-------------------------------------------------------------------------------
registerEthPhyIntfCapsInfoClass( EthPhyDefaultCaps )

def ueGuard( mode, token ):
   ''' Guard for UnconnectedEthernet when there are none on the system '''
   if UnconnectedEthPhyIntf.getAllPhysical( mode ):
      return None
   return CliParser.guardNotThisPlatform

class UnconnectedEthPhyIntf( EthPhyIntf ):
   '''A subclass of EthPhyIntf class for physical Unconnected Ethernet interfaces.'''

   def __init__( self, name, mode ):
      ''' Creates a new UnconnectedEthPhyIntf instance of the specified name. '''
      t0( "UnconnectedEthPhyIntf::__init__", name )
      # Deal with the fact that short prefix of UnconnectedEthernet is Ue (not Un)
      if name.lower().startswith( 'ue' ):
         suffix = name[ len( 'ue' ): ]
         name = 'UnconnectedEthernet' + suffix
      EthPhyIntf.__init__( self, name, mode )

   def isLinkModeSupported( self, desiredLinkMode, advertisedModes=None ):
      if desiredLinkMode != 'linkModeUnknown' and self.lookupPhysical():
         if not self.status().linkModeSupported[ desiredLinkMode ]:
            # desired linkMode is not supported
            self.mode_.addError( 'Speed and duplex settings are not available on '
                                 'interface %s.' % self.name )
            return False
         elif( desiredLinkMode == 'linkModeAutoneg' and
               not self.isAdvertisedModesSupported( advertisedModes ) ):
            self.mode_.addError( 'Cannot advertise desired speeds and '
                                 'duplex settings for interface %s.'
                                 % self.name )
            return False
      return True

   def switchportEligible( self ):
      return True

   #----------------------------------------------------------------------------
   # The rule for matching Ethernet unconnected interface names.  When this pattern
   # matches, it returns an instance of the UnconnectedEthPhyIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = PhysicalIntfRule.PhysicalIntfMatcher( 'UnconnectedEthernet',
                                              alternates=[ 'Ue' ],
                                              value=lambda mode, intf:
                                              UnconnectedEthPhyIntf( intf, mode ),
                                              guard=ueGuard )

   def linkModeCapabilitiesToList( self ):
      # There are no xcvrs on unconnected ports
      assert not self.intfXcvrPresent()

      # Just return the default link mode
      caps = self.status().linkModeCapabilities
      defaultMode = self.config().defaultConfig.linkModeLocal
      return EthIntfLib.linkModeCapabilitiesToList( caps, autonegCapable=False,
                                                    defaultMode=defaultMode,
                                                    showLanes=self.showLanes() )

   @staticmethod
   def getAllPhysical( mode ):
      """
      Returns an unsorted list of UnconnectedEthPhyIntf objects representing all the
      existing Interface::EthPhyIntfStatus objects for unconnected ports.
      """
      intfs = []
      for name in ethPhyIntfStatusDir.intfStatus:
         if not name.startswith( "UnconnectedEthernet" ):
            continue
         intf = UnconnectedEthPhyIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

#-------------------------------------------------------------------------------
# Register the UnconnectedEthPhyIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( UnconnectedEthPhyIntf,
                                  UnconnectedEthPhyAutoIntfType,
                                  withIpSupport=True,
                                  withRoutingProtoSupport=True )

IntfRange.registerIntfTypeGuard( UnconnectedEthPhyAutoIntfType, ueGuard )

#-------------------------------------------------------------------------------
# Register the EthPhyIntf class as able to participate in interface ranges.
#-------------------------------------------------------------------------------

def _updateEthIntfHelpDescForModular():
   """ Updates the EthPhyAutoIntfType to have the correct help description
   for slot/port numbers. """
   EthPhyAutoIntfType.helpDesc = [ 'Slot number', 'Port number' ]

FruCli.registerModularSystemCallback( _updateEthIntfHelpDescForModular )

EthPhyAutoIntfType.registerEthPhyIntfClass( EthPhyIntf )
UnconnectedEthPhyAutoIntfType.registerEthPhyIntfClass( UnconnectedEthPhyIntf )

def _updateMgmtPhyIntfHelpDescForModular():
   """ Updates the MgmtPhyAutoIntfType to have the correct help description
   for slot/port numbers. """
   MgmtPhyAutoIntfType.helpDesc = [ 'Slot number', 'Port number' ]
FruCli.registerModularSystemCallback( _updateMgmtPhyIntfHelpDescForModular )

class DataplaneIntfModelet( CliParser.Modelet ):
   """
   Adds dataplan-specific (e.g. not management) CLI commands to the
   "config-if" mode.
   """
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, EthPhyIntf ) and
               mode.intf.switchportEligible() )

class EthIntfModelet( CliParser.Modelet ):
   """
   Adds Ethernet-specific CLI commands to the "config-if" mode.
   """
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, EthPhyIntf )


class MacAddrCmd( CliCommand.CliCommandClass ):
   """
   [ no | default ] mac-address router MAC_ADDR
   [ no | default ] mac-address MAC_ADDR

   Note that "mac-address 0.0.0" will reset the MAC address to the default.
   Remarkably, this is consistent with the industry-standard.
   """
   syntax = 'mac-address ( MAC_ADDR | ( router ( MAC_ADDR | physical ) ) )'
   noOrDefaultSyntax = 'mac-address [ router ] ...'
   data = {
      'mac-address': IntfCli.matcherMacAddress,
      'router': IntfCli.nodeRouter,
      'MAC_ADDR': MacAddr.macAddrMatcher,
      'physical': IntfCli.nodePhysical,
   }

   @staticmethod
   def handler( mode, args ):
      isPhysical = 'physical' in args
      macAddr = None if isPhysical else args[ 'MAC_ADDR' ]
      mode.intf.setMacAddr( macAddr, routerMacConfigured='router' in args,
                            physical=isPhysical )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setMacAddr( None )

EthIntfModelet.addCommandClass( MacAddrCmd )

class LinkDebounceCmd( CliCommand.CliCommandClass ):
   """
   [ no| default ] link-debounce time LINK_DEBOUNCE_TIME [ LINK_DOWN_DEBOUNCE_TIME ]

   Note that a debounce time of 0ms will disable link debouncing. Link
   debouncing is disabled by default.
   """
   syntax = 'link-debounce time LINK_DEBOUNCE_TIME [ LINK_DOWN_DEBOUNCE_TIME ]'
   noOrDefaultSyntax = 'link-debounce ...'
   data = {
      'link-debounce' : 'Configure a link debounce timer for this interface',
      'time' : 'Specify the debounce time period (in milliseconds)',
      'LINK_DEBOUNCE_TIME' : CliMatcher.IntegerMatcher( 0, 1800000,
         helpdesc='Debounce time for both link-up and link-down in milliseconds' ),
      'LINK_DOWN_DEBOUNCE_TIME' : CliMatcher.IntegerMatcher( 0, 1800000,
         helpdesc='Debounce time for link-down only transitions in milliseconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      linkDebounceTime = args[ 'LINK_DEBOUNCE_TIME' ]
      linkDownDebounceOpt = args.get( 'LINK_DOWN_DEBOUNCE_TIME' )

      # Use a seperate link down debounce time only if it has been specified
      if linkDownDebounceOpt is None:
         linkDownDebounceOpt = linkDebounceTime

      # Set the link-up and link-down debounce times for the interface
      mode.intf.setLinkDebounceTimes( linkDebounceTime, linkDownDebounceOpt )
      Tracing.trace0( "Set", mode.intf.name, "link-up debounce time to",
                      str( linkDebounceTime ), "ms and link-down debounce time to ",
                      str( linkDownDebounceOpt ), "ms" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Disable debouncing by setting both link-up and
      # link-down debounce times to zero
      mode.intf.setLinkDebounceTimes( 0, 0 )
      Tracing.trace0( "No Set " + mode.intf.name + " link debounce time" )

EthIntfModelet.addCommandClass( LinkDebounceCmd )

#-------------------------------------------------------------------------------
# The "speed" commands:
#   speed { 100full }
#   speed forced { 10half | 10full | 100half | 100full | 1000half | 1000full }
#   no speed [ forced [ 10half | 10full | 100half | 100full | 1000half | 1000full ] ]
#
#   speed auto
#   no speed auto
#
# BUG96410 : Deprecate the "speed forced" command in favor of "speed <speed>"
#            commands. This will be done in stages with 100Full being the first
#            "speed <speed>" command.
#
# BUG282: allow users to optionally specify the capabilities that are
#         advertised during autoneg, i.e., implement
#
#   speed auto [ 10half, 10full, 100half, 100full, 1000half, 1000full, 10000full ]
#   no speed auto [ 10half, 10full, 100half, 100full, 1000half, 1000full, 10000full ]
#-------------------------------------------------------------------------------
matcherSpeed = CliMatcher.KeywordMatcher( 'speed',
      helpdesc='Configure autoneg and speed/duplex/flowcontrol' )
matcherAuto = CliMatcher.KeywordMatcher( 'auto',
      helpdesc='Enable autoneg for speed, duplex, and flowcontrol' )

def checkChangeCancelHooks( mode, intfList, linkMode, advertisedModes ):
   ''' Execute all the registered hooks for the link mode change and confirm that
       the change can be made. If any of the hooks return true then the user aborted
       the change and the request has to be ignored.
   '''
   if mode.session.commandConfirmation():
      for hook in canPromptForAbortHook.extensions():
         if hook( mode, intfList, linkMode, advertisedModes ):
            Tracing.trace0( "LinkMode change for " +
                            ','.join( intf.name for intf in intfList ) +
                            " cancelled by " + str( hook ) )
            # One of the registered hooks returned true so the linkmode change
            # is cancelled
            return True

   # We are good to go with the link mode change
   return False

def checkChangeCancelOnSfp28( mode, intfList, linkMode, advertisedModes ):
   promptTemplate = "Changing the speed setting on master interface {0} may cause " \
                    "slave interfaces in the speed group to flap."
   warningPrompt = ""
   if mode.session.commandConfirmation():
      masterIntfs = []
      for intf in intfList:
         groupStatus = intf.speedGroupStatus()
         if( groupStatus and
             intf.name == groupStatus.masterIntf and
             not groupStatus.setting and
             linkMode != intf.config().linkModeLocal and
             advertisedModes != intf.config().advertisedModesLocal ):
            masterIntfs.append( groupStatus.masterIntf )
      if masterIntfs:
         masterIntfString = IntfRange.intfListToCanonical( masterIntfs,
                                                           noHoleRange=True )[ 0 ]
         warningPrompt = promptTemplate.format( masterIntfString )

   if warningPrompt:
      mode.addWarning( warningPrompt )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )

      # See if user cancelled the command.
      if not ans:
         intfString = IntfRange.intfListToCanonical( intfList,
                                                     noHoleRange=True )[ 0 ]
         abortMsg = "Command aborted by user for " + intfString
         mode.addMessage( abortMsg )
         return True

   return False

noSpeedChangeHook = CliExtensions.CliHook()

def configureLinkMode( mode, linkMode, advertisedModes=None,
                       legacySpeedCmdSyntax=False ):
   # First run through the registered hooks and confirm that we can go this linkMode
   if checkChangeCancelHooks( mode, [ mode.intf ], linkMode, advertisedModes ):
      return
   if checkChangeCancelOnSfp28( mode, [ mode.intf ], linkMode, advertisedModes ):
      return

   mode.intf.setLinkMode( linkMode, advertisedModes, legacySpeedCmdSyntax )
   Tracing.trace0( "Set " + mode.intf.name + " speed/duplex to " +
                   str( linkMode ) + " ('None' means auto)" )

def setForcedLinkMode( mode, args ):
   legacySpeedCmdSyntax = False
   if args[ 'SPEED_TYPE' ] == '400gfull':
      legacySpeedCmdSyntax = True

   speedTypeDict = speedForceLinkMode if 'forced' in args else speedLinkMode

   configureLinkMode( mode, speedTypeDict[ args[ 'SPEED_TYPE' ] ],
                      legacySpeedCmdSyntax=legacySpeedCmdSyntax )

def setAutoLinkMode( mode, args ):
   # TODO: refactor to reuse for both intf config and intf range config modes.
   choice = speedAutoLinkMode.get( args.get( 'SPEED_TYPE' ) )

   linkMode = 'linkModeAutoneg'
   if choice == 'mode40GbpsFull': # Special case for `linkMode`.
      # We should to get rid of linkModeAuto40GbpsFull and use advertisedModes to
      # turn on autoneg and advertise 40GFull. BUG97475 has more details.
      linkMode = 'linkModeAuto40GbpsFull'

   doConfigLinkMode = partial( configureLinkMode, mode, linkMode=linkMode )

   if choice:
      advertisedModes = EthLinkModeSet()
      try:
         setattr( advertisedModes, choice, True )
         doConfigLinkMode( advertisedModes=advertisedModes )
      except AttributeError:
         mode.addErrorAndStop( 'Cannot advertise the given speed and duplex' )
   else:
      # Speed type not specified; reset.
      doConfigLinkMode()

def noSpeed( mode, args ):
   configureLinkMode( mode, linkMode='linkModeUnknown' )
   Tracing.trace0( "Set " + mode.intf.name + " speed to default" )

#--------
# Implement the speed <speed> commands.
#--------

speedLinkMode = {
   '10half': 'linkMode10MbpsHalf',
   '10mhalf': 'linkMode10MbpsHalf',
   '10full': 'linkMode10MbpsFull',
   '10mfull': 'linkMode10MbpsFull',
   '100half': 'linkMode100MbpsHalf',
   '100mhalf': 'linkMode100MbpsHalf',
   '100full': 'linkMode100MbpsFull',
   '100mfull': 'linkMode100MbpsFull',
   '1g': 'linkModeForced1GbpsFull',
   '2p5g': 'linkModeForced2p5GbpsFull',
   '10g': 'linkModeForced10GbpsFull',
   '25g': 'linkModeForced25GbpsFull',
   '40g': 'linkModeForced40GbpsFull',
   '50g': 'linkModeForced50GbpsFull',
   '50g-1': 'linkModeForced50GbpsFull1Lane',
   '50g-2': 'linkModeForced50GbpsFull',
   '100g': 'linkModeForced100GbpsFull',
   '100g-1': 'linkModeForced100GbpsFull1Lane',
   '100g-2': 'linkModeForced100GbpsFull2Lane',
   '100g-4': 'linkModeForced100GbpsFull',
   '200g': 'linkModeForced200GbpsFull4Lane',
   '200g-2': 'linkModeForced200GbpsFull2Lane',
   '200g-4': 'linkModeForced200GbpsFull4Lane',
   '400g': 'linkModeForced400GbpsFull8Lane',
   '400g-4': 'linkModeForced400GbpsFull4Lane',
   '400g-8': 'linkModeForced400GbpsFull8Lane',
   '800g-8': 'linkModeForced800GbpsFull8Lane',
   'linkModeUnknown': 'linkModeUnknown',
}

speedForceLinkMode = {
   '10half': 'linkModeForced10MbpsHalf',
   '10full': 'linkModeForced10MbpsFull',
   '100half': 'linkModeForced100MbpsHalf',
   '100full': 'linkModeForced100MbpsFull',
   '1000half': 'linkModeForced1GbpsHalf',
   '1000full': 'linkModeForced1GbpsFull',
   '10000full': 'linkModeForced10GbpsFull',
   '25gfull': 'linkModeForced25GbpsFull',
   '40gfull': 'linkModeForced40GbpsFull',
   '50gfull': 'linkModeForced50GbpsFull',
   '100gfull': 'linkModeForced100GbpsFull',
   '10mhalf': 'linkModeForced10MbpsHalf',
   '10mfull': 'linkModeForced10MbpsFull',
   '100mhalf': 'linkModeForced100MbpsHalf',
   '100mfull': 'linkModeForced100MbpsFull',
   '1ghalf': 'linkModeForced1GbpsHalf',
   '1gfull': 'linkModeForced1GbpsFull',
   '10gfull': 'linkModeForced10GbpsFull',
   '25000full': 'linkModeForced25GbpsFull',
   '40000full': 'linkModeForced40GbpsFull',
   '50000full': 'linkModeForced50GbpsFull',
   '100000full': 'linkModeForced100GbpsFull',
   'linkModeUnknown': 'linkModeUnknown',
   # We add this hidden token to meet customer's need
   '400gfull': 'linkModeForced400GbpsFull8Lane',
}

speedAutoLinkMode = {
   '10000full': 'mode10GbpsFull',
   '1000full': 'mode1GbpsFull',
   '100full': 'mode100MbpsFull',
   '100g-1': 'mode100GbpsFull1Lane',
   '100g-2': 'mode100GbpsFull2Lane',
   '100g-4': 'mode100GbpsFull',
   '100gfull': 'mode100GbpsFull',
   '100half': 'mode100MbpsHalf',
   '10full': 'mode10MbpsFull',
   '10gfull': 'mode10GbpsFull',
   '10half': 'mode10MbpsHalf',
   '1gfull': 'mode1GbpsFull',
   '2.5gfull': 'mode2p5GbpsFull',
   '200g-2': 'mode200GbpsFull2Lane',
   '200g-4': 'mode200GbpsFull4Lane',
   '25gfull': 'mode25GbpsFull',
   '400g-4': 'mode400GbpsFull4Lane',
   '400g-8': 'mode400GbpsFull8Lane',
   '800g-8': 'mode800GbpsFull8Lane',
   '40gfull': 'mode40GbpsFull',
   '50g-1': 'mode50GbpsFull1Lane',
   '50g-2': 'mode50GbpsFull',
   '50gfull': 'mode50GbpsFull',
   '5gfull': 'mode5GbpsFull',
   '10mhalf': 'mode10MbpsHalf',
   '10mfull': 'mode10MbpsFull',
   '100mhalf': 'mode100MbpsHalf',
   '100mfull': 'mode100MbpsFull',
   '25000full': 'mode25GbpsFull',
   '40000full': 'mode40GbpsFull',
   '50000full': 'mode50GbpsFull',
   '100000full': 'mode100GbpsFull',
}

class speedForceExpression( CliCommand.CliExpression ):
   expression = 'SPEED_FORCE_NODE | SPEED_FORCE_HIDDEN_NODE'
   data = {
      'SPEED_FORCE_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10000full': 'Disable autoneg and force 10 Gbps/full duplex operation',
            '1000half': 'Disable autoneg and force 1 Gbps/half duplex operation',
            '1000full': 'Disable autoneg and force 1 Gbps/full duplex operation',
            '100full': 'Disable autoneg and force 100 Mbps/full duplex operation',
            '100gfull': 'Disable autoneg and force 100 Gbps/full duplex operation',
            '100half': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '10full': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '10half': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '25gfull': 'Disable autoneg and force 25 Gbps/full duplex operation',
            '40gfull': 'Disable autoneg and force 40 Gbps/full duplex operation',
            '50gfull': 'Disable autoneg and force 50 Gbps/full duplex operation',
       } ),
         alias='SPEED_TYPE' ),
      'SPEED_FORCE_HIDDEN_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10mhalf': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '10mfull': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '100mhalf': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '100mfull': 'Disable autoneg and force 100 Mbps/full duplex operation',
            '1ghalf': 'Disable autoneg and force 1 Gbps/half duplex operation',
            '1gfull': 'Disable autoneg and force 1 Gbps/full duplex operation',
            '10gfull': 'Disable autoneg and force 10 Gbps/full duplex operation',
            '25000full': 'Disable autoneg and force 25 Gbps/full duplex operation',
            '40000full': 'Disable autoneg and force 40 Gbps/full duplex operation',
            '50000full': 'Disable autoneg and force 50 Gbps/full duplex operation',
            '100000full': 'Disable autoneg and force 100 Gbps/full duplex operation',
            # We add this hidden token to meet customer's need
            '400gfull': 'Disable autoneg and force 400 Gbps/full duplex operation'
                   ' over 8 lanes',
        } ),
         alias='SPEED_TYPE',
         hidden=True ),
}


class SpeedForcedCmd( CliCommand.CliCommandClass ):
   syntax = 'speed forced SPEED_TYPE'
   data = {
      'speed': matcherSpeed,
      'forced': 'Disable autoneg and force speed/duplex/flowcontrol',
      'SPEED_TYPE': speedForceExpression,
   }

   handler = setForcedLinkMode

EthIntfModelet.addCommandClass( SpeedForcedCmd )

class SpeedAutoExpression( CliCommand.CliExpression ):
   expression = 'SPEEDAUTO_NODE | SPEEDAUTO_HIDDEN_NODE'
   data = {
      'SPEEDAUTO_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10000full': 'Enable autoneg for 10 Gbps/full duplex operation',
            '1000full': 'Enable autoneg for 1 Gbps/full duplex operation',
            '100full': 'Enable autoneg for 100 Mbps/full duplex operation',
            '100g-1': 'Enable autoneg for 100 Gbps/full duplex operation'
                      ' over 1 lane',
            '100g-2': 'Enable autoneg for 100 Gbps/full duplex operation'
                      ' over 2 lanes',
            '100g-4': 'Enable autoneg for 100 Gbps/full duplex operation'
                      ' over 4 lanes',
            '100gfull': 'Enable autoneg for 100 Gbps/full duplex operation',
            '100half': 'Enable autoneg for 100 Mbps/half duplex operation',
            '10full': 'Enable autoneg for 10 Mbps/full duplex operation' ,
            '10gfull': 'Enable autoneg for 10 Gbps/full duplex operation',
            '10half': 'Enable autoneg for 10 Mbps/half duplex operation',
            '1gfull': 'Enable autoneg for 1 Gbps/full duplex operation',
            '2.5gfull': 'Enable autoneg for 2.5 Gbps/full duplex operation',
            '200g-2': 'Enable autoneg for 200 Gbps/full duplex operation'
                      ' over 2 lanes',
            '200g-4': 'Enable autoneg for 200 Gbps/full duplex operation'
                      ' over 4 lanes',
            '25gfull': 'Enable autoneg for 25 Gbps/full duplex operation',
            '400g-4': 'Enable autoneg for 400 Gbps/full duplex operation'
                      ' over 4 lanes',
            '400g-8': 'Enable autoneg for 400 Gbps/full duplex operation'
                      ' over 8 lanes',
            '40gfull': 'Enable autoneg for 40 Gbps/full duplex operation',
            '50g-1': 'Enable autoneg for 50 Gbps/full duplex operation'
                     ' over 1 lane',
            '50g-2': 'Enable autoneg for 50 Gbps/full duplex operation'
                     ' over 2 lanes',
            '50gfull': 'Enable autoneg for 50 Gbps/full duplex operation',
            '5gfull': 'Enable autoneg for 5 Gbps/full duplex operation',
            '800g-8': 'Enable autoneg for 800 Gbps/full duplex operation'
                      ' over 8 lanes',
       } ),
         alias='SPEED_TYPE' ),
      'SPEEDAUTO_HIDDEN_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10mhalf': 'Enable autoneg for 10 Mbps/half duplex operation',
            '10mfull': 'Enable autoneg for 10 Mbps/full duplex operation',
            '100mhalf': 'Enable autoneg for 100 Mbps/half duplex operation',
            '100mfull': 'Enable autoneg for 100 Mbps/full duplex operation',
            '25000full': 'Enable autoneg for 25 Gbps/full duplex operation',
            '40000full': 'Enable autoneg for 40 Gbps/full duplex operation',
            '50000full': 'Enable autoneg for 50 Gbps/full duplex operation',
            '100000full': 'Enable autoneg for 100 Gbps/full duplex operation',
        } ),
         alias='SPEED_TYPE',
         hidden=True ),
}

class SpeedAutoCmd( CliCommand.CliCommandClass ):
   syntax = 'speed auto [ SPEED_TYPE ]'
   data = {
      'speed': matcherSpeed,
      'auto': matcherAuto,
      'SPEED_TYPE': SpeedAutoExpression,
   }

   handler = setAutoLinkMode

EthIntfModelet.addCommandClass( SpeedAutoCmd )

class speedExpression( CliCommand.CliExpression ):
   expression = 'SPEED_NODE | SPEED_HIDDEN_NODE'
   data = {
      'SPEED_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10half': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '10full': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '100half': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '100full': 'Disable autoneg and force 100 Mbps/full duplex operation',
            '1g':  'Disable autoneg and force 1 Gbps/full duplex operation'
                   ' over 1 lane',
            '10g': 'Disable autoneg and force 10 Gbps/full duplex operation'
                   ' over 1 lane',
            '25g': 'Disable autoneg and force 25 Gbps/full duplex operation'
                   ' over 1 lane',
            '40g': 'Disable autoneg and force 40 Gbps/full duplex operation'
                   ' over 4 lanes',
            '50g': 'Disable autoneg and force 50 Gbps/full duplex operation'
                   ' over 2 lanes',
            '50g-1': 'Disable autoneg and force 50 Gbps/full duplex operation'
                   ' over 1 lane',
            '50g-2': 'Disable autoneg and force 50 Gbps/full duplex operation'
                   ' over 2 lane',
            '100g': 'Disable autoneg and force 100 Gbps/full duplex operation'
                    ' over 4 or 10 lanes',
            '100g-1': 'Disable autoneg and force 100 Gbps/full duplex operation'
                    ' over 1 lane',
            '100g-2': 'Disable autoneg and force 100 Gbps/full duplex operation'
                   ' over 2 lanes',
            '100g-4': 'Disable autoneg and force 100 Gbps/full duplex operation'
                   ' over 4 lanes',
            '200g': 'Disable autoneg and force 200 Gbps/full duplex operation'
                   ' over 4 lanes',
            '200g-2': 'Disable autoneg and force 200 Gbps/full duplex operation'
                    ' over 2 lanes',
            '200g-4': 'Disable autoneg and force 200 Gbps/full duplex operation'
                   ' over 4 lanes',
            '400g': 'Disable autoneg and force 400 Gbps/full duplex operation'
                   ' over 8 lanes',
            '400g-4': 'Disable autoneg and force 400 Gbps/full duplex operation'
                      ' over 4 lanes',
            '400g-8': 'Disable autoneg and force 400 Gbps/full duplex operation'
                   ' over 8 lanes',
            '800g-8': 'Disable autoneg and force 800 Gbps/full duplex operation'
                      ' over 8 lanes',
       } ),
         alias='SPEED_TYPE' ),
      'SPEED_HIDDEN_NODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            '10mhalf': 'Disable autoneg and force 10 Mbps/half duplex operation',
            '10mfull': 'Disable autoneg and force 10 Mbps/full duplex operation',
            '100mhalf': 'Disable autoneg and force 100 Mbps/half duplex operation',
            '100mfull': 'Disable autoneg and force 100 Mbps/full duplex operation',
        } ),
         alias='SPEED_TYPE',
         hidden=True ),
}

class SpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'speed SPEED_TYPE'
   noOrDefaultSyntax = 'speed ...'
   data = {
      'speed': matcherSpeed,
      'SPEED_TYPE': speedExpression
   }

   handler = setForcedLinkMode
   noOrDefaultHandler = noSpeed

EthIntfModelet.addCommandClass( SpeedCmd )

class SpeedSfp1000BasetAuto100FullCmd( CliCommand.CliCommandClass ):
   """
   speed sfp-1000baset auto 100full

   Special case to allow the user to configure whether to put a
   1000BASE-T SFP into "auto-100/full" mode. Putting the SFP into
   "forced-100/full" mode is done using the "speed forced 100full"
   command.

   I would really just like to implement this as part of implementing
   BUG282 (allow specifying the set of capabilities to advertise during
   autoneg), but this is a fairly big project, because it affects the
   platform code (drivers) for all our PHYs and MACs/switch chips.
   """
   syntax = 'speed sfp-1000baset auto 100full'
   data = {
      'speed' : matcherSpeed,
      'sfp-1000baset' : 'Configure autoneg and speed/duplex on 1000BASE-T SFP',
      'auto' : matcherAuto,
      '100full' : ( 'Advertise 100 Mbps/full duplex as the only capability during '
                    'autoneg' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setLinkModeSfp100BaseTAuto100Full()

EthIntfModelet.addCommandClass( SpeedSfp1000BasetAuto100FullCmd )

# A canSetFlowcontrolHook accepts three arguments: the CLI mode, the
# name of the physical interface, and the requested flowcontrol
# value. If any of the hooks return false, the request is ignored.
canSetFlowcontrolHook = CliExtensions.CliHook()

class FlowcontrolCmd( CliCommand.CliCommandClass ):
   ''' [ no | default ] flowcontrol DIRECTION VALUE '''
   syntax = 'flowcontrol DIRECTION VALUE'
   noOrDefaultSyntax = 'flowcontrol DIRECTION ...'
   data = {
      'flowcontrol' : 'Configure flow operation',
      'DIRECTION' : CliMatcher.EnumMatcher( {
         'send' : 'Configure transmit flow operation',
         'receive' : 'Configure receiving flow operation',
      } ),
      'VALUE' : CliMatcher.EnumMatcher( {
         'on' : 'Require flow-control capable link partner',
         'off' : 'Forbid flow-control capable link partner',
         'desired' : 'Allow but do not require flow-control capable link partner',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      direction = args[ 'DIRECTION' ]
      flowcontrolVal = 'flowControlConfig' + args[ 'VALUE' ].capitalize()
      for hook in canSetFlowcontrolHook.extensions():
         if not hook( mode, mode.intf.name, direction, flowcontrolVal ):
            Tracing.trace0( "Flowcontrol for " + mode.intf.name +
                            " cancelled by " + str(hook) )
            return
      mode.intf.setFlowcontrol( direction, flowcontrolVal )
      Tracing.trace0( "Set " + mode.intf.name + " " + direction +
                      " flowcontrol to " + flowcontrolVal )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      direction = args[ 'DIRECTION' ]
      mode.intf.setFlowcontrol( direction, 'flowControlConfigUnknown' )
      Tracing.trace0( "Set " + mode.intf.name + " " + direction +
                      " flowcontrol to default." )

EthIntfModelet.addCommandClass( FlowcontrolCmd )

LogDirectionType = TacLazyType( "Interface::CongestionDropsLogDirection" )
def congestionDropsGuard( mode, token ):
   if intfHwStatus.dropsLogDirectionSupport == \
          LogDirectionType.directionNone:
      return CliParser.guardNotThisPlatform
   else:
      return None

congestionDropsKw = CliCommand.guardedKeyword(
   'congestion-drops',
   'Drops due to congestion',
   congestionDropsGuard )

def enableGlobalCongestionDropsLogging( mode, logInterval ):
   assert logInterval != 0
   ethPhyIntfDefaultConfigDir.congestionDropsLogInterval = logInterval

def disableGlobalCongestionDropsLogging( mode ):
   ethPhyIntfDefaultConfigDir.congestionDropsLogInterval = 0

class LoggingEventCongestionDropsGlobal( LoggingEventGlobalCmd ):
   """
   The "[no|default] logging event congestion-drops interval <val>" command,
   in "config" mode.
   """
   syntax = 'logging event congestion-drops interval INTERVAL'
   noOrDefaultSyntax = 'logging event congestion-drops ...'
   data = {
      'congestion-drops' : congestionDropsKw,
      'interval' : 'Logging interval',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 65535,
                                             helpdesc='Logging interval in seconds' )
      }
   @staticmethod
   def handler( mode, args ):
      interval = args.get( 'INTERVAL', 0 )
      ethPhyIntfDefaultConfigDir.congestionDropsLogInterval = interval

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventCongestionDropsGlobal )

class LoggingEventCongestionDropsIntf( LoggingEventIntfCmd ):
   """
   logging event congestion-drops [ use-global ]
   no|default logging event congestion-drops
   """
   syntax = "logging event congestion-drops"
   data = {
      'congestion-drops' : congestionDropsKw,
   }

   @classmethod
   def _enableHandler( cls, mode, args ):
      mode.intf.setCongestionDropsLoggingMode( 'on' )

   @classmethod
   def _disableHandler( cls, mode, args ):
      mode.intf.setCongestionDropsLoggingMode( 'off' )

   @classmethod
   def _useGlobalHandler( cls, mode, args ):
      mode.intf.setCongestionDropsLoggingMode( 'useGlobal' )

EthIntfModelet.addCommandClass( LoggingEventCongestionDropsIntf )

# A canSetUnidirectionalLinkModeHook accepts two arguments:
# CLI mode and intfList( could be a single or bunch of interfaces ).
canSetUnidirectionalLinkModeHook = CliExtensions.CliHook()

def uniLinkSupportedGuard( mode, token ):
   if mode.intf.unidirectionalLinkSupported():
      return None
   return CliParser.guardNotThisPlatform

LINK_MODE_ENUM = {
   'send-only' : 'uniLinkModeSendOnly',
   'receive-only' : 'uniLinkModeReceiveOnly',
   'send-receive' : 'uniLinkModeSendReceive'
}

class UnidirectionalCmd( CliCommand.CliCommandClass ):
   ''' [ no | default ] unidirectional LINK_MODE '''
   syntax = 'unidirectional LINK_MODE'
   noOrDefaultSyntax = 'unidirectional [ LINK_MODE ]'
   data = {
      'unidirectional' : CliCommand.guardedKeyword( 'unidirectional',
         helpdesc='Configure unidirectional link mode',
         guard=uniLinkSupportedGuard ),
      'LINK_MODE' : CliMatcher.EnumMatcher( {
         'send-only' : 'Configure intf send only mode',
         'receive-only' : 'Configure intf receive only mode',
         'send-receive' : 'Configure intf send/recv mode',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'LINK_MODE' in args:
         unidirectionalLinkMode = LINK_MODE_ENUM[ args[ 'LINK_MODE' ] ]
      else:
         unidirectionalLinkMode = None

      for i in canSetUnidirectionalLinkModeHook.extensions():
         if not mode.maybeIntfRangeCliHook( i, 'canSetUnidirectionalLinkModeHook' ):
            return

      if CliCommand.isNoOrDefaultCmd( args ):
         if not CliCommand.isDefaultCmd( args ) or unidirectionalLinkMode is None:
            unidirectionalLinkMode = 'uniLinkModeDisabled'
      # Set the UnidirectionalLinkMode for this intf
      mode.intf.setUnidirectionalLinkMode( unidirectionalLinkMode )

   noOrDefaultHandler = handler

EthIntfModelet.addCommandClass( UnidirectionalCmd )

def showIntfsUnidirectional( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   intfsUnidirectional = EthIntfModel.InterfacesUnidirectional()

   for intf in intfs:
      if intf.unidirectionalLinkSupported():
         unidirectionalData = EthIntfModel.InterfaceUnidirectional()
         unidirectionalData.status = intf.linkStatus()
         unidirectionalData.uniLinkMode = intf.showUnidirectionalLinkMode()
         intfsUnidirectional.interfaces[ intf.name ] = unidirectionalData

   return intfsUnidirectional

class ShowUnidirectional( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces unidirectional'
   data = dict( unidirectional='Show unidirectional link mode' )
   moduleAtEnd = True
   handler = showIntfsUnidirectional
   cliModel = EthIntfModel.InterfacesUnidirectional

BasicCli.addShowCommandClass( ShowUnidirectional )

def supportsL2Mtu( mode, token ):
   if MgmtIntfId.isMgmtIntfId( mode.intf.name ):
      return CliParser.guardNotThisInterface # BUG815828
   if bridgingHwCapabilities.l2MtuSupported:
      return None
   return CliParser.guardNotThisPlatform

def rangeL2MtuFn( mode, context ):
   return bridgingHwCapabilities.l2MtuMin, bridgingHwCapabilities.l2MtuMax

class L2MtuCmd( CliCommand.CliCommandClass ):
   ''' [ no | default ] l2 mtu MTU '''
   syntax = 'l2 mtu MTU'
   noOrDefaultSyntax = 'l2 mtu ...'
   data = {
      'l2' : 'Configure L2',
      'mtu' : CliCommand.guardedKeyword( 'mtu',
         helpdesc='Set L2 maximum transmission unit in bytes', guard=supportsL2Mtu ),
      'MTU' : CliMatcher.DynamicIntegerMatcher( rangeL2MtuFn,
         helpdesc='L2 MTU (bytes)' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setL2Mtu( args[ 'MTU' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setL2Mtu( 0 )

EthIntfModelet.addCommandClass( L2MtuCmd )

def supportsL2Mru( mode, token ):
   if MgmtIntfId.isMgmtIntfId( mode.intf.name ):
      return CliParser.guardNotThisInterface # BUG815828
   if bridgingHwCapabilities.l2MruSupported:
      return None
   return CliParser.guardNotThisPlatform

def rangeL2MruFn( mode, context ):
   return bridgingHwCapabilities.l2MruMin, bridgingHwCapabilities.l2MruMax

class L2MruCmd( CliCommand.CliCommandClass ):
   ''' [ no | default ] l2 mru MRU '''
   syntax = 'l2 mru MRU'
   noOrDefaultSyntax = 'l2 mru ...'
   data = {
      'l2' : 'Configure L2',
      'mru' : CliCommand.guardedKeyword( 'mru',
         helpdesc='Set L2 maximum receive unit in bytes', guard=supportsL2Mru ),
      'MRU' : CliMatcher.DynamicIntegerMatcher( rangeL2MruFn,
         helpdesc='L2 MRU (bytes)' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.setL2Mru( args[ 'MRU' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setL2Mru( 0 )

EthIntfModelet.addCommandClass( L2MruCmd )

#-------------------------------------------------------------------------------
# Associate the DataplaneIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( DataplaneIntfModelet )

#-------------------------------------------------------------------------------
# Associate the EthIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( EthIntfModelet )

#-------------------------------------------------------------------------------
# The "show flowcontrol [ interface <name> ]" command, in enable mode.
#   show flowcontrol [ interface <name> ]
#   show flowcontrol module <modnum>
#-------------------------------------------------------------------------------
def getAllIntfs( mode, intf, mod, clazz, exposeInactive=False,
                 exposeUnconnected=False, exposeInternal=False ):
   intfs = IntfCli.Intf.getAll( mode, intf, mod, clazz,
                                exposeInactive=exposeInactive,
                                exposeUnconnected=exposeUnconnected,
                                exposeInternal=exposeInternal )
   if intfs:
      intfs = [ i for i in intfs if i.lookup() ] # See BUG9124
   return intfs

def showInterfacesFlowcontrol( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   intfsFlowControl = EthIntfModel.InterfacesFlowControl()

   def _convertFcState( state ):
      retState = "unsupported" if state == "Unsupp." else state
      return retState

   if ethFlowControlGlobalConfig.flowControlSyslogTimeout != \
         ethFlowControlGlobalConfig.invalidFlowControlSyslogTimeout:
      intfsFlowControl.syslogInterval = \
            ethFlowControlGlobalConfig.flowControlSyslogTimeout

   if intfs:
      for x in intfs:
         flowControl = EthIntfModel.FlowControlData()
         flowControl.txAdminState = _convertFcState( x.flowcontrolTxAdminState() )
         flowControl.txOperState = _convertFcState( x.flowcontrolTxOperState() )
         flowControl.rxAdminState = _convertFcState( x.flowcontrolRxAdminState() )
         flowControl.rxOperState = _convertFcState( x.flowcontrolRxOperState() )
         flowControl.rxPause = x.flowcontrolRxPause()
         flowControl.txPause = x.flowcontrolTxPause()
         intfsFlowControl.interfaceFlowControls[ x.name ] = flowControl
   return intfsFlowControl

class ShowFlowcontrol( ShowCommand.ShowCliCommandClass ):
   syntax = "show flowcontrol [ ( interface INTF ) | MOD ]"
   data = dict( flowcontrol='Details on flow control',
                interface='Show flowcontrol information for interfaces',
                INTF=IntfCli.Intf.rangeMatcher,
                MOD=ModuleIntfCli.ModuleExpressionFactory() )
   cliModel = EthIntfModel.InterfacesFlowControl
   handler = showInterfacesFlowcontrol

BasicCli.addShowCommandClass( ShowFlowcontrol )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] flow-control" command, in enable mode.
#   show interfaces [<name>] flow-control
#   show interfaces module <modnum> flow-control
#   show interfaces flow-control module <modnum>
#  legacy:
#   show interfaces [<name>] flowcontrol
#   show interfaces module <modnum> flowcontrol
#   show interfaces flowcontrol module <modnum>
#-------------------------------------------------------------------------------
flowControlDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'flowcontrol',
                              helpdesc='Details on flow control' ),
   deprecatedByCmd='show interface flow-control' )

class ShowIntfFlowcontrol( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces flow-control | flowcontrol'
   data = { 'flow-control' : 'Show interface flow-control information',
            'flowcontrol' : flowControlDeprecated }
   handler = showInterfacesFlowcontrol
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesFlowControl

BasicCli.addShowCommandClass( ShowIntfFlowcontrol )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] debounce" command, in enable mode.
#   show interfaces [<name>] debounce
#   show interfaces debounce module <modnum>
#   show interfaces module <modnum> debounce
#-------------------------------------------------------------------------------
def showInterfacesDebounceTime( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   result = EthIntfModel.InterfacesDebounceTime()

   intfs = getAllIntfs( mode, intf, mod, L1Intf )
   if not intfs:
      return result

   for x in intfs:
      debounceInfo = EthIntfModel.InterfaceDebounceTime()
      debounceInfo.linkUpTime = x.linkUpDebouncePeriod()
      debounceInfo.linkDownTime = x.linkDownDebouncePeriod()

      result.interfaces[ x.name ] = debounceInfo

   return result

class ShowIntfDebounce( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces debounce'
   data = dict( debounce='Show interface link-debounce information' )
   handler = showInterfacesDebounceTime
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesDebounceTime

BasicCli.addShowCommandClass( ShowIntfDebounce )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] l2 mtu" command
#   show interfaces [<name>] l2 mtu
#   show interfaces l2 mtu
#-------------------------------------------------------------------------------
def showInterfacesL2Mtu( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   # Retrieve all interfaces that support L2 mtu
   intfs = getAllIntfs( mode, intf, mod, gv.l2MtuValidIntfType )
   intfL2Mtu = EthIntfModel.InterfacesL2Mtu()
   if not intfs:
      return intfL2Mtu

   for intf in intfs:
      l2Mtu = EthIntfModel.L2Mtu()
      l2Mtu.l2MtuCfg = intf.config().l2Mtu
      l2Mtu.l2MtuStatus = intf.status().l2Mtu
      intfL2Mtu.interfaceL2Mtus[ intf.name ] = l2Mtu
   return intfL2Mtu

class ShowIntfL2Mtu( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces l2 mtu'
   data = dict( l2='Show interface l2 information',
                mtu='Show interface l2 MTU information' )
   handler = showInterfacesL2Mtu
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesL2Mtu

BasicCli.addShowCommandClass( ShowIntfL2Mtu )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] counters bins" command.
#   show interfaces [<name>] counters bins
#   show interfaces module <modnum> counters bins
#-------------------------------------------------------------------------------
def showInterfacesBinsCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   def stat( attr, counter, ckpt ):
      sysdbValue = getattr( counter.ethStatistics, attr, 0 ) or 0

      if ckpt:
         checkpointValue = getattr( ckpt.ethStatistics, attr, 0 ) or 0
      else:
         checkpointValue = 0
      return sysdbValue - checkpointValue

   def getCountersValues( x, direction ):

      counter = x.counter()

      if getattr( counter, direction + '64OctetFrames', None ) is None:
         return None
         # Use the absence of value for 64OctetFrames as an indication
         # that this interface does not support bins

      ckpt = x.getLatestCounterCheckpoint()

      binsCounters  = EthIntfModel.InterfaceBinsCounters()

      binsCounters.frames64Octet = stat( direction + '64OctetFrames', counter, ckpt )
      binsCounters.frames65To127Octet = stat( direction + '65To127OctetFrames',
                              counter, ckpt )
      binsCounters.frames128To255Octet = stat( direction + '128To255OctetFrames',
                              counter, ckpt )
      binsCounters.frames256To511Octet = stat( direction + '256To511OctetFrames',
                              counter, ckpt )
      binsCounters.frames512To1023Octet = stat( direction + '512To1023OctetFrames',
                              counter, ckpt )
      binsCounters.frames1024To1522Octet = stat( direction +'1024To1522OctetFrames',
                              counter, ckpt )
      binsCounters.frames1523ToMaxOctet = stat( direction + '1523ToMaxOctetFrames',
                              counter, ckpt )
      return binsCounters

   result = EthIntfModel.InterfacesBinsCounters()

   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   if not intfs:
      return result

   for x in intfs:

      inOutBinsCounters = EthIntfModel.InterfaceInOutBinsCounters()

      inOutBinsCounters.inBinsCounters = getCountersValues( x, 'in' )

      inOutBinsCounters.outBinsCounters = getCountersValues( x, 'out' )

      result.interfaces[ x.name ] = inOutBinsCounters

   return result

class ShowIntfCountersBins( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces counters bins'
   data = dict( counters=IntfCli.countersKw,
                bins='Packet length bin counters' )
   handler = showInterfacesBinsCounters
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesBinsCounters

BasicCli.addShowCommandClass( ShowIntfCountersBins )

#-------------------------------------------------------------------------------
# The "show interfaces hardware" commands in enable mode:
#     show interfaces [<name>] hardware
#     show interfaces hardware module <modnum>
#     show interfaces module <modnum> hardware
#  legacy:
#     show interfaces [<name>] capabilities
#     show interfaces capabilities module <modnum>
#     show interfaces module <modnum> capabilities
#
# The "default" token, in enable mode, shows capabilities of a port regardless
# of any installed transceiver. The output captures the shared capabilities of
# every asic/phy leading up to the front panel.
#     show interfaces [<name>] hardware default
#     show interfaces hardware default module <modnum>
#     show interfaces module <modnum> hardware default
#  legacy:
#     show interfaces [<name>] capabilities default
#     show interfaces capabilities default module <modnum>
#     show interfaces module <modnum> capabilities default
#-------------------------------------------------------------------------------
# Pre-compute the format string so that things are properly indented
showIntCapabilitiesLbls = [ 'Model:', 'Type:', 'Speed/Duplex:', 'Flowcontrol:' ]

phyModulationLbls = [ 'Modulation:' ]

errorCorrectionEncodingToLbls = OrderedDict( [ ( 'disabled', '  Disabled: ' ),
                                               ( 'fire-code', '  Fire-code:   ' ),
                                               ( 'reed-solomon', '  Reed-Solomon:' ),
                                               ] )
errorCorrectionLbls = list( errorCorrectionEncodingToLbls.values() )
# Size the output to accomodate fec values even if they don't apply to some
# interfaces.
maxShowIntCapLabelLen = len( max( showIntCapabilitiesLbls + errorCorrectionLbls +
                                  phyModulationLbls, key=len ) )
# This label sits on a line by itself so not included in the above max calc.
errorCorrectionHeading = 'Error Correction:'

showIntCapFmt = "  %-" + ( "%d" % maxShowIntCapLabelLen ) + "s %s"

def showInterfacesCapabilities( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfCaps = EthIntfModel.InterfacesCapabilities()
   intfs = getAllIntfs( mode, intf, mod, L1Intf,
                        exposeInternal=( 'all' in args ) )
   if not intfs:
      return intfCaps

   allPhyCoherentStatus = {}
   # Get all coherent status' at once.
   if 'Modulation' in interfaceCapabilityFns:
      allPhyCoherentStatus = interfaceCapabilityFns[ 'Modulation' ].getAll()

   # Get all default capabilities at once.
   capsInfos = getIntfCapsInfo( intfs )

   for x in intfs:
      icdd = EthIntfModel.InterfaceCapabilitiesData()

      # Model:
      icdd.modelIs( modelStr( x.name ) )

      # Type:
      xcvrPresent = x.intfXcvrPresent()
      icdd.typeIs( xcvrPresent, x.xcvrTypeStr() )

      # Speed/Duplex:
      icdd.speedLanesDuplexIs( x.linkModeCapabilitiesToList() )

      groupStatus = x.speedGroupStatus()
      if groupStatus:
         icdd.speedGroupIs( groupStatus.name, groupStatus.memberIntf,
                            groupStatus.masterIntf )

      # Flowcontrol:
      icdd.flowControlIs( x.status().rxFlowcontrolCapabilities,
                          x.status().autonegCapabilities.rxPause,
                          rx=True )
      icdd.flowControlIs( x.status().txFlowcontrolCapabilities,
                          x.status().autonegCapabilities.txPause,
                          rx=False )

      if ( 'Modulation' in interfaceCapabilityFns and
            not x.cmisCoherentErrorCorrectionSupported() ):
         # The modulation for CMIS coherent modules is being picked up by the Cli
         # model from the coherent error correction capabilities, instead of
         # modulation capabilities.
         modStr = interfaceCapabilityFns[ 'Modulation' ].getMods(
                  allPhyCoherentStatus, x.name,
                  phyModulationLbls[ 0 ], sysCap=False )
         icdd.modulationIs( modStr )

      if 'LineRate' in interfaceCapabilityFns:
         lrDict = interfaceCapabilityFns[ 'LineRate' ].getMods(
            allPhyCoherentStatus, x.name, '', sysCap=False )
         icdd.lineRateIs( lrDict )

      if 'Tributary' in interfaceCapabilityFns:
         tmDict = interfaceCapabilityFns[ 'Tributary' ].getMods( None, x.name,
                                                                 '', sysCap=False )
         icdd.tributaryMappingIs( tmDict )

      # Error Correction:
      # Client FEC
      if xcvrPresent and not x.cmisCoherentErrorCorrectionSupported():
         for encoding in errorCorrectionEncodingToLbls:
            if x.errorCorrectionEncodingSupported( encoding ):
               caps = x.errorCorrectionCapabilitiesToList( encoding )
               icdd.errorCorrectionIs( caps, defaultCheck=True )
      # Coherent FEC
      if ( x.coherentErrorCorrectionSupported() or
           x.cmisCoherentErrorCorrectionSupported() ):
         if 'coherent' in errorCorrectionShCapFns:
            fecDict = \
               errorCorrectionShCapFns[ 'coherent' ].getFecs( allPhyCoherentStatus,
                                                              x,
                                                              False )
         icdd.coherentErrorCorrectionIs( fecDict )

      # Autoneg info is stored in CAPI model but will not be printed by 'show
      # interface et* capabilities'
      for ( epif, epdc ) in capsInfos:
         if x.name == epif.name:
            autonegMode = epif.autonegCapabilitiesMode()
            autonegCaps = epif.autonegCapabilitiesToList()
            # Clause 28
            if autonegMode == 'anegModeClause28':
               icdd.clause28Is( autonegCaps )

            # Clause 37
            if autonegMode == 'anegModeClause37':
               icdd.clause37Is( autonegCaps )

            icdd.clause73Is( [], [] )
            # Clause 73 has both IEEE and 25G Consortium standards
            if autonegMode == 'anegModeClause73':
               # Preserve the order of capabilities in the list.
               icdd.clause73Is(
                  [ cap for cap in epdc.anegCl73ModeCapabilitiesToList( 'ieee' )
                    if cap in autonegCaps ],
                  [ cap for cap in
                    epdc.anegCl73ModeCapabilitiesToList( 'consortium' )
                    if cap in autonegCaps ] )

      intfCaps.interfaces[ x.name ] = icdd
   return intfCaps


capabilitiesDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'capabilities',
                              helpdesc='Show interface capabilities information' ),
   deprecatedByCmd='show interface hardware' )

hardwareKw = CliMatcher.KeywordMatcher(
   'hardware',
   helpdesc='Show interface hardware capabilities information' )

class ShowIntfHardware( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces capabilities | hardware'
   data = dict( capabilities=capabilitiesDeprecated,
                hardware=hardwareKw )
   handler = showInterfacesCapabilities
   allowAllSyntax = True
   moduleAtEnd = True
   cliModel= EthIntfModel.InterfacesCapabilities

BasicCli.addShowCommandClass( ShowIntfHardware )

def showInterfacesCapabilitiesDefault( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   capsDefaults = EthIntfModel.InterfacesCapabilitiesDefault()
   intfs = getAllIntfs( mode, intf, mod, L1Intf,
                        exposeInternal=( 'all' in args ) )
   if not intfs:
      return capsDefaults

   capsInfos = getIntfCapsInfo( intfs )

   allPhyCoherentStatus = {}
   # Get all coherent status' at once.
   if 'Modulation' in interfaceCapabilityFns:
      allPhyCoherentStatus = interfaceCapabilityFns[ 'Modulation' ].getAll()

   for epis, epdc in capsInfos:
      icdd = EthIntfModel.InterfaceCapabilitiesDefaultData()

      # Model:
      icdd.modelIs( modelStr( epis.name ) )

      # Type:
      icdd.typeIs( epis.intfXcvrPresent(),
                   epis.xcvrTypeStr() )

      # Speed/Duplex:
      icdd.speedLanesDuplexIs( epdc.linkModeCapabilitiesToList() )

      # Speed Group:
      groupStatus = epis.speedGroupStatus()
      if groupStatus:
         icdd.speedGroupIs( groupStatus.name, groupStatus.memberIntf,
                            groupStatus.masterIntf )

      # Flowcontrol:
      icdd.flowControlIs( epis.status().rxFlowcontrolCapabilities,
                          epis.status().autonegCapabilities.rxPause,
                          rx=True )
      icdd.flowControlIs( epis.status().txFlowcontrolCapabilities,
                          epis.status().autonegCapabilities.txPause,
                          rx=False )

      # Autoneg labels
      # Clause 28
      icdd.clause28Is( epdc.autonegCapabilitiesToList( 'anegModeClause28' ) )

      # Clause 37
      icdd.clause37Is( epdc.autonegCapabilitiesToList( 'anegModeClause37' ) )

      # Clause 73 has both IEEE and 25G Consortium standards
      icdd.clause73Is( epdc.anegCl73ModeCapabilitiesToList( 'ieee' ),
                       epdc.anegCl73ModeCapabilitiesToList( 'consortium' ) )

      icdd.hasAutoneg = epdc.checkForAutoneg()
      if 'Modulation' in interfaceCapabilityFns:
         modStr = interfaceCapabilityFns[ 'Modulation' ].getMods(
                  allPhyCoherentStatus, epis.name,
                  phyModulationLbls[ 0 ], sysCap=True )
         # If we have a CMIS coherent capable slot or module we don't want to
         # display the module's modulation at all. This is supposed to be a
         # Host capability.
         if epis.cmisCoherentErrorCorrectionSupported():
            icdd._hasCmisCoherentIntf = True
         icdd.modulationIs( modStr )

      # Error Correction:
      # Client FEC
      for encoding in epdc.errorCorrectionEncodings():
         fecLinkModes = epdc.errorCorrectionLinkModes( encoding )
         icdd.errorCorrectionIs( fecLinkModes )

      # Coherent FEC
      if epis.coherentErrorCorrectionSupported():
         if 'coherent' in errorCorrectionShCapFns:
            fecDict = \
               errorCorrectionShCapFns[ 'coherent' ].getFecs( allPhyCoherentStatus,
                                                              epis,
                                                              True )
         icdd.coherentErrorCorrectionIs( fecDict )

      capsDefaults.interfaces[ epis.name ] = icdd
   return capsDefaults

hardwareDefaultKw = CliMatcher.KeywordMatcher(
   'default',
   helpdesc='Show default port capabilities' )

class ShowIntfHardwareDefault( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces capabilities | hardware default'
   data = dict( capabilities=capabilitiesDeprecated,
                hardware=hardwareKw,
                default=hardwareDefaultKw )
   handler = showInterfacesCapabilitiesDefault
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesCapabilitiesDefault

BasicCli.addShowCommandClass( ShowIntfHardwareDefault )

class ShowIntfAllHardwareDefault( ShowCommand.ShowCliCommandClass ):
   syntax = 'show interfaces all hardware default'
   data = dict( interfaces=IntfCli.interfacesShowKw,
                all=IntfCli.interfacesShowAllKw,
                hardware=hardwareKw,
                default=hardwareDefaultKw )
   handler = showInterfacesCapabilities
   cliModel = EthIntfModel.InterfacesCapabilities

BasicCli.addShowCommandClass( ShowIntfAllHardwareDefault )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] transceiver properties" command, in enable mode.
#   show interfaces [<name>] transceiver properties
#   show interfaces module <modnum> transceiver properties
#-------------------------------------------------------------------------------
EthLinkMode = TacLazyType( 'Interface::EthLinkMode' )
EthFecEncoding = TacLazyType( "Interface::EthFecEncoding" )
fecEncodingToStr = {
      EthFecEncoding.fecEncodingReedSolomon : 'reedSolomon',
      EthFecEncoding.fecEncodingReedSolomon544 : 'reedSolomon',
      EthFecEncoding.fecEncodingFireCode : 'fireCode',
      EthFecEncoding.fecEncodingDisabled : 'disabled',
      EthFecEncoding.fecEncodingUnknown : 'unsupported'
}

def getIntfCapabilitiesDir( intfName ):
   # On a vEOS platform, interface do not have XcvrCapabilities set.
   if isVEos():
      return None

   isModularSys = Cell.cellType() == 'supervisor'
   isManagement = TacLazyType( 'Arnet::MgmtIntfId' ).isMgmtIntfId( intfName )
   if isModularSys:
      if isManagement:
         return None
      else:
         sliceName = EthIntfLib.sliceName( intfName )
   else:
      sliceName = 'FixedSystem'
   capabilitiesDir = LazyMount.mount( entityManager,
      'interface/archer/status/eth/capabilities/input/slice/%s/xcvr' % sliceName,
      'Interface::Capabilities::InputCapabilitiesDir', 'r' )
   return capabilitiesDir

def getLinkModeCapabilitiesCaps( intfName, inputDir ):
   ''' return: List of linkModeCapabilities of the transceiver '''
   linkModeCapsList = None
   lanes = False
   if inputDir and intfName in inputDir.intfCapabilities:
      intfCapabilities = inputDir.intfCapabilities[ intfName ]
      lanes = EthPhyIntf._showLanes( intfCapabilities )
      linkModeCaps = intfCapabilities.linkModeCapabilities
      linkModeCapsList = EthIntfLib.linkModeCapabilitiesToList( linkModeCaps,
                         showLanes=lanes )
      if not linkModeCapsList:
         linkModeCapsList = 'none'
   return linkModeCapsList

def getFecCapabilities( intfName, intfCapabilitiesDir ):
   '''
   Getting the transceiver's fecCapabilities for all linkModes.

   Parameters
   ----------
   intfName: str
   intfCapabilitiesDir: Interface::Capabilities::InputCapabilitiesDir

   Return
   ------
   fecCapabilities: Dict of fecCapabilities of the transceiver.
                    ex. { linkMode : fecCapSet }
                    linkMode: Interface::EthLinkMode
                    fecCapSet: Interface::EthFecCapabilitySet
   '''
   fecCapabilities = {}
   if intfCapabilitiesDir and intfName in intfCapabilitiesDir.intfCapabilities:
      intfCap = intfCapabilitiesDir.intfCapabilities[ intfName ]
      for linkMode, fecCapSet in intfCap.fecCapabilities.items():
         fecCapabilities[ linkMode ] = fecCapSet
   return fecCapabilities

def checkVetoFec( fecCapabilities ):
   '''
   The system vetoes the FEC on some transceivers by setting
   the fecCapabilitiesSet to false at certain speeds, like QSFP 100GE-DWDM2,
   100GBASE-SRBD/DR/FR.

   Parameters
   ----------
   fecCapabilities: Dict, EthFecCapabilitySet[EthLinkMode]

   Return
   ------
   Bool: Return true when no FEC is allowed by the transceiver at certain speeds,
         and then False otherwise
   '''
   disabledFecSet = Tac.Value( 'Interface::EthFecCapabilitySet' )
   disabledFecSet.fecEncodingDisabled = True
   # Return True when all EthFecCapabilitySets in the collection
   # are equal to the disabledFecSet
   return ( bool( fecCapabilities ) and
            all( ethFecSet == disabledFecSet
                 for ethFecSet in fecCapabilities.values() ) )

def getIntfXcvrStatusDir( intfName ):
   return intfXcvrStatusDir.get( intfName )

def getMediaFecCap( intfName, intfXcvrStatus ):
   '''
   Getting the mediaFecStandard for each interface with transceiver installed.

   Parameters
   ----------
   intfName: str
   intfXcvrStatus: Interface::EthIntfXcvrStatus

   Return
   ------
   mediaFecCap: Dict of the mediaFecStandard of the transceiver..
                    ex. { speed : fecEncodingStr }
                    speed : str, ex.25Gps
                    fecEncodingStr: str, ex.reedSolomon
   '''
   mediaFecCap = {}
   if intfXcvrStatus:
      for linkMode, fecEncoding in intfXcvrStatus.mediaFecStandard.items():
         speed, _, _ = EthIntfLib.linkModeToSpeedLanesDuplex[ linkMode ]
         speed = speed[ len( 'speed' ): ]
         if fecEncoding in fecEncodingToStr:
            mediaFecCap[ speed ] = fecEncodingToStr[ fecEncoding ]
   return mediaFecCap

def getSwappableXcvr( intfName, inputDir ):
   isSwappableXcvr = False
   if inputDir and intfName in inputDir.intfCapabilities:
      intfCapabilities = inputDir.intfCapabilities[ intfName ]
      isSwappableXcvr = intfCapabilities.isSwappableXcvr
   return isSwappableXcvr

def showInterfacesXcvrProperties( mode, args ):
   interface = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   model = XcvrCapabilitiesModel.TransceiverPropertiesBase()
   intfs = getAllIntfs( mode, interface, mod, L1Intf )

   if not intfs:
      return model

   # TODO BUG587806: Switch interfaces are currently treated as internal interfaces
   #                 by some CLI commands and external interfaces by others. As such,
   #                 they presently report as external interfaces. Once they make the
   #                 transition to internal interfaces we should replace this with
   #                 exposeInternal=False.
   intfs = [ intf for intf in intfs if not intf.name.startswith( 'Sw' ) and
             not intf.internal() ]

   for intf in intfs:
      tp = XcvrCapabilitiesModel.TransceiverProperties()
      tp.mediaType = intf.xcvrTypeStr()
      xcvrPresent = intf.intfXcvrPresent()
      tp.transceiverPresenceIs( xcvrPresent )
      stringFormat = 'Format2'
      # Passing the info of speed capabilities when the transceiver is installed.
      # If there is no transceiver installed, no speed capabilities in the output
      # of the command.
      if xcvrPresent:
         intfCapabilitiesDir = getIntfCapabilitiesDir( intf.name )
         mediaFecCap = getMediaFecCap( intf.name, getIntfXcvrStatusDir( intf.name ) )
         linkModeCaps = getLinkModeCapabilitiesCaps( intf.name, intfCapabilitiesDir )
         fecCap = getFecCapabilities( intf.name, intfCapabilitiesDir )
         tp.xcvrCapListIs( linkModeCaps, mediaFecCap, checkVetoFec( fecCap ) )
         tp.isSwappableXcvrIs( getSwappableXcvr( intf.name, intfCapabilitiesDir ) )
      else:
         tp.xcvrCapListIs( None, None, False  )
      if intf.autonegActive():
         # If the transceiver auto-negotiation is active, the administrative speed
         # will be 'auto' or 'Auto-speed', the laneCount will be 0.
         tp.adminSpeedIs( intf.adminSpeedStr(), 0, intf.adminDuplexStr(),
                          intf.showLanes(), intf.autonegActive() )
         if intf.status().speed == EthSpeed.speedUnknown:
            # If the transceiver auto negotiation is active and speed is unknown,
            # the operational speed will be 'auto' or 'Auto-speed', the laneCount
            # will be 0.
            # Sometimes the interface speed will change from 'speedUnknown' to
            # 'speed1Gbps' or other speed in function 'intf.operSpeed',
            # the return value of 'operSpeed' in this case is '1G'.
            # '1G' is not an avaible value for CLI model, we expect 'speed1Gbps'.
            # Double check the value of 'operSpeed', if not 'auto', it means the
            # interface speed was changed, we need to reset the operSpeed and
            # operLanes by calling funtion 'setOperSpeedInProperties'.
            operSpeed = intf.operSpeed( stringFormat )
            if operSpeed in [ 'auto' ]:
               operLanes = 0
            else:
               operSpeed, operLanes = intf.setOperSpeedInProperties()
            tp.operSpeedIs( operSpeed, operLanes,
                            intf.operDuplex( stringFormat ), intf.showLanes(),
                            intf.autonegActive(), intf._autonegSuffix() )
         else:
            operSpeed, operLanes = intf.setOperSpeedInProperties()
            tp.operSpeedIs( operSpeed, operLanes, intf.operDuplex( stringFormat ),
                            intf.showLanes(), intf.autonegActive(),
                            intf._autonegSuffix() )
      else:
         adminSpeed, adminLanes = intf.adminSpeed( stringFormat )
         tp.adminSpeedIs( adminSpeed, adminLanes, intf.adminDuplexStr(),
                          intf.showLanes(), intf.autonegActive() )
         operSpeed, operLanes = intf.setOperSpeedInProperties()
         tp.operSpeedIs( operSpeed, operLanes, intf.operDuplex( stringFormat ),
                         intf.showLanes(), intf.autonegActive(),
                         intf._autonegSuffix() )
      model.interfaces[ intf.name ] = tp
   return model

class ShowIntfXcvrProperties( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces transceiver properties'
   data = dict( transceiver='Details on transceivers',
                properties='Show interface transceiver properties' )
   handler = showInterfacesXcvrProperties
   cliModel = XcvrCapabilitiesModel.TransceiverPropertiesBase

BasicCli.addShowCommandClass( ShowIntfXcvrProperties )

#-------------------------------------------------------------------------------
# The "show interfaces [<name>] negotiation [detail]" command, in enable mode.
#   show interfaces [<name>] negotiation [detail]
#   show interfaces module <modnum> negotiation [detail]
#-------------------------------------------------------------------------------

def showInterfacesNegotiationDetails( intfs ):

   lineFmt = '  %-21.21s %-52.52s'
   negFmt = '%-20.20s %-10.10s %-20.20s'
   resolvedFmt = '%-20.20s %-10.10s %-25.25s'

   for i in intfs:

      state = i.autonegStateStr( longStr=True )
      mode = i.autonegMode()
      downshiftStatus = i.speedDownshiftStatus()

      localSpeed = i.localSpeedSet( detail=True )
      localDuplex = i.localDuplexSet( detail=True )
      localPause = i.localAnegPause()
      localAdStr = negFmt % ( localSpeed, localDuplex, localPause )

      partnerSpeed = i.partnerSpeedSet( detail=True )
      partnerDuplex = i.partnerDuplexSet( detail=True )
      partnerPause = i.partnerAnegPause()
      partnerAdStr = negFmt % ( partnerSpeed, partnerDuplex, partnerPause )

      advert = i.status().autonegStatus.advertisement
      active = advert.linkModes.anyCapsSet()
      resolvedSpeed = i.operSpeed( 'Format1' ) if active else 'None'
      resolvedDuplex = i.operDuplex( 'Format2' ) if active else 'None'
      if advert.pause != 'pauseUnknown':
         resolvedPause = 'Rx=%s,Tx=%s' % ( i.flowcontrolRxOperState(),
                                           i.flowcontrolTxOperState() )
      else:
         resolvedPause = 'None'
      resolvedStr = resolvedFmt % ( resolvedSpeed, resolvedDuplex, resolvedPause )

      print( i.name, ":\n" )
      print( '%-25.25s %-40.40s' % ( 'Auto-Negotiation Mode', mode  ) )
      print( '%-25.25s %-40.40s' % ( 'Auto-Negotiation Status', state ) )
      print( '%-25.25s %-40.40s' % ( 'Speed Downshifting', downshiftStatus ) )
      print()

      # Only display further details if autoneg is active
      autonegStatus = i.status().autonegStatus
      if ( i.config().enabled and
           autonegStatus.state not in [ 'anegStateUnknown', 'anegStateOff' ] ):
         negHdr = negFmt % ( 'Speed', 'Duplex', 'Pause' )
         print( lineFmt % ( 'Advertisements', negHdr ) )

         negHdr = negFmt % ( '-' * 20, '-' * 10, '-' * 20 )
         print( lineFmt % ( ' ', negHdr ) )

         print( lineFmt % ( ' ' * 4 + 'Local', localAdStr ) )
         print( lineFmt % ( ' ' * 4 + 'Link Partner', partnerAdStr ) )
         print()

         print( lineFmt % ( 'Resolution', resolvedStr ) )
         print()

      print()

def showInterfacesNegotiation( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   exposeInternal = 'all' in args
   intfs = getAllIntfs( mode, intf, mod, L1Intf, exposeInternal=exposeInternal )
   if not intfs:
      return

   hdrFmt  = '%-10.10s  %-7.7s  %-11.11s  %-20.20s'
   print( hdrFmt % ( 'Port', 'Autoneg', ' ', 'Negotiated Settings' ) )

   fmt = '%-10.10s  %-7.7s  %-5.5s  %-5.5s  %-6.6s  %-8.8s  %-8.8s  %-4.4s'
   print( fmt % ( ' ', 'Status', 'Speed', 'Lanes', 'Duplex',
                 'Rx Pause', 'Tx Pause', 'FEC' ) )
   print( fmt % ( '-' * 10, '-' * 7, '-' * 5, '-' * 5, '-' * 6,
                 '-' * 8, '-' * 8, '-' * 4 ) )

   for i in intfs:

      s = i.status()
      name = i.shortname
      anegStatus = s.autonegStatus
      state = i.autonegStateStr()

      if state == 'off' or state == 'unknown':
         speed = '-'
         duplex = '-'
         lanes = '-'
         rxFlowControl = '-'
         txFlowControl = '-'
         fec = '-'
      else:
         advert = anegStatus.advertisement
         active = advert.linkModes.anyCapsSet()

         speed = i.operSpeed( 'Format2' ) if active else '-'
         duplex = i.operDuplex( 'Format2' ) if active else '-'
         laneCount = EthTypesApi.laneCountNumber( anegStatus.laneCount )
         laneCount = laneCount if laneCount else 'auto'
         lanes = laneCount if active else '-'

         if advert.pause != 'pauseUnknown':
            rxFlowControl = i.flowcontrolRxOperState()
            txFlowControl = i.flowcontrolTxOperState()
         else:
            rxFlowControl = txFlowControl = '-'

         fec = i.anegFecEncodingStrMap[ anegStatus.fecEncoding ] if active else '-'

      print( fmt % ( name, state, speed, lanes,
                    duplex, rxFlowControl, txFlowControl, fec ) )

class ShowIntfNegotiation( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces negotiation'
   data = dict( negotiation='Show interface Auto-Negotiation status' )
   allowAllSyntax = True
   handler = showInterfacesNegotiation

BasicCli.addShowCommandClass( ShowIntfNegotiation )

# ------------------------------------------------------------------------------
# configure; interface defaults; ethernet
# ------------------------------------------------------------------------------
class InterfaceDefaultsEthernetConfigMode( InterfaceDefaultsEthernetMode,
                                           BasicCli.ConfigModeBase ):
   name = "Interface Defaults Ethernet Configuration"

   def __init__( self, parent, session ):
      InterfaceDefaultsEthernetMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# ethernet
#--------------------------------------------------------------------------------
class EthernetCmd( CliCommand.CliCommandClass ):
   syntax = 'ethernet'
   noOrDefaultSyntax = syntax
   data = {
      'ethernet' : 'Set default values for new interfaces of type ethernet',
   }

   @staticmethod
   def handler( mode, args ):
      newSubMode = mode.childMode( InterfaceDefaultsEthernetConfigMode )
      mode.session_.gotoChildMode( newSubMode )

   noOrDefaultHandler = InterfaceDefaultsEthernetConfigMode.clear

IntfCli.InterfaceDefaultsConfigMode.addCommandClass( EthernetCmd )

# ------------------------------------------------------------------------------
# configure; interface defaults; ethernet; [no] shutdown
# ------------------------------------------------------
# This is used to set what the admin state of a just instanciated ethernet interface
# should be. The default admin state has historically been 'up', but some customers
# want interfaces to say "silent" until they are configured. The only way to change
# this default behavior is by shipping the EOS switch with a factory preset file at
# /mnt/flash/kickstart-config which contains this command. It was decided not to ship
# EOS hardware with such preset kickstart files, so if this security feature is
# needed, then the customer has to run this command on the first day of service, and
# if he uses "rollback clean-config" later, remember to reconfigure it each time.
# Security is not the only reason to want inert interfaces on first power up, there
# are issues about not sending the wrong frequency/color into an optical muxer.
# Typically switches get connected to cables before they get configured, and since
# we dont support kickstart-config, they need to run this command before cabling.
# ------------------------------------------------------------------------------
IntfEnabledState = TacLazyType( 'Interface::IntfEnabledState' )

def shutdownHandler( mode, args ):
   shutdownCommonHandler( mode, True )

def shutdownNoDefHandler( mode, args ):
   shutdownCommonHandler( mode, False )

def shutdownCommonHandler( mode, targetShutdownState ):
   if IntfCli.globalIntfConfig.defaultEthernetShutdown != targetShutdownState:
      # since defaults will change soon, we have to explicitely set the actual admin
      # status of interfaces that were still 'unset' and relied on the default, else
      # the 'show run' output might say 'down' while "show interf status" says 'up'.
      # Note the sub-interfaces are not in the ethPhyIntfConfigSliceDir (and those
      # are not subject to this default settings, only physical interfaces are).
      # This feature will actually forcefully shut any interface that was still not
      # explicitely configured before, so warm the user and give him a way out.
      # Note that disabling the default shutdown feature will not affect any intfs
      # since while it is enabled both shut & unshut states are in running-config.
      def getListOfAffectedInterfaces():
         aIntfs = []
         for sliceVal in ethPhyIntfConfigSliceDir.values():
            for intfId, intf in sliceVal.intfConfig.items():
               if intfId.startswith( "Ethernet" ): # exclude management eth
                  if ( intf.adminEnabledStateLocal ==
                       IntfEnabledState.unknownEnabledState ):
                     aIntfs.append( intf )
         return aIntfs
      def proceedWithCommand( aIntfs ):
         mode.addWarning(
            "This command will change the administrative status of every Ethernet "
            "interface that has not yet been specifically configured.\n"
            "This might include the interface over which this session runs!\n"
            "Here is the list of affected interfaces:\n%s\n" % aIntfs )
         prompt = 'Do you wish to proceed with this command?'
         return BasicCliUtil.getChoice( mode, prompt, [ 'yes', 'no' ], 'yes' )
      aIntfs = getListOfAffectedInterfaces()
      answer = 'yes' # in case there are no affected interfaces
      if aIntfs:
         intfNames = " ".join( sorted( [ IntfCli.Intf.getShortname( i.intfId )
                                         for i in aIntfs ] ) )
         answer = proceedWithCommand( intfNames ).lower()
      if not ( answer == 'yes' or answer == 'ye' or answer == 'y' ):
         # cohab tests return an empty answer so take those as a 'yes'
         if not ( os.environ.get( "A4_CHROOT" ) and answer == "" ):
            print( "Command aborted" )
            return
      adminEnabledState = 'shutdown' if targetShutdownState else 'enabled'
      for intf in aIntfs:
         if intf.adminEnabledStateLocal == IntfEnabledState.unknownEnabledState:
            intf.adminEnabledStateLocal = adminEnabledState
      # now do the actual set to change the default admin status of new interfaces
      IntfCli.globalIntfConfig.defaultEthernetShutdown = targetShutdownState

class ShutdownCmd( CliCommand.CliCommandClass ):
   ''' [ no | default ] shutdown '''
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : "Set default admin state of new eth interfaces to 'disabled'",
   }

   handler = shutdownHandler
   noOrDefaultHandler = shutdownNoDefHandler

InterfaceDefaultsEthernetConfigMode.addCommandClass( ShutdownCmd )

def mountCounters( ):
   global ethIntfCounterReaderSm, ethPhyIntfCounterDir_
   mg = entityManager.mountGroup()
   mg.mount( "interface/counter/eth/intf", "Interface::AllIntfCounterDir", "w" )
   mg.close( callback=None )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   ethIntfCounterReaderSm = Tac.newInstance( "Interface::EthIntfCounterReaderSm",
                                             smashEm, ethIntfCounterWriterStatusDir )
   ethIntfCounterReaderSm.enableLegacyShmemManSupport()
   ethIntfCounterReaderSm.checkpointCounterDir = checkpointCounterDir
   ethIntfCounterReaderSm.handleInitialized()
   ethPhyIntfCounterDir_ = \
         ethIntfCounterReaderSm.legacyShmemManCounterAccessorInternal

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( em ):
   global entityManager
   global ethPhyIntfConfigSliceDir
   global ethPhyIntfDefaultConfigDir
   global ethIntfSliceDefaultConfigDir
   global ethIntfStatusDir, ethPhyIntfStatusDir, entmib, intfHwStatus
   global ethIntfGenerationSliceDir, ethIntfModeSliceDir
   global ethIntfCounterWriterStatusDir, checkpointCounterDir
   global ethPhyDefaultCapsSliceDir
   global intfXcvrStatusDir
   global internalSissSliceDir
   global bridgingHwCapabilities
   global entityMib
   global ethFlowControlGlobalConfig
   global resourceMgrStatusSliceDir
   global linkFlapDampingConfig
   global linkFlapDampingIntfStatusDir
   global l1CapsSlotDir
   global defaultL1CapsSlotDir

   entityManager = em
   mg = entityManager.mountGroup()
   ethIntfCounterWriterStatusDir = mg.mount(
      "interface/ethIntfCounter/writerStatus", "Tac::Dir", "ri" )
   # We use a cell specific path to store the Cli clear counter snapshot because the
   # current snapshot is stored in Smash which is not synchronized across
   # supervisors. Thus, after a switchover we don't want to use obsolete values (see
   # BUG217544). This needs to be revisited if we ever use synchronization of Smash
   # tables between supervisors for the current snapshot.
   checkpointCounterDir = mg.mount( Cell.path( "interface/cli/counters/eth" ),
                                    "Interface::AllEthIntfCounterBaseDir", "wf" )
   mg.close( mountCounters )

   ethPhyIntfDefaultConfigDir = ConfigMount.mount(
                         entityManager,
                         "interface/config/eth/phy/globalDefault",
                         "Interface::EthPhyIntfGlobalDefaultConfigDir", "w" )

   ethPhyIntfConfigSliceDir = ConfigMount.mount( entityManager,
                                        "interface/config/eth/phy/slice",
                                        "Tac::Dir", "wi" )
   ethIntfStatusDir = LazyMount.mount( entityManager,
                                "interface/status/eth/intf",
                                "Interface::EthIntfStatusDir", "r" )

   ethIntfSliceDefaultConfigDir = LazyMount.mount( entityManager,
                                "interface/archer/config/eth/phy/sliceDefault",
                                "Interface::EthIntfSliceDefaultConfigDir", "r" )

   ethPhyIntfStatusDir = LazyMount.mount( entityManager,
                                          "interface/status/eth/phy/all",
                                          "Interface::AllEthPhyIntfStatusDir", "r" )

   ethIntfGenerationSliceDir = LazyMount.mount( entityManager,
                                "interface/archer/config/eth/phy/generation/slice",
                                "Tac::Dir", "ri" )

   ethIntfModeSliceDir = LazyMount.mount( entityManager,
                                "interface/archer/status/eth/phy/mode/slice",
                                "Tac::Dir", "ri" )

   resourceMgrStatusSliceDir = LazyMount.mount( entityManager,
                                "interface/resources/status/slice",
                                "Tac::Dir", "ri" )

   # The only reason this mount is here is to ensure that
   # CliSessionMgr will have it when certain CliSavePlugin files
   # request this path.
   _ethPhyIntfConfigDir = LazyMount.mount( entityManager,
                                           "interface/config/eth/phy/all",
                                           "Interface::AllEthPhyIntfConfigDir",
                                           "r" )

   intfXcvrStatusDir = XcvrEthIntfDir.ethIntfXcvrStatusDir( entityManager )

   internalSissSliceDir = LazyMount.mount( entityManager,
                                           "interface/intfSlot/status/internal/slot",
                                           "Tac::Dir", "ri" )

   intfHwStatus = LazyMount.mount( entityManager, "interface/hwstatus",
                                   "Interface::HwStatus", "r" )
   entmib = LazyMount.mount( entityManager, "hardware/entmib",
                             "EntityMib::Status", "r" )

   ethPhyDefaultCapsSliceDir = LazyMount.mount( entityManager,
                        "interface/archer/status/eth/phy/capabilities/default/slice",
                        "Tac::Dir", "ri" )

   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )

   ethFlowControlGlobalConfig = LazyMount.mount(
                          em,
                          "interface/flowControl/flowControlGlobalConfig",
                          "Interface::EthFlowControlGlobalConfig", "r" )

   entityMib = LazyMount.mount(
      entityManager,
      'hardware/entmib',
      'EntityMib::Status', 'r' )

   linkFlapDampingConfig = LazyMount.mount(
      entityManager,
      "interface/archer/config/eth/linkFlapDamping/dampingConfig",
      "LinkFlap::DampingConfig", "r" )

   linkFlapDampingIntfStatusDir = LazyMount.mount(
      entityManager,
      'interface/archer/status/eth/linkFlapDamping/slice',
      'Tac::Dir',
      'ri',
   )

   l1CapsSlotDir = LazyMount.mount(
      entityManager,
      'hardware/l1/capabilities/slice',
      'Tac::Dir',
      'ri',
   )

   defaultL1CapsSlotDir = LazyMount.mount(
      entityManager,
      'hardware/l1/capabilities/default/slice',
      'Tac::Dir',
      'ri',
   )

   gv.l1ProductConfig = LazyMount.mount(
      entityManager,
      L1MountConstants.productConfigRootDirPath(),
      L1ProductConfig.tacType.fullTypeName,
      'r' )


   global systemName
   systemName = entityManager.sysname()

capsTuple = namedtuple( 'capabilityFn', 'getAll printStr getMods' )
interfaceCapabilityFns = {}
def registerInterfaceCapabilities( capability, getAllFn, printStrFn, getModsFn ):
   ''' A getAllFn can be called to gather all the information from
   Sysdb at once.'''
   interfaceCapabilityFns[ capability ] = capsTuple( getAllFn, printStrFn,
                                                     getModsFn )

fecTuple = namedtuple( 'errorCorrectionCfgFn', 'configFn' )
errorCorrectionCfgFns = {}
def registerErrorCorrectionCfgFn( errorCorrection, configFn ):
   # Configure the corresponding error-correction depending on the
   # interface type
   errorCorrectionCfgFns[ errorCorrection ] = fecTuple( configFn )

fecShowCapTuple = namedtuple( 'errorCorrectionShCapFn',
                              'checkFn checkFec printStr getFecs' )
errorCorrectionShCapFns = {}
def registerErrorCorrectionShCapFn( errorCorrection, checkFn, checkFec, printStr,
                                    getFecs ):
   """
   Print the corresponding capabilities for
   "show interfaces capabilities [ default ]"
   """
   errorCorrectionShCapFns[ errorCorrection ] = fecShowCapTuple( checkFn, checkFec,
                                                                 printStr, getFecs )

fecShowErrTuple = namedtuple( 'errorCorrectionShErrFn', 'showErr' )
errorCorrectionShErrFns = {}
def registerErrorCorrectionShErrFn( errorCorrection, showErr ):
   """
   Set the corresponding error-correction attrs for
   "show interfaces error-correction"
   """
   errorCorrectionShErrFns[ errorCorrection ] = fecShowErrTuple( showErr )

class EthIntfRangeModelet( CliParser.Modelet ):
   ''' Modelet for IntfRangeConfigMode '''
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.individualIntfModes_[ 0 ].intf, EthPhyIntf )

def _setIntfRangeLinkMode( mode, args ):
   ''' action for the cli "speed forced" in the "config-if-range" mode '''
   legacySpeedCmdSyntax = args.get( 'SPEED_TYPE' ) == '400gfull'

   # First run through the registered hooks and confirm that we can go this linkMode
   speedTypeDict = speedForceLinkMode if 'forced' in args else speedLinkMode
   linkMode = speedTypeDict[ args.get( 'SPEED_TYPE', 'linkModeUnknown' ) ]
   if checkChangeCancelHooks( mode, [ m.intf for m in mode.individualIntfModes_ ],
                              linkMode, None ):
      return
   if checkChangeCancelOnSfp28( mode,
                                [ m.intf for m in mode.individualIntfModes_ ],
                                linkMode,
                                None ):
      return

   for m in mode.individualIntfModes_:
      m.intf.setLinkMode( linkMode, legacySpeedCmdSyntax=legacySpeedCmdSyntax )
      Tracing.trace0( "Set " + m.intf.name + " speed/duplex to " +
                   str( linkMode ) + " ('None' means auto)" )

def _setIntfRangeAutoLinkMode( mode, args ):
   choice = speedAutoLinkMode.get( args.get( 'SPEED_TYPE' ) )

   linkMode = 'linkModeAutoneg'
   if choice == 'mode40GbpsFull':
      # As of 11/20/2015, we still need to use the special 40G link mode
      # for FocalPoint (perhaps others as well).  See BUG97475
      linkMode = 'linkModeAuto40GbpsFull'

   if choice:
      advertisedModes = EthLinkModeSet()
      try:
         setattr( advertisedModes, choice, True )
      except AttributeError:
         mode.addErrorAndStop( 'Cannot advertise the given speed and duplex' )
   else:
      advertisedModes = None

   if mode.session.commandConfirmation():
      for hook in canPromptForAbortHook.extensions():
         if hook( mode, [ m.intf for m in mode.individualIntfModes_ ], linkMode,
                  advertisedModes ):
            Tracing.trace0( "LinkMode change for " + mode.longModeKey +
                            " cancelled by " + str( hook ) )
            return

   for m in mode.individualIntfModes_:
      m.intf.setLinkMode( linkMode, advertisedModes )
      Tracing.trace0( "Set " + m.intf.name + " speed/duplex to " +
                   str( linkMode ) + " ('None' means auto)" )

def noSpeedIntfRange( mode, args ):
   _setIntfRangeLinkMode( mode, args )
   Tracing.trace0( "Set " + mode.longModeKey + " speed to default" )

class SpeedForcedRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'speed forced SPEED_TYPE'
   data = {
      'speed' : matcherSpeed,
      'forced' : 'Disable autoneg and force speed/duplex/flowcontrol',
      'SPEED_TYPE': speedForceExpression,
   }

   handler = _setIntfRangeLinkMode

EthIntfRangeModelet.addCommandClass( SpeedForcedRangeCmd )


class SpeedAutoRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'speed auto [ SPEED_TYPE ]'
   data = {
      'speed' : matcherSpeed,
      'auto' : matcherAuto,
      'SPEED_TYPE' : SpeedAutoExpression,
   }

   handler = _setIntfRangeAutoLinkMode

EthIntfRangeModelet.addCommandClass( SpeedAutoRangeCmd )


class SpeedRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'speed SPEED_TYPE'
   noOrDefaultSyntax = 'speed ...'
   data = {
      'speed' : matcherSpeed,
      'SPEED_TYPE': speedExpression
   }

   handler = _setIntfRangeLinkMode
   noOrDefaultHandler = noSpeedIntfRange

EthIntfRangeModelet.addCommandClass( SpeedRangeCmd )

#-------------------------------------------------------------------------------
# Associate the EthIntfRangeModelet with the "config-if-range" mode.
#-------------------------------------------------------------------------------
IntfRangeCli.IntfRangeConfigMode.addModelet( EthIntfRangeModelet )

#-------------------------------------------------------------------------------
# The "[no|default] transceiver qsfp default-mode <mode>" command, in "config" mode.
#-------------------------------------------------------------------------------
xcvrShowKw = CliMatcher.KeywordMatcher( 'transceiver',
                                        helpdesc='Show transceiver information' )

xcvrKw = CliMatcher.KeywordMatcher( 'transceiver',
                                    helpdesc='configure transceiver settings' )

qsfpKw = CliMatcher.KeywordMatcher( 'qsfp', helpdesc='QSFP transceiver' )

class XcvrQsfpDefaultMode( CliCommand.CliCommandClass ):
   syntax = 'transceiver qsfp default-mode 4x10G'
   noOrDefaultSyntax = 'transceiver qsfp default-mode ...'
   data = {
      'transceiver' : xcvrKw,
      'qsfp' : qsfpKw,
      'default-mode' : 'default mode',
      '4x10G': '4-by-10G mode',
   }

   @staticmethod
   def handler( mode, args ):
      qsfpConfig = ethPhyIntfDefaultConfigDir.xcvrModeConfig[ 'qsfp' ]
      if qsfpConfig.isSestoFixedSystem():
         qsfpConfig.globalMode = XcvrMode.noXcvrMode
      else:
         qsfpConfig.globalMode = XcvrMode.xcvrMode4x10G

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      qsfpConfig = ethPhyIntfDefaultConfigDir.xcvrModeConfig[ 'qsfp' ]
      qsfpConfig.globalMode = XcvrMode.noXcvrMode

BasicCli.GlobalConfigMode.addCommandClass( XcvrQsfpDefaultMode )

#-------------------------------------------------------------------------------
# Forward Error Correction (FEC) commands:

# "[no|default] error-correction encoding clause-91|clause-74"
# The no and default forms have different actions for this command.
#-------------------------------------------------------------------------------
# Allow the coherent keywords to show up if coherentErrorCorrection is
# supported on that interface.
def coherentFecTokenGuard( mode, token ):
   if ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.errorCorrection:
      return None
   if mode.intf.coherentErrorCorrectionSupported():
      return None
   return CliParser.guardNotThisPlatform

def cmisCoherentFecTokenGuard( mode, token ):
   if ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.errorCorrection:
      return None
   if mode.intf.cmisCoherentErrorCorrectionSupported():
      if ( token == 'concatenated' and
           mode.intf.coherentErrorCorrectionEncodingSupp( 'fecEncodingCfec' ) ):
         return None
      if ( token == 'open' and
           mode.intf.coherentErrorCorrectionEncodingSupp( 'fecEncodingOfec' ) ):
         return None
   return CliParser.guardNotThisPlatform

def nonCoherentFecTokenGuard( mode, token ):
   """
   Allow 'reed-solomon' keyword to show up on a coherent interface if the interface
   has 'g709' capability else reject all the client FECs. If the client FECs are
   configured on a empty DCO slots we need to allow them.
   """
   if ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.errorCorrection:
      return None
   if mode.intf.coherentErrorCorrectionSupported():
      if ( ( token == 'reed-solomon' and
             mode.intf.coherentErrorCorrectionEncodingSupp( 'fecEncodingG709' ) ) or
             mode.intf.errorCorrectionSupported() ):
         return None
      return CliParser.guardNotThisPlatform

   # Make sure we are not displaying client FEC for CMIS coherent modules unless we
   # have an empty CMIS coherent capable port (i.e. osfp/qsfpDd).
   if mode.intf.cmisCoherentErrorCorrectionSupported():
      if ( mode.intf.intfXcvrStatus().xcvrPresence == 'xcvrNotPresent' and
           mode.intf.errorCorrectionSupported() ):
         return None
      return CliParser.guardNotThisPlatform
   return None

nodeCFec = CliCommand.guardedKeyword( 'concatenated',
      helpdesc='Configure concatenated (C-FEC) forward error correction',
      guard=cmisCoherentFecTokenGuard )
nodeOFec = CliCommand.guardedKeyword( 'open',
      helpdesc='Configure open (O-FEC) forward error correction',
      guard=cmisCoherentFecTokenGuard )
nodeReedSolomon = CliCommand.guardedKeyword( 'reed-solomon',
      helpdesc='Configure Reed-Solomon (RS-FEC) forward error correction',
      guard=nonCoherentFecTokenGuard )
nodeSoftDecision = CliCommand.guardedKeyword( 'soft-decision',
      helpdesc='Configure soft decision based forward error correction',
      guard=coherentFecTokenGuard )
matcherG709 = CliMatcher.KeywordMatcher( 'g709',
      helpdesc='Configure Reed-Solomon G.709 forward error correction',
      value=lambda mode,match: 'coherentFecEncodingCfgG709' )
nodeG709ReedSolomon = CliCommand.Node( matcher=matcherG709,
      guard=coherentFecTokenGuard )
matcherOverhead = CliMatcher.KeywordMatcher( 'overhead',
      helpdesc='Configure the overhead for forward error correction' )
matcherPercent = CliMatcher.KeywordMatcher( 'percent',
      helpdesc='Percent' )
matcherBch = CliMatcher.KeywordMatcher( 'bch',
      helpdesc='Configure forward error correction with outer BCH code',
      value=lambda mode,match: 'coherentFecEncodingCfgSd25Bch' )
matcherOverhead7 = CliMatcher.KeywordMatcher( '7',
      helpdesc='Configure 7% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgHd7' )
matcherOverhead15 = CliMatcher.KeywordMatcher( '15',
      helpdesc='Configure 15% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgSd15' )
matcherOverhead20 = CliMatcher.KeywordMatcher( '20',
      helpdesc='Configure 20% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgSd20' )
matcherOverhead25 = CliMatcher.KeywordMatcher( '25',
      helpdesc='Configure 25% overhead for forward error correction',
      value=lambda mode, match: 'coherentFecEncodingCfgSd25' )

def setErrorCorrectionEncoding( mode, args ):
   noOrDefault = 'default' if CliCommand.isDefaultCmd( args ) \
         else CliCommand.isNoCmd( args )
   encodingValue = ( args.get( 'fire-code' ) or
                  args.get( 'g709' ) or
                  args.get( 'reed-solomon' ) or
                  args.get( '7' ) or
                  args.get( 'bch' ) or
                  args.get( '25' ) or
                  args.get( '20' ) or
                  args.get( '15' ) or
                  args.get( 'concatenated' ) or
                  args.get( 'open' ) )

   # The other FEC tokens make use of the KeywordMatcher "value" attribute which
   # can override the token value if matched. For C/O-FEC we used a guardedKeyword
   # which does not accept this attribute.
   cmisCoherentFecs = { 'concatenated' : 'coherentFecEncodingCfgCfec',
                        'open' : 'coherentFecEncodingCfgOfec',
                      }
   encodingValue = cmisCoherentFecs.get( encodingValue, encodingValue )

   mode.intf.setErrorCorrectionEncoding( encodingValue, noOrDefault )

#--------------------------------------------------------------------------------
# [ no | default ] error-correction encoding ( fire-code | reed-solomon [ g709 ] |
# ( soft-decision overhead ( 15 | 20 ) percent ) |
# soft-decision overhead 25 percent [ bch ] |
# ( staircase overhead 7 percent ) |
# ( concatenated ) |
# ( open ) )
#--------------------------------------------------------------------------------
class ErrorCorrectionEncodingCmd( CliCommand.CliCommandClass ):
   syntax = '''error-correction encoding
            ( ( fire-code )
            | ( reed-solomon [ g709 ] )
            | ( soft-decision overhead ( 15 | 20 ) percent )
            | ( soft-decision overhead 25 percent [ bch ] )
            | ( staircase overhead 7 percent )
            | ( concatenated )
            | ( open ) )'''
   noOrDefaultSyntax = '''error-correction encoding
            [ ( fire-code )
            | ( reed-solomon [ g709 ] )
            | ( soft-decision overhead ( 15 | 20 ) percent )
            | ( soft-decision overhead 25 percent [ bch ] )
            | ( staircase overhead 7 percent )
            | ( concatenated )
            | ( open ) ]'''
   data = {
      'error-correction' : 'Configure forward error correction options',
      'encoding' : 'Configure forward error encoding',
      'fire-code' : CliCommand.guardedKeyword( 'fire-code',
         helpdesc='Configure Fire code (BASE-R) forward error correction',
         guard=nonCoherentFecTokenGuard ),
      'reed-solomon' : CliCommand.guardedKeyword( 'reed-solomon',
         helpdesc='Configure Reed-Solomon (RS-FEC) forward error correction',
         guard=nonCoherentFecTokenGuard ),
      'g709' : nodeG709ReedSolomon,
      '7' : matcherOverhead7,
      'soft-decision' : nodeSoftDecision,
      'overhead' : matcherOverhead,
      '15' : matcherOverhead15,
      '20' : matcherOverhead20,
      'percent' : matcherPercent,
      '25' : matcherOverhead25,
      'bch' : matcherBch,
      'staircase' : CliCommand.guardedKeyword( 'staircase',
         helpdesc='Configure a staircase based forward error correction',
         guard=coherentFecTokenGuard ),
      'concatenated' : nodeCFec,
      'open' : nodeOFec,
   }

   handler = setErrorCorrectionEncoding
   noOrDefaultHandler = handler

EthIntfModelet.addCommandClass( ErrorCorrectionEncodingCmd )

class ErrorCorrectionReedSolomonBypassCmd( CliCommand.CliCommandClass ):
   ''' [ no | default ] error-correction reed-solomon bypass BYPASS_VALUE '''
   syntax = 'error-correction reed-solomon bypass BYPASS_VALUE'
   noOrDefaultSyntax = 'error-correction reed-solomon bypass ...'
   data = {
      'error-correction' : 'Configure forward error correction options',
      'reed-solomon' : nodeReedSolomon,
      'bypass' : 'Configure error correction bypass options',
      'BYPASS_VALUE' : CliMatcher.EnumMatcher( {
         'correction' : 'Disable correction of correctable errors',
         'indication' : 'Disable per frame indication of uncorrectable errors.',
      } )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.intf.setErrorCorrectionBypass( args[ 'BYPASS_VALUE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.noDefaultErrorCorrectionBypass()

EthIntfModelet.addCommandClass( ErrorCorrectionReedSolomonBypassCmd )

#-------------------------------------------------------------------------------
# "show interfaces error-correction" command.
#-------------------------------------------------------------------------------

# Order is important. Coherent FEC data needs to be filled in before
# client FEC data. This allows coherent to be set first which sets
# all the client FEC attributes to false. Having the client FEC code
# run after coherent FEC code allows re-writing the client FEC
# attributes if required. If not, they're correctly set to false.
def showErrorCorrection( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   interfacesErrCorr = EthIntfModel.InterfacesErrorCorrectionStatus()
   intfs = IntfCli.Intf.getAll( mode, intf, mod, intfType=L1Intf )
   if not intfs:
      return interfacesErrCorr

   FecEncodingStat = TacLazyType( 'Interface::EthFecEncoding' )
   fecStatusString = { FecEncodingStat.fecEncodingUnknown : 'unknown',
                       FecEncodingStat.fecEncodingDisabled : 'disabled',
                       FecEncodingStat.fecEncodingReedSolomon544 : 'reedSolomon',
                       FecEncodingStat.fecEncodingReedSolomon : 'reedSolomon',
                       FecEncodingStat.fecEncodingFireCode : 'fireCode' }

   fecEncodingConfigAttrXlate = {
      'fecEncodingDisabled' : 'fecEncodingConfigDisabled',
      'fecEncodingReedSolomon544' : 'fecEncodingConfigReedSolomon544Enabled',
      'fecEncodingReedSolomon' : 'fecEncodingConfigReedSolomonEnabled',
      'fecEncodingFireCode' : 'fecEncodingConfigFireCodeEnabled'
      }

   for x in intfs: # pylint: disable=too-many-nested-blocks
      if ( x.errorCorrectionSupported() or x.coherentErrorCorrectionSupported() or
           x.cmisCoherentErrorCorrectionSupported() ):
         errCorrStatus = EthIntfModel.InterfaceErrorCorrectionStatus()
         if ( x.coherentErrorCorrectionSupported() or
              x.cmisCoherentErrorCorrectionSupported() ):
            if "coherent" in errorCorrectionShErrFns:
               errCorrStatus = errorCorrectionShErrFns[ 'coherent' ].showErr(
                               x, errCorrStatus )
               if x.coherentErrorCorrectionSupported():
                  interfacesErrCorr._hasCoherentIntf = True

         if ( x.errorCorrectionSupported() and not
              x.cmisCoherentErrorCorrectionSupported() ):
            errCorrStatus._nonCoherentIntf = True # pylint:disable-msg=W0212
            errCorrStatus.errCorrEncodingStatus = fecStatusString[
                  x.status().fecStatus ]
            for tacAttr, modelAttr in fecEncodingConfigAttrXlate.items():
               tacVal = getattr( x.config().fecEncodingConfig, tacAttr )
               setattr( errCorrStatus, modelAttr, tacVal )

            # An encoding is available if fecEncodingDisabled is False,
            # the phy claims support for the encoding at the current speed,
            # and the configuration allows it. If the speed is unknown, we
            # are more lenient and use the logical or of the caps at all
            # speeds.
            if x.status().speed == EthSpeed.speedUnknown:
               # All caps will be initially False.
               fecCaps = Tac.Value( 'Interface::EthFecCapabilitySet' )
               for phyCaps in x.status().phyFecCapabilities.values():
                  for attr in phyCaps.attributes:
                     if attr in [ 'stringValue', 'fecEncodingDisabled',
                                  'setFecCapabilities' ]:
                        continue
                     val = getattr( fecCaps, attr ) or getattr( phyCaps, attr )
                     setattr( fecCaps, attr, val )
            elif x.status().autonegActive:
               fecCaps = None
               for i, j in EthIntfLib.autonegToforcedMapping.items():
                  if getattr( x.status().autonegStatus.advertisement.linkModes, i ):
                     fecCaps = x.status().phyFecCapabilities.get( j )
                     break
            else:
               try:
                  fecCaps = \
                          x.status().phyFecCapabilities[ x.status().linkModeStatus ]
               except KeyError:
                  fecCaps = None

            fecEncodingAvailXlate = { 'fecEncodingReedSolomon544' :
                                         'fecEncodingReedSolomon544Available',
                                      'fecEncodingReedSolomon' :
                                         'fecEncodingReedSolomonAvailable',
                                      'fecEncodingFireCode' :
                                         'fecEncodingFireCodeAvailable' }

            for tacAttr, modelAttr in fecEncodingAvailXlate.items():
               setattr( errCorrStatus, modelAttr, False )
               if fecCaps is not None and getattr( fecCaps, tacAttr ):
                  setattr( errCorrStatus, modelAttr, True )
         interfacesErrCorr.interfaceErrCorrStatuses[ x.name ] = errCorrStatus
   return interfacesErrCorr

class ShowIntfErrorCorrection( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces error-correction'
   data = { 'error-correction' : 'Show interface error correction information' }
   handler = showErrorCorrection
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfacesErrorCorrectionStatus

BasicCli.addShowCommandClass( ShowIntfErrorCorrection )

# -------------------------------------------------------------------------------
# "show interfaces link-flap damping" command.
# -------------------------------------------------------------------------------

# Look up interface damping status by interface ID
def getIntfStatus( sliceId, intfId ):
   intfStatusDir = linkFlapDampingIntfStatusDir.get( sliceId )
   if intfStatusDir is None:
      return None
   return intfStatusDir.intfStatus.get( intfId )

def decayedDemerit( oldDemerit, duration, decayHalfLife ):
   """
   Calculate a new demerit value considering that the
   "duration" seconds ago the demerit value was equal
   to "oldDemerit"
   """
   ratio = math.pow( 2, duration / decayHalfLife )
   if ratio == 0.0:
      return 0
   return int( oldDemerit / ratio )

def showInterfacesLinkFlapDamping( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   result = EthIntfModel.InterfaceLinkFlapDamping()

   intfs = getAllIntfs( mode, intf, mod, EthPhyIntf )
   if not intfs:
      return result

   for x in intfs:
      # CliPlugin.EthIntfCli.EthPhyIntf.slotId() can return None
      # for some types of interfaces, and Python is not happy about
      # dict lookups with None as a key
      if x.slotId() is None:
         continue
      intfStatus = getIntfStatus( x.slotId(), x.name )
      if intfStatus is None:
         continue
      intfData = EthIntfModel.LinkFlapDampingIntfData()
      utcCorrection = Tac.utcNow() - Tac.now()
      for profileName, profileStatus in intfStatus.profileStatus.items():
         profileConfig = linkFlapDampingConfig.profileConfig[ profileName ]
         profileData = EthIntfModel.LinkFlapDampingProfileData()
         profileData.demerit = profileStatus.totalDemerit
         if profileData.demerit:
            profileData.demerit = decayedDemerit(
               profileData.demerit,
               Tac.now() - profileStatus.totalDemeritLastUpdated,
               profileConfig.parameters.decayHalfLife )
         if profileStatus.damped:
            profileData.suppressed = profileStatus.lastSuppressTime + utcCorrection
            profileData.reuse = profileStatus.reuseTime + utcCorrection
         else:
            profileData.suppressed = 0.0
            profileData.reuse = 0.0
         intfData.profiles[ profileName ] = profileData

      result.interfaces[ x.name ] = intfData

   return result

class ShowIntfLinkFlapDamping( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces link-flap damping'
   data = {
      'link-flap': 'Display link flap information',
      'damping': 'Display link flap damping information',
   }
   handler = showInterfacesLinkFlapDamping
   moduleAtEnd = True
   cliModel = EthIntfModel.InterfaceLinkFlapDamping

BasicCli.addShowCommandClass( ShowIntfLinkFlapDamping )

#--------------------------------------------------------------------------------
# Add loopback CLI commands to the "config-if" mode
# traffic-loopback source system device phy
# [ no | default ] traffic-loopback source ( ( system device ( phy | mac ) ) |
#                                            ( network device phy ) )
#--------------------------------------------------------------------------------
def loopbackSystemMacSupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported( lbMode=LoopbackMode.loopbackMac ):
      return None
   return CliParser.guardNotThisPlatform

def loopbackSystemPhySupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported( lbMode=LoopbackMode.loopbackPhy ):
      return None
   return CliParser.guardNotThisPlatform

def loopbackSystemSupportedGuard( mode, token ):
   guard = CliParser.guardNotThisPlatform
   if ( loopbackSystemPhySupportedGuard( mode, token ) != guard or
        loopbackSystemMacSupportedGuard( mode, token ) != guard ):
      return None
   return CliParser.guardNotThisPlatform

def loopbackNetworkPhySupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported( lbMode=LoopbackMode.loopbackPhyRemote ):
      return None
   return CliParser.guardNotThisPlatform

def loopbackModeSupportedGuard( mode, token ):
   if mode.intf.loopbackModeSupported():
      return None
   return CliParser.guardNotThisPlatform

class SystemLoopbackExpression( CliCommand.CliExpression ):
   expression = ( 'system device ( SYSTEMPHY | mac )' )
   data = {
      'system' : CliCommand.guardedKeyword( 'system',
         helpdesc='Traffic from local host',
         guard=loopbackSystemSupportedGuard ),
      'device' : 'Configure loopback device',
      'SYSTEMPHY' : CliCommand.guardedKeyword( 'phy',
         helpdesc='implement loopback in PHY layer',
         guard=loopbackSystemPhySupportedGuard ),
      'mac' : CliCommand.guardedKeyword( 'mac',
         helpdesc='implement loopback in MAC layer',
         guard=loopbackSystemMacSupportedGuard ),
   }

class NetworkLoopbackExpression( CliCommand.CliExpression ):
   expression = ( 'network device NETWORKPHY' )
   data = {
      'device' : 'Configure loopback device',
      'network' : CliCommand.guardedKeyword( 'network',
         helpdesc='traffic from peer host',
         guard=loopbackNetworkPhySupportedGuard ),
      'NETWORKPHY' : CliCommand.guardedKeyword( 'phy',
         helpdesc='implement loopback in PHY layer',
         guard=loopbackNetworkPhySupportedGuard ),
   }

class LoopbackModeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traffic-loopback source ( SYSTEM | NETWORK )' )
   noOrDefaultSyntax = 'traffic-loopback ...'
   data = {
      'traffic-loopback' : CliCommand.guardedKeyword( 'traffic-loopback',
         helpdesc='Configure loopback',
         guard=loopbackModeSupportedGuard ),
      'source' : 'Configure traffic source',
      'SYSTEM' : SystemLoopbackExpression,
      'NETWORK' : NetworkLoopbackExpression
   }

   @staticmethod
   def handler( mode, args ):
      if 'system' in args:
         if 'SYSTEMPHY' in args:
            mode.intf.setLoopbackMode( 'system', 'phy' )
         else:
            mode.intf.setLoopbackMode( 'system', 'mac' )
      else:
         mode.intf.setLoopbackMode( 'network', 'phy' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setNoLoopbackMode()

EthIntfModelet.addCommandClass( LoopbackModeCmd )

#--------------------------------------------------------------------------------
# The CLI command to configure autoneg 25g/50g consortium operation
# [ no | default ] phy media 25gbase-cr negotiation standard { consortium | ieee }
#--------------------------------------------------------------------------------
def autonegConsortiumSupportedGuard( mode, token ):
   if mode.intf.autonegStandardSupported( 'consortium25g' ):
      return None
   return CliParser.guardNotThisPlatform

def autonegIeeeSupportedGuard( mode, token ):
   if mode.intf.autonegStandardSupported( 'ieee25g' ):
      return None
   return CliParser.guardNotThisPlatform

class PhyMedia25GbaseCrNegotiationStandardCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media 25gbase-cr negotiation standard { consortium | ieee }'
   noOrDefaultSyntax = 'phy media 25gbase-cr negotiation standard ...'
   data = {
      'phy' : 'Configure phy-specific parameters',
      'media' : 'Configure media specific options',
      '25gbase-cr' : 'Options for 25gBase-cr media',
      'negotiation' : 'Configure auto-negotiation options',
      'standard' : 'Configure auto-negotiation standards',
      'consortium' : CliCommand.guardedKeyword( 'consortium',
         helpdesc='Configure 25G/50G ethernet consortium operation',
         guard=autonegConsortiumSupportedGuard, maxMatches=1 ),
      'ieee' : CliCommand.guardedKeyword( 'ieee',
         helpdesc='Configure IEEE 802.3 Clause 73 standard operation',
         guard=autonegIeeeSupportedGuard, maxMatches=1 ),
   }

   @staticmethod
   def handler( mode, args ):
      aneg25gModes = []
      if 'consortium' in args:
         aneg25gModes.append( 'consortium25g' )
      if 'ieee' in args:
         aneg25gModes.append( 'ieee25g' )
      mode.intf.setAutonegStandard( aneg25gModes )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.setAutonegStandard( None )

EthIntfModelet.addCommandClass( PhyMedia25GbaseCrNegotiationStandardCmd )
