#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
from CliCommand import (
      CliExpression,
      Node,
      singleKeyword
)
import ShowCommand
from CliPlugin.ArBgpCli import (
      ShowBgpDebugPolicyBase,
)
from CliPlugin.BgpCliModels import BgpSummary
from CliPlugin.BgpVpnCli import (
      showExtCommunity,
      LargeCommValuesAndExactExpression
)
from CliPlugin.EvpnCli import (
   doShowBgpEvpn,
   doShowBgpEvpnL2,
   doShowBgpEvpnSummary,
)
from CliPlugin.Esid import esiMatcher
from CliPlugin.Ip6AddrMatcher import (
      Ip6AddrMatcher,
      Ip6PrefixMatcher,
)
from CliPlugin.IpAddrMatcher import (
      IpAddrMatcher,
      IpPrefixMatcher
)
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.RouteMapCli import (
      RtSooExtCommCliMatcher
)
from CliPlugin.RouteDistinguisher import RdDistinguisherMatcher
from CliPlugin.RoutingBgpCli import (
      V4V6PeerKeyCliExpression )
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.VxlanCli import vniMatcher
from CliToken.Evpn import evpnShow
from CliToken.RoutingBgpShowCliTokens import (
      neighbors,
      communityExact,
      received,
      detail,
      bgpAfterShow,
      ipv4AfterShowBgp,
      ipv6AfterShowBgp,
      longerPrefixes,
      advertised,
      receivedAll,
      summary,
      CommunityValuesAndExactRule
)
from Toggles import ArBgpToggleLib

matcherAll = CliMatcher.KeywordMatcher( 'all',
      helpdesc='All configured VRFs' )
matcherAllActive = CliMatcher.KeywordMatcher( 'all-active',
      helpdesc='All-Active redundancy mode' )
# matcher for different route types
matcherAutoDiscovery = CliMatcher.KeywordMatcher( 'auto-discovery',
      helpdesc='Filter by Ethernet auto-discovery (A-D) route (type 1)' )
matcherMacIp = CliMatcher.KeywordMatcher( 'mac-ip',
      helpdesc='Filter by MAC/IP advertisement route (type 2)' )
matcherImet = CliMatcher.KeywordMatcher( 'imet',
      helpdesc='Filter by inclusive multicast Ethernet tag route (type 3)' )
matcherEthernetSegment = CliMatcher.KeywordMatcher( 'ethernet-segment',
      helpdesc='Filter by Ethernet segment route (type 4)' )
matcherIpPrefix = CliMatcher.KeywordMatcher( 'ip-prefix',
      helpdesc='Filter by IP prefix route (type 5)' )
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
      helpdesc='Limit address family to IPv4' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='Limit address family to IPv6' )
matcherSmet = CliMatcher.KeywordMatcher( 'smet',
      helpdesc='Filter by selective multicast Ethernet tag route (type 6)' )
matcherJoinSync = CliMatcher.KeywordMatcher( 'join-sync',
      helpdesc='Filter by multicast join sync route (type 7)' )
matcherLeaveSync = CliMatcher.KeywordMatcher( 'leave-sync',
      helpdesc='Filter by multicast leave sync route (type 8)' )
matcherSpmsi = CliMatcher.KeywordMatcher( 'spmsi',
      helpdesc='Filter by selective PMSI auto discovery route (type 10)' )
matcherRemote = CliMatcher.KeywordMatcher( 'remote',
      helpdesc='remote domain' )
matcherLocal = CliMatcher.KeywordMatcher( 'local',
      helpdesc='local domain' )
matcherDomain = CliMatcher.KeywordMatcher( 'domain',
      helpdesc='Filter by route domain' )

matcherCommunity = CliMatcher.KeywordMatcher( 'community',
      helpdesc='Match BGP community list' )
matcherCount = CliMatcher.KeywordMatcher( 'count',
      helpdesc='Route-type based path count' )
matcherEsi = CliMatcher.KeywordMatcher( 'esi',
      helpdesc='Filter by Ethernet Segment Identifier' )
matcherGshut = CliMatcher.KeywordMatcher( 'GSHUT',
      helpdesc='Graceful Shutdown (well-known community)' )
matcherInternet = CliMatcher.KeywordMatcher( 'internet',
      helpdesc='Internet (well-known community)' )
matcherLargeCommunity = CliMatcher.KeywordMatcher( 'LARGE_COMMUNITY',
      helpdesc='Match BGP large community list' )
matcherLocalAs = CliMatcher.KeywordMatcher( 'local-as',
      helpdesc='Do not send outside local AS' )
matcherNextHop = CliMatcher.KeywordMatcher( 'next-hop',
      helpdesc='Filter by next hop (remote VTEP) IPv4 or IPv6 address' )
matcherNoAdvertise = CliMatcher.KeywordMatcher( 'no-advertise',
      helpdesc='Do not advertise to any peer' )
matcherNoExport = CliMatcher.KeywordMatcher( 'no-export',
      helpdesc='Do not export to next AS' )
matcherAdAliasing = CliMatcher.KeywordMatcher( 'aliasing',
      helpdesc='Show only multi-homing aliasing routes' )
matcherAdFastConvergence = CliMatcher.KeywordMatcher( 'fast-convergence',
      helpdesc='Show only multi-homing fast convergence routes' )
matcherRd = CliMatcher.KeywordMatcher( 'rd',
      helpdesc='Filter by route distinguisher' )
matcherRouteType = CliMatcher.KeywordMatcher( 'route-type',
      helpdesc='Filter by NLRI route type' )
matcherRt = CliMatcher.KeywordMatcher( 'rt',
      helpdesc='Route Target' )
matcherSingleActive = CliMatcher.KeywordMatcher( 'single-active',
      helpdesc='Single-Active redundancy mode' )
matcherSticky = CliMatcher.KeywordMatcher( 'sticky',
      helpdesc='Sticky MAC' )
matcherVni = CliMatcher.KeywordMatcher( 'vni',
      helpdesc='Filter by VXLAN network identifier' )
matcherVrf = CliMatcher.KeywordMatcher( 'vrf',
      helpdesc='VRF name' )
matcherVxlan = CliMatcher.KeywordMatcher( 'vxlan',
      helpdesc='VXLAN Tunnel' )

vlanIdMatcher = CliMatcher.IntegerMatcher( 1, 4094,
                                           helpdesc='Identifier for a Virtual LAN' )

nodeTunnelEncap = singleKeyword( 'tunnel-encap', helpdesc='Tunnel Encap' )

nodeEsiLabel = singleKeyword( 'esi-label', helpdesc='ESI Label' )

nodeRouterMac = singleKeyword( 'router-mac', helpdesc='Router MAC' )

nodeMacMobility = singleKeyword( 'mac-mobility', helpdesc='MAC Mobility' )

redundancyMatcher = CliMatcher.EnumMatcher( {
   'single-active' : 'Single-Active redundancy mode',
   'all-active' : 'All-Active redundancy mode'
   } )

class EvpnExtCommunityConstExpr( CliExpression ):
   expression = ''' ( rt RT_VAL ) | 
                    ( tunnel-encap vxlan ) | 
                    ( esi-label ESI_LABEL_VAL REDUN_MATCHER ) | 
                    ( router-mac ROUTER_MAC_VAL ) | 
                    ( mac-mobility ( sticky | SEQ_NO ) )'''
   data = {
      'rt' : 'Route Target',
      'RT_VAL' : RtSooExtCommCliMatcher( 'Route Target' ),
      'tunnel-encap' : nodeTunnelEncap,
      'vxlan' : matcherVxlan,
      'esi-label' : nodeEsiLabel,
      'ESI_LABEL_VAL' : CliMatcher.IntegerMatcher( 0, 16777215,
         helpdesc='ESI Label value' ),
      'REDUN_MATCHER' : redundancyMatcher,
      'router-mac' : nodeRouterMac,
      'ROUTER_MAC_VAL' : MacAddrMatcher( helpdesc='Ethernet address' ),
      'mac-mobility' : nodeMacMobility,
      'sticky' : matcherSticky,
      'SEQ_NO' : CliMatcher.IntegerMatcher( 0, 2**32 - 1,
         helpdesc='Index in the sequence' ),
   }

   @staticmethod
   def adapter( mode, args, argList ):
      extCommunityValuesDict = { 'extCommunityValues' : None, 
                                 'exact' : None }
      extCommuniltyList = []

      rt = args.pop( 'rt', None )
      if rt:
         rtList = []
         RT_VAL = args.pop( 'RT_VAL', None )
         if RT_VAL:
            rtList.append( ( 'rt', RT_VAL ) )
            extCommuniltyList = extCommuniltyList + rtList

      tunnelEncap = args.pop( 'tunnel-encap', None )
      if tunnelEncap:
         vxlan = args.pop( 'vxlan', None )
         if vxlan:
            t = ( 'tunnelEncap', 'vxlan' )
            extCommuniltyList.append( t )

      esiLabel = args.pop( 'esi-label', None )
      if esiLabel:
         esiLabelDict = { 'esiLabelRedundancyMode' : None, 'esiLabelValue' : None }
         ESI_LABEL_VAL = args.pop( 'ESI_LABEL_VAL', None )
         if ESI_LABEL_VAL:
            esiLabelDict[ 'esiLabelValue' ] = ESI_LABEL_VAL[ 0 ]
            esiLabelRedundancyMode = args.get( 'REDUN_MATCHER' )
            esiLabelDict[ 'esiLabelRedundancyMode' ] = esiLabelRedundancyMode[ 0 ]
         t = ( 'esiLabel', esiLabelDict )
         extCommuniltyList.append( t )

      routerMac = args.pop( 'router-mac', None )
      if routerMac:
         ROUTER_MAC_VAL = args.pop( 'ROUTER_MAC_VAL', None )
         if ROUTER_MAC_VAL:
            t = ( 'routerMac', ROUTER_MAC_VAL[ 0 ] )
            extCommuniltyList.append( t )

      macMobility = args.pop( 'mac-mobility', None )
      if macMobility:
         if args.pop( 'sticky', None ):
            t = ( 'macMobility', 'sticky' )
            extCommuniltyList.append( t )

         SEQ_NO = args.pop( 'SEQ_NO', None )
         if SEQ_NO:
            t = ( 'macMobility', SEQ_NO[ 0 ] )
            extCommuniltyList.append( t )

      if extCommuniltyList:
         extCommunityValuesDict[ 'extCommunityValues' ] = extCommuniltyList
         args[ 'extCommValuesAndExact' ] = extCommunityValuesDict

class EvpnExtCommValuesAndExactExpression( CliExpression ):
   expression = 'extcommunity { EXT_COMM_VALUE } [ EXT_COM_EXACT ] '
   data = {
         'extcommunity' : showExtCommunity,
         'EXT_COMM_VALUE' : EvpnExtCommunityConstExpr,
         'EXT_COM_EXACT' : communityExact,
   }

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( 'extcommunity', None ):
         return
      if args.pop( 'EXT_COM_EXACT', None ):
         args[ 'extCommValuesAndExact' ][ 'exact' ] = 'exact'

multicastRouteExpression = '''| ( smet |
                                 join-sync |
                                 leave-sync |
                                 spmsi
                                 [ ( ipv4Token | ipv6Token ) ]
                               )'''

multicastRoutesDict = {
   'smet' : matcherSmet,
   'join-sync' : matcherJoinSync,
   'ipv4Token' : ipv4AfterShowBgp,
   'ipv6Token' : ipv6AfterShowBgp,
   'leave-sync' : matcherLeaveSync,
   'spmsi' : matcherSpmsi
}

class EvpnRouteTypeValueAndPrefixExpression( CliExpression ):
   # pylint: disable-next=consider-using-f-string
   expression = '''route-type
         ( ( auto-discovery [ aliasing | fast-convergence ] ) | 
           ( mac-ip [  MACADDR | IPADDR | IP6ADDR ] ) |
           ( imet [ ( ORIGIP | ORIGIP6 ) ] ) |
           ( ethernet-segment ) |
           ( ip-prefix 
            ( ( ipv4 ) | 
              ( ipv6 ) |
              ( PREFIX_V4 [ longer-prefixes ] ) |
              ( PREFIX_V6 [ longer-prefixes ] ) ) 
           ) %s 
         )''' % multicastRouteExpression 

   data = {
      'route-type' : matcherRouteType,
      'auto-discovery' : matcherAutoDiscovery,
      'aliasing' : matcherAdAliasing,
      'fast-convergence' : matcherAdFastConvergence,
      'mac-ip' : matcherMacIp,
      'MACADDR' : MacAddrMatcher( helpdesc='Ethernet address' ),
      'IPADDR' : IpAddrMatcher( helpdesc='Filter by IPv4 address' ),
      'IP6ADDR' : Ip6AddrMatcher( helpdesc='Filter by IPv6 address' ),
      'imet' : matcherImet,
      'ORIGIP' : IpAddrMatcher( helpdesc='Originating router IPv4 address' ),
      'ORIGIP6' : Ip6AddrMatcher( helpdesc='Originating router IPv6 address' ),
      'ethernet-segment' : matcherEthernetSegment,
      'ip-prefix' : matcherIpPrefix,
      'ipv4' : matcherIpv4,
      'ipv6' : matcherIpv6,
      'PREFIX_V4' : IpPrefixMatcher( 'IPv4 address prefix' ),
      'PREFIX_V6' : Ip6PrefixMatcher( 'IPv6 address prefix' ),
      'longer-prefixes' : longerPrefixes,
   }
   data.update( multicastRoutesDict )

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( 'route-type', None ):
         return
      nlriTypeAndPrefixDict = { 'nlriTypeAndPrefixValues' :
         { 'nlriType' : None,
           'autoDiscoveryType' : None,
           'addressFamily' : None,
           'macOrIpAddress' : None,
           'origIp' : None,
           'prefix' : { 'longerPrefixes' : None, 'prefixValue' : None } } }

      def parseAddrFamily():
         if args.pop( 'ipv4Token', None ):
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'addressFamily' ]\
                  = 'ipv4'
         if args.pop( 'ipv6Token', None ):
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'addressFamily' ]\
                  = 'ipv6'

      def parseLongerPrefixes():
         longerPrefixesToken = args.pop( 'longer-prefixes', None )
         if longerPrefixesToken:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'prefix' ]\
                  [ 'longerPrefixes' ] = longerPrefixesToken

      if args.pop( 'auto-discovery', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType1'
         aliasing = args.pop( 'aliasing', None )
         if aliasing:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ]\
                  [ 'autoDiscoveryType' ] = 'aliasing'
         fastConvergence = args.pop( 'fast-convergence', None )
         if fastConvergence:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ]\
                  [ 'autoDiscoveryType' ] = 'fast-convergence'

      elif args.pop( 'mac-ip', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType2'
         MACADDR = args.pop( 'MACADDR', None )
         if MACADDR:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ]\
                  [ 'macOrIpAddress' ] = MACADDR
         IPADDR = args.pop( 'IPADDR', None )
         if IPADDR:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ]\
                  [ 'macOrIpAddress' ] = IPADDR
         IP6ADDR = args.pop( 'IP6ADDR', None )
         if IP6ADDR:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ]\
                  [ 'macOrIpAddress' ] = IP6ADDR

      elif args.pop( 'imet', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType3'
         ORIGIP = args.pop( 'ORIGIP', None )
         if ORIGIP:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'origIp' ] = ORIGIP
         ORIGIP6 = args.pop( 'ORIGIP6', None )
         if ORIGIP6:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'origIp' ] = ORIGIP6

      elif args.pop( 'ethernet-segment', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType4'

      elif args.pop( 'ip-prefix', None ):
         ipv4 = args.pop( 'ipv4', None )
         if ipv4:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] =\
                  'evpnType5Ipv4'
         ipv6 = args.pop( 'ipv6', None )
         if ipv6:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] =\
                  'evpnType5Ipv6'
         PREFIX_V4 = args.pop( 'PREFIX_V4', None )
         if PREFIX_V4:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] =\
                  'evpnType5Ipv4'
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'prefix' ]\
                  [ 'prefixValue' ] = PREFIX_V4
            parseLongerPrefixes()

         PREFIX_V6 = args.pop( 'PREFIX_V6', None )

         # Ip6PrefixMatcher returns an Arnet::Ip6AddrWithMask value,
         # which evaluates to False if the default value is passed (::/0).
         # So instead, we check against None
         if PREFIX_V6 is not None:
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] =\
                  'evpnType5Ipv6'

            # If the default value (::/0) is passed, we convert to the route string.
            nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'prefix' ]\
                  [ 'prefixValue' ] = PREFIX_V6 if PREFIX_V6 else \
                                      PREFIX_V6.stringValue
            parseLongerPrefixes()

      elif args.pop( 'smet', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType6'
         parseAddrFamily()

      elif args.pop( 'join-sync', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType7'
         parseAddrFamily()

      elif args.pop( 'leave-sync', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType8'
         parseAddrFamily()

      elif args.pop( 'spmsi', None ):
         nlriTypeAndPrefixDict[ 'nlriTypeAndPrefixValues' ][ 'nlriType' ] = \
               'evpnType10'
         parseAddrFamily()

      args.update( nlriTypeAndPrefixDict )

matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='Filter by multicast group and/or source' )

class EvpnMulticastExpression( CliExpression ):
   expression = 'multicast FIRST_VAL [ SECOND_VAL ]'
   data = {
      'multicast' : matcherMulticast,
      'FIRST_VAL' : IpGenAddrMatcher(
         helpdesc='Multicast group or source address' ),
      'SECOND_VAL' : IpGenAddrMatcher(
         helpdesc='Multicast group or source address' )
   }

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( 'multicast', None ):
         return

      FIRST_VAL = args.pop( 'FIRST_VAL', None )
      if FIRST_VAL:
         args[ 'multicast' ] = { 'first' : FIRST_VAL.stringValue }

      SECOND_VAL = args.pop( 'SECOND_VAL', None )
      if SECOND_VAL:
         args[ 'multicast' ].update( { 'second' : SECOND_VAL.stringValue } )

multicastExpression = '[ Multicast ]'

multicastDict = {
   'Multicast' : EvpnMulticastExpression
}

class EvpnNexthop( CliExpression ):
   expression = 'next-hop ( NEXTHOPV4 | NEXTHOPV6 )'
   data = {
      'next-hop' : matcherNextHop,
      'NEXTHOPV4' : IpAddrMatcher( helpdesc='IPv4 address' ),
      'NEXTHOPV6' : Ip6AddrMatcher( helpdesc='IPv6 address' ),
   }

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( 'next-hop', None ):
         return

      NEXTHOPV4 = args.pop( 'NEXTHOPV4', None )
      if NEXTHOPV4:
         args[ 'nexthopValue' ] = NEXTHOPV4

      NEXTHOPV6 = args.pop( 'NEXTHOPV6', None )
      if NEXTHOPV6:
         args[ 'nexthopValue' ] = NEXTHOPV6

class EvpnDomain( CliExpression ):
   expression = 'domain ( remote | local )'
   data = {
      'domain' : matcherDomain,
      'remote' : matcherRemote,
      'local' : matcherLocal,
   }

   @staticmethod
   def adapter( mode, args, argList ):
      if 'domain' not in args:
         return
      if 'remote' in args:
         args[ 'domain' ] = 'remote'
      else:
         args[ 'domain' ] = 'local'

class EvpnOptions( CliExpression ):
   # pylint: disable-next=consider-using-f-string
   expression = '''[ community ] 
                   [ EXT_COM ] 
                   [ LARGE_COMMUNITY ] 
                   [ RT_VALUE_AND_PREFIX ] 
                   [ rd RD ] 
                   [ vni VNI ] 
                   [ NEXTHOP ] 
                   [ esi ESI ] 
                   [ DOMAIN ]
                   %s  
                   [ detail ]''' % multicastExpression
   data = {
      'community' : CommunityValuesAndExactRule,
      'EXT_COM' : EvpnExtCommValuesAndExactExpression,
      'LARGE_COMMUNITY' : LargeCommValuesAndExactExpression,
      'RT_VALUE_AND_PREFIX' : EvpnRouteTypeValueAndPrefixExpression,
      'rd' : matcherRd,
      'RD' : RdDistinguisherMatcher( helpdesc='BGP route distinguisher' ),
      'vni' : matcherVni,
      'VNI' : vniMatcher,
      'NEXTHOP' : EvpnNexthop,
      'esi' : matcherEsi,
      'ESI' : esiMatcher,
      'DOMAIN' : EvpnDomain,
      'detail' : detail,
   }
   data.update( multicastDict )

   @staticmethod
   def adapter( mode, args, argList ):
      LARGE_COMMUNITIES = args.pop( 'LARGE_COMMUNITIES', None )
      if LARGE_COMMUNITIES:
         args[ 'largeCommValuesAndExact' ] = LARGE_COMMUNITIES
      communityValuesAndExact = args.pop( 'communityValuesAndExact', None )
      if communityValuesAndExact:
         args[ 'commValuesAndExact' ] = communityValuesAndExact
      RD = args.pop( 'RD', None )
      if RD :
         args['rdValue'] = RD
      ESI = args.pop( 'ESI', None )
      if ESI :
         args['esiValue'] = ESI 
      VNI = args.pop( 'VNI', None )
      if VNI :
         args['vniValue'] = VNI 

class BgpEvpnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp evpn [EVPN_OPTIONS]'
   data = {
      'bgp' : bgpAfterShow,
      'evpn' : evpnShow,
      'EVPN_OPTIONS' : EvpnOptions
   }

   handler = doShowBgpEvpn
   cliModel = "EvpnCliModels.EvpnRoutes"

BasicCli.addShowCommandClass( BgpEvpnCmd )

class BgpEvpnL2Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp evpn ( arp | mac ) [ vni VNI ]'
   data = {
      'bgp' : bgpAfterShow,
      'evpn' : evpnShow,
      'arp' : 'Display EVPN ARP table',
      'mac' : 'Display EVPN MAC table',
      'vni' : matcherVni,
      'VNI' : vniMatcher,
   }
   handler = doShowBgpEvpnL2
   cliModel = "EvpnCliModels.EvpnRoutesL2"

if ArBgpToggleLib.toggleBgpBribExportL2TableEnabled():
   BasicCli.addShowCommandClass( BgpEvpnL2Cmd )

class BgpEvpnRouteTypeCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp evpn route-type count'
   data = {
      'bgp' : bgpAfterShow,
      'evpn' : evpnShow,
      'route-type' : matcherRouteType,
      'count' : matcherCount,
   }

   handler = "EvpnShowCommandsHandler.doShowBgpEvpnRouteTypeCount"
   cliModel = "EvpnCliModels.EvpnPathRouteTypeCount"

BasicCli.addShowCommandClass( BgpEvpnRouteTypeCountCmd )

multicastRouteTypeExpression = '| smet | join-sync | leave-sync | spmsi'
multicastRouteTypeDict = {
   'smet' : matcherSmet,
   'join-sync' : matcherJoinSync,
   'leave-sync' : matcherLeaveSync,
   'spmsi' : matcherSpmsi
}

class EvpnRouteTypeExpression( CliExpression ):
   # pylint: disable-next=consider-using-f-string
   expression = '''route-type
         ( auto-discovery |
           mac-ip |
           imet |
           ethernet-segment |
           ( ip-prefix ( ipv4 | ipv6 ) )
           %s )''' % multicastRouteTypeExpression 

   data = {
      'route-type' : matcherRouteType,
      'auto-discovery' : matcherAutoDiscovery,
      'mac-ip' : matcherMacIp,
      'imet' : matcherImet,
      'ethernet-segment' : matcherEthernetSegment,
      'ip-prefix' : matcherIpPrefix,
      'ipv4' : matcherIpv4,
      'ipv6' : matcherIpv6
   }
   data.update( multicastRouteTypeDict )

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( 'route-type', None ):
         return
      nlriType = None
      if args.pop( 'auto-discovery', None ):
         nlriType = 'evpnType1'
      if args.pop( 'mac-ip', None ):
         nlriType = 'evpnType2'
      if args.pop( 'imet', None ):
         nlriType = 'evpnType3'
      if args.pop( 'ethernet-segment', None ):
         nlriType = 'evpnType4'
      if args.pop( 'ip-prefix', None ):
         ipv4 = args.pop( 'ipv4', None )
         if ipv4:
            nlriType = 'evpnType5Ipv4'
         ipv6 = args.pop( 'ipv6', None )
         if ipv6:
            nlriType = 'evpnType5Ipv6'
      if args.pop( 'smet', None ):
         nlriType = 'evpnType6'
      if args.pop( 'join-sync', None ):
         nlriType = 'evpnType7'
      if args.pop( 'leave-sync', None ):
         nlriType = 'evpnType8'
      if args.pop( 'spmsi', None ):
         nlriType = 'evpnType10'
      args[ 'nlriTypeValue' ] = { 'nlriType' : nlriType }

# -------------------------------------------------------------------------------
# "show bgp evpn summary [ route-type auto-discovery | mac-ip | imet |
#                                     ethernet-segment | ( ip-prefix ipv4 | ipv6 ) |
#                                     smet | spmsi |join-sync | leave-sync ]"
# -------------------------------------------------------------------------------
# Show BGP EVPN peers and optionally filter by route-type.
# When route-type is not specified, summary statistics reflect the number of routes
# received and accepted for all evpn route-types.
# When a route-type is specified, summary statistics reflect the number of routes
# received and accepted for the specified route-type.
class BgpEvpnSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp evpn summary [ROUTE_TYPE]'
   data = {
      'bgp' : bgpAfterShow,
      'evpn' : evpnShow,
      'summary' : summary,
      'ROUTE_TYPE' : EvpnRouteTypeExpression
   }

   handler = doShowBgpEvpnSummary
   cliModel = BgpSummary

BasicCli.addShowCommandClass( BgpEvpnSummaryCmd )

class BgpEvpnInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp evpn RT_VALUE_AND_PREFIX [ esi ESI ] internal"""
   data = {
      'bgp' : bgpAfterShow,
      'evpn' : evpnShow,
      'RT_VALUE_AND_PREFIX' : EvpnRouteTypeValueAndPrefixExpression,
      'esi' : matcherEsi,
      'ESI' : esiMatcher,
      'internal' : Node(
         matcher=CliMatcher.KeywordMatcher( 'internal',
            helpdesc='Dump internal state' ), hidden=True )
   }
   handler = doShowBgpEvpn
   cliModel = "EvpnCliModels.EvpnRoutes"

BasicCli.addShowCommandClass( BgpEvpnInternalCmd )

class BgpNeighborsEvpnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp neighbors PEER_ADDR evpn 
               ( EVPN_ADVT_ROUTES | EVPN_ROUTES | EVPN_RECV_ROUTES ) 
               [EVPN_OPTIONS]'''

   data = {
      'bgp' : bgpAfterShow,
      'neighbors' : neighbors,
      'PEER_ADDR' : V4V6PeerKeyCliExpression,
      'evpn' : evpnShow,
      'EVPN_ADVT_ROUTES' : advertised,
      'EVPN_ROUTES' : received,
      'EVPN_RECV_ROUTES' : receivedAll,
      'EVPN_OPTIONS' : EvpnOptions
   }

   handler = doShowBgpEvpn
   cliModel = "EvpnCliModels.EvpnRoutes"

   @staticmethod
   def adapter( mode, args, argList ):
      if args.pop( 'EVPN_ROUTES', None ):
         args[ 'bgpRouteTypeValue' ] = 'routes'
      if args.pop( 'EVPN_ADVT_ROUTES', None ):
         args[ 'bgpRouteTypeValue' ] = 'advertised-routes'
      if args.pop( 'EVPN_RECV_ROUTES', None ):
         args[ 'bgpRouteTypeValue' ] = 'received-routes'
      PEER = args.pop( 'PEER', None )
      if PEER:
         args[ 'peerAddrValue' ] = PEER.stringValue

BasicCli.addShowCommandClass( BgpNeighborsEvpnCmd )

# --------------------------------------------------------------------------------
# show bgp evpn sanity [ brief | detail ]
# --------------------------------------------------------------------------------
class EvpnSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp evpn sanity [ brief | detail ]'
   data = {
      'bgp' : bgpAfterShow,
      'evpn' : evpnShow,
      'sanity' : 'EVPN Sanity',
      'brief' : 'Show brief information',
      'detail' : 'Show comprehensive output'
   }
   handler = "EvpnShowCommandsHandler.showEvpnSanity"
   cliModel = "EvpnCliModels.EvpnSanityModel"

BasicCli.addShowCommandClass( EvpnSanityCmd )

class ShowBgpDebugPolicyEvpnBase( ShowBgpDebugPolicyBase ):
   data = ShowBgpDebugPolicyBase.data.copy()
   data.update( {
      'export' : 'Debug export policy application',
      'evpn' : evpnShow,
      'route-type' : matcherRouteType,
      'ip-prefix' : matcherIpPrefix,
      'rd' : matcherRd,
      'RD' : RdDistinguisherMatcher( helpdesc='BGP route distinguisher' ) } )
   handler = "EvpnShowCommandsHandler.policyDebugEvpnHandler"

# --------------------------------------------------------------------------------
# show bgp debug policy ( inbound | outbound ) neighbor ( ADDR | ADDR6 | all )
# evpn [ rcf RCF_FUNC_NAME() ] route-type ip-prefix ( PREFIX | PREFIX6 ) rd RD
# --------------------------------------------------------------------------------
class ShowBgpDebugPolicyEvpnInbound( ShowBgpDebugPolicyEvpnBase ):
   syntax = 'show bgp debug policy ' \
            'inbound neighbor ( ADDR | ADDR6 | all ) evpn [ rcf RCF_FUNC_NAME ] ' \
            'route-type ip-prefix ( PREFIX | PREFIX6 ) rd RD'
   data = ShowBgpDebugPolicyEvpnBase.data.copy()

BasicCli.addShowCommandClass( ShowBgpDebugPolicyEvpnInbound )

class ShowBgpDebugPolicyEvpnOutbound( ShowBgpDebugPolicyEvpnBase ):
   syntax = 'show bgp debug policy ' \
            'outbound neighbor ( ADDR | ADDR6 ) evpn [ rcf RCF_FUNC_NAME ] ' \
            'route-type ip-prefix ( PREFIX | PREFIX6 ) rd RD'
   data = ShowBgpDebugPolicyEvpnBase.data.copy()

BasicCli.addShowCommandClass( ShowBgpDebugPolicyEvpnOutbound )

# --------------------------------------------------------------------------------
# show bgp debug policy export evpn [ vrf <vrf name> ] [ rcf <function name>() ]
#    <prefix>
# --------------------------------------------------------------------------------
class ShowBgpDebugPolicyEvpnExport( ShowBgpDebugPolicyEvpnBase ):
   syntax = '''show bgp debug policy export
            evpn [ VRF ] [ rcf RCF_FUNC_NAME ] ( PREFIX | PREFIX6 )'''
   data = ShowBgpDebugPolicyEvpnBase.data.copy()

BasicCli.addShowCommandClass( ShowBgpDebugPolicyEvpnExport )
