#!/usr/bin/env python3
# Copyright (c) 2009-2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#-------------------------------------------------------------------------------
# This module implements EOS extension management.
#
#     show extensions [detail]
#     show installed-extensions
#
# In privileged exec mode:
#
#     [no] extension <extension-name> [force]
#-------------------------------------------------------------------------------

import errno
import os
import sys
import threading

import Ark
import AgentDirectory
import BasicCli
import BasicCliModes
import BasicCliUtil
import CliCommand
import CliMode.Package
import CliMatcher
import CliParser
import CliPlugin.TechSupportCli
from CliPlugin import ConfigMgmtMode
from CliPlugin import WaitForWarmupCli
from CliPlugin.ExtensionMgrCliModel import BootExtensions
from CliPlugin.ExtensionMgrCliModel import Extension
from CliPlugin.ExtensionMgrCliModel import Extensions
from CliPlugin.ExtensionMgrCliModel import InstallationStatus
from CliPlugin.ExtensionMgrCliModel import InstallationStatuses
from CliPlugin.ExtensionMgrCliModel import PackageInfo
from CliPlugin.ExtensionMgrCliModel import Repository
from CliPlugin.ExtensionMgrCliModel import RepositoryEntry
import ConfigMount
import ExtensionMgrLib
import FileCliUtil
import LazyMount
import ShowCommand
import SwiSignLib
import Tac
import Url

from ExtensionMgr import errors
from UrlPlugin import ExtensionUrl

extensionRepoConfig = None
extensionConfig = None
extensionStatus = None
mgmtSecurityConfig = None
mgmtSslConfig = None

ERR_CFG_NO_REPOS = 'No package repositories configured'

PkgStatusType = Tac.Type( "Extension::PkgStatus" )

def getFileNames():
   return [ v.filename for v in extensionStatus.info.values() ]

# Need to specify a new pattern becasue the original pattern does not have "."
# in it. And sample extension names are Test1.i386.rpm with "." in them.
# Also exlude keyword 'migrate' used in command 'extension migrate dirve'
extensionNamePattern = BasicCliUtil.notAPrefixOf( 'migrate' ) + r'[A-Za-z0-9_.:+-]+'
extNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: getFileNames(),
                                                'Extension name',
                                                pattern=extensionNamePattern )

def extensionFromInfo( info, model, certs=None ):
   extension = Extension()
   r = info.package.get( info.primaryPkg )
   if r is None:
      # This shouldn't happen unless there is a bug in the code that
      # populates Sysdb with extension info.
      extension.error = True
      model.extensions[ info.filename ] = extension
      return
   extension.numPackages = len( info.package )
   extension.error = False
   extension.version = r.version
   extension.release = r.release
   extension.presence = info.presence
   if info.presenceDetail:
      extension.presenceDetail = info.presenceDetail
   extension.status = info.status
   extension.boot = info.boot != ExtensionMgrLib.BootStatus.notBoot
   if info.statusDetail:
      extension.statusDetail = info.statusDetail
   extension.vendor = r.vendor
   extension.summary = r.summary
   extension.installedSize = 0
   for k in info.installedPackage:
      rr = info.package[ k ]
      pkg = PackageInfo( version=rr.version, release=rr.release )
      extension.installedSize += rr.installedSize
      extension.packages[ rr.filename ] = pkg
   if info.affectedAgents:
      for k in info.affectedAgents:
         extension.affectedAgents.append( info.affectedAgents[ k ] )
   if info.agentsToRestart:
      for k in info.agentsToRestart:
         extension.agentsToRestart.append( info.agentsToRestart[ k ] )
   extension.description = r.description
   # Optionally add signature information
   if certs and info.format == 'formatSwix':
      sigValid, sigErr, _ = SwiSignLib.verifySwixSignature( info.filepath, certs )
      extension.signed = sigValid
      if not sigValid:
         extension.signedDetail = sigErr
   model.extensions[ info.filename ] = extension

#-------------------------------------------------------------------------------
# "show extensions [ detail ]" in enable mode
#-------------------------------------------------------------------------------
def getModelInstance( mode ):
   model = Extensions()
   for info in extensionStatus.info.values():
      extensionFromInfo( info, model )
   return model

class ShowExtensionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show extensions [ detail ] """
   data = {
      'extensions': 'EOS extensions present on this device',
      'detail': 'Additional details about each extension',
   }
   cliModel = Extensions


   @staticmethod
   def handler( mode, args ):
      certs = None
      if ExtensionMgrLib.signatureVerificationEnabled( mgmtSecurityConfig ):
         try:
            certs = ExtensionMgrLib.signatureVerificationCerts( mgmtSecurityConfig,
                                                                mgmtSslConfig )
         except errors.SignatureVerificationError as e:
            mode.addWarning( str( e ) )

      model = Extensions()
      for info in extensionStatus.info.values():
         extensionFromInfo( info, model, certs )
      model._renderDetail = 'detail' in args # pylint: disable=protected-access

      if not extensionStatus.info:
         mode.addWarning( "No extensions are available" )

      if checkExtensionDirMounted():
         model.extensionStoredDir = 'drive:'
      return model

BasicCli.addShowCommandClass( ShowExtensionCmd )

#-------------------------------------------------------------------------------
# "show installed-extensions" in enable mode
#-------------------------------------------------------------------------------
class ShowInstalledExtensions( ShowCommand.ShowCliCommandClass ):
   syntax = "show installed-extensions"
   data = { 'installed-extensions' : 'Display installed EOS extensions' }
   cliModel = InstallationStatuses

   @staticmethod
   def handler( mode, args ):
      installed = InstallationStatuses()
      for i in extensionConfig.installation.values():
         extension = InstallationStatus()
         extension.forced = i.force
         installed.extensions[ i.filename ] = extension
      return installed

BasicCli.addShowCommandClass( ShowInstalledExtensions )

#-------------------------------------------------------------------------------
# "[no] extension <extension-name> [ { force |
#                                      ( signature-verification ignored ) } ]" 
# in enable mode
#-------------------------------------------------------------------------------
def _run( mode, cmd, **kwargs ):
   # In breadth tests I want to use arsudo instead of regular sudo to preserve
   # LD_LIBRARY_PATH.
   if kwargs.get( "asRoot" ):
      arsudo = "/usr/bin/arsudo"
      if os.path.exists( arsudo ):
         kwargs[ "asRoot" ] = False
         cmd.insert( 0, arsudo )
   try:
      Tac.run( cmd, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, **kwargs )
      return True
   except Tac.SystemCommandError as e:
      mode.addError( e.output )
      return False

def resetInstallationState( name ):
   info = ExtensionMgrLib.latestExtensionForName( name, extensionStatus )
   if info is not None:
      os.environ[ "EXTENSIONMGR_EXTENSION" ] = name
      if info.affectedAgents: # not unconditional since could be rpm driven
         info.agentsToRestart.clear()
         for i in info.affectedAgents:
            info.agentsToRestart[ i ] = info.affectedAgents[ i ]

def alreadyInstalled( mode, name ):
   info = ExtensionMgrLib.latestExtensionForName( name, extensionStatus )
   if info and info.status in ( 'installed', 'forceInstalled' ):
      return True
   return False

def alreadyUninstalled( mode, name ):
   info = ExtensionMgrLib.installedExtensionForName( name, extensionStatus )
   if info is None:
      return ( True, None )
   return ( False, info )

__extensionLock__ = threading.Lock()

@Ark.synchronized( __extensionLock__ )
def installExtension( mode, name, force=False, signatureIgnored=False ):
   info = ExtensionMgrLib.latestExtensionForName( name, extensionStatus )
   if info is None:
      mode.addError( "Extension not found: %s" % name )
      return False
   checkSig = ExtensionMgrLib.signatureVerificationEnabled( mgmtSecurityConfig )
   if checkSig and not signatureIgnored:
      swixfile = info.filepath
      if ExtensionMgrLib.getPackageFormat( swixfile ) == 'formatSwix':
         try:
            if ( not ExtensionMgrLib.verifySignature( swixfile, mgmtSecurityConfig,
                                                      mgmtSslConfig ) ):
               err = "Failed to install extension '%s': Invalid or missing signature"
               mode.addError( err % name )
               return False
         except errors.SignatureVerificationError as e:
            mode.addError( str( e ) )
            return False
   okay, message = ExtensionMgrLib.checkInstallPrerequisites( info )
   if not okay:
      mode.addError( f"Failed to install extension '{name}':\n{message}\n" )
      return False
   sysname = mode.session_.sysname
   cmd = [ sys.executable, "/usr/bin/InstallExtension", "--sysname", sysname, name ]
   if force:
      cmd.append( "--force" )
   if signatureIgnored:
      # Mark the fact that we should ignore this Swix's signature
      # It will be used when determining whether to save the extension 
      # to boot-extensions
      cmd.append( "--signatureIgnored" )
   ret = _run( mode, cmd, asRoot=True )
   return ret

@Ark.synchronized( __extensionLock__ )
def uninstallExtension( mode, name, info, force=False ):
   okay, message = ExtensionMgrLib.checkUninstallPrerequisites( info )
   if not okay:
      mode.addError(
         f"Failed to uninstall extension '{name}':\n{message}\n" )
      return False

   version, release = ExtensionMgrLib.getInstalledVersion( info )
   pkg = info.package[ info.primaryPkg ]
   if version != pkg.version or release != pkg.release:
      mode.addWarning( 'Expected to uninstall package version %s%s but uninstalling '
            'version %s%s instead' % ( pkg.version, pkg.release, version, release ) )

   sysname = mode.session_.sysname
   cmd = [ sys.executable, "/usr/bin/UninstallExtension", "--sysname", sysname,
      name ]
   if force:
      cmd.append( "--force" )
   ret = _run( mode, cmd, asRoot=True )
   return ret

def getPids( agentsIn ):
   pids = []
   agents = []
   notRunning = []
   
   pidDict = { a[ "name" ] : a[ "pid" ] for a in AgentDirectory.agentList() } 

   
   for a in agentsIn:
      try:
         if a in pidDict:
            agents.append( a )
            pids.append( pidDict[ a ] )
            continue
         pidofs = Tac.run( [ 'pidof', a ], stdout=Tac.CAPTURE ).strip( "\n" )
         pidofs = [ int( e ) for e in pidofs.split() ]
         pids.extend( pidofs )
         agents.append( a )
      except Tac.SystemCommandError:
         notRunning.append( a )
   return ( agents, pids, notRunning )
      
def recordPidsOfAffectedAgents( ext ):
   for info in extensionStatus.info.values():
      if info.filename == ext:
         info.agentPids.clear()
         agents = list( info.agentsToRestart.values() )
         _, pids, _ = getPids( agents )
         for pid in pids:
            info.agentPids[ pid ] = True
         break
      
def getRecordedPidsOfAffectedAgents( ext ):
   for info in extensionStatus.info.values():
      if info.filename == ext:
         return info.agentPids.members()
   return []

def finalizeOrWarn( mode, ext, finalizeNow ):
   # If same time (un)install & finalization is required, do finalize, else warn 
   # operator about the need for further actions to finalize the (un)install.
   # But if there was an error during %post, don't warn or restart anything.
   WaitForWarmupCli.doWaitForWarmup( mode, [ 'Sysdb' ], timeout=10 )
   extInfo = getModelInstance( mode ).extensions.get( ext )
   if extInfo:
      if finalizeNow:
         finalize( mode, ext )
         return
      if extInfo.agentsToRestart:
         print( "Agents that need restarting for changes to become effective:" )
         print( "  ", ", ".join( extInfo.agentsToRestart ) )
         print( "To restart those agents, when opportune (possible service " +
                "interruption) run this cli command:" )
         print( "  extension restart-agents" )
         print( "or, to restart only the agents needed for a particular extension:" )
         print( "  [no] extension <name>" )
      return

def nothingToFinalize( mode, ext ):
   extInfo = getModelInstance( mode ).extensions.get( ext )
   return not extInfo.agentsToRestart if extInfo else True

class InstallExtensionCmd( CliCommand.CliCommandClass ):
   syntax = "extension EXTENSION [ { force | " + \
            "( signature-verification ignored ) | delay-restart-agents } ]"
   noOrDefaultSyntax = "extension EXTENSION [ { force | delay-restart-agents } ] ..."
   data = {
      'extension' : 'Install an EOS extension',
      'EXTENSION' : extNameMatcher,
      'force' : CliCommand.singleKeyword( 'force',
                     helpdesc= 'Override dependency checks' ),
      'signature-verification' : CliCommand.singleKeyword( 'signature-verification',
                     helpdesc='Configure whether to verify SWIX signatures' ),
      'ignored' : 'Ignore SWIX signatures',
      'delay-restart-agents' : CliCommand.singleKeyword( 'delay-restart-agents',
                     helpdesc='If an agent needs to be restarted for the ' + \
                       'extension to become effective (software patch), ' + \
                       'delay it until "extension restart-agents" cli is run.' ),
      }
   @staticmethod
   def handler( mode, args ):
      name = args[ 'EXTENSION' ]
      finalizeNow = not args.get( 'delay-restart-agents' )
      if not alreadyInstalled( mode, name ):
         resetInstallationState( name )
         ok = installExtension( mode, name, force=( 'force' in args ),
                                 signatureIgnored=( 'ignored' in args ) )
         if not ok:
            return
         # In case of delayed restarts, don't restart agents that already did
         recordPidsOfAffectedAgents( name )
      finalizeOrWarn( mode, name, finalizeNow )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'EXTENSION' ]
      finalizeNow = not args.get( 'delay-restart-agents' )
      uninstalled, info = alreadyUninstalled( mode, name )
      if not uninstalled:
         resetInstallationState( name )
         ok = uninstallExtension( mode, name, info, force=( 'force' in args ) )
         if not ok:
            return
         # In case of delayed restart, don't restart agents that already did
         recordPidsOfAffectedAgents( name )
      elif not finalizeNow or nothingToFinalize( mode, name ):
         mode.addError( "Extension not installed: %s" % name )
         return
      # finalization can happen when extension was already un-installed...
      finalizeOrWarn( mode, name, finalizeNow )

BasicCli.EnableMode.addCommandClass( InstallExtensionCmd )

class FinalizeExtensionsCmd( CliCommand.CliCommandClass ):
   syntax = "extension restart-agents"
   data = {
      'extension' : 'Install an EOS extension',
      'restart-agents' : 'Restart any agent that still needs a restart for any' + \
                   'recently installed extensions to become activated.',
   }

   @staticmethod
   def handler( mode, args ):
      finalize( mode, "" )

BasicCli.EnableMode.addCommandClass( FinalizeExtensionsCmd )

#--------------------------------------------------------------------------------
# boot extension EXTENSION
#--------------------------------------------------------------------------------
def compatibleWithBootExts( bootExtName, r, info ):
   """
   Check extension status info.boot to make sure that the newly configured
   bootExtName does not have the same primary package and version/release
   as existing boot-extension.
   Return a warning message if incompatible; return '' otherwise.
   """
   for extInfo in extensionStatus.info.values():
      if extInfo.filename == bootExtName:
         continue
      if extInfo.boot == ExtensionMgrLib.BootStatus.notBoot:
         continue
      if extInfo.primaryPkg == info.primaryPkg:
         extR = info.package.get( extInfo.primaryPkg )
         if extR.version == r.version and extR.release == r.release:
            return ( 'Existing boot extension ' + extInfo.filename +
                     ' has the same primary rpm as ' + bootExtName )
   return ''

def writeBootExtensions( mode ):
   # Lines in boot-extensions file have four fields seperated by space
   # "extension-name force-and-boot extension-format [dependencies]"
   # force-and-boot field is used to indicate both force and boot information.
   # Possible values of force-and-boot field is as follows:
   #    If an extension is added to boot-extensions file by:
   #    - "copy installed-extensions boot-extensions" : "force" or "no"
   #    - "boot extension EXTENSION" : "boot"
   #    - both the above commands: "force,boot" or "no,boot"
   try:
      bootExtUrl = ExtensionUrl.bootExtensions( mode )
      sysdbRoot = None
      if bootExtUrl.context and bootExtUrl.context.entityManager:
         sysdbRoot = bootExtUrl.context.entityManager.root()
      with bootExtUrl.open( mode='wb' ) as f:
         ExtensionMgrLib.saveBootExtensions( sysdbRoot, f )
   except OSError:
      mode.addError( 'Failed to write to boot-extensions file' )

def handleBootExtension( mode, args ):
   bootExtName = args[ 'EXTENSION' ]
   info = ExtensionMgrLib.latestExtensionForName( bootExtName, extensionStatus )
   if info is None:
      mode.addError( 'Extension not found: %s' % bootExtName )
      return
   r = info.package.get( info.primaryPkg )
   warningMessage = compatibleWithBootExts( bootExtName, r, info )

   if warningMessage:
      mode.addWarning( warningMessage )
      promptText = 'Do you wish to proceed with this command? [y/N]'
      if not BasicCliUtil.confirm( mode, promptText, answerForReturn=False ):
         mode.addError( 'Command aborted by user' )
         return

   info.boot = ExtensionMgrLib.BootStatus.bootByConfig
   writeBootExtensions( mode )

def handleNoBootExtension( mode, args ):
   """
   Remove the specified EXTENSION from boot-extensions file whether
   it was added by "boot extension EXTENSION" or by
   "copy installed-extensions boot-extensions" command
   If no EXTENSION is specified, remove all extensions in boot-extensions file
   """
   bootExtName = args.get( 'EXTENSION' )
   if bootExtName:
      info = ExtensionMgrLib.latestExtensionForName( bootExtName, extensionStatus )
      if info is None:
         mode.addError( 'Extension not found: %s' % bootExtName )
      else:
         info.boot = ExtensionMgrLib.BootStatus.notBoot
   else:
      for info in extensionStatus.info.values():
         info.boot = ExtensionMgrLib.BootStatus.notBoot
   writeBootExtensions( mode )

class BootExtensionCmd( CliCommand.CliCommandClass ):
   syntax = 'boot extension EXTENSION'
   noOrDefaultSyntax = 'boot extension [ EXTENSION ]'
   data = {
      'boot': 'System boot configuration',
      'extension' : 'Configure an EOS boot-extension',
      'EXTENSION' : extNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      handleBootExtension( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      handleNoBootExtension( mode, args )

BasicCliModes.GlobalConfigMode.addCommandClass( BootExtensionCmd )

# Remove from 'agentsToRestart' any agent that is not already running. This is called
# during 'finalization' only: if at that time the agent is not running, we are done
# for ever trying to restart it. If it does start later, it will use the new code.
def rmAgentsToRestart( rpm, agents ):
   for info in extensionStatus.info.values():
      if rpm != "" and info.filename != rpm: # pylint: disable=consider-using-in
         continue
      for i in info.agentsToRestart:
         a = info.agentsToRestart[ i ] 
         if a in agents:
            del info.agentsToRestart[ i ]

def finalize( mode, rpm ):
   model = getModelInstance( mode )
   agents = set()
   pidsAtInstall = set()
   for filename, extension in model.extensions.items():
      if rpm in ( "", filename ): # TODO: pre-filter.
         if extension.agentsToRestart:
            agents.update( extension.agentsToRestart )
            pidsAtInstall.update( getRecordedPidsOfAffectedAgents( filename ) )

   if not agents:
      return

   agents = sorted( agents )
   print( "Agents to be restarted:", ", ".join( agents ) )
   if "ConfigAgent" in agents:
      print( "Cli sessions on ssh or console will be logged out." )

   agents, pids, notRunning = getPids( agents ) # Sorted: `agents`, `notRunning`.
   if notRunning:
      print( "Those agents are not running:", ", ".join( notRunning ) )
      rmAgentsToRestart( rpm, notRunning )

   alreadyRestarted = set()
   for i, pid in enumerate( pids ):
      # TODO: `pidof` can return multiple PIDs per proc name,
      # so we may hit `IndexError` here if `len( pids ) > len( agents )`?
      # Not sure if that is possible for agents,
      # but perhaps `getPids` should return a map: { agent: pids/agent }.
      if pid not in pidsAtInstall:
         # since installation of code, agent already restarted: we are done
         rmAgentsToRestart( rpm, [ agents[ i ] ] )
         alreadyRestarted.add( agents[ i ] )

   if alreadyRestarted:
      # Some agents have already restarted; mention that, then filter them out.
      agents = [ a for a in agents if a not in alreadyRestarted ]
      alreadyRestarted = ", ".join( sorted( alreadyRestarted ) )
      print( "Those agents restarted already:", alreadyRestarted )

   if not agents:
      print( "No agents to restart" )
      return

   rmAgentsToRestart( "", agents )

   # Run the killing daemonized: this is so the cli prompt returns even if
   # ConfigAgent is restarted. Not that important for interactive Cli, but for
   # interfaces like Capi there is no other way to find out your command reached
   # the box. From this point, chances for errors are pretty low, and if one still
   # happens, the corresponding Agent will not be removed from the pending restarts
   # and 'show extension details' may have a corresponding error detail.
   try:
      Tac.run( [ 'daemonize', '/usr/bin/arsudo', sys.executable,
                 '/usr/bin/ExtensionMgrUtils', 'restart', rpm ] + agents )
   except Tac.SystemCommandError as e:
      mode.addError( "Failed to restart agents: %s" % e )

#-------------------------------------------------------------------------------
# extension migrate drive: 
# EXEC command in enable mode
#-------------------------------------------------------------------------------
def checkExtensionDirMounted():
   # check if /mnt/flash/.extensions has been mounted
   # TODO: Rename this? It's used to set `model.extensionStoredDir`
   # from the default 'flash:' to 'drive:'.
   extensionFilesystem = Url.getFilesystem( 'extension:' )
   driveFilesystem = Url.getFilesystem( 'drive:' )
   extensionDev, extensionMntpnt, _ = extensionFilesystem.mountInfo()
   
   return ( extensionMntpnt == extensionFilesystem.location_ and
            driveFilesystem and extensionDev == driveFilesystem.mountInfo()[ 0 ] )

def migrateExtensions( mode ):
   if checkExtensionDirMounted(): 
      return

   # print a warning message about concurrent extension acces 
   # and ask for confirmation in interactive mode.
   mode.addWarning( "Concurrent extension access could result in error "
                    "during running the migration command. Please note: "
                    "no other extension command like adding/removing action "
                    "can be executed at the same time." )
   prompt = "Do you wish to proceed with this command? [y/N]"
   if not BasicCliUtil.confirm( mode, prompt ):
      return

   try:
      Tac.run( [ "ExtensionMigration" ], asRoot=True,
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      mode.addError( e.output )

def guardMigrateWithSSD( mode, token ):
   if Url.getFilesystem( 'drive:' ):
      return None
   return CliParser.guardNotThisPlatform

class MigrateExtensionCmd( CliCommand.CliCommandClass ):
   syntax = 'extension migrate drive:'
   data = {
      'extension' : 'Install an EOS extension', 
      'migrate' : CliCommand.guardedKeyword( 'migrate', 
         helpdesc='Migrate EOS extensions to a specified location',
         guard=guardMigrateWithSSD ),
      'drive:' : 'The destination filesystem to migrate EOS extensions'
   }

   @staticmethod
   def handler( mode, args ):
      migrateExtensions( mode )

BasicCli.EnableMode.addCommandClass( MigrateExtensionCmd )

#-------------------------------------------------------------------------------
# Add support for copying to/from boot-extensions and from installed-extensions
#-------------------------------------------------------------------------------
FileCliUtil.registerCopySource( 'boot-extensions',
                                CliMatcher.KeywordMatcher(
                                   'boot-extensions',
                                   helpdesc='Copy boot extensions configuration',
                                   value=lambda mode, match: \
                                   ExtensionUrl.bootExtensions( mode ) ),
                                'boot-extensions' )
FileCliUtil.registerCopySource( 'installed-extensions',
                                CliMatcher.KeywordMatcher(
                                   'installed-extensions',
                                   helpdesc='Copy installed extensions status',
                                   value=lambda mode, match: \
                                   ExtensionUrl.installedExtensions( mode ) ),
                                'installed-extensions' )

def validateCopyToBootExtension( mode, srcUrl, dstUrl, commandSource ):
   # ignore it if dstUrl is not boot-extensions
   if dstUrl.url != "flash:/boot-extensions":
      return True

   if srcUrl.url.endswith( ( ".rpm", ".swix", ".tar.gz" ) ):
      mode.addError( "In order to configure an extension to be installed"
            " at boot time, first install the extension, then run"
            " 'copy installed-extensions boot-extensions'" )
      return False
   localPath = srcUrl.localFilename()
   if localPath:
      try:
         stat = os.stat( localPath )
         if stat.st_size > 100 * 1024:  # Arbitrary limit
            # If the file is large, it's unlikely to be a list of extensions.
            # Proceed anyway, but emit a warning to tell the user that he's
            # most likely doing something he didn't intend to.
            mode.addWarning( "%s is fairly large (%dKB), are you sure it's a"
                  " list of extensions?" % ( srcUrl.url, stat.st_size // 1024 ) )
      except OSError:
         pass  # the copy will most likely fail elsewhere.
   return True

FileCliUtil.copyFileCheckers.addExtension( validateCopyToBootExtension )

FileCliUtil.registerCopyDestination(
   'boot-extensions',
   CliMatcher.KeywordMatcher(
      'boot-extensions',
      helpdesc='Copy to boot extensions configuration',
      value=lambda mode, match: ExtensionUrl.bootExtensions( mode ) ),
   'boot-extensions' )

#-------------------------------------------------------------------------------
# "show boot-extensions" in enable mode
#-------------------------------------------------------------------------------
class ShowBootExtensions( ShowCommand.ShowCliCommandClass ):
   syntax = "show boot-extensions"
   data = { 'boot-extensions' : 'Display contents of boot extensions configuration' }
   cliModel = BootExtensions

   @staticmethod
   def handler( mode, args ):
      bootExtensions = BootExtensions()
      extensionUrl = ExtensionUrl.bootExtensions( mode )
      try:
         with extensionUrl.open() as f: # pylint: disable-msg=E1103
            extensions = []
            for line in f.readlines():
               extensions.append( line.strip().split( ' ' )[ 0 ] )
            bootExtensions.extensions = extensions
      except OSError as e:
         if e.errno != errno.ENOENT:
            raise
      return bootExtensions

BasicCli.addShowCommandClass( ShowBootExtensions )

#-------------------------------------------------------------
# Register "show extensions detail" into "show tech-support"
#-------------------------------------------------------------

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:24:00',
   cmds=[ 'show extensions detail' ],
   summaryCmds=[ 'show extensions' ] )

# Generic CLI tokens we use - do not accept pipe (|) characters as the CLI
# will ingest them as part of this pattern rule rather than piping the output
repoNamePattern = CliMatcher.PatternMatcher( '[^|]+', helpname='WORD',
                                         helpdesc='Repository name' )

# BUG94891 Enabled this once we support pip packages, add 'pypi': 'PyPi format' )
repoTypeMatcher = CliMatcher.EnumMatcher( { 'yum': 'Yum format' } )

def getRepo( name ):
   """
   Gets the repository Tac object from the config.
   """
   return extensionRepoConfig.repo.get( name )

def maybeUpdateRepo( repo ):
   """
   Calles ExtensionMgrLib.updateRepository if the minimum
   information to create a repository is there.
   """
   if repo.format in ( 'formatUnknown', '' ):
      return
   elif not repo.url:
      return

   ExtensionMgrLib.updateRepository( repo )

#------------------------------------------
# The "package" mode command
#------------------------------------------
class PackageConfigMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = 'Package extensions config mode'

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, 'package' )

class EnterPackageConfigMode( CliCommand.CliCommandClass ):
   syntax = """management package"""
   noOrDefaultSyntax = syntax
   data = {
            "management": ConfigMgmtMode.managementKwMatcher,
            "package": 'Package configuration'
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( PackageConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      extensionRepoConfig.enable = True
      for repo in extensionRepoConfig.repo:
         ExtensionMgrLib.deleteRepository( getRepo( repo ) )
         del extensionRepoConfig.repo[ repo ]

BasicCli.GlobalConfigMode.addCommandClass( EnterPackageConfigMode )

#-------------------------------------------------
# [ no | default ] shutdown
#-------------------------------------------------
class EnableRepoModeCmd( CliCommand.CliCommandClass ):
   syntax = """shutdown"""
   noOrDefaultSyntax = syntax
   data = {
            "shutdown" : 'Enable/disable package management'
          }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      extensionRepoConfig.enable = False

   @staticmethod
   def noHandler( mode, args ):
      extensionRepoConfig.enable = True

   defaultHandler = handler

PackageConfigMode.addCommandClass( EnableRepoModeCmd )

#-------------------------------------------------
# [ no | default ] repository REPO_NAME 
#-------------------------------------------------
class RepositoryConfigMode( CliMode.Package.RepositoryConfigMode,
                            BasicCli.ConfigModeBase ):
   name = 'Repository config mode'

   def __init__( self, parent, session, repoName=None ):
      CliMode.Package.RepositoryConfigMode.__init__(self, repoName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.repoName = repoName

   def onExit( self ):
      pass

def repoTypeToEnum( t ):
   if t == "yum":
      return "formatYum"
   elif t == "pypi":
      return "formatPyPi"
   elif t == "swix":
      return "formatSwix"
   return "formatUnknown"

class RepositoryHandler( CliCommand.CliCommandClass ):
   syntax = """repository REPO_NAME""" 
   noOrDefaultSyntax = syntax

   data = {
            "repository": "Configure a package repository",
            "REPO_NAME": repoNamePattern,
          }

   @staticmethod
   def handler( mode, args ):
      # create the repo in sysdb
      name = args[ 'REPO_NAME' ]
      extensionRepoConfig.repo.newMember( name )
      # go to the submode
      childMode = mode.childMode( RepositoryConfigMode, repoName=name )
      mode.session_.gotoChildMode( childMode )
   
   @staticmethod
   def noOrDefaultHandler(  mode, args ):
      repo = args[ 'REPO_NAME' ]
      ExtensionMgrLib.deleteRepository( getRepo( repo ) )
      del extensionRepoConfig.repo[ repo ]
      
PackageConfigMode.addCommandClass( RepositoryHandler )

#-------------------------
# type TYPE
#-------------------------
class RepoTypeCmd( CliCommand.CliCommandClass ):
   syntax = """type TYPE"""
   data = {
            "type": "Repository type",
            "TYPE": repoTypeMatcher
          }

   @staticmethod
   def handler( mode, args ):
      repo = getRepo( mode.repoName )
      newfmt = repoTypeToEnum( args[ 'TYPE' ] )
      # pylint: disable-next=consider-using-in
      if newfmt != "formatUnknown" and newfmt != repo.format:
         # if the types aren't the same we need to delete the old one then recreate
         ExtensionMgrLib.deleteRepository( repo )
      repo.format = newfmt
      maybeUpdateRepo( repo )

RepositoryConfigMode.addCommandClass( RepoTypeCmd )

#-------------------------
# url URL
#-------------------------
class RepoUrlCmd( CliCommand.CliCommandClass ):
   syntax = """url URL"""
   data = {
            'url': 'Repository URL',
            'URL': CliMatcher.PatternMatcher( 'http(s)?://.+',
                                              helpname='Repository URL',
                                              helpdesc='Repository URL' )
          }

   @staticmethod
   def handler( mode, args ):
      repo = getRepo( mode.repoName )
      newurl = args[ 'URL' ]
      if newurl != repo.url:
         repo.url = newurl
      maybeUpdateRepo( repo )

RepositoryConfigMode.addCommandClass( RepoUrlCmd )

#----------------------------
# description DESCRIPTION
#----------------------------
class RepoDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = """description DESCRIPTION"""
   data = {
      "description": "Repository description",
      "DESCRIPTION": CliMatcher.StringMatcher( helpname='LINE',
                                          helpdesc='Description of this repository' )
   }

   @staticmethod
   def handler( mode, args ):
      repo = getRepo( mode.repoName )
      repo.description = args[ 'DESCRIPTION' ]

RepositoryConfigMode.addCommandClass( RepoDescriptionCmd )

#-----------------------------------------------
# package download <pypi|yum> REPO_NAME
#-----------------------------------------------
class PackageDownloadHandler( CliCommand.CliCommandClass ):
   syntax = """package download TYPE REPO_NAME"""
   data = {
      "package": "Display and search for software packages",
      "download": "Download a package from a repository",
      "TYPE": repoTypeMatcher,
      "REPO_NAME": CliMatcher.PatternMatcher( '.+', helpname='PACKAGE',
                                             helpdesc='Package name' )
   }

   @staticmethod
   def handler( mode, args ):
      if not extensionRepoConfig.enable:
         mode.addError( "The package manager is disabled, please enable it first." )
         return

      repotype = repoTypeToEnum( args[ 'TYPE' ] )
      name = args[ 'REPO_NAME' ]
      repos = [ r for r in extensionRepoConfig.repo.values()
                if r.format == repotype ]
      if not repos:
         mode.addError( ERR_CFG_NO_REPOS )
         return

      try:
         ExtensionMgrLib.packageDownload( name, repos, repotype, extensionStatus )
      except errors.NoRepoError:
         mode.addError( ERR_CFG_NO_REPOS )
      except errors.PackageNotAvailableError as pkgname:
         mode.addError( "No package named '%s' was available to download" % pkgname )
      except errors.PackageNameDownloadMismatchError as pkgname:
         mode.addError( "Package named '%s' was not downloaded" % pkgname )
      except errors.Error as e:
         mode.addError( f"Download of package '{name}' failed: {e}" )

BasicCli.EnableMode.addCommandClass( PackageDownloadHandler )

def _appendEntry( model, repo ):
   entry = RepositoryEntry()
   entry.name = repo.name
   entry.url = repo.url
   entry.rType = ExtensionMgrLib.repoTypeStrMap[ repo.format ]
   entry.description = repo.description
   model.repositories[ entry.name ] = entry

#---------------------------------------------
# show management package repository [ REPO_NAME ]
#---------------------------------------------
class ShowPackageRepositoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show management package repository [ REPO_NAME ]"""
   data = {
      "management": ConfigMgmtMode.managementShowKwMatcher,
      "package": "Display and search for software packages",
      "repository": "Package repository information",
      "REPO_NAME": repoNamePattern
   }
   cliModel = Repository


   @staticmethod
   def handler( mode, args ):
      model = Repository()
      name = args.get( 'REPO_NAME' )
      if name:
         repo = getRepo( name )
         model._renderAll = False # pylint: disable=protected-access
         if repo:
            _appendEntry( model, repo )
         else:
            mode.addError( "Unknown repository %s" % name )
            return None
      else:
         for repo in extensionRepoConfig.repo.values():
            _appendEntry( model, repo )

      return model

BasicCli.addShowCommandClass( ShowPackageRepositoryCmd )

#----------------------------------------------------------------------
# show management package [ PKG_NAME ] [ detail ]
#----------------------------------------------------------------------
class ShowPackageCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show management package [ PKG_NAME ] [ detail ]"""
   data = {
      "management": ConfigMgmtMode.managementShowKwMatcher,
      "package": "Display and search for software packages",
      "PKG_NAME": CliMatcher.PatternMatcher(
                        BasicCliUtil.notAPrefixOf( 'repository' ) + '[^|]+',
                        helpname='PACKAGE',
                        helpdesc='Package name' ),
      "detail": "Show extended package information"
   }
   cliModel = Extensions

   @staticmethod
   def handler( mode, args ):
      pkgName = args.get( 'PKG_NAME', '' )

      packages = {}
      if pkgName:
         query = pkgName.lower()
         repos = list( extensionRepoConfig.repo.values() )
         infos = ExtensionMgrLib.packageSearch( query, repos ) or []
         # remove duplicates based on generation
         for info in infos:
            if info.filename in packages:
               if info.generation > packages[ info.filename ].generation:
                  packages[ info.filename ] = info
            else:
               packages[ info.filename ] = info
      else:
         # if they haven't supplied a pkgName they just want 
         # the installed packages
         for infoKey in extensionStatus.info:
            info = extensionStatus.info[ infoKey ]
            packages[ info.filename ] = info

      model = Extensions()
      for info in packages.values():
         extensionFromInfo( info, model )
      model._renderDetail = 'detail' in args # pylint: disable=protected-access

      if checkExtensionDirMounted():
         model.extensionStoredDir = 'drive:'
      return model

BasicCli.addShowCommandClass( ShowPackageCmd )

def Plugin( entityManager ):
   global extensionRepoConfig
   global extensionConfig
   global extensionStatus
   global mgmtSecurityConfig
   global mgmtSslConfig
   extensionRepoConfig = ConfigMount.mount( entityManager, 
                                            ExtensionMgrLib.repoConfigPath(),
                                            "Extension::Repo", "w" )
   extensionConfig = entityManager.mount( ExtensionMgrLib.configPath(),
                                          "Extension::Config", "wf" )
   extensionStatus = entityManager.mount( ExtensionMgrLib.statusPath(),
                                          "Extension::Status", "wf" )
   mgmtSecurityConfig = LazyMount.mount( entityManager,
                                         "mgmt/security/config",
                                         "Mgmt::Security::Config", "r" )
   mgmtSslConfig = LazyMount.mount( entityManager, 
                                    "mgmt/security/ssl/config",
                                    "Mgmt::Security::Ssl::Config", "r" )

