# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import ExtensionMgrLib
import Tac
import TableOutput
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str

NA_STRING = 'N/A'

_statusMap = {
   'notInstalled': ( '', 'not installed' ),
   'installed': ( 'I', 'installed' ),
   'forceInstalled': ( 'F', 'force installed' ),
   }

def strForStatus( statusEnum, abbrev=True ):
   """Generates a string representing the status given an
   Extension::Info::Status enum."""
   elem = 0 if abbrev else 1
   return _statusMap[ statusEnum ][ elem ]

_presenceMap = {
   'present': ( 'A', 'available' ),
   'absent': ( 'NA', 'not available' ),
   }

def strForPresence( presenceEnum, abbrev=True ):
   """Generates a string representing the status given an
   Extension::Info::Presence enum."""
   elem = 0 if abbrev else 1
   return _presenceMap[ presenceEnum ][ elem ]

_bootMap = {
   True: ( 'B', 'install at boot' ),
   False: ( '', 'not install at boot' ),
   }

def strForBoot( bootVal, abbrev=True ):
   """Generates a string representing boot given a boolean value."""
   elem = 0 if abbrev else 1
   return _bootMap[ bootVal ][ elem ]

_signatureMap = {
   True: ( 'S', 'valid' ),
   False: ( 'NS', 'invalid' ),
   }

def strForSignature( signatureVal, abbrev=True ):
   """Generates a string representing the signature given
   a boolean value."""
   elem = 0 if abbrev else 1
   return _signatureMap[ signatureVal ][ elem ]

class PackageInfo( Model ):
   version = Str( help='Version of this particular extension' )
   release = Str( help='Release of this particular extension' )

class Extension( Model ):
   __revision__ = 2

   version = Str( help='Version of the extension', default=NA_STRING )
   release = Str( help='Release of the extension', default=NA_STRING )
   presence = Enum( default='present', values= ( 'present', 'absent' ),
         help='Whether or not the extension is still present on the switch.'
         '  An extension can be installed and then removed, which would cause'
         ' it to remain installed and become absent.' )
   status = Enum( default='notInstalled', 
         values=( 'installed', 'notInstalled', 'forceInstalled' ),
         help='Whether the extension is installed' )
   boot = Bool( help='Install at boot.' )
   numPackages = Int( optional=True, help='How many packages are installed' )
   error = Bool( default=False, 
         help='Indicates that the extension is in an error state, for'
         ' instance if the system believes the extension was installed but one'
         ' or more of its packages got manually uninstalled' )

   # optional field if SWIX signing is enabled
   signed = Bool( help='The extension has a valid signature.',
                  optional=True )

   # all fields below are optional and only printed when 'detail' is included
   presenceDetail = Str( help='Contains an error string explaining the'
         ' presence of the extension.  For instance the extension may be'
         ' corrupted or otherwise unusable, which would cause the system'
         ' to consider the extension is absent because it cannot be used.',
         optional=True )
   statusDetail = Str( help='Contains an error string explaining the'
         ' status of the extensions.  For instance the extension may have'
         ' failed to be (un)installed on the system, and this string would'
         ' contain additional information explaining why.',
         optional=True )
   signedDetail = Str( help='Contains an error string explaining why'
         ' the extension is signed incorrectly.', optional=True )
   vendor = Str( help='Vendor of this extension', optional=True )
   summary = Str( help='Summary to this extension', optional=True )
   installedSize = Int( help='Installed size (in bytes) of the extension', 
                        optional=True )
   packages = Dict( keyType=str, valueType=PackageInfo, 
         help='Info about the extension keyed by names of the extensions', 
         optional=True )
   description = Str( help='Description to the extension', optional=True )
   affectedAgents = List( help='Agents that are affected by this extension ' + \
                               'in the sence that they need to be restarted for ' + \
                               'the extension to take effect (static info)', 
                           valueType=str, optional=True )
   agentsToRestart = List( help='Agents that are pending a restart to finalize ' + \
                                'the extension; will be reset after restart', 
                           valueType=str, optional=True )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'numRpms' ] = dictRepr.pop( 'numPackages', None )
         dictRepr[ 'rpms' ] = dictRepr.pop( 'packages', None )

      return dictRepr

class Extensions( Model ):
   __revision__ = 2

   extensions = Dict( valueType=Extension, 
                      help='Maps filenames to extension details' )
   extensionStoredDir = Str( help='The filesystem that extensions are '
                                  'actually stored in', default='flash:' )

   _renderDetail = Bool(
         help="Determines whether additional information should be rendered",
         default=False )

   def render( self ):
      if self.extensions:
         if self._renderDetail:
            self.renderDetail()
         else:
            self.renderBasic()
      
      storeLoc = 'SSD' if self.extensionStoredDir == 'drive:' else 'internal flash'
      print( "The extensions are stored on {} ({})".format( storeLoc,
                                                       self.extensionStoredDir ) )

   def renderBasic( self ):
      headings = [ 'Name', 'Version/Release', 'Status', 'Extension' ]
      table = TableOutput.createTable( headings )
      nameFmt = TableOutput.Format( justify='left', maxWidth=42, wrap=True )
      nameFmt.noPadLeftIs( True )
      nameFmt.noTrailingSpaceIs( True )
      versionFmt = TableOutput.Format( justify='left', maxWidth=15, wrap=True )
      versionFmt.noPadLeftIs( True )
      versionFmt.noTrailingSpaceIs( True )
      statusFmt = TableOutput.Format( justify='left', maxWidth=14, wrap=True )
      statusFmt.noPadLeftIs( True )
      statusFmt.noTrailingSpaceIs( True )
      extensionFmt = TableOutput.Format( justify='left', maxWidth=9, wrap=True )
      extensionFmt.noPadLeftIs( True )
      extensionFmt.noTrailingSpaceIs( True )
      table.formatColumns( nameFmt, versionFmt, statusFmt, extensionFmt )

      for filename, extension in sorted( self.extensions.items() ):
         if extension.error:
            statusStr = "Error: extension is not available"
         else:
            statusStr = ", ".join( filter( None,
               [ strForPresence( extension.presence ),
                 strForStatus( extension.status ),
                 strForBoot( extension.boot ) ] ) )
            if extension.signed != None: # pylint: disable=singleton-comparison
               statusStr += ", %s" % strForSignature( extension.signed )
         version = f"{extension.version}/{extension.release}"
         table.newRow( filename, version, statusStr, extension.numPackages )

      print( table.output() )
      print( "" )
      print( "A: available | NA: not available | I: installed | "
             "F: forced | B: install at boot\n"
             "S: valid signature | NS: invalid signature" )

      # Second table about agents needing a restart for extension to be actived.
      # (if no restart is needed, will print nothing at all)
      headings = [ 'Name', 'Agents needing a restart to activate' ]
      table = TableOutput.createTable( headings )
      nameFmt = TableOutput.Format( justify='left', maxWidth=39, wrap=True )
      nameFmt.noPadLeftIs( True )
      nameFmt.noTrailingSpaceIs( True )
      agentsFmt = TableOutput.Format( justify='left', maxWidth=39, wrap=True )
      agentsFmt.noPadLeftIs( True )
      agentsFmt.noTrailingSpaceIs( True )
      table.formatColumns( nameFmt, agentsFmt )

      havePendings = False
      for filename, extension in sorted( self.extensions.items() ):
         if extension.agentsToRestart:
            agents = ", ".join( extension.agentsToRestart )
            table.newRow( filename, agents )
            havePendings = True

      if havePendings:
         print( "" )
         print( table.output() )

   def renderDetail( self ):
      for filename, extension in sorted( self.extensions.items() ):
         if extension.error:
            print( "Error: extension %s was not properly loaded" % filename )
            continue
         print( "       Name: %s" % filename )
         print( "    Version: %s" % extension.version )
         print( "    Release: %s" % extension.release )
         detail = ''
         if extension.presenceDetail:
            detail = "(%s)" % extension.presenceDetail
         print( "   Presence: {} {}".format( strForPresence( extension.presence,
                                          abbrev=False ), detail ) )
         detail = ''
         if extension.statusDetail:
            detail = "(%s)" % extension.statusDetail
         print( "     Status: {} {}".format( strForStatus( extension.status,
            abbrev=False ), detail ) )
         detail = ''
         if extension.signedDetail:
            detail = "(%s)" % extension.signedDetail
         if extension.signed != None: # pylint: disable=singleton-comparison
            print( "  Signature: {} {}".format( strForSignature( extension.signed,
               abbrev=False ), detail ) )
         print( "       Boot: %s" % strForBoot( extension.boot, abbrev=False ) )
         print( "     Vendor: %s" % extension.vendor )
         print( "    Summary: %s" % extension.summary )
         print( "   Packages: %s" % "\n             ".join( "%s %s/%s" %
               ( name, pkg.version, pkg.release ) for ( name, pkg ) in
                     extension.packages.items() ) )
         print( " Total size: %d bytes" % extension.installedSize )
         if extension.affectedAgents:
            print( "Affected Agents : %s" %
                   ', '.join( extension.affectedAgents ) )
         if extension.agentsToRestart:
            print( "Agents in need of restart: %s" %
                   ', '.join( extension.agentsToRestart ) )
         
         if extension.description:
            print( "Description:\n%s\n" % extension.description )

class InstallationStatus( Model ):

   forced = Bool( help='Whether the installed extension is forced' )

class InstallationStatuses( Model ):

   extensions = Dict( keyType=str, valueType=InstallationStatus,
         help='Maps the extension names to their statuses' )

   def render( self ):
      if self.extensions:
         for filename, status in sorted( self.extensions.items() ):
            if status.forced:
               print( filename, "force" )
            else:
               print( filename )
      else:
         print( "No extensions are installed" )

class BootExtensions( Model ):

   extensions = List( valueType=str, help='List of extensions that will '
         'be installed and run at startup' )

   def render( self ):
      for extension in self.extensions:
         print( extension )

class RepositoryEntry( Model ):
   name = Str( help='Repository name' )
   url = Str( help='Repository URL' )
   rType = Enum( values=list( ExtensionMgrLib.repoTypeStrMap.values() ),
                 help='Repository type (yum or pypi)' )
   description = Str( help='Repository description' )

class Repository( Model ):
   repositories = Dict( keyType=str, valueType=RepositoryEntry,
         help='List of extension repositories' )

   _renderAll = Bool(
         help="Determines whether additional information should be rendered",
         default=True )

   def render( self ):
      if self._renderAll:
         return self.renderAll()
      else:
         return self.renderOne()

   def renderOne( self ):
      # repositories should only be populated with one entry
      entry = list( self.repositories.values() )[ 0 ]
      print( "Name: %s" % entry.name )
      print( "URL: %s" % entry.url )
      print( "Type: %s" % entry.rType )
      print( "Description: %s" % entry.description )

   def renderAll( self ):
      headings = [ 'Name', 'Type', 'URL' ]
      table = TableOutput.createTable( headings )
      nameFmt = TableOutput.Format( justify='left', maxWidth=20, wrap=True )
      nameFmt.noPadLeftIs( True )
      nameFmt.noTrailingSpaceIs( True )
      typeFmt = TableOutput.Format( justify='left', maxWidth=10, wrap=True )
      typeFmt.noPadLeftIs( True )
      typeFmt.noTrailingSpaceIs( True )
      urlFmt = TableOutput.Format( justify='left', maxWidth=75, wrap=True )
      urlFmt.noPadLeftIs( True )
      urlFmt.noTrailingSpaceIs( True )
      table.formatColumns( nameFmt, typeFmt, urlFmt )

      for r in self.repositories.values():
         table.newRow( r.name, r.rType, r.url )

      print( table.output() )
