#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Cell
import CliGlobal
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Switch
import ConfigMount
import LazyMount

gv = CliGlobal.CliGlobal( dict( fabricCliCapabilities=None,
                                insufficientBandwidthConfig=None ) )


def insufficientBandwidthActionGuard( mode, token ):
   if gv.fabricCliCapabilities.insufficientBandwidthCliSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def linecardShutdownGuard( mode, token ):
   # This follows the original guard in FabricSystemCapabilities.tin 
   # where the insufficient-bandwidth tree is disabled on FixedSystems
   if Cell.cellType() == 'fixed':
      return CliParser.guardNotThisPlatform
   return None

def logGuard( mode, token ):
   if gv.fabricCliCapabilities.insufficientBandwidthLogCliSupported:
      return None
   return CliParser.guardNotThisPlatform

matcherAction = CliMatcher.KeywordMatcher( 'action',
      helpdesc='Configure insufficient-bandwidth fabric action' )
matcherFabric = CliMatcher.KeywordMatcher( 'fabric',
      helpdesc='Configure fabric parameters' )
matcherInsufficientBandwidth = CliMatcher.KeywordMatcher( 'insufficient-bandwidth',
      helpdesc='Configure insufficient-bandwidth fabric parameter' )
matcherPercent = CliMatcher.KeywordMatcher( 'percent',
      helpdesc='Percentage-based threshold' )
nodeInsufficientBandwidth = CliCommand.guardedKeyword( 'insufficient-bandwidth',
      helpdesc='Configure insufficient-bandwidth fabric parameter',
      guard=insufficientBandwidthActionGuard )
nodeLinecardShutdownAction = CliCommand.guardedKeyword( 'linecard-shutdown',
      helpdesc='Power off linecard as a result of fabric bandwidth loss',
      guard=linecardShutdownGuard )
nodeLogAction = CliCommand.guardedKeyword( 'log',
      helpdesc='Log a warning as a result of fabric bandwidth loss',
      guard=logGuard )

#--------------------------------------------------------------------------------
# [ no | default ] switch fabric insufficient-bandwidth action
#                                          ( linecard-shutdown threshold THRESHOLD )
#                                          | ( log threshold THRESHOLD percent )
#--------------------------------------------------------------------------------
class InsufficientBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = ( 'switch fabric insufficient-bandwidth action '
             '( ( linecard-shutdown threshold THRESHOLD ) '
             '| ( log threshold THRESHOLD percent ) )' )
   noOrDefaultSyntax = 'switch fabric insufficient-bandwidth action ...'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'fabric' : matcherFabric,
      'insufficient-bandwidth' : nodeInsufficientBandwidth,
      'action' : matcherAction,
      'linecard-shutdown' : nodeLinecardShutdownAction,
      'log' : nodeLogAction,
      'threshold' : ( 'Set the insufficient-bandwidth threshold corresponding to '
                      'the action' ),
      'THRESHOLD' : CliMatcher.IntegerMatcher( 5, 95,
         helpdesc='% of bandwidth loss allowed' ),
      'percent' : matcherPercent,
   }

   @staticmethod
   def handler( mode, args ):
      gv.insufficientBandwidthConfig.linkFailureActionIgnore = False
      gv.insufficientBandwidthConfig.linkFailureThresholdPercent = args.get(
         'THRESHOLD', 80 )
      if 'linecard-shutdown' in args:
         gv.insufficientBandwidthConfig.linkFailureAction = 'linecardShutdown'
      elif 'log' in args:
         gv.insufficientBandwidthConfig.linkFailureAction = 'log'
      else:
         gv.insufficientBandwidthConfig.linkFailureAction = 'defaultAction'

   @staticmethod
   def noHandler( mode, args ):
      gv.insufficientBandwidthConfig.linkFailureActionIgnore = True
      gv.insufficientBandwidthConfig.linkFailureAction = 'ignore'

   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( InsufficientBandwidthCmd )

#--------------------------------------------------------------------------------
# switch fabric insufficient-bandwidth action ignore
#--------------------------------------------------------------------------------
class InsufficientBandwidthIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'switch fabric insufficient-bandwidth action ignore'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'fabric' : matcherFabric,
      'insufficient-bandwidth' : nodeInsufficientBandwidth,
      'action' : matcherAction,
      'ignore' : 'Take no action as a result of fabric bandwidth loss',
   }

   @staticmethod
   def handler( mode, args ):
      gv.insufficientBandwidthConfig.linkFailureActionIgnore = True
      gv.insufficientBandwidthConfig.linkFailureAction = 'ignore'

BasicCli.GlobalConfigMode.addCommandClass( InsufficientBandwidthIgnoreCmd )

def Plugin( em ):
   gv.fabricCliCapabilities = LazyMount.mount(
      em,
      'hardware/fabric/capabilities/cli',
      'Fabric::CliCapabilities',
      'r' )
   gv.insufficientBandwidthConfig = ConfigMount.mount(
      em,
      'hardware/fabric/cli/config/bandwidth',
      'Fabric::InsufficientBandwidthConfig',
      'w' )
