# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# The following is needed to introduce a dependency between the CLI and other RPMs
# pkgdeps: rpm EthIntf
# pkgdeps: rpm Intf

'''This module defines and registers the Fabric Interface CLI model.'''

import CliCommand
import CliGlobal
import CliParser
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IntfModel as IntfModel
import ConfigMount
import FabricIntfLib
import FabricIntfStatusMatcher
import IntfRangePlugin.FabricIntf as FabricIntfRangePlugin
import LazyMount
import Tac

EthDuplex = Tac.Type( 'Interface::EthDuplex' )
EthSpeedApi = Tac.Value( 'Interface::EthSpeedApi' )
IntfEnabledState = Tac.Type( 'Interface::IntfEnabledState' )
IntfIsolatedState = Tac.Type( 'Interface::IntfIsolatedState' )
IntfOperStatus = Tac.Type( 'Interface::IntfOperStatus' )
LinkStatus = Tac.Type( 'Interface::LinkStatus' )

gv = CliGlobal.CliGlobal( dict( intfConfigDir=None,
                                allFabricIntfStatusDir=None,
                                configRequestDir=None,
                                intfDescRootDir=None,
                                entMibStatus=None ) )

def valueFunc( mode, intf ):
   return FabricIntf( intf, mode )

class FabricIntf( EthIntfCli.L1Intf ):
   '''Defines the CLI type which represents Fabric Interfaces.

   Fabric interfaces represent a class of interfaces which is primarily used to
   connect one switch to another via a proprietary protocol so that both switches
   behave as one.

   They can either be internal to the switch or exist on the front panel making their
   interface IDs particularly tricky to parse. In general it is impossible to deduce
   the location of the fabric interface from it's interface ID. The port role must
   instead be used.

   Their configuration is limited but it does include speed, enabled state, etc.
   In the cases where the fabric interfaces are exposed to the front panel the
   interface are considered user configurable, else the FWD / PHY agents are
   responsible for their configurations.

   This limited CLI configuration is guarded behind configIfModeSupported in
   FabricCliCapabilities.

   TODO:
      BUG522169 : Add support for show interfaces counters CLI
      BUG532409 : Add support for show interfaces l2 mtu CLI
   '''

   def __init__( self, name, mode ):
      self.allFabricIntfStatusDir = gv.allFabricIntfStatusDir
      self.intfDescRootDir = gv.intfDescRootDir
      self.entMibStatus = gv.entMibStatus
      self.intfConfig_ = None
      self.intfStatus_ = None
      self.intfConfigDir = gv.intfConfigDir
      # The super class init needs to go below the intfConfig and intfStatus
      # attributes, so the internal() method can be safely invoked in the parent
      # class __init__ method. The internal() method will determine if an
      # intfXcvrStatusDir exists for fabric interfaces on this system.
      super().__init__( name, mode )

   # The rule for matching fabric interface names
   matcher = FabricIntfStatusMatcher.FabricIntfStatusMatcher(
      'Fabric',
      FabricIntfLib.Constants.localAllFabricIntfStatusPath(),
      helpdesc="Fabric Interfaces",
      value=valueFunc,
      alternates=[ 'Fa' ] )

   #---------------------------------------------------------------------------------
   # Interface lifespan controls
   #
   # Fabric interfaces are created and are completely managed by the platform
   # agents. The CLI does not have any bearing or impact on their lifespan.
   # However, to apply config on startup, CLI can pre-emptively create an intfConfig
   # for the interface - but this is only allowed when in startup-config mode.
   #---------------------------------------------------------------------------------
   def createInterfaceConfigurationRequest( self ):
      gv.configRequestDir.intf.add( self.name )

   def deleteInterfaceConfigurationRequest( self ):
      del gv.configRequestDir.intf[ self.name ]

   def createInterfaceConfiguration( self ):
      return self.intfConfigDir.intfConfig.newMember( self.name )

   def deleteInterfaceConfiguration( self ):
      del self.intfConfigDir.intfConfig[ self.name ]


   #---------------------------------------------------------------------------------
   #  CLI enable toggles
   #
   #  These methods control whether or not fabric interfaces are acceptable by
   #  various CLI commands.
   #---------------------------------------------------------------------------------

   def intfCreationCurrentlySupported( self, mode ):
      return True

   def countersSupported( self ):
      return False

   def routingSupported( self ):
      return False

   def routingCurrentlySupported( self ):
      return False

   def vrfSupported( self ):
      return False

   #---------------------------------------------------------------------------------
   # Interface related lookups
   #---------------------------------------------------------------------------------

   def lookup( self ):
      return self.lookupPhysical()

   def lookupPhysical( self ):
      for intfDescDir in self.intfDescRootDir.values():
         if self.name in intfDescDir.ethPhyIntfDesc:
            return True

      return False

   def config( self ):
      if self.intfConfig_ is None:
         self.intfConfig_ = self.intfConfigDir.intfConfig.get( self.name )

      return self.intfConfig_

   def status( self ):
      if self.intfStatus_ is None:
         self.intfStatus_  = self.allFabricIntfStatusDir.intfStatus.get( self.name )

      return self.intfStatus_

   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for intfDescDir in gv.intfDescRootDir.values():
         for intfId in intfDescDir.ethPhyIntfDesc:
            intfType = FabricIntf( intfId, mode )
            intfs.append( intfType )

      return intfs

   #---------------------------------------------------------------------------------
   # Basic status reporting methods required for Intf show commands
   #---------------------------------------------------------------------------------

   def slotId( self ):
      intfStatus = self.status()
      if not intfStatus:
         return None

      return intfStatus.slotId

   def active( self ):
      intfStatus = self.status()
      return not intfStatus or intfStatus.active

   def unconnected( self ):
      '''Fabric interfaces are always connected interfaces.'''
      return False

   def showPhysical( self, mode, intfStatusModel ):
      return

   def addrStr( self ):
      return None

   def hardware( self ):
      return "fabric"

   def bandwidth( self ):
      if not self.lookup():
         return 0

      return EthSpeedApi.speedKbps( self.status().speed )

   def lineProtocolState( self ):
      '''Fabric interfaces do not use oper status. Instead linkStatus is converted to
      oper status.
      '''
      if not self.lookup():
         return IntfModel.intfOperStatusToEnum( IntfOperStatus.intfOperUnknown )

      sts = self.status()
      if sts.linkStatus == LinkStatus.linkUp:
         if sts.isolatedState != IntfIsolatedState.unIsolated:
            return IntfModel.intfOperStatusToEnum( IntfOperStatus.intfOperDown )
         return IntfModel.intfOperStatusToEnum( IntfOperStatus.intfOperUp )
      elif sts.linkStatus == LinkStatus.linkDown:
         return IntfModel.intfOperStatusToEnum( IntfOperStatus.intfOperDown )
      else:
         return IntfModel.intfOperStatusToEnum( IntfOperStatus.intfOperUnknown )

   def getStatus( self ):
      intfStatus = self.status()
      lineProtocolStatus = self.lineProtocolState()

      if intfStatus is None:
         interfaceStatus = 'uninitialized'
      elif not intfStatus.active:
         interfaceStatus = 'inactive'
      elif intfStatus.enabledState != IntfEnabledState.enabled:
         interfaceStatus = 'disabled'
      elif intfStatus.isolatedState == IntfIsolatedState.adminIsolated:
         interfaceStatus = "adminIsolated"
      elif intfStatus.isolatedState == IntfIsolatedState.autoIsolated:
         interfaceStatus = "autoIsolated"
      else:
         interfaceStatus = self.linkStatus()

      return ( lineProtocolStatus, interfaceStatus )

   def getIntfState( self ):
      if self.status().linkStatus == LinkStatus.linkUp:
         if self.status().isolatedState == IntfIsolatedState.adminIsolated:
            return "adminIsolated"
         elif self.status().isolatedState == IntfIsolatedState.autoIsolated:
            return "autoIsolated"
         else:
            return 'up'
      else:
         return 'down'

   #---------------------------------------------------------------------------------
   # Interface status reporting methods required for compatibility with EthIntf show
   # commands
   #---------------------------------------------------------------------------------

   def getIntfDuplexStr( self ):
      if not self.lookup():
         return EthDuplex.duplexUnknown()

      return self.status().duplex

   def linkStatus( self ):
      if not self.config().enabled:
         if ( not self.config().adminEnabled ) and self.status().active:
            return 'disabled'
         else:
            reason = self.config().enabledStateReason
            return reason if reason else 'unknown'

      elif self.status().isolatedState == IntfIsolatedState.adminIsolated:
         return "adminIsolated"
      elif self.status().isolatedState == IntfIsolatedState.autoIsolated:
         return "autoIsolated"

      linkStatusMap = EthIntfCli.EthIntf.linkStatusStrings.get( 'Format2' )
      return linkStatusMap[ self.status().linkStatus ]

   def ethPhyDefaultCaps( self ):
      if self.internal():
         return None
      sliceName = self.slotId()
      ethPhyDefaultCapsDirs = EthIntfCli.ethPhyDefaultCapsSliceDir.get( sliceName )
      if ethPhyDefaultCapsDirs:
         for ethPhyDefaultCapsDir in ethPhyDefaultCapsDirs.values():
            ethPhyDefaultCaps = ethPhyDefaultCapsDir.portCaps.get( self.name )
            if ethPhyDefaultCaps:
               return ethPhyDefaultCaps
      return None

   def ethIntfMode( self ):
      return self.status().mode

   #---------------------------------------------------------------------------------
   # Counters related methods
   #---------------------------------------------------------------------------------

   def getIntfCounterDir( self ):
      return None

   #---------------------------------------------------------------------------------
   # Configuration methods
   #---------------------------------------------------------------------------------

   def setDefault( self ):
      # pylint: disable=useless-super-delegation
      # Thank you pylint, but the comment below begs to differ.
      #
      # TODO: Once fabric interfaces start supporting more and more commands they
      #       either need to be added here or to the common L1Intf base
      super().setDefault()

# Register fabric interfaces with the interface configuration modelet
class FabricModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, FabricIntf )

# Modelet for IntfRangeConfigMode
class FabricIntfRangeModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.individualIntfModes_[ 0 ].intf, FabricIntf )

# Register all supported fabric interface configuration commands
#-------------------------------------------------------------------------------
# "[no|default] isolated" config command
#-------------------------------------------------------------------------------
class IsolatedCmd( CliCommand.CliCommandClass ):
   syntax = "isolated"
   noOrDefaultSyntax = syntax
   data = {
      'isolated' : "Configure fabric interface isolation",
   }

   @staticmethod
   def handler( mode, args ):
      mode.intf.config().adminIsolated = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.intf.config().adminIsolated = False

FabricModelet.addCommandClass( IsolatedCmd )

FabricModelet.addCommandClass( EthIntfCli.SpeedForcedCmd )
FabricModelet.addCommandClass( EthIntfCli.SpeedCmd )
FabricModelet.addCommandClass( EthIntfCli.LoopbackModeCmd )
FabricModelet.addCommandClass( EthIntfCli.LinkDebounceCmd )
FabricModelet.addCommandClass( EthIntfCli.ErrorCorrectionEncodingCmd  )

FabricIntfRangeModelet.addCommandClass( EthIntfCli.SpeedForcedRangeCmd )
FabricIntfRangeModelet.addCommandClass( EthIntfCli.SpeedRangeCmd )
FabricIntfRangeModelet.addCommandClass( EthIntfCli.LinkDebounceCmd )
FabricIntfRangeModelet.addCommandClass( EthIntfCli.ErrorCorrectionEncodingCmd  )

IntfCli.IntfConfigMode.addModelet( FabricModelet )

# Register fabric interfaces with the interface range matchers
IntfCli.Intf.addPhysicalIntfType( FabricIntf,
                                  FabricIntfRangePlugin.fabricAutoIntfType )

def Plugin( em ):
   gv.intfConfigDir = ConfigMount.mount(
      em,
      FabricIntfLib.Constants.intfConfigPath(),
      'Interface::FabricIntfConfigDir',
      'w' )

   gv.allFabricIntfStatusDir = LazyMount.mount(
      em,
      FabricIntfLib.Constants.localAllFabricIntfStatusPath(),
      'Interface::FabricIntfStatusPtrConstDir',
      'r' )

   gv.configRequestDir = ConfigMount.mount(
      em,
      FabricIntfLib.Constants.intfConfigRequestCliPath(),
      'Interface::IntfConfigRequestDir',
      'w' )

   gv.intfDescRootDir = LazyMount.mount(
      em,
      FabricIntfLib.Constants.intfDescRootPath(),
      'Tac::Dir',
      'ri' )

   gv.entMibStatus = LazyMount.mount(
      em,
      "hardware/entmib",
      "EntityMib::Status",
      "r" )
