# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import IpUtils
import Ethernet

from CliModel import Model
from CliModel import List
from CliModel import Int
from CliModel import Str
from CliModel import Enum
from CliModel import Bool
from CliModel import Float
from CliModel import Submodel
from CliModel import Dict

import Ark
from ArnetModel import Ip4Address
from ArnetModel import IpAddrAndMask
from ArnetModel import Ip6Address
from ArnetModel import Ip6AddrAndMask
from ArnetModel import MacAddress
from CliPlugin import IntfCli
from CliPlugin.IntfModel import InterfaceStatus
from CliPlugin.IntfModel import InterfacesDescriptions
from IntfModels import Interface
import Tac
import TableOutput
import functools

ethAddrZero = '00:00:00:00:00:00'
defaultMask = 'ff:ff:ff:ff:ff:ff'
_briefFields = [ 'Interface', 'VRF', 'ID', 'Ver', 'Pri', 'Time', 'State',
                 'Last Transition', 'VR IP Addresses' ]
_briefTypes = [ 's', 's', 'd', 'd', 'd', 'd', 's', 's', 's' ]
_briefWidths = [ 9, 10, 3, 3, 3, 5, 6, 19, 15 ]
defaultV4Addr = '0.0.0.0'
defaultV6Addr = '::'

protocolStatusDict = {
   "up": "U",
   "down": "D",
   "testing": "T",
   "unknown": "UN",
   "notpresent": "NP",
   "lowerlayerdown": "LLD"
}

def renderTransitionTime( transTime, brief=False ):
   timeSinceTransition = int( Tac.utcNow() - transTime )
   weeks = timeSinceTransition // 604800
   timeSinceTransition %= 604800
   days = timeSinceTransition // 86400
   timeSinceTransition %= 86400
   hours = timeSinceTransition // 3600
   timeSinceTransition %= 3600
   minutes = timeSinceTransition // 60
   timeSinceTransition %= 60
   seconds = timeSinceTransition
   output = ""
   if weeks:
      output = "%d" % weeks
      output += "%s " % ( "w" if brief else " weeks," )
   if days:
      output += "%d" % days
      output += "%s " % ( "d" if brief else " days," )
   output += "%02d:%02d:%02d ago" % ( hours, minutes, seconds )
   return output

class VrrpRouterBase( Model ):
   interface = Interface( help='Name of the interface' )
   vrfName = Str( help='Name of VRF', optional=True )
   groupId = Int( help='VRRP group ID' )
   version = Int( help='VRRP version' )
   priority = Int( help='Priority of the virtual router' )
   state = Enum( values=( 'stopped', 'backup', 'master' ),
                 help='stopped -- the virtual router has been shutdown, '
                 'backup -- the virtual router is in backup mode, '
                 'master -- the virtual router is the master of the VRRP group.' )
   masterDownInterval = Int( help='The master down interval of the '
                                  'virtual router in milliseconds' )
   stateTransitionTime = Float( help='Timestamp in UTC of last VRRP state '
                                     'transition' )

class VrrpBriefModel( Model ):
   class VrrpRouterBriefBase( VrrpRouterBase ):
      def renderBrief( self, ipAddr ):
         data = [ IntfCli.Intf.getShortname( self.interface.stringValue ),
                  self.vrfName[ : 10 ], self.groupId, self.version, self.priority,
                  self.masterDownInterval, self.state.title(),
                  renderTransitionTime( self.stateTransitionTime, brief=True ),
                  ipAddr ]
         for field, width, type_ in zip( data, _briefWidths, _briefTypes ):
            fmt = '%%-%d%s' % ( width, type_ )
            print( fmt % field, end=' ' )
         print( '' )

   class VrrpRouterBriefV4( VrrpRouterBriefBase ):
      virtualIp = Ip4Address( help='The primary virtual Ipv4 address' )

      def render( self ):
         self.renderBrief( self.virtualIp )

   class VrrpRouterBriefV6( VrrpRouterBriefBase ):
      virtualIps = List( valueType=Ip6Address,
                        help='Virtual Ipv6 addresses' )

      def render( self ):
         ipAddr = self.virtualIps[ 0 ]
         self.renderBrief( ipAddr )

         spaces = sum( _briefWidths[ : -1 ] ) + len( _briefWidths ) - 2
         for ip in sorted( self.virtualIps[ 1 : ],
                           key=functools.cmp_to_key( IpUtils.compareIp6Address ) ):
            print( spaces * ' ', ip )

   virtualRouters = List( valueType=VrrpRouterBriefBase,
                          help='Virtual router status' )

   def render( self ):
      self.renderHeader()
      for vr in self.virtualRouters:
         vr.render()

   def renderHeader( self ):
      for field, width in zip( _briefFields, _briefWidths ):
         rep = '%-*s' % ( width, field )
         print( rep, end=' ' )
      print( '' )
      print( ' '.join( '-' * w for w in _briefWidths ) )

class TrackedObject( Model ):
   trackedObject = Str( help="Tracked object" )
   interface = Interface( help='Name of the interface' )
   shutdown = Bool( help='Shutdown action' )
   priority = Int( help='Priority of the tracked object' )

   def render( self ):
      if self.shutdown:
         print( '  Tracking object %s with shutdown action' % self.trackedObject )
      else:
         # pylint: disable-next=bad-string-format-type
         print( '  Tracking object %s with priority %d' % ( self.trackedObject,
                                                            self.priority ) )
      print( '    Interfaces: %s' % self.interface.stringValue )

class VrrpDetailModel( Model ):
   class VrrpRouterDetailBase( VrrpRouterBase ):
      description = Str( help='Vrrp description' )
      virtualMac = MacAddress( help='The virtual mac address of this vrrp group' )
      macAddressInterval = Int( help='The time in seconds between sending neighbor '
                                'discovery or arp of the virtual mac address.' )
      vrrpAdvertInterval = Int( help='The time in seconds between sending Vrrp '
                                'packets.' )
      preempt = Bool( help='Preemption status' )
      preemptDelay = Int( help='Preemption delay time in seconds' )
      preemptReload = Int( help='Preemption reload delay time in seconds' )
      masterPriority = Int( help='Priority of the master virtual router' )
      masterInterval = Int( help='The master advertisement interval in seconds' )
      skewTime = Float( help='The skew of the master down interval in seconds' )
      vrIdDisabled = Bool( help='Virtual router id is disabled' )
      vrIdDisabledReason = Str( help='Virtual router id disabled description' )
      trackedObjects = List( valueType=TrackedObject,
                             help='Vrrp tracked objects' )

      def render( self ):
         if self.vrIdDisabled:
            print( self.interface.stringValue, '-', 'Group', self.groupId,
                   "(Disabled. %s.)" % self.vrIdDisabledReason )
         else:
            print( self.interface.stringValue, '-', 'Group', self.groupId )

         if len( self.description ):
            print( self.description )

         print( '  VRF is', self.vrfName )
         print( '  VRRP Version', self.version )
         print( '  State is', self.state.title() )
         print( '  Last state transition was',
                Ark.utcTimeRelativeToNowStr( self.stateTransitionTime ) )

         self.renderAddrs()

         print( '  Virtual MAC address is', end=' ' )
         print( Ethernet.convertMacAddrToDisplay( str( self.virtualMac ) ) )

         self.renderMacWarning()

         print( '  Mac Address Advertisement interval is %ds' %
                self.macAddressInterval )
         print( '  VRRP Advertisement interval is %ds' % self.vrrpAdvertInterval )

         if self.preempt:
            print( '  Preemption is enabled' )
         else:
            print( '  Preemption is disabled' )

         print( '  Preemption delay is %ds' % self.preemptDelay )
         print( '  Preemption reload delay is %ds' % self.preemptReload )
         print( '  Priority is', self.priority )

         self.renderAuth()

         if self.masterPriority == 0 and self.state == 'master':
            print( "  Master Router is %s (local)" % self.masterAddr )
         elif self.masterPriority == 0:
            print( "  Master Router is %s" % self.masterAddr )
         elif self.state == 'master':
            # pylint: disable-next=bad-string-format-type
            print( "  Master Router is %s (local), priority is %d" %
                   ( self.masterAddr, self.masterPriority ) )
         else:
            # pylint: disable-next=bad-string-format-type
            print( "  Master Router is %s, priority is %d" %
                   ( self.masterAddr, self.masterPriority ) )

         print( '  Master Advertisement interval is %ds' % self.masterInterval )
         print( '  Skew time is %.3fs' % self.skewTime )
         print( '  Master Down interval is %.3fs' % ( self.masterDownInterval /
                                                      1000.0 ) )

         self.renderBfdPeer()

         for obj in self.trackedObjects:
            obj.render()

      # Need to have function declared
      def renderAddrs( self ):
         pass

      def renderAuth( self ):
         pass

      def renderMacWarning( self ):
         pass

      def renderBfdPeer( self ):
         pass

   class VrrpRouterDetailV4( VrrpRouterDetailBase ):
      virtualIp = Ip4Address( help='Configured primary virtual Ipv4 address' )
      virtualIpSecondary = List( valueType=Ip4Address,
                                 help='Configured secondary virtual Ipv4 addresses' )
      masterAddr = Ip4Address( help='Current address of the master virtual router' )
      authType = Enum( values=( 'text', 'md5' ), optional=True,
                       help='The type of authentication being used' )
      authSringEncrytType = Enum( values=( '7', '8a' ), optional=True,
                        help='The authentication password encryption algorithm' )
      authString = Str( optional=True,
                        help='The authentication password encrypted text' )
      bfdPeerAddr = Ip4Address( help='Bfd peer of the virtual router' )
      bfdPeerState = Enum( values=( 'down', 'up', 'shuttingDown' ), optional=True,
                                    help='Bfd peer state of the virtual router' )

      def renderAddrs( self ):
         print( '  Virtual IPv4 address is', self.virtualIp )
         for addr in self.virtualIpSecondary:
            if addr != self.virtualIp:
               print( '    Secondary Virtual IPv4 address is', addr )

      def renderAuth( self ):
         if self.version == 2:
            if self.authType:
               authStr = f'{self.authSringEncrytType} {self.authString}'
               if self.authType == 'text':
                  authTypeStr = 'text, string'
               else:
                  authTypeStr = 'MD5, key-string'
               print( f' Authentication {authTypeStr} \"{authStr}\"' )

      def renderBfdPeer( self ):
         if str( self.bfdPeerAddr ) != defaultV4Addr:
            print( '  BFD peer address is %s' % str( self.bfdPeerAddr ) )
            print( '  BFD state is %s' % str( self.bfdPeerState ) )

   class VrrpRouterDetailV6( VrrpRouterDetailBase ):
      virtualIpv6 = List( valueType=Ip6Address,
                         help='Configured secondary virtual Ipv6' )
      masterAddr = Ip6Address( help='Current address of the master virtual router' )
      linkLocalAddr = Ip6Address( help='Derived link-local Ipv6 from the VRRP mac '
                                       'or the configured link-local Ipv6 address' )
      bfdPeerAddr = Ip6Address( help='Bfd peer of the virtual router' )
      bfdPeerState = Enum( values=( 'down', 'up', 'shuttingDown' ), optional=True,
                                    help='Bfd peer state of the virtual router' )


      def renderAddrs( self ):
         print( '  Virtual IPv6 link-local address is', self.linkLocalAddr )
         for addr in self.virtualIpv6:
            print( '  Virtual IPv6 address is', addr )

      def renderMacWarning( self ):
         if ( ( self.virtualMac.word2 & 0xFF00 ) >> 8 ) == 1:
            print( '    Using VRRP IPv4 MAC prefix' )

      def renderBfdPeer( self ):
         if str( self.bfdPeerAddr ) != defaultV6Addr:
            print( '  BFD peer address is %s' % str( self.bfdPeerAddr ) )
            print( '  BFD state is %s' % str( self.bfdPeerState ) )

   virtualRouters = List( valueType=VrrpRouterDetailBase,
                          help='Virtual router status' )

   def render( self ):
      first = True
      for vr in self.virtualRouters:
         if first:
            first = False
         else:
            print()
         vr.render()

class VirtualRouterMacModel( Model ):
   macAddress = MacAddress( help='Virtual router MAC address' )
   macType = Enum( values=( 'varp', 'mlag' ),
                   help='varp -- VARP MAC address, '
                   'mlag -- Mlag peer router MAC address.' )

   def render( self ):
      pass

class VarpModel( Model ):
   __revision__ = 2

   class VarpRouterMac( VirtualRouterMacModel ):
      mask = MacAddress( help='Virtual router MAC address mask', optional=True )
      subnetRoutes = Bool( help='Virtual subnet routes enabled' )

      def render( self ):
         varpMac = str( self.macAddress )
         varpMask = str( self.mask )
         if varpMac == ethAddrZero and varpMask == defaultMask:
            print( 'IP virtual router MAC address is not configured' )
         else:
            print( 'IP virtual router is configured with MAC address:', end=' ' )
            print( Ethernet.convertMacAddrToDisplay( varpMac ) )
            if varpMask != defaultMask:
               print( 'IP virtual router is configured with MAC mask:', end=' ' )
               print( Ethernet.convertMacAddrToDisplay( varpMask ) )
            print( 'IP virtual router address subnet routes {}enabled'.format(
               '' if self.subnetRoutes else 'not ' ) )

   class VarpRouterBase( Model ):
      interface = Interface( help='Name of Interface' )
      vrfName = Str( help='Name of VRF' )
      interfaceStatus = InterfacesDescriptions.InterfaceDescription.interfaceStatus
      protocolStatus = InterfaceStatus.lineProtocolStatus
      state = Enum( values=( 'stopped', 'active' ),
                    help='stopped -- the virtual router is inactive, '
                    'active -- the virtual router is active', optional=True )

      def renderVr( self, last, table ):
         pass

   class VarpRouterV4( VarpRouterBase ):
      ipAddress = Submodel( valueType=IpAddrAndMask, help='Interface address' )
      virtualIps = List( valueType=IpAddrAndMask, help='Virtual IPv4 addresses' )

      def renderVr( self, last, table ):
         for vip in self.virtualIps:
            if vip.mask == 32:
               vipStr = vip.ip
            else:
               vipStr = '%s/%s' % ( vip.ip, vip.mask )

            line = ( IntfCli.Intf.getShortname( self.interface.stringValue ),
                     self.vrfName, vipStr,
                     protocolStatusDict[ self.protocolStatus.lower() ],
                     self.state )
            table.newRow( *line )

         if last:
            print( table.output() )

   class VarpRouterV6( VarpRouterBase ):
      ipv6Addresses = List( valueType=Ip6AddrAndMask,
                            help='Ipv6 interface addresses' )
      virtualIpv6 = List( valueType=Ip6Address,
                          help='Virtual Ipv6 addresses' )

      def renderVr( self, last, table ):
         for vAddr in sorted(
               self.virtualIpv6,
               key=functools.cmp_to_key( IpUtils.compareIp6Address ) ):
            line = ( IntfCli.Intf.getShortname( self.interface.stringValue ),
                     self.vrfName, vAddr,
                     protocolStatusDict[ self.protocolStatus.lower() ],
                     self.state )
            table.newRow( *line )

         if last:
            print( table.output() )

   virtualMacs = List( valueType=VirtualRouterMacModel,
                       help='Virtual MAC addresses configured.' )
   advertiseInterval = Int( help='Mac address advertisement interval' )
   virtualRouters = List( valueType=VarpRouterBase,
                          help='Virtual Routers configured' )

   def render( self ):
      varpMac = None
      for v in self.virtualMacs:
         v.render()
         if v.macType == 'varp':
            varpMac = str( v.macAddress )

      if varpMac and varpMac != ethAddrZero:
         print( 'MAC address advertisement interval:', end=' ' )
         print( self.advertiseInterval, 'seconds' )
         if len( self.virtualRouters ):
            print( "" )
            print( "Protocol: U - Up, D - Down, T - Testing, UN - Unknown" )
            print( "          NP - Not Present, LLD - Lower Layer Down" )
            print( "" )

            first = True
            last = False
            for vr in self.virtualRouters:
               if first == True: # pylint: disable=singleton-comparison
                  first = False
                  if ( vr.__class__.__name__ ) == 'VarpRouterV4':
                     virtIpVersion = 'Virtual IP Address'
                  else:
                     virtIpVersion = 'Virtual IPv6 Address'
                  tableHeadings = ( 'Interface', 'Vrf', virtIpVersion,
                                    'Protocol', 'State' )
                  table = TableOutput.createTable( tableHeadings )
                  colLens = [ 10, 14, 40, 10, 8 ]
                  colFormats = []
                  for l in colLens:
                     f = TableOutput.Format( justify='left', maxWidth=l, wrap=True )
                     f.noPadLeftIs( True )
                     colFormats.append( f )
                  table.formatColumns( *colFormats )

               if vr == self.virtualRouters[ -1 ]:
                  last = True
               vr.renderVr( last, table )
         else:
            print( 'No interface with virtual IP address' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # take varp from virtualMacs -> to virtualMac
         dictRepr[ 'virtualMac' ] = ethAddrZero
         for vmac in dictRepr[ 'virtualMacs' ]:
            if vmac[ 'macType' ] == 'varp':
               dictRepr[ 'virtualMac' ] = vmac[ 'macAddress' ]
         del dictRepr[ 'virtualMacs' ]

         # convert list of virtualIps with mask -> list of virtualIps
         for vr in dictRepr[ 'virtualRouters' ]:
            if 'virtualIps' in vr:
               vips = []
               for vip in vr[ 'virtualIps' ]:
                  vips.append( vip[ 'ip' ] )
               vr[ 'virtualIps' ] = vips
            if 'state' in vr:
               del vr[ 'state' ]
      return dictRepr

class VarpSourceNatV4Model( Model ):
   vrfs = Dict( keyType=str, valueType=Ip4Address,
                help='A mapping of VRF names to their source NAT IP address' )

   def render( self ):
      headings = ( 'VRF Name', 'Source NAT IP Address' )
      t = TableOutput.createTable( ( h, 'l' ) for h in headings )

      for vrf, ipAddr in self.vrfs.items():
         t.newRow( vrf, ipAddr )

      print( t.output() )

class VarpSourceNatV6Model( Model ):
   vrfs = Dict( keyType=str, valueType=Ip6Address,
                help='A mapping of VRF names to their source NAT IPv6 address' )

   def render( self ):
      headings = ( 'VRF Name', 'Source NAT IPv6 Address' )
      t = TableOutput.createTable( ( h, 'l' ) for h in headings )

      for vrf, ip6Addr in self.vrfs.items():
         t.newRow( vrf, ip6Addr )

      print( t.output() )
