# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
from CliPlugin.ClassificationCliLib import applicationModeGuards
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
import CliToken.Clear
import CliToken.Router
import LazyMount
import ShowCommand

# pkgdeps: rpm FirewallLib-lib
# pkgdeps: rpm MatchList-lib

# --------------------------------------------------------------------------
# Firewall globals
# --------------------------------------------------------------------------

gv = CliGlobal.CliGlobal(
   firewallConfig=None,
   hwCapability=None,
   hwConfig=None,
)

# --------------------------------------------------------------------------
# Utility functions
# --------------------------------------------------------------------------
def segSecSupportedGuard( mode, token ):
   if gv.hwCapability.segSecSupported:
      return None
   return CliParser.guardNotThisPlatform

def vrfSupportedGuard( mode, token ):
   if gv.hwCapability.vrfSupported:
      return None
   return CliParser.guardNotThisPlatform

def clearVrfSessionsSupportedGuard( mode, token ):
   if gv.hwCapability.clearVrfSessionsSupported and gv.hwCapability.vrfSupported:
      return None
   return CliParser.guardNotThisPlatform

def segSecCountersSupportedGuard( mode, token ):
   if gv.hwCapability.segSecCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

def errorReportingSupportedGuard( mode, token ):
   if gv.hwCapability.errorReportingSupported:
      return None
   return CliParser.guardNotThisPlatform

def applicationModePlugin( token ):
   if not gv.hwCapability.segSecSupported:
      return False
   if token == "ipv4": # enable `application ipv4 APPNAME` if these are supported
      return ( gv.hwCapability.segmentMatchInterfaceSupported and
               gv.hwCapability.segSecConfigurablePolicySupported )
   if token == "l4": # enable `application l4 APPNAME`
      return gv.hwCapability.segSecConfigurablePolicySupported
   return False

# --------------------------------------------------------------------------
# Helper Functions
# --------------------------------------------------------------------------
def getAllServiceName( mode ):
   return list( gv.hwConfig.services )

def getAllPoliciesName( mode ):
   return list( gv.firewallConfig.policy )

# --------------------------------------------------------------------------
# Common nodes and Matchers
# --------------------------------------------------------------------------
firewallPolicyNameMatcher = CliMatcher.DynamicNameMatcher( getAllPoliciesName,
                                                           helpdesc='Policy name' )

segmentNameMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                               helpdesc='Segment name',
                                               helpname='WORD' )

applicationNameMatcher = CliMatcher.DynamicNameMatcher( getAllServiceName,
                                                       helpdesc='Application name' )

ipv4Matcher = IpAddrMatcher( helpdesc='IPv4 address' )

ipv6Matcher = Ip6AddrMatcher( 'IPv6 address' )

segmentSecurityNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'segment-security',
         helpdesc='Show segment security information' ),
      guard=segSecSupportedGuard )

segmentSecurityNodeForClear = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'segment-security',
         helpdesc='Clear segment security information' ),
      guard=segSecSupportedGuard )

segmentSecurityCountersNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'counters',
         helpdesc='Segment security counters' ),
      guard=segSecCountersSupportedGuard )

vrfExprFactoryForShow = VrfExprFactory(
      helpdesc='Show segment security information in VRF',
      guard=vrfSupportedGuard )

vrfExprFactoryForClear = VrfExprFactory(
      helpdesc='Clear segment security information in VRF',
      guard=vrfSupportedGuard )

# --------------------------------------------------------------------------
# 'router segment-security' to enter config-router-seg-sec mode
# --------------------------------------------------------------------------
class RouterSegmentSecurityModeCmd( CliCommand.CliCommandClass ):
   syntax = 'router segment-security'
   noOrDefaultSyntax = syntax
   data = {
         'router': CliToken.Router.routerMatcherForConfig,
         'segment-security': segmentSecurityNode,
   }

   handler = 'FirewallCliHandler.routerSegmentSecurityHandler'
   noOrDefaultHandler = 'FirewallCliHandler.routerSegmentSecurityNoOrDefaultHandler'

BasicCli.GlobalConfigMode.addCommandClass( RouterSegmentSecurityModeCmd )

# --------------------------------------------------------------------------
# 'show segment-security [ VRF ] [ segment <segment-name> ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurity( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security [ VRF ] [ segment SEGMENT_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'VRF': vrfExprFactoryForShow,
         'segment': 'Show segment security segment information',
         'SEGMENT_NAME': segmentNameMatcher,
   }
   cliModel = 'FirewallModels.FirewallInfo'
   handler = 'FirewallCliShowHandler.showSegmentSecurityHandler'

BasicCli.addShowCommandClass( ShowSegmentSecurity )

# --------------------------------------------------------------------------
# 'show segment-security [ VRF ] to ( ( DST_IP [ from SRC_IP ] )'\
#     ' | ( DST_IPV6 [ from SRC_IPV6 ] ) )' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurityPolicyTo( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security [ VRF ] to ( ( DST_IP [ from SRC_IP ] )'\
      ' | ( DST_IPV6 [ from SRC_IPV6 ] ) )'
   data = {
         'segment-security': segmentSecurityNode,
         'VRF': vrfExprFactoryForShow,
         'to': 'Show segment security policy to destination host',
         'from': 'Show segment security policy from source host',
         'ip': 'Match IPv4 address',
         'ipv6': 'Match IPv6 address',
         'DST_IP': ipv4Matcher,
         'SRC_IP': ipv4Matcher,
         'DST_IPV6': ipv6Matcher,
         'SRC_IPV6': ipv6Matcher,
   }
   cliModel = 'FirewallModels.FwPolicyToInfo'
   handler = 'FirewallCliShowHandler.showSegmentSecurityPolicyToHandler'

BasicCli.addShowCommandClass( ShowSegmentSecurityPolicyTo )

# --------------------------------------------------------------------------
# 'show segment-security policy [ policy-name ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurityPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security policy [ POLICY_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'policy': 'Show segment security policy',
         'POLICY_NAME': firewallPolicyNameMatcher,
   }
   cliModel = 'FirewallModels.FwPolicyInfo'
   handler = 'FirewallCliShowHandler.showSegmentSecurityPolicyHandler'

BasicCli.addShowCommandClass( ShowSegmentSecurityPolicy )

# --------------------------------------------------------------------------
# 'show segment-security application [ application-name ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurityApplication( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security application [ APPLICATION_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'application': 'Show segment security application',
         'APPLICATION_NAME': applicationNameMatcher,
   }
   cliModel = 'FirewallModels.FwServiceInfo'
   handler = 'FirewallCliShowHandler.showSegmentSecurityApplicationHandler'

BasicCli.addShowCommandClass( ShowSegmentSecurityApplication )

# --------------------------------------------------------------------------
# clear segment-security sessions [ VRF ]
# --------------------------------------------------------------------------
class ClearSegmentSecuritySessions( CliCommand.CliCommandClass ):
   syntax = 'clear segment-security sessions [ VRF ]'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'segment-security': segmentSecurityNodeForClear,
         'sessions': 'Segment security sessions',
         'VRF': VrfExprFactory(
                       helpdesc='Clear sessions belonging to a particular VRF',
                       guard=clearVrfSessionsSupportedGuard ),
   }

   handler = 'FirewallCliHandler.clearSegmentSecuritySessionsHandler'

BasicCli.EnableMode.addCommandClass( ClearSegmentSecuritySessions )

# --------------------------------------------------------------------------
# show segment-security sessions [ VRF ]
# --------------------------------------------------------------------------
class ShowSegmentSecuritySessions( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security sessions [ VRF ]'
   data = {
         'segment-security': segmentSecurityNode,
         'sessions': 'Segment security sessions',
         'VRF': vrfExprFactoryForShow,
   }
   handler = 'FirewallCliShowHandler.showSegmentSecuritySessionsHandler'

BasicCli.addShowCommandClass( ShowSegmentSecuritySessions )

# --------------------------------------------------------------------------
# 'show segment-security counters [ VRF ]' CLI
# --------------------------------------------------------------------------
class ShowSegmentSecurityCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security counters [ VRF ]'
   data = {
         'segment-security': segmentSecurityNode,
         'counters': segmentSecurityCountersNode,
         'VRF': vrfExprFactoryForShow,
   }
   cliModel = 'FirewallModels.FwCounterInfo'
   handler = 'FirewallCliShowHandler.showSegmentSecurityCountersHandler'

BasicCli.addShowCommandClass( ShowSegmentSecurityCounters )

# -------------------------------------------------------------------------------
# 'show segment-security status [ VRF ] [ segment <segment-name> ]'
# -------------------------------------------------------------------------------
class ShowSegmentSecurityStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security status [ VRF ] [ segment SEGMENT_NAME ]'
   data = {
         'segment-security': segmentSecurityNode,
         'status': 'Show status of segment security',
         'VRF': vrfExprFactoryForShow,
         'segment': 'Show status of segment',
         'SEGMENT_NAME': segmentNameMatcher,
   }
   cliModel = 'FirewallModels.FwHwStatusInfo'
   handler = 'FirewallCliShowHandler.showSegmentSecurityStatusHandler'

BasicCli.addShowCommandClass( ShowSegmentSecurityStatus )

# -------------------------------------------------------------------------------
# 'show segment-security errors [ VRF [ segment <segment-name> ] ]'
# -------------------------------------------------------------------------------
class ShowSegmentSecurityErrors( ShowCommand.ShowCliCommandClass ):
   syntax = 'show segment-security errors [ VRF [ segment SEGMENT_NAME ] ]'
   data = {
         'segment-security': segmentSecurityNode,
         'errors': CliCommand.guardedKeyword(
               'errors', helpdesc='Show errors with segment-security',
               guard=errorReportingSupportedGuard ),
         'VRF': vrfExprFactoryForShow,
         'segment': 'Show errors with segment',
         'SEGMENT_NAME': segmentNameMatcher,
   }
   cliModel = 'FirewallModels.FwHwErrorsInfo'
   handler = 'FirewallCliShowHandler.showSegmentSecurityErrorsHandler'

BasicCli.addShowCommandClass( ShowSegmentSecurityErrors )

# --------------------------------------------------------------------------
# clear segment-security counters [ VRF ]
# --------------------------------------------------------------------------
class ClearSegmentSecurityCounters( CliCommand.CliCommandClass ):
   syntax = 'clear segment-security counters [ VRF ]'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'segment-security': segmentSecurityNodeForClear,
         'counters': segmentSecurityCountersNode,
         'VRF': vrfExprFactoryForClear,
   }

   handler = 'FirewallCliHandler.clearSegmentSecurityCountersHandler'

BasicCli.EnableMode.addCommandClass( ClearSegmentSecurityCounters )

applicationModeGuards.addExtension( applicationModePlugin )

def Plugin( entityManager ):
   gv.firewallConfig = LazyMount.mount(
         entityManager, 'firewall/config/cli', 'Firewall::Config', 'r' )
   gv.hwCapability = LazyMount.mount(
         entityManager, 'firewall/hw/capability', 'Firewall::HwCapability', 'r' )
   gv.hwConfig = LazyMount.mount(
         entityManager, 'firewall/hw/config', 'Firewall::HwConfig', 'r' )
