# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMode.FlexEncap
import CliMode.Intf
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import MultiRangeRule
import Tac
from Toggles.EbraToggleLib import (
   toggleFlexEncapUntaggedEnabled,
   toggleFlexEncapNetworkClientInnerEnabled,
   toggleFlexEncapSubIntfMultiEncapEnabled,
   toggleFlexEncapSubIntfClientIdEnabled, )
from TypeFuture import TacLazyType
from CliPlugin.SubIntfCli import ( SubIntfModelet, theQinQL3SubintfSupportedGuard )
from CliPlugin.VlanCli import ( vlanIdMatcher, vlanIdListFunc, VlanSet )

intfCapability = None
subIntfConfigDir = None

Dot1qEncap = TacLazyType( 'Bridging::Dot1qEncap' )
Dot1qEncapConfigMode = TacLazyType( 'Interface::SubIntfDot1qEncapConfigMode' )
FlexEncapConstants = TacLazyType( 'Ebra::FlexEncapConstants' )
FlexEncapRange = TacLazyType( 'Ebra::FlexEncapRange' )
FlexEncapField = TacLazyType( 'Ebra::FlexEncapField' )
FieldType = TacLazyType( 'Ebra::FlexEncapField::FieldType' )
FlexEncapSpec = TacLazyType( 'Ebra::FlexEncapSpec' )
FlexEncapEntry = TacLazyType( 'Ebra::FlexEncapEntry' )
ReservedClientIds = TacLazyType( 'Ebra::FlexEncapConstants::ReservedClientIds' )
SpecType = TacLazyType( 'Ebra::FlexEncapSpec::SpecType' )
VlanEncapReason = TacLazyType( 'Interface::SubIntfConfigDir::VlanEncapReason' )
VlanTpid = TacLazyType( 'Bridging::VlanTpid' )

ENCAP_ASYMMETRIC_STR = ( "client and network encapsulations do not have the same "
                         "number of VLAN tags" )
ENCAP_INVALID_STR = ( "encapsulation is not a valid or supported format" )
ENCAP_OVERLAP_STR = ( "client or network encapsulation contains overlapping or "
                      "descending VLAN tags" )
MODE_CONFLICT_STR = ( "encapsulation submode cannot be used in conjunction "
                      "with encapsulation dot1q vlan command" )
MULTI_ENCAP_UNSUPPORTED_STR = ( "platform does not support multiple encapsulations "
                                "for a subinterface" )
UNMATCHED_CONFLICT_STR = ( "'client unmatched' cannot be specified for more than "
                           "one subinterface under a parent interface" )
UNTAGGED_CONFLICT_STR = ( "'client untagged ...' cannot be specified for more than "
                          "one subinterface under a parent interface" )

dot1q = 'dot1q'
dot1ad = 'dot1ad'
tpid = 'tpid'

#-------------------------------------------------------------------------------
# Rule to match the token class <vlan_set>, returning a corresponding VlanSet
# object. This limits the number tag specifications to 1.
#-------------------------------------------------------------------------------
flexEncapVlanRangeMatcher = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, 4094 ),
   False,
   'VLAN ID or range of VLAN IDs',
   maxRanges=1,
   value=vlanIdListFunc )

vlanTagMatcher = \
   flexEncapVlanRangeMatcher if toggleFlexEncapSubIntfMultiEncapEnabled() else \
   vlanIdMatcher

def vlanIdFromMatcher( vlanObj ):
   if not vlanObj:
      return None

   if isinstance( vlanObj, VlanSet ):
      return vlanObj
   else:
      return vlanObj.id

def field( value, tpidNameOrValue ):
   if value is None:
      return FlexEncapField.createAnyOrNone()
   elif value == 'client':
      return FlexEncapField.createClient()
   else:
      if tpidNameOrValue == dot1q:
         flexEncapField = FlexEncapField.createDot1qTagged()
      elif tpidNameOrValue == dot1ad:
         flexEncapField = FlexEncapField.createDot1adTagged()
      elif isinstance( tpidNameOrValue, int ):
         flexEncapField = FlexEncapField.createUserDefinedTpidTagged(
            tpidNameOrValue )
      else:
         assert False, f"invalid tpid {tpidNameOrValue!r}"
      fld = Tac.nonConst( flexEncapField )
      if isinstance( value, VlanSet ):
         rangeStrings = str( value ).split( ',' )
         for rangeString in rangeStrings:
            parts = [ int( p ) for p in rangeString.split( '-' ) ]
            startTag = parts[ 0 ]
            endTag = parts[ -1 ]
            fld.insertTaggedRange( FlexEncapRange( startTag, endTag ) )
      else:
         # We only support single tagged value at the moment
         assert isinstance( value, int )
         fld.insertTaggedRange( FlexEncapRange( value, value ) )
      return fld

def getOuterAndInnerTpidFromArgs( args, keywordPrefix ):
   outerTpid = ( args.get( f'{keywordPrefix}_TPID_DOT1Q_KEYWORD' )
                 or args.get( f'{keywordPrefix}_TPID_DOT1AD_KEYWORD' )
                 or args[ f'{keywordPrefix}_TPID_TPID_VALUE' ] )
   innerTpid = ( args.get( f'{keywordPrefix}_DOUBLE_INNER_TPID_DOT1Q_KEYWORD' )
                 or args.get( f'{keywordPrefix}_DOUBLE_INNER_TPID_DOT1AD_KEYWORD' )
                 or args.get( f'{keywordPrefix}_DOUBLE_INNER_TPID_TPID_VALUE' )
                 or outerTpid )
   assert outerTpid is not None and innerTpid is not None
   return ( outerTpid, innerTpid )

def parseClientEncapsulation( args ):
   """
   Parser for ClientExpression; the argument keys are a concatenation of
   of the key specified in ClientExpression and the corresponding
   SingleTagExprFactory/DoubleTagExprFactory
   """
   if 'unmatched' in args:
      return FlexEncapSpec.createUnmatched()
   if 'untagged' in args:
      return FlexEncapSpec.createUntagged()
   else:
      outerValue = vlanIdFromMatcher( args.get( 'CLIENT_SINGLE_TAG' ) or
                                      args.get( 'CLIENT_DOUBLE_OUTER_TAG' ) )
      ( outerTpid, innerTpid ) = getOuterAndInnerTpidFromArgs( args, 'CLIENT' )
      outer = field( outerValue, outerTpid )
      inner = field( vlanIdFromMatcher( args.get( 'CLIENT_DOUBLE_INNER_TAG' ) ),
                     innerTpid )
      return FlexEncapSpec.createTagged( outer, inner )

def parseNetworkEncapsulation( args ):
   if ( 'unmatched' in args and 'NETWORK_SINGLE_TAG' not in args and
         'NETWORK_DOUBLE_OUTER_TAG' not in args ):
      return FlexEncapSpec.createClient()
   elif 'NETWORK_CLIENT' in args:
      if 'NETWORK_INNER' in args:
         return FlexEncapSpec.createClientInner()
      else:
         return FlexEncapSpec.createClient()
   elif 'NETWORK_UNTAGGED' in args:
      return FlexEncapSpec.createUntagged()
   else:
      outerValue = vlanIdFromMatcher( args.get( 'NETWORK_SINGLE_TAG' ) or
                                      args.get( 'NETWORK_DOUBLE_OUTER_TAG' ) )
      if outerValue is not None:
         innerValue = ( args.get( 'NETWORK_DOUBLE_INNER_CLIENT' ) or
                        vlanIdFromMatcher( args.get( 'NETWORK_DOUBLE_INNER_TAG' ) ) )
         ( outerTpid, innerTpid ) = getOuterAndInnerTpidFromArgs( args, 'NETWORK' )
         outer = field( outerValue, outerTpid )
         inner = field( innerValue, innerTpid )
         return FlexEncapSpec.createTagged( outer, inner )
      else:
         return FlexEncapSpec.createAnyOrNone()

def setEncap( mode, args ):
   """
   handler used by different variants of EncapsulationCmd(s) to parse
   the data and set the encapsulation
   """
   client = parseClientEncapsulation( args )
   if client.size > 1 and not intfCapability.pwSubIntfMultiEncapSupported:
      mode.addWarning( MULTI_ENCAP_UNSUPPORTED_STR )
   network = parseNetworkEncapsulation( args )
   entry = FlexEncapEntry( client, network )

   if not entry.isValid():
      if entry.isAsymmetric():
         errorStr = ENCAP_ASYMMETRIC_STR
      elif entry.hasOverlap():
         errorStr = ENCAP_OVERLAP_STR
      else:
         errorStr = ENCAP_INVALID_STR
      mode.addErrorAndStop( errorStr )

   clientId = args.get( 'CLIENT_ID', FlexEncapConstants.defaultClientId )
   flexEncapMode = Dot1qEncapConfigMode.subIntfDot1qEncapConfigFlexEncap
   # If we're working with unmatched entries, if multiple unmatched entries are
   # configured under the same parentIntf, although we will treat this as a conflict
   # further downstream, it is better for the user if we can block this via CLI
   # itself
   checkSiblingConflict = client.type in ( SpecType.unmatched, SpecType.untagged )
   status = subIntfConfigDir.setVlanEncap(
         mode.intf.name, clientId, entry, flexEncapMode, checkSiblingConflict )
   if status.reason == VlanEncapReason.modeConflict:
      mode.addError( MODE_CONFLICT_STR )
   elif status.reason == VlanEncapReason.siblingConflict:
      # Except for unmatched and untagged subinterfaces, we will not support CLI
      # revertive behavior for dot1q conflict because this logic will become
      # extremely complicated once we support VLAN ranges, and multiple flex encap
      # entries.
      #
      # Even though it's easy to detect dot1q conflict in the initial phase,
      # we want to maintain a consistent user experience across phases.
      if client.type == SpecType.unmatched:
         mode.addError( UNMATCHED_CONFLICT_STR )
      elif client.type == SpecType.untagged:
         mode.addError( UNTAGGED_CONFLICT_STR )
      else:
         mode.addError( f"Unexpected failure reason: {status.reason}" )

def noOrDefaultEncap( mode, args ):
   """
   no or default handler used by different variants of EncapsulationCmd(s) to parse
   the data and delete the encapsulation

   The deletion does not do wild card matching, so we will only delete the
   encapsulation entry if there's an exact match
   """
   client = parseClientEncapsulation( args )
   network = parseNetworkEncapsulation( args )
   entry = FlexEncapEntry( client, network )
   clientId = args.get( 'CLIENT_ID', FlexEncapConstants.defaultClientId )
   flexEncapMode = Dot1qEncapConfigMode.subIntfDot1qEncapConfigFlexEncap
   reason = subIntfConfigDir.deleteVlanEncap(
         mode.intf.name, clientId, entry, flexEncapMode )
   if reason == VlanEncapReason.modeConflict:
      mode.addError( MODE_CONFLICT_STR )
   elif reason != VlanEncapReason.success:
      # sanity check
      mode.addError( f"Unexpected failure reason: {reason}" )

# Guards for TPIDs:
def flexEncapNonDot1qSupportedGuard( mode, token ):
   "Only check the capability if non-dot1q TPID"
   if token == dot1q or intfCapability.flexEncapNonDot1qSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def flexEncapCombinedGuard( mode, token ):
   '''
   Combines both non-dot1q and network translation guards, but the former is only
   applied if it's a non-dot1q TPID.
   '''
   if token == dot1q or intfCapability.flexEncapNonDot1qSupported:
      return vlanXlateFullySupportedGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def vlanXlateSupportedGuard( mode, token ):
   if intfCapability.flexEncapVlanXlateSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

# Other guards based on toggles or compatability:
def vlanXlateFullySupportedGuard( mode, token ):
   if intfCapability.flexEncapVlanXlateFullySupported:
      return vlanXlateSupportedGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def flexEncapClientIdSupportGuard( mode, token ):
   if ( toggleFlexEncapSubIntfClientIdEnabled() and
        intfCapability.pwSubIntfMultiEncapSupported ):
      # If multi-encap is supported, any non-reserved token is allowed.
      return None
   else:
      # Otherwise, client ID is not supported.
      return CliParser.guardNotThisPlatform

def flexEncapNetworkClientInnerSupportedGuard( mode, token ):
   if toggleFlexEncapNetworkClientInnerEnabled():
      return None
   else:
      return CliParser.guardNotThisEosVersion

def raiseInvalidInputError( msg ):
   raise CliParser.InvalidInputError( ": " + msg )

def getClientIds( mode ):
   subIntfConfig = subIntfConfigDir.intfConfig.get( mode.intf.name, None )
   if subIntfConfig and subIntfConfig.flexEncap:
      return subIntfConfig.flexEncap.entry
   return {}

class EncapsulationMode( CliMode.FlexEncap.EncapsulationBaseMode,
                         BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, intf ):
      CliMode.FlexEncap.EncapsulationBaseMode.__init__( self, intf.name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.intf = intf

#-------------------------------------------------------------------------------
# The "[no] encapsulation vlan" command
#-------------------------------------------------------------------------------
class EncapsulationsCmd( CliCommand.CliCommandClass ):
   syntax = "encapsulation vlan"
   noOrDefaultSyntax = syntax
   data = {
      "encapsulation": 'configure encapsulation',
      "vlan": 'VLAN encapsulation',
   }

   @staticmethod
   def handler( mode, args ):
      if not isinstance( mode.session_.mode_, CliMode.Intf.IntfMode ):
         raiseInvalidInputError( "not supported in interface range mode" )
      childMode = mode.childMode( EncapsulationMode, intf=mode.intf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      flexEncapMode = Dot1qEncapConfigMode.subIntfDot1qEncapConfigFlexEncap
      reason = subIntfConfigDir.deleteVlanEncap(
            mode.intf.name, FlexEncapConstants.defaultClientId, FlexEncapEntry(),
            flexEncapMode )
      if reason == VlanEncapReason.modeConflict:
         mode.addWarning( MODE_CONFLICT_STR )
      elif reason == VlanEncapReason.success:
         childMode = mode.childMode( EncapsulationMode, intf=mode.intf )
         childMode.removeComment()
      else:
         # sanity check
         mode.addError( f"Unexpected failure reason: {reason}" )

SubIntfModelet.addCommandClass( EncapsulationsCmd )

#-------------------------------------------------------------------------------
# Submode: encapsulation vlan
# - There are several versions of the commands allowed in this mode.  Refer to
#   the following command classes for syntax detail.
# - flexEncapVlanXlateSupported hwCapability is required to enable the optional
#   network commands
# - flexEncapVlanXlateFullySupported hwCapability is further required to enable
#   all VLAN translations
# - flexEncapNonDot1qSupported hwCapability is required to enable encapsulation
#   options besides just dot1q single/double tagged client expressions
#-------------------------------------------------------------------------------

class TpidExprFactory( CliCommand.CliExpressionFactory ):

   def __init__( self, useCombinedGuard=False ):
      super().__init__()
      self.useCombinedGuard = useCombinedGuard

   def guard( self, mode, token ):
      if self.useCombinedGuard:
         return flexEncapCombinedGuard( mode, token )
      else:
         return flexEncapNonDot1qSupportedGuard( mode, token )

   def generate( self, name ):
      dot1qKeyword = f"{name}_DOT1Q_KEYWORD"
      dot1adKeyword = f"{name}_DOT1AD_KEYWORD"
      tpidKeyword = f"{name}_TPID_KEYWORD"
      tpidValue = f"{name}_TPID_VALUE"

      class TpidExpression( CliCommand.CliExpression ):
         expression = f'{dot1qKeyword} | {dot1adKeyword}' \
            f' | ( {tpidKeyword} {tpidValue} )'
         data = {
            dot1qKeyword: CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( dot1q, 'Dot1q encapsulation' ),
               guard=self.guard ),
            dot1adKeyword: CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( dot1ad, 'Dot1ad encapsulation' ),
               guard=self.guard ),
            tpidKeyword: CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( tpid, 'User-defined TPID' ),
               guard=self.guard ),
            tpidValue: CliMatcher.IntegerMatcher( VlanTpid.min, VlanTpid.max,
                                                  helpdesc='The TPID value' ),
         }

      return TpidExpression

# Client ID allows everything except the following reserved words.  Note that a
# numeric ID is allowed.
reservedTokens = list( ReservedClientIds.attributes )
clientIdRegex = '(?!' + '|'.join( reservedTokens ) + r')[A-Za-z0-9_:{}\[\]-]+'
matcherClientId = CliMatcher.DynamicNameMatcher( getClientIds,
                                                 helpdesc='Client ID',
                                                 pattern=clientIdRegex )
nodeClientIdWithGuard = CliCommand.Node( matcher=matcherClientId,
                                         guard=flexEncapClientIdSupportGuard )

outerMatcher = CliMatcher.KeywordMatcher( 'outer', 'Outer VLAN tags' )
nodeOuter = CliCommand.Node( matcher=outerMatcher,
                             guard=theQinQL3SubintfSupportedGuard, noResult=True )

nodeOuterWithGuard = CliCommand.Node( matcher=outerMatcher,
                                      guard=vlanXlateFullySupportedGuard,
                                      noResult=True )
nodeNetwork = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher(
         "network", "Network encapsulation" ),
         guard=vlanXlateSupportedGuard,
      noResult=True )
nodeNetworkSideClient = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher(
         "client", "Retain client encapsulation" ) )
nodeInnerWithGuard = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher(
         'inner', 'Retain client inner tag' ),
      guard=flexEncapNetworkClientInnerSupportedGuard )
nodeUntagged = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( "untagged", "No encapsulation" ),
      guard=flexEncapNonDot1qSupportedGuard )
nodeUnmatched = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher(
         'unmatched',
         ( 'Match packets that do not match any other client '
           'encapsulation on the port' ) ),
      guard=flexEncapNonDot1qSupportedGuard )

class SingleTagExprFactory( CliCommand.CliExpressionFactory ):
   """
   A single tag expression that can be used by both the client and network portion
   ie.
      TAG-SPEC
   """
   def generate( self, name ):
      tagName = name + '_TAG'
      class SingleTagExpression( CliCommand.CliExpression ):
         expression = f'{tagName}'
         data = {
            tagName: vlanTagMatcher,
         }
      return SingleTagExpression

class DoubleTagExprFactory( CliCommand.CliExpressionFactory ):
   """
   A double tag expression that can be used by both the client and network portion
   ie.
      outer TAG-SPEC inner [ TPID ] ( TAG-SPEC | client )

   Note: 'outer' node is defined as a class variable (outerNode) because in some
         variants, we need the node to be guarded
   """
   def __init__( self, innerClient=False ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.innerClient = innerClient
      self.outerNode = nodeOuter

   def generate( self, name ):
      outerNodeKey = name + '_outer'
      outerTagName = name + '_OUTER_TAG'
      innerNodeKey = name + '_inner'
      innerTagName = name + '_INNER_TAG'
      innerTpid = name + '_INNER_TPID'
      innerClientName = name + '_INNER_CLIENT'
      if self.innerClient:
         innerTag = f'( {innerTagName} | {innerClientName} )'
      else:
         innerTag = innerTagName
      class DoubleTagExpression( CliCommand.CliExpression ):
         expression = f'{outerNodeKey} {outerTagName} {innerNodeKey} ' \
            f'[ {innerTpid} ] {innerTag}'
         data = {
            outerNodeKey : self.outerNode,
            outerTagName: vlanTagMatcher,
            innerNodeKey : CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher(
                                 'inner', 'Inner VLAN tags' ),
                              noResult=True ),
            innerTagName: vlanTagMatcher,
            innerTpid: TpidExprFactory(),
            innerClientName: nodeNetworkSideClient,
         }
      return DoubleTagExpression

class GuardedDoubleTagExprFactory( DoubleTagExprFactory ):
   """ outer node is guarded with vlanXlateFullySupportedGuard """
   def __init__( self, innerClient=False ):
      DoubleTagExprFactory.__init__( self )
      self.innerClient = innerClient
      self.outerNode = nodeOuterWithGuard

class ClientExpression( CliCommand.CliExpression ):
   expression = 'client [ CLIENT_ID ]'
   data = {
      'client': 'Client encapsulation',
      'CLIENT_ID': nodeClientIdWithGuard,
   }

# Command classes:
class UnmatchedClientCommand( CliCommand.CliCommandClass ):
   """
   client [ CLIENT_ID ] unmatched
   [ network ( TPID ( TAG-SPEC | ( outer TAG-SPEC inner [ TPID ] TAG-SPEC ) ) ) ]
   """
   syntax = """CLIENT_EXPR unmatched [ network ( NETWORK_TPID NETWORK_SINGLE |
                                                        NETWORK_DOUBLE ) ]"""
   noOrDefaultSyntax = syntax

   data = {
      'CLIENT_EXPR': ClientExpression,
      'unmatched': nodeUnmatched,
      "network": nodeNetwork,
      "NETWORK_TPID": TpidExprFactory( useCombinedGuard=True ),
      "NETWORK_SINGLE": SingleTagExprFactory(),
      "NETWORK_DOUBLE": DoubleTagExprFactory(),
   }

   handler = setEncap
   noOrDefaultHandler = noOrDefaultEncap

class UntaggedClientCommand( CliCommand.CliCommandClass ):
   """
   client [ CLIENT_ID ] untagged
   [ network ( untagged | TPID ( TAG-SPEC |
                                 ( outer TAG-SPEC inner [ TPID ] TAG-SPEC ) ) ) ]
   """
   syntax = """CLIENT_EXPR untagged [ network NETWORK_UNTAGGED |
                                         ( NETWORK_TPID NETWORK_SINGLE |
                                                        NETWORK_DOUBLE ) ]"""
   noOrDefaultSyntax = syntax

   data = {
      "CLIENT_EXPR": ClientExpression,
      "untagged": nodeUntagged,
      "network": nodeNetwork,
      "NETWORK_UNTAGGED": nodeUntagged,
      "NETWORK_TPID": TpidExprFactory( useCombinedGuard=True ),
      "NETWORK_SINGLE": SingleTagExprFactory(),
      "NETWORK_DOUBLE": DoubleTagExprFactory(),
   }

   handler = setEncap
   noOrDefaultHandler = noOrDefaultEncap

class SingleTagClientCmd( CliCommand.CliCommandClass ):
   """
   client TPID TAG-SPEC
   [ network ( client |
               TPID TAG-SPEC |
               ( outer TAG-SPEC inner [ TPID ] TAG-SPEC ) ) ]

   node network is guarded with vlanXlateSupportedGuard
   node (network) outer is guarded with vlanXlateFullySupportedGuarded
   """
   syntax = """CLIENT_EXPR CLIENT_TPID CLIENT_SINGLE
               [ network NETWORK_CLIENT |
                         ( NETWORK_TPID NETWORK_SINGLE | NETWORK_DOUBLE ) ]"""
   noOrDefaultSyntax = syntax

   data = {
      'CLIENT_EXPR' : ClientExpression,
      'CLIENT_TPID': TpidExprFactory(),
      'CLIENT_SINGLE' : SingleTagExprFactory(),
      "network" : nodeNetwork,
      "NETWORK_CLIENT" : nodeNetworkSideClient,
      "NETWORK_TPID": TpidExprFactory(),
      "NETWORK_SINGLE" : SingleTagExprFactory(),
      "NETWORK_DOUBLE": GuardedDoubleTagExprFactory( innerClient=True ),
   }

   handler = setEncap
   noOrDefaultHandler = noOrDefaultEncap

class DoubleTagClientCmd( CliCommand.CliCommandClass ):
   """
   client TPID outer TAG-SPEC inner TAG-SPEC
   [ network client [ inner ] | TPID TAG-SPEC |
                      ( outer TAG-SPEC inner [ TPID ] TAG-SPEC )

   node network is guarded with vlanXlateSupportedGuard
   network-side node dot1q is guarded with vlanXlateFullySupportedGuarded
   """
   syntax = """CLIENT_EXPR CLIENT_TPID CLIENT_DOUBLE
               [ network ( NETWORK_CLIENT [ NETWORK_INNER ] ) |
                         ( NETWORK_TPID NETWORK_SINGLE | NETWORK_DOUBLE ) ]"""
   noOrDefaultSyntax = syntax

   data = {
      'CLIENT_EXPR' : ClientExpression,
      'CLIENT_TPID': TpidExprFactory(),
      'CLIENT_DOUBLE' : DoubleTagExprFactory(),
      "network" : nodeNetwork,
      "NETWORK_CLIENT" : nodeNetworkSideClient,
      "NETWORK_INNER": nodeInnerWithGuard,
      "NETWORK_TPID": TpidExprFactory( useCombinedGuard=True ),
      "NETWORK_SINGLE" : SingleTagExprFactory(),
      "NETWORK_DOUBLE" : DoubleTagExprFactory(),
   }

   handler = setEncap
   noOrDefaultHandler = noOrDefaultEncap

# Commands are all added here.
EncapsulationMode.addCommandClass( SingleTagClientCmd )
EncapsulationMode.addCommandClass( DoubleTagClientCmd )
EncapsulationMode.addCommandClass( UnmatchedClientCommand )
if toggleFlexEncapUntaggedEnabled():
   EncapsulationMode.addCommandClass( UntaggedClientCommand )

def Plugin( entityManager ):
   global intfCapability
   global subIntfConfigDir
   intfCapability = LazyMount.mount( entityManager,
                                     "interface/hardware/capability",
                                     "Interface::Hardware::Capability", "r" )
   subIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/subintf",
                                         "Interface::SubIntfConfigDir", "w" )
