# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for FlowTracker configuration commands

import difflib
import io
import sys

import BasicCliModes
import CliCommand
import CliParser
import CliMatcher
from CliPlugin.FlowCongestionCli import (
   CongestionContext,
   CongestionMode,
)
from CliPlugin.FlowExporterCli import (
   ExporterContext,
   ExporterMode,
)
from CliPlugin.FlowGroupsCli import (
   GroupContext,
   GroupMode,
   GroupsMode,
)
from CliPlugin.FlowTrackingCliLib import (
   FlowTrackingMode,
   TrackerMode,
   guardCongestionExport,
   guardExportOnTcpState,
   guardFlowKw,
   guardFlowTableSize,
   guardIntfFtrHardware,
   guardIntfFtrDfw,
   guardIntfFtrSampled,
   guardEgressIntfSampling,
   guardIngressIntfFilteredSampling,
   guardMplsExport,
   getMplsHwCaps,
   getFtrCaps,
   getFtrSwCaps,
   trackerNameMatcher,
   guardInterval,
   guardActiveInterval,
   guardInactive,
   guardMonitor,
)
import CliPlugin.IntfCli as IntfCli
import CliPlugin.SubIntfCli as SubIntfCli
from CliPlugin.TunnelIntfCli import TunnelIntfConfigModelet
from CliPlugin.EthIntfCli import EthIntfModelet
from CliPlugin.SwitchIntfCli import SwitchIntfModelet
from CliPlugin.LagIntfCli import LagIntfConfigModelet
from CliPlugin.DpsIntfCli import DpsIntfConfigModelet
import CliToken.Cli
from CliToken.Flow import flowMatcherForConfigIf
import ConfigMount
from FlowTrackerCliUtil import (
   egressStr,
   ftrConfigType,
   ftrMirrorOnDropConfigType,
   ftrTypeHardware,
   ftrTypes,
   ftrTypeSampled,
   ftrTypeDfw,
   ftrTypeMirrorOnDrop,
   getFtrTypeFromArgs,
   getFlowTrackingCliConfigPath,
   showActiveGroups,
   showCongestion,
   showExporter,
   showFtr,
   showGroup,
   ftrTypeKwStr,
)
from FlowTrackerConst import (
   activeInterval,
   constants,
   inactiveTimeout,
   flowTableSize,
   spaceConst,
)
from ShowCommand import ShowCliCommandClass
import Tac
import Tracing
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'FlowTrackerCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2
t3 = traceHandle.trace3

FtConst = TacLazyType( 'FlowTracking::Constants' )

trackingConfig = {}

# enums used in FlowTrackerConfig.tac
sampleModeAll = Tac.enumValue( "FlowTracking::FlowTrackerSampleMode",
                               "sampleModeAll" )
sampleModeFiltered = Tac.enumValue( "FlowTracking::FlowTrackerSampleMode",
                                    "sampleModeFiltered" )

#---------------------------------------------------------
# Flow Tracker Context
#  - Python object for Tracker config thats pending commit
#---------------------------------------------------------

class TrackerContext:
   def __init__( self, config, ftrType, trName, trConfig=None ):
      self.ftrConfig_ = config
      self.ftrType_ = ftrType
      self.name_ = trName
      self.trConfig = trConfig
      self.groupCtx = {}
      self.exporterCtx = {}
      self.congestionCtx = None
      self.groupToSeqnoMap = {}
      self.monitoredCpuQueue = set()
      self.seqnoToGroupMap = {}
      self.groupResequencingInProgress = False
      self.changed = False
      self.inactiveTimeout = None
      self.activeInterval = None
      self.tcpStateChangeExport = None
      self.mplsExport = None
      self.flowTableSize = None
      t0( 'creating new tracker context', ftrType, trName, trConfig )

      if not trConfig:
         # new tracker
         self.initFromDefaults()
      else:
         self.initFromConfig()

   def ftrType( self ):
      return self.ftrType_

   def initFromDefaults( self ):
      self.inactiveTimeout = inactiveTimeout.timeoutDefault
      self.activeInterval = activeInterval.intervalDefault
      self.tcpStateChangeExport = constants.tcpStateChangeExportDefault
      self.mplsExport = constants.mplsExportDefault
      self.flowTableSize = flowTableSize.sizeDefault
      self.setChanged()

   def initFromConfig( self ):
      self.inactiveTimeout = self.trConfig.inactiveTimeout
      self.activeInterval = self.trConfig.activeInterval
      self.tcpStateChangeExport = self.trConfig.tcpStateChangeExport
      self.mplsExport = self.trConfig.mplsExport
      self.flowTableSize = self.trConfig.flowTableSize
      if self.trConfig.congestionConfig:
         self.congestionCtx = CongestionContext( self,
                                                 self.trConfig.congestionConfig )
      for expName, expConfig in self.trConfig.expConfig.items():
         t0( 'Creating new exporter context', expName )
         self.exporterCtx[ expName ] = \
                  ExporterContext( self, expName, expConfig )
      self.groupToSeqnoMap = {}
      self.seqnoToGroupMap = {}
      for fgName, fgConfig in self.trConfig.fgConfig.items():
         t0( 'creating new group context', fgName )
         self.groupCtx[ fgName ] = \
               GroupContext( self, fgName, fgConfig )
         seqno = fgConfig.seqno
         self.groupToSeqnoMap[ fgName ] = seqno
         self.seqnoToGroupMap[ seqno ] = fgName
      self.monitoredCpuQueue.update( self.trConfig.monitoredCpuQueue )
      if self.trConfig.groupResequencingInProgress:
         self.resequenceExistingFlowGroups()
      if self.trConfig.groupsCommitInProgress:
         # It is possible that group commit was in progress earlier such that all
         # the groups were actually committed in Sysdb but CLI crashes before
         # groupsCommitted flag was set to true. In this case when the system
         # comes up, everything will be there in Sysdb config, so
         # GroupContext.changed_ flag will not be set to true for any group.
         # For such cases we use the flag groupsCommitInProgress which will preserve
         # state through crashes. So when the system comes up after the crash,
         # if groupsCommitInProgress was true, it means that groupsCommitted was not
         # able to be set to true earlier, so we do it now.
         self.setChanged()

   def resequenceExistingFlowGroups( self ):
      self.setResequencingInProgress( True )
      self.groupToSeqnoMap = {}
      newSeqnoToGroupMap = {}
      # Re-assign seqno at default interval
      newSeqno = constants.fgSeqnoInterval
      for seqno in sorted( self.seqnoToGroupMap ):
         fgName = self.seqnoToGroupMap[ seqno ]
         self.groupContext( fgName ).seqno_ = newSeqno
         self.groupContext( fgName ).setChanged()
         self.groupToSeqnoMap[ fgName ] = newSeqno
         newSeqnoToGroupMap[ newSeqno ] = fgName
         newSeqno += constants.fgSeqnoInterval
      self.seqnoToGroupMap = newSeqnoToGroupMap

   def setChanged( self ):
      self.changed = True

   def setResequencingInProgress( self, state ):
      self.groupResequencingInProgress = state

   def commit( self ):
      if not self.changed:
         t0( 'commit: No changes found' )
         return
      if not self.trConfig:
         t0( 'commit: creating new FlowTrackerConfig', self.name_ )
         self.trConfig = trackingConfig[ 
                           self.ftrType_ ].newFlowTrackerConfig( self.name_ )

      self.trConfig.inactiveTimeout = self.inactiveTimeout
      self.trConfig.activeInterval = self.activeInterval
      self.trConfig.tcpStateChangeExport = self.tcpStateChangeExport
      self.trConfig.mplsExport = self.mplsExport
      self.trConfig.flowTableSize = self.flowTableSize

      if self.congestionCtx:
         self.congestionCtx.commitCongestion( self.trConfig )
         if self.congestionCtx.deleted():
            self.congestionCtx = None

      expToDelete = []
      for expName, ctx in self.exporterCtx.items():
         ctx.commitExporter( self.trConfig )
         if ctx.deleted():
            expToDelete.append( expName )
      t0( 'commit: deleting Exporter context', expToDelete )
      for expName in expToDelete:
         del self.exporterCtx[ expName ]

      self.trConfig.groupsCommitted = False
      self.trConfig.groupResequencingDone = False
      self.trConfig.groupResequencingInProgress = self.groupResequencingInProgress
      fgToDelete = []

      for seqno in sorted( self.seqnoToGroupMap, reverse=True ):
         fgName = self.seqnoToGroupMap[ seqno ]
         ctx = self.groupCtx[ fgName ]
         ctx.commitGroup( self.trConfig )
         if ctx.deleted():
            fgToDelete.append( fgName )
         if ctx.deleted() or ctx.changed():
            self.trConfig.groupsCommitInProgress = True
      t0( 'commit: deleting FlowGroup context', fgToDelete )
      for fgName in fgToDelete:
         del self.groupCtx[ fgName ]
         seqno = self.groupToSeqnoMap[ fgName ]
         del self.seqnoToGroupMap[ seqno ]
         del self.groupToSeqnoMap[ fgName ]

      # Reconcile monitored cpu queue
      for cpuQueueName in self.trConfig.monitoredCpuQueue:
         if cpuQueueName not in self.monitoredCpuQueue:
            del self.trConfig.monitoredCpuQueue[ cpuQueueName ]
      for cpuQueueName in self.monitoredCpuQueue:
         self.trConfig.monitoredCpuQueue.add( cpuQueueName )

      if self.groupResequencingInProgress:
         self.setResequencingInProgress( False )
         self.trConfig.groupResequencingDone = True
         self.trConfig.groupResequencingInProgress = self.groupResequencingInProgress

      if self.trConfig.groupsCommitInProgress:
         self.trConfig.groupsCommitted = True
         self.trConfig.groupsCommitInProgress = False

   def newExporterContext( self, expName ):
      expConfig = None
      if self.trConfig is not None:
         # Config could changed after entering flow-tracking mode.
         # Create context with existing Sysdb config.
         expConfig = self.trConfig.expConfig.get( expName )
      self.exporterCtx[ expName ] = ExporterContext( self, expName, expConfig )
      return self.exporterCtx[ expName ]

   def activeIntervalIs( self, _activeInterval ):
      if self.activeInterval != _activeInterval:
         self.activeInterval = _activeInterval
         self.setChanged()

   def inactiveTimeoutIs( self, timeout ):
      if self.inactiveTimeout != timeout:
         self.inactiveTimeout = timeout
         self.setChanged()

   def tcpStateChangeExportIs( self, enable ):
      if self.tcpStateChangeExport != enable:
         self.tcpStateChangeExport = enable
         self.setChanged()

   def mplsExportIs( self, enable ):
      if self.mplsExport != enable:
         self.mplsExport = enable
         self.setChanged()

   def flowTableSizeIs( self, size ):
      if self.flowTableSize != size:
         self.flowTableSize = size
         self.setChanged()

   def monitorCpuQueueIs( self, cpuQueueName, no=False ):
      if no:
         if cpuQueueName not in self.monitoredCpuQueue:
            return
         self.monitoredCpuQueue.remove( cpuQueueName )
      else:
         self.monitoredCpuQueue.add( cpuQueueName )
      self.setChanged()

   def trackerName( self ):
      return self.name_

   def ftrConfig( self ):
      return self.ftrConfig_

   def exporterConfig( self, expName ):
      if self.trConfig:
         if expName in self.trConfig.expConfig:
            return self.trConfig.expConfig[ expName ]
      return None

   def exporterContext( self, expName ):
      if expName in self.exporterCtx:
         return self.exporterCtx[ expName ]
      else:
         return None

   def newGroupContext( self, fgName ):
      fgConfig = None
      if self.trConfig is not None:
         # Config could be changed after entering flow-tracking mode.
         # Create context with existing Sysdb config.
         fgConfig = self.trConfig.fgConfig.get( fgName )
      self.groupCtx[ fgName ] = GroupContext( self, fgName, fgConfig )
      return self.groupCtx[ fgName ]

   def groupConfig( self, fgName ):
      if self.trConfig:
         return self.trConfig.fgConfig.get( fgName )
      return None

   def groupContext( self, fgName ):
      if fgName in self.groupCtx:
         return self.groupCtx[ fgName ]
      else:
         return None

   def gotoCongestionMode( self, mode ):
      t1( 'gotoCongestionMode of', self.trackerName )

      if self.congestionCtx is None:
         t0( 'No congestionContext found, creating new congestion context' )
         congestionConfig = self.trConfig.congestionConfig if self.trConfig else None
         self.congestionCtx = CongestionContext( self, congestionConfig )
      elif self.congestionCtx.deleted() or self.congestionCtx.congestionConfig():
         t0( 'Re-entering a deleted congestionContext' )
         self.congestionCtx.deletedIs( False )

      childMode = mode.childMode( CongestionMode, context=self.congestionCtx )
      mode.session_.gotoChildMode( childMode )

   def noCongestionMode( self ):
      t1( 'noCongestionMode of', self.trackerName )

      if self.congestionCtx is None:
         return
      self.congestionCtx.deletedIs( True )

#-------------------------------------------------------------------------------
# "[no|default] tracker <flow-tracker-name>" command
#-------------------------------------------------------------------------------

def gotoTrackerMode( mode, trackerName ):
   t1( 'gotoTrackerMode of', trackerName )
   if len( trackerName ) > FtConst.confNameMaxLen:
      mode.addError(
         f'Tracker name is too long (maximum {FtConst.confNameMaxLen})' )
      return

   if trackerName == 'egress':
      mode.addError( "The tracker name '%s' is reserved." % trackerName )
      return

   ftrCtx = mode.context()
   trCtx = ftrCtx.trackerContext( trackerName )
   if trCtx is None:
      t0( 'Creating new tracker context for', trackerName )
      trCtx = ftrCtx.newTrackerContext( trackerName )
   elif trCtx.trConfig:
      # Re-enter tracker mode, update context with latest config.
      t0( 'Updating tracker context for', trackerName )
      trCtx.initFromConfig()
   childMode = mode.childMode( TrackerMode, ftrCtx=ftrCtx, trCtx=trCtx )
   mode.session_.gotoChildMode( childMode )

def noTrackerMode( mode, trackerName ):
   t1( 'noTrackerMode of', trackerName )
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, TrackerMode ) and lastMode.context()
         and lastMode.context().trackerName() == trackerName ):
      t0( 'lastMode is: TrackerMode' )
      # If no tracker is issued in ftr mode then delete trConfig from
      # context to avoid committing it again
      lastMode.context().trConfig = None
      lastMode.context_ = None
   elif ( isinstance( lastMode, ( ExporterMode, GroupMode, GroupsMode ) ) and
         lastMode.context() and
         lastMode.context().trackerName() == trackerName ):
      t0( 'lastMode is: ', lastMode )
      # If no tracker is issued in exporter/groups/group mode then delete
      # trConfig from ftrMode
      lastMode.context().trCtx_.trConfig = None

   del trackingConfig[ mode.context().ftrType ].flowTrackerConfig[ trackerName ]

   ftrCtx = mode.context()
   if trackerName in ftrCtx.trCtx:
      del ftrCtx.trCtx[ trackerName ]

trackerMatcherForConfig = CliMatcher.KeywordMatcher(
                                    'tracker',
                                    helpdesc='Configure flow tracker' )

def getTrackerNames( mode ):
   # trCtx is a dict mapping tracker name -> TrackerContext
   return mode.context().trCtx

class TrackerCmd( CliCommand.CliCommandClass ):
   syntax = '''tracker TRACKER_NAME'''
   noOrDefaultSyntax = syntax

   data = {
      'tracker' : trackerMatcherForConfig,
      'TRACKER_NAME' : CliMatcher.DynamicNameMatcher( getTrackerNames,
                                          "Flow tracker name" ),
   }

   @staticmethod
   def handler( mode, args ):
      gotoTrackerMode( mode, trackerName=args[ "TRACKER_NAME" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noTrackerMode( mode, trackerName=args[ "TRACKER_NAME" ] )

FlowTrackingMode.addCommandClass( TrackerCmd )

#-------------------------------------------------------------------------------
# "[no|default] record export on interval <interval>" command,
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

recordMatcher = CliMatcher.KeywordMatcher(
                  'record', helpdesc='Configure flow record export' )

exportMatcher = CliMatcher.KeywordMatcher(
                  'export', helpdesc='Configure flow record export' )

onMatcher = CliMatcher.KeywordMatcher(
                  'on', helpdesc='Configure flow record export' )

intervalMatcher = CliCommand.guardedKeyword( 'interval',
                     helpdesc='Configure flow record export interval',
                     guard=guardInterval )

activeIntervalKw = CliCommand.guardedKeyword( '0',
                     helpdesc='Disable active flow record export on interval',
                     guard=guardActiveInterval )

class ActiveIntervalExpression( CliCommand.CliExpression ):
   expression = 'INTERVAL | 0'
   data = {
         'INTERVAL' : CliMatcher.IntegerMatcher(
            activeInterval.minInterval,
            activeInterval.maxInterval,
            helpdesc='Flow record export interval in milliseconds' ),
         '0' : activeIntervalKw
         }

   @staticmethod
   def adapter( mode, args, argList ):
      # Convert '0' to an INTERVAL
      if '0' in args:
         assert 'INTERVAL' not in args
         args[ 'INTERVAL' ] = 0
         del args[ '0' ]

class ActiveIntervalCommand( CliCommand.CliCommandClass ):
   syntax = '''record export on interval INTERVAL'''
   noOrDefaultSyntax = '''record export on interval ...'''

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'on' : onMatcher,
      'interval' : intervalMatcher,
      'INTERVAL' : ActiveIntervalExpression,
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().activeIntervalIs(
            args.get( 'INTERVAL', activeInterval.intervalDefault ) )

   noOrDefaultHandler = handler

TrackerMode.addCommandClass( ActiveIntervalCommand )

#-------------------------------------------------------------------------------
# "[no|default] record export on inactive timeout <timeout>" command,
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

inactiveMatcher = CliCommand.guardedKeyword( 'inactive',
                     helpdesc='Configure flow record export inactive timeout',
                     guard=guardInactive )

class InactiveTimeoutCommand( CliCommand.CliCommandClass ):
   syntax = '''record export on inactive timeout TIMEOUT'''
   noOrDefaultSyntax = '''record export on inactive timeout ...'''

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'on' : onMatcher,
      'inactive' : inactiveMatcher,
      'timeout' : 'Configure flow record export inactive timeout',
      'TIMEOUT' : CliMatcher.IntegerMatcher(
                     inactiveTimeout.minTimeout,
                     inactiveTimeout.maxTimeout,
                     helpdesc='Flow record inactive export timeout in milliseconds' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().inactiveTimeoutIs(
            args.get( 'TIMEOUT', inactiveTimeout.timeoutDefault ) )

   noOrDefaultHandler = handler

TrackerMode.addCommandClass( InactiveTimeoutCommand )

#-------------------------------------------------------------------------------
# "[no|default] record export on tcp state change" command,
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

tcpMatcher = CliCommand.guardedKeyword( 'tcp',
                  helpdesc='Configure flow record export on TCP state',
                  guard=guardExportOnTcpState )

class ExportOnTcpStateCmd( CliCommand.CliCommandClass ):
   syntax = '''record export on tcp state change'''
   noOrDefaultSyntax = syntax

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'on' : onMatcher,
      'tcp' : tcpMatcher,
      'state' : 'Configure flow record export on TCP state',
      'change' : 'Configure flow record export on TCP state change',
   }

   @staticmethod
   def handler( mode, args ):
      t2( "setExportOnTcpState: ", mode, mode.ftrTypeStr )
      mode.context().tcpStateChangeExportIs( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().tcpStateChangeExportIs(
         constants.tcpStateChangeExportDefault )

TrackerMode.addCommandClass( ExportOnTcpStateCmd )

#-------------------------------------------------------------------------------
# [no|default] record export mpls
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------
mplsExportKw = CliCommand.guardedKeyword( 'mpls',
                                          'Export MPLS forwarding information',
                                          guardMplsExport )

class ExportEncapsCmd( CliCommand.CliCommandClass ):
   syntax = "record export mpls"
   noOrDefaultSyntax = syntax

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'mpls' : mplsExportKw,
   }

   @staticmethod
   def handler( mode, args ):
      t2( "set 'record export mpls'", mode, mode.ftrTypeStr )
      mode.context().mplsExportIs( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t2( "reset 'record export mpls'", mode, mode.ftrTypeStr )
      mode.context().mplsExportIs( constants.mplsExportDefault )

TrackerMode.addCommandClass( ExportEncapsCmd )

# -------------------------------------------------------------------------------
# [no|default] record export on congestion
# in "config-ftr-tr" mode.
# -------------------------------------------------------------------------------
congestionMatcher = CliCommand.guardedKeyword( 'congestion',
                       helpdesc='Export congestion information',
                       guard=guardCongestionExport )

class CongestionCmd( CliCommand.CliCommandClass ):
   syntax = "record export on congestion"
   noOrDefaultSyntax = syntax

   data = {
      'record' : recordMatcher,
      'export' : exportMatcher,
      'on' : onMatcher,
      'congestion' : congestionMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().gotoCongestionMode( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().noCongestionMode()

TrackerMode.addCommandClass( CongestionCmd )

#-------------------------------------------------------------------------------
# [no|default] flow table size <size> entries
# in "config-ftr-tr" mode.
#-------------------------------------------------------------------------------

flowKw = CliCommand.guardedKeyword( 'flow', 'Configure flow parameters',
                                    guardFlowKw )
flowTableSizeKw = CliCommand.guardedKeyword( 'size',
                     'Configure maximum flow table size', guardFlowTableSize )

class FlowTableSizeCmd( CliCommand.CliCommandClass ):
   syntax = '''flow table size SIZE entries'''
   noOrDefaultSyntax = '''flow table size ...'''

   data = {
      'flow' : flowKw,
      'table' : 'Configure flow table parameters',
      'size' : flowTableSizeKw,
      'SIZE' : CliMatcher.IntegerMatcher(
                  flowTableSize.minSize,
                  flowTableSize.maxSize,
                  helpdesc='Maximum number of entries in flow table' ),
      'entries' : 'Flow table entries'
   }
   @staticmethod
   def handler( mode, args ):
      size = args.get( 'SIZE', flowTableSize.sizeDefault )
      t2( "set maximum flow table size", mode, mode.ftrTypeStr, size )
      mode.context().flowTableSizeIs( size )

   noOrDefaultHandler = handler

TrackerMode.addCommandClass( FlowTableSizeCmd )

class AbortCmd( CliCommand.CliCommandClass ):
   syntax = '''abort'''
   data = {
         'abort': CliToken.Cli.abortMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

TrackerMode.addCommandClass( AbortCmd )

# -------------------------------------------------------------------------------
#  [no|default] monitor cpu-queue <cpuQueue>
#  in "config-ftr-tr" mode.
# -------------------------------------------------------------------------------
monitorMatcher = CliCommand.guardedKeyword( 'monitor',
                    helpdesc='Configure CPU queue monitoring',
                    guard=guardMonitor )

def supportedMonitorCpuQueueNames( mode ):
   ftrType = mode.context().ftrType()
   ftrCaps = getFtrCaps( ftrType )
   return ftrCaps.monitoredCpuQueueSupported.keys()

cpuQueueCompletion = [ CliParser.Completion( 'WORD', 'CPU queue to monitor',
                                             literal=False ) ]

class MonitorCpuQueueCmd( CliCommand.CliCommandClass ):
   syntax = "monitor cpu-queue CPU_QUEUE"
   noOrDefaultSyntax = syntax

   data = {
      'monitor' : monitorMatcher,
      'cpu-queue' : 'Configure CPU queue to monitor',
      'CPU_QUEUE' : CliMatcher.DynamicKeywordMatcher(
                                       supportedMonitorCpuQueueNames,
                                       emptyTokenCompletion=cpuQueueCompletion,
                                       alwaysMatchInStartupConfig=True )
   }

   @staticmethod
   def handler( mode, args ):
      mode.context().monitorCpuQueueIs( args[ 'CPU_QUEUE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context().monitorCpuQueueIs( args[ 'CPU_QUEUE' ], no=True )

TrackerMode.addCommandClass( MonitorCpuQueueCmd )

#----------------
# interface
#---------------

def setFlowTracker( mode, intfName, ftrType, direction, sampleMode=sampleModeAll,
                    no=None, trackerName=None ):
   intfId = Tac.Value( 'Arnet::IntfId', intfName )
   trcfg = trackingConfig[ ftrType ]
   config = trcfg.flowTrackerIntfConfig if direction != egressStr else\
            trcfg.flowTrackerEgressIntfConfig
   ftrIntf = config.get( intfId )
   if no:
      if not trackerName or not ftrIntf or trackerName == ftrIntf.trackerName:
         if ftrIntf and ftrIntf.trackerName:
            t1( 'delete tracker', ftrType, ftrIntf.trackerName, 'from', intfName )
         del config[ intfId ]
   else:
      if not trackerName:
         return
      if ftrIntf and ( ftrIntf.trackerName != trackerName
                       or ftrIntf.sampleMode != sampleMode ):
         t1( 'delete tracker', ftrType, ftrIntf.trackerName, 'from', intfName )
         del config[ intfId ]
         ftrIntf = None
      t1( 'set tracker', ftrType, trackerName, 'on', intfName )
      if ftrIntf:
         assert trackerName == ftrIntf.trackerName
         assert sampleMode == ftrIntf.sampleMode
      else:
         if trackerName not in trcfg.flowTrackerConfig:
            mode.addWarning(
                  "Flow tracker: %s doesn't exist. The configuration will "
                  "not take effect until the flow tracker is configured." %
                  trackerName )
         if direction == egressStr:
            ftrIntf = trcfg.newFlowTrackerEgressIntfConfig( intfId,
                                                            trackerName, sampleMode )
         else:
            ftrIntf = trcfg.newFlowTrackerIntfConfig( intfId, trackerName,
                                                      sampleMode )
   return

sampledIfConfigKw = CliCommand.guardedKeyword( 'sampled',
                        helpdesc='Configure flow tracker sampled',
                        guard=guardIntfFtrSampled,
                        storeSharedResult=True )

filteredSampleIngressIfConfigKw = CliCommand.guardedKeyword( 'filtered',
                        helpdesc='Configure filtered sampling',
                        guard=guardIngressIntfFilteredSampling )

sampledEgressIfConfigKw = CliCommand.guardedKeyword( egressStr,
                     helpdesc='Configure flow tracker sampled on egress interface',
                     guard=guardEgressIntfSampling,
                     storeSharedResult=True )

hardwareIfConfigKw = CliCommand.guardedKeyword( 'hardware',
                        helpdesc='Configure flow tracker hardware',
                        guard=guardIntfFtrHardware,
                        storeSharedResult=True )

dfwIfConfigFwKw = CliCommand.guardedKeyword( 'firewall',
                        helpdesc='Configure distributed firewall flow tracker',
                        guard=guardIntfFtrDfw,
                        storeSharedResult=True )

dfwIfConfigDistKw = CliCommand.guardedKeyword( 'distributed',
                        helpdesc='Configure distributed firewall flow tracker',
                        guard=guardIntfFtrDfw,
                        storeSharedResult=True )

class FlowTrackerIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''flow tracker
               ( ( sampled TRACKER_NAME [ filtered ] ) |
                 ( sampled egress TRACKER_NAME ) |
                 ( hardware TRACKER_NAME ) |
                 ( firewall distributed TRACKER_NAME ) )'''
   noOrDefaultSyntax = '''flow tracker
                          ( ( sampled [ egress ] ) | hardware |
                            ( firewall distributed ) ) ...'''

   data = {
         'flow' : flowMatcherForConfigIf,
         'tracker' : trackerMatcherForConfig,
         'sampled' : sampledIfConfigKw,
         'egress' : sampledEgressIfConfigKw,
         'hardware' : hardwareIfConfigKw,
         'firewall' : dfwIfConfigFwKw,
         'distributed' : dfwIfConfigDistKw,
         'TRACKER_NAME' : trackerNameMatcher.matcher_,
         'filtered' : filteredSampleIngressIfConfigKw,
   }

   @staticmethod
   def handler( mode, args ):
      ftrType = getFtrTypeFromArgs( args )
      direction = args.get( 'egress', '' )
      sampleMode = sampleModeFiltered if 'filtered' in args else sampleModeAll
      setFlowTracker( mode, mode.intf.name, ftrType, direction,
                      sampleMode, trackerName=args[ "TRACKER_NAME" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ftrType = getFtrTypeFromArgs( args )
      direction = args.get( 'egress', '' )
      setFlowTracker( mode, mode.intf.name, ftrType, direction,
                      no=True )

SwitchIntfModelet.addCommandClass( FlowTrackerIntfConfigCmd )
LagIntfConfigModelet.addCommandClass( FlowTrackerIntfConfigCmd )
EthIntfModelet.addCommandClass( FlowTrackerIntfConfigCmd )
TunnelIntfConfigModelet.addCommandClass( FlowTrackerIntfConfigCmd )
SubIntfCli.SubIntfModelet.addCommandClass( FlowTrackerIntfConfigCmd )
DpsIntfConfigModelet.addCommandClass( FlowTrackerIntfConfigCmd )

#-----------------------------------------------------------------
#              show active|pending|diff
#-----------------------------------------------------------------

def showPendingTracker( trCtx, output=None, space='' ):
   if trCtx is None:
      return
   if output is None:
      output = sys.stdout
   ftrType = trCtx.ftrType()
   ftrCaps = getFtrCaps( ftrType )
   ftrSwCaps = getFtrSwCaps( ftrType )
   showFtr( output, trCtx.name_, trCtx, ftrType, getMplsHwCaps(),
            ftrTypeCaps=ftrCaps, ftrTypeSwCaps=ftrSwCaps,
            cliSave=False, space=space )
   if trCtx.congestionCtx and not trCtx.congestionCtx.deleted():
      showCongestion( output, trCtx.congestionCtx, ftrSwCaps,
                      cliSave=False, space=space )
   t3( 'trCtx.exporterCtx: ', trCtx.exporterCtx )
   for expName in sorted( trCtx.exporterCtx ):
      showExporter( output, expName, trCtx.exporterCtx[ expName ],
                    ftrTypeKwStr[ ftrType ], ftrCaps, ftrSwCaps, cliSave=False,
                    space=space )
   if trCtx.groupCtx:
      space += spaceConst
      output.write( '%sgroups\n' % space )
      space += spaceConst
      for groupSeqno in sorted( trCtx.seqnoToGroupMap ):
         groupName = trCtx.seqnoToGroupMap[ groupSeqno ]
         showGroup( output, groupName, trCtx.groupCtx[ groupName ], cliSave=False,
                    space=space )

def showActiveTracker( ftrType, trName, trConfig, output=None, space='' ):
   if trConfig is None:
      return
   if output is None:
      output = sys.stdout
   ftrCaps = getFtrCaps( ftrType )
   ftrSwCaps = getFtrSwCaps( ftrType )
   showFtr( output, trName, trConfig, ftrType, getMplsHwCaps(),
            ftrTypeCaps=ftrCaps, ftrTypeSwCaps=ftrSwCaps,
            cliSave=False, space=space )
   if trConfig.congestionConfig:
      showCongestion( output, trConfig.congestionConfig,
                      ftrSwCaps, cliSave=False, space=space )
   for expName, expConfig in sorted( trConfig.expConfig.items() ):
      showExporter( output, expName, expConfig, ftrTypeKwStr[ ftrType ],
                    ftrCaps, ftrSwCaps, cliSave=False, space=space )
   showActiveGroups( output, trConfig, space=space )

def _showDiff( trName, trCtx, trConfig ):
   # generate diff between active and pending
   activeOutput = io.StringIO()
   showActiveTracker( trCtx.ftrType(), trName, trConfig, output=activeOutput )
   pendingOutput = io.StringIO()
   showPendingTracker( trCtx, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(),
                                lineterm='' )
   print( '\n'.join( diff ) )

class ShowTracker( ShowCliCommandClass ):
   syntax = 'show ( pending | active | diff )'

   data = {
      'active' : BasicCliModes.showActiveNode,
      'pending': 'Display the new tracker configuration to be applied',
      'diff': ( 'Display the diff between active tracker configuration'
                'to be applied' ),
    }
 
   @staticmethod
   def handler( mode, args ):
      trName = mode.trackerName()
      trCtx = mode.context()
      trConfig = trCtx.ftrConfig().flowTrackerConfig.get( trName )
      if 'diff' in args:
         _showDiff( trName, trCtx, trConfig )
      elif 'active' in args:
         showActiveTracker( trCtx.ftrType(), trName, trConfig )
      else:
         showPendingTracker( trCtx )
 
TrackerMode.addShowCommandClass( ShowTracker )

#-------------------------------------------------------------------------------
# The FlowTrackerIntf class is used to remove the FlowTracker IntfConfig object
# when an interface is deleted.
#-------------------------------------------------------------------------------
class FlowTrackerIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      for ftrType in [ ftrTypeSampled, ftrTypeHardware, ftrTypeDfw ]:
         for dirStr in [ '', egressStr ]:
            setFlowTracker( None, self.intf_.name, ftrType, direction=dirStr,
                            no=True )

#---------------------------------------------------------
def Plugin( em ):
   for ftrType in ftrTypes:
      ftrEntityType = ( ftrMirrorOnDropConfigType if ftrType == ftrTypeMirrorOnDrop
                        else ftrConfigType )
      trackingConfig[ ftrType ] = \
         ConfigMount.mount( em,
                            getFlowTrackingCliConfigPath( ftrType ),
                            ftrEntityType, 'wi' )

   IntfCli.Intf.registerDependentClass( FlowTrackerIntf )
